/*
#   mp4tag.c: reads/writes mp4 metadata and gets playback time
#   Copyright (C) 2006 Stephen Fairchild (s-fairchild@users.sourceforge.net)
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program in the file entitled COPYING.
#   If not, see <http://www.gnu.org/licenses/>.
*/

#include "../config.h"
#ifdef HAVE_MP4V2

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <mp4.h>
#include "mp4tag.h"

int mp4_file_info(char *pathname)
   {
   MP4FileHandle h;
   char *artist;
   char *name;
   int duration;
   
   if (!(h = MP4Read(pathname, 0)))
      {
      fprintf(stderr, "failed to open file %s\n", pathname);
      printf("idjcmixer: mp4fileinfo Not Valid\n");
      fflush(stdout);
      return 0;
      }

   if (MP4GetMetadataArtist(h, &artist) && artist != NULL)
      {
      fprintf(stderr, "artist=%s\n", artist);
      printf("idjcmixer: mp4fileinfo artist=%s\n", artist);
      free(artist);
      }

   if (MP4GetMetadataName(h, &name) && name != NULL)
      {
      fprintf(stderr, "title=%s\n", name);
      printf("idjcmixer: mp4fileinfo title=%s\n", name);
      free(name);
      }

   duration = MP4GetDuration(h) / MP4GetTimeScale(h);
   fprintf(stderr, "duration = %d\n", duration);
   printf("idjcmixer: mp4fileinfo length=%d\n"
          "idjcmixer: mp4fileinfo end\n", duration);
   fflush(stdout);
   MP4Close(h);
   return 1;
   }

int mp4_tag_read(char *pathname)
   {
   MP4FileHandle h;
   char *s;
   u_int16_t i1, i2;
   char *header = "idjcmixer: mp4tagread ";

   if (!(h = MP4Read(pathname, 0)))
      {
      fprintf(stderr, "failed to open file %s\n", pathname);
      printf("idjcmixer: mp4fileinfo Not Valid\n");
      fflush(stdout);
      return 0;
      }

   if (MP4GetMetadataArtist(h, &s))
      {
      printf("%sartist=%s\n", header, s);
      free(s);
      }

   if (MP4GetMetadataName(h, &s))
      {
      printf("%sname=%s\n", header, s);
      free(s);
      }

   if (MP4GetMetadataAlbum(h, &s))
      {
      printf("%salbum=%s\n", header, s);
      free(s);
      }

   if (MP4GetMetadataYear(h, &s))
      {
      printf("%syear=%s\n", header, s);
      free(s);
      }

   if (MP4GetMetadataTrack(h, &i1, &i2))
      {
      if (i1)
         printf("%stracknum=%u\n", header, i1);
      if (i2)
         printf("%stracktotal=%u\n", header, i2);
      }

   if (MP4GetMetadataGenre(h, &s))
      {
      printf("%sgenre=%s\n", header,s);
      free(s);
      }

   if (MP4GetMetadataComment(h, &s))
      {
      printf("%scomment=%s\n", header, s);
      free(s);
      }

   printf("idjcmixer: mp4tagread end\n");
   fflush(stdout);
   MP4Close(h);
   return 1;
   }

int mp4_tag_write(char *pathname, char *taglist)
   {
   MP4FileHandle h;
   char *saveptr, *k, *v;
   int i, tracknum = 0, tracktotal = 0;

   if (!(h = MP4Modify(pathname, 0, 0)))
      {
      fprintf(stderr, "failed to open file %s\n", pathname);
      printf("idjcmixer: mp4fileinfo Not Valid\n");
      fflush(stdout);
      return 0;
      }

   for (i = 0; (k = strtok_r(i ? NULL : taglist, "=", &saveptr)) != '\0'; i++)
      {
      v = strtok_r(NULL, "\n", &saveptr);
      if (!strcmp(k, "artist"))
         {
         if (*v)
            MP4SetMetadataArtist(h, v);
         else
            MP4DeleteMetadataArtist(h);
         }
  
      if (!strcmp(k, "name"))
         {
         if (*v)
            MP4SetMetadataName(h, v);
         else
            MP4DeleteMetadataName(h);
         }

      if (!strcmp(k, "album"))
         {
         if (*v)
            MP4SetMetadataAlbum(h, v);
         else
            MP4DeleteMetadataAlbum(h);
         }

      if (!strcmp(k, "year"))
         {
         if (*v)
            MP4SetMetadataYear(h, v);
         else
            MP4DeleteMetadataYear(h);
         }

      if (!strcmp(k, "tracknum"))
         tracknum = atoi(v);

      if (!strcmp(k, "tracktotal"))
         tracktotal = atoi(v);

      if (!strcmp(k, "genre"))
         {
         if (*v)
            MP4SetMetadataGenre(h, v);
         else
            MP4DeleteMetadataGenre(h);
         }

      if (!strcmp(k, "comment"))
         {
         if (*v)
            MP4SetMetadataComment(h, v);
         else
            MP4DeleteMetadataComment(h);
         }
      }

   if (tracknum < 0)
      tracknum = 0;
   if (tracktotal < 0)
      tracktotal = 0;

   if (tracknum > 0 || tracktotal > 0)
      MP4SetMetadataTrack(h, tracknum, tracktotal);
   else
      MP4DeleteMetadataTrack(h);

   MP4Close(h);
   return 1;
   }

#endif

