# -*- coding: iso-8859-1 -*-
#
#-------------------------------------------------------------------------------
#
#     This file is part of the Code_Saturne User Interface, element of the
#     Code_Saturne CFD tool.
#
#     Copyright (C) 1998-2008 EDF S.A., France
#
#     contact: saturne-support@edf.fr
#
#     The Code_Saturne User Interface is free software; you can redistribute it
#     and/or modify it under the terms of the GNU General Public License
#     as published by the Free Software Foundation; either version 2 of
#     the License, or (at your option) any later version.
#
#     The Code_Saturne User Interface is distributed in the hope that it will be
#     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
#     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
#
#     You should have received a copy of the GNU General Public License
#     along with the Code_Saturne Kernel; if not, write to the
#     Free Software Foundation, Inc.,
#     51 Franklin St, Fifth Floor,
#     Boston, MA  02110-1301  USA
#
#-------------------------------------------------------------------------------

"""
This module contains the following classes:
- StandardItemModelAverage
- TimeAveragesView
"""

#-------------------------------------------------------------------------------
# Standard modules
#-------------------------------------------------------------------------------

import string
import logging

#-------------------------------------------------------------------------------
# Third-party modules
#-------------------------------------------------------------------------------

from PyQt4.QtCore import *
from PyQt4.QtGui  import *

#-------------------------------------------------------------------------------
# Application modules import
#-------------------------------------------------------------------------------

from Base.Toolbox import GuiParam
from TimeAveragesForm import Ui_TimeAveragesForm
import Base.QtPage as QtPage
from Pages.StartRestartModel import StartRestartModel
from Pages.TimeAveragesModel import TimeAveragesModel

#-------------------------------------------------------------------------------
# log config
#-------------------------------------------------------------------------------

logging.basicConfig()
log = logging.getLogger("TimeAveragesView")
log.setLevel(GuiParam.DEBUG)

#-------------------------------------------------------------------------------
# StandarItemModel class for time averages
#-------------------------------------------------------------------------------

class StandardItemModelAverage(QStandardItemModel):

    def __init__(self, parent):
        """
        """
        QStandardItemModel.__init__(self)
        self.parent = parent
        self.headers = [self.tr("Number"), self.tr("Name"), self.tr("Start"),
                        self.tr("Restart"), self.tr("Time average")]
        self.setColumnCount(len(self.headers))
        self.dataAverage = [] 

    def data(self, index, role):
        if not index.isValid():
            return QVariant()
        if role == Qt.DisplayRole:
            return QVariant(self.dataAverage[index.row()][index.column()])
        return QVariant()
    
    def flags(self, index):
        if not index.isValid():
            return Qt.ItemIsEnabled
        else:
            return Qt.ItemIsEnabled | Qt.ItemIsSelectable 
    
    def headerData(self, section, orientation, role):
        if orientation == Qt.Horizontal and role == Qt.DisplayRole:
            return QVariant(self.headers[section])
        return QVariant()

    def setData(self, index, value, role):
        self.emit(SIGNAL("dataChanged(const QModelIndex &, const QModelIndex &)"), index, index)
        return True

    def addItem(self, imom, label, ntdmom, imoold, list):
        """
        Add a row in the table. 
        """
        item = [imom, label, ntdmom, imoold, list]
        self.dataAverage.append(item)
        row = self.rowCount()
        if row +1 > 5:
            title = self.tr("Information")
            msg = self.tr("The maximal number of time averages cannot exceed 50. ")
            QMessageBox.information(self.parent, title, msg)
        else:
            self.setRowCount(row+1)

    def replaceItem(self, row, imom, label, ntdmom, imoold, list):
        """
        Replace a row in the table. 
        """
        item = [imom, label, ntdmom, imoold, list]
        self.dataAverage[row] = item

    def deleteRow(self, row):
        """
        Delete the row in the model.
        """
        del self.dataAverage[row]
        row = self.rowCount()
        self.setRowCount(row-1)

    def getItem(self, row):
        """
        Returns the name of the mesh file.
        """
        return self.dataAverage[row]

#-------------------------------------------------------------------------------
# Main class
#-------------------------------------------------------------------------------

class TimeAveragesView(QWidget, Ui_TimeAveragesForm):
    """
    """
    def __init__(self, parent, case, stbar):
        """
        Constructor
        """
        QWidget.__init__(self, parent)

        Ui_TimeAveragesForm.__init__(self)
        self.setupUi(self)
        
        self.case = case
        self.mdl = TimeAveragesModel(self.case)
        self.entriesNumber = 0

        # Create the Page layout.

        # Models
        self.modelAverage = StandardItemModelAverage(self)
        self.treeViewAverage.setModel(self.modelAverage)
        self.treeViewAverage.setAlternatingRowColors(True)
        self.treeViewAverage.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.treeViewAverage.setSelectionMode(QAbstractItemView.SingleSelection)
        self.treeViewAverage.setEditTriggers(QAbstractItemView.DoubleClicked)
        
        self.modelDrag = QStringListModel()
        self.modelDrop = QStringListModel()
        self.listViewDrag.setModel(self.modelDrag)
        self.listViewDrop.setModel(self.modelDrop)
        
        # Drag ...
        self.listViewDrag.setDragDropMode(QAbstractItemView.DragOnly)
        self.listViewDrop.setDragDropOverwriteMode(False)
        self.listViewDrag.setAlternatingRowColors(True)
        self.listViewDrag.setDragEnabled(True)
        #self.listViewDrag.setAcceptDrops(True)
        # ... and Drop
        self.listViewDrop.setDragDropMode(QAbstractItemView.DragDrop)
        self.listViewDrop.setAlternatingRowColors(True)
        self.listViewDrop.setAcceptDrops(True)
        self.listViewDrop.setDragEnabled(True)
        self.listViewDrop.setDragDropOverwriteMode(False)

        # Connections
        self.connect(self.pushButtonAdd,    SIGNAL("clicked()"), self.slotAddAverage)
        self.connect(self.pushButtonEdit,   SIGNAL("clicked()"), self.slotEditAverage)
        self.connect(self.pushButtonDelete, SIGNAL("clicked()"), self.slotDeleteAverage)
        self.connect(self.treeViewAverage,  SIGNAL("pressed(const QModelIndex &)"), self.selectAverage)

        # Validators
        validatorStart = QtPage.IntValidator(self.lineEditStart)        
        self.lineEditStart.setValidator(validatorStart)
        
        validatorRestart = QtPage.IntValidator(self.lineEditRestart, min=50)
        validatorRestart.setExclusiveMin(True)
        self.lineEditRestart.setValidator(validatorRestart)

        # Initialize

        self.average    = []
        self.label2Item = {}

        #update list of variables, properties, scalars ...
        #self.mdl.updateDicoLabelName()
        liste_label = QStringList()
        for label in self.mdl.dicoLabel2Name.keys():
            liste_label.append(label)
        
        self.modelDrag.setStringList(liste_label)
            
        #is it a following calculation ?
        if StartRestartModel(self.case).getRestart() == 'off':
            self.labelRestart.setDisabled(True)
            self.lineEditRestart.setDisabled(True)
        
        #update list of averages for view from xml file
        for nb in self.mdl.getAverageList():
            self.entriesNumber = self.entriesNumber + 1
            label, start, restart, list = self.mdl.getAverageInformations(nb)
            if restart == 0:
                restart = ''
            self.insertAverage(str(nb), label, start, restart, list)


    def getLabel(self):
        """
        Return label of average.
        """
        label = str(self.lineEditAverage.text())
        if label in self.mdl.getAverageLabelsList():
            #windowTitle = "Checking of label"
            default = {}
            default['label'] = label
            default['list'] = self.mdl.getAverageLabelsList()
            from VerifyExistenceLabelDialogView import VerifyExistenceLabelDialogView
            dialog = VerifyExistenceLabelDialogView(self, default)
            if dialog.exec_():
                result = dialog.get_result() # User clicked 'OK'
                label = result['label']
                if result['label'] == default['label']:
                    label = ""
        return label


    def getStart(self):
        """
        Return an integer for ntdmom, value of start of calculation.
        """
        start = 1
        try:
            start, ok = self.lineEditStart.text().toInt()
        except:
            title = self.tr("Warning")
            msg   = self.tr("Warning, this value must be strictly up to 0")
            QMessageBox.warning(self, title, msg)
        return start


    def getRestart(self):
        """
        Return an integer for imoold, value of restart of calculation.
        """
        restart = -999
        try:
            restart, ok = self.lineEditRestart.text().toInt()
        except:
            title = self.tr("Warning")
            msg   = self.tr("Warning, this value must be:\n" \
                            "ever between 1 and 50\n"        \
                            "ever equal to -1")
            QMessageBox.warning(self, title, msg)
        return restart


    def averageInfo(self):
        """
        Return info from the argument entry.
        """
        selectionModel = self.treeViewAverage.selectionModel()
        # How many entries in the selection ?
        for index in selectionModel.selectedIndexes():
            row = index.row()
        return self.modelAverage.getItem(row)


    def insertAverage(self, imom, label, ntdmom, imoold, list):
        """
        Insert values in Hlist.
        """
        idfmom = string.join(list,'*')
        idfmom_view = "<" + idfmom +">"

        index = 'item' + str(imom)
        self.label2Item[label] = index
        if imoold == None:
            imoold = -1
        self.modelAverage.addItem(imom, label, ntdmom, imoold, idfmom_view)

        
    @pyqtSignature("")
    def slotAddAverage(self):
        """
        Set in view IMOM, NTDMOM, IMOOLD, IDFMOM
        """
        ok = 1
        ntdmom = self.getStart()
        imoold = ''
        if StartRestartModel(self.case).getRestart() != 'off':
            imoold = self.getRestart()
            if imoold == -999:
                ok = 0
            if imoold == '':
                imoold = -1
                imoold_xml = imoold
        else:
            imoold_xml = imoold
            if imoold == '':
                imoold_xml = 0

        self.average = [str(s) for s in self.modelDrop.stringList()]
        idfmom = string.join(self.average,'*')
        
        if idfmom == '':
            title = self.tr("Warning")
            msg   = self.tr("You must select at least one variable or property from list")
            QMessageBox.information(self, title, msg)
            ok = 0

        if ok == 0: 
            return
        else: 
            label = self.getLabel()
            if label in self.mdl.getAverageLabelsList():
                msg = self.tr("This label already exists: another is choosen")
                self.stbar.showMessage(msg, 2000)
                label = ''

            self.entriesNumber = self.entriesNumber+1
            if label == '':
                label = 'Moy' + repr(self.entriesNumber)
            self.insertAverage(self.entriesNumber, label, ntdmom, imoold, self.average)
            average = string.split(idfmom,'*')
            self.mdl.setAverage(self.entriesNumber, label, ntdmom, imoold_xml, average)
        self.eraseEntries()


    @pyqtSignature("")
    def slotEditAverage(self):
        """
        Edit average to modify the selected average from the list.
        """
        self.selectAverage()
        
        try:
            [imom, label, ntdmom, imoold, idfmom] = self.averageInfo()
        except:
            title = self.tr("Warning")
            msg   = self.tr("To modify, you must select one time average")
            QMessageBox.information(self, title, msg)
            return
        self.setEditedAverage(imom, label, ntdmom, imoold, idfmom)
        self.cancelSelection()
        self.eraseEntries()


    @pyqtSignature("")
    def slotDeleteAverage(self):
        """
        Delete the selected average from the list (one by one).
        """
        self.selectAverage()
        selectionModel = self.treeViewAverage.selectionModel()
        # How many entries in the selection ?
        if len(selectionModel.selectedIndexes()) == 0:
            title = self.tr("Warning")
            msg   = self.tr("You must select at least one variable or property from list")
            QMessageBox.information(self, title, msg)
            return
            
        for index in selectionModel.selectedIndexes():
            row = index.row()
            [imom, label, ntdmom, imoold, idfmom] = self.averageInfo()
            print "imom = ", imom, type(imom)
            self.mdl.deleteAverage(int(imom)) # ???
            print "slotDeleteAverage row = %i "%row
            self.modelAverage.deleteRow(row)


    def setEditedAverage(self, imom, old_label, old_start, old_restart, old_average):
        """
        Modify the selected average from the list.
        """
        iok = 1
        new_label = self.getLabel()
        
        if new_label in self.mdl.getAverageLabelsList() and new_label != old_label: # 
            msg = self.tr("This label already exists: another is choosen")
            self.stbar.showMessage(msg, 2000)
            new_label = ''

        if new_label == '':
            new_label = old_label

        new_start = self.getStart()
        new_restart = ''
        if StartRestartModel(self.case).getRestart() != 'off':
            new_restart = self.getRestart()
            if new_restart == -999:
                ok = 0
            new_restart_xml = new_restart
            if new_restart == '':
                new_restart = -1
                new_restart_xml = -1
        else:
            if new_restart == '':
                new_restart_xml = 0
        
        if len(self.average) == 0 or self.average[0] == '': 
            title = self.tr("Warning")
            msg   = self.tr("You must select at least one variable or property from list")
            QMessageBox.information(self, title, msg)
            iok = 0

        if iok == 0:
            return
        else:
            new_average = self.average
            self.replaceAverage(imom, new_label, new_start, new_restart, new_average)
            self.mdl.replaceAverage(int(imom), new_label, new_start, new_restart_xml, new_average)


    def replaceAverage(self, imom, label, start, restart, average):
        """
        Replace values on the selected average.
        """
        log.debug("replaceAverage")
        

    def selectAverage(self):
        """
        Return the selected item from the Hlist.
        """
        selectionModel = self.treeViewAverage.selectionModel()
        for index in selectionModel.selectedIndexes():
            row = index.row()
            [imom, label, ntdmom, imoold, idfmom] = self.averageInfo()
            self.lineEditAverage.setText(QString(label))
            self.lineEditStart.setText(QString(str(ntdmom)))
            if StartRestartModel(self.case).getRestart() != 'off':
                if imoold == '':
                    imoold = -1
                self.lineEditRestart.setText(QString(str(imoold)))
            liste = [QString(s) for s in idfmom.replace('>','').replace('<','').split('*')]
            self.modelDrop.setStringList(liste)


    def eraseEntries(self):
        """
        Delete all caracters in the entries.
        """
        self.lineEditAverage.setText(str(""))
        self.lineEditStart.setText(str(""))
        if StartRestartModel(self.case).getRestart() == 'on':
            self.lineEditRestart.setText(str(""))
        self.modelDrop.setStringList(QStringList())
        self.average = []


    def insertListVarP(self, name):
        """
        Insert name of variable or property into the list (one by one).
        """
        log.debug("insertListVarP")


    def tr(self, text):
        """
        Translation
        """
        return text 
        
#-------------------------------------------------------------------------------
# Testing part
#-------------------------------------------------------------------------------

if __name__ == "__main__":
    pass

#-------------------------------------------------------------------------------
# End
#-------------------------------------------------------------------------------
