/**
 * IceMC - a menu editor for IceWM
 * Copyright (c) 2000 Georg Mittendorfer
 */

#include "IceMainWindow.h"
#include "IceMenuInCompiler.h"
#include "IceMenuSave.h"
#include "IceFileSearch.h"
#include "IcePathDialog.h"
#include "IceClipboard.h"

#include <qfiledialog.h>
#include <qfontdialog.h>
#include <qpixmap.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>

#include "fileopen.xpm"
#include "filesave.xpm"
#include "cut.xpm"
#include "copy.xpm"
#include "paste.xpm"

/*
 * The main window of the application. Handles menubar and menu operations.
 */

IceMainWindow::IceMainWindow() 
{
  setGeometry(100,100,500,400);

  /* pseudo clipboard */
  clipboard = new IceClipboard();

  /* menubar --> MainWindow */
  QPopupMenu* file = new QPopupMenu(this);
  file->insertItem("&Open",this,SLOT(open()),ALT+Key_O);
  file->insertItem("&Save",this,SLOT(save()),ALT+Key_S);
  file->insertItem("Save &As",this,SLOT(save_as()),ALT+Key_A);
  file->insertSeparator();
  file->insertItem("&Close",this,SLOT(close()));
  file->insertItem("E&xit",this,SLOT(exit()));

  edit = new QPopupMenu(this);
  edit->insertItem("Cut",this,SLOT(cutItem()),CTRL+Key_X);
  edit->insertItem("Copy",this,SLOT(copyItem()),CTRL+Key_C);
  edit->insertItem("Paste",this,SLOT(pasteItem()),CTRL+Key_V);
  edit->insertItem("&Delete",this,SLOT(deleteItem()),Key_Delete);
  edit->insertSeparator();
  edit->insertItem("Insert &Application",this,SLOT(insertApp()));
  edit->insertItem("Insert &Folder",this,SLOT(insertFolder()));
  edit->insertItem("Insert &Separator",this,SLOT(insertSeparator()));
  edit->insertItem("Insert &Restart",this,SLOT(insertRestart()));
  edit->insertSeparator();
  edit->insertItem("&Unselect Item",this,SLOT(unselect()));

  view = new QPopupMenu(this);
  view->insertItem("&Toolbar",this,SLOT(tool()),0,1);
  view->insertItem("&Statusbar",this,SLOT(statusbar()),0,2);
  view->insertSeparator();
  view->insertItem("s&ortable",this,SLOT(sort()),0,3);
  view->insertSeparator();
  view->insertItem("&Refresh",this,SLOT(refresh()),0,4);
  view->setCheckable(true);
  view->setItemChecked(1,true);
  view->setItemChecked(2,true);

  QPopupMenu* options = new QPopupMenu(this);
  options->insertItem("&Paths",this,SLOT(setCustomPaths()));
  options->insertItem("&Font",this,SLOT(setCustomFont()));

  // TODO: Help menu  
//    QPopupMenu* help = new QPopupMenu(this);
//    help->insertItem("&Contents",this,SLOT(contents()));
//    help->insertSeparator();
//    help->insertItem("&About",this,SLOT(about()));
  
  QMenuBar* menubar = new QMenuBar(this); 
  menubar->insertItem("&File", file);
  menubar->insertItem("E&dit", edit);
  menubar->insertItem("&View", view);
  menubar->insertItem("O&ptions", options);
  //  menubar->insertItem("&Help", help);

  /* toolbar & icons */
  QPixmap openicon, saveicon;
  QPixmap cuticon, copyicon, pasteicon;
  openicon = QPixmap(fileopen);
  saveicon = QPixmap(filesave);
  cuticon = QPixmap(cut_xpm);
  copyicon = QPixmap(copy_xpm);
  pasteicon = QPixmap(paste_xpm);

  toolbar = new QToolBar(this);
  QToolButton* b;
  b = new QToolButton(openicon,"Open File","Open File",this,SLOT(open()),toolbar);
  b->setAccel(ALT+Key_O);
  b = new QToolButton(saveicon,"Save File","Save File",this,SLOT(save()),toolbar);
  b->setAccel(ALT+Key_S);
  toolbar->addSeparator();
  b = new QToolButton(cuticon,"Cut","Cut",this,SLOT(cutItem()),toolbar);
  b->setAccel(CTRL+Key_X);
  b = new QToolButton(copyicon,"Copy","Copy",this,SLOT(copyItem()),toolbar);
  b->setAccel(CTRL+Key_C);
  b = new QToolButton(pasteicon,"Paste","Paste",this,SLOT(pasteItem()),toolbar);
  b->setAccel(CTRL+Key_V);

  this->statusBar();
  
  /* central widget */
  mainwidget = new IceMainWidget(this);
  setCentralWidget(mainwidget);

  connect(mainwidget->mainview, SIGNAL(rightButtonPressed(QListViewItem*,const QPoint &,int)),
				       this,SLOT(popupEdit(QListViewItem*,const QPoint &,int)));
	currentFile = "";
	this->setCaption(APP_NAME);
}

IceMainWindow::~IceMainWindow()
{
  // Qt does the job (i hope ;)
}

/////////////////////////////////////////////////////
// public slots
/////////////////////////////////////////////////////

void IceMainWindow::popupEdit(QListViewItem* item,const QPoint& pos,int coloumn)
{
  edit->popup(pos);
}

void IceMainWindow::open()
{
  ;//qDebug("DEBUG: IceMainWindow::open() ");
  QString file;
  int ok;
  IceFileSearch* search = new IceFileSearch();
  static QString openPath = search->dirReadable(0,mainwidget->getLibPath()); // startdir for filedialog
  delete search;
  QFileDialog* dialog = new QFileDialog(openPath,QString::null,this,"openFile",true);
  dialog->setShowHiddenFiles(true);
  dialog->setMode(QFileDialog::ExistingFile);
  dialog->setCaption("Please Select Menu Configuration File");
  ok = dialog->exec();

  file = dialog->selectedFile();
  if ( (!file.isEmpty()) && (ok) ) {
    // the user selected/entered a filename
    openPath = dialog->dirPath(); // for next use

    QFileInfo* myFile = new QFileInfo(); // check if exists/readable
    myFile->setCaching(false);
    myFile->setFile(file);
    if (myFile->isReadable()) {
      // TODO:
      // gibt es eine offene liste?
      // wenn ja: ist diese liste schon gespeichert? -> close()
      close();
      IceMenuInParser* parser = new IceMenuInParser();
      parser->parseFile(mainwidget->mainview, file);
      if (!parser->successful()) {
	// qDebug("DEBUG: IceMainWindow.open: Syntaxerror");
	int lnr, cnr;
	parser->errorPos(lnr,cnr);
	if (view->isItemChecked(2))
	  this->statusBar()->message("ERROR: Syntax error in specified file", 5000);
	QString msg = "Not a valid menu configuration file:\n"
	  "Syntax error around line " + QString::number(lnr) + " character " + QString::number(cnr) + ".";
	errorWarning("... to err is human ...", msg);
	file = "";
      }
      delete parser;
      mainwidget->mainview->updateList();
      mainwidget->mainview->show();
			currentFile = file;
			this->setCaption(APP_NAME + currentFile);
    }
    else {
      if (view->isItemChecked(2))
				this->statusBar()->message("ERROR: Can't open specified file", 5000);
      errorWarning("File Not Found", "Can't open specified file");
    }
    delete myFile;
  }
  else {
    // canceled
  }
  delete dialog;
}

void IceMainWindow::save()
{
  ;//qDebug("DEBUG: IceMainWindow::save() ");
  if (currentFile.isEmpty()) {
    save_as();
  } else {
    QFileInfo* info = new QFileInfo(currentFile);
    QString dir;
  
    dir = info->dirPath();
    if (((info->exists()) && (!info->isWritable())) || (!QFileInfo(dir).isWritable())) {
      qWarning("WARNING: can't write file!"); // can't write
      if (view->isItemChecked(2))
	this->statusBar()->message("couldn't write file: " + currentFile, 5000);
    } else {
      ;//qDebug("DEBUG: try to write file");
      // write file
      IceMenuSave* parser = new IceMenuSave();
      parser->write(mainwidget->mainview,currentFile);
      if (view->isItemChecked(2))
	this->statusBar()->message("wrote file: " + currentFile, 5000);
      ;//qDebug("DEBUG: wrote file: " + currentFile);
      delete parser;
    }
    delete info;
  }
} // TODO: bool einbauen ob file veraendert und nach save zuruecksetzen

void IceMainWindow::save_as()
{
  ;//qDebug("DEBUG: IceMainWindow::save_as() ");
  QString file;
	QFileInfo* info = new QFileInfo(currentFile);
	QString dir = info->dirPath();

	file = QFileDialog::getSaveFileName(dir, QString::null, this);
	if ( !file.isEmpty() ) {
		// the user gave a file name
		info->setFile(file);
		dir = info->dirPath();
		if (((info->exists()) && (!info->isWritable())) || (!QFileInfo(dir).isWritable())) {
			qWarning("WARNING: can't write file!"); // can't write
			if (view->isItemChecked(2))
				this->statusBar()->message("couldn't write file: " + file, 5000);
		} else {
			;//qDebug("DEBUG: try to write file");
			// write file
			IceMenuSave* parser = new IceMenuSave();
			parser->write(mainwidget->mainview,file);
      if (view->isItemChecked(2))
				this->statusBar()->message("wrote file: " + file, 5000);
			;//qDebug("DEBUG: wrote file: " + file);
			currentFile = info->filePath();
			delete parser;
			this->setCaption(APP_NAME + currentFile);
		}
	} else {
		// the user cancelled the dialog
	}
	delete info;
}

void IceMainWindow::close() 
{
  ;//qDebug("DEBUG: IceMainWindow::close() ");
  mainwidget->mainview->clear();
  mainwidget->clearLineEdits();
	currentFile = "";
	this->setCaption(APP_NAME);
}

void IceMainWindow::exit()
{
  ;//qDebug("DEBUG: IceMainWindow::exit() ");
  close();
  qApp->quit();
}

void IceMainWindow::cutItem()
{
  ;//qDebug("DEBUG: IceMainWindow::cutItem() ");


  if (IceListViewItem* item = mainwidget->mainview->selectedItem()) {
    IceListViewItem* parent;
    IceListViewItem* next;

    parent = item->parent();
    if (!(next = item->nextSibling()))
      next = item->itemAbove();

    clipboard->insertItem(item);

    if (!parent)
      mainwidget->mainview->takeItem(item);
    else {
      parent->takeItem(item);
      ;//qDebug("DEBUG: cut() took item");
      mainwidget->mainview->updateRootFolder(parent);
    }

    if (next) {
      mainwidget->mainview->setSelected(next,true);
      // ??   mainwidget->mainview->setCurrentItem(next);
    }

    delete item;
  }
}

void IceMainWindow::copyItem()
{
  ;//qDebug("DEBUG: IceMainWindow::copyItem() ");

  if (IceListViewItem* item = mainwidget->mainview->selectedItem()) {
    clipboard->insertItem(item);
  }
}

void IceMainWindow::pasteItem()
{
  ;//qDebug("DEBUG: IceMainWindow::pasteItem() ");
  QApplication::setOverrideCursor(Qt::waitCursor);

  IceListViewItem* item;

  item = mainwidget->mainview->createItem(mainwidget->mainview->selectedItem());

  clipboard->getItem(item);
  
  if (item->getInfo() == "menu") {
      if (item->parent())
	mainwidget->mainview->updateRootFolder(item);
       else
	 mainwidget->mainview->updateFolder(item);
  }
  else
    mainwidget->updateListItem(item);
  QApplication::restoreOverrideCursor();
}

void IceMainWindow::deleteItem() 
{
  ;//qDebug("DEBUG: IceMainWindow::deleteItem() ");
  if (mainwidget->mainview->currentItem()) {
    IceListViewItem* item;
    IceListViewItem* parent;
    item = mainwidget->mainview->currentItem();
    parent = item->parent();
    if (!parent)
      mainwidget->mainview->takeItem(item);
    else {
      bool exec = (item->getColor() == IceListView::executable);
      parent->takeItem(item);
      if (exec)
	mainwidget->mainview->updateRootFolder(parent);
    }
    delete item;
  }
}

void IceMainWindow::insertApp() 
{
  ;//qDebug("DEBUG: IceMainWindow::insertApp() ");
  IceListViewItem* item = mainwidget->mainview->createItem(mainwidget->mainview->selectedItem(),"prog");
  item->setText(0,"New Application");
  mainwidget->updateListItem(item);
}

void IceMainWindow::insertRestart()
{
  ;//qDebug("DEBUG: IceMainWindow::insertRestart() ");
  IceListViewItem* item = mainwidget->mainview->createItem(mainwidget->mainview->selectedItem(),"restart");
  item->setText(0,"New Application (restart)");
  mainwidget->updateListItem(item);
}

void IceMainWindow::insertFolder()
{
  ;//qDebug("DEBUG: IceMainWindow::insertFolder() ");
  IceListViewItem* item = mainwidget->mainview->createItem(mainwidget->mainview->selectedItem(),"menu");
  item->setText(0,"New Folder");
  item->setText(1,"folder");
  mainwidget->updateListItem(item);
}

void IceMainWindow::insertSeparator()
{
  ;//qDebug("DEBUG: IceMainWindow::insertSeparator() ");
  IceListViewItem* item = mainwidget->mainview->createItem(mainwidget->mainview->selectedItem(),"separator");
  item->setText(0," ------ ");
}

void IceMainWindow::unselect()
{
  ;//qDebug("DEBUG: IceMainWindow::unselect() ");
  mainwidget->mainview->clearSelection();
}

void IceMainWindow::tool()
{
  ;//qDebug("DEBUG: IceMainWindow::tool() ");

  if (!view->isItemChecked(1)) {
    view->setItemChecked(1,true);
    addToolBar(toolbar);
  }
  else {
    view->setItemChecked(1,false);
    removeToolBar(toolbar);
  }
}

void IceMainWindow::statusbar()
{
  ;//qDebug("DEBUG: IceMainWindow::statusbar() ");
 
  // attention: whenever statusbar() is called somewhere else a new 
  //            statusbar will be created if there doesn't exist one
  //            therefore: keep consistent with checked popupmenu!
  if (!view->isItemChecked(2)) {
    view->setItemChecked(2,true);
    this->statusBar();
  }
  else {
    view->setItemChecked(2,false);
    delete this->statusBar();
  }
}

void IceMainWindow::sort()
{
  // TODO: implement usefull sort criterias ;-)
  ;//qDebug("DEBUG: IceMainWindow::sort() ");
  if (!view->isItemChecked(3)) {
    view->setItemChecked(3,true);
    mainwidget->mainview->setShowSortIndicator(false);
    mainwidget->mainview->setSorting(666,true); // out of range!! not sorting!!
  }
  else {
    view->setItemChecked(3,false);
    mainwidget->mainview->setSorting(-1);
  }
}

void IceMainWindow::refresh()
{
  ;//qDebug("DEBUG: IceMainWindow::refresh() ");
  mainwidget->mainview->updateList();
  mainwidget->mainview->triggerUpdate();
}

void IceMainWindow::setCustomPaths()
{
  ;//qDebug("DEBUG: IceMainWindow::setCustomPaths() ");
  IcePathDialog* pref = new IcePathDialog(this,0,true);

  int button = pref->exec();

  if (button == IcePathDialog::Accepted) {
    mainwidget->setLibPath(pref->getCustomLibPath());
    mainwidget->setIconPath(pref->getCustomIconPath());
  }
  else if (button == IcePathDialog::Rejected) {
    // do nothing
  }
  else
    qWarning("WARNING: IceMainWindow::setCustomPaths -> button: wrong return value");

  delete pref;
}

void IceMainWindow::setCustomFont()
{
  ;//qDebug("DEBUG: IceMainWindow::setCustomFont() ");
  bool ok;
  QFont f = QFontDialog::getFont( &ok, this->font(), this );
  if ( ok ) {
    QApplication::setFont(f,true);
  } else {
    // the user cancelled the dialog
  }
}

void IceMainWindow::contents()
{
  ;//qDebug("DEBUG: IceMainWindow::contents() ");
}

void IceMainWindow::about()
{
  ;//qDebug("DEBUG: IceMainWindow::about() ");
} 

//////////// private methods ////////////////////////////////////////////////////////////
  
void IceMainWindow::errorWarning(QString caption, QString msg)
{
  //QMessageBox warnBox(caption, msg, QMessageBox::Warning, QMessageBox::Ok | QMessageBox::Default, 0, 0);
  QMessageBox::warning( this, caption, msg, "Ok");
  //warnBox.show(); 
}
