/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: Mixer.cpp,v 1.96 2004/09/06 09:41:48 comix Exp $
 *
 */

#include "qinputdialog.h"

#include "lib/Hydrogen.h"
#include "lib/Song.h"

#include "Mixer.h"
#include "Button.h"
#include "MixerLine.h"
#include "HydrogenApp.h"
#include "LadspaFXProperties.h"
#include "PatternEditorPanel.h"

#define MIXER_STRIP_WIDTH	44

Mixer::Mixer( QWidget* parent )
 : QWidget( parent, "Mixer" )
 , Object( "Mixer" )
{

	unsigned nMinimumVisibleFadersSize;

	m_nMixerHeight = 270;
	nMinimumVisibleFadersSize = 5 + MIXER_STRIP_WIDTH * 16 + 5;
	uint fadersSize = 5 + MIXER_STRIP_WIDTH * MAX_INSTRUMENTS;

	m_nMixerWidth = 5 + MIXER_STRIP_WIDTH * 4 + 70;	// 4 instruments visible
	setMinimumSize( m_nMixerWidth, m_nMixerHeight );

	setMaximumSize( fadersSize +  70, m_nMixerHeight );
	m_nMixerWidth = 5 + MIXER_STRIP_WIDTH * 16 + 70;	// 16 instruments visible
	resize( m_nMixerWidth, m_nMixerHeight );
	setCaption( trUtf8( "Mixer" ) );
	setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );

	// Background image
	string background_path = string( IMG_PATH ) + string( "/img/mixerPanel/mixer_background.png" );
	bool ok = m_backgroundPixmap.load( background_path.c_str() );
	if( ok == false ){
		errorLog( "Error loading pixmap " + background_path );
	}


	// fader scroll view
	m_pFaderScrollView = new QScrollView( this );
	m_pFaderScrollView->move( 0, 0 );
	m_pFaderScrollView->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pFaderScrollView->setHScrollBarMode( QScrollView::AlwaysOn );
	m_pFaderScrollView->resize( nMinimumVisibleFadersSize, m_nMixerHeight );

	// fader frame
	m_pFaderFrame = new QFrame( m_pFaderScrollView->viewport() );
	m_pFaderFrame->resize( fadersSize, m_nMixerHeight );
	m_pFaderScrollView->addChild( m_pFaderFrame );
	m_pFaderFrame->setBackgroundPixmap( m_backgroundPixmap );


	// fx scroll view
	m_pFXScrollView = new QScrollView( this );
	m_pFXScrollView->move( 0, 0 );
	m_pFXScrollView->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pFXScrollView->setHScrollBarMode( QScrollView::AlwaysOn );
	m_pFXScrollView->resize( nMinimumVisibleFadersSize, m_nMixerHeight );

	// fX frame
	m_pFXFrame = new QFrame( m_pFXScrollView->viewport() );
	m_pFXFrame->resize( fadersSize, m_nMixerHeight );
	m_pFXScrollView->addChild( m_pFXFrame );
	m_pFXFrame->setBackgroundPixmap( m_backgroundPixmap );



	this->setBackgroundPixmap( m_backgroundPixmap );


	m_pFaderScrollView->show();
	m_pFXScrollView->hide();

	setupMixer();

	QTimer *pTimer = new QTimer( this );
	connect( pTimer, SIGNAL( timeout() ), this, SLOT( updateMixer() ) );
	pTimer->start(50);

	( Hydrogen::getInstance() )->addEngineListener( this );
}



Mixer::~Mixer()
{
}



/// Setup the mixer strips
void Mixer::setupMixer()
{
	uint nMaster_X = m_nMixerWidth - MIXER_STRIP_WIDTH - 5;

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	uint xPos = 0;
	uint nInstruments = instrList->getSize();
	for ( uint i = 0; i < MAX_INSTRUMENTS; i++ ) {	// MIXER LINE
		xPos = 5 + MIXER_STRIP_WIDTH * i;

		float volume = 0.2;
		bool mute = false;
		bool solo = false;

		Instrument *instr = NULL;
		if (i < nInstruments) {
			instr = instrList->get( i );
			volume = instr->getVolume();
			mute = instr->isMuted();
		}

		m_pMixerLine[i] = new MixerLine( m_pFaderFrame );
		m_pMixerLine[i]->move( xPos, 4 );
		m_pMixerLine[i]->setVolume( volume );
		m_pMixerLine[i]->setMuteClicked( mute );
		m_pMixerLine[i]->setSoloClicked( solo );
		m_pMixerLine[i]->updateMixerLine();

		connect( m_pMixerLine[i], SIGNAL( noteOnClicked(MixerLine*) ), this, SLOT( noteOnClicked(MixerLine*) ) );
		connect( m_pMixerLine[i], SIGNAL( noteOffClicked(MixerLine*) ), this, SLOT( noteOffClicked(MixerLine*) ) );
		connect( m_pMixerLine[i], SIGNAL( muteBtnClicked(MixerLine*) ), this, SLOT( muteClicked(MixerLine*) ) );
		connect( m_pMixerLine[i], SIGNAL( soloBtnClicked(MixerLine*) ), this, SLOT( soloClicked(MixerLine*) ) );
		connect( m_pMixerLine[i], SIGNAL( volumeChanged(MixerLine*) ), this, SLOT( volumeChanged(MixerLine*) ) );
		connect( m_pMixerLine[i], SIGNAL( instrumentNameClicked(MixerLine*) ), this, SLOT( nameClicked(MixerLine*) ) );
		connect( m_pMixerLine[i], SIGNAL( instrumentNameSelected(MixerLine*) ), this, SLOT( nameSelected(MixerLine*) ) );
		connect( m_pMixerLine[i], SIGNAL( panChanged(MixerLine*) ), this, SLOT( panChanged( MixerLine*) ) );
		connect( m_pMixerLine[i], SIGNAL( knobChanged(MixerLine*, int) ), this, SLOT( knobChanged( MixerLine*, int) ) );
	}

	m_pSwingFactorLine = new FxMixerLine( m_pFXFrame );
	m_pSwingFactorLine->move( 5, 4 );
	m_pSwingFactorLine->setName( trUtf8( "Swing factor" ) );
	connect( m_pSwingFactorLine, SIGNAL( volumeChanged(FxMixerLine*) ), this, SLOT( swingChanged(FxMixerLine*) ) );
	connect( m_pSwingFactorLine, SIGNAL( activeBtnClicked(FxMixerLine*) ), this, SLOT( activeBtnChanged( FxMixerLine*) ) );

	m_pHumanizeTimeLine = new FxMixerLine( m_pFXFrame );
	m_pHumanizeTimeLine->move( 5 + MIXER_STRIP_WIDTH, 4 );
	m_pHumanizeTimeLine->setName( trUtf8( "Human time" ) );
	connect( m_pHumanizeTimeLine, SIGNAL( volumeChanged(FxMixerLine*) ), this, SLOT( humanizeChanged(FxMixerLine*) ) );
	connect( m_pHumanizeTimeLine, SIGNAL( activeBtnClicked(FxMixerLine*) ), this, SLOT( activeBtnChanged( FxMixerLine*) ) );

	m_pHumanizeVelocityLine = new FxMixerLine( m_pFXFrame );
	m_pHumanizeVelocityLine->move( 5 + MIXER_STRIP_WIDTH * 2, 4 );
	m_pHumanizeVelocityLine->setName( trUtf8( "Human velocity" ) );
	connect( m_pHumanizeVelocityLine, SIGNAL( volumeChanged(FxMixerLine*) ), this, SLOT( humanizeChanged(FxMixerLine*) ) );
	connect( m_pHumanizeVelocityLine, SIGNAL( activeBtnClicked(FxMixerLine*) ), this, SLOT( activeBtnChanged( FxMixerLine*) ) );

	// LADSPA
	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		m_pLadspaFXLine[nFX] = new LadspaFXMixerLine( m_pFXFrame );
		m_pLadspaFXLine[nFX]->move( 5 + MIXER_STRIP_WIDTH * (4 + nFX), 4 );
		connect( m_pLadspaFXLine[nFX], SIGNAL( activeBtnClicked(LadspaFXMixerLine*) ), this, SLOT( ladspaActiveBtnClicked( LadspaFXMixerLine*) ) );
		connect( m_pLadspaFXLine[nFX], SIGNAL( editBtnClicked(LadspaFXMixerLine*) ), this, SLOT( ladspaEditBtnClicked( LadspaFXMixerLine*) ) );
		connect( m_pLadspaFXLine[nFX], SIGNAL( volumeChanged(LadspaFXMixerLine*) ), this, SLOT( ladspaVolumeChanged( LadspaFXMixerLine*) ) );
		connect( m_pLadspaFXLine[nFX], SIGNAL( ladspaNameClicked(LadspaFXMixerLine*) ), this, SLOT( ladspaEditBtnClicked( LadspaFXMixerLine*) ) );
	}
	// ~LADSPA


	m_pMasterLine = new MasterMixerLine( this );
	m_pMasterLine->move( nMaster_X, 4 );
	connect( m_pMasterLine, SIGNAL( volumeChanged(MasterMixerLine*) ), this, SLOT( masterVolumeChanged(MasterMixerLine*) ) );

	string showMixer_on_path = string(IMG_PATH).append("/img/mixerPanel/showMixer_on.png");
	string showMixer_off_path = string(IMG_PATH).append("/img/mixerPanel/showMixer_off.png");
	string showMixer_over_path = string(IMG_PATH).append("/img/mixerPanel/showMixer_over.png");
	m_pShowFaderPanelBtn = new ToggleButton( this, QSize(40, 12), showMixer_on_path, showMixer_off_path, showMixer_over_path );
	m_pShowFaderPanelBtn->move( nMaster_X, 20 );
	m_pShowFaderPanelBtn->setPressed(true);
	QToolTip::add( m_pShowFaderPanelBtn, trUtf8( "Show faders panel" ) );
	connect( m_pShowFaderPanelBtn, SIGNAL(clicked(Button*)), this, SLOT(showFadersPanelClicked(Button*)));

	string showFX_on_path = string(IMG_PATH).append("/img/mixerPanel/showFX_on.png");
	string showFX_off_path = string(IMG_PATH).append("/img/mixerPanel/showFX_off.png");
	string showFX_over_path = string(IMG_PATH).append("/img/mixerPanel/showFX_over.png");
	m_pShowFXPanelBtn = new ToggleButton( this, QSize(40, 12), showFX_on_path, showFX_off_path, showFX_over_path );
	m_pShowFXPanelBtn->move( nMaster_X, 40 );
	m_pShowFXPanelBtn->setPressed(false);
	QToolTip::add( m_pShowFXPanelBtn, trUtf8( "Show FX panel" ) );
	connect( m_pShowFXPanelBtn, SIGNAL(clicked(Button*)), this, SLOT( showFXPanelClicked(Button*)));

	string showPeaks_on_path = string(IMG_PATH).append("/img/mixerPanel/showPeaks_on.png");
	string showPeaks_off_path = string(IMG_PATH).append("/img/mixerPanel/showPeaks_off.png");
	string showPeaks_over_path = string(IMG_PATH).append("/img/mixerPanel/showPeaks_over.png");
	m_pShowPeaksBtn = new ToggleButton( this, QSize(40, 12), showPeaks_on_path, showPeaks_off_path, showPeaks_over_path );
	m_pShowPeaksBtn->move( nMaster_X, 60 );
	m_pShowPeaksBtn->setPressed( (Preferences::getInstance())->showInstrumentPeaks() );
	QToolTip::add( m_pShowPeaksBtn, trUtf8( "Show instrument peaks" ) );
	connect( m_pShowPeaksBtn, SIGNAL(clicked(Button*)), this, SLOT( showPeaksBtnClicked(Button*)));

	m_pShowFaderPanelBtn->setPressed(true);
}



void Mixer::muteClicked(MixerLine* ref) {
	int nLine = findMixerLineByRef(ref);
	Hydrogen::getInstance()->setSelectedInstrumentNumber( nLine );
	bool isMuteClicked = ref->isMuteClicked();

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Instrument *instr = instrList->get(nLine);
	instr->setMuted(isMuteClicked);
	//(HydrogenApp::getInstance())->setSelectedInstrument(nLine);
	Hydrogen::getInstance()->setSelectedInstrumentNumber(nLine);
}



void Mixer::soloClicked(MixerLine* ref) {
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();


	int nLine = findMixerLineByRef(ref);
	Hydrogen::getInstance()->setSelectedInstrumentNumber( nLine );
	bool isSoloClicked = ref->isSoloClicked();

	if (isSoloClicked) {
		for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
			m_pMixerLine[i]->setSoloClicked( false );
			m_pMixerLine[i]->setMuteClicked( true );

			Instrument *instr = instrList->get( i );
			instr->setMuted( true );
		}
		m_pMixerLine[nLine]->setSoloClicked( true );
		m_pMixerLine[nLine]->setMuteClicked( false );
		Instrument *instr = instrList->get( nLine );
		instr->setMuted( false );
	}
	else {
		for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
			m_pMixerLine[i]->setMuteClicked( false );
			m_pMixerLine[i]->setSoloClicked( false );

			Instrument *instr = instrList->get( i );
			instr->setMuted( false );
		}
	}

	Hydrogen::getInstance()->setSelectedInstrumentNumber(nLine);
}



/// used in PatternEditorInstrumentList
void Mixer::soloClicked(uint nLine)
{
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();


	bool isSoloClicked = (m_pMixerLine[nLine])->isSoloClicked();

	if (!isSoloClicked) {
		for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
			m_pMixerLine[i]->setSoloClicked( false );
			m_pMixerLine[i]->setMuteClicked( true );

			Instrument *instr = instrList->get( i );
			instr->setMuted( true );
		}
		m_pMixerLine[nLine]->setSoloClicked( true );
		m_pMixerLine[nLine]->setMuteClicked( false );
		Instrument *instr = instrList->get( nLine );
		instr->setMuted( false );
	}
	else {
		for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
			m_pMixerLine[i]->setMuteClicked( false );
			m_pMixerLine[i]->setSoloClicked( false );

			Instrument *instr = instrList->get( i );
			instr->setMuted( false );
		}
	}

}



void Mixer::noteOnClicked( MixerLine* ref )
{
	int nLine = findMixerLineByRef( ref );
	Hydrogen::getInstance()->setSelectedInstrumentNumber( nLine );

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Note *note = new Note( 0, 1.0, 1.0f, 1.0f, -1 );
	note->setInstrument( instrList->get(nLine) );

	engine->noteOn( note );

	Hydrogen::getInstance()->setSelectedInstrumentNumber(nLine);
}



/// Play sample button, right-clicked (note off)
 void Mixer::noteOffClicked( MixerLine* ref )
{
	int nLine = findMixerLineByRef( ref );
	Hydrogen::getInstance()->setSelectedInstrumentNumber( nLine );

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Note *note = new Note( 0, 1.0, 1.0, 1.0, -1 );
	note->setInstrument( instrList->get( nLine ) );

	engine->noteOff( note );

	Hydrogen::getInstance()->setSelectedInstrumentNumber(nLine);
}



uint Mixer::findMixerLineByRef(MixerLine* ref) {
	for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
		if (m_pMixerLine[i] == ref) {
			return i;
		}
	}
	return 0;
}



void Mixer::volumeChanged(MixerLine* ref) {

	int nLine = findMixerLineByRef(ref);
	Hydrogen::getInstance()->setSelectedInstrumentNumber( nLine );

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Instrument *instr = instrList->get(nLine);

	instr->setVolume(ref->getVolume());

	Hydrogen::getInstance()->setSelectedInstrumentNumber(nLine);
}




void Mixer::masterVolumeChanged(MasterMixerLine* ref) {
	float volume = ref->getVolume();
	Song *song = (HydrogenApp::getInstance())->getSong();
	song->setVolume(volume);
}



void Mixer::updateMixer() {
//	if(!isVisible()) {
//		return;
//	}

	Preferences *pPref = Preferences::getInstance();
	bool bShowPeaks = pPref->showInstrumentPeaks();

	Hydrogen *engine = Hydrogen::getInstance();
	Song *pSong = engine->getSong();
	InstrumentList *instrList = pSong->getInstrumentList();

	//float fallOff = 1.08;	// slow
	//float fallOff = 1.1;	// normal
	//float fallOff = 1.3;	// fast

	uint nSelectedInstr = Hydrogen::getInstance()->getSelectedInstrumentNumber();

	float fallOff = (Preferences::getInstance())->getMixerFalloffSpeed();

	uint nMuteClicked = 0;
	uint nInstruments = instrList->getSize();
	for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
		Instrument *instr = instrList->get(i);

		// fader
		float oldPeak_L = m_pMixerLine[i]->getPeak_L();
		float newPeak_L = instr->getPeak_L();
		instr->setPeak_L( 0.0f );	// reset instrument peak

		float oldPeak_R = m_pMixerLine[i]->getPeak_R();
		float newPeak_R = instr->getPeak_R();
		instr->setPeak_R( 0.0f );	// reset instrument peak

		if (!bShowPeaks) {
			newPeak_L = 0.0f;
			newPeak_R = 0.0f;
		}

		if ( newPeak_L >= oldPeak_L) {	// LEFT peak
			m_pMixerLine[i]->setPeak_L( newPeak_L );
		}
		else {
			m_pMixerLine[i]->setPeak_L( oldPeak_L / fallOff );
		}
		if ( newPeak_R >= oldPeak_R) {	// Right peak
			m_pMixerLine[i]->setPeak_R( newPeak_R );
		}
		else {
			m_pMixerLine[i]->setPeak_R( oldPeak_R / fallOff );
		}

		// fader position
		float newVolume = instr->getVolume();
		m_pMixerLine[i]->setVolume(newVolume);

		// mute
		bool muted = instr->isMuted();
		if (muted) {
			nMuteClicked++;
		}
		m_pMixerLine[i]->setMuteClicked( muted );

		// instr name
		m_pMixerLine[i]->setName( instr->getName().c_str() );

		// pan
		float panValue = 0.0;
		float pan_L = instr->getPan_L();
		float pan_R = instr->getPan_R();
		if (pan_R == 1.0) {
			panValue = 1.0 - (pan_L / 2.0);
		}
		else {
			panValue = pan_R / 2.0;
		}
		panValue = panValue * 100.0;

		m_pMixerLine[i]->setPan((int)panValue);	/// \todo perche' setPan prende un'intero???

		// activity
		if ( m_pMixerLine[i]->getActivity() > 0 ) {
			m_pMixerLine[i]->setActivity( m_pMixerLine[i]->getActivity() - 30 );
			m_pMixerLine[i]->setPlayClicked( true );
		}
		else {
			m_pMixerLine[i]->setPlayClicked( false );
		}

		for (uint nFX = 0; nFX < MAX_FX; nFX++) {
			m_pMixerLine[i]->setFXLevel( nFX, instr->getFXLevel( nFX ) );
		}

		if ( i == nSelectedInstr) {
			m_pMixerLine[i]->setSelected( true );
		}
		else {
			m_pMixerLine[i]->setSelected( false );
		}

		m_pMixerLine[i]->updateMixerLine();
	}

	if (nMuteClicked == MAX_INSTRUMENTS - 1) {
		// find the not muted button
		for (uint i = 0; i < nInstruments; i++) {
			Instrument *instr = instrList->get(i);
			if (instr->isMuted() == false) {
				m_pMixerLine[i]->setSoloClicked(true);
				break;
			}
		}
	}
	else {
		for (uint i = 0; i < nInstruments; i++) {
			m_pMixerLine[i]->setSoloClicked(false);
		}
	}


	// update MasterPeak
	float oldPeak_L = m_pMasterLine->getPeak_L();
	float newPeak_L = engine->getMasterPeak_L();
	engine->setMasterPeak_L(0.0);
	float oldPeak_R = m_pMasterLine->getPeak_R();
	float newPeak_R = engine->getMasterPeak_R();
	engine->setMasterPeak_R(0.0);

	if (!bShowPeaks) {
		newPeak_L = 0.0;
		newPeak_R = 0.0;
	}

	if (newPeak_L >= oldPeak_L) {
		m_pMasterLine->setPeak_L( newPeak_L );
	}
	else {
		m_pMasterLine->setPeak_L( oldPeak_L / fallOff );
	}
	if (newPeak_R >= oldPeak_R) {
		m_pMasterLine->setPeak_R(newPeak_R);
	}
	else {
		m_pMasterLine->setPeak_R( oldPeak_R / fallOff );
	}




	// set master fader position
	float newVolume = pSong->getVolume();
	float oldVolume = m_pMasterLine->getVolume();
	if (oldVolume != newVolume) {
		m_pMasterLine->setVolume(newVolume);
	}
	m_pMasterLine->updateMixerLine();


	// set humanize fx
	m_pHumanizeTimeLine->setFxActive( pSong->isHumanizeTimeEnabled() );
	if (m_pHumanizeTimeLine->getVolume() != pSong->getHumanizeTimeValue() ) {
		m_pHumanizeTimeLine->setVolume( pSong->getHumanizeTimeValue() );
	}
	m_pHumanizeTimeLine->setPeak_L( pSong->getHumanizeTimeValue() );
	m_pHumanizeTimeLine->setPeak_R( pSong->getHumanizeTimeValue() );
	m_pHumanizeTimeLine->updateMixerLine();


	m_pHumanizeVelocityLine->setFxActive( pSong->isHumanizeVelocityEnabled() );
	if ( m_pHumanizeVelocityLine->getVolume() != pSong->getHumanizeVelocityValue() ) {
		m_pHumanizeVelocityLine->setVolume( pSong->getHumanizeVelocityValue() );
	}
	m_pHumanizeVelocityLine->setPeak_L( pSong->getHumanizeVelocityValue() );
	m_pHumanizeVelocityLine->setPeak_R( pSong->getHumanizeVelocityValue() );
	m_pHumanizeVelocityLine->updateMixerLine();

	// swing factor
	m_pSwingFactorLine->setFxActive( pSong->isSwingEnabled() );
	if ( m_pSwingFactorLine->getVolume() != pSong->getSwingFactor() ) {
		m_pSwingFactorLine->setVolume( pSong->getSwingFactor() );
	}
	m_pSwingFactorLine->setPeak_L ( pSong->getSwingFactor() );
	m_pSwingFactorLine->setPeak_R ( pSong->getSwingFactor() );
	m_pSwingFactorLine->updateMixerLine();


	// LADSPA
	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		LadspaFX *pFX = pSong->getLadspaFX( nFX );
		if ( pFX ) {
			m_pLadspaFXLine[nFX]->setName( pFX->getPluginName().c_str() );
			float fNewPeak_L = 0.0;
			float fNewPeak_R = 0.0;
			engine->getLadspaFXPeak( nFX, &fNewPeak_L, &fNewPeak_R );
			engine->setLadspaFXPeak( nFX, 0.0, 0.0 );	// reset

			float fOldPeak_L = 0.0;
			float fOldPeak_R = 0.0;
			m_pLadspaFXLine[nFX]->getPeaks( &fOldPeak_L, &fOldPeak_R );

			if (fNewPeak_L < fOldPeak_L)	fNewPeak_L = fOldPeak_L / fallOff;
			if (fNewPeak_R < fOldPeak_R)	fNewPeak_R = fOldPeak_R / fallOff;
			m_pLadspaFXLine[nFX]->setPeaks( fNewPeak_L, fNewPeak_R );
			m_pLadspaFXLine[nFX]->setFxActive( pFX->isEnabled() );
			m_pLadspaFXLine[nFX]->setVolume( pFX->getVolume() );
		}
		else {
			m_pLadspaFXLine[nFX]->setName( trUtf8("No plugin") );
			m_pLadspaFXLine[nFX]->setFxActive( false );
			m_pLadspaFXLine[nFX]->setVolume( 0.0 );
		}
	}
	// ~LADSPA
}



/// show event
void Mixer::showEvent ( QShowEvent *ev ) {
	updateMixer();
}



/// hide event
void Mixer::hideEvent ( QHideEvent *ev ) {
}



void Mixer::nameClicked(MixerLine* ref) {
	Hydrogen *pEngine = Hydrogen::getInstance();

	Song *song = pEngine->getSong();
	InstrumentList *instrList = song->getInstrumentList();
	int nSelectedInstr = pEngine->getSelectedInstrumentNumber();
	if (nSelectedInstr == -1) {
		return;
	}

	Instrument *pInstr = instrList->get( nSelectedInstr );
	if (pInstr) {
		bool bIsOkPressed;
		QString sOldName = QString( pInstr->getName().c_str() );
		QString text = QInputDialog::getText( "Hydrogen", trUtf8( "Instrument name" ), QLineEdit::Normal, sOldName, &bIsOkPressed, this );
		if ( bIsOkPressed && !text.isEmpty() ) {
			// user entered something and pressed OK
			pEngine->lockEngine( "InstrumentPropertiesDialog::nameClicked" );
			pInstr->setName( text.latin1() );
			pEngine->unlockEngine();

			HydrogenApp::getInstance()->getPatternEditorPanel()->getInstrumentList()->updateEditor();
		}
		else {
			// user entered nothing or pressed Cancel
		}
	}
}



void Mixer::nameSelected(MixerLine* ref)
{
	int nLine = findMixerLineByRef(ref);
	Hydrogen::getInstance()->setSelectedInstrumentNumber( nLine );

	Hydrogen::getInstance()->setSelectedInstrumentNumber(nLine);
}



void Mixer::panChanged(MixerLine* ref) {
	float panValue = ref->getPan();

	float pan_L = (100.0 - panValue) / 100.0;
	float pan_R = panValue / 100.0;

	panValue = panValue / 100.0;

	if (panValue >= 0.5) {
		pan_L = (1.0 - panValue) * 2;
		pan_R = 1.0;
	}
	else {
		pan_L = 1.0;
		pan_R = ( 1.0 - ( 1.0 - panValue) ) * 2;
	}

	int nLine = findMixerLineByRef(ref);
	Hydrogen::getInstance()->setSelectedInstrumentNumber( nLine );

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Instrument *instr = instrList->get(nLine);
	instr->setPan_L( pan_L );
	instr->setPan_R( pan_R );


	Hydrogen::getInstance()->setSelectedInstrumentNumber(nLine);
}



void Mixer::knobChanged(MixerLine* ref, int nKnob) {
	int nLine = findMixerLineByRef(ref);
	Hydrogen::getInstance()->setSelectedInstrumentNumber( nLine );

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();
	Instrument *pInstr = instrList->get(nLine);
	pInstr->setFXLevel( nKnob, ref->getFXLevel(nKnob) );
	QString sInfo = trUtf8( "Set FX %1 level ").arg( nKnob );
	( HydrogenApp::getInstance() )->setStatusBarMessage( sInfo+ QString( "[%1]" ).arg( ref->getFXLevel(nKnob), 0, 'f', 2 ), 2000 );

	Hydrogen::getInstance()->setSelectedInstrumentNumber(nLine);
}



void Mixer::humanizeChanged(FxMixerLine* ref) {
	Hydrogen *engine = Hydrogen::getInstance();
	ref->setPeak_L( ref->getVolume() );
	ref->setPeak_R( ref->getVolume() );

	char faderPos[100];
	float value = ref->getVolume();
	sprintf( faderPos, "%#.2f",  value);

	if (ref == m_pHumanizeTimeLine) {
		float humanizeValue = ref->getVolume();
		engine->lockEngine("Mixer::humanizeChanged");
		engine->getSong()->setHumanizeTimeValue( humanizeValue );
		engine->unlockEngine();
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Set humanize time parameter [%1]").arg( faderPos ), 2000 );
	}
	else if (ref == m_pHumanizeVelocityLine) {
		float humanizeValue = ref->getVolume();
		engine->lockEngine("Mixer::humanizeChanged");
		engine->getSong()->setHumanizeVelocityValue( humanizeValue );
		engine->unlockEngine();
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Set humanize velocity parameter [%1]").arg( faderPos ), 2000 );
	}
}



void Mixer::swingChanged( FxMixerLine* ref) {
	ref->setPeak_L( ref->getVolume() );
	ref->setPeak_R( ref->getVolume() );
	Hydrogen *engine = Hydrogen::getInstance();
	float swingFactor = ref->getVolume();
	engine->lockEngine("Mixer::swingChanged");
	engine->getSong()->setSwingFactor( swingFactor );
	engine->unlockEngine();

	char faderPos[100];
	float value = ref->getVolume();
	sprintf( faderPos, "%#.2f",  value);
	( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Set swing factor [%1]").arg( faderPos ), 2000 );
}



/// This method is called from another thread (audio engine)
void Mixer::noteOn( Note *note ) {
	H2TextEvent *ev = new H2TextEvent( "noteOn", note );
	QApplication::postEvent(this, ev);
}



void Mixer::customEvent( QCustomEvent *ev ) {
	if ( ev->type() != H2_TEXT_EVENT ) {	// Must be a H2TextEvent
		return;
	}
	QString message = ( (H2TextEvent*) ev )->getText();

	if (message == QString( "noteOn" )) {
		Note *note = ( (H2TextEvent*) ev )->getNote();

		Hydrogen *engine = Hydrogen::getInstance();
		Song *song = engine->getSong();
		InstrumentList *instrList = song->getInstrumentList();

		// search the instrument line
		int instrLine = -1;
		for (uint i = 0; i < instrList->getSize(); i++) {
			Instrument* instr = instrList->get(i);
			if (note->getInstrument() == instr) {
				instrLine = i;
				break;
			}
		}
		if (instrLine != -1) {
			m_pMixerLine[instrLine]->setActivity( 100 );
		}

		delete note;
		note = NULL;
	}

}



void Mixer::activeBtnChanged( FxMixerLine *ref ) {
	bool isActive = ref->isFxActive();
	Song *song = ( Hydrogen::getInstance() )->getSong();
	if ( ref == m_pHumanizeTimeLine ) {
		song->setHumanizeTimeEnabled( isActive );
		if (isActive) {
			( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Humanize Time FX enabled" ), 2000 );
		}
		else {
			( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Humanize Time FX disabled" ), 2000 );
		}
	}
	else if ( ref == m_pHumanizeVelocityLine ) {
		song->setHumanizeVelocityEnabled( isActive );
		if (isActive) {
			( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Humanize Velocity FX enabled" ), 2000 );
		}
		else {
			( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Humanize Velocity FX disabled" ), 2000 );
		}
	}
	else if ( ref == m_pSwingFactorLine ) {
		song->setSwingEnabled( isActive );
		if (isActive) {
			( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Swing factor FX enabled" ), 2000 );
		}
		else {
			( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Swing factor FX disabled" ), 2000 );
		}
	}
	else {
		cout << "Mixer::activeBtnChanged ??" << endl;
	}
}



void Mixer::resizeEvent ( QResizeEvent *ev )
{
	m_pFaderScrollView->resize( width() - 60 - 5, m_nMixerHeight );
	m_pFXScrollView->resize( width() - 60 - 5, m_nMixerHeight );

	uint nMaster_X = width() - MIXER_STRIP_WIDTH - 5;

	m_pMasterLine->move( nMaster_X, 4 );
	m_pShowFaderPanelBtn->move( nMaster_X + 1, 20 );
	m_pShowFXPanelBtn->move( nMaster_X + 1, 40 );
	m_pShowPeaksBtn->move( nMaster_X + 1, 60 );

	resize(width(), m_nMixerHeight);	// qt bug workaround
}



void Mixer::showFadersPanelClicked(Button* ref)
{
	m_pShowFaderPanelBtn->setPressed(true);
	m_pShowFXPanelBtn->setPressed(false);
	m_pFaderScrollView->show();
	m_pFXScrollView->hide();
}



void Mixer::showFXPanelClicked(Button* ref)
{
	m_pShowFXPanelBtn->setPressed(true);
	m_pShowFaderPanelBtn->setPressed(false);
	m_pFXScrollView->show();
	m_pFaderScrollView->hide();
}



void Mixer::showPeaksBtnClicked(Button* ref)
{
	Preferences *pPref = Preferences::getInstance();

	if ( ref->isPressed() ) {
		pPref->setInstrumentPeaks( true );
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Show instrument peaks = On"), 2000 );
	}
	else {
		pPref->setInstrumentPeaks( false );
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Show instrument peaks = Off"), 2000 );
	}
}



void Mixer::ladspaActiveBtnClicked( LadspaFXMixerLine* ref )
{
	bool bActive = ref->isFxActive();

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		if (ref == m_pLadspaFXLine[ nFX ] ) {
			LadspaFX *pFX = song->getLadspaFX(nFX);
			if (pFX) {
				pFX->setEnabled( bActive );
			}
			break;
		}
	}
}



void Mixer::ladspaEditBtnClicked( LadspaFXMixerLine *ref )
{
	infoLog( "ladspaEditBtnClicked" );
	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		if (ref == m_pLadspaFXLine[ nFX ] ) {
			( HydrogenApp::getInstance() )->getLadspaFXProperties(nFX)->hide();
			( HydrogenApp::getInstance() )->getLadspaFXProperties(nFX)->show();
		}
	}
	(Hydrogen::getInstance() )->getSong()->setModified( true );
}



void Mixer::ladspaVolumeChanged( LadspaFXMixerLine* ref)
{
	Song *pSong = (Hydrogen::getInstance() )->getSong();
	pSong->setModified( true );

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		if (ref == m_pLadspaFXLine[ nFX ] ) {
			LadspaFX *pFX = pSong->getLadspaFX(nFX);
			if (pFX) {
				pFX->setVolume( ref->getVolume() );
				QString sInfo = trUtf8( "Set LADSPA FX ( %1 ) volume").arg( QString(pFX->getPluginName().c_str()) );
				( HydrogenApp::getInstance() )->setStatusBarMessage( sInfo+ QString( " [%1]" ).arg( ref->getVolume(), 0, 'f', 2 ), 2000 );
			}
		}
	}


}



void Mixer::selectedInstrumentChanged()
{
	//infoLog( "[selectedInstrumentChanged]" );
	// do nothing: the selected instrument will be redrawn at next update cicle
}



void Mixer::getPeaksInMixerLine( uint nMixerLine, float& fPeak_L, float& fPeak_R )
{
	if ( (nMixerLine >= 0 ) && (nMixerLine < MAX_INSTRUMENTS) ) {
		fPeak_L = m_pMixerLine[ nMixerLine ]->getPeak_L();
		fPeak_R = m_pMixerLine[ nMixerLine ]->getPeak_R();
	}
	else {
		fPeak_L = 0;
		fPeak_R = 0;
	}
}



