/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: Song.h,v 1.6 2004/01/26 11:16:59 comix Exp $
 *
 */

#ifndef SONG_H
#define SONG_H


#include <string>
using std::string;

#include <vector>
using std::vector;

#include <map>
using std::map;

#include "Object.h"
#include "Sample.h"
#include "LocalFileMng.h"
#include "LadspaFX.h"


class Note;
class Instrument;
class Sequence;
class Pattern;
class Song;


//-----------------------------------------------------------------------------
/**
 * Sequence List
 */
class SequenceList : public Object {
	private:
		vector<Sequence*> list;
	public:
		SequenceList();
		~SequenceList();

		void add(Sequence* newSequence);
		Sequence* get(unsigned int pos);
		unsigned int getSize();

		SequenceList* copy();
};




//-----------------------------------------------------------------------------
/**
 * Pattern List
 */
class PatternList : public Object {
	private:
		vector<Pattern*> list;

	public:
		PatternList();
		~PatternList();

		void add(Pattern* newPattern);
		Pattern* get(unsigned int pos);
		unsigned int getSize();
		void clear();

		void replace( Pattern* newPattern, unsigned int pos );

		void del(Pattern *pattern);
		void del(uint index);
};




//-----------------------------------------------------------------------------
/**
 * Instrument List
 */
class InstrumentList : public Object {
	private:
		vector<Instrument*> list;
		map<Instrument*, unsigned int> posmap;
	public:
		InstrumentList();
		~InstrumentList();

		void add(Instrument* newPattern);
		Instrument* get(unsigned int pos);
		int getPos(Instrument* inst);
		unsigned int getSize();
		void replaceSample(Instrument* newInstrument, int pos);

};





//----------------------------------------------------------------------------
/**
 * A note...
 */
class Note : public Object {
	public:
		float samplePosition;

		/** Constructor. */
		Note(unsigned int position,
			float velocity,
			float pan_L,
			float pan_R,
			float nPitch = 0.0
		);

		/** Destructor */
		~Note();

		unsigned int getPosition(){	return position;	}
		void setPosition(unsigned int pos) {	this->position = pos;	}

		float getPan_L(){	return pan_L;	}
		void setPan_L( float val );

		float getPan_R(){	return pan_R;	}
		void setPan_R( float val );

		void setInstrument(Instrument* instrument);
		Instrument* getInstrument();

		Note* copy();

		float getVelocity(){	return velocity;	}
		void setVelocity( float val );

		uint getHumanizeDelay() {	return humanizeDelay;	}
		void setHumanizeDelay( uint val ) {	humanizeDelay = val;	}

		float getPitch() {	return m_nPitch;	}
		void setPitch( float nPitch ) {	m_nPitch = nPitch;	}
	private:
		/** Note position inside the song */
		unsigned int position;

		/** Velocity (intensity) of the note */
		float velocity;

		/** Pan of the note (left volume) */
		float pan_L;

		/** Pan of the note (right volume) */
		float pan_R;

		Instrument* instrument;

		/** Used in "humanize" function */
		uint humanizeDelay;

		float m_nPitch;
};



//----------------------------------------------------------------------------
/**
 * Instrument class (sample + info)
 */
class Instrument : public Object{
	public:
		/** Constructor. */
		Instrument( string id, string name, string author, int type, float volume );

		/** Destructor */
		~Instrument();

		/** Return the volume */
		float getVolume(){	return volume;	}
		void setVolume(float volume);

		void setId(string id) {	this->id = id;	}
		string getId(){	return id;	}

		string getName(){	return name;	}
		void setName(string name) {	this->name = name;	}

		string getAuthor(){	return author;	}

		void setSample(Sample *newSample){	sample = newSample;	}
		Sample* getSample(){	return sample;	}

		bool isMuted() {	return muted;	}
		void setMuted(bool muted) {	this->muted = muted;	}

		static Instrument* load(string filename);

		/** \todo implementation missing */
		void save(string filename);

		void setPeak_L( float fPeak );
		float getPeak_L() {	return peak_L;	}

		void setPeak_R( float fPeak );
		float getPeak_R() {	return peak_R;	}

		float getPan_L(){	return pan_L;	}
		void setPan_L( float val );

		float getPan_R(){	return pan_R;	}
		void setPan_R( float val );

		// LADSPA FX
		float getFXLevel(uint nFX){	return m_fFXLevel[nFX];	}
		void setFXLevel( uint nFX, float value ) {	m_fFXLevel[nFX] = value;	}

		void setDrumkitName( string drumkitName ) {	this->drumkitName = drumkitName;	}
		string getDrumkitName() {	return drumkitName;	}

		// FIXME: passare il riferimento
//		vector<Instrument*> getExcludeVect() {	return excludeVect;	}

		/** exclude this instruments */
		vector<Instrument*> excludeVect;

		vector<int> excludeVectId;

	private:
		/** ID of the instrument */
		string id;

		/** Volume of the instrument */
		float volume;

		/** Instrument name */
		string name;

		/** author of the Instrument */
		string author;

		Sample *sample;

		bool muted;

		/** current peak value */
		float peak_L;

		/** current peak value */
		float peak_R;

		/** Pan of the instrument (left volume) */
		float pan_L;

		/** Pan of the instrument (right volume) */
		float pan_R;

		/** Drumkit name */
		string drumkitName;

		/** Ladspa FX level */
		float m_fFXLevel[MAX_FX];
};





//----------------------------------------------------------------------------
/**
 * Sequence of notes
 */
class Sequence : public Object{
	public:
		Note* noteList[MAX_NOTES];

		/** Constructor. */
		Sequence(string name);

		/** Destructor */
		~Sequence();

		string getName(){	return name;	}
		void setName(string newName){	name = newName;	}

		Sequence* copy();

	private:
		string name;
};





//----------------------------------------------------------------------------
/**
 * Pattern (a container of sequences)
 */
class Pattern : public Object
{
	public:
		/** Constructor */
		Pattern(string name, uint nPatternSize = 192);

		/** Destructor */
		~Pattern();

		void setName(string name);
		string getName();

		SequenceList* getSequenceList(){	return sequenceList;	}
		void setSequenceList( SequenceList *vett ){	sequenceList = vett;	}

		static Pattern* getEmptyPattern();
		Pattern* copy();

		void setSize(uint nSize) {	m_nSize = nSize;	}
		uint getSize() {	return m_nSize;	}
	private:
		string name;
		uint m_nSize;
		SequenceList *sequenceList;

};






//----------------------------------------------------------------------------
/**
 * Song class.
 */
class Song : public Object{
	public:
		/** Constructor */
		Song(string id, string name, string author, float bpm, float volume);

		/** Destructor */
		~Song();

		string getId(){	return id;	}

		float getBpm(){	return bpm;	}
		void setBpm(float bpm);

		void setVolume(float volume) {	this->volume = volume;	}
		float getVolume() {	return volume;	}

		void setMetronomeVolume(float volume) {	this->metronomeVolume = volume;	}
		float getMetronomeVolume() {	return metronomeVolume;	}

		unsigned int getResolution(){	return resolution;	}

		string getName(){	return name;	}
		void setName(string name) {	this->name = name;	}

		void setAuthor(string author) {	this->author = author;	}
		string getAuthor(){	return author;	}

		PatternList* getPatternList(){	return patternList;	}
		void setPatternList( PatternList *vett ){	patternList = vett;	}

		PatternList* getPatternSequence(){	return patternSequence;	}
		void setPatternSequence( PatternList *vett ){	patternSequence = vett;	}

		static Song* load(string filename);
		void save(string filename);

		InstrumentList* getInstrumentList(){	return instrumentList;	}
		void setInstrumentList( InstrumentList *vett ){	instrumentList = vett;	}

		/** return the total number of ticks in the song */
		unsigned int estimateSongSize();

		static Song* getEmptySong();

		void setNotes(string newNotes) {	notes = newNotes;	}
		string getNotes() {	return notes;	}

		string getFilename() {	return filename;	}
		void setFilename(string filename) {	this->filename = filename;	}

		bool isModified() {	return modified;	}
		void setModified(bool status) {	this->modified = status;	}

		bool isLoopEnabled() {	return loopEnabled;	}
		void setLoopEnabled( bool enabled ) {	loopEnabled = enabled;	}

		float getHumanizeTimeValue();
		void setHumanizeTimeValue(float value);

		float getHumanizeVelocityValue();
		void setHumanizeVelocityValue(float value);

		float getSwingFactor() {	return swingFactor;	}
		void setSwingFactor(float factor);

		int getMode() {	return mode;	}
		void setMode( int newMode ) {	mode = newMode;	}

		bool isHumanizeTimeEnabled() {	return humanizeTimeEnabled;	}
		void setHumanizeTimeEnabled( bool enabled ) {	humanizeTimeEnabled = enabled;	}

		bool isHumanizeVelocityEnabled() {	return humanizeVelocityEnabled;	}
		void setHumanizeVelocityEnabled( bool enabled ) {	humanizeVelocityEnabled = enabled;	}

		bool isSwingEnabled() {	return swingEnabled;	}
		void setSwingEnabled( bool enabled ) {	swingEnabled = enabled;	}

		LadspaFX* getLadspaFX(uint nFX) {	return m_pLadspaFX[nFX];	}
		void setLadspaFX( uint nFX, LadspaFX* pLadspaFX ) {	m_pLadspaFX[nFX] = pLadspaFX;	}

	private:
		/** ID of the song */
		string id;

		/** Beats per minute */
		float bpm;

		/** Resolution of the song */
		unsigned int resolution;

		/* volume of the song (0.0..1.0)*/
		float volume;

		/* Metronome volume */
		float metronomeVolume;

		/** song name */
		string name;

		/** author of the song */
		string author;

		/** Notes */
		string notes;

		/** List of patterns */
		PatternList *patternList;

		/** Sequence of patterns */
		PatternList *patternSequence;

		/** Track instruments */
		InstrumentList *instrumentList;

		/** Filename */
		string filename;

		/** is modified? */
		bool modified;

		bool loopEnabled;

		bool humanizeTimeEnabled;
		float humanizeTimeValue;

		bool humanizeVelocityEnabled;
		float humanizeVelocityValue;

		bool swingEnabled;
		float swingFactor;

		/** Mode: PATTERN_MODE or SONG_MODE */
		int mode;

		LadspaFX* m_pLadspaFX[MAX_FX];
};





/**
 * Drumkit info
 */
class DrumkitInfo : public Object {
	public:
		/** Constructor */
		DrumkitInfo();

		/** Destructor */
		~DrumkitInfo();

		InstrumentList *getInstrumentList() {	return instrumentList;	}
		void setInstrumentList( InstrumentList* instr ) {	this->instrumentList = instr;	}

		void setName( string name ) {	this->name = name;	}
		string getName() {	return name;	}

		void setAuthor( string author ) {	this->author = author;	}
		string getAuthor() {	return author;	}

		void setInfo( string info ) {	this->info = info;	}
		string getInfo() {	return info;	}

		void dump();
		
	private:
		InstrumentList *instrumentList;
		string name;
		string author;
		string info;
};


#endif


