/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: PatternEditorPanel.cpp,v 1.55 2004/02/20 14:09:13 comix Exp $
 *
 */


#include "PatternEditorPanel.h"

#include "lib/PreferencesMng.h"
#include "lib/Hydrogen.h"

#include <cmath>

/**
 * Constructor
 */
PatternEditorPanel::PatternEditorPanel(QWidget *parent) : QWidget(parent)
{
//	cout << "PatternEditorPanel INIT" << endl;

	m_nPanelWidth = 706;
	m_nPanelHeight = 302;

	int nMaxHeight = 494;
	m_nSelectedInstrument = 0;

	setMinimumSize( 200, 200 );
	setMaximumSize( m_nPanelWidth, nMaxHeight );

	resize( m_nPanelWidth, m_nPanelHeight);

	setCaption( trUtf8( "Pattern Editor" ) );
	setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );





	// RULER____________________________________

	// Ruler ScrollView
	m_pRulerScrollView = new QScrollView( this );
	m_pRulerScrollView->setFrameShape( QFrame::NoFrame );
	m_pRulerScrollView->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pRulerScrollView->setHScrollBarMode( QScrollView::AlwaysOff );

	// Ruler frame
	m_pRulerFrame = new QFrame( m_pRulerScrollView->viewport() );

	// Ruler
	m_pPatternEditorRuler = new PatternEditorRuler( m_pRulerFrame );
	m_pPatternEditorRuler->move( 0, 0 );

	m_pRulerFrame->resize( m_pPatternEditorRuler->width(), m_pPatternEditorRuler->height() );
	m_pRulerScrollView->addChild( m_pRulerFrame );
	m_pRulerScrollView->move( 105, 30 );

	//_________________________________________





	// EDITOR _____________________________________

	// Editor scrollview
	m_pEditorScrollView = new QScrollView( this );
	m_pEditorScrollView->setFrameShape( QFrame::NoFrame );
	m_pEditorScrollView->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pEditorScrollView->setHScrollBarMode( QScrollView::AlwaysOff );
	m_pEditorScrollView->move( 105, 45 );

	// Editor frame
	m_pEditorFrame = new QFrame( m_pEditorScrollView->viewport() );

	// Editor
	m_pPatternEditor = new PatternEditor( m_pEditorFrame, this );
	m_pPatternEditor->move( 0, 0 );

	m_pEditorFrame->resize( m_pPatternEditor->width(), m_pPatternEditor->height() );
	m_pEditorScrollView->addChild( m_pEditorFrame );


	QScrollBar *pEditorVScroll = m_pEditorScrollView->verticalScrollBar();
	connect( pEditorVScroll , SIGNAL(valueChanged(int)), this, SLOT( contentsMoving(int) ) );

	//_______________________________________________





	// INSTRUMENT LIST________________________________________
	// Instrument list scrollview
	m_pInstrListScrollView = new QScrollView( this );
	m_pInstrListScrollView->setFrameShape( QFrame::NoFrame );
	m_pInstrListScrollView->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pInstrListScrollView->setHScrollBarMode( QScrollView::AlwaysOff );
	m_pInstrListScrollView->move( 5, 45 );

	// Instrument list frame
	m_pInstrListFrame = new QFrame( m_pInstrListScrollView->viewport() );

	// Instrument list
	m_pInstrumentList = new PatternEditorInstrumentList( m_pInstrListFrame, m_pPatternEditor, this );
	m_pInstrumentList->move( 0, 0 );

	m_pInstrListFrame->resize( m_pInstrumentList->width(), m_pPatternEditor->height() );
	m_pInstrListScrollView->addChild( m_pInstrListFrame );

	m_pPatternEditor->setPatternEditorInstrumentList( m_pInstrumentList );

	QScrollBar *pInstrListVScroll = m_pInstrListScrollView->verticalScrollBar();
	connect( pInstrListVScroll , SIGNAL(valueChanged(int)), this, SLOT( contentsMoving(int) ) );

	//____________________________________________________________





	// VELOCITY RULER______________________________________________
	m_pVelocityRulerScrollView = new QScrollView( this );
	m_pVelocityRulerScrollView->setFrameShape( QFrame::NoFrame );
	m_pVelocityRulerScrollView->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pVelocityRulerScrollView->setHScrollBarMode( QScrollView::AlwaysOff );
	m_pVelocityRulerScrollView->move( 5, 600 );


	m_pVelocityRulerFrame = new QFrame( m_pVelocityRulerScrollView->viewport() );


	m_pVelocityRuler = new PatternEditorVelocityRuler( m_pVelocityRulerFrame, m_pPatternEditor, this );
	m_pVelocityRuler->move(0,0);

	m_pVelocityRulerFrame->resize( m_pVelocityRuler->width(), m_pVelocityRuler->height() );

	m_pVelocityRulerScrollView->addChild( m_pVelocityRulerFrame );


	m_pPatternEditor->setPatternEditorVelocityRuler( m_pVelocityRuler );
	//_____________________________________________________________






	// external horizontal scrollbar
	m_pPatternEditorHScrollBar = new QScrollBar( this, "m_pPatternEditorHScrollBar" );
	m_pPatternEditorHScrollBar->setOrientation( Qt::Horizontal );
	connect( m_pPatternEditorHScrollBar, SIGNAL(valueChanged(int)), this, SLOT( syncToExternalHorizontalScrollbar(int) ) );

	// external vertical scrollbar
	m_pPatternEditorVScrollBar = new QScrollBar( this, "m_pPatternEditorVScrollBar" );
	m_pPatternEditorVScrollBar->setOrientation( Qt::Vertical );
	connect( m_pPatternEditorVScrollBar, SIGNAL(valueChanged(int)), this, SLOT( syncToExternalHorizontalScrollbar(int) ) );



	// Background image
	string background_path = string(IMG_PATH) + string("/img/patternEditor/patternEditor_background.png");
	bool ok = background.load(background_path.c_str());
	if( ok == false ){
		qWarning( "PatternEditorPanel: Error loading pixmap" );
	}
	setBackgroundPixmap(background);




	QLabel *gridResolutionLbl = new QLabel(this);
	gridResolutionLbl->setText(trUtf8( "Grid resolution" ));
	gridResolutionLbl->move( 10, 5 );
	gridResolutionLbl->resize( 100, 20 );
	gridResolutionLbl->setBackgroundPixmap(background);


	string hearBtn_on_path = string(IMG_PATH).append("/img/patternEditor/hearBtn_on.png");
	string hearBtn_off_path = string(IMG_PATH).append("/img/patternEditor/hearBtn_off.png");
	string hearBtn_over_path = string(IMG_PATH).append("/img/patternEditor/hearBtn_over.png");
	hearNotesBtn = new ToggleButton(this, QSize(20, 20), hearBtn_on_path, hearBtn_off_path, hearBtn_over_path);
	hearNotesBtn->move(200, 5);
	QToolTip::add( hearNotesBtn, trUtf8( "Hear new notes" ) );
	connect( hearNotesBtn, SIGNAL(clicked(Button*)), this, SLOT( hearNotesBtnClick(Button*)));


	// shows the pattern name
	patternNameLbl = new QLabel(this);
	patternNameLbl->setText(trUtf8( "Pattern name") );
	patternNameLbl->move( 284, 5 );
	patternNameLbl->resize( 300, 20 );
	patternNameLbl->setBackgroundPixmap(background);


	// Grid resolution combo
	gridResolutionCombo = new QComboBox( FALSE, this, "gridResolutionComboBox" );
	gridResolutionCombo->insertItem( trUtf8( "4" ) );
	gridResolutionCombo->insertItem( trUtf8( "8" ) );
	gridResolutionCombo->insertItem( trUtf8( "16" ) );
	gridResolutionCombo->insertItem( trUtf8( "32" ) );
	gridResolutionCombo->insertItem( trUtf8( "64" ) );

	gridResolutionCombo->insertItem( trUtf8( "4T" ) );	// triplets..
	gridResolutionCombo->insertItem( trUtf8( "8T" ) );
	gridResolutionCombo->insertItem( trUtf8( "16T" ) );
	gridResolutionCombo->insertItem( trUtf8( "32T" ) );

	gridResolutionCombo->move( 130, 5 );
	gridResolutionCombo->resize( 60, 20 );
	QToolTip::add( gridResolutionCombo, trUtf8( "Grid resolution" ) );
	connect( gridResolutionCombo, SIGNAL( activated(const QString&) ), this, SLOT( gridResolutionChanged() ) );

	PreferencesMng *pref = ( PreferencesMng::getInstance() );

	// restore grid resolution
	if ( pref->isPatternEditorUsingTriplets() == false ) {
		switch ( pref->getPatternEditorGridResolution() ) {
			case 4:
				gridResolutionCombo->setCurrentItem( 0 );
				break;

			case 8:
				gridResolutionCombo->setCurrentItem( 1 );
				break;

			case 16:
				gridResolutionCombo->setCurrentItem( 2 );
				break;

			case 32:
				gridResolutionCombo->setCurrentItem( 3 );
				break;

			case 64:
				gridResolutionCombo->setCurrentItem( 4 );
				break;

			default:
				gridResolutionCombo->setCurrentItem( 1 );
		}
	}
	else {
		switch ( pref->getPatternEditorGridResolution() ) {
			case 4:
				gridResolutionCombo->setCurrentItem( 5 );
				break;

			case 8:
				gridResolutionCombo->setCurrentItem( 6 );
				break;

			case 16:
				gridResolutionCombo->setCurrentItem( 7 );
				break;

			case 32:
				gridResolutionCombo->setCurrentItem( 8 );
				break;

			default:
				gridResolutionCombo->setCurrentItem( 5 );
		}
	}
	gridResolutionChanged();

	// restore hear new notes button state
	hearNotesBtn->setPressed( pref->getHearNewNotes() );


	// Pattern size combo
	pPatternSizeCombo = new QComboBox( FALSE, this, "pPatternSizeCombo" );
//	pPatternSizeCombo->setEditable( true );
	pPatternSizeCombo->insertItem( trUtf8( "1" ) );
	pPatternSizeCombo->insertItem( trUtf8( "2" ) );
	pPatternSizeCombo->insertItem( trUtf8( "3" ) );
	pPatternSizeCombo->insertItem( trUtf8( "4" ) );
	pPatternSizeCombo->insertItem( trUtf8( "5" ) );
	pPatternSizeCombo->insertItem( trUtf8( "6" ) );
	pPatternSizeCombo->insertItem( trUtf8( "7" ) );
	pPatternSizeCombo->insertItem( trUtf8( "8" ) );
	pPatternSizeCombo->move( 550, 5 );
	pPatternSizeCombo->resize( 80, 20 );
	pPatternSizeCombo->setCurrentItem( 7 );
	QToolTip::add( pPatternSizeCombo, trUtf8( "Pattern size" ) );
	connect( pPatternSizeCombo, SIGNAL( activated(const QString&) ), this, SLOT( patternSizeComboChanged() ) );


	string recBtn_on_path = string(IMG_PATH).append("/img/patternEditor/recBtn_on.png");
	string recBtn_off_path = string(IMG_PATH).append("/img/patternEditor/recBtn_off.png");
	string recBtn_over_path = string(IMG_PATH).append("/img/patternEditor/recBtn_over.png");
	recordEventsBtn = new ToggleButton(this, QSize(20, 20), recBtn_on_path, recBtn_off_path, recBtn_over_path);
	recordEventsBtn->move(230, 5);
	recordEventsBtn->setPressed(pref->getRecordEvents());
	QToolTip::add( recordEventsBtn, trUtf8( "Record keyboard/midi events" ) );
	connect( recordEventsBtn, SIGNAL(clicked(Button*)), this, SLOT( recordEventsBtnClick(Button*)));



	string quantBtn_on_path = string(IMG_PATH).append("/img/patternEditor/quantBtn_on.png");
	string quantBtn_off_path = string(IMG_PATH).append("/img/patternEditor/quantBtn_off.png");
	string quantBtn_over_path = string(IMG_PATH).append("/img/patternEditor/quantBtn_over.png");
	quantizeEventsBtn = new ToggleButton(this, QSize(20, 20), quantBtn_on_path, quantBtn_off_path, quantBtn_over_path);
	quantizeEventsBtn->move(254, 5);
	quantizeEventsBtn->setPressed(pref->getQuantizeEvents());
	QToolTip::add( quantizeEventsBtn, trUtf8( "Quantize keyboard/midi events to grid" ) );
	connect( quantizeEventsBtn, SIGNAL(clicked(Button*)), this, SLOT( quantizeEventsBtnClick(Button*)));

	// we need to do all this to support the keyboard playing
	// for all the window modes
	pPatternSizeCombo->installEventFilter(HydrogenApp::getInstance()->getMainForm());
	gridResolutionCombo->installEventFilter(HydrogenApp::getInstance()->getMainForm());

	(Hydrogen::getInstance())->addEngineListener(this);

}




/**
 * Destructor
 */
PatternEditorPanel::~PatternEditorPanel()
{
	delete m_pPatternEditor;
	delete m_pPatternEditorRuler;
	delete m_pInstrumentList;
	delete m_pVelocityRuler;
//	cout << "PatternEditorPanel DESTROY" << endl;
}





/**
 *
 */
void PatternEditorPanel::syncToExternalHorizontalScrollbar(int)
{
	int nValue;

	// Editor
	nValue = m_pPatternEditorHScrollBar->value();
	QScrollBar *pEditorHScroll = m_pEditorScrollView->horizontalScrollBar();
	if (pEditorHScroll->value() != nValue ) {
		pEditorHScroll->setValue( nValue );
	}

	nValue = m_pPatternEditorVScrollBar->value();
	QScrollBar *pEditorVScroll = m_pEditorScrollView->verticalScrollBar();
	if (pEditorVScroll->value() != nValue) {
		pEditorVScroll->setValue( nValue );
	}


	// Ruler
	nValue = m_pPatternEditorHScrollBar->value();
	QScrollBar *pRulerHScroll = m_pRulerScrollView->horizontalScrollBar();
	if ( pRulerHScroll->value() != nValue ) {
		pRulerHScroll->setValue( nValue );
	}

	// Instrument list
	nValue = m_pPatternEditorVScrollBar->value();
	QScrollBar *pInstrListVScroll = m_pInstrListScrollView->verticalScrollBar();
	if (pInstrListVScroll->value() != nValue ) {
		pInstrListVScroll->setValue( nValue );
	}

	// Velocity ruler
	nValue = m_pPatternEditorHScrollBar->value();
	QScrollBar *pVelocityRulerHScroll = m_pVelocityRulerScrollView->horizontalScrollBar();
	if ( pVelocityRulerHScroll->value() != nValue ) {
		pVelocityRulerHScroll->setValue( nValue );
	}
}




/**
 *
 */
void PatternEditorPanel::gridResolutionChanged()
{

	string selected = (gridResolutionCombo->currentText()).latin1();

	uint nResolution;
	bool bUseTriplets = false;
	if (selected == "4") {
		nResolution = 4;
	}
	else if (selected == "8") {
		nResolution = 8;
	}
	else if (selected == "16") {
		nResolution = 16;
	}
	else if (selected == "32") {
		nResolution = 32;
	}
	else if (selected == "64") {
		nResolution = 64;
	}
	else if (selected == "4T") {
		nResolution = 8;
		bUseTriplets = true;
	}
	else if (selected == "8T") {
		nResolution = 16;
		bUseTriplets = true;
	}
	else if (selected == "16T") {
		nResolution = 32;
		bUseTriplets = true;
	}
	else if (selected == "32T") {
		nResolution = 64;
		bUseTriplets = true;
	}
	m_pPatternEditor->setResolution( nResolution, bUseTriplets );

	( PreferencesMng::getInstance() )->setPatternEditorGridResolution( atoi( selected.c_str() ) );
	( PreferencesMng::getInstance() )->setPatternEditorUsingTriplets( bUseTriplets );
}




///
/// This method will be called only when the engine is not playing. Engine lock/unlock not needed
///
void PatternEditorPanel::patternSizeComboChanged()
{
	string selected = (pPatternSizeCombo->currentText()).latin1();

	uint nQuarter = 192 / 8;

	uint nSize;
	if (selected == "1") {
		nSize = nQuarter;
	}
	else if (selected == "2") {
		nSize = nQuarter * 2;
	}
	else if (selected == "3") {
		nSize = nQuarter * 3;
	}
	else if (selected == "4") {
		nSize = nQuarter * 4;
	}
	else if (selected == "5") {
		nSize = nQuarter * 5;
	}
	else if (selected == "6") {
		nSize = nQuarter * 6;
	}
	else if (selected == "7") {
		nSize = nQuarter * 7;
	}
	else if (selected == "8") {
		nSize = nQuarter * 8;
	}
	else {
		cout << "[PatternEditorPanel::patternSizeComboChanged()] Unhandled case " << selected << endl;
	}


	Pattern *currentPattern = (Hydrogen::getInstance())->getCurrentPattern();
	if (currentPattern != NULL) {
		currentPattern->setSize( nSize );
	}

	m_pPatternEditorRuler->updateEditor( true );	// redraw all
}






/**
 *
 */
void PatternEditorPanel::updateStart( bool bStart )
{
	m_pPatternEditorRuler->updateStart( bStart );
}




/**
 * This method is called from another thread (audio engine)
 */
void PatternEditorPanel::patternChanged()
{
	H2TextEvent *ev = new H2TextEvent("patternChanged");
	QApplication::postEvent(this, ev);
}

/**
 * This method is called from another thread (audio engine)
 */
void PatternEditorPanel::patternModified()
{
	H2TextEvent *ev = new H2TextEvent("patternModified");
	QApplication::postEvent(this, ev);
}


/**
 *
 */
void PatternEditorPanel::customEvent( QCustomEvent *ev )
{
	if ( ev->type() != H2_TEXT_EVENT ) {	// Must be a H2TextEvent
		return;
	}
	QString message = ((H2TextEvent*)ev)->getText();

	if (message == QString( "patternChanged" ) ) {
		// Audio engine current pattern
		string currentPatternName;
		Pattern *currentPattern = (Hydrogen::getInstance())->getCurrentPattern();
		if (currentPattern != NULL) {
			currentPatternName = currentPattern->getName();
			this->setCaption( ( trUtf8( "Pattern editor - %1").arg(currentPatternName.c_str()) ) );

			// update pattern size combobox
			int nPatternSize = currentPattern->getSize();
			int nQuarter = 192 / 8;
			if (nPatternSize == nQuarter) {
				pPatternSizeCombo->setCurrentItem( 0 );
			}
			else if (nPatternSize == nQuarter * 2) {
				pPatternSizeCombo->setCurrentItem( 1 );
			}
			else if (nPatternSize == nQuarter * 3) {
				pPatternSizeCombo->setCurrentItem( 2 );
			}
			else if ( nPatternSize == nQuarter * 4 ) {
				pPatternSizeCombo->setCurrentItem( 3 );
			}
			else if ( nPatternSize == nQuarter * 5 ) {
				pPatternSizeCombo->setCurrentItem( 4 );
			}
			else if ( nPatternSize == nQuarter * 6 ) {
				pPatternSizeCombo->setCurrentItem( 5 );
			}
			else if ( nPatternSize == nQuarter * 7 ) {
				pPatternSizeCombo->setCurrentItem( 6 );
			}
			else if ( nPatternSize == nQuarter * 8 ) {
				pPatternSizeCombo->setCurrentItem( 7 );
			}
			else {
				cout << "[PatternEditorPanel::customEvent()] unhandled case: nPatternSize = " << nPatternSize;
			}
		}
		else {
			currentPatternName = (trUtf8( "<b>No pattern selected! (NULL pattern)</b>" ) ).latin1();
			this->setCaption( ( trUtf8( "Pattern editor - ") + trUtf8( "No pattern selected! (NULL pattern)")  ) );
		}
		patternNameLbl->setText( trUtf8( "Pattern name: <b>") + QString(currentPatternName.c_str()) + "</b>" );
	}
	else if (message == "enablePatternSizeCombo") {
		pPatternSizeCombo->setEnabled( true );
	}
	else if (message == "disablePatternSizeCombo") {
		pPatternSizeCombo->setEnabled( false );
	}
	else if (message == "patternModified") {
		// full update of display
		m_pPatternEditor->updateEditor(true);
	}
}




/**
 *
 */
void PatternEditorPanel::hearNotesBtnClick(Button *ref)
{
	PreferencesMng *pref = ( PreferencesMng::getInstance() );
	pref->setHearNewNotes( ref->isPressed() );

	if (ref->isPressed() ) {
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Hear new notes = On" ), 2000 );
	}
	else {
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Hear new notes = Off" ), 2000 );
	}

}


/**
 *
 */
void PatternEditorPanel::recordEventsBtnClick(Button *ref)
{
	PreferencesMng *pref = ( PreferencesMng::getInstance() );
	pref->setRecordEvents( ref->isPressed() );

	if (ref->isPressed() ) {
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Record keyboard/midi events = On" ), 2000 );
	}
	else {
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Record keyboard/midi events = Off" ), 2000 );
	}

}

/**
 *
 */
void PatternEditorPanel::quantizeEventsBtnClick(Button *ref)
{
	PreferencesMng *pref = ( PreferencesMng::getInstance() );
	pref->setQuantizeEvents( ref->isPressed() );

	if (ref->isPressed() ) {
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Quantize incoming keyboard/midi events = On" ), 2000 );
	}
	else {
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Quantize incoming keyboard/midi events = Off" ), 2000 );
	}
}




/**
 *
 */
void PatternEditorPanel::stateChanged(int state)
{
	if ( state == READY) {
		H2TextEvent *ev = new H2TextEvent("enablePatternSizeCombo");
		QApplication::postEvent(this, ev);
	}
	else {
		H2TextEvent *ev = new H2TextEvent("disablePatternSizeCombo");
		QApplication::postEvent(this, ev);
	}
}



/**
 *
 */
void PatternEditorPanel::resizeEvent( QResizeEvent *ev )
{

	m_nPanelHeight = height();
	m_nPanelWidth = width();

	m_pEditorScrollView->resize( m_nPanelWidth - 120, m_nPanelHeight - 110 );
	m_pRulerScrollView->resize( m_pEditorScrollView->width(), m_pRulerScrollView->height() );
	m_pInstrListScrollView->resize( m_pInstrListScrollView->width(), m_pEditorScrollView->height() );

	// pattern editor Horizontal scroll bar
	m_pPatternEditorHScrollBar->resize( m_pEditorScrollView->width(), 16 );
	m_pPatternEditorHScrollBar->move( 105, m_nPanelHeight - 16 );
	m_pPatternEditorHScrollBar->setMinValue( 0 );
	m_pPatternEditorHScrollBar->setMaxValue( m_pEditorFrame->width() - m_pEditorScrollView->width() );
	m_pPatternEditorHScrollBar->setLineStep(20);
	m_pPatternEditorHScrollBar->setPageStep(300);

	// pattern editor Vertical scroll bar
	m_pPatternEditorVScrollBar->resize( 16, m_pEditorScrollView->height() );
	m_pPatternEditorVScrollBar->move( m_nPanelWidth - 16, m_pEditorScrollView->y() );
	m_pPatternEditorVScrollBar->setMinValue( 0 );
	m_pPatternEditorVScrollBar->setMaxValue( m_pEditorFrame->height() - m_pEditorScrollView->height() );
	m_pPatternEditorVScrollBar->setLineStep(20);
	m_pPatternEditorVScrollBar->setPageStep(300);


	// Velocity ruler
	m_pVelocityRulerScrollView->move( 105, m_nPanelHeight - 65 );
	m_pVelocityRulerScrollView->resize( m_pEditorScrollView->width(), m_pVelocityRuler->height() );

}




void PatternEditorPanel::showEvent ( QShowEvent *ev )
{
	m_pPatternEditorVScrollBar->setValue( m_pPatternEditorVScrollBar->maxValue() );
}


/// richiamato dall'uso dello scroll del mouse
void PatternEditorPanel::contentsMoving(int dummy)
{
	syncToExternalHorizontalScrollbar(0);
}




void PatternEditorPanel::setSelectedInstrument( int nInstr )
{
	m_nSelectedInstrument = nInstr;

	m_pPatternEditor->updateEditor(true);	// force repaint
	m_pInstrumentList->updateEditor();
	m_pVelocityRuler->updateEditor();

}


