/*
 * commands_pop3.c -- POP3 specific command handlers
 * This file contains all the POP3 specific command implementations. Should be
 * used in conjunction with commands.c which contains all the POP3/IMAP
 * independent commands.
 *
 * Created by: David Smith <courierdave@users.sourceforge.net> 27-Nov-2003
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "httpmail.h"
#include "hotwayd.h"
#include "libghttp-1.0.9-mod/ghttp.h"
#include "libghttp-1.0.9-mod/ghttp_constants.h"
#include "commands.h"
#include "xmlstuff.h"

extern char *serverurl;
extern int unauth_cnt;
extern ghttp_request *request;
extern char buffer[N_BUFFER];
extern char *folder;

static int listdownloaded = 0; /* got the inbox list yet? */
static int set_read = 0; /* should we update the readmark after downloading? */
static FOLDER_STRUCT *inbox_props; /* stores all properties for opened folder */ 

static int top_lines = -1;
int pastheader = 0; /* used for displaying correctly when using top command */
static int lazy_dump(char *pos, int len, int *curpos_p, int *curline_p, int maxlines);
extern char *UserName;  /* allocate space depending on size later */
extern char *PassWord;
extern const char *http_hdr_Translate;
static char *msg_data = NULL;
static int getmessage(ghttp_request *request, int num_msg, FOLDER_STRUCT *folder_props, int lines);

/* functions from commands.c */
extern int getlist(ghttp_request *request, FOLDER_STRUCT *folder_props);
extern char *get_moved_url(ghttp_request *);
extern int cp_mv_msg(ghttp_request *request, int num_msg, FOLDER_STRUCT *folder_props, char *dest_href, int cmd);
extern int handlereplies(ghttp_request *request, int reply);
extern int dologin(ghttp_request *request, int protocol);
extern int propfind(ghttp_request *request, char *myurl, int whichrequest);
extern void request_folder_props(ghttp_request *request);

void set_readupdate_flag(int flag);

/* Function: inboxprops_destroy
 * Destroy the inbox_props data structure
 */
void inboxprops_destroy(void)
{
  /* Free memory */
  if (inbox_props != NULL) {
    free(inbox_props->href);
    /* free(inbox_props->trash); */
    free(inbox_props->name);
    if (inbox_props->visiblecount > 0 && listdownloaded) {
      int i;
      for (i = 0; i < inbox_props->visiblecount; i++) {
	if( inbox_props->msg_props[i]->href ){
	  free(inbox_props->msg_props[i]->href);
	}
	if( inbox_props->msg_props[i]->message ){
	  free(inbox_props->msg_props[i]->message);
	}
	if (inbox_props->msg_props[i]->date)
	  free(inbox_props->msg_props[i]->date);
	free(inbox_props->msg_props[i]);
      }
    }
    listdownloaded=0; /* 20020924: reset listdownloaded variable */
    free(inbox_props->msg_props);
    inbox_props->msg_props= NULL;
    free(inbox_props->unread_props);
    inbox_props->unread_props=NULL;
    free(inbox_props);
  }
  inbox_props= NULL;
}



void httpmail_dele(cmdtable *command, char *arg)
{
  long i;

  register_command( NULL, NULL, 0, NULL, NULL );	
  if ((i = strtoul(arg, NULL, 10)) && (i <= inbox_props->listcount) && (i > 0)) {
    if (cp_mv_msg(request, i-1, inbox_props, NULL, TRASH)) {
      PSOUT("+OK Message deleted");
      PCRLF;
    }
    else {
      PSOUT("-ERR Error deleting message");
      PCRLF;
    }
  }
  else {
    PSOUT("-ERR No such message");
    PCRLF;
  }
}


void httpmail_list(cmdtable *command, char *arg)
{
  long i;
  short cache_flag=1;

  /*  if (((!listdownloaded || (time(NULL)-time_getlist > GETLIST_TIMEOUT)) &&
      inbox_props->href)) { */
  if (!listdownloaded && inbox_props->href) {
    getlist(request, inbox_props);
    listdownloaded = 1;
    cache_flag--;
  }

  if (arg == NULL) {
    if (!cache_flag) {
      PSOUT("+OK Mailbox scan listing follows");
    } else {
      PSOUT("+OK Cached mailbox scan listing follows");
    }
    PCRLF;
    for (i = 0; i < inbox_props->listcount; i++) {
      snprintf(buffer, N_BUFFER, "%lu %u\015\012", (i + 1), inbox_props->list_props[i]->length);
      PSOUT(buffer);
    }
    PSOUT(".");
    PCRLF;
    return;
  }
  if ((i = strtoul(arg, NULL, 10)) && (i <= inbox_props->listcount) && (i > 0)) {
    snprintf(buffer, N_BUFFER, "+OK %lu %u", i, inbox_props->list_props[i - 1]->length);
    register_command( command, buffer, 1, arg, NULL );
    PSOUT(buffer);
    PCRLF; 
  } else {
    register_command( NULL, NULL, 0, NULL, NULL );
    PSOUT("-ERR No such message");
    PCRLF;
  }
}


long httpmail_nemails(void)
{
  return inbox_props->listcount;
}


void httpmail_retr(cmdtable *command, char *arg)
{
  long i;

  if ((i = strtoul(arg, NULL, 10)) && (i <= inbox_props->listcount) && (i > 0)) {
    if (getmessage(request, i-1, inbox_props, -1) != E_GETMSG_OK)
      PSOUT("-ERR Internal error downloading message");
    else if (set_read)
      update_readmark(inbox_props->list_props[i-1], 1);
  }
  else {
    PSOUT("-ERR No such message");
    PCRLF;
    register_command( NULL, NULL, 0, NULL, NULL );
  }
}



void httpmail_rset(cmdtable *command)
{
  register_command( NULL, NULL, 0, NULL, NULL );
  PSOUT("+OK Mailbox has been reset");
  PCRLF;
}

/* Function: httpmail_stat
 * Implementation of the POP-3 STAT command. It returns the number of messages
 * in the mailbox and the size (in octets) of the mailbox. Size is pretty much
 * invalid because of the values that hotmail reports, so we hope the client
 * doesn't worry about this value too much.
 */
void httpmail_stat(cmdtable *command)
{
  long maildrop_size = 0, n_not_deleted_emails = 0, i;

  /* GET MESSAGES LIST */
  if (inbox_props->href) {
    getlist(request, inbox_props);
    listdownloaded = 1;
    for (i = 0; i < inbox_props->listcount; i++) {
      /* 20030130 RJVB: FIXME: deleted mails seem to show up as empty messages, at the end of the list
	 \ can we just not increment n_not_deleted_emails when length==0, or only when there is no more
	 \ non-zero-length message *after* the current one?
	 \ But in fact, visiblecount should not have included empty
	 messages! */
      /* 20030421 DAVE: fixed visiblecount inside xmlstuff.c. stat
	 \ and list run update_visiblecount before filling the message
	 \structs now. Added a check if running in debug mode also.
      */
      n_not_deleted_emails++;
      maildrop_size += inbox_props->list_props[i]->length;
    }
  }
  
  snprintf(buffer, N_BUFFER, "+OK %lu %lu", n_not_deleted_emails, maildrop_size);
  register_command( command, buffer, 0, NULL, NULL );
  PSOUT(buffer);
  PCRLF;
}



/* 20020923 RJVB: implementation for the top command, as I understood it from a true pop server */
void httpmail_top(cmdtable *command, char *arg, char *arg2)
{
  unsigned long i;
  long lines= strtol(arg2, NULL, 10);

  if (lines < 0) {
    /* For an invalid lines count, just return the whole body */
    httpmail_retr(command,arg);
  } else {
    if ((i = strtoul(arg, NULL, 10)) && (i <= inbox_props->listcount) && (i > 0)) {
      if (getmessage(request, i-1, inbox_props, lines) != E_GETMSG_OK)
	PSOUT("-ERR Internal error downloading message");
      else if (set_read)
	update_readmark(inbox_props->list_props[i-1], 1);	
    } else {
      PSOUT("-ERR No such message");
      PCRLF;
      register_command( NULL, NULL, 0, NULL, NULL );
    }
  }
}


void httpmail_uidl(cmdtable *command, char *arg)
{
  char buf[32];
  long i;

  if (!listdownloaded && inbox_props->href) {
    getlist(request, inbox_props);
    listdownloaded = 1;
  }

  if (arg == NULL) {
    char *response= (char*) calloc(inbox_props->listcount* N_BUFFER+2, sizeof(char));
    snprintf(buffer, N_BUFFER, "+OK unique-id listing follows\015\012");
    if (response) strcat(response, buffer);
    PSOUT(buffer);
    for (i = 0; i < inbox_props->listcount; i++) {
      snprintf(buffer, N_BUFFER, "%lu %s\015\012", (i + 1), (char *)gen_uidl(buf, inbox_props->list_props[i]->href));
      if( response ){
	strcat( response, buffer );
      }
      PSOUT(buffer);
    }
    PSOUT(".");
    if( response ){
      strcat( response, "." );
    }
    PCRLF;
    register_command( command, response, 0, NULL, NULL );
    if( response ){
      free(response);
    }
    return;
  }
  else if ((i = strtoul(arg, NULL, 10)) && (i <= inbox_props->listcount) && (i > 0)) {
    snprintf(buffer, N_BUFFER, "+OK %lu %s\015\012", i, (char *)gen_uidl(buf, inbox_props->list_props[i - 1]->href));
    PSOUT(buffer);
    /* Don't need the PCRLF here as buffer already has CR/LF added */
  } else {
    register_command( NULL, NULL, 0, NULL, NULL );
    PSOUT("-ERR No such message");
    PCRLF;
  }
}


/* Function: open_folder
 * Upon login we need to try and open the requested folder. This function is
 * called by dologin in commands.c to try and open the folder (which the
 * default is inbox). We need to do this to do the implicit STAT on the
 * mailbox which is required by the POP3 protocol. Before this function is
 * called we should already be logged into the mailbox.
 * Returns 1 on success, 0 on failure.
 */
int open_folder(void) {
  extern ghttp_request *request; /* FIXME: make local */
  
  /* allocate inbox_props structure */
  if (!inbox_props) inbox_props = (FOLDER_STRUCT *) calloc(1, sizeof(FOLDER_STRUCT));
  /* fill the data in */
  request_folder_props(request);
    /* Lets parse the XML and see if we can find the folder we were after */
  return ParseXMLFolderProps(request, folder, inbox_props);
}

void set_readupdate_flag(int flag) {
  set_read = flag;
}

/* Function: getmessage
 * Checks to see if we have already downlowded the message. If so it simply
 * spews out the cached message, otherwise it sends a HTTPMail request for the
 * message and subsequently spews it out.
 */
static int getmessage(ghttp_request *request, int num_msg, FOLDER_STRUCT *folder_props, int lines)
{
  pastheader = 0;
  if (folder_props && folder_props->list_props && folder_props->list_props[num_msg]->message) {
    int curpos = 0;
    int curline = 0;
    PSOUT("+OK We have that cached\r\n");
    lazy_dump(folder_props->list_props[num_msg]->message, -1, &curpos, &curline, lines);
    PSOUT("\r\n.\r\n");
    return E_GETMSG_OK;
  }
  else if(folder_props && folder_props->list_props && folder_props->list_props[num_msg]->href) {
    top_lines = lines; /* set global variable */
    /* Settings of our connection */
    ghttp_clean(request);
    ghttp_set_uri(request, folder_props->list_props[num_msg]->href);
    ghttp_set_type(request, ghttp_type_lazy);
    
    /* Set Headers */
    ghttp_set_header(request, http_hdr_Accept, "message/rfc822, */*");
    ghttp_set_header(request, http_hdr_Translate, "f");
    set_authinfo(request, UserName, PassWord, NULL, 1);

    /* Prepare and process the connection */
    prepare_and_send(request, folder_props->msg_props[num_msg]->href);
    
    free(folder_props->msg_props[num_msg]->message);
      
    if (msg_data)
      folder_props->msg_props[num_msg]->message = msg_data;
    
    msg_data = NULL;
    
    return E_GETMSG_OK;
  } else {
    return E_GETMSG_ERROR;
  }
}

/*
 * Function lazy_dump
 *
 * This function is used by lazy_handler to print out the buffer collected by
 * lazy_handler line by line. The reason we do it like this is two fold. First
 * so that we can implement TOP and second so that if we are downloading a big
 * message it means the mail reader won't time out waiting for hotwayd to spew
 * the message out in a big chunk. It also makes the problem more complex so
 * why not :-)
 */
static int lazy_dump(char *pos, int len, int *curpos_p, int *curline_p, int maxlines) 
 {
  char foo[LAZY_BUFSIZE+1];

  char *cureol = NULL;

  int curpos = *curpos_p;
  int curline = *curline_p;

  int i;

  int just_sent_line = 1;
  int will_send_line = 0;

  foo[LAZY_BUFSIZE] = '\0';

  if(len == -1)
    len = strlen(pos);

  /* hack hack hack */
  if(maxlines == 0 && curline == 0)
    curline = -1;

  /* flush from pos */
  while(((maxlines == -1) || (curline < maxlines)) && (curpos < len)) {
	char *next_n = NULL;
	char *next_r = NULL;

	next_n=strstr(pos+curpos, "\n");
	next_r=strstr(pos+curpos, "\r");

	if (next_r && (*(next_r+1) == '\n')) next_r++; /* handle \r\n */

	/* check which is the next to come, a \r, a \n or a \r\n */
	if (next_r && next_n && (next_r < next_n)) {
	  cureol = next_r;
	} else if (next_n) {
	  cureol = next_n;
	} else {
	  cureol = NULL;
	}
	  
	if(cureol) 
      i = MIN(cureol-(pos+curpos)+1, LAZY_BUFSIZE);

    if(cureol && (i == (cureol-(pos+curpos)+1))) {

      if(maxlines && pastheader) {
		curline++; /* we are past the header, start counting lines */
	  }

      if( !pastheader && /* still in header */
	  just_sent_line &&  /* just flushed to a \r\n */
	  ( (pos[curpos] == '\r' && pos[curpos+1] == '\n') ||
	    (pos[curpos] == '\n') ) ) { /* and find ourselves at another \r\n */ 
		pastheader++; /* right, now we can start counting lines.. */
	if(maxlines == 0 && curline == -1)
	   curline = 0;
      }
      
      will_send_line = 1;
    } else {
      i = MIN(len-curpos, LAZY_BUFSIZE);
      just_sent_line = 0;
    }

    memcpy(foo, pos+curpos, i);
    foo[i] = '\0';

    if(just_sent_line && foo[0] == '.')
	printf(".");

    printf("%s", foo);

    just_sent_line = will_send_line;

    will_send_line = 0;

    curpos += i;
  }

  *curline_p = curline;
  *curpos_p = curpos;

  if(curpos >= len) {
	return 1;
  }
  if(curline >= maxlines) {
    return 2;
  }
  return 0;
}

/*
 * Function lazy_handler
 *
 * We use this function to progressively read a message from the mail
 * server. It works by filling an internal buffer (of LAZY_BUFSIZE) and
 * sending chunks to lazy_dump to be output. 
 * 
 * what i said about "it was a mess when i got here" being no excuse..
 * i was lying through my libghttping[1] teeth
 *  -jbm
 * [1]: yes, it's now officially profanity.
 */
int lazy_handler(char *pos, int len, int totallen, int fd)
{
  char foo[LAZY_BUFSIZE + 1];
  char *dest = NULL;
  int curpos = 0;
  int i;
  int kg = KG_STATE_READ;
  int curline = 0;
  int curlen = 0;
  foo[LAZY_BUFSIZE] = '\0';

  /* this is a bit of a nasty hack, but it does the job ;) */
  /*  if (buffer == 1)
      return lazy_handler_buffer(pos, len, totallen, fd); */

  PSOUT("+OK Going to Lazy-download that one\r\n");
  
  if(!(dest = grow_and_copy(dest, 0, pos, len))) {
    printf("-ERR OR damn, i dunno: %s\r\n", strerror(errno));
    return -1;
  }
  curlen += len;

  while(kg > KG_STATE_CACHE) {
    if(kg == KG_STATE_READ) {
      /* better not to call read at end of buffer so we'll use totallen which
       * was discovered from Content-Length of headers to see if we have
       * finished reading all of the message. It is also possible to use
       * select here, but I'd rather leave it as the default timeout incase
       * someone has a very slow connection */
      if (curlen < totallen)
	i = read(fd, foo, LAZY_BUFSIZE);
      else
	i = 0;
      if(i < 0) {
	kg = KG_STATE_UNCACHE;
	continue;
      }

      if(i == 0)
	kg = KG_STATE_DUMP;

      foo[i] = '\0';

      if(!(dest = grow_and_copy(dest, curlen, foo, i))) { 
	printf("-ERR OR damn, i dunno: %s\r\n", strerror(errno));
	return -1;
      }
      curlen += i;
    }

    if(kg == KG_STATE_DUMP || kg == KG_STATE_READ) {
      i = lazy_dump(dest, curlen, &curpos, &curline, top_lines);

      if(kg == KG_STATE_DUMP && i == 1)
	kg = KG_STATE_CACHE;
      if(i == 2) /* to handle TOP as well as RETR -- top probably won't hit DUMP*/
	kg = KG_STATE_UNCACHE;

    }
  }

  PSOUT("\r\n.\r\n"); /* required by POP3 standard at end of message */
  
  if(kg == KG_STATE_CACHE)
    msg_data = dest;
  else {
    free(dest);
    msg_data = NULL;
  }


  return 1;
}
