<?php

require_once dirname(__FILE__) . '/spamd.php';

/**
 * Sam storage implementation for LDAP backend.
 * Requires SpamAssassin patch found at:
 * http://bugzilla.spamassassin.org/show_bug.cgi?id=2205
 *
 * Required parameters:<pre>
 *   'ldapserver'       The hostname of the ldap server.
 *   'basedn'           The password associated with 'username'.
 *   'attribute'        The database type (ie. 'pgsql', 'mysql', etc.).
 *   'uid'              The communication protocol ('tcp', 'unix', etc.).</pre>
 *
 * $Horde: sam/lib/Driver/spamd_ldap.php,v 1.14 2006/01/01 21:11:50 jan Exp $
 *
 * Copyright 2003-2006 Chris Bowlby <excalibur@hub.org>
 * Copyright 2003-2006 Max Kalika <max@horde.org>
 * Copyright 2003-2006 Neil Sequeira <neil@ncsconsulting.com>
 * Copyright 2004-2006 Jan Schneider <jan@horde.org>
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Chris Bowlby <excalibur@hub.org>
 * @author  Max Kalika <max@horde.org>
 * @author  Neil Sequeira <neil@ncsconsulting.com>
 * @author  Jan Schneider <jan@horde.org>
 * @since   SAM 0.1
 * @package Sam
 */
class SAM_Driver_spamd_ldap extends SAM_Driver {

    /**
     * Handle for the current LDAP connection.
     *
     * @var resource
     */
    var $_linkid;

    /**
     * Boolean indicating whether or not we're connected to the LDAP server.
     *
     * @var boolean
     */
    var $_connected = false;

    /**
     * Constructs a new LDAP storage object.
     *
     * @param string $user   The user who owns these SPAM options.
     * @param array $params  A hash containing connection parameters.
     */
    function SAM_Driver_spamd_ldap($user, $params = array())
    {
        $this->_user = $user;
        $this->_params = $params;
    }

    /**
     * Retrieve an option set from the storage backend.
     *
     * @access private
     */
    function retrieve()
    {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        /* Set default values. */
        $this->_setDefaults();
        $eres = false;
        $user = $this->_user;
        $attrib = strtolower($this->_params['attribute']);
        $filter = sprintf('(%s=%s)', $this->_params['uid'], $user);
        $res = ldap_search($this->_linkid, $this->_params['basedn'],
                           $filter, array($attrib));
        if ($res) {
            $eres = ldap_get_entries($this->_linkid, $res);
        }
        if ($eres && isset($eres[0][$attrib])) {
            for ($i = 0; $i < $eres[0][$attrib]['count']; $i++) {
                list($a, $v) = explode(' ', $eres[0][$attrib][$i]);
                $ra = $this->_mapOptionToAttribute($a);
                if (is_numeric($v)) {
                    if (strstr($v, '.')) {
                        $newoptions[$ra][] = (float) $v;
                    } else {
                        $newoptions[$ra][] = (int) $v;
                    }
                } else {
                    $newoptions[$ra][] = $v;
                }
            }

            /* go through new options and pull single values
             * out of their arrays */
            foreach ($newoptions as $k => $v) {
                if (count($v) > 1) {
                    $this->_options[$k] = $v;
                } else {
                    $this->_options[$k] = $v[0];
                }
            }
        }
    }

    /**
     * Set default values.
     *
     * @access private
     */
    function _setDefaults()
    {
        $this->_options = array_merge($this->_options, $this->_params['defaults']);
    }

    /**
     * Store an option set in the storage backend.
     *
     * @access private
     *
     * @return boolean    True on success or false on failure.
     */
    function store()
    {
        /* Make sure we have a valid database connection. */
        if (!$this->_connect()) {
            return false;
        }
        $user = $this->_user;
        $attrib = $this->_params['attribute'];
        $userdn = sprintf('%s=%s,%s', $this->_params['uid'], $user,$this->_params['basedn']);
        $store = $this->_options;
        foreach ($store as $a => $v) {
            $sa = $this->_mapAttributeToOption($a);
            if (is_array($v)) {
                foreach ($v as $av) {
                    $entry[$attrib][] = $sa . ' ' . $av;
                }
            } else {
                $entry[$attrib][] = $sa . ' ' . $v;
            }
        }

        return ldap_modify($this->_linkid, $userdn, $entry);
    }

    /**
     * Attempts to open a connection to the LDAP server.
     *
     * @access private
     *
     * @return boolean    True on success or false on failure.
     */
    function _connect()
    {
        if ($this->_connected) {
            return true;
        }

        $bindpass = Auth::getCredential('password');
        $user = $this->_user;
        $binddn = sprintf('%s=%s,%s', $this->_params['uid'], $user, $this->_params['basedn']);
        Horde::assertDriverConfig($this->_params, 'spamd_ldap',
                                  array('ldapserver', 'basedn', 'attribute', 'uid'),
                                  'SAM backend', 'backends.php', '$backends');

        /* try three times */
        for ($tries = 3; $tries > 0; $tries--) {
            $lc = ldap_connect($this->_params['ldapserver']);
            if ($lc) {
                $lb = ldap_bind($lc, $binddn, $bindpass);
                if ($lb) {
                    $this->_linkid = $lc;
                    $this->connected = true;
                    return true;
                } else  {
                    ldap_unbind($lc);
                }
            }
        }
        return false;
    }

    /**
     * Disconnect from the LDAP server and clean up the connection.
     *
     * @return boolean  True on success, false on failure.
     */
    function _disconnect()
    {
        if ($this->_connected) {
            $this->_connected = false;
            return ldap_unbind($this->_linkid);
        }

        return true;
    }

}
