/*
 *  JLib - Jacob's Library.
 *  Copyright (C) 2003, 2004  Juan Carlos Seijo Prez
 * 
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 * 
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 * 
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Juan Carlos Seijo Prez
 *  jacob@mainreactor.net
 */

/** Arbol simple recursivo.
 * <pre>
 * El rbol se organiza as:
 *
 *(parent de cur y root de todo el rbol)
 * |  (cur, iterador aqu)
 * |   |  
 * v   v  
 * O===O===O <-- (child)
 *     |
 *     O <-- (next)
 *     |
 *     O===O
 *     |   |
 *     O   O
 *     |
 *     O===O===O
 *     ^
 *     |
 *  (branch)
 * </pre>
 * 
 * @file    JTree.h.
 * @author  Juan Carlos Seijo Prez
 * @date    26/10/2003
 * @version 0.0.1 - 26/10/2003 - Primera versin.
 */

#ifndef _JTREE_INCLUDED
#define _JTREE_INCLUDED

#include <JLib/Util/JTypes.h>
#include <JLib/Util/JObject.h>

/** Arbol simple recursivo. Est pensado para no ser muy grande,
 * si se necesitan muchos niveles (ms de 5) de profundidad, se
 * debera usar otro tipo de rbol.
 */
template <typename T>
class JTree : public JObject
{
  T* data;                 /**< Datos del nodo */
  JTree<T>* prev;          /**< Nodo anterior del arbol */
  JTree<T>* next;          /**< Nodo siguiente del arbol */
  JTree<T>* child;         /**< Nodo hijo del arbol */
  JTree<T>* parent;        /**< Nodo padre del arbol */

public:

  /** Crea un nuevo nodo de arbol, que a su vez es un rbol.
   * @param  _data Datos de este nodo.
   * @param  _prev Nodo anterior.
   * @param  _next Nodo posterior.
   * @param  _parent Nodo padre.
   */
  JTree(T *_data = 0, JTree<T> *_prev = 0, JTree<T> *_next = 0, JTree<T> *_parent = 0)
    : data(_data), prev(_prev), next(_next), parent(_parent), child(0)
  {}

  /** Libera la memoria asociada.
   */
  virtual ~JTree()
  {
    JDELETE(child);
    JDELETE(next);
  }

  /** Iterador de rbol.
   */
  class Iterator
  {
  public:
    JTree<T>* node;        /**< Nodo actual del arbol. */
    
    /** Crea un iterador a partir de un rbol.
     */
    Iterator(JTree<T> &tree)
    {
      node = &tree;
    }

    /** Asigna los datos del nodo actual.
     * @param data Nuevos datos.
     */
    void Data(T* data)
    {
      if (node->data)
        delete node->data;

      node->data = data;
    }

    /** Devuelve los datos del nodo actual.
     * @return Datos del nodo actual.
     */
    T* Data()
    {
      return node->data;
    }

    /** Va al nodo siguiente.
     * @return <b>true</b> si el nodo exista, <b>false</b> si no.
     */
    bool Next()
    {
      if (node->next)
      {
        node = node->next;
        return true;
      }

      return false;
    }
    
    /** Va al nodo anterior.
     * @return <b>true</b> si el nodo exista, <b>false</b> si no.
     */
    bool Prev()
    {
      if (node->prev)
      {
        node = node->prev;
        return true;
      }

      return false;
    }
    
    /** Va al nodo padre.
     * @return <b>true</b> si el nodo exista, <b>false</b> si no.
     */
    bool Parent()
    {
      if (node->parent)
      {
        node = node->parent;
        return true;
      }

      return false;
    }
    
    /** Va al nodo hijo.
     * @return <b>true</b> si el nodo exista, <b>false</b> si no.
     */
    bool Child()
    {
      if (node->child)
      {
        node = node->child;
        return true;
      }

      return false;
    }

    /** Aade un nodo anterior o posterior.
     * @param  nodeData Datos asociados al nuevo nodo.
     * @param  after <b>true</b> para ponerlo despus del actual,
     * <b>false</b> para ponerlo antes.
     */
    void AddNode(T* nodeData, bool after = true)
    {
      if (after)
        node->next = new JTree<T>(nodeData, node, node->next, node->parent);
      else
        node->prev = new JTree<T>(nodeData, node->prev, node, node->parent);
    }

    /** Aade un nodo anterior o posterior y se sita en l.
     * @param  nodeData Datos asociados al nuevo nodo.
     * @param  after <b>true</b> para ponerlo despus del actual,
     * <b>false</b> para ponerlo antes.
     */
    void AddNodeGo(T* nodeData, bool after = true)
    {
      if (after)
      {
        node->next = new JTree<T>(nodeData, node, node->next, node->parent);
        node = node->next;
      }
      else
      {
        node->prev = new JTree<T>(nodeData, node->prev, node, node->parent);
        node = node->prev;
      }
    }
    
    /** Aade un nodo hijo, si no existe ya.
     * @param  nodeData Dtos del nuevo nodo hijo.
     */
    void AddBranch(T* nodeData)
    {
      if (!node->child)
        node->child = new JTree<T>(nodeData, 0, 0, node);
    }

    /** Aade un nodo hijo, si no existe ya, y se sita en l.
     * @param  nodeData Dtos del nuevo nodo hijo.
     */
    void AddBranchGo(T* nodeData)
    {
      if (!node->child)
      {
        node->child = new JTree<T>(nodeData, 0, 0, node);
        node = node->child;
      }
    }

    /** Borra un nodo y sus hijos
     */
    void RemoveNode()
    {
      if (node->next)
        node->next->prev = node->prev;

      if (node->prev)
        node->prev->next = node->next;

      JDELETE(node);
    }
    
    /** Va al primer nodo de la rama.
     */
    void FirstInBranch()
    {
      while (Prev());
    }

    /** Va al timo nodo de la rama.
     */
    void LastInBranch()
    {
      while (Next());
    }

    /** Va al primer nodo del rbol.
     */
    void FirstInTree()
    {
      while (Parent())
			{}

      FirstInBranch();
    }

    /** Va al primer nodo del rbol es lo mismo que FirstInTree().
     */
    void Root()
    {
      FirstInTree();
    }

    /** Determina si tiene hijos.
		 * @return <b>true</b> En caso de que los tenga, <b>false</b> si no.
     */
    bool HasChilds()
    {
      return (0 != node->child);
    }
  };

	/** Devuelve un nuevo iterador. El llamador debe borrar el objeto
	 * al terminar de usarlo.
	 * @return Iterador posicionado en el elemento raz.
	 */
	Iterator * NewIterator() {return new Iterator(*this);}
	
  friend class Iterator;
};

#endif  // _JTREE_INCLUDED
