/*
 *  JLib - Jacob's Library.
 *  Copyright (C) 2003, 2004  Juan Carlos Seijo Prez
 * 
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 * 
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 * 
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Juan Carlos Seijo Prez
 *  jacob@mainreactor.net
 */

///////////////////////////////////////////////////////////////////////////////
// @author: Juan Carlos Seijo Prez
// @date: 15/06/2003
// @description: Clase base de ficheros
///////////////////////////////////////////////////////////////////////////////

#include <JLib/Util/JObject.h>
#include <JLib/Util/JFile.h>

// Constructor
JFile::JFile(const s8 *filename) : file(0), buff(0), size(0), buffSize(0), name(0)
{
	if (filename != 0)
	{
		name = new s8 [strlen(filename) + 1];
		strcpy(name, filename);

		Refresh();
	}
}

// Destructor
JFile::~JFile()
{
  Close();    // Quiz no sea necesario, ya que puede hacerlo el programador
	JDELETE_ARRAY(name);
}

// Vuelve a consultar la informacin del fichero
s32 JFile::Refresh()
{
	if (name)
	{
		s32 ret;
		if (0 != (ret = stat(name, &statInfo)))
		{
			if (errno == ENOENT)
				exists = false;
		}
		else
		{
			exists = true;
		}

		return ret;
	}

	return -2;
}

// Abre el fichero. mode como en open() de C. Si no existe actualiza 
// el flag de existencia.
bool JFile::Open(const s8 *filename, const s8 *mode)
{
  Close();

  if (filename)
	{
		JDELETE_ARRAY(name);

		name = new s8 [strlen(filename) + 1];
		strcpy(name, filename);

		// Refresca la informacin de fichero 'statInfo'
		Refresh();
	}

	if (name != 0)
	{
		if (0 != (file = fopen(name, mode)))
		{
			return true;
		}
	}

  return false;
}
  
// Lee el fichero. Si no se especifica readsize o es cero, lo lee entero
u32 JFile::Read(u32 readSize)
{
  if (!readSize)
    readSize = statInfo.st_size;
  
  FreeBuffer();

  buff = new s8[readSize];

  return (buffSize = (u32)fread(buff, sizeof(s8), readSize, file));
}

// Carga todo el fichero. Equivalente a Open() + Read(0).
bool JFile::Load(const s8 *filename, const s8 *mode)
{
  if (Open(filename, mode))
	{
    return 0 < Read();
	}

  return false;
}

// Lee del fichero
u32 JFile::Read(void *buffer, u32 size)
{
  return (u32)fread(buffer, sizeof(s8), size, file);
}
  
// Lee del fichero al buffer dado y deja el puntero 
// en su posicin original
u32 JFile::Peep(void *buffer, u32 size)
{
  s32 p = Pos();
  u32 r = (u32)fread(buffer, size, 1, file);
  Pos(p);

  return r;
}


// Escribe en el fichero
u32 JFile::Write(void *buffer, u32 size)
{
  return (u32) fwrite(buffer, sizeof(s8), size, file);
}
  
// Cierra el fichero. Devuelve true si liber memoria, false si no lo hizo.
bool JFile::Close()
{
  if (!(file || buff))
    return false;

  if (file)
  {
    fclose(file);
    file = 0;
  }

  FreeBuffer();

  return true;
}

// Situa el puntero al inicio
void JFile::Rewind()
{
  rewind(file);
}

// Libera la memoria del buffer de forma segura
void JFile::FreeBuffer()
{
  if (buff)
  {
    delete[] buff;
    buff = 0;
  }
}

// Determina si existe el fichero
bool JFile::Exists(const s8 *name)
{
	if (name != 0)
  {
	  struct stat s;
	  if (0 != stat(name, &s))
	  {
		  if (errno == ENOENT)
		  {
			  return false;
		  }
	  }
		
		return true;
  }

  return false;
}

// Determina si existe el fichero
u32 JFile::Size(const s8 *name)
{
	if (name == 0)
		return false;

	struct stat s;
	if (0 == stat(name, &s))
	{
		return s.st_size;
	}
	
	return 0;
}

// Determina si el fichero dado es del tipo dado
bool JFile::IsOfType(const s8 *name, mode_t mode)
{
	if (name == 0)
		return false;

	struct stat s;
	if (0 == stat(name, &s))
	{
		return 0 != (s.st_mode & mode);
	}
	
	return false;
}

// Determina si existe el fichero
bool JFile::IsFile(const s8 *name)
{
	return IsOfType(name, S_IFREG);
}

// Determina si existe el fichero
bool JFile::IsDir(const s8 *name)
{
	return IsOfType(name, S_IFDIR);
}

// Determina si existe el fichero
bool JFile::IsLink(const s8 *name)
{
	return IsOfType(name, S_IFLNK);
}

// Determina si existe el fichero
bool JFile::IsCharDev(const s8 *name)
{
	return IsOfType(name, S_IFCHR);
}

// Determina si existe el fichero
bool JFile::IsBlockDev(const s8 *name)
{
	return IsOfType(name, S_IFBLK);
}
