// ****************************************************************************
// copyright (c) 2000-2004 Horst Knorr <hk_classes@knoda.org>  
// This file is part of the hk_classes library.
// This file may be distributed and/or modified under the terms of the
// GNU Library Public License version 2 as published by the Free Software
// Foundation and appearing in the file COPYING included in the
// packaging of this file.
// This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
// ****************************************************************************
#ifndef HK_CONNECTION
#define HK_CONNECTION
#include "hk_class.h"
#include <list>
#include <vector>
#include <algorithm>
using namespace std;
class hk_drivermanager;
class hk_connectionprivate;

class hk_database;
/**
 *@short connects to the SQL Server
 *@version $Revision: 1.43 $
 *@author Horst Knorr (hk_classes@knoda.org)
 *
 *connects to the SQL Server. Set @ref set_host,@ref set_user, @ref set_password ,
 *@ref set_tcp_port if necessary and then call @ref connect.

 */
class hk_connection:public hk_class
{
    friend class hk_database;
    friend class hk_drivermanager;
    public:
/**
 * When the connection will be destroyed, the database created with @ref new_database() and all
 * its children will be destroyed
 **/
        virtual     ~hk_connection();
/**
 * name of the host where the SQL Server is. Default is "localhost".
 */
        void    set_host(const hk_string& h);
        hk_string   host(void);
/**
 * user name for the SQL Server.
 */
        void    set_user(const hk_string& u);
        hk_string   user(void);
/**
 * password for the SQL Server
 */
        void    set_password(const hk_string& p);
        hk_string   password(void);
/**
 *  TCP Port number of the SQL Server. Default is 3306.
 */
        void        set_tcp_port(unsigned int t);
        unsigned int    tcp_port(void);
/**
 * returns the default tcp port of the the database driver;
 */
        virtual unsigned int    default_tcp_port(void) const;
/**
 * tries to connect to the SQL Server. If some information is missing (i.e. password)
 * a dialog will ask for these values. You can set a GUI dependend dialog with @ref set_passworddialog
 * @return true if connecting was successful, else false.
 */
        bool    connect(enum_interaction c=interactive);
/**
 * disconnects from the SQL Server
 * @return true if successful.
 */
        bool    disconnect(void);
/**
 *@return true  if @ref connect was successful
 */
        bool    is_connected(void);
/**
 *if connected it returns a hk_stringlist of all existing databases on the SQL Server.
 *The list will be created new every time you call this function.
 *So be careful. Never use i.e. for_each(dblist()->begin(),dblist()->end(),anyfunction)
 *Use instead:
 *list<hk_string>* mylist=dblist();
 *for_each(mylist->begin(),mylist->end(),anyfunction)

 */
        vector<hk_string>* dblist(void);
/**
 * if you want to use a database on the SQL Server create a hk_database element
 * with this class. For each connection you can define one database. If you define another
 * one the old one will be destroyed.

 */
        hk_database* new_database(const hk_string& name="");

	bool rename_database(const hk_string& originalname, const hk_string& new_name);
/**
 *see also @ref serversupports. Defines what abilities and fieldtypes a database driver supports
 */
        enum support_enum
        {
            SUPPORTS_AUTOINCCOLUMN=1,
            SUPPORTS_BOOLCOLUMN=2,
            SUPPORTS_DATECOLUMN=3,
            SUPPORTS_TIMECOLUMN=4,
            SUPPORTS_DATETIMECOLUMN=5,
            SUPPORTS_BINARYCOLUMN=6,
            SUPPORTS_MEMOCOLUMN=7,
            SUPPORTS_TIMESTAMPCOLUMN=8,

            SUPPORTS_SQL=100,
            SUPPORTS_TRANSACTIONS=101,
            SUPPORTS_REFERENTIALINTEGRITY=102,
            SUPPORTS_NEW_DATABASE=103,
            SUPPORTS_DELETE_DATABASE=104,
            SUPPORTS_NEW_TABLE=105,
            SUPPORTS_ALTER_TABLE=106,
            SUPPORTS_DELETE_TABLE=107,
            SUPPORTS_CREATE_INDEX=108,
            SUPPORTS_DELETE_INDEX=109,
            SUPPORTS_CHANGE_PASSWORD=110,
            SUPPORTS_RENAME_DATABASE=111,
            SUPPORTS_RENAME_TABLE=112
        };

/**
 *give information about the database server. See @ref hk_definitions.h for details
 */
        virtual bool server_supports(support_enum) const{return false;}
/**
 *see @ref server_needs
 */
        enum need_enum
        {

            NEEDS_NOTHING=0,
            NEEDS_HOST=1,
            NEEDS_USERNAME=2,
            NEEDS_PASSWORD=3,
            NEEDS_SQLDELIMITER=4,
            NEEDS_PORT=5,
            NEEDS_DATABASENAME=6,
	    NEEDS_BOOLEANEMULATION=7,//e.g. Mysql  doesn't have natively a boolean type, so tinyint is used
	    NEEDS_NULL_TERMINATED_SQL=8 //e.g. Postgres needs null terminated strings in PQExec
        };
/**
 *shows what information the database server needs (e.g. password).
 */
        virtual bool server_needs(need_enum) const {return false;}
/**
 * the passworddialog will be called from @ref connect if necessary. You can define a GUI dependend
 *dialog and set it with this function. What exactly is asked in the dialog is database driver dependend
 *(i.e. the Mysql driver needs user, password, host and tcp_port)
 */
        static  void    set_passworddialog(password_dialogtype*);
/**
 * tries to create a new database. If successful it returns true else false
 */
virtual        bool    create_database(const hk_string& dbase);
/**
 * tries to delete a database.
 *@param dbase name of an existing database
 *@return true if successful else flase
 */
virtual        bool    delete_database(const hk_string& dbase);
/**
 * checks whether a database exists
 */
        bool    database_exists(const hk_string& databasename);
/**
 * sets a new password at the SQL server.
 *@param newpassword the new password
 *@return true if successful else flase

 */
        bool set_newpassword(const hk_string& newpassword);
/**
 *shows a password dialog and sets the new password by calling @ref set_newpassword
 *If no passworddialog is set with @ref set_newpassworddialog the necessary information
 *will be queried at the standard output
 */
        void show_newpassworddialog(void);
/**
 * It is possible to define a newpassworddialog,
 * which will be shown when @ref set_newpassword is called
 */
        static void set_newpassworddialog(newpassword_dialogtype*);
/*
 *shows a password dialog
 *@return false if it has been cancelled else true
 */
        bool show_passworddialog(void);
/**
 *returns the name of the driver (e.g Mysql,Postgres etc.)
 */

        virtual hk_string drivername(void) const;
/**
*If the database does not have a native boolean type(like Mysql),
*but hk_classes should use one datatype to emulate, set this to true (default) else switch it off.
*
*/
	void set_booleanemulation(bool emulate);
        bool booleanemulation(void) const;
        hk_drivermanager* drivermanager(void);
/**
* copies whole databases
*@param fromdatabase the database that should be copied
*@param schema_and_data if true both the table schemas will be created and the data will be copied, if false only the table will be created
*@param copy_local_files if true queries,forms and reports will be also copied
*@param progressdialog if set this function will be regularly called to show the progress
*/
        bool copy_database(hk_database* fromdatabase,bool schema_and_data,bool copy_local_files,progress_dialogtype* progressdialog=NULL);
        hk_string last_servermessage(void){return p_lastservermessage;}
        hk_string databasepath(void);

    protected:
        hk_connection(hk_drivermanager* c);
        bool    p_connected;
        virtual bool    driver_specific_connect(void){return false;}
        virtual bool    driver_specific_disconnect(void){return false;}
	virtual bool    driver_specific_rename_database(const hk_string& oldname,const hk_string& newname){return false;}
        virtual  vector<hk_string>* driver_specific_dblist(void){return NULL;}
        virtual hk_database* driver_specific_new_database(void){return NULL;}
        virtual bool    driver_specific_new_password(const hk_string&){return false;}
        virtual bool    driver_specific_delete_database(const hk_string& dbase);

        void    db_remove(hk_database* d);
        hk_string sqldelimiter(void) ;
        void inform_databases_connect(void);
        void inform_databases_disconnect(void);
        vector<hk_string> p_databaselist;
/**
 *saves the tags <HOST><USER><TCP-PORT><SQLDELIMITER> with the equivalent values
 */
        virtual void savedata(ostream& s);
/**
 *load the tags <HOST><USER><TCP-PORT><SQLDELIMITER> with the equivalent values
 */
        virtual void loaddata(const hk_string& definition);

        void set_last_servermessage(const hk_string& m);
        hk_database* database(void) const {return p_database;}
    private:
        void load_configuration(void);
        void save_configuration(void);
        bool copy_local_files(hk_database* fromdb,hk_database* todb,objecttype f,progress_dialogtype* progressdialog);
        hk_string ask_dbname(void);
        void make_databasedir(const hk_string& dbname);
        void set_classespath(hk_string& p);
        void delete_databasedirectory(const hk_string& db);
        void delete_directory(const hk_string& d);

        hk_database* p_database;
        unsigned int p_debug_counter;
        static  password_dialogtype* p_passworddialog;
        static  newpassword_dialogtype* p_newpassworddialog;
        hk_drivermanager* p_drivermanager;
        hk_string   p_lastservermessage;
      hk_connectionprivate* p_private;
} ;
#endif
