/////////////////////////////////////////////////////////////////////////////
//	file		:	cmdline.h
//  copyright	:	(C) 2002-2003 by Benjamin Kaufmann
//  email		:	hume@c-plusplus.de
//	internet	:	http://bens.c-plusplus.info/
//   Added operator[](char),  hasOption(char ShortOptionName) -- André Simon, 17.03.04
//
//	a simple command-line parser.
//	The class parses short form (-v) and and long form (--verbose) options.
//	It also allows options with associated values
//	for example (-c 4, --count 4, --count=2).
//
//	This class is only intended for few and simple command line options.
//	If your program uses lots of options, go and use GNU getopt or GNU gengetopt
//
/////////////////////////////////////////////////////////////////////////////
//
/****************************************************************************
 *                                                                         	*
 *	This program is free software; you can redistribute it and/or modify  	*
 *	it under the terms of the GNU General Public License as published by  	*
 *	the Free Software Foundation; either version 2 of the License, or     	*
 *	(at your option) any later version.                                   	*
 *																			*
 * 	This program is distributed in the hope that it will be useful,			*
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of			*
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *	GNU General Public License for more details.                           	*
 *																			*
 ****************************************************************************/

#ifndef CMDLINE__H_INCLUDED
#define CMDLINE__H_INCLUDED

#include "includeme.h"
#include <stdexcept>
#include <string>
#include <map>
#include <cstdlib>

struct ArgumentParseException : public std::invalid_argument
{
	ArgumentParseException(const std::string& msg) : std::invalid_argument(msg) {}
};


class ValueProxy
{
	friend class CommandLineParser;
	public:
		int toInt()				{ return atoi(value_.c_str()); }
		std::string toString()	{ return value_; }
		bool toBool()			{ return value_ == "true";}
		operator std::string()	{ return toString(); }
		operator int()			{ return toInt(); }
		operator bool()			{ return toBool();}
	private:
		const std::string& value_;
		ValueProxy(const std::string& str) : value_(str) {}
};

class CommandLineParser
{

	public:
	CommandLineParser();
		~CommandLineParser();
		void addIntOption	(char shortName, const std::string& longName,
							 const std::string& defVal = "-1");

		void addSwitchOption(char shortName, const std::string& longName);

		void addStringOption(char shortName, const std::string& longName,
							 const std::string& defVal = "");

		void parse(int& argc, char** argv);

		const std::string& getAt(const std::string& optionName) const;
		ValueProxy operator[] (const std::string& optionName) const;
		ValueProxy operator[] (char ShortOptionName) const;

		bool hasOption(const std::string& optionName) const;
		bool hasOption(char ShortOptionName) const;

	    unsigned int getArgCount();
	private:
	  
	    unsigned int argCount; 
		// base class representing an command line option
		class Option
		{
			public:
				Option (char shortName, const std::string& longName,
						const std::string& defVal, bool expectsValue);
				virtual ~Option() {}
				const std::string& getShortName() const {return shortName_;}
				const std::string& getLongName() const {return longName_;}
				const std::string& getDefValue() const {return defaultValue_;}
				const std::string& getValue() const {return value_;}
				virtual std::string& setValue(std::string& );


				bool expectsValue() const {return expectsValue_;}

			private:
				Option (const Option&);
				Option& operator=(const Option&);
				bool		expectsValue_;
				std::string shortName_;
				std::string longName_;
				std::string defaultValue_;
				std::string value_;
		};

		class IntOption : public Option
		{
			public:
				IntOption ( char shortName, const std::string& longName,
							const std::string& defVal);
				~IntOption() {}
				std::string& setValue(std::string& );
		};

		typedef std::map<std::string, Option*> OptionMap;
		OptionMap	options_;

		void addOption(Option* pOption);


};

#endif

