/***************************************************************************
                          main.cpp  -  description
                             -------------------
    begin                : Die Apr 23 22:16:35 CEST 2002
    copyright            : (C) 2002-2004 by André Simon
    email                : andre.simon1@gmx.de


   Highlight is a universal source code to HTML converter. Syntax highlighting
   is formatted by Cascading Style Sheets. It's possible to easily enhance
   highlight's parsing database.

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "main.h"
 
using namespace std;

void HighlightApp::printVersionInfo()
{
  cout << "\n highlight version " 
       << HIGHLIGHT_VERSION
#ifdef USE_PORTABLE_CMDLINE_PARSER
       << "\n (Build with USE_PORTABLE_CMDLINE_PARSER flag set.)"
#endif      
       << "\n Copyright (C) 2002-2004 Andre Simon <andre.simon1@gmx.de>"
       << "\n\n Artistic Style Classes (1.15.3)"
       << "\n Copyright (C) 1998-2002 Tal Davidson <davidsont@bigfoot.com>"
       << "\n\n DirStream Class"
#ifdef USE_PORTABLE_CMDLINE_PARSER
       << ", CommandLineParser Class"
#endif
       << "\n Copyright (C) 2002 Benjamin Kaufmann <hume@c-plusplus.de>"
       << "\n\n This software is released under the terms of the GNU General "
       << "Public License."
       << "\n For more information about these matters, see the file named "
       << "COPYING.\n"
       << endl;
}

void HighlightApp::printBadInstallationInfo()
{
  cerr << "highlight: Data directory not found. Bad installation or wrong "
       << "data-dir parameter."
       << "\n\nCopy the highlight files (*.lang, *.style) into one of the "
       << "directories listed in INSTALL.\n"       
       << "You may also set the data directory with --data-dir and "
       << "--add-data-dir.\n";
}

bool HighlightApp::listInstalledFiles(bool themes)
{
  // Liste der Eingabedateien
  vector <string> inFileList;
  unsigned int suffixLength=(themes)?6:5;
  string searchDir = (themes)?dataDir.getThemeDir()+
                     "*.style":dataDir.getLangDefDir()+"*.lang";
  // Liste der Eingabedateien fuellen:
  bool directoryOK = DirectoryReader::getEntries (inFileList, searchDir);
  if (!directoryOK)
    {
     cerr << "highlight: Could not access directory "
          <<  searchDir
          << ", aborted.\n";
      return false;
    }

  cout << "\n  Installed "
       << ((themes)? "themes":"language definitions ")
       << "(located in "
       << ((themes)?dataDir.getThemeDir():dataDir.getLangDefDir())
       << ") :\n"
       << endl;

  string  temp;
  for (unsigned int i=0;i< inFileList.size(); i++)
    {
      if (themes)
       temp = (inFileList[i]).substr(dataDir.getThemeDir().length());
      else
       temp = (inFileList[i]).substr(dataDir.getLangDefDir().length());
      cout << "  "<<temp.substr(0, temp.length()- suffixLength) << endl;
    }
   cout  <<"\n  Use name of the desired "
         << ((themes)?"theme":"language")
         << " with the -"
         << ((themes)?"s":"S")
         << " option.\n" << endl;
   return true;
}

void HighlightApp::printParseError(highlight::ParseError err,
                                   const string &inFile,
                                   const string &outFile){
  switch (err){
    case highlight::BAD_OUTPUT:
          cerr << "highlight: Could not write "
               << outFile
               << "."
               << endl;
          break;
     case highlight::BAD_INPUT:
           cerr << "highlight: Input file "
                << inFile
                << " not found."
                << endl;
          break;     
     default:; 
  }
 }
 
 void HighlightApp::printDebugInfo( highlight::LanguageDefinition &lang, 
                      const string & language)
{    
   cout << "\nLANGUAGE: " 
        << language
        << "\n\nSYMBOLS: "  
        << lang.getSymbolString();   
   cout << "\n\nKEYWORDS: "; 
   lang.printKeys();      
   cout << "\n\nTYPES:    ";      
   lang.printTypes(); 
   cout <<"\n\n"; 
}

string HighlightApp::getFileSuffix(const string &fileName) 
  {
    unsigned int ptPos=fileName.rfind(".");
    return (ptPos == string::npos) ? 
            "" : fileName.substr(ptPos+1, fileName.length());  
}


bool HighlightApp::loadFileExtensions(){
  ConfigurationReader extensionsConfig(dataDir.getDir() + "extensions.conf", 
                                       true);  
  if (extensionsConfig.found())
  {
    stringstream values;
    string paramName, paramVal;
    for (unsigned int i=0;i<extensionsConfig.getParameterNames().size();i++)    
     {
       paramName = extensionsConfig.getParameterNames()[i];
       values.str(extensionsConfig.getParameter(paramName)) ;        
       while (values >> paramVal) {         
          extensions[ paramVal ] = paramName;     
       }
       values.clear();
      }
      return true;
    }
  return false;
}


/** gibt bei mehreren gebraeuchlichen Dateiendungen den passenden Namen der
Sprachdefinitionsdatei zurueck */
string HighlightApp::getFileType(const string& suffix)
{
  return (extensions.count(suffix)) ? extensions[suffix] : suffix ;
}

int HighlightApp::getNumDigits(int i){
  int res=0;
  while (i){
    i/=10;
    ++res; 
  }
  return res;
}

void HighlightApp::printProgressBar(int count, int index){
  if (!count) return;
  int p=100*index / count;
  int numProgressItems=p/10;
  cout << "\r[";
  for (int i=0;i<10;i++){
    cout <<((i<numProgressItems)?"#":" ");
  }  
  cout<< "] " <<setw(3)<<p<<"%, "<<index << " / " << count << "  " <<flush;
  if (p==100) {
    cout << endl;
  }
}

void HighlightApp::printCurrentAction(const string&outfilePath, 
                                      int count, int index, int countWidth){
  cout << "Writing file " 
       << setw(countWidth)<< index
       << " of "
       << count
       << ": "              
       << outfilePath 
       << "\n";      
}

int HighlightApp::run(int argc, char**argv){

  //get command line options
  CmdLineOptions options(argc, argv);
 
  // set data directory path, where /langDefs and /themes reside
  string highlightRootDir;
#ifdef _WIN32
  highlightRootDir = argv[0];
  string::size_type pos = highlightRootDir.find_last_of(PATH_SEPARATOR_CHAR);
  highlightRootDir = highlightRootDir.substr(0, pos+1);    
#endif

  // Highlight-Verzeichnis bestimmen
  if (! dataDir.searchDataDir((options.dataDirGiven())?
                                options.getDataDir(): highlightRootDir))
    {
      printBadInstallationInfo();	  
      return EXIT_FAILURE;
    }

  if (options.additionalDataDirGiven()){
     dataDir.setAdditionalDataDir(options.getAdditionalDataDir());
   }
   
  loadFileExtensions();
  
  if (options.printVersion())
    {
      printVersionInfo();
      return EXIT_SUCCESS;
    }

  if  (options.printHelp())
    {
      Help::printHelp(dataDir.getHelpMsgDir() + options.getHelpLang());
      return EXIT_SUCCESS;
    }

  if (options.showThemes() || options.showLangdefs())
    {      
      return listInstalledFiles(options.showThemes())?
               EXIT_SUCCESS:EXIT_FAILURE;
    }
      
  highlight::CodeParser *parser = NULL;
  
  // list of input files
  const  vector <string> inFileList=options.getInputFileNames();
  string stylePath=dataDir.searchForTheme(options.getStyleName());
  
  // get an appropriate code parser instance
  if (options.texFormat())
    {
      parser = new highlight::TexCode (stylePath,
                                       options.printLineNumbers(),
                                       options.fragmentOutput(),
                                       options.getNumberSpaces(),
                                       options.getWrappingStyle());
    }
  else if (options.latexFormat())
    {
      parser = new highlight::LatexCode( stylePath,
                                        options.printLineNumbers(),
                                        options.fragmentOutput(),
                                        options.getNumberSpaces(),
                                        options.getWrappingStyle(),
                                        options.replaceQuotes());
    }
  else if (options.rtfFormat())
    {
      parser = new highlight::RtfCode (stylePath,
                                       options.printLineNumbers(),
                                       options.fragmentOutput(),
                                       options.getNumberSpaces(),
                                       options.getWrappingStyle());
    }
  else if (options.xslfoFormat())
    {
      parser = new highlight::XslFoCode(stylePath,
                                        options.printLineNumbers(),
                                        options.fragmentOutput(),                             
                                        options.getNumberSpaces(),
                                        options.getWrappingStyle(),
                                        options.fopCompatible());
    }
  else
    {
      parser = new highlight::HtmlCode(stylePath,
                                       options.printLineNumbers(),
                                       options.fragmentOutput(),
                                       options.getNumberSpaces(),
                                       options.getWrappingStyle(),
                                       options.getCssOutFilename(),
                                       options.includeCssDefinition(),
                                       options.attachLineAnchors(),
                                       options.xhtmlFormat());
      // print external css file
      if (!options.includeCssDefinition()){
         ((highlight::HtmlCode*)parser)->printCSSFile(options.getCssInFilename(),
                                           options.getOutDirectory() + 
                                           options.getCssOutFilename());
      }      
      if (options.printIndexFile()){
         ((highlight::HtmlCode*)parser) -> printIndexFile(inFileList, 
                                             options.getOutDirectory());
      }
    }
    
  string outfilePath;
   
  enum highlight::ParseError error;
  unsigned int fileCount=inFileList.size(), 
               numErrors=0,
               fileCountWidth=getNumDigits(fileCount);
  string suffix;
  bool fatalError=!parser->styleValid();
  unsigned int i=0;

  while (i< fileCount && !fatalError)
  {     
    suffix = (options.syntaxGiven())?options.getLanguage() :                   
                                     getFileType(getFileSuffix(inFileList[i]));
    fatalError=!initializeParser(parser, options, suffix);        

    if (fatalError) break;                 
    if (options.enableBatchMode()){
      string::size_type pos=(inFileList[i]).find_last_of(PATH_SEPARATOR_CHAR);          
      outfilePath = options.getOutDirectory();      
      outfilePath += inFileList[i].substr(pos+1);
      outfilePath += options.getOutFileSuffix();         

      if (!options.quietMode()) {
        if (options.printProgress()){
          printProgressBar(fileCount, i+1);
        } else {
          printCurrentAction(outfilePath, fileCount, i+1, fileCountWidth);
        }
      }
    }
     else {         
       outfilePath = options.getSingleOutFilename();         
    }
    error = parser->printOutput(inFileList[i], outfilePath);
    if (error != highlight::PARSE_OK){
      printParseError(error, inFileList[i], outfilePath);
      ++numErrors;
    }    
    ++i;   
    parser->reset();    
  }
  
  if (numErrors && fileCount > 1 && !fatalError){    
    cerr << "highlight: Was not able to convert "
         << numErrors
         << " file" << ((numErrors>1)?"s":"")         
         << "."
         << endl;
  }
  delete parser;  
  return EXIT_SUCCESS;  
}

bool HighlightApp::initializeParser(highlight::CodeParser* parser,
                                    CmdLineOptions &options,
                                    const string & suffix){
                        
  if (suffix==".lang")   {
      cerr<<"highlight: Undefined language definition. Use -S option."<<endl;
      return false;      
   }
  string langDefPath=dataDir.searchForLangDef( suffix+".lang");
  highlight::LoadResult loadRes= parser->setLanguage(langDefPath);
  if (options.printDebugInfo() && loadRes==highlight::LOAD_NEW){    
      cout << "\nLoading new language definition: "
           << langDefPath
           <<endl;
      printDebugInfo(parser->getLanguage(), suffix);
  }

  if (loadRes==highlight::LOAD_FAILED){
    cerr << "highlight: Unknown source file extension \""
         << suffix
         << "\".\n"; 
    return false;
  }
            
  if (options.formattingEnabled()){  
    bool formatok= parser->enableReformatting(options.getFormatStyle(), suffix);
    if (!formatok) {
      cerr << "highlight: Code reformatting only enabled with "
           << "C, C++, C# or Java input files.\n";
    }
  }               
  return true;                            
}

int main(int argc, char **argv)
{
  HighlightApp app;
  return app.run(argc, argv);
}
