/***************************************************************************
                          htmlcode.cpp  -  description
                             -------------------
    begin                : Wed Nov 28 2001
    copyright            : (C) 2001 by Andr�é Simon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "htmlcode.h"

using namespace std;

namespace highlight {

// CSS-Kommentar
const string HtmlCode :: cssComment =
  "\t/* Used abbreviations: */\n"
  "\t/* key:  Keywords */\n"
  "\t/* typ:  Types and type modifiers */\n"
  "\t/* str:  Strings */\n"
  "\t/* num:  Numbers */\n"
  "\t/* com:  Comments */\n"
  "\t/* esc:  Escape characters */\n"
  "\t/* dir:  Directive lines */\n"
  "\t/* dstr: Strings within directive lines */\n"
  "\t/* sym:  Symbols */\n"
  "\t/* line: Line numbers */\n\n";


HtmlCode::HtmlCode(void)
{}

string  HtmlCode::formatStyleAttributes(const string & elemName, 
                                        const ElementStyle & elem)
{
  ostringstream s;
  s  << "\t."<<elemName<<"\t{ color:#"
     << (elem.getColour().getHexRedValue())
     << (elem.getColour().getHexGreenValue())
     << (elem.getColour().getHexBlueValue()  )
     << ( elem.isBold() ?"; font-weight:bold" :"" )
     << ( elem.isItalic() ?"; font-style:italic" :"" )
     << ( elem.isUnderline() ?"; text-decoration:underline" :"" )
     << "; }\n" ;
  return  s.str();
}

string  HtmlCode::getOpenTag(const string& styleName ){
  return "<span class=\""+styleName+"\">";
}

HtmlCode::HtmlCode (
  const string &cssStyle,  
  bool printLineNumbers, 
  bool fragment,  
  int numSpaces, 
  WrapMode wrapStyle,
  const string &cssFilename,
  bool includeCssDef, 
  bool withAnchors, 
  bool xhtmlFormat)
    : CodeParser( printLineNumbers, cssStyle, numSpaces, wrapStyle, fragment),
    attachAnchors(withAnchors),
    cssFile(cssFilename),
    includeCss(includeCssDef),
    xhtmlOutput(xhtmlFormat)
{
  for (int i=1;i<NUMBER_OF_STYLES; i++)
    {
      styleTagClose[i] = "</span>";
    }
  //styleTagClose[STANDARD] = "";
  
  //styleTagOpen[STANDARD] = "";
  styleTagOpen[KEYWORD] = getOpenTag("key");
  styleTagOpen[TYPE] = getOpenTag("typ");
  styleTagOpen[STRING] = getOpenTag("str");
  styleTagOpen[NUMBER] = getOpenTag("num");
  styleTagOpen[SINGLE_LINE_COMMENT] = getOpenTag("com");
  styleTagOpen[MULTI_LINE_COMMENT_BEGIN] = getOpenTag("com");
  styleTagOpen[ESC_CHAR] = getOpenTag("esc");
  styleTagOpen[DIRECTIVE_LINE] = getOpenTag("dir");
  styleTagOpen[DIRECTIVE_STRING] = getOpenTag("dstr");
  styleTagOpen[LINENUMBER] = getOpenTag("line");
  styleTagOpen[SYMBOL] = getOpenTag("sym");
  newLineTag = "\n";
  spacer = " ";  
}

string HtmlCode::getStyleDefinition()
{
  if (styleDefinitionCache.empty()){  
    ostringstream os;
    os << "\tpre\t{ color:#"
       << (docStyle.getDefaultStyle().getColour().getHexRedValue())
       << (docStyle.getDefaultStyle().getColour().getHexGreenValue())
       << (docStyle.getDefaultStyle().getColour().getHexBlueValue()  )
       << "; background-color:#"
       << (docStyle.getBgColour().getHexRedValue())
       << (docStyle.getBgColour().getHexGreenValue())
       << (docStyle.getBgColour().getHexBlueValue()  )
       << "; font-size:"
       << docStyle.getFontSize()
       << "pt; font-family:Courier;}\n";
    os << formatStyleAttributes("key", docStyle.getKeywordStyle())
       << formatStyleAttributes("num", docStyle.getNumberStyle())
       << formatStyleAttributes("typ", docStyle.getTypeStyle())
       << formatStyleAttributes("esc", docStyle.getEscapeCharStyle())
       << formatStyleAttributes("str", docStyle.getStringStyle())
       << formatStyleAttributes("dstr", docStyle.getDirectiveStringStyle())
       << formatStyleAttributes("com", docStyle.getCommentStyle())
       << formatStyleAttributes("dir", docStyle.getDirectiveStyle())
       << formatStyleAttributes("sym", docStyle.getSymbolStyle())
       << formatStyleAttributes("line", docStyle.getLineStyle());
    styleDefinitionCache=os.str();
  }  
  return styleDefinitionCache;
}

string HtmlCode::getHeader(const string &title)
{
  if (headerCache.empty()){
  ostringstream osPart1, osPart2;
  if (xhtmlOutput)   //XHTML Ausgabe:
    {
      osPart1   << "<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n"
           << "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\""
           << "  \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n"
           << "<html xmlns=\"http://www.w3.org/1999/xhtml\">" 
           << endl;
    }
  else     //HTML Ausgabe:
    {
      osPart1  << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">"
           << "\n<html>" 
           << endl;
    }

  osPart1   <<"<head>\n\t<title>" ;

  osPart2 << "</title>" << endl;

  if (langInfo.getSyntaxHighlight())
    {
      if (includeCss)    //CSS-Definition in HTML-<head> einfuegen
        {
          osPart2  << "\t<style type=\"text/css\">\n";
          if (!xhtmlOutput){
            osPart2 << "\t<!--\n";
          }
          //XHTML-Deklaration von PCDATA im Scriptbereich:
          osPart2 << ((xhtmlOutput) ? "\t<![CDATA[\n" : "")
             << cssComment;
          osPart2 << getStyleDefinition();
          osPart2 << ((xhtmlOutput) ? "\t]]>\n" : "\t//-->\n");
          osPart2 <<"\t</style>" << endl;
        }
      else  //Referenz auf CSS-Datei einfuegen
        {
          osPart2 << "\t<link rel=\"stylesheet\" type=\"text/css\" href=\""
             << cssFile
             << "\""
             << ((xhtmlOutput) ? "/" : "")
             << ">"
             << endl;         
        }
    }
  osPart2 << "</head>\n<body>\n<pre>";
    headerCache=osPart1.str();
    headerCache2=osPart2.str();
  }
  return headerCache + ((title.empty())?"Source file":title) + headerCache2;
}

string HtmlCode::getFooter()
{
  if (footerCache.empty()){
    ostringstream os;
    os  << "</pre>\n</body>\n</html>\n<!--"
        << ((xhtmlOutput) ? "X" : "")
        << "HTML generated by highlight "
        << HIGHLIGHT_VERSION
        << ", "
        << HIGHLIGHT_URL
        << "-->"
        << endl;
    footerCache=os.str();
  }
  return footerCache;
}


void HtmlCode::printBody()
{
  processStandardState();
}

bool HtmlCode::printCSSFile(const string &cssInFileName, 
                            const string &cssOutFileName)
{
  if (!includeCss && langInfo.getSyntaxHighlight())
    {
      ofstream cssFile(cssOutFileName.c_str());
      if (cssFile)
        {
          cssFile << "/* CSS definition file generated by highlight "
                  << HIGHLIGHT_VERSION
                  << ", "
                  << HIGHLIGHT_URL
                  << " */\n\n";

          if (!cssInFileName.empty())
            {
              ifstream cssUserDef(cssInFileName.c_str());
              if (cssUserDef)
                {
                  cssFile << "/* Content of "<<cssInFileName<<": */\n";
                  string line;
                  while (getline(cssUserDef, line))
                    {
                      cssFile << line << "\n";                      
                    }
                  cssUserDef.close();
                }
              else
                {
                  cssFile << "/* ERROR: Could not include "
                          << cssInFileName
                          << ".*/\n";
                }
            }
          cssFile << "\n\n/* Highlighting theme definition: */\n\n"          
                  << cssComment
                  << getStyleDefinition();
          cssFile << endl;
          cssFile.close();
        }
      else
        {
          cerr << "highlight: Could not write "               
               << cssOutFileName<<".\n";
          return false;
        }
    }
    return true;
}

string HtmlCode::maskCharacter(unsigned char c)
{
  switch (c) {
    case '<' :
      return "&lt;";
      break;
    case '>' :
      return "&gt;";
      break;
    case '&' :
      return "&amp;";
      break;
    case '\"' :
      return "&quot;";
      break;
    case '@' :
      return "&#64;"; 
      break;      
    case AUML_LC :  
      return "&auml;";  
      break;
    case OUML_LC :
      return "&ouml;";
      break;
    case UUML_LC :
      return "&uuml;";
      break;
    case AUML_UC :
      return "&Auml;";
      break;
    case OUML_UC :
      return "&Ouml;";
      break;
    case UUML_UC :
      return "&Uuml;";
      break;
    case SZLIG :
      return "&szlig;";
      break;
      
    case AACUTE_LC :
      return "&aacute;";
      break;
    case EACUTE_LC:
      return "&eacute;";
      break;    
    case OACUTE_LC:
      return "&oacute;";
      break;
    case UACUTE_LC:
      return "&uacute;";
      break;
            
    case AACUTE_UC:
      return "&Aacute;";
      break;
    case EACUTE_UC:
      return "&Eacute;";
      break;    
    case OACUTE_UC:
      return "&Oacute;";
      break;
    case UACUTE_UC :
      return "&Uacute;"; 
      break;
    
    case AGRAVE_LC:
      return "&agrave;";
      break;
    case EGRAVE_LC:
      return "&egrave;";
      break;    
    case OGRAVE_LC:
      return "&ograve;";
      break;
    case UGRAVE_LC:
      return "&ugrave;";
      break;
    
    case AGRAVE_UC:
      return "&Agrave;";
      break;
    case EGRAVE_UC:
      return "&Egrave;";
      break;     
    case OGRAVE_UC:
      return "&Ograve;";
      break;
    case UGRAVE_UC:
      return "&Ugrave;";
      break; 
    
    // skip  first byte of multibyte characters
    #ifndef _WIN32
    case 195:
      return string("");
      break;  
    #endif  
  
    default :
      string m;
      m += c;
      return m;
    }
}

/** fuegt am Anfang der neuen Zeile eine fortlaufende Nummer ein
Parameter: State envokingState
           Zustand des Aufrufers
*/ 
void HtmlCode::insertLineNumber (bool insertNewLine)
{
  if (insertNewLine){
    *out << getNewLine();          
  }
  if (showLineNumbers)
    {
      // Falls Zustand != Standardzustand, entsprechende Tags schliessen...
      if (currentState != STANDARD)
        {
          *out << styleTagClose[currentState];
        }      
      if (attachAnchors)
        {
          *out << "<a "
               << ((xhtmlOutput)?"id": "name")
               << "=\"l_" 
	           << lineNumber
	           << "\">";
        }
      ostringstream os;      
      os <<setw(LINE_NUMBER_WIDTH)<<right<< lineNumber;
      *out << styleTagOpen[LINENUMBER] 
           << os.str()  
           << spacer 
	       << styleTagClose[LINENUMBER];
      if (attachAnchors)
        {
          *out << "</a>";
        }

      // ... und wieder oeffnen
      if (currentState != STANDARD)
        {
          *out << styleTagOpen[currentState];
        }
    }
}

/** schreibt Index-Datei ins Ausgabeverzeichnis */
bool HtmlCode::printIndexFile(const 
#ifdef USE_WX_LIBRARY
wxListBox*
#else
vector<string> &
#endif
fileList, const string &outPath ){
  string suffix = (xhtmlOutput)? ".xhtml" : ".html";
  string outFilePath = outPath + "index" + suffix;
 
  ofstream indexfile(outFilePath.c_str());
      
  if (!indexfile.fail()){
     string outFileName;     
     indexfile << getHeader("Source Index");
     indexfile << "<h1> Source Index</h1>\n"
               << ((xhtmlOutput)?"<hr />":"<hr>")
               <<  "<ul>"<<endl;

     #ifdef USE_WX_LIBRARY     
     for (int i=0; i < fileList->GetCount(); i++){
         outFileName = string(fileList->GetString(i).AfterLast(PATH_SEPARATOR_CHAR).c_str());
     #else
     string::size_type pos;
     for (unsigned int i=0; i < fileList.size();  i++){           
         pos=(fileList[i]).find_last_of(PATH_SEPARATOR_CHAR);     
         outFileName =(fileList[i]).substr(pos+1);
     #endif
         indexfile << "<li><a href=\""<< outFileName<<suffix<<"\">";
         indexfile << outFileName<<suffix <<"</a></li>"<<endl;
     }
     indexfile << "</ul>"
          << ((xhtmlOutput)?"<hr />":"<hr>")
          << ((xhtmlOutput)?"<br />":"<br>" )
          << "<small>Generated by highlight "
          << HIGHLIGHT_VERSION
          << ", "
          << "<a href=\""<<HIGHLIGHT_URL << "\" target=\"new\">"
          << HIGHLIGHT_URL << "</a></small>"
          << endl;
     indexfile << getFooter();       
  } else {
    cerr << "highlight: Could not write "               
         << outFilePath<<".\n";
    return false;
  }
  return true;
}

}
