-- Very Simple Expression (SExps, but even more simple!)
module VSExp
  ( VSExp(..)
  , CondType(..)
  , NodeExp(..)
  , nodeExpForNode
  , findLambdas
  , removeVerticalCycles
  , limitNExpDepth
  , compressClosures
  , unevalVSExp
  , partEvalVSExp
  , fullEvalVSExp
  , cutoffEvalVSExp
  , condEvalVSExp
  , cutoffFindMinOffset
  , cutoffFindNodes
  , condFindNodes
  , hideFunction
  , limitDepth
  , removeHorizontalCycles
  , vsExp2sExp
  ) where

import LowLevel hiding    ( nil )
import qualified LowLevel ( nil )
import List               ( elemIndex )
import SExp               ( SExp (..), SFixity(..), transFixity, QName(..)
                          , showQN )

data CondType = IfCond | CaseCond | GuardCond --deriving Show

-- Represents an ART file in it's raw form
data NodeExp
  = NExpApp        FileNode NodeExp [NodeExp] NodeExp
  | NExpIdentifier FileNode QName SFixity
  | NExpLambda     FileNode (Maybe String) [NodeExp]
  | NExpLiteral    FileNode String
  | NExpCond       FileNode CondType NodeExp NodeExp
  | NExpFieldExp   FileNode NodeExp [(String, NodeExp)]
  | NExpProjection FileNode NodeExp
  | NExpForward    FileNode NodeExp
  | NExpHidden     FileNode NodeExp
  | NExpInterrupt
  | NExpUneval
  | NExpBottom
  | NExpNotShown
  | NExpVerticalCycle
--  deriving Show

instance Eq NodeExp where
  (==) exp1 exp2
    = if isSimpleNodeType exp1 || isSimpleNodeType exp2
        then simpleComp exp1 exp2
        else (getNode exp1) == (getNode exp2)

instance Ord NodeExp where
  (>) exp1 exp2
    = if isSimpleNodeType exp1 then False
      else if isSimpleNodeType exp2 then True
      else (getNode exp1) > (getNode exp2)

getNode :: NodeExp -> FileNode
getNode (NExpApp node _ _ _) = node
getNode (NExpIdentifier node _ _) = node
getNode (NExpLambda node _ _) = node
getNode (NExpLiteral node _) = node
getNode (NExpCond node _ _ _) = node
getNode (NExpFieldExp node _ _) = node
getNode (NExpProjection node _) = node
getNode (NExpForward node _) = node
getNode (NExpHidden node _) = node
getNode _ = LowLevel.nil

isSimpleNodeType :: NodeExp -> Bool
isSimpleNodeType NExpInterrupt = True
isSimpleNodeType NExpBottom = True
isSimpleNodeType NExpNotShown = True
isSimpleNodeType NExpVerticalCycle = True
isSimpleNodeType _ = False

simpleComp :: NodeExp -> NodeExp -> Bool
simpleComp NExpInterrupt NExpInterrupt = True
simpleComp NExpBottom NExpBottom = True
simpleComp NExpNotShown NExpNotShown = True
simpleComp NExpVerticalCycle NExpVerticalCycle = True
simpleComp _ _ = False

-- All VSExps have the file node at which they are found and some other
-- arguments specific to the expression type.
data VSExp
    -- Applications have the expression being applied, and a list of items
    -- to which it is being applied.
  = VSApp        FileNode VSExp [VSExp]
    -- Named items, may be identifiers or literal values.
  | VSIdentifier FileNode QName SFixity
    -- Lambda expressions store the applications of themselves.
  | VSLambda     FileNode (Maybe String) [([VSExp], VSExp)]
  | VSLiteral    FileNode String
    -- All kinds of conditionals
    -- Conditional expressions have their condition
  | VSCond       FileNode CondType VSExp VSExp
    -- Field updates and Field definitions have the expression being
    -- updated, and pairs representing bindings.
  | VSFieldExp   FileNode VSExp [(String, VSExp)]
  | VSInterrupt
  | VSBottom
  | VSUneval
  | VSNotShown
  | VSVerticalCycle
  | VSHorizontalCycle
--  deriving Show -- For debugging

getVSNode :: VSExp -> FileNode
getVSNode (VSApp node _ _) = node
getVSNode (VSIdentifier node _ _) = node
getVSNode (VSLambda node _ _) = node
getVSNode (VSLiteral node _) = node
getVSNode (VSCond node _ _ _) = node
getVSNode (VSFieldExp node _ _) = node

nodeExpForNode :: FileNode -> NodeExp
nodeExpForNode node =
  if node == LowLevel.unevaluated
    then NExpUneval
    else
      case simpleNodeType node of
        NodeAtom ->
          NExpIdentifier node
                         (Qualified (getAtomMod node) (getAtom node))
                         (transFixity (getAtomFixity node))
        NodeApplication ->
          NExpApp node funExp argExps resExp
          where
            funExp = nodeExpForNode function
            argExps = map nodeExpForNode args
            resExp = nodeExpForNode (peekResult node)
            (function:args) = peekSubExprs node
        NodeBasicValue ->
          NExpLiteral node (getValue node)
        NodeCAF ->
          NExpIdentifier node
                         (Qualified (getValueMod node) (getValue node))
                         (transFixity (getFixity node))
        NodeConditional ->
          let
            condType = case nodeType node of
              ExpIf -> IfCond
              ExpCase -> CaseCond
              ExpGuard -> GuardCond
          in
            NExpCond node
                     condType
                     (nodeExpForNode (head (peekSubExprs node)))
                     (nodeExpForNode (peekResult node))
        NodeIdentifier->
          NExpIdentifier node
                         (Qualified mod id)
                         (grabFixity mod id)
          where
            mod = getValueMod node
            id = getValue node
            grabFixity m i =
              case i of
                "."   | m == "Prelude" -> SAssoc 9 i
                "++"  | m == "Prelude" -> SAssoc 5 i
                "&&"  | m == "Prelude" -> SAssoc 3 i
                "||"  | m == "Prelude" -> SAssoc 2 i
                "*"   | m == "Prelude" -> SAssoc 7 i
                "+"   | m == "Prelude" -> SAssoc 6 i
                ">>"  | m == "Prelude" -> SAssoc 1 i
                ">>=" | m == "Prelude" -> SAssoc 1 i
                _                      -> transFixity (getFixity node)
        NodeSpecial ->
          case nodeType node of
            ExpProjection ->
              NExpProjection node
                             (nodeExpForNode (peekResult node))
            ExpHidden ->
              NExpHidden node (nodeExpForNode (peekResult node))
            ExpForward ->
              NExpForward node (nodeExpForNode (peekResult node))
        NodeSugar ->
          case nodeType node of
            ExpDoStmt ->
              NExpLiteral node "{do stmt}"
            ExpFieldUpdate ->
              NExpFieldExp node exp (zip (getFieldLabels node) range)
              where
                (exp:range) = map nodeExpForNode (peekSubExprs node)

findLambdas :: NodeExp -> NodeExp
findLambdas (NExpApp node fun args res) =
  NExpApp node
          (findLambdas fun)
          (map findLambdas args)
          (findLambdas res)
findLambdas (NExpIdentifier node (Qualified "" "(\\..)") _) =
  NExpLambda node Nothing (map findLambdas (findApplications node))
findLambdas (NExpIdentifier node (Qualified mod "(\\..)") _) =
  NExpLambda node (Just mod) (map findLambdas (findApplications node))
findLambdas (NExpIdentifier node (Plain "(\\..)") _) =
  NExpLambda node Nothing (map findLambdas (findApplications node))
findLambdas (NExpLambda node mod apps) =
  NExpLambda node mod (map findLambdas apps)
findLambdas (NExpCond node condType cond res) =
  NExpCond node
           condType
           (findLambdas cond)
           (findLambdas res)
findLambdas (NExpFieldExp node exp pairs) =
  NExpFieldExp node
               (findLambdas exp)
               (map findInMappings pairs)
  where
    findInMappings :: (String, NodeExp) -> (String, NodeExp)
    findInMappings (x,y) = (x, findLambdas y)
findLambdas (NExpProjection node proj) =
  NExpProjection node (findLambdas proj)
findLambdas (NExpForward node forw) =
  NExpForward node (findLambdas forw)
findLambdas (NExpHidden node hidd) =
  NExpHidden node (findLambdas hidd)
findLambdas x = x

findApplications :: FileNode -> [NodeExp]
findApplications node = []

removeVerticalCycles :: NodeExp -> NodeExp
removeVerticalCycles = remCycles []
  where
    remCycles :: [FileNode] -> NodeExp -> NodeExp
    remCycles xs (NExpApp node fun args res)
      = if xs `contains` (getNode res)
          then NExpApp node
                       (remCycles (node:xs) fun)
                       (map (remCycles (node:xs)) args)
                       NExpVerticalCycle
          else NExpApp node
                       (remCycles (node:xs) fun)
                       (map (remCycles (node:xs)) args)
                       (remCycles (node:xs) res)
    remCycles xs (NExpCond node condType cond res)
      = if xs `contains` (getNode res)
          then NExpCond node
                        condType
                        (remCycles (node:xs) cond)
                        NExpVerticalCycle
          else NExpCond node
                        condType
                        (remCycles (node:xs) cond)
                        (remCycles (node:xs) res)
    remCycles xs (NExpProjection node res)
      = if xs `contains` (getNode res)
          then NExpProjection node
                              NExpVerticalCycle
          else NExpProjection node
                              (remCycles (node:xs) res)
    remCycles xs (NExpForward node res)
      = if xs `contains` (getNode res)
          then NExpForward node NExpVerticalCycle
          else NExpForward node (remCycles (node:xs) res)
    remCycles xs (NExpHidden node res)
      = if xs `contains` (getNode res)
          then NExpHidden node NExpVerticalCycle
          else NExpHidden node (remCycles (node:xs) res)
    remCycles xs (NExpFieldExp node exp mappings)
      = NExpFieldExp node
                     (remCycles (node:xs) exp)
                     (map remCycs mappings)
        where
          remCycs :: (String, NodeExp) -> (String, NodeExp)
          remCycs (x,y) = (x, remCycles (node:xs) y)
    remCycles xs y = y

limitNExpDepth :: Int -> NodeExp -> NodeExp
limitNExpDepth 0 _ = NExpNotShown
limitNExpDepth n (NExpApp node fun args res)
  = NExpApp node
            (limitNExpDepth (n-1) fun)
            (map (limitNExpDepth (n-1)) args)
            (limitNExpDepth (n-1) res)
limitNExpDepth n (NExpCond node condType cond res)
  = NExpCond node
             condType
             (limitNExpDepth (n-1) cond)
             (limitNExpDepth (n-1) res)
limitNExpDepth n (NExpFieldExp node exp mappings)
  = NExpFieldExp node
                 (limitNExpDepth (n-1) exp)
                 (map limitMapDepth mappings)
    where
      limitMapDepth :: (String, NodeExp) -> (String, NodeExp)
      limitMapDepth (x, y) = (x, limitNExpDepth (n-1) y)
limitNExpDepth n (NExpProjection node res)
  = NExpProjection node
                   (limitNExpDepth (n-1) res)
limitNExpDepth n (NExpForward node res)
  = NExpForward node (limitNExpDepth (n-1) res)
limitNExpDepth n (NExpHidden node res)
  = NExpHidden node
               (limitNExpDepth (n-1) res)
limitNExpDepth n x = x

compressClosures :: NodeExp -> NodeExp
compressClosures (NExpApp node
                          (NExpApp innerNode innerFun innerArgs innerRes)
                          args
                          res) =
  NExpApp node
          innerFun
          (map compressClosures (innerArgs ++ args))
          (compressClosures res)
compressClosures (NExpApp node fun args res) =
  NExpApp node
          (compressClosures fun)
          (map compressClosures args)
          (compressClosures res)
compressClosures (NExpCond node condType cond res) =
  NExpCond node
           condType
           (compressClosures cond)
           (compressClosures res)
compressClosures (NExpFieldExp node exp mappings) =
  NExpFieldExp node
               (compressClosures exp)
               (map compressMapping mappings)
  where
    compressMapping :: (String, NodeExp) -> (String, NodeExp)
    compressMapping (x, y) = (x, compressClosures y)
compressClosures (NExpProjection node res) =
  NExpProjection node (compressClosures res)
compressClosures (NExpForward node res) =
  NExpForward node (compressClosures res)
compressClosures (NExpHidden node res) =
  NExpHidden node (compressClosures res)
compressClosures x = x

unevalVSExp :: NodeExp -> VSExp
unevalVSExp = flatVSExp unevalVSExp

partEvalVSExp :: NodeExp -> VSExp
partEvalVSExp = flatVSExp fullEvalVSExp

fullEvalVSExp :: NodeExp -> VSExp
fullEvalVSExp = condEvalVSExp (\x -> True)

-- Only for use internally by VSExp.  Takes a function for retieving the
-- subexpressions of 
flatVSExp :: (NodeExp -> VSExp) -> NodeExp -> VSExp
flatVSExp buildSubs (NExpApp node fun args res)
  = VSApp node funExp argExps
      where
        funExp = buildSubs fun
        argExps = map buildSubs args
flatVSExp buildSubs (NExpIdentifier node id fix)
  = VSIdentifier node id fix
flatVSExp buildSubs (NExpLambda node mod apps)
  = VSLambda node mod (map getPairs apps)
    where
      getPairs :: NodeExp -> ([VSExp], VSExp)
      getPairs (NExpApp node fun args res) =
        ((map buildSubs args), buildSubs res)
flatVSExp buildSubs (NExpLiteral node nm)
  = VSLiteral node nm
flatVSExp buildSubs (NExpCond node condType cond res)
  = VSCond node condType (buildSubs cond) (buildSubs res)
flatVSExp buildSubs (NExpFieldExp node exp mappings)
  = VSFieldExp node procExp procMappings
      where
        procExp = buildSubs exp
        procMappings = map buildSub mappings
        buildSub :: (String, NodeExp) -> (String, VSExp)
        buildSub (x, y) = (x, buildSubs y)
flatVSExp buildSubs (NExpProjection node res)
  = buildSubs res
flatVSExp buildSubs (NExpForward node res)
  = buildSubs res
flatVSExp buildSubs (NExpHidden node res)
  = VSLiteral node "{?}"
flatVSExp _ NExpInterrupt
  = VSInterrupt
flatVSExp _ NExpBottom
  = VSBottom
flatVSExp _ NExpNotShown
  = VSNotShown
flatVSExp _ NExpVerticalCycle
  = VSVerticalCycle
flatVSExp _ NExpUneval
  = VSUneval

-- Will chase result pointers until it hits the specified offest.
cutoffEvalVSExp :: FileNode -> NodeExp -> VSExp
cutoffEvalVSExp cutoff node = condEvalVSExp (notUnevalOrLessThan cutoff) node

notUnevalOrLessThan :: FileNode -> NodeExp -> Bool
notUnevalOrLessThan fn ne = (ne /= NExpUneval) && (lessThanNode fn ne)

lessThanNode :: FileNode -> NodeExp -> Bool
lessThanNode node (NExpVerticalCycle) = False
lessThanNode node exp
  = if isSimpleNodeType exp then True
    else if caredNode == Nothing then True
    else (unMaybe caredNode) < node
    where
      caredNode = (getFirstCaredNode exp)

unMaybe :: Maybe a -> a
unMaybe (Just x) = x

fetchOffset :: NodeExp -> FileNode
fetchOffset (NExpApp node _ _ _) = node
fetchOffset (NExpIdentifier node _ _) = node
fetchOffset (NExpLiteral node _) = node
fetchOffset (NExpCond node _ _ _) = node
fetchOffset (NExpFieldExp node _ _) = node
fetchOffset (NExpProjection node _) = node
fetchOffset (NExpForward _ res) = fetchOffset res
fetchOffset (NExpHidden _ res) = fetchOffset res

getFirstCaredNode :: NodeExp -> Maybe FileNode
getFirstCaredNode (NExpApp node _ _ res)
  = if res == NExpVerticalCycle || res == NExpUneval
      then Nothing
      else Just node
getFirstCaredNode (NExpIdentifier node _ _) = Just node
getFirstCaredNode (NExpLiteral node _) = Just node
getFirstCaredNode (NExpCond node _ _ _) = Just node
getFirstCaredNode (NExpFieldExp node _ _) = Just node
getFirstCaredNode (NExpProjection node _) = Just node
getFirstCaredNode (NExpForward _ res) = getFirstCaredNode res
getFirstCaredNode (NExpHidden _ res) = getFirstCaredNode res
getFirstCaredNode _ = Nothing

-- Will chase result pointers until the predicate passed returns False.
condEvalVSExp :: (NodeExp -> Bool) -> NodeExp -> VSExp
condEvalVSExp pred nExp@(NExpApp node fun args res)
  = if pred res
      then condEvalVSExp pred res
      else VSApp node
                 (condEvalVSExp pred fun)
                 (map (condEvalVSExp pred) args)
condEvalVSExp pred nExp@(NExpCond node condType cond res)
  = if pred res
      then condEvalVSExp pred res
      else VSCond node
                  condType
                  (condEvalVSExp pred cond)
                  (condEvalVSExp pred res)
condEvalVSExp pred nExp@(NExpFieldExp node exp mappings)
  = VSFieldExp node evalExp evalMappings
    where
      evalExp = condEvalVSExp pred exp
      evalMappings = map evalMapping mappings
      evalMapping :: (String, NodeExp) -> (String, VSExp)
      evalMapping (x,y) = (x, condEvalVSExp pred y)
condEvalVSExp pred nExp@(NExpProjection node res)
  = condEvalVSExp pred res
condEvalVSExp pred nExp@(NExpForward _ res)
  = condEvalVSExp pred res
condEvalVSExp pred nExp@(NExpHidden _ res)
  = condEvalVSExp pred res
condEvalVSExp _ x = unevalVSExp x

cutoffFindMinOffset :: FileNode -> NodeExp -> [String] -> Maybe FileNode
cutoffFindMinOffset cutoff node hides
  = if offsets == [] then Nothing
    else Just (minimum offsets)
    where offsets = (map fetchOffset
                         (filter (not . isSimpleNodeType)
                                 (cutoffFindNodes cutoff hides node)))

cutoffFindNodes :: FileNode -> [String] -> NodeExp -> [NodeExp]
cutoffFindNodes cutoff = condFindNodes (lessThanNode cutoff)

condFindNodes :: (NodeExp -> Bool) -> [String] -> NodeExp -> [NodeExp]
condFindNodes pred hides nExp@(NExpApp _ fun args res)
  = case (condEvalVSExp pred fun) of
      (VSIdentifier _ name _) ->
        if hides `contains` (showQN True name) ||
           hides `contains` (showQN False name) then []
        else if pred res then condFindNodes pred hides res
        else (res:((condFindNodes pred hides fun)
                   ++ (foldr ((++) . (condFindNodes pred hides)) [] args)))
      _ -> if pred res then condFindNodes pred hides res
           else (res:((condFindNodes pred hides fun)
                      ++ (foldr ((++) . (condFindNodes pred hides)) [] args)))
condFindNodes pred hides nExp@(NExpCond _ condType cond res)
  = let
      shouldHide =
        case condType of
          IfCond    -> (hides `contains` "if")
          CaseCond  -> (hides `contains` "case")
          GuardCond -> (hides `contains` "|")
    in
      if shouldHide || pred res then condFindNodes pred hides res
      else (res:(condFindNodes pred hides cond))
condFindNodes pred hides nExp@(NExpFieldExp _ exp mappings)
  = (condFindNodes pred hides exp)
     ++ (foldr ((++) . (condFindNodes pred hides) . snd) [] mappings)
condFindNodes pred hides nExp@(NExpProjection _ res)
  = (condFindNodes pred hides res)
condFindNodes pred hides nExp@(NExpForward _ res)
  = (condFindNodes pred hides res)
condFindNodes pred hides nExp@(NExpHidden _ res)
  = (condFindNodes pred hides res)
condFindNodes _ _ x = []

-- Trusts a function i.e. follows the result pointer for each application
-- of the function and replaces the node.  The result VSExp is created using
-- the import function.
hideFunction :: String -> (FileNode -> VSExp) -> VSExp -> VSExp
hideFunction name imp app@(VSApp node fun@(VSIdentifier _ funName fix) args)
  = if ((showQN True funName) == name || (showQN False funName) == name)
       && nodeType node /= ExpValueApp
      then hideFunction name imp (imp (peekResult node))
      else VSApp node fun (map (hideFunction name imp) args)
hideFunction name imp app@(VSApp node fun args)
  = VSApp node (hideFunction name imp fun)
               (map (hideFunction name imp) args)
hideFunction "if" imp ifCon@(VSCond node IfCond _ res)
  = hideFunction "if" imp res
hideFunction name imp ifCon@(VSCond node IfCond cond res)
  = VSCond node IfCond (hideFunction name imp cond) (hideFunction name imp res)
hideFunction "case" imp ifCon@(VSCond node CaseCond _ res)
  = hideFunction "case" imp res
hideFunction name imp ifCon@(VSCond node CaseCond cond res)
  = VSCond node CaseCond (hideFunction name imp cond) (hideFunction name imp res)
hideFunction "|" imp ifCon@(VSCond node GuardCond cond res)
  = hideFunction "|" imp res
hideFunction name imp ifCon@(VSCond node GuardCond cond res)
  = VSCond node GuardCond (hideFunction name imp cond) (hideFunction name imp res)
hideFunction name imp fieldExp@(VSFieldExp node exp mappings)
  = VSFieldExp node (hideFunction name imp exp) (map hideMapping mappings)
    where
      hideMapping :: (String, VSExp) -> (String, VSExp)
      hideMapping (x, y) = (x, hideFunction name imp y)
hideFunction name imp anythingElse
  = anythingElse
  
-- Limits the depth of a specified Very Simple Expression
-- will replace all VSExps that go beyond this depth limit with VSNotShown
limitDepth :: Int -> VSExp -> VSExp
limitDepth 0 _ = VSNotShown
limitDepth n (VSApp node fun args)
  = VSApp node (limitDepth (n-1) fun) (map (limitDepth (n-1)) args)
limitDepth n (VSCond node condType cond res) 
  = VSCond node condType (limitDepth (n-1) cond) (limitDepth (n-1) res)
limitDepth n (VSFieldExp node exp mappings)
  = VSFieldExp node (limitDepth (n-1) exp)
               (map (limitSnd (n-1)) mappings)
limitDepth n x = x

limitSnd :: Int -> (String, VSExp) -> (String, VSExp)
limitSnd n (x, y) = (x,limitDepth n y)

-- Detects cycles in a VSExp and removes them, replacing the offending nodes
-- with VSNotShown
removeHorizontalCycles :: VSExp -> VSExp
removeHorizontalCycles = remCycles []
  where
    remCycles :: [FileNode] -> VSExp -> VSExp
    remCycles visited (VSApp node fun args)
      = if visited `contains` node
          then VSHorizontalCycle
          else VSApp node
                     (remCycles (node:visited) fun)
                     (map (remCycles (node:visited)) args)
    remCycles visited (VSCond node condType cond res)
      = if visited `contains` node
          then VSHorizontalCycle
          else VSCond node condType (remCycles (node:visited) cond) (remCycles (node:visited) res)
    remCycles visited (VSFieldExp node exp mappings)
      = if visited `contains` node
          then VSFieldExp node
                          (remCycles (node:visited) exp)
                          (map remCycs mappings)
          else VSHorizontalCycle
        where
          remCycs :: (String, VSExp) -> (String, VSExp)
          remCycs (x,y) = (x, remCycles (node:visited) y)
    remCycles _ y = y

contains :: Eq a => [a] -> a -> Bool
contains [] _ = False
contains (x:xs) y | x == y    = True
                  | otherwise = contains xs y

vsExp2sExp :: VSExp -> SExp String
vsExp2sExp vsApp@(VSApp _ fun args)
  = if isAppString vsApp
      then SString "" (buildString vsApp) False
      else SApp "" ((vsExp2sExp fun):(map vsExp2sExp args))
vsExp2sExp (VSIdentifier _ id fix)
  = SId "" id fix
vsExp2sExp (VSLiteral _ lit)
  = SLiteral "" lit
vsExp2sExp (VSCond _ IfCond cond res)
  = SIf "" (vsExp2sExp cond) (Just (vsExp2sExp res))
vsExp2sExp (VSCond _ CaseCond cond res)
  = SCase "" (vsExp2sExp cond) (Just (vsExp2sExp res))
vsExp2sExp (VSCond _ GuardCond cond res)
  = SGuard "" (vsExp2sExp cond) (Just (vsExp2sExp res))
vsExp2sExp (VSFieldExp _ exp mappings)
  = SFieldExpr ""
               (vsExp2sExp exp)
               (fst (unzip mappings))
               (map vsExp2sExp (snd (unzip mappings)))
vsExp2sExp VSInterrupt = SInterrupted ""
vsExp2sExp VSBottom = SBottom ""
vsExp2sExp VSUneval = SUnevaluated ""
vsExp2sExp VSNotShown = SCut ""
vsExp2sExp VSVerticalCycle = SCut ""
vsExp2sExp VSHorizontalCycle = SCut ""

isAppString :: VSExp -> Bool
isAppString (VSApp node fun@(VSIdentifier idNode name fix) args)
  = if name == (Qualified "Prelude" ":")
      then ((nodeType (getVSNode (head args))) == ExpChar)
           && ((isAppString (head (tail args)))
              || isEmptyList (head (tail args)))
      else False
isAppString exp = False

buildString :: VSExp -> String
buildString (VSApp node fun args)
  = (:) (getCharFromLit (head args))
        (if (isEmptyList (head (tail args)))
          then []
          else buildString (head (tail args)))

isEmptyList :: VSExp -> Bool
isEmptyList (VSIdentifier _ (Qualified "Prelude" "[]") _) = True
isEmptyList exp = False

getCharFromLit :: VSExp -> Char
getCharFromLit (VSLiteral node charList)
  = (head (tail charList))
