/* Implementation of the column view.
 *
 * Copyright (c) 2003, 2004 Ole Laursen.
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA.
 */

#include <deque>
#include <cmath>

#include <libgnomecanvasmm/pixbuf.h>
#include <gconfmm/client.h>

#include "column-view.hpp"
#include "applet.hpp"
#include "monitor.hpp"

#include "pixbuf-drawing.hpp"


//
// class ColumnGraph - represents the columns in a column diagram
//

class ColumnGraph
{
public:
  ColumnGraph(Monitor *monitor, const Glib::ustring &mon_dir);
  ~ColumnGraph();

  void update(unsigned int max_samples); // gather info from monitor
  void draw(Gnome::Canvas::Canvas &canvas, // redraw columns on canvas
	    Applet *applet, int width, int height, int samples,
	    double time_offset); 

  Monitor *monitor;
  
private:
  // a pixbuf is used for the columns
  std::auto_ptr<Gnome::Canvas::Pixbuf> columns;

  // we need to store the past values
  typedef std::deque<double> value_sequence;
  typedef value_sequence::iterator value_iterator;
  value_sequence values;

  Glib::ustring monitor_dir;
};

ColumnGraph::ColumnGraph(Monitor *m, const Glib::ustring &mon_dir)
  : monitor(m), monitor_dir(mon_dir)
{
}

ColumnGraph::~ColumnGraph()
{
}

void ColumnGraph::update(unsigned int max_samples)
{
  double val = monitor->value();
  values.push_front(val);
  
  // get rid of extra samples (there may be more than one if user changes
  // configuration)
  while (values.size() > max_samples)
    values.pop_back();
}

void ColumnGraph::draw(Gnome::Canvas::Canvas &canvas,
		       Applet *applet, int width, int height, int samples,
		       double time_offset)
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet->get_gconf_client();
    
  // ensure there are the correct no. of rectangles
  value_iterator vi = values.begin(), vend = values.end();

  if (vi == vend)		// there must be at least one point
    return;

  
  // get colour
  unsigned int color;

  Gnome::Conf::Value v;
  
  v = client->get(monitor_dir + "/color");
  if (v.get_type() == Gnome::Conf::VALUE_INT)
    color = v.get_int();
  else {
    color = applet->get_fg_color();
    client->set(monitor_dir + "/color", int(color));
  }

  // make sure we got a pixbuf and that it has the right size
  Glib::RefPtr<Gdk::Pixbuf> pixbuf;

  if (columns.get() == 0)
    pixbuf = Gdk::Pixbuf::create(Gdk::COLORSPACE_RGB, true, 8, width, height);
  else {
    pixbuf = columns->property_pixbuf();

    // but perhaps the dimensions have changed
    if (pixbuf->get_width() != width || pixbuf->get_height() != height)
      pixbuf = Gdk::Pixbuf::create(Gdk::COLORSPACE_RGB, true, 8, width, height);
  }

  pixbuf->fill(color & 0xFFFFFF00);
  

  double max = std::max(monitor->max(), 0.000001); // avoid division by zero
  double pixels_per_sample = double(width) / (samples - 1);

  // start from right
  double l = width - pixels_per_sample * time_offset;

  do {
    if (*vi <= 0)
      continue;
    
    // FIXME: the uppermost pixel should be scaled down too to avoid aliasing
    double r = l + pixels_per_sample;
    int
      t = int((1 - (*vi / max)) * (height - 1)),
      b = height - 1;

    if (t < 0)
      t = 0;
    
    for (int x = std::max(int(l), 0); x < std::min(r, double(width)); ++x) {
      PixelPosition pos = get_position(pixbuf, x, t);

      // anti-aliasing effect; if we are partly on a pixel, scale alpha down
      double scale = 1.0;
      if (x < l)
	scale -= l - std::floor(l);
      if (x + 1 > r)
	scale -= std::ceil(r) - r;

      int alpha = int((color & 0xFF) * scale);
      
      for (int y = t; y <= b; ++y, pos.down())
	pos.pixel().alpha() = std::min(pos.pixel().alpha() + alpha, 255);
    }
    
    l -= pixels_per_sample;
  } while (++vi != vend);
  
  // update columns
  if (columns.get() == 0)
    columns.reset(new Gnome::Canvas::Pixbuf(*canvas.root(), 0, 0, pixbuf));
  else
    columns->property_pixbuf() = pixbuf;
}


//
// class ColumnView
//

ColumnView::ColumnView()
  : draws_since_update(0)
{
}

ColumnView::~ColumnView()
{
  for (column_iterator i = columns.begin(), end = columns.end(); i != end; ++i)
    delete *i;
}

void ColumnView::do_update()
{
  CanvasView::do_update();
  
  // first update the configured attributes
  Glib::RefPtr<Gnome::Conf::Client> &client = applet->get_gconf_client();
  Glib::ustring dir = applet->get_gconf_dir();

  if (client->get(dir + "/viewer/samples").get_type() == Gnome::Conf::VALUE_INT)
    samples = client->get_int(dir + "/viewer/samples");
  else {
    samples = 30;
    client->set(dir + "/viewer/samples", int(samples));
  }
  
  // then loop through each column
  for (column_iterator i = columns.begin(), end = columns.end(); i != end; ++i)
    (*i)->update(samples);
  
  draws_since_update = 0;
}

void ColumnView::do_attach(Monitor *monitor, const Glib::ustring &mon_dir)
{
  columns.push_back(new ColumnGraph(monitor, mon_dir));
}

void ColumnView::do_detach(Monitor *monitor)
{
  for (column_iterator i = columns.begin(), end = columns.end(); i != end; ++i)
    if ((*i)->monitor == monitor) {
      delete *i;
      columns.erase(i);
      return;
    }

  g_assert_not_reached();
}

void ColumnView::do_draw_loop()
{
  double time_offset =
    double(draws_since_update) * CanvasView::draw_interval
    / applet->get_update_interval();
  
  for (column_iterator i = columns.begin(), end = columns.end(); i != end; ++i)
    (*i)->draw(*canvas, applet, width(), height(), samples, time_offset);
  
  ++draws_since_update;
}
