%pure_parser
%{
/*
 * $Id: macro.y,v 1.30 2004/04/02 11:14:23 rglab Exp $
 */

/*
 * Harbour Project source code:
 * Macro compiler YACC rules and actions
 *
 * Copyright 1999 Antonio Linares <alinares@fivetech.com>
 * www - http://www.harbour-project.org
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA (or visit the web site http://www.gnu.org/).
 *
 * As a special exception, the Harbour Project gives permission for
 * additional uses of the text contained in its release of Harbour.
 *
 * The exception is that, if you link the Harbour libraries with other
 * files to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the Harbour library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * This exception applies only to the code released by the Harbour
 * Project under the name Harbour.  If you copy code from other
 * Harbour Project or Free Software Foundation releases into a copy of
 * Harbour, as the General Public License permits, the exception does
 * not apply to the code that you add in this way.  To avoid misleading
 * anyone as to the status of such modified files, you must delete
 * this exception notice from them.
 *
 * If you write modifications of your own for Harbour, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.
 *
 */

/* TODO list
 * 1) jumps longer then 2^15 bytes
 * 2) Change the pcode generated by ::cVar from Self:cVar to QSELF():cVar
 * 3) Support this syntax: nPtr := @Hello()
 */

/* this #define HAVE TO be placed before all #include directives
 */
#define  HB_MACRO_SUPPORT

#include "hbmacro.h"
#include "hbcomp.h"

/* Compile using: bison -d -p hb_comp macro.y */

/* NOTE: these symbols are used internally in bison.simple
 */
#undef alloca
#define alloca  hb_xgrab
#undef malloc
#define malloc  hb_xgrab
#undef free
#define free hb_xfree
/* This is workaround of yyparse() declaration bug in bison.simple
*/

#if !defined(__GNUC__) && !defined(__IBMCPP__)
   #if 0
      /* This makes BCC 551 fail with Bison 1.30, even with the
         supplied harbour.simple file, which makes Bison 1.30 blow.
         [vszakats] */
      void __yy_memcpy ( char*, const char*, unsigned int ); /* to satisfy Borland compiler */
   #endif
#endif

#ifdef __GNUC__
#undef __GNUC__
#endif

/* yacc/lex related definitions
 */
#undef YYPARSE_PARAM
#define YYPARSE_PARAM HB_MACRO_PARAM    /* parameter passed to yyparse function - it have to be of 'void *' type */
#undef YYLEX_PARAM
#define YYLEX_PARAM   ( (HB_MACRO_PTR)YYPARSE_PARAM ) /* additional parameter passed to yylex */

extern int yyparse( void * );	/* to make happy some purist compiler */
extern void * hb_compFlexNew( HB_MACRO_PTR );
extern void hb_compFlexDelete( void * );

extern void yyerror( char * ); /* parsing error management function */

/* Standard checking for valid expression creation
 */
#define  HB_MACRO_CHECK( pExpr ) \
   if( ! ( HB_MACRO_DATA->status & HB_MACRO_CONT ) ) \
   { \
      YYABORT; \
   }

#define HB_MACRO_IFENABLED( pSet, pExpr, flag ) \
   if( HB_MACRO_DATA->supported & (flag) ) \
     { pSet = (pExpr); }\
   else \
   { \
      YYABORT; \
   }

%}

%union                  /* special structure used by lex and yacc to share info */
{
   char * string;       /* to hold a string returned by lex */
   int    iNumber;      /* to hold a temporary integer number */
   long   lNumber;      /* to hold a temporary long number */
   struct
   {
      int    iNumber;      /* to hold a number returned by lex */
      char * szValue;
   } valInteger;
   struct
   {
      long   lNumber;      /* to hold a long number returned by lex */
      char * szValue;
   } valLong;
   struct
   {
      double dNumber;   /* to hold a double number returned by lex */
      /* NOTE: Intentionally using "unsigned char" instead of "BYTE" */
      unsigned char bWidth; /* to hold the width of the value */
      unsigned char bDec; /* to hold the number of decimal points in the value */
      char * szValue;
   } valDouble;
   HB_EXPR_PTR asExpr;
   void * pVoid;        /* to hold any memory structure we may need */
};

%{
/* This must be placed after the above union - the union is
 * typedef-ined to YYSTYPE
 */
int yylex( YYSTYPE *, HB_MACRO_PTR );
%}


%token IDENTIFIER NIL NUM_DOUBLE INASSIGN NUM_LONG
%token IIF IF LITERAL TRUEVALUE FALSEVALUE
%token AND OR NOT EQ NE1 NE2 INC DEC ALIASOP SELF
%token LE GE FIELD MACROVAR MACROTEXT
%token PLUSEQ MINUSEQ MULTEQ DIVEQ POWER EXPEQ MODEQ

/*the lowest precedence*/
/*postincrement and postdecrement*/
%left  POST
/*assigment - from right to left*/
%right INASSIGN
%right  PLUSEQ MINUSEQ
%right  MULTEQ DIVEQ MODEQ
%right  EXPEQ
/*logical operators*/
%right  OR
%right  AND
%right  NOT
/*relational operators*/
%right  '=' '<' '>' EQ NE1 NE2 LE GE '$'
/*mathematical operators*/
%right  '+' '-'
%right  '*' '/' '%'
%right  POWER
%right UNARY
/*preincrement and predecrement*/
%right  PRE
/*special operators*/
%right  ALIASOP '&' '@'
%right ','
/*the highest precedence*/

%type <string>  IDENTIFIER LITERAL MACROVAR MACROTEXT
%type <valDouble>  NUM_DOUBLE
%type <valLong>    NUM_LONG
%type <asExpr>  Argument ArgList ElemList BlockExpList BlockVarList BlockNoVar
%type <asExpr>  NumValue NumAlias
%type <asExpr>  NilValue
%type <asExpr>  LiteralValue
%type <asExpr>  CodeBlock
%type <asExpr>  Logical
%type <asExpr>  SelfValue
%type <asExpr>  Array
%type <asExpr>  ArrayAt
%type <asExpr>  Variable VarAlias
%type <asExpr>  MacroVar MacroVarAlias
%type <asExpr>  MacroExpr MacroExprAlias
%type <asExpr>  AliasId AliasVar AliasExpr
%type <asExpr>  VariableAt
%type <asExpr>  FunCall
%type <asExpr>  ObjectData
%type <asExpr>  ObjectMethod
%type <asExpr>  IfInline
%type <asExpr>  ExpList PareExpList PareExpListAlias AsParamList RootParamList
%type <asExpr>  Expression SimpleExpression
%type <asExpr>  EmptyExpression
%type <asExpr>  ExprAssign ExprOperEq ExprPreOp ExprPostOp
%type <asExpr>  ExprMath ExprBool ExprRelation ExprUnary
%type <asExpr>  ExprPlusEq ExprMinusEq ExprMultEq ExprDivEq ExprModEq ExprExpEq
%type <asExpr>  ArrayIndex IndexList
%type <asExpr>  FieldAlias FieldVarAlias
%type <asExpr>  PostOp

%%

Main : Expression '\n' {
                           HB_MACRO_DATA->exprType = hb_compExprType( $1 );
                           if( HB_MACRO_DATA->Flags &  HB_MACRO_GEN_PUSH )
                              hb_compExprGenPush( $1, HB_MACRO_PARAM );
                           else
                              hb_compExprGenPop( $1, HB_MACRO_PARAM );
                           hb_compGenPCode1( HB_P_ENDPROC, HB_MACRO_PARAM );
                        }

     | Expression      {
                           HB_MACRO_DATA->exprType = hb_compExprType( $1 );
                           if( HB_MACRO_DATA->Flags &  HB_MACRO_GEN_PUSH )
                              hb_compExprGenPush( $1, HB_MACRO_PARAM );
                           else
                              hb_compExprGenPop( $1, HB_MACRO_PARAM );
                           hb_compGenPCode1( HB_P_ENDPROC, HB_MACRO_PARAM );
                        }
     | AsParamList      {
                           HB_MACRO_DATA->exprType = hb_compExprType( $1 );
                           if( HB_MACRO_DATA->Flags &  HB_MACRO_GEN_PUSH )
                              hb_compExprGenPush( $1, HB_MACRO_PARAM );
                           else
                              hb_compExprGenPop( $1, HB_MACRO_PARAM );
                           hb_compGenPCode1( HB_P_ENDPROC, HB_MACRO_PARAM );
                        }
     | Expression error   {
                 HB_TRACE(HB_TR_DEBUG, ("macro -> invalid expression: %s", HB_MACRO_DATA->string));
                 hb_macroError( EG_SYNTAX, HB_MACRO_PARAM );
		           YYABORT;
	       }
     | error   {
                 HB_TRACE(HB_TR_DEBUG, ("macro -> invalid syntax: %s", HB_MACRO_DATA->string));
                 hb_macroError( EG_SYNTAX, HB_MACRO_PARAM );
		           YYABORT;
	       }
;

/* Numeric values
 */
NumValue   : NUM_DOUBLE      { $$ = hb_compExprNewDouble( $1.dNumber, $1.bWidth, $1.bDec ); }
           | NUM_LONG        { $$ = hb_compExprNewLong( $1.lNumber ); }
           ;

NumAlias   : NUM_LONG ALIASOP      { $$ = hb_compExprNewLong( $1.lNumber ); }
;

/* NIL value
 */
NilValue   : NIL             { $$ = hb_compExprNewNil(); }
;

/* Literal string value
 */
LiteralValue : LITERAL       { $$ = hb_compExprNewString( $1 ); }
;

/* Logical value
 */
Logical    : TRUEVALUE       { $$ = hb_compExprNewLogical( TRUE ); }
           | FALSEVALUE      { $$ = hb_compExprNewLogical( FALSE ); }
           ;

/* SELF value and expressions
 */
SelfValue  : SELF            { $$ = hb_compExprNewSelf(); }
;

/* Literal array
 */
Array      : '{' ElemList '}'          { $$ = hb_compExprNewArray( $2 ); }
           ;

/* Literal array access
 */
ArrayAt     : Array ArrayIndex   { $$ = $2; }
;

/* Variables
 */
Variable    : IDENTIFIER         { $$ = hb_compExprNewVar( $1 ); }
;

VarAlias    : IDENTIFIER ALIASOP      { $$ = hb_compExprNewAlias( $1 ); }
;

/* Macro variables - this can signal compilation errors
 */
MacroVar    : MACROVAR        { $$ = hb_compExprNewMacro( NULL, '&', $1 );
                                HB_MACRO_CHECK( $$ );
                              }
            | MACROTEXT       {  ULONG ulLen = strlen( $1 );
                                 char * szVarName = hb_macroTextSubst( $1, &ulLen );
                                 if( hb_macroIsIdent( szVarName ) )
                                 {
                                    $$ = hb_compExprNewVar( szVarName );
                                    hb_xfree( $1 );
                                    HB_MACRO_CHECK( $$ );
                                 }
                                 else
                                 {
                                    /* invalid variable name
                                     */
				    HB_TRACE(HB_TR_DEBUG, ("macro -> invalid variable name: %s", $1));

                                    hb_xfree( $1 );
                                    YYABORT;
                                 }
                              }
;

MacroVarAlias  : MacroVar ALIASOP   { $$ = $1; }
;

/* Macro expressions
 */
MacroExpr  : '&' PareExpList       { $$ = hb_compExprNewMacro( $2, 0, NULL ); }
;

MacroExprAlias : MacroExpr ALIASOP     { $$ = $1; }
;

/* Aliased variables
 */
/* special case: _FIELD-> and FIELD-> can be nested
 */
FieldAlias  : FIELD ALIASOP               { $$ = hb_compExprNewAlias( hb_strdup( "FIELD") ); }
            | FIELD ALIASOP FieldAlias    { $$ = $3; }
            ;

/* ignore _FIELD-> or FIELD-> if a real alias is specified
 */
FieldVarAlias  : FieldAlias VarAlias            { $$ = $2; }
               | FieldAlias NumAlias            { $$ = $2; }
               | FieldAlias PareExpListAlias    { $$ = $2; }
               | FieldAlias MacroVarAlias       { $$ = $2; }
               | FieldAlias MacroExprAlias      { $$ = $2; }
               ;

AliasId     : IDENTIFIER      { $$ = hb_compExprNewVar( $1 ); }
            | MacroVar        { $$ = $1; }
            ;

AliasVar   : NumAlias AliasId          { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           | MacroVarAlias AliasId     { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           | MacroExprAlias AliasId    { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           | PareExpListAlias AliasId  { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           | VarAlias AliasId          { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           | FieldAlias AliasId        { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           | FieldVarAlias AliasId     { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           ;

/* Aliased expressions
 */
/* NOTE: In the case:
 * alias->( Expression )
 * alias always selects a workarea at runtime
 */
AliasExpr  : NumAlias PareExpList         { $$ = hb_compExprNewAliasExpr( $1, $2 ); }
           | VarAlias PareExpList         { $$ = hb_compExprNewAliasExpr( $1, $2 ); }
           | MacroVarAlias PareExpList    { $$ = hb_compExprNewAliasExpr( $1, $2 ); }
           | MacroExprAlias PareExpList   { $$ = hb_compExprNewAliasExpr( $1, $2 ); }
           | PareExpListAlias PareExpList { $$ = hb_compExprNewAliasExpr( $1, $2 ); }
           ;

/* Array expressions access
 */
VariableAt  : NilValue      ArrayIndex    { $$ = $2; }
            | LiteralValue  ArrayIndex    { $$ = $2; }
            | CodeBlock     ArrayIndex    { $$ = $2; }
            | Logical       ArrayIndex    { $$ = $2; }
            | SelfValue     ArrayIndex    { $$ = $2; }
            | Variable      ArrayIndex    { $$ = $2; }
            | AliasVar      ArrayIndex    { $$ = $2; }
            | AliasExpr     ArrayIndex    { $$ = $2; }
            | MacroVar      ArrayIndex    { $$ = $2; }
            | MacroExpr     ArrayIndex    { $$ = $2; }
            | ObjectData    ArrayIndex    { $$ = $2; }
            | ObjectMethod  ArrayIndex    { $$ = $2; }
            | FunCall       ArrayIndex    { $$ = $2; }
            | IfInline      ArrayIndex    { $$ = $2; }
            | PareExpList   ArrayIndex    { $$ = $2; }
            ;

/* Function call
 */
FunCall    : IDENTIFIER '(' ArgList ')'   { $$ = hb_compExprNewFunCall( hb_compExprNewFunName( $1 ), $3, HB_MACRO_PARAM );
                                            HB_MACRO_CHECK( $$ );
                                          }
           | MacroVar '(' ArgList ')'     { $$ = hb_compExprNewFunCall( $1, $3, HB_MACRO_PARAM );
                                            HB_MACRO_CHECK( $$ );
                                          }
;

ArgList    : Argument                     { $$ = hb_compExprNewArgList( $1 ); }
           | ArgList ',' Argument         { $$ = hb_compExprAddListExpr( $1, $3 ); }
           ;

Argument   : EmptyExpression                   { $$ = $1; }
           | '@' IDENTIFIER                    { $$ = hb_compExprNewVarRef( $2 ); }
           | '@' AliasVar                      { $$ = hb_compExprNewRef( $2 ); }
           | '@' IDENTIFIER '(' ')'            { $$ = hb_compExprNewFunRef( $2 ); }
           ;

/* Object's instance variable
 */
ObjectData  : NumValue ':' IDENTIFIER        { $$ = hb_compExprNewSend( $1, $3 ); }
            | NilValue ':' IDENTIFIER        { $$ = hb_compExprNewSend( $1, $3 ); }
            | LiteralValue ':' IDENTIFIER    { $$ = hb_compExprNewSend( $1, $3 ); }
            | CodeBlock ':' IDENTIFIER       { $$ = hb_compExprNewSend( $1, $3 ); }
            | Logical ':' IDENTIFIER         { $$ = hb_compExprNewSend( $1, $3 ); }
            | SelfValue ':' IDENTIFIER       { $$ = hb_compExprNewSend( $1, $3 ); }
            | Array ':' IDENTIFIER           { $$ = hb_compExprNewSend( $1, $3 ); }
            | ArrayAt ':' IDENTIFIER         { $$ = hb_compExprNewSend( $1, $3 ); }
            | Variable ':' IDENTIFIER        { $$ = hb_compExprNewSend( $1, $3 ); }
            | AliasVar ':' IDENTIFIER        { $$ = hb_compExprNewSend( $1, $3 ); }
            | AliasExpr ':' IDENTIFIER       { $$ = hb_compExprNewSend( $1, $3 ); }
            | MacroVar ':' IDENTIFIER        { $$ = hb_compExprNewSend( $1, $3 ); }
            | MacroExpr ':' IDENTIFIER       { $$ = hb_compExprNewSend( $1, $3 ); }
            | FunCall ':' IDENTIFIER         { $$ = hb_compExprNewSend( $1, $3 ); }
            | IfInline ':' IDENTIFIER        { $$ = hb_compExprNewSend( $1, $3 ); }
            | PareExpList ':' IDENTIFIER     { $$ = hb_compExprNewSend( $1, $3 ); }
            | VariableAt ':' IDENTIFIER      { $$ = hb_compExprNewSend( $1, $3 ); }
            | ObjectMethod ':' IDENTIFIER    { $$ = hb_compExprNewSend( $1, $3 ); }
            | ObjectData ':' IDENTIFIER      { $$ = hb_compExprNewSend( $1, $3 ); }
            ;

/* Object's method
 */
ObjectMethod : ObjectData '(' ArgList ')'    { $$ = hb_compExprNewMethodCall( $1, $3 ); }
            ;

SimpleExpression :
             NumValue
           | NilValue                         { $$ = $1; }
           | LiteralValue                     { $$ = $1; }
           | CodeBlock                        { $$ = $1; }
           | Logical                          { $$ = $1; }
           | SelfValue                        { $$ = $1; }
           | Array                            { $$ = $1; }
           | ArrayAt                          { $$ = $1; }
           | AliasVar                         { $$ = $1; }
           | MacroVar                         { $$ = $1; }
           | MacroExpr                        { $$ = $1; }
           | Variable                         { $$ = $1; }
           | VariableAt                       { $$ = $1; }
           | FunCall                          { $$ = $1; }
           | IfInline                         { $$ = $1; }
           | ObjectData                       { $$ = $1; }
           | ObjectMethod                     { $$ = $1; }
           | AliasExpr                        { $$ = $1; }
           | ExprAssign                       { $$ = $1; }
           | ExprOperEq                       { HB_MACRO_IFENABLED( $$, $1, HB_SM_HARBOUR ); }
           | ExprPostOp                       { HB_MACRO_IFENABLED( $$, $1, HB_SM_HARBOUR ); }
           | ExprPreOp                        { HB_MACRO_IFENABLED( $$, $1, HB_SM_HARBOUR ); }
           | ExprUnary                        { $$ = $1; }
           | ExprMath                         { $$ = $1; }
           | ExprBool                         { $$ = $1; }
           | ExprRelation                     { $$ = $1; }
;

Expression : SimpleExpression                 { $$ = $1; HB_MACRO_CHECK( $$ ); }
           | PareExpList                      { $$ = $1; HB_MACRO_CHECK( $$ ); }
;

RootParamList : EmptyExpression ',' {
                                      if( !(HB_MACRO_DATA->Flags & HB_MACRO_GEN_LIST) )
                                      {
                                         HB_TRACE(HB_TR_DEBUG, ("macro -> invalid expression: %s", HB_MACRO_DATA->string));
                                         hb_macroError( EG_SYNTAX, HB_MACRO_PARAM );
                                         YYABORT;
                                      }
                                    }
                EmptyExpression     {
                                      HB_MACRO_DATA->iListElements = 1;
                                      $$ = hb_compExprAddListExpr( ( HB_MACRO_DATA->Flags & HB_MACRO_GEN_PARE ) ? hb_compExprNewList( $1 ) : hb_compExprNewArgList( $1 ), $4 );
                                    }
              ;

AsParamList  : RootParamList                    { $$ = $1; }
             | AsParamList ',' EmptyExpression  { HB_MACRO_DATA->iListElements++; $$ = hb_compExprAddListExpr( $1, $3 ); }
;

EmptyExpression: /* nothing => nil */        { $$ = hb_compExprNewEmpty(); }
           | Expression
;

/* NOTE: PostOp can be used in one context only - it uses $0 rule
 *    (the rule that stands before PostOp)
 */
PostOp      : INC    { $$ = hb_compExprNewPostInc( $<asExpr>0 ); }
            | DEC    { $$ = hb_compExprNewPostDec( $<asExpr>0 ); }
            ;

/* NOTE: We cannot use 'Expression PostOp' because it caused
 * shift/reduce conflicts
 */
ExprPostOp  : NumValue     PostOp %prec POST  { $$ = $2; }
            | NilValue     PostOp %prec POST  { $$ = $2; }
            | LiteralValue PostOp %prec POST  { $$ = $2; }
            | CodeBlock    PostOp %prec POST  { $$ = $2; }
            | Logical      PostOp %prec POST  { $$ = $2; }
            | SelfValue    PostOp %prec POST  { $$ = $2; }
            | Array        PostOp %prec POST  { $$ = $2; }
            | ArrayAt      PostOp %prec POST  { $$ = $2; }
            | Variable     PostOp %prec POST  { $$ = $2; }
            | MacroVar     PostOp %prec POST  { $$ = $2; }
            | MacroExpr    PostOp %prec POST  { $$ = $2; }
            | AliasVar     PostOp %prec POST  { $$ = $2; }
            | AliasExpr    PostOp %prec POST  { $$ = $2; }
            | VariableAt   PostOp %prec POST  { $$ = $2; }
            | PareExpList  PostOp %prec POST  { $$ = $2; }
            | IfInline     PostOp %prec POST  { $$ = $2; }
            | FunCall      PostOp %prec POST  { $$ = $2; }
            | ObjectData   PostOp %prec POST  { $$ = $2; }
            | ObjectMethod PostOp %prec POST  { $$ = $2; }
            ;

ExprPreOp   : INC Expression  %prec PRE      { $$ = hb_compExprNewPreInc( $2 ); }
            | DEC Expression  %prec PRE      { $$ = hb_compExprNewPreDec( $2 ); }
            ;

ExprUnary   : NOT Expression                 { $$ = hb_compExprNewNot( $2 ); }
            | '-' Expression  %prec UNARY    { $$ = hb_compExprNewNegate( $2 ); }
            | '+' Expression  %prec UNARY    { $$ = $2; }
            ;

ExprAssign  : NumValue     INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | NilValue     INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | LiteralValue INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | CodeBlock    INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | Logical      INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | SelfValue    INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | Array        INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | ArrayAt      INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | Variable     INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | MacroVar     INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | MacroExpr    INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | AliasVar     INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | AliasExpr    INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | VariableAt   INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | PareExpList  INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | IfInline     INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | FunCall      INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | ObjectData   INASSIGN Expression   { HB_MACRO_IFENABLED( $$, hb_compExprAssign( $1, $3 ), HB_SM_HARBOUR ); }
            | ObjectMethod INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            ;

ExprPlusEq  : NumValue     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | NilValue     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | LiteralValue PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | CodeBlock    PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Logical      PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | SelfValue    PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Array        PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ArrayAt      PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Variable     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroVar     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroExpr    PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasVar     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasExpr    PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | VariableAt   PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | PareExpList  PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | IfInline     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | FunCall      PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectData   PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectMethod PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprMinusEq : NumValue     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | NilValue     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | LiteralValue MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | CodeBlock    MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Logical      MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | SelfValue    MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Array        MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ArrayAt      MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Variable     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroVar     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroExpr    MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasVar     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasExpr    MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | VariableAt   MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | PareExpList  MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | IfInline     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | FunCall      MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectData   MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectMethod MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprMultEq  : NumValue     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | NilValue     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | LiteralValue MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | CodeBlock    MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Logical      MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | SelfValue    MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Array        MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ArrayAt      MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Variable     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroVar     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroExpr    MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasVar     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasExpr    MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | VariableAt   MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | PareExpList  MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | IfInline     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | FunCall      MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectData   MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectMethod MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprDivEq   : NumValue     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | NilValue     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | LiteralValue DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | CodeBlock    DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Logical      DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | SelfValue    DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Array        DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ArrayAt      DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Variable     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroVar     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroExpr    DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasVar     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasExpr    DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | VariableAt   DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | PareExpList  DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | IfInline     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | FunCall      DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectData   DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectMethod DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprModEq   : NumValue     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | NilValue     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | LiteralValue MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | CodeBlock    MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Logical      MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | SelfValue    MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Array        MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ArrayAt      MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Variable     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroVar     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroExpr    MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasVar     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasExpr    MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | VariableAt   MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | PareExpList  MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | IfInline     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | FunCall      MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectData   MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectMethod MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprExpEq   : NumValue     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | NilValue     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | LiteralValue EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | CodeBlock    EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Logical      EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | SelfValue    EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Array        EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ArrayAt      EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Variable     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroVar     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroExpr    EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasVar     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasExpr    EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | VariableAt   EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | PareExpList  EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | IfInline     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | FunCall      EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectData   EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectMethod EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprOperEq  : ExprPlusEq      { $$ = $1; }
            | ExprMinusEq     { $$ = $1; }
            | ExprMultEq      { $$ = $1; }
            | ExprDivEq       { $$ = $1; }
            | ExprModEq       { $$ = $1; }
            | ExprExpEq       { $$ = $1; }
            ;

ExprMath    : Expression '+' Expression     { $$ = hb_compExprSetOperand( hb_compExprNewPlus( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '-' Expression     { $$ = hb_compExprSetOperand( hb_compExprNewMinus( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '*' Expression     { $$ = hb_compExprSetOperand( hb_compExprNewMult( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '/' Expression     { $$ = hb_compExprSetOperand( hb_compExprNewDiv( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '%' Expression     { $$ = hb_compExprSetOperand( hb_compExprNewMod( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression POWER Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPower( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprBool    : Expression AND Expression   { $$ = hb_compExprSetOperand( hb_compExprNewAnd( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression OR  Expression   { $$ = hb_compExprSetOperand( hb_compExprNewOr( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprRelation: Expression EQ  Expression   { $$ = hb_compExprSetOperand( hb_compExprNewEQ( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '<' Expression   { $$ = hb_compExprSetOperand( hb_compExprNewLT( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '>' Expression   { $$ = hb_compExprSetOperand( hb_compExprNewGT( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression LE  Expression   { $$ = hb_compExprSetOperand( hb_compExprNewLE( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression GE  Expression   { $$ = hb_compExprSetOperand( hb_compExprNewGE( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression NE1 Expression   { $$ = hb_compExprSetOperand( hb_compExprNewNE( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression NE2 Expression   { $$ = hb_compExprSetOperand( hb_compExprNewNE( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '$' Expression   { $$ = hb_compExprSetOperand( hb_compExprNewIN( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '=' Expression   { $$ = hb_compExprSetOperand( hb_compExprNewEqual( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ArrayIndex : IndexList ']'                   { $$ = $1; }
           ;

/* NOTE: $0 represents the expression before ArrayIndex
 *    Don't use ArrayIndex in other context than as an array index!
 */
IndexList  : '[' Expression               { $$ = hb_compExprNewArrayAt( $<asExpr>0, $2, HB_MACRO_PARAM ); }
           | IndexList ',' Expression     { $$ = hb_compExprNewArrayAt( $1, $3, HB_MACRO_PARAM ); }
           | IndexList ']' '[' Expression { $$ = hb_compExprNewArrayAt( $1, $4, HB_MACRO_PARAM ); }
           ;

ElemList   : EmptyExpression                 { $$ = hb_compExprNewList( $1 ); }
           | ElemList ',' EmptyExpression    { $$ = hb_compExprAddListExpr( $1, $3 ); }
           ;

CodeBlock  : '{' '|'
                  { $<asExpr>$ = hb_compExprNewCodeBlock(NULL,FALSE,FALSE); } BlockNoVar
             '|' BlockExpList '}'
                  { $$ = $<asExpr>3; }
           | '{' '|'
                  { $<asExpr>$ = hb_compExprNewCodeBlock(NULL,FALSE,FALSE); }
             BlockVarList
             '|' BlockExpList '}'
                  { $$ = $<asExpr>3; }
           ;

/* NOTE: This uses $-2 then don't use BlockExpList in other context
 */
BlockExpList : Expression                { $$ = hb_compExprAddCodeblockExpr( $<asExpr>-2, $1 ); }
           | BlockExpList ',' Expression { $$ = hb_compExprAddCodeblockExpr( $<asExpr>-2, $3 ); }
           ;

/* NOTE: This is really not needed however it allows the use of $-2 item
 * in BlockExpList to refer the same rule defined in Codeblock
 */
BlockNoVar : /* empty list */    { $$ = NULL; }
;

BlockVarList : IDENTIFIER                 { $$ = hb_compExprCBVarAdd( $<asExpr>0, $1, HB_MACRO_PARAM ); }
           | BlockVarList ',' IDENTIFIER  { $$ = hb_compExprCBVarAdd( $<asExpr>0, $3, HB_MACRO_PARAM ); HB_MACRO_CHECK( $$ ); }
           ;

ExpList    : '(' EmptyExpression          { $$ = hb_compExprNewList( $2 ); }
           | ExpList ',' EmptyExpression  { $$ = hb_compExprAddListExpr( $1, $3 ); }
;

PareExpList : ExpList ')'                 { $$ = $1; }
;

PareExpListAlias : PareExpList ALIASOP     { $$ = $1; }
;

IfInline   : IIF '(' Expression ',' EmptyExpression ','
             { $<asExpr>$ = hb_compExprAddListExpr( hb_compExprNewList( $3 ), $5 ); }
             EmptyExpression ')'
             { $$ = hb_compExprNewIIF( hb_compExprAddListExpr( $<asExpr>7, $8 ) ); }

           | IF '(' Expression ',' EmptyExpression ','
             { $<asExpr>$ = hb_compExprAddListExpr( hb_compExprNewList( $3 ), $5 ); }
             EmptyExpression ')'
             { $$ = hb_compExprNewIIF( hb_compExprAddListExpr( $<asExpr>7, $8 ) ); }
           ;

%%

/*
 ** ------------------------------------------------------------------------ **
 */

int hb_macroYYParse( HB_MACRO_PTR pMacro )
{
   int iResult;
   void * lexBuffer;

   lexBuffer = hb_compFlexNew( pMacro );

   pMacro->status = HB_MACRO_CONT;
   /* NOTE: bison requires (void *) pointer
    */
   iResult = yyparse( ( void * ) pMacro );

   hb_compFlexDelete( lexBuffer );

   return iResult;
}

/* ************************************************************************* */

void yyerror( char * s )
{
   HB_SYMBOL_UNUSED( s );
}

