/*
 * Copyright (C) 2007 the xine-project
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * System tray icon for gxine.
 */

#include "globals.h"
#include "systray.h"

#include <stdlib.h>

#include "drag_drop.h"
#include "engine.h"
#include "menu.h"
#include "noskin_window.h"
#include "ui.h"

#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <X11/Xlib.h>

static GtkWidget *tray_icon, *tray_menu;
static Window tray = None;

static struct {
  Atom data, manager, opcode, selection;
} atoms;

static void systray_update (void);

static void
send_opcode (glong d1, glong d2, glong d3, glong d4)
{
  GdkDisplay *display = gtk_widget_get_display (tray_icon);
  Display *xdisplay = GDK_DISPLAY_XDISPLAY (display);

  XClientMessageEvent xev = {
    ClientMessage, 0, True,
    xdisplay, tray, atoms.opcode,
    32, { .l = { gdk_x11_get_server_time (tray_icon->window), d1, d2, d3, d4 } }
  };

  gdk_error_trap_push ();
  XSendEvent (xdisplay, tray, False, NoEventMask, (XEvent *) &xev);
  XSync (xdisplay, FALSE);
  gdk_error_trap_pop ();
}

static GdkFilterReturn
systray_filter (GdkXEvent *gxev, GdkEvent *ev, gpointer data)
{
  XEvent *xev = (XEvent *) gxev;

  switch (xev->type)
  {
  case ClientMessage:
    /* tray appeared? */
    if (xev->xclient.message_type == atoms.manager
	&& xev->xclient.data.l[1] == atoms.selection)
      systray_update ();
    break;

  case DestroyNotify:
    /* tray disappeared? */
    if (xev->xclient.window == tray)
      systray_update ();
    break;
  }

  return GDK_FILTER_CONTINUE;
}

static void
systray_update (void)
{
  GdkDisplay *display = gtk_widget_get_display (app);
  Display *xdisplay = GDK_DISPLAY_XDISPLAY (display);

  if (tray != None)
  {
    GdkWindow *window = gdk_window_lookup_for_display (display, tray);
    gdk_window_remove_filter (window, systray_filter, NULL);
  }

  XGrabServer (xdisplay);
  tray = XGetSelectionOwner (xdisplay, atoms.selection);
  if (tray != None)
    XSelectInput (xdisplay, tray, StructureNotifyMask);
  XUngrabServer (xdisplay);
  XFlush (xdisplay);

  if (tray != None)
  {
    GdkWindow *window = gdk_window_lookup_for_display (display, tray);
    gdk_window_add_filter (window, systray_filter, NULL);
    /* request dock */
    send_opcode (0, gtk_plug_get_id (GTK_PLUG (tray_icon)), 0, 0);      
  }
}

/* Standard callback (invokes Javascript) */
#define JS_CB(func, command) \
  static void js_##func##_cb (GtkAction *action, gpointer data) \
  { \
    char *fn = g_strdup_printf (_("menu function %s"), #func); \
    engine_exec ((command), NULL, NULL, fn); \
    free (fn); \
  }

JS_CB (PrevTrack, "playlist_play (playlist_get_item () - 1);")
JS_CB (NextTrack, "playlist_play (playlist_get_item () + 1);")
JS_CB (Play, "play ();")
JS_CB (Pause, "if (!is_live_stream ()) pause ();")
JS_CB (FFwd, "if (!is_live_stream ()) av_speed.v = 5;")
JS_CB (Stop, "stop ();")
JS_CB (Playlist, "playlist_show ();")
JS_CB (Quit, "exit ();")

#define CB(name, stock, label, key) \
  { #name, (stock), (label), (key), NULL, G_CALLBACK(js_##name##_cb) }

static const GtkActionEntry popup_actions[] = {
  CB(PrevTrack,	GTK_STOCK_MEDIA_PREVIOUS,	N_("Pre_vious track"),	NULL),
  CB(Play,	GTK_STOCK_MEDIA_PLAY,		N_("_Play"),		NULL),
  CB(Pause,	GTK_STOCK_MEDIA_PAUSE,		N_("_Pause"),		NULL),
  CB(FFwd,	GTK_STOCK_MEDIA_FORWARD,	N_("_Fast forward"),	NULL),
  CB(Stop,	GTK_STOCK_MEDIA_STOP,		N_("_Stop"),		NULL),
  CB(NextTrack,	GTK_STOCK_MEDIA_NEXT,		N_("_Next track"),	NULL),
  CB(Playlist,	GTK_STOCK_INDEX,		N_("Play_list..."),	NULL),
  CB(Quit,	GTK_STOCK_QUIT,			N_("_Quit"),		NULL),
};

static const char popup_menu_xml[] =
	"<ui><popup name='tray'>\n"
	 "<menuitem action='PrevTrack' />\n"
	 "<menuitem action='Play' />\n"
	 "<menuitem action='Pause' />\n"
	 "<menuitem action='FFwd' />\n"
	 "<menuitem action='Stop' />\n"
	 "<menuitem action='NextTrack' />\n"
	 "<menuitem action='Playlist' />\n"
	 "<separator />\n"
	 "<menuitem action='Quit' />\n"
	"</popup></ui>";

static gboolean
button_cb (GtkWidget *widget, GdkEventButton *ev, gpointer data)
{
  switch (ev->button)
  {
  case 1:
    gtk_menu_popup (GTK_MENU (tray_menu), NULL, NULL, NULL, NULL,
		    ev->button, ev->time);
    break;
  case 2:
    if (GTK_WIDGET_MAPPED (app))
      app_hide ();
    else
      app_show ();
    break;
  case 3:
    gtk_menu_popup (GTK_MENU (popup_menu), NULL, NULL, NULL, NULL,
		    ev->button, ev->time);
    break;
  }

  return TRUE;
}

void systray_init (void)
{
  tray_icon = gtk_plug_new (0);
  gtk_container_add (GTK_CONTAINER (tray_icon),
		     gtk_image_new_from_icon_name ("gxine", GTK_ICON_SIZE_LARGE_TOOLBAR));
  gtk_widget_show_all (tray_icon);
  drag_drop_setup (tray_icon, TRUE);

  GdkScreen *screen = gtk_widget_get_screen (tray_icon);
  Screen *xscreen = GDK_SCREEN_XSCREEN (screen);
  GdkDisplay *display = gtk_widget_get_display (tray_icon);
  Display *xdisplay = GDK_DISPLAY_XDISPLAY (display);

  char selection[32];
  snprintf (selection, sizeof (selection), "_NET_SYSTEM_TRAY_S%d",
	    XScreenNumberOfScreen (xscreen));
  atoms.data = XInternAtom (xdisplay, "_NET_SYSTEM_TRAY_MESSAGE_DATA", False);
  atoms.manager = XInternAtom (xdisplay, "MANAGER", False);
  atoms.opcode = XInternAtom (xdisplay, "_NET_SYSTEM_TRAY_OPCODE", False);
  atoms.selection = XInternAtom (xdisplay, selection, False);

  systray_update ();

  gdk_window_add_filter (gdk_screen_get_root_window (screen),
			 systray_filter, NULL);

  GtkUIManager *st_ui = ui_create_manager ("systray", tray_icon);
  GtkActionGroup *action = ui_get_action_group (st_ui);
  gtk_action_group_add_actions (action, popup_actions,
				G_N_ELEMENTS (popup_actions), NULL);
  GError *error = NULL;
  gtk_ui_manager_add_ui_from_string (st_ui, popup_menu_xml, -1, &error);
  tray_menu = gtk_ui_manager_get_widget (st_ui, "/ui/tray");

  static const char items[][16] = {
    [Control_STOP] = "/ui/tray/Stop",
    [Control_PAUSE] = "/ui/tray/Pause",
    [Control_PLAY] = "/ui/tray/Play",
    [Control_FASTFWD] = "/ui/tray/FFwd",
   };
  for (int i = 0; i < G_N_ELEMENTS (items); ++i)
    if (items[i][0])
      ui_register_control_button (i, gtk_ui_manager_get_widget (st_ui, items[i]));

  gdk_window_set_events (tray_icon->window,
			 gdk_window_get_events (tray_icon->window) | GDK_BUTTON_PRESS_MASK);
  g_signal_connect (G_OBJECT (tray_icon), "button-press-event",
		    G_CALLBACK (button_cb), NULL);
}
