/* 
 * Copyright (C) 2003-2005 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: script_engine.c,v 1.65 2005/11/11 20:51:14 dsalt Exp $
 *
 * gate to the spider monkey javascript interpreter
 *
 * provides functions to generate interpreter instances
 * customized for controlling the xine engine and the gxine
 * frontend. this is done by defining a set of global objects
 * and functions so any xine and gxine function can be invoked from
 * (java)scripts.
 *
 */

#include "globals.h"

#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <pthread.h>
#include <stdarg.h>

#include <gdk/gdkkeysyms.h>
#include <jsstr.h>

#include "script_engine.h"
#include "ui.h"
#include "utils.h"
#include "mediamarks.h"
#include "playlist.h"
#include "settings.h"
#include "preferences.h"
#include "key_events.h"
#include "log_window.h"
#include "stream_info.h"
#include "open_mrl.h"
#include "server.h"
#include "wizards.h"
#include "snapshot.h"
#include "engine.h"
/*
#define LOG
*/
static JSRuntime *rt=NULL; /* global */

static GSList *se_chain;

static GtkListStore *se_error_store;
static GtkWidget *se_window, *eval_box;
static GtkTreeView *tree_view;

static char *eval_text = NULL;

static void se_store_text (const char *type, const char *text)
{
  GtkTreeIter iter;
  GtkTreePath *path;
  gtk_list_store_append (GTK_LIST_STORE (se_error_store), &iter);
  gtk_list_store_set (GTK_LIST_STORE (se_error_store), &iter,
		      0, type, 1, text, -1);
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (se_error_store), &iter);
  gtk_tree_view_scroll_to_cell (tree_view, path, NULL, TRUE, 0, 1);
  gtk_tree_path_free (path);
}

void se_result_cb (void *data, const char *format, ...)
{
  va_list ap;
  char *msg, *text;

  va_start (ap, format);
  msg = g_strdup_vprintf (format, ap);
  va_end (ap);
  text = g_markup_printf_escaped ("%s", msg);
  free (msg);
  se_store_text (GTK_STOCK_DIALOG_INFO, text);
  free (text);
}

void se_error_cb (const char *msg, const JSErrorReport *report)
{
  static const char *const type[] = {
    GTK_STOCK_DIALOG_ERROR, GTK_STOCK_DIALOG_WARNING,
    GTK_STOCK_DIALOG_ERROR, GTK_STOCK_DIALOG_QUESTION
  };

  char *message = g_markup_printf_escaped ("<i>%s</i>", msg);
  char *file = report->filename
	     ? g_markup_printf_escaped (_("<b>Source:</b> %s"), report->filename)
	     : NULL;
  char *line = report->lineno
	     ? g_markup_printf_escaped (_("<b>Line:</b> %d"), report->lineno)
	     : NULL;
  char *stmt = report->linebuf
	     ? g_markup_printf_escaped (_("<b>Line:</b> %s"), report->linebuf)
	     : NULL;

  char *text = g_strconcat (message, file ? "\n" : "", file ? : "",
			    line ? "\n" : "", line ? : "",
			    stmt ? "\n" : "", stmt ? : "", NULL);

  free (message);
  free (line);
  free (file);
  free (stmt);
  se_store_text (type[report->flags & 3], text);
  free (text);
}

int se_eval_ext (se_t *se, const gchar *script, se_o_t *obj,
		 se_print_cb_t print_cb, void *print_cb_data,
		 se_error_cb_t error_cb, const char *src)
{
  se->print_cb      = print_cb;
  se->print_cb_data = print_cb_data;
  se->error_cb      = error_cb;
  return JS_EvaluateScript (se->cx, obj ? obj->obj : se->global, script,
			    strlen (script), src ? : "gxine", 0, &se->rval);
}

int se_eval (se_t *se, const gchar *script, se_o_t *obj,
	     se_print_cb_t print_cb, void *print_cb_data, const char *src)
{
  return se_eval_ext (se, script, obj, print_cb, print_cb_data, se_error_cb,
		      src);
}

gchar *se_result_str (se_t *se)
{
  if (!JSVAL_IS_STRING (se->rval))
    return NULL;

  se->str = JS_ValueToString (se->cx, se->rval);
  return JS_GetStringBytes (se->str);
}

int se_result_int (se_t *se, JSInt32 *num)
{
  if (JSVAL_IS_INT (se->rval))
    return JS_ValueToInt32 (se->cx, se->rval, num);
  return 0;
}

int se_result_double (se_t *se, JSFloat64 *num)
{
  if (JSVAL_IS_DOUBLE (se->rval))
    return JS_ValueToNumber (se->cx, se->rval, num);
  return 0;
}

int se_result_bool (se_t *se, JSBool *num)
{
  if (JSVAL_IS_BOOLEAN (se->rval))
    return JS_ValueToBoolean (se->cx, se->rval, num);
  return 0;
}

int se_result_num_as_int (se_t *se, JSInt32 *num)
{
  if (JSVAL_IS_DOUBLE (se->rval))
  {
    JSFloat64 f;
    JS_ValueToNumber (se->cx, se->rval, &f);
    *num = floor (f);
    return 1;
  }
  return JS_ValueToInt32 (se->cx, se->rval, num);
}

int se_result_num_as_double (se_t *se, JSFloat64 *num)
{
  if (JSVAL_IS_INT (se->rval))
  {
    JSInt32 i;
    JS_ValueToInt32 (se->cx, se->rval, &i);
    *num = i;
    return 1;
  }
  return JS_ValueToNumber (se->cx, se->rval, num);
}

/* 
 * methods
 */

static JSBool controls_exit (JSContext *cx, JSObject *obj, uintN argc, 
			     jsval *argv, jsval *rval)
{
  gchar *fname;

  se_log_fncall_checkinit ("controls.exit");

#if 0
  stop_server();
  printf ("script_engine: server stopped\n");
#endif

  if (gtk_main_level() < 1)
  {
    engine_exec ("settings_clear ();", NULL, NULL, NULL);
    exit (2);
  }

  mm_save (); 
  playlist_save (NULL);
  save_key_bindings ();
  save_startup_script ();

  logprintf ("script_engine: saving config...\n");

#if 0
  {
    xine_cfg_entry_t entry;
    if (xine_config_lookup_entry (xine, "audio.volume.remember_volume", &entry) &&
	entry.num_value &&
	xine_config_lookup_entry (xine, "audio.volume.mixer_volume", &entry))
    {
      entry.num_value = xine_get_param (stream, XINE_PARAM_AUDIO_VOLUME);
      xine_config_update_entry (xine, &entry);
    }
  }
#endif

  fname = g_strconcat(g_get_home_dir(), "/.gxine/config", NULL);
  xine_config_save (xine, fname);
  g_free (fname);

  { /* lie about remembering the volume... */
    xine_cfg_entry_t entry;
    if (xine_config_lookup_entry (xine, "audio.volume.remember_volume", &entry))
    {
      entry.num_value = 0;
      xine_config_update_entry (xine, &entry);
    }
  }
  engine_exec ("settings_clear ();", NULL, NULL, NULL);

  gtk_main_quit();

  exit (0);

  return JS_TRUE;
}

static JSBool show_about (JSContext *cx, JSObject *obj, uintN argc, 
			  jsval *argv, jsval *rval)
{
  static const gchar *const authors[] = {
    "Guenter Bartsch <guenter@users.sourceforge.net>",
    "Darren Salt <dsalt@users.sourceforge.net>",
    NULL
  };
  static const gchar *const documenters[] = {
    "Darren Salt <dsalt@users.sourceforge.net>",
    "Craig Sanders <cas@taz.net.au>",
    "Philipp Matthias Hahn <pmhahn@titan.lahn.de>",
    NULL
  };
  static GtkWidget *about = NULL;

  se_log_fncall_checkinit ("show_about");

  if (!about)
  {
    GtkWidget *label;
    GtkBox *vbox;
    const gchar *translator = _("TRANSLATOR");

    about = gtk_about_dialog_new ();
    g_object_set (G_OBJECT(about),
	"name", "gxine",
	"version", VERSION,
	"copyright", _("© 2002-2005 the xine project team"),
	"comments", _("A GTK media player front end for xine-lib"),
	"license", /* roughly formatted for a typical variable-width font */
	  /* Translators: rewrap this as required */
          _("This program is free software; you can redistribute it and/or\n"
	    "modify it under the terms of the GNU General Public Licence\n"
	    "as published by the Free Software Foundation; either version\n"
	    "2 of the Licence, or (at your option) any later version.\n"
	    "\n"
	    "This program is distributed in the hope that it will be useful,\n"
	    "but WITHOUT ANY WARRANTY; without even the implied\n"
	    "warranty of MERCHANTABILITY or FITNESS FOR A\n"
	    "PARTICULAR PURPOSE. See the GNU General Public\n"
	    "Licence for more details.\n"
	    "\n"
	    "You should have received a copy of the GNU General Public\n"
	    "Licence along with this program; if not, write to the Free\n"
	    "Software Foundation, Inc., 51 Franklin St, Fifth Floor,\n"
	    "Boston, MA  02110-1301  USA\n"),
	"website", "http://xinehq.de/",
	"authors", authors,
	"documenters", documenters,
	"translator-credits",
	  strcmp (translator, "TRANSLATOR") ? translator : NULL,
	"logo-icon-name", GXINE_LOGO,
	NULL);

    label = gtk_label_new ("<small><i>" VENDOR_PKG_VERSION "</i></small>");
    gtk_label_set_use_markup (GTK_LABEL(label), TRUE);
    vbox = GTK_BOX(GTK_DIALOG(about)->vbox);
    gtk_box_pack_end (vbox, label, FALSE, FALSE, 2);
  }

  window_show (about, NULL);
  return JS_TRUE;
}

static JSBool show_js_console (JSContext *cx, JSObject *obj, uintN argc,
			       jsval *argv, jsval *rval)
{
  se_log_fncall_checkinit ("show_js_console");
  if (GTK_WIDGET_VISIBLE (se_window))
    gtk_widget_hide (se_window);
  else
    window_show (se_window, NULL);
  return JS_TRUE;
}

static gboolean
js_callback_execute (gpointer *data)
{
  se_eval_ext (gse, data[0], data[1], NULL, NULL, NULL, _("JS callback"));
  free (data);
  return FALSE;
}

static JSBool
js_callback (JSContext *cx, JSObject *obj,
	     uintN argc, jsval *argv, jsval *rval)
{
  se_t *se = (se_t *) JS_GetContextPrivate (cx);
  gpointer *buf;
  se_log_fncall ("callback");
  se_argc_check_range (1, 2, "callback");
  se_arg_is_string (0, "callback");
  if (argc > 1)
    se_arg_is_object (1, "callback");
  /* need a temporary buffer for two pointers, to script and object */
  buf = calloc (2, sizeof (void *));
  buf[0] = JS_GetStringBytes (JS_ValueToString (cx, argv[0]));
  buf[1] = JS_GetPrivate (cx, argc > 1 ? JSVAL_TO_OBJECT (argv[1])
				       : se->global);
  g_idle_add ((GSourceFunc)js_callback_execute, buf);
  return JS_TRUE;  
}

#if 0
static JSBool dummy (JSContext *cx, JSObject *obj, uintN argc, 
		     jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

  se_log_fncall ("dummy");

  /*
    argc

    JSString *str;
    str = JS_ValueToString (cx, argv[0]);
    mrl = JS_GetStringBytes (str);

    JS_ValueToInt32 (cx, argv[0], &pos);

    se->print_cb (se->print_cb_data, 
		  "error: \n");

    JSCustomer *p = JS_GetPrivate(cx, obj);
    *rval = INT_TO_JSVAL(5);
  */

  return JS_TRUE;
}
#endif

static char *show_help_int (se_t *se, se_o_t *obj, se_group_t selector)
{
  const GList *objs;
  char *help = NULL;

  foreach_glist (objs, obj->functions)
  {
    se_f_t *cmd = objs->data;
    if (cmd->group == selector)
    {
      char *prefix = NULL;
      se_o_t *parent = obj;
      while (parent && strcmp (parent->id, "_global_"))
      {
	asreprintf (&prefix, "%s.%s", parent->id, prefix ? : "");
	parent = parent->parent;
      }
      asreprintf (&help, "%s    %s%s (%s)%s%s%s\n",
		  help ? : "", prefix ? prefix : "", cmd->id,
		  cmd->arghelp ? : "", cmd->funchelp ? "  /* " : "",
		  cmd->funchelp ? : "", cmd->funchelp ? " */" : "");
      free (prefix);
    }
  }

  foreach_glist (objs, obj->children)
  {
    char *group = show_help_int (se, objs->data, selector);
    if (group)
    {
      asreprintf (&help, "%s%s", help ? : "", group);
      free (group);
    }
  }

  foreach_glist (objs, obj->children)
  {
    se_o_t *o = objs->data;
    if (o->group == selector)
    {
      char *prefix = NULL;
      se_o_t *parent = obj;
      while (parent && strcmp (parent->id, "_global_"))
      {
	asreprintf (&prefix, "%s.%s", parent->id, prefix ? : "");
	parent = parent->parent;
      }
      asreprintf (&help, "%s    %s%s\t%s%s%s\n",
		  help ? : "", prefix ? prefix : "", o->id,
		  o->help ? "  /* " : "", o->help ? : "",
		  o->help ? " */" : "");
      free (prefix);
    }
  }

  return help;
}

static JSBool show_help (JSContext *cx, JSObject *obj, uintN argc, 
			  jsval *argv, jsval *rval)
{
  static char *const group_id[] = {
    NULL,
    N_("Engine control:"),
    N_("Dialogue boxes:"),
    N_("Playlist control:"),
    N_("File access:"),
    N_("Status:"),
    N_("Properties:"),
    N_("Input event generation (DVD menus etc.):"),
    N_("External program support:"),
    NULL
  };

  int i;
  se_t *se = (se_t *) JS_GetContextPrivate(cx);

  se_log_fncall_checkinit ("help");

  se->print_cb (se->print_cb_data, "\n%s",
		_("Available commands (gxine Javascript interface):"));

  i = 0;
  while (group_id[++i])
  {
    char *help = NULL;
    const GSList *pse;

    foreach_glist (pse, se_chain)
    {
      char *group = show_help_int (se, ((se_t *)pse->data)->g, i);
      if (group)
      {
	asreprintf (&help, "%s%s  %s\n%s", help ? : "", help ? "\n" : "",
		    gettext (group_id[i]), group);
	free (group);
      }
    }

    if (!help)
      continue;
    help[strlen (help) - 1] = 0;
    se->print_cb (se->print_cb_data, "%s", help);
    free (help);
  }

  /* set_audio_channel (int);
   * set_spu_channel (int);
   * int get_pos ();
   * int get_time ();
   * int get_length ();
   */
  return JS_TRUE;
}

static int get_prop_int (se_t *se, se_o_t *o, se_prop_t *p)
{
  int n;

  if (p->reader)
  {
    se_prop_read_t v;
    p->reader (p, &v);
    return v.i;
  }

  if (sscanf (p->value, "%d", &n) == 1)
  {
    logprintf ("script_engine: this is a simple integer %d\n", n);
    return n;
  }
  if (!strncasecmp (p->value, "jscript:", 8))
  {
    char *script = p->value + 8;
    logprintf ("script_engine: this is a jscript => evaluating '%s'...\n",
               script);
    se_eval (se, script, o, NULL, NULL, "JS property");
    if (se_result_num_as_int (se, &n))
    {
      logprintf ("script_engine: result: %d\n", n);
      return n;
    }
    fprintf (stderr, "script_engine: error, non-numeric result in %s=='%s'",
	     p->id, p->value);
  } else
    fprintf (stderr, "script_engine: error, non-numeric property %s=='%s'",
	     p->id, p->value);

  return 0;
}

static double get_prop_double (se_t *se, se_o_t *o, se_prop_t *p)
{
  double n;
  
  if (p->reader)
  {
    se_prop_read_t v;
    p->reader (p, &v);
    return v.d;
  }

  if (sscanf (p->value, "%lf", &n) == 1)
  {
    logprintf ("script_engine: this is a simple float %lf\n", n);
    return n;
  }
  if (!strncasecmp (p->value, "jscript:", 8))
  {
    char *script = p->value + 8;
    logprintf ("script_engine: this is a jscript => evaluating '%s'...\n",
               script);
    se_eval (se, script, o, NULL, NULL, "JS property");
    if (se_result_num_as_double (se, &n))
    {
      logprintf ("script_engine: result: %lf\n", n);
      return n;
    }
    fprintf (stderr, "script_engine: error, non-numeric result in %s=='%s'",
	     p->id, p->value);
  } else
    fprintf (stderr, "script_engine: error, non-numeric property %s=='%s'",
	     p->id, p->value);

  return 0;
}

static JSBool get_prop_bool (se_t *se, se_o_t *o, se_prop_t *p)
{
  int n;

  if (p->reader)
  {
    se_prop_read_t v;
    p->reader (p, &v);
    return v.i ? JS_TRUE : JS_FALSE;
  }

  if (sscanf (p->value, "%d", &n) == 1)
  {
    n = !!n;
    logprintf ("script_engine: this is a simple boolean %d\n", n);
    return n ? JS_TRUE : JS_FALSE;
  }
  return strcasecmp (p->value, "false") ? JS_TRUE : JS_FALSE;
}

static jsval get_prop_jsval (se_t *se, se_o_t *o, se_prop_t *p)
{
  logprintf ("script_engine: getting value of %s=='%s'\n", p->id, p->value);

  switch (p->se_type)
  {
  case SE_TYPE_INT:
    {
      int n = get_prop_int (se, o, p);
      return INT_TO_JSVAL (n);
    }

  case SE_TYPE_DOUBLE:
    {
      double d = get_prop_int (se, o, p);
      return DOUBLE_TO_JSVAL (d);
    }

  case SE_TYPE_BOOL:
    {
      JSBool b = get_prop_bool (se, o, p);
      return BOOLEAN_TO_JSVAL (b);
    }
 
  case SE_TYPE_STRING:
    {
      JSString *jstr;
      char     *bytes;
      int       len;

      logprintf ("script_engine: this is a string\n");

      len = strlen (p->value);
      bytes = JS_malloc (se->cx, len);
      memcpy (bytes, p->value, len);
       
      jstr = JS_NewString (se->cx, bytes, len);
      return STRING_TO_JSVAL(jstr);
    }

  default:
    fprintf (stderr, "script_engine: error, property %s has unknown type %d\n",
	     p->id, p->se_type);
  }
 
 
  return BOOLEAN_TO_JSVAL (0);
}

static void set_prop_jsval (se_t *se, se_o_t *o, se_prop_t *p, const jsval *v)
{
  logprintf ("script_engine: setting value of %s\n", p->id);

  switch (p->se_type)
  {
  case SE_TYPE_INT:
    if (JSVAL_IS_NUMBER(*v))
      se_prop_set_int (se, o, p->id, JSVAL_TO_INT(*v));
    else
      se->print_cb (se->print_cb_data, _("\n%s.%s: value must be numeric\n"),
		    o->id, p->id);
    return;

  case SE_TYPE_DOUBLE:
    if (JSVAL_IS_NUMBER(*v))
      se_prop_set_double (se, o, p->id, *JSVAL_TO_DOUBLE(*v));
    else
      se->print_cb (se->print_cb_data, _("\n%s.%s: value must be numeric\n"),
		    o->id, p->id);
    return;

  case SE_TYPE_BOOL:
    if (JSVAL_IS_BOOLEAN(*v))
      se_prop_set_bool (se, o, p->id, JSVAL_TO_BOOLEAN(*v));
    else
      se->print_cb (se->print_cb_data, _("\n%s.%s: value must be boolean\n"),
		    o->id, p->id);
    return;
 
  case SE_TYPE_STRING:
    {
      JSString *str = JSVAL_TO_STRING(*v);
      char     *string = JS_GetStringBytes (str);
      se_prop_set (se, o, p->id, string);
    }
    return;

  default:
    fprintf (stderr, "script_engine: error, property %s has unknown type %d\n",
	     p->id, p->se_type);
  }
}
 
  
/*
 * function to create and maintain js objects
 */

JSBool generic_JSGetProperty (JSContext *cx, JSObject *obj, jsval id, jsval *vp) {
 
  se_t   *se = (se_t *) JS_GetContextPrivate(cx);
  se_o_t *o;
 
  se_log_fncall ("generic get property");
 
  o = JS_GetPrivate (cx, obj);

  if (JSVAL_IS_STRING (id)) {
    JSString *str = JS_ValueToString (cx, id);
    char     *prop = str ? JS_GetStringBytes (str) : "";
    GList    *n;

    logprintf ("script_engine: looking for generic property '%s' in '%s'\n",
	       prop, o->id);

    /* look through properties */
 
    foreach_glist (n, o->properties)
    {
      se_prop_t *p = (se_prop_t *) n->data;
      if (!strcasecmp (p->id, prop)) {
        *vp = get_prop_jsval (se, o, p);
        return JS_TRUE;
      }
    }
     
    /* look through objects for default value */
 
    foreach_glist (n, o->children)
    {
      se_o_t *p = (se_o_t *) n->data;
      if (!strcasecmp (p->id, prop))
      {
        static const jschar name[] = { '.', 0 };
	static const struct JSString
	  __attribute__ ((aligned (1 << JSVAL_TAGBITS))) /* ugh */
	  prop = { 1, name };
	if (!generic_JSGetProperty (cx, p->obj, STRING_TO_JSVAL (&prop), vp))
	  *vp = OBJECT_TO_JSVAL (p->obj);
	return JS_TRUE;
      }
    }

    if (*prop && o->parent)
      return generic_JSGetProperty (cx, o->parent->obj, id, vp);
 
    return JS_TRUE;
  }
 
  *vp = JSVAL_VOID;
 
  return JS_FALSE;
}

JSBool generic_JSSetProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
  se_t   *se = (se_t *) JS_GetContextPrivate(cx);
  se_o_t *o;
 
  se_log_fncall ("generic set property");
 
  o = JS_GetPrivate (cx, obj);
 
  if (JSVAL_IS_STRING (id)) {
    JSString *str = JS_ValueToString (cx, id);
    char     *prop = JS_GetStringBytes (str);
    GList    *n;

    logprintf ("script_engine: looking for generic property '%s' in '%s'\n",
	       prop, o->id);

    /* look through properties */
 
    foreach_glist (n, o->properties)
    {
      se_prop_t *p = (se_prop_t *) n->data;
      if (!strcasecmp (p->id, prop)) {
	if (!p->constant)
	  set_prop_jsval (se, o, p, vp);
        return JS_TRUE;
      }
    }

    /* look through objects for default value */
 
    foreach_glist (n, o->children)
    {
      se_o_t *p = (se_o_t *) n->data;
      if (!strcasecmp (p->id, prop))
      {
	if (!generic_JSSetProperty (cx, p->obj, STRING_TO_JSVAL("."), vp))
	  *vp = OBJECT_TO_JSVAL (p->obj);
	return JS_TRUE;
      }
    }

    if (*prop && o->parent)
      return generic_JSSetProperty (cx, o->parent->obj, id, vp);

    return JS_TRUE;
  }
 
  *vp = JSVAL_VOID;
  
  return JS_FALSE;
}

static void generic_JSDestructor (JSContext *cx, JSObject *obj) {
  se_log_fncall ("generic destructor");
}

static JSClass generic_JSClass = 
  {
    "view", JSCLASS_HAS_PRIVATE,
    JS_PropertyStub, JS_PropertyStub,
    generic_JSGetProperty, generic_JSSetProperty,
    JS_EnumerateStub, JS_ResolveStub, 
    JS_ConvertStub, generic_JSDestructor
  };

se_o_t *se_create_object (se_t *se, se_o_t *parent /* may be NULL */,
			  const gchar *name, void *user_data,
			  se_group_t group, const gchar *help)
{
  se_o_t *o;

#ifdef LOG
  { 
    se_o_t *p;
    printf ("script_engine: creating object %s parent:", name);

    p = parent;
    while (p) {
      printf (".%s", p->id);

      p = p->parent; 
    }
    printf ("\n");
  }
#endif  

  o = (se_o_t *) malloc (sizeof (se_o_t));

  if (!parent)
    o->obj = JS_DefineObject (se->cx, se->global, name,
			      &generic_JSClass, NULL, 0);
  else
    o->obj = JS_DefineObject (se->cx, parent->obj, name,
			      &generic_JSClass, NULL, 0);

  o->user_data   = user_data;
  o->parent      = parent;
  o->id          = strdup (name);
  o->children    = NULL;
  o->properties  = NULL;
  o->functions   = NULL;
  o->group	 = group;
  o->help	 = help;

  JS_SetPrivate (se->cx, o->obj, o);

  if (parent)
    parent->children = g_list_append (parent->children, o);
  else
    se->g->children = g_list_append (se->g->children, o);

  return o;
}

se_o_t *se_find_object (se_t *se, se_o_t *parent, const gchar *name)
{
  GList *objs;
  while (strchr (name, '.'))
  {
    int index = strchr (name, '.') - name;
    foreach_glist (objs, parent ? parent->children : se->g->children)
    {
      se_o_t *o = objs->data;
      if (!strncasecmp (o->id, name, index) && !o->id[index])
      {
        name += index + 1;
        parent = o;
        break;
      }
    }
    if (!objs)
      return NULL;
  }
  foreach_glist (objs, parent ? parent->children : se->g->children)
  {
    se_o_t *o = objs->data;
    if (!strcasecmp (o->id, name))
      return o;
  }
  return NULL;
}

static se_prop_t *se_prop_find (se_t *se, se_o_t *o, const char *id)
{
  GList *node;
  foreach_glist (node, o ? o->properties : se->g->properties)
  {
    se_prop_t *p = node->data;
    if (!strcasecmp (p->id, id))
      return p;
  }
  return NULL;
}

static void se_prop_listeners (se_t *se, se_o_t *o, se_prop_t *p, se_prop_read_t r)
{
  GList *node;
  foreach_glist (node, p->listeners)
  {
    const se_prop_listener_t *l = node->data;
    l->cb (l->user_data, se, o, p, r);
  }
}
 
void se_prop_set (se_t *se, se_o_t *o, const char *id, const char *value)
{
  se_prop_t *p;

  if (!o)
    o = se->g;

  p = se_prop_find (se, o, id);
  if (p)
  {
    se_prop_read_t v;
    if (p->reader && p->se_type != SE_TYPE_STRING)
    {
      fprintf (stderr, "script_engine: se_prop_set%s: can't change type of property %s.%s\n",
	       "", o->id, id);
      return;
    }
    free (p->value);
    v.s = p->value = strdup (value);
    p->se_type = SE_TYPE_STRING;
    se_prop_listeners (se, o, p, v);
  }
  else
    se_prop_create (se, o, id, value, NULL, 0, SE_TYPE_STRING, FALSE);
}

void se_prop_set_int (se_t *se, se_o_t *o, const char *id, int value)
{
  se_prop_t *p;

  if (!o)
    o = se->g;

  p = se_prop_find (se, o, id);
  if (p)
  {
    se_prop_read_t v;
    if (p->reader && p->se_type != SE_TYPE_INT)
    {
      fprintf (stderr, "script_engine: se_prop_set%s: can't change type of property %s.%s\n",
	       "_int", o->id, id);
      return;
    }
    free (p->value);
    p->value = g_strdup_printf ("%d", v.i = value);
    p->se_type = SE_TYPE_INT;
    se_prop_listeners (se, o, p, v);
  }
  else
    se_prop_create_int (se, o, id, value, FALSE);
}

void se_prop_set_double (se_t *se, se_o_t *o, const char *id, double value)
{
  se_prop_t *p;

  if (!o)
    o = se->g;

  p = se_prop_find (se, o, id);
  if (p)
  {
    se_prop_read_t v;
    if (p->reader && p->se_type != SE_TYPE_DOUBLE)
    {
      fprintf (stderr, "script_engine: se_prop_set%s: can't change type of property %s.%s\n",
	      "_double", o->id, id);
      return;
    }
    free (p->value);
    p->value = g_strdup_printf ("%la", v.d = value);
    p->se_type = SE_TYPE_DOUBLE;
    se_prop_listeners (se, o, p, v);
  }
  else
    se_prop_create_double (se, o, id, value, FALSE);
}

void se_prop_set_bool (se_t *se, se_o_t *o, const char *id, int value)
{
  se_prop_t *p;

  if (!o)
    o = se->g;

  p = se_prop_find (se, o, id);
  if (p)
  {
    se_prop_read_t v;
    if (p->reader && p->se_type != SE_TYPE_BOOL)
    {
      fprintf (stderr, "script_engine: se_prop_set%s: can't change type of property %s.%s\n",
	       "_bool", o->id, id);
      return;
    }
    free (p->value);
    p->value = g_strdup_printf ("%d", v.i = !!value);
    p->se_type = SE_TYPE_BOOL;
    se_prop_listeners (se, o, p, v);
  }
  else
    se_prop_create_bool (se, o, id, value, FALSE);
}

char *se_prop_get (se_t *se, se_o_t *o, const char *id) {
 
  se_prop_t *p;
 
  if (!o)
    o = se->g;
 
  p = se_prop_find (se, o, id);
 
  if (!p) {
    fprintf (stderr, "script_engine: se_prop_get%s error, property %s.%s doesn't exist\n",
	     "", o->id, id);
    return NULL;
  }

  if (p->reader)
  {
    se_prop_read_t v;
    p->reader (p, &v);
    switch (p->se_type)
    {
    case SE_TYPE_INT:
      free (p->value);
      return p->value = g_strdup_printf ("%d", v.i);
    case SE_TYPE_BOOL:
      free (p->value);
      return p->value = strdup (v.i ? "true" : "false");
    case SE_TYPE_DOUBLE:
      free (p->value);
      return p->value = g_strdup_printf ("%la", v.d);
    case SE_TYPE_STRING:
      free (p->value);
      return p->value = strdup (v.s);
    default:;
    }
  }

  return p->value;
}

int se_prop_get_int (se_t *se, se_o_t *o, const char *id)
{
  se_prop_t *p;

  if (!o)
    o = se->g;

  p = se_prop_find (se, o, id);
  if (!p) {
    fprintf (stderr, "script_engine: se_prop_get%s error, property %s.%s doesn't exist\n",
	     "_int", o->id, id);
    return 0;
  }

  logprintf ("script_engine: getting value of %s.%s=='%s'\n",
	     o->id, p->id, p->value);

  switch (p->se_type)
  {
  case SE_TYPE_INT:
    return get_prop_int (se, o, p);

  case SE_TYPE_DOUBLE:
    return floor (get_prop_double (se, o, p));

  case SE_TYPE_BOOL:
    return get_prop_bool (se, o, p);

  default:
    fprintf (stderr, "script_engine: se_get_prop%s error, %s.%s is not numeric\n",
	     "_int", o->id, id);
  }
  return 0;
}

double se_prop_get_double (se_t *se, se_o_t *o, const char *id)
{
  se_prop_t *p;

  if (!o)
    o = se->g;

  p = se_prop_find (se, o, id);
  if (!p) {
    fprintf (stderr, "script_engine: se_prop_get%s error, property %s.%s doesn't exist\n",
	     "_double", o->id, id);
    return 0;
  }

  logprintf ("script_engine: getting value of %s.%s=='%s'\n",
	     o->id, p->id, p->value);
 
  switch (p->se_type)
  {
  case SE_TYPE_INT:
    return 0.0 + get_prop_int (se, o, p);

  case SE_TYPE_DOUBLE:
    return get_prop_double (se, o, p);

  case SE_TYPE_BOOL:
    return 0.0 + get_prop_bool (se, o, p);

  default:
    fprintf (stderr, "script_engine: se_get_prop%s error, %s.%s is not numeric\n",
	     "_double", o->id, id);
  }
  return 0;
}

JSBool se_prop_get_bool (se_t *se, se_o_t *o, const char *id)
{
  se_prop_t *p;

  if (!o)
    o = se->g;

  p = se_prop_find (se, o, id);
  if (!p) {
    fprintf (stderr, "script_engine: se_prop_get%s error, property %s.%s doesn't exist\n",
	     "_bool", o->id, id);
    return JS_FALSE;
  }

  logprintf ("script_engine: getting value of %s.%s=='%s'\n",
	     o->id, p->id, p->value);
 
  switch (p->se_type)
  {
  case SE_TYPE_INT:
    return get_prop_int (se, o, p) ? JS_TRUE : JS_FALSE;

  case SE_TYPE_DOUBLE:
    return (get_prop_double (se, o, p) != 0) ? JS_TRUE : JS_FALSE;

  case SE_TYPE_BOOL:
    return get_prop_bool (se, o, p);

  default:
    fprintf (stderr, "script_engine: se_get_prop%s error, %s.%s is not numeric\n",
	     "_bool", o->id, id);
  }
  return JS_FALSE;
}

void se_prop_create (se_t *se, se_o_t *o,
                     const char *id, const char *value,
		     const char *xine_id, int xine_param,
		     se_prop_type_t se_type, gboolean constant)
{
  se_prop_t *p;
 
  if (!o)
    o = se->g;

  logprintf ("script_engine: setting %s.%s to '%s'\n", o->id, id, value);

  p = se_prop_find (se, o, id);
  if (!p) {
    logprintf ("script_engine: creating new property %s.%s\n", o->id, id);
    p = malloc (sizeof (se_prop_t));
    p->id = strdup (id);
    p->value = strdup (value);
    p->xine_id = xine_id ? strdup (xine_id) : NULL;
    p->xine_param = xine_param;
    p->listeners = NULL;
    o->properties = g_list_append (o->properties, p);
  }

  p->se_type = se_type;
  p->constant = constant;
}

void se_prop_create_int (se_t *se, se_o_t *o, const char *id, int value,
			 gboolean constant)
{
  char str[24];
  snprintf (str, sizeof (str), "%d", value);
  se_prop_create (se, o, id, str, NULL, 0, SE_TYPE_INT, constant);
}

void se_prop_create_double (se_t *se, se_o_t *o, const char *id, double value, 
			    gboolean constant)
{
  char str[24];
  snprintf (str, sizeof (str), "%la", value); /* C99 hex float */
  se_prop_create (se, o, id, str, NULL, 0, SE_TYPE_DOUBLE, constant);
}

void se_prop_create_bool (se_t *se, se_o_t *o, const char *id, int value, 
			  gboolean constant)
{
  char str[24];
  snprintf (str, sizeof (str), "%d", !!value);
  se_prop_create (se, o, id, str, NULL, 0, SE_TYPE_BOOL, constant);
}

static void se_prop_read_xine_id_int (se_prop_t *prop, se_prop_read_t *value)
{
  xine_cfg_entry_t entry;
  if (xine_config_lookup_entry (xine, prop->xine_id, &entry))
    value->i = entry.num_value;
  else
    value->i = 0;
}

static void se_prop_read_xine_id_string (se_prop_t *prop, se_prop_read_t *value)
{
  xine_cfg_entry_t entry;
  if (xine_config_lookup_entry (xine, prop->xine_id, &entry))
    value->s = entry.str_value;
  else
    value->s = "";
}

void se_prop_create_xine_id (se_t *se, se_o_t *o,
			     const char *id, const char *xine_id)
{
  xine_cfg_entry_t entry = { NULL };
  xine_config_lookup_entry (xine, xine_id, &entry);
  switch (entry.type)
  {
  case XINE_CONFIG_TYPE_RANGE:
  case XINE_CONFIG_TYPE_NUM:
    se_prop_create (se, o, id, ".", xine_id, 0, SE_TYPE_INT, FALSE);
    se_prop_set_reader_func (se, o, id, se_prop_read_xine_id_int);
    return;
  case XINE_CONFIG_TYPE_ENUM:
    puts ("script_engine: can't create property for xine config enum type");
    return;
  case XINE_CONFIG_TYPE_STRING:
    se_prop_create (se, o, id, "", xine_id, 0, SE_TYPE_STRING, FALSE);
    se_prop_set_reader_func (se, o, id, se_prop_read_xine_id_string);
    return;
  case XINE_CONFIG_TYPE_BOOL:
    se_prop_create (se, o, id, "", xine_id, 0, SE_TYPE_BOOL, FALSE);
    se_prop_set_reader_func (se, o, id, se_prop_read_xine_id_int);
    return;
  default:
    puts ("script_engine: can't create property for xine config unknown type");
  }
}

static void se_prop_read_xine_param (se_prop_t *prop, se_prop_read_t *value)
{
  value->i = xine_get_param (stream, prop->xine_param);
}

void se_prop_create_xine_param (se_t *se, se_o_t *o,
			        const char *id, int xine_param,
				se_prop_type_t type)
{
  switch (type)
  {
  case SE_TYPE_BOOL:
  case SE_TYPE_INT:
    se_prop_create (se, o, id, "", NULL, xine_param, type, FALSE);
    se_prop_set_reader_func (se, o, id, se_prop_read_xine_param);
    return;
  default:
    puts ("script_engine: can't create property for xine param non-int type");
  }
}

void se_prop_add_listener (se_t *se, se_o_t *o, const char *id,
			   se_prop_cb_t prop_cb, void *user_data)
{
  se_prop_t *p;
 
  if (!o)
    o = se->g;

  logprintf ("script_engine: adding listener to %s.%s\n", o->id, id);

  p = se_prop_find (se, o, id);
  if (!p)
    fprintf (stderr, "script_engine: error in se_prop_add_listener: property %s.%s not found\n", o->id, id);
  else
  {
    se_prop_listener_t *listener = malloc (sizeof (se_prop_listener_t));
    listener->cb = prop_cb;
    listener->user_data = user_data;
    p->listeners = g_list_append (p->listeners, listener);
  }
}

void se_prop_set_reader_func (se_t *se, se_o_t *o, const char *id,
			      se_prop_reader_t reader)
{
  se_prop_t *p;
 
  if (!o)
    o = se->g;

  logprintf ("script_engine: setting reader for %s.%s\n", o->id, id);

  p = se_prop_find (se, o, id);
  if (!p)
    fprintf (stderr, "script_engine: error in se_prop_set_reader_func: property %s.%s not found\n", o->id, id);
  else
    p->reader = reader;  
}

static void error_reporter (JSContext *cx, const char *message,
			    JSErrorReport *report)
{
  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  if (se->error_cb)
    se->error_cb (message, report);
  if (se->print_cb && se->print_cb != se_result_cb)
    se->print_cb (se->print_cb_data, "%s", message);
  logprintf ("script_engine: JSError '%s'\n", message);
}

se_f_t *se_defun (se_t *se, se_o_t *o /* may be NULL */,
                  const char *name, JSNative fun, uintN nargs, uintN attrs,
                  se_group_t group, const char *arghelp, const char *funchelp)
{
  se_f_t *f;
  f = malloc (sizeof(se_f_t));
  f->id = strdup (name);

  if (o) {
    f->fun = JS_DefineFunction (se->cx, o->obj, name, fun, nargs, attrs);
    o->functions = g_list_append (o->functions, f);
  } else {
    f->fun = JS_DefineFunction (se->cx, se->global, name, fun, nargs, attrs);
    se->g->functions = g_list_append (se->g->functions, f);
  }

  f->group = group;
  f->arghelp = arghelp ? gettext (arghelp) : NULL;
  f->funchelp = funchelp ? gettext (funchelp) : NULL;

  return f;
}

void se_defuns (se_t *se, se_o_t *o /* may be NULL */, const se_f_def_t defs[])
{
  int i = -1;
  while (defs[++i].name)
    se_defun (se, o, defs[i].name, defs[i].func, defs[i].nargs, defs[i].attrs,
	      defs[i].group, defs[i].arghelp, defs[i].funchelp);
}

static void se_execute_cb (GtkCellEditable *cell, gpointer data)
{
  char *js = gtk_editable_get_chars
	       (GTK_EDITABLE (GTK_BIN (data)->child), 0, -1);
  if (js)
  {
    g_strstrip (js);
    if (*js)
    {
      GtkTreeModel *tree = gtk_combo_box_get_model (data);
      gint textcol = gtk_combo_box_entry_get_text_column (data);
      GtkTreeIter iter;
      gboolean found = FALSE;
      if (gtk_tree_model_get_iter_first (tree, &iter))
      {
	do
	{
	  GValue v = {0};
	  const char *str;
	  gtk_tree_model_get_value (tree, &iter, textcol, &v);
	  str = g_value_peek_pointer (&v);
	  if (*str == *js && !strcmp (str, js))
	    found = TRUE;
	  g_value_unset (&v);
	} while (!found && gtk_tree_model_iter_next (tree, &iter));
      }
      if (!found)
	gtk_combo_box_prepend_text (data, js);
      engine_exec (js, se_result_cb, NULL, _("Javascript console"));
    }
    free (js);
  }
  free (eval_text);
  eval_text = NULL;
}

static void se_exec_clear_cb (GtkWidget *widget, gpointer data)
{
  free (eval_text);
  eval_text = NULL;
  gtk_combo_box_set_active (data, -1);
  gtk_list_store_clear ((GtkListStore *)gtk_combo_box_get_model (data));
}

static void se_response_cb (GtkWidget *widget, int response, gpointer data)
{
  switch (response)
  {
  case GTK_RESPONSE_REJECT:
    gtk_list_store_clear (GTK_LIST_STORE (se_error_store));
    break;
  default:
    gtk_widget_hide (se_window);
  }
}

static void
set_eval_iter (GtkScrollType dir)
{
  GtkEditable *edit = GTK_EDITABLE (GTK_BIN (eval_box)->child);
  int item = gtk_combo_box_get_active (GTK_COMBO_BOX (eval_box));
  int prev = item;
  int last = gtk_tree_model_iter_n_children
	       (gtk_combo_box_get_model (GTK_COMBO_BOX (eval_box)), NULL) - 1;

  switch (dir)
  {
  case GTK_SCROLL_STEP_BACKWARD:
    if (item >= 0)
      --item;
    break;

  case GTK_SCROLL_STEP_FORWARD:
    if (item < last)
      ++item;
    break;

  case GTK_SCROLL_START:
    item = 0;
    break;

  case GTK_SCROLL_END:
    item = last;
    break;

  default:
    fprintf (stderr, "gxine: eek, unexpected scroll type %d\n", dir);
    abort ();
  }

  if (prev == item)
    return;

  if (prev < 0)
  {
    free (eval_text);
    eval_text = gtk_editable_get_chars (edit, 0, -1);
  }

  gtk_combo_box_set_active (GTK_COMBO_BOX (eval_box), item);

  if (item < 0)
  {
    gint pos = 0;
    gtk_editable_delete_text (edit, 0, -1);
    if (eval_text)
      gtk_editable_insert_text (edit, eval_text, -1, &pos);
  }
}

static gboolean
console_scroll_cb (GtkWidget *w, GdkEventKey *e, GtkWidget *t)
{
  GtkAdjustment *adj = gtk_tree_view_get_vadjustment (tree_view);
  gdouble value = adj->value;

  switch (e->state & GXINE_MODIFIER_MASK)
  {
  default:
    return FALSE;

  case 0:
    switch (e->keyval)
    {
    default:		return FALSE;
    case GDK_Up:	set_eval_iter (GTK_SCROLL_STEP_BACKWARD); break;
    case GDK_Down:	set_eval_iter (GTK_SCROLL_STEP_FORWARD);  break;
    case GDK_Page_Up:	value -= adj->page_increment; break;
    case GDK_Page_Down:	value += adj->page_increment; break;
    }
    break;

  case GDK_SHIFT_MASK:
    switch (e->keyval)
    {
    default:		return FALSE;
    case GDK_Up:	value -= adj->step_increment; break;
    case GDK_Down:	value += adj->step_increment; break;
    case GDK_Page_Up:	value -= adj->page_increment; break;
    case GDK_Page_Down:	value += adj->page_increment; break;
    }
    break;

  case GDK_CONTROL_MASK:
    switch (e->keyval)
    {
    default:		return FALSE;
    case GDK_Up:	set_eval_iter (GTK_SCROLL_START); return TRUE;
    case GDK_Down:	set_eval_iter (GTK_SCROLL_END); return TRUE;
    case GDK_Page_Up:	value = adj->lower; break;
    case GDK_Page_Down:	value = adj->upper; break;
    }
    break;
  }

  if (value < adj->lower)
    value = adj->lower;
  else if (value > adj->upper - adj->page_size)
    value = adj->upper - adj->page_size;

  gtk_adjustment_set_value (adj, value);
  return TRUE;
}

static void
se_create_console (void)
{
  GtkWidget *w, *b;
  GtkCellRenderer *cell;
  GtkTreeViewColumn *column;

  se_window = gtk_dialog_new_with_buttons (_("Javascript console"), NULL, 0,
				GTK_STOCK_CLEAR, GTK_RESPONSE_REJECT,
				GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
				NULL);
  gtk_window_set_default_size (GTK_WINDOW (se_window), 480, 250);
  g_object_connect (se_window,
	"signal::delete-event", G_CALLBACK (gtk_widget_hide_on_delete), NULL,
	"signal::response", G_CALLBACK (se_response_cb), NULL,
	NULL);

  w = gtk_hbox_new (FALSE, 2);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (se_window)->vbox), w,
		      FALSE, FALSE, 2);

  eval_box = gtk_combo_box_entry_new_text ();
  gtk_box_pack_start_defaults (GTK_BOX (w), eval_box);
  g_signal_connect (GTK_BIN (eval_box)->child, "activate",
		    G_CALLBACK (se_execute_cb), eval_box);

  b = gtk_button_new_from_stock (GTK_STOCK_EXECUTE);
  gtk_box_pack_start (GTK_BOX (w), b, FALSE, FALSE, 0);
  g_signal_connect (b, "clicked", G_CALLBACK (se_execute_cb), eval_box);

  b = ui_button_new_stock_mnemonic (GTK_STOCK_CLEAR, _("_Flush"));
  gtk_box_pack_start (GTK_BOX (w), b, FALSE, FALSE, 0);
  g_signal_connect (b, "clicked", G_CALLBACK (se_exec_clear_cb), eval_box);

  se_error_store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  tree_view = GTK_TREE_VIEW (gtk_tree_view_new_with_model
			       (GTK_TREE_MODEL (se_error_store)));
  gtk_tree_view_set_rules_hint (tree_view, TRUE);
  gtk_tree_view_set_reorderable (tree_view, FALSE);
  gtk_tree_view_set_headers_visible (tree_view, FALSE);

  cell = gtk_cell_renderer_pixbuf_new ();
  g_object_set (cell, "stock-size", (guint) GTK_ICON_SIZE_DIALOG, NULL);
  column = gtk_tree_view_column_new_with_attributes
	     ("", cell, "stock-id", 0, NULL);
  gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN (column),
				   GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_append_column (tree_view, GTK_TREE_VIEW_COLUMN (column));

  cell = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes
	     ("", cell, "markup", 1, NULL);
  gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN (column),
				   GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_append_column (tree_view, GTK_TREE_VIEW_COLUMN (column));

  w = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (w),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (w), GTK_WIDGET (tree_view));

  g_signal_connect (G_OBJECT (eval_box), "key-press-event",
		    G_CALLBACK (console_scroll_cb), w);

  gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (se_window)->vbox), w);
}

se_t *se_new (void)
{
  se_t    *se;
  static JSClass global_class = {
    "global",0,
    JS_PropertyStub,JS_PropertyStub,JS_PropertyStub,JS_PropertyStub,
    JS_EnumerateStub,JS_ResolveStub,JS_ConvertStub,JS_FinalizeStub
  };

  /*
   * create the JS console window
   */

  if (!se_window)
    se_create_console ();

  /*
   * create a js engine instance 
   */

  if (!rt) 
    rt = JS_NewRuntime(0x100000);

  se = malloc (sizeof (se_t));

  se->cx       = JS_NewContext (rt, 0x1000);
  se->global   = JS_NewObject (se->cx, &global_class, NULL, NULL);
  JS_InitStandardClasses (se->cx, se->global);
  JS_SetErrorReporter (se->cx, error_reporter);
  JS_SetContextPrivate (se->cx, se);

  /*
   * create fake global se_o
   */
 
  se->g = malloc (sizeof(se_o_t));
  se->g->obj        = se->global;
  se->g->parent     = NULL;
  se->g->id         = strdup("_global_");
  se->g->children   = NULL;
  se->g->functions  = NULL;
  se->g->properties = NULL;

  se_chain = g_slist_append (se_chain, se);

  {
    static const se_f_def_t defs[] = {
      { "exit", controls_exit, 0, 0, SE_GROUP_ENGINE, NULL, NULL },
      { "about_show", show_about, 0, 0, SE_GROUP_DIALOGUE, NULL, NULL },
      { "js_console_show", show_js_console, 0, 0, SE_GROUP_DIALOGUE, NULL, NULL },
      { "help", show_help, 0, 0, SE_GROUP_HIDDEN, NULL, NULL },
      { "callback", js_callback, 0, 0, SE_GROUP_ENGINE, NULL, NULL },
      { NULL }
    };
    se_defuns (se, se->g, defs);
  }

  return se;
}

JSBool se_warn_initialisation (JSContext *cx, const char *func)
{
  JS_ReportWarning
    (cx, _("Attempt to call function “%s ()” during initialisation"), func);
  return JS_FALSE;
}
