/*
 * Copyright (C) 2001-2005 the xine-project
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: utils.c,v 1.50 2005/01/13 19:53:38 dsalt Exp $
 *
 * needful things, and various UI bits:
 * - button registry and update (state display etc.)
 * - slider/spinner adjustment creation and update
 */

#include "globals.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>

#include <pthread.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include "utils.h"
#include "engine.h"
#include "http.h"
#include "player.h"
#include "playlist.h"
#include "gtkvideo.h"

void ensure_path_exists (char *path) {

  char *sep, *start;

  start = path;
  while ( (sep = strchr (start, '/')) ) {

    *sep = 0;

    printf ("mkdir %s...\n", path);

    mkdir (path, 0755);
    
    *sep = '/';

    start = sep+1;
  }

  printf ("mkdir %s...\n", path);

  mkdir (path, 0755);
}

char *read_entire_file (const char *mrl, int *file_size) {

  char        *buf;
  struct stat  statb;
  int          fd;
  int	       size;

  if (!strncasecmp (mrl, "http://", 7)) 
    return http_download (mrl, file_size);

  if (file_size)
    *file_size = 0;

  if (stat (mrl, &statb) < 0) {
    /*printf ("utils: cannot stat '%s'\n", mrl); */
    return NULL;
  }

  if (S_ISDIR (statb.st_mode))
  {
    errno = EISDIR;
    return NULL;
  }

  size = statb.st_size;

  fd = open (mrl, O_RDONLY);
  if (fd<0)
    return NULL;

  /* two extra bytes to ensure LF+NUL termination */
  buf = malloc (size + 2);
  if (!buf)
    return NULL;

  size = read (fd, buf, size);
  if (size < 0)
  {
    /* read error */
    free (buf);
    return NULL;
  }

  if (file_size)
    *file_size = size;

  if (size && buf[size - 1] != '\n')
    buf[size++] = '\n';
  buf[size] = 0;

  close (fd);

  return buf;
}

GdkPixbuf *load_icon (const char *filename) {

  GdkPixbuf *pix;
  gchar     *pathname;

  pathname = g_strconcat (pixmapdir, "/", filename, NULL);

  pix = gdk_pixbuf_new_from_file (pathname, NULL);

  g_free (pathname);

  return pix;
}

static void response_cb (GtkWidget *dialog, gint response, gpointer data) {
  char *msg;
  
  msg = g_object_get_data (G_OBJECT (dialog), "msg");
  g_free (msg);
  gtk_widget_destroy (dialog);
}


void v_display_message (const gchar *title, GtkMessageType type,
			const gchar *fmt, va_list ap)
{
  GtkWidget *dialog;
  gchar     *msg;
  gboolean   modal = (fmt == NULL);

  if (modal)
    fmt = va_arg (ap, const gchar *);

  msg = g_strdup_vprintf (fmt, ap);
  va_end (ap);

  dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_DESTROY_WITH_PARENT, type,
				   GTK_BUTTONS_CLOSE, msg);
  gtk_window_set_title (GTK_WINDOW (dialog), title);
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);

  if (modal)
    gtk_window_set_modal (GTK_WINDOW(dialog), TRUE);

  g_signal_connect (G_OBJECT (dialog), "response",
		    G_CALLBACK (response_cb), NULL);
  g_object_set_data (G_OBJECT (dialog), "msg", msg);
  gtk_widget_show (dialog);
}


void display_error (const gchar *title, const gchar *fmt, ...)
{
  va_list ap;
  va_start (ap, fmt);
  v_display_message (title, GTK_MESSAGE_ERROR, fmt, ap);
}

void display_warning (const gchar *title, const gchar *fmt, ...)
{
  va_list ap;
  va_start (ap, fmt);
  v_display_message (title, GTK_MESSAGE_WARNING, fmt, ap);
}

void display_info (const gchar *title, const gchar *fmt, ...)
{
  va_list ap;
  va_start (ap, fmt);
  v_display_message (title, GTK_MESSAGE_INFO, fmt, ap);
}

void int_to_timestring (gint int_time, char* string_time, gint length) {
  gint hours, minutes, seconds;

  seconds = int_time % 60;
  int_time /= 60;

  minutes = int_time % 60;
  int_time /= 60;

  hours = int_time;

  snprintf(string_time, length, "%d:%02d:%02d", hours, minutes, seconds);
}

int parse_timestring (const char *string_time) {

  gint hours = 0, minutes = 0, seconds = 0;

  sscanf (string_time, "%d:%d:%d", &hours, &minutes, &seconds);

  return hours * 3600 + minutes * 60 + seconds;
}

static int        answer;

static gboolean ok_cb (GtkFileSelection *file_selector, gpointer user_data) {
  answer = 1;
  gtk_main_quit();
  
  return TRUE;
}

static gboolean cancel_cb (GtkFileSelection *file_selector, gpointer user_data) {
  answer = 0;
  gtk_main_quit();

  return TRUE;
}

static GtkWidget *file_dlg;


static void check_file_origin_path (void)
{
  static int done = 0;
  xine_cfg_entry_t entry;

  if (done)
    return;
  done = 1;
  if (xine_config_lookup_entry (xine, "media.files.origin_path", &entry)
      && entry.type == XINE_CONFIG_TYPE_STRING)
  {
    if (entry.str_value[0] &&
	entry.str_value[strlen (entry.str_value) - 1] != '/')
    {
      char *txt = g_strdup_printf ("%s/", entry.str_value);
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (file_dlg), txt);
      free (txt);
    }
    else
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (file_dlg),
				       entry.str_value);
  }
}


gchar *modal_file_dialog (const char *title, const gchar *pattern) {

  int        was_modal;

  check_file_origin_path ();

  gtk_window_set_title (GTK_WINDOW (file_dlg), title);

  if (pattern)
    gtk_file_selection_complete (GTK_FILE_SELECTION (file_dlg), pattern);
 
  gtk_file_selection_set_select_multiple (GTK_FILE_SELECTION (file_dlg), FALSE);

  answer = 0;

  /* Display that dialog */
  
  gtk_widget_show_all (file_dlg);

  /* run as a modal dialog */

  was_modal = GTK_WINDOW(file_dlg)->modal;
  if (!was_modal) 
    gtk_window_set_modal(GTK_WINDOW(file_dlg),TRUE);

  gtk_main();

  gtk_widget_hide (file_dlg);

  if (answer)
    return g_strdup (gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_dlg)));
  else
    return NULL;
}

gchar *modal_file_dialog_with_buttons (const char *title,
				       const gchar *pattern, ...)
{
  va_list ap, del;
  GtkWidget *widget;
  gchar *ret;

  va_start (ap, pattern);
  va_copy (del, ap);
  while ((widget = va_arg (ap, GtkWidget *)) != NULL)
    gtk_box_pack_end (GTK_BOX(GTK_FILE_SELECTION(file_dlg)->main_vbox),
		      widget, FALSE, FALSE, 2);
  va_end (ap);

  ret = modal_file_dialog (title, pattern);

  while ((widget = va_arg (del, GtkWidget *)) != NULL)
    gtk_container_remove
      (GTK_CONTAINER(GTK_FILE_SELECTION(file_dlg)->main_vbox), widget);
  va_end (del);

  return ret;
}

gchar **modal_multi_file_dialog (const char *title, const gchar *pattern) {

  int        was_modal;

  check_file_origin_path ();

  gtk_window_set_title (GTK_WINDOW (file_dlg), title);

  if (pattern)
    gtk_file_selection_complete (GTK_FILE_SELECTION (file_dlg), pattern);
 
  gtk_file_selection_set_select_multiple (GTK_FILE_SELECTION (file_dlg), TRUE);

  answer = 0;

  /* Display that dialog */
  
  gtk_widget_show_all (file_dlg);

  /* run as a modal dialog */

  was_modal = GTK_WINDOW(file_dlg)->modal;
  if (!was_modal) 
    gtk_window_set_modal(GTK_WINDOW(file_dlg),TRUE);

  gtk_main();

  gtk_widget_hide (file_dlg);

  if (answer)
    return gtk_file_selection_get_selections (GTK_FILE_SELECTION (file_dlg));
  else
    return NULL;
}

static gboolean close_cb (GtkWidget *widget, gpointer data)
{
  if (data)
    *((gboolean *)data) = FALSE;
  gtk_widget_hide (widget);
  return TRUE;
}

void hide_on_delete (GtkWidget *widget, gboolean *visible)
{
  g_signal_connect (GTK_OBJECT(widget), "delete-event",
		    GTK_SIGNAL_FUNC(close_cb), visible);
}

void utils_init (void) {

  /* Create the selector */

  file_dlg = gtk_file_selection_new ("Quaak");
   
  g_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (file_dlg)->ok_button),
		    "clicked",
		    G_CALLBACK (ok_cb),
		    file_dlg);
  
  g_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (file_dlg)->cancel_button),
		    "clicked",
		    G_CALLBACK (cancel_cb),
		    NULL);

  g_signal_connect( GTK_OBJECT (file_dlg), "delete-event",
		    G_CALLBACK (cancel_cb), NULL );
}

void add_table_row_items (GtkWidget *table, gint row,
			  const gchar *labeltext, int widgets, ...)
{
  /* Pads out last widget to fill remaining space.
   * First widget is expandable, rest aren't.
   */
  int column = 1;
  GtkWidget *label = gtk_label_new (labeltext);
  GtkAttachOptions attach = GTK_EXPAND | GTK_SHRINK | GTK_FILL;
  va_list ap;

  gtk_table_attach (GTK_TABLE (table), label, 0, 1, row, row + 1, 0, 0, 2, 2);
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);

  va_start (ap, widgets);
  while (widgets--)
  {
    GtkWidget *widget = va_arg (ap, GtkWidget *);
    gtk_table_attach (GTK_TABLE (table), widget, column,
		      widgets ? column + 1 : GTK_TABLE (table)->ncols,
		      row, row + 1, attach, attach, 2, 2);
    attach = GTK_FILL;
    ++column;
  }
  va_end (ap);
}

char *asreprintf (char **str, const char *fmt, ...)
{
  char *ret;
  va_list ap;

  va_start (ap, fmt);
  ret = g_strdup_vprintf (fmt, ap);
  free (*str);
  return *str = ret;
}

char *unique_name (char *base) {
 
  static int  i = 0;
  char       *name;
 
  name = malloc (strlen(base)+10);
 
  sprintf (name, "%s_%d", base, i);
 
  i++;
 
  return name;
}


/*
 * Playback status
 * UI button & adjustment handling
 */

gboolean no_recursion = FALSE;

gboolean toolbar_at_top = FALSE;
gboolean toolbar_visible = FALSE;

static ui_status_t status = UI_CURRENT_STATE;
pthread_mutex_t status_lock = PTHREAD_MUTEX_INITIALIZER;

static GSList *c_buttons[Control_Buttons] = { NULL };
static GtkObject *c_adjustments[Control_Adjustments] = { NULL };

#define foreach_glist(var,list) for (var = (list); var; var = var->next)

/* (Internal) Reflect play/pause/stop status in control buttons */

static void ui_int_set_status (ui_status_t status)
{
  char states[Control_PlayerButtons] = { 0 };
  int live = player_live_stream () || playlist_showing_logo ();
  int i;

  switch (status)
  {
  case UI_STOP:		states[Control_STOP]	= 1; break;
  case UI_PAUSE:	states[Control_PAUSE]	= 1; break;
  case UI_PLAY_SLOW:	states[Control_PLAY]	= 1;
			states[Control_PAUSE]	= 1; break;
  case UI_PLAY:		states[Control_PLAY]	= 1; break;
  case UI_FAST_FORWARD:	states[Control_FASTFWD]	= 1; break;
  default:;
  }

  no_recursion = TRUE;
  for (i = 0; i < Control_PlayerButtons; ++i)
  {
    GSList *btn;
    foreach_glist (btn, c_buttons[i])
      g_object_set (G_OBJECT(btn->data), "active", states[i], NULL);
    if (i == Control_PAUSE || i == Control_REWIND || i == Control_FASTFWD)
      foreach_glist (btn, c_buttons[i])
	g_object_set (G_OBJECT(btn->data), "sensitive", !live, NULL);
  }
  no_recursion = FALSE;
}

/* (Internal) Reflect mute status in control button */

static void ui_int_set_mute (int mute)
{
  GSList *btn;
  no_recursion = TRUE;
  foreach_glist (btn, c_buttons[Control_MUTE])
    g_object_set (G_OBJECT(btn->data), "active", !mute, NULL);
  no_recursion = FALSE;
}

/* (Internal) Toolbar status */

static void ui_int_toolbar_state (void)
{
  toolbar_item[0][0]->active = toolbar_visible;
  toolbar_item[0][1]->active = toolbar_visible;
  toolbar_item[1][0]->active = !toolbar_at_top;
  toolbar_item[1][1]->active = !toolbar_at_top;
  toolbar_item[2][0]->active = toolbar_at_top;
  toolbar_item[2][1]->active = toolbar_at_top;
}

/* Set/show play/pause/stop/mute status in control buttons */

void ui_set_status (ui_status_t newstatus)
{
  pthread_mutex_lock (&status_lock);


  switch (newstatus)
  {
  case UI_CURRENT_STATE:
    ui_int_set_status (status);
    break;
  case UI_STOP:
  case UI_PAUSE:
  case UI_PLAY_SLOW:
  case UI_PLAY:
  case UI_FAST_FORWARD:
    if (playlist_showing_logo ())
      newstatus = UI_STOP;
    ui_int_set_status (status = newstatus);
    break;
  case UI_AUDIO_UNMUTE:	ui_int_set_mute (FALSE); break;
  case UI_AUDIO_MUTE:	ui_int_set_mute (TRUE); break;
  case UI_TOOLBAR:	ui_int_toolbar_state (); break;
  default:;
  }

  pthread_mutex_unlock (&status_lock);
}

/* Register a button for the above status functions */

void ui_register_control_button (control_button_t item, GtkWidget *widget)
{
  g_return_if_fail (item >= 0 && item < Control_Buttons);
  c_buttons[item] = g_slist_append (c_buttons[item], widget);
}

/* Get the adjustment object for a given adjustment type */

typedef struct {
  gdouble start, min, max, step, page, pagesize;
  const char *setting;
  int param;
} ui_adjustable_t;

static const ui_adjustable_t ranges[] = {
  [Control_SEEKER]		= {     0,       0,  65535,    1,    10, 1, NULL },
  [Control_AUDIO_CHANNEL]	= {    -1,      -1,     32,    1,     1, 1, NULL,		 XINE_PARAM_AUDIO_CHANNEL_LOGICAL },
  [Control_VOLUME]		= {    50,       0,    100,    1,    10, 1, "audio.volume.mixer_volume", XINE_PARAM_AUDIO_VOLUME },
  [Control_COMPRESSOR]		= {   100,     100,   1000,   10,   100, 1, "gui.ao_compressor", XINE_PARAM_AUDIO_COMPR_LEVEL },
  [Control_AMPLIFIER]		= {   100,       0,    200,    1,    10, 1, "gui.ao_amplifier",  XINE_PARAM_AUDIO_AMP_LEVEL },
  [Control_AV_SYNC]		= {     0, -180000, 180000, 1000, 10000, 1, "gui.av_sync",	 XINE_PARAM_AV_OFFSET },
  [Control_HUE]			= { 32768,       0,  65535,  100,  1000, 1, "gui.vo_hue",	 XINE_PARAM_VO_HUE },
  [Control_SATURATION]		= { 32768,       0,  65535,  100,  1000, 1, "gui.vo_saturation", XINE_PARAM_VO_SATURATION },
  [Control_CONTRAST]		= { 32768,       0,  65535,  100,  1000, 1, "gui.vo_contrast",   XINE_PARAM_VO_CONTRAST },
  [Control_BRIGHTNESS]		= { 32768,       0,  65535,  100,  1000, 1, "gui.vo_brightness", XINE_PARAM_VO_BRIGHTNESS },
};
static gdouble starts[sizeof (ranges) / sizeof (ranges[0])];
static gdouble inits[sizeof (ranges) / sizeof (ranges[0])];

/* true if not volume setting or audio.remember_volume is set */
static int check_remember_volume (const ui_adjustable_t *info)
{
  xine_cfg_entry_t entry;
  return info->param != XINE_PARAM_AUDIO_VOLUME ||
         !xine_config_lookup_entry (xine, "audio.volume.remember_volume", &entry) ||
         entry.num_value;
}

static void ui_adjustment_value_changed_cb (GtkAdjustment *adj,
					    const ui_adjustable_t *info)
{
  xine_cfg_entry_t entry;

  if (no_recursion)
    return;

  if (info->param)
    xine_set_param (stream, info->param, adj->value);

  if (info->param == XINE_PARAM_AUDIO_VOLUME)
    return;
  if (info->setting && xine_config_lookup_entry (xine, info->setting, &entry))
  {
    entry.num_value = adj->value;
    xine_config_update_entry (xine, &entry);
  }
}

GtkObject *ui_register_control_adjustment (control_adjustment_t item)
{
  g_return_val_if_fail (item >= 0 && item < Control_Adjustments, NULL);

  if (!c_adjustments[item])
  {
    xine_cfg_entry_t entry;
    int start = ranges[item].start;

    if (ranges[item].setting && check_remember_volume (&ranges[item]) &&
	xine_config_lookup_entry (xine, ranges[item].setting, &entry))
      start = entry.num_value;

    starts[item] = start;

    if (ranges[item].param)
    {
      inits[item] = start = xine_get_param (stream, ranges[item].param);
      if (item != Control_VOLUME)
        xine_set_param (stream, ranges[item].param, start);
    }

    c_adjustments[item] =
      gtk_adjustment_new (start, ranges[item].min, ranges[item].max + 1,
			  ranges[item].step, ranges[item].page,
			  ranges[item].pagesize);
    if (ranges[item].param || ranges[item].setting)
      g_signal_connect (c_adjustments[item], "value-changed",
			G_CALLBACK(ui_adjustment_value_changed_cb),
			(gpointer) &ranges[item]);
  }
  return c_adjustments[item];
}

/* Set the state of every button in a class, e.g. every play button */

void ui_set_control_button (control_button_t item, gboolean state)
{
  GSList *btn;
  g_return_if_fail (item >= Control_PlayerButtons && item < Control_Buttons);
  no_recursion = TRUE;
  if (c_buttons[item] &&
      (GTK_TOGGLE_BUTTON(c_buttons[item]->data))->active != state)
    foreach_glist (btn, c_buttons[item])
      g_object_set (G_OBJECT(btn->data), "active", state, NULL);
  no_recursion = FALSE;
}

/* Set the value of an adjustment */

void ui_set_control_adjustment (control_adjustment_t item, gdouble value)
{
  g_return_if_fail (item >= 0 && item < Control_Adjustments);
  no_recursion = TRUE;
  if ((GTK_ADJUSTMENT(c_adjustments[item]))->value != value)
    g_object_set (G_OBJECT(c_adjustments[item]), "value", value, NULL);
  no_recursion = FALSE;
}

void ui_reset_control_adjustment (control_adjustment_t item)
{
  g_return_if_fail (item >= 0 && item < Control_Adjustments);
  g_object_set (G_OBJECT(c_adjustments[item]), "value",
		ranges[item].start, NULL);
}

void ui_revert_control_adjustment (control_adjustment_t item)
{
  g_return_if_fail (item >= 0 && item < Control_Adjustments);
  g_object_set (G_OBJECT(c_adjustments[item]), "value", starts[item], NULL);
}

void ui_clear_control_adjustment (control_adjustment_t item)
{
  g_return_if_fail (item >= 0 && item < Control_Adjustments);
  xine_set_param (stream, ranges[item].param, inits[item]);
}

/* Update all sliders & spinners which use this adjustment */

void ui_xine_set_param_from_adjustment (control_adjustment_t item)
{
  g_return_if_fail (item >= 0 && item < Control_Adjustments);
  gtk_adjustment_value_changed (GTK_ADJUSTMENT(c_adjustments[item]));
}

void ui_toolbar_toggle (void)
{
  window_toolbar_show (!toolbar_visible);
}

static JSBool js_toolbar_show (JSContext *cx, JSObject *obj, uintN argc, 
			       jsval *argv, jsval *rval)
{
  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  JSBool show;

  se_log_fncall ("toolbar_show");
  se_argc_check_max (1, "toolbar_show");

  if (argc == 1)
  {
    se_arg_is_int (0, "toolbar_show");
    JS_ValueToBoolean (cx, argv[0], &show);
  }
  else
    show = !toolbar_visible;

  if (engine_mutex_trylock (se))
    return JS_TRUE;
  window_toolbar_show (show);
  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool js_toolbar_position (JSContext *cx, JSObject *obj, uintN argc, 
				   jsval *argv, jsval *rval)
{
  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  JSBool top;

  se_log_fncall ("set_toolbar_position");
  se_argc_check_max (1, "set_toolbar_position");

  if (argc == 1)
  {
    se_arg_is_int (0, "set_toolbar_position");
    JS_ValueToBoolean (cx, argv[0], &top);
  }
  else
    top = !toolbar_at_top;

  if (engine_mutex_trylock (se))
    return JS_TRUE;
  window_toolbar_position (top);
  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static void post_deinterlace_plugin_cb (void *data, xine_cfg_entry_t *cfg)
{
  if (gtv)
    gtk_video_set_tvtime (GTK_VIDEO(gtv), cfg->str_value);
}

static void post_plugins_video_cb (void *data, xine_cfg_entry_t *cfg)
{
  if (gtv)
    gtk_video_set_post_plugins_video (GTK_VIDEO(gtv), cfg->str_value);
}

static void post_plugins_audio_cb (void *data, xine_cfg_entry_t *cfg)
{
  if (gtv)
    gtk_video_set_post_plugins_audio (GTK_VIDEO(gtv), cfg->str_value,
				      audio_port);
}

void ui_preferences_register (xine_t *this)
{
  static char *experience_labels[] = {
    N_("Beginner"), N_("Advanced"), N_("Expert"),
    N_("Master of the known universe"), NULL
  };
  static char *tbar_pos_labels[] = {
    N_("Top, hidden"), N_("Bottom, hidden"),
    N_("Top, visible"), N_("Bottom, visible"),
    NULL
  };
  const char **vis_labels, *const *vis_src;
  int i;

  /* Register the experience level setting */
  xine_config_register_enum
    (this, "gui.experience_level", 0, experience_labels,
     _("Display of configuration settings"),
     _("Controls whether more advanced configuration settings are shown."),
     0, NULL, CONFIG_DATA_NONE);

  /* Register a few audio/video config items */
  for (i = 0; i < sizeof (ranges) / sizeof (ranges[0]); ++i)
  {
    if (ranges[i].setting && !strncmp (ranges[i].setting, "gui.", 4))
      xine_config_register_range (this, ranges[i].setting, ranges[i].start,
				  ranges[i].min, ranges[i].max, NULL, NULL,
				  40, NULL, NULL);
    starts[i] = inits[i] = ranges[i].param;
  }

  /* Register some front-end widget options */
  xine_config_register_string
    (this, "gui.post_plugins.deinterlace",
     "tvtime:method=LinearBlend,cheap_mode=1,pulldown=none,use_progressive_frame_flag=1",
     _("Deinterlace plugin name and parameters"),
     _("Format: tvtime:arg=value,arg=value,..."),
     40, post_deinterlace_plugin_cb, CONFIG_DATA_NONE);

  xine_config_register_bool
    (this, "gui.post_plugins.deinterlace_enable", 1,
     _("Enable deinterlacing at startup"), NULL,
     30, NULL, NULL);

  xine_config_register_string
    (this, "gui.post_plugins.video", "",
     _("Video post-processing plugins' names and parameters"),
     _("Format: plugin:arg=value,arg=value,...;plugin:..."),
     40, post_plugins_video_cb, CONFIG_DATA_NONE);

  xine_config_register_bool
    (this, "gui.post_plugins.video_enable", 0,
     _("Enable video post-processing at startup"), NULL,
     30, NULL, NULL);

  xine_config_register_string
    (this, "gui.post_plugins.audio", "",
     _("Audio post-processing plugins' names and parameters"),
     _("Format: plugin:arg=value,arg=value,...;plugin:..."),
     40, post_plugins_audio_cb, CONFIG_DATA_NONE);

  xine_config_register_bool
    (this, "gui.post_plugins.audio_enable", 0,
     _("Enable audio post-processing at startup"), NULL,
     30, NULL, NULL);

  vis_src = xine_list_post_plugins_typed
	      (xine, XINE_POST_TYPE_AUDIO_VISUALIZATION);
  for (i = 0; vis_src[i]; ++i) /**/;
  vis_labels = calloc (i + 2, sizeof (char *));
  vis_labels[0] = N_("None"); /* translate for display, not for config file */
  memcpy (vis_labels + 1, vis_src, (i + 1) * sizeof (char *));

  xine_config_register_enum
    (this, "gui.post_plugins.audio_visualisation", 1, vis_labels,
     _("Default audio visualisation plugin"),
     _("Post-plugin to be used when playing streams without video"),
     30, NULL, NULL);

  xine_config_register_enum
    (this, "gui.fullscreen_toolbar", 1, tbar_pos_labels,
     _("Default position & visibility of the full-screen toolbar"),
     NULL, 0, NULL, NULL);
}

void ui_init (void)
{
  static const se_f_def_t defs[] = {
    { "toolbar_show", js_toolbar_show, 0, 0,
      SE_GROUP_DIALOGUE, N_("[bool]"), NULL },
    { "set_toolbar_position", js_toolbar_position, 0, 0,
      SE_GROUP_DIALOGUE, N_("bool"), N_("at top if true") },
    { NULL }
  };
  xine_cfg_entry_t entry;

  se_defuns (gse, NULL, defs);

  if (xine_config_lookup_entry (xine, "gui.fullscreen_toolbar", &entry))
  {
    toolbar_at_top = !(entry.num_value & 1);
    toolbar_visible = (entry.num_value >> 1) & 1;
  }
}
