/*
 * Copyright (C) 2002-2004 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: playlist.c,v 1.112.2.4 2005/04/26 00:12:30 dsalt Exp $
 *
 * playlist implementation
 *
 * .pls parser stolen from totem (C) 2002 Bastien Nocera
 */

#include "globals.h"

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include <X11/Xlib.h>

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>
#include "xmlparser.h"

#include <pthread.h>

#include "playlist.h"
#include "http.h"
#include "utils.h"
#include "play_item.h"
#include "engine.h"
#include "player.h"
#include "drag_drop.h"

/*
#define LOG
*/

static int             is_visible;
static GtkListStore   *pl_store;
static GtkWidget      *dlg, *tree_view;
static GtkWidget      *repeat_button, *random_button;
static char            logo_mrl[1024];
static int             logo_mode;
static int             cur_list_pos;
static int	       ref_list_pos;

#define unmark_play_item() set_mark_play_item (FALSE)
#define mark_play_item()   set_mark_play_item (TRUE)
static void set_mark_play_item (gboolean state);
static void play_next (void);

static inline void remove_trailing_cr (char *str)
{
  str = strrchr (str, '\r');
  if (str && str[1] == 0)
    *str = 0;
}

static int item_marked_current (GtkTreeIter *iter)
{
  const char *p;
  GValue v;
  int ret;
  memset (&v, 0, sizeof (GValue));
  gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store), iter, 3, &v);
  p = g_value_peek_pointer (&v);
  ret = p && *p;
  g_value_unset (&v);
  return ret;
}

static gboolean close_cb (GtkWidget* widget, gpointer data) {
  is_visible = FALSE;
  gtk_widget_hide (dlg);

  return TRUE;
}

void playlist_clear (void)
{
  if (!logo_mode)
    se_eval (gse, "stop()", NULL, NULL, NULL);
  gtk_list_store_clear (pl_store);
}

static void clear_cb (GtkWidget* widget, gpointer data) {
  playlist_clear ();
  playlist_logo ();
}

static void del_cb (GtkWidget* widget, gpointer data) {

  GtkTreeView       *tree = (GtkTreeView *) data;
  GtkTreeSelection  *sel;
  GtkTreeIter        iter;  

  sel = gtk_tree_view_get_selection (tree);

  if (gtk_tree_selection_get_selected (sel, NULL, &iter)) {

    gint        *indices;
    GtkTreePath *path;
    gint         pos;

    path = gtk_tree_model_get_path (GTK_TREE_MODEL (pl_store), &iter);
    indices = gtk_tree_path_get_indices (path);
    pos = indices[0];

    if (pos == cur_list_pos)
      play_next ();

    if (pos <= cur_list_pos)
      --cur_list_pos;
    gtk_list_store_remove (pl_store, &iter);

    gtk_tree_selection_select_path (sel, path);
    gtk_tree_path_free (path);
  }
}

static void add_cb (GtkWidget* widget, gpointer data) {

  gchar **files;

  files = modal_multi_file_dialog (_("Select files to add to playlist"), NULL);

  if (files) {

    int i;
    gchar *mrl;
    
    i = 0;
    while ((mrl = files[i]) ) {

      playlist_add_mrl (mrl, -1);

      i++;
    }

    g_strfreev (files);

  }
}

static void up_cb (GtkWidget* widget, gpointer data)
{
  GtkTreeIter        iter;
  GtkTreeSelection  *sel;

  sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));

  if (gtk_tree_selection_get_selected (sel, NULL, &iter))
  {
    GtkTreePath *path;
    GtkTreeIter  prev;
    int          pos;

    path = gtk_tree_model_get_path (GTK_TREE_MODEL (pl_store), &iter);
    pos = gtk_tree_path_get_indices (path) [0];
    gtk_tree_path_free (path);

    logprintf ("playlist: got selection, pos=%d\n", pos);

    if (pos < 1)
      return;

    if (pos == cur_list_pos)
      --cur_list_pos; /* we're moving the item being played */
    else if (pos == cur_list_pos + 1)
      ++cur_list_pos; /* we're swapping the item with the one being played */

    gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (pl_store), &prev, NULL,
				   pos - 1);
    gtk_list_store_swap (GTK_LIST_STORE(pl_store), &iter, &prev);
  } else
    logprintf ("playlist: got no selection\n");
}

static void down_cb (GtkWidget* widget, gpointer data)
{
  GtkTreeIter        iter;
  GtkTreeSelection  *sel;

  sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));

  if (gtk_tree_selection_get_selected (sel, NULL, &iter))
  {
    GtkTreePath *path;
    GtkTreeIter  next = iter;
    int          pos;

    path = gtk_tree_model_get_path (GTK_TREE_MODEL(pl_store), &iter);
    pos = gtk_tree_path_get_indices (path) [0];
    gtk_tree_path_free (path);

    logprintf ("playlist: got selection, pos=%d\n", pos);

    if (!gtk_tree_model_iter_next (GTK_TREE_MODEL(pl_store), &next))
      return;

    if (pos == cur_list_pos)
      ++cur_list_pos; /* we're moving the item being played */
    else if (pos == cur_list_pos - 1)
      --cur_list_pos; /* we're swapping the item with the one being played */

    gtk_list_store_swap (GTK_LIST_STORE(pl_store), &iter, &next);
  } else
    logprintf ("playlist: got no selection\n");
}

static void edit_cb (GtkWidget* widget, gpointer data) {

  GtkTreeView       *tree = (GtkTreeView *) data;
  GtkTreeSelection  *sel = gtk_tree_view_get_selection (tree);
  GtkTreeIter        iter;

  if (gtk_tree_selection_get_selected (sel, NULL, &iter)) {
    GValue       v;
    play_item_t *play_item;
    const char  *title;

    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store), &iter, 2, &v);
    play_item = g_value_peek_pointer (&v);

    title = item_marked_current (&iter)
	    ? xine_get_meta_info (stream, XINE_META_INFO_TITLE) : NULL;

    if (play_item_edit (play_item, PLAY_ITEM_LIST, title))
      /* FIXME: what if the MRL is changed? */
      gtk_list_store_set (pl_store, &iter,
			  0, play_item->title,
			  1, play_item->mrl, 2, play_item, -1);

    g_value_unset (&v);
  }
}

static int playlist_load (char *fname) {

  char  *plfile;

  if (!fname)
    fname = g_strconcat(g_get_home_dir(), "/.gxine/playlist", NULL);

  plfile = read_entire_file (fname, NULL);

  if (plfile) {

    xml_node_t *node;

    xml_parser_init (plfile, strlen (plfile), XML_PARSER_CASE_INSENSITIVE);

    if (xml_parser_build_tree (&node)>=0) {

      if (!strcasecmp (node->name, "asx")) {

	playlist_clear ();

	node = node->child;

	while (node) {

	  if (!strcasecmp (node->name, "entry"))
	  {
	    play_item_t *play_item = play_item_load (node->child);
	    GtkTreeIter  iter;

	    gtk_list_store_append (pl_store, &iter);

	    gtk_list_store_set (pl_store, &iter, 0, play_item->title, 1, 
				play_item->mrl, 2, play_item, 3, "", -1);
	  }
	  else if (!strcasecmp (node->name, "settings"))
	  {
	    gtk_toggle_button_set_active
	      (GTK_TOGGLE_BUTTON (random_button),
	       xml_parser_get_property_bool (node, "RANDOM", 0));
	    gtk_toggle_button_set_active
	      (GTK_TOGGLE_BUTTON (repeat_button),
	       xml_parser_get_property_bool (node, "REPEAT", 0));
	  }

	  node = node->next;
	}
      } else {
	display_error (_("Playlist load failed"),
		       _("Loading of playlist file failed:\n"
		       "%s is not an ASX file"), fname);
	return 0;
      }
    } else {
      display_error (_("Playlist load failed"),
		     _("Loading of playlist file failed:\n"
		     "%s is not a valid XML/ASX file"), fname);
      return 0;
    }

  } else {
    if (errno != ENOENT)
      display_error (_("Playlist load failed"),
		     _("Loading of playlist file failed:\n"
		     "Failed to open file '%s'\n%s"), fname, strerror (errno));
    return 0;
  }
  return 1;
}

void playlist_save (char *fname) {

  FILE *f;

  if (!fname)
    fname = g_strconcat(g_get_home_dir(), "/.gxine/playlist", NULL);

  f = fopen (fname, "w");
  if (f)
  {
    static const char tf[][8] = { "false", "true" };

    GtkTreeIter iter;

    fprintf
      (f, "<ASX VERSION=\"3.0\">\n"
          "  <SETTINGS REPEAT=\"%s\" RANDOM=\"%s\"/>\n",
       tf[!!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(repeat_button))],
       tf[!!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(random_button))]);

    if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (pl_store), &iter)) {

      do {
	GValue vi;
	memset (&vi, 0, sizeof (GValue));
	gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store),
				  &iter, 2, &vi);
	play_item_save (g_value_peek_pointer (&vi), f, 1);

	g_value_unset (&vi);
      } while (gtk_tree_model_iter_next (GTK_TREE_MODEL(pl_store), &iter));
    }

    fprintf (f, "</ASX>\n");
    if (ferror (f))
      printf (_("playlist: save failed: %s\n"), strerror (errno));
    if (fclose (f))
      printf (_("playlist: save failed: %s\n"), strerror (errno));
  } else {
    printf (_("playlist: save failed: %s\n"), strerror (errno));
  }
}

static void open_cb (GtkWidget* widget, gpointer data) {

  char *filename;

  filename = modal_file_dialog (_("Select playlist..."), "*.asx");

  if (!filename)
    return;

  playlist_load (filename);
    
}

static void save_as_cb (GtkWidget* widget, gpointer data) {

  char *filename;

  filename = modal_file_dialog (_("Save playlist as..."), "*.asx");

  if (!filename)
    return;

  playlist_save (filename);
}

play_item_t *playlist_get_item (int pos) {

  GtkTreeIter        iter;  

  if (pos<0)
    pos = 0;

  if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (pl_store),
				     &iter, NULL, pos)) {

    GValue       v;
    play_item_t *play_item;
    
    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store),
			      &iter, 2, &v);
    play_item = g_value_peek_pointer (&v);
    g_value_unset (&v);
      
    return play_item;
 } else
   return NULL;
}

int playlist_get_list_pos (void) {
  return cur_list_pos;
}

int playlist_size (void) {
  return gtk_tree_model_iter_n_children (GTK_TREE_MODEL (pl_store), NULL);
}

static int read_ini_line_int (char **lines, const char *key) {
  int retval = -1;
  int i;

  if (lines == NULL || key == NULL)
    return -1;

  for (i = 0; (lines[i] != NULL && retval == -1); i++) {
    if (g_strncasecmp (lines[i], key, strlen (key)) == 0) {
      char **bits;

      bits = g_strsplit (lines[i], "=", 2);
      if (bits[0] == NULL || bits [1] == NULL) {
	g_strfreev (bits);
	return -1;
      }

      retval = (gint) g_strtod (bits[1], NULL);
      g_strfreev (bits);
    }
  }

  return retval;
}

static char* read_ini_line_string (char **lines, const char *key) {
  char *retval = NULL;
  int i;

  if (lines == NULL || key == NULL)
    return NULL;

  for (i = 0; (lines[i] != NULL && retval == NULL); i++) {
    if (g_strncasecmp (lines[i], key, strlen (key)) == 0) {
      char **bits;

      bits = g_strsplit (lines[i], "=", 2);
      if (bits[0] == NULL || bits [1] == NULL) {
	g_strfreev (bits);
	return NULL;
      }

      retval = g_strdup (bits[1]);

      g_strfreev (bits);
    }
  }

  return retval;
}

static int playlist_load_m3u (const char *mrl, gint ins_pos) {
  gboolean  retval = FALSE;
  char     *contents, **lines;
  int       size, i, first;
  char     *extension;
  char     *suffix = NULL;

  first = -1;

  if (! (contents = read_entire_file (mrl, &size)) )
    return -1;

  lines = g_strsplit (contents, "\n", 0);
  g_free (contents);

  for (i = 0; lines[i] != NULL; ++i) {
    remove_trailing_cr (lines[i]);
    /* Either it's a URI, or it has a proper path */
    if (strstr(lines[i], "://") != NULL
	|| lines[i][0] == G_DIR_SEPARATOR)  {
      extension = strrchr(lines[i], '.');
      if (extension && !strncasecmp (extension, ".mp3", 4)) {
          suffix="#demux:mp3";
      }
      else if (extension && !strncasecmp (extension, ".ogg", 4)) {
          suffix="#demux:ogg";
      }

      if (first >= 0) {
	playlist_add_mrl (g_strconcat(lines[i],suffix,NULL), ins_pos);
      }
      else {
	first = playlist_add_mrl (g_strconcat(lines[i],suffix,NULL), ins_pos);
      }

      retval = TRUE;
    }
  }

  g_strfreev (lines);

  if (retval)
    return first;
  else
    return -1;
}

static int playlist_load_smil (const char *mrl, gint ins_pos) {
  gboolean  retval = FALSE;
  char     *contents, **lines;
  int       size, i, first;
  char     *begin, *end;

  first = -1;

  if (! (contents = read_entire_file (mrl, &size)) )
    return -1;

  lines = g_strsplit (contents, "\n", 0);
  g_free (contents);

  for (i = 0; lines[i] != NULL; ++i) {

    if (strstr(lines[i], "<audio src=") != NULL
	|| lines[i][0] == G_DIR_SEPARATOR)  {
      begin=strstr(lines[i], "<audio src=")+12;
      end=strchr(begin, '"');
      if (!end) end=strchr(begin, '\'');
      if (!end) return -1;
      *end=0;
      logprintf("playlist: got '%s'\n", begin);
      if (first >= 0) {
	playlist_add_mrl (begin, ins_pos);
      }
      else {
	first = playlist_add_mrl (begin, ins_pos);
      }

      retval = TRUE;
    }
    if (strstr(lines[i], "<video src=") != NULL
	|| lines[i][0] == G_DIR_SEPARATOR)  {
      begin=strstr(lines[i], "<video src=")+12;
      end=strchr(begin, '"');
      if (!end) end=strchr(begin, '\'');
      if (!end) return -1;
      *end=0;
      logprintf ("playlist: got '%s'\n", begin);
      if (first >= 0) {
	playlist_add_mrl (begin, ins_pos);
      }
      else {
	first = playlist_add_mrl (begin, ins_pos);
      }

      retval = TRUE;
    }

  }

  g_strfreev (lines);

  if (retval)
    return first;
  else
    return -1;
}

int playlist_add (play_item_t *play_item, gint ins_pos) {

  GtkTreeIter  iter;
  GtkTreePath *path;
  gint        *indices;
  gint         ret;

  if (ins_pos<0)
    gtk_list_store_append (pl_store, &iter);
  else
    gtk_list_store_insert (pl_store, &iter, ins_pos);
    
  gtk_list_store_set (pl_store, &iter, 0, play_item->title, 
		      1, play_item->mrl, 2, play_item, 3, "", -1);

  /*
   * find out entry number in list
   */

  path = gtk_tree_model_get_path (GTK_TREE_MODEL (pl_store), &iter);
  indices = gtk_tree_path_get_indices (path);
  ret = indices[0];
  gtk_tree_path_free (path);

  return ret;
}

static gint playlist_add_int (const char *mrl, gint ins_pos)
{
  play_item_t *play_item;
  int i;

  /* don't add duplicate items */
  if (ins_pos < 0)
    for (i = 0; i < playlist_size (); ++i)
      if (!strcmp (playlist_get_item (i)->mrl, mrl))
        return i;

  play_item = play_item_new (NULL, mrl, 0);
  return playlist_add (play_item, ins_pos);
}

static int playlist_load_ram (const char *mrl, gint ins_pos) {
  gboolean  retval = FALSE;
  char      contents[1024], mime_type[256], **lines;
  char      *contents_ptr;
  int       size, i, first;
  int       fd;
  int       pos; 

  first = -1;
  sprintf(mime_type,"none");

  /* get first 4 bytes to decide whether it is a real media
   * file or a playlist.
   */
  size = 4;

  if (!strncasecmp (mrl, "http://", 7)) {
  
    size = http_peek (mrl,size, contents, mime_type);
    if (size <= 0)
      return -1;

  } else {

    fd = open (mrl, O_RDONLY);
    if (fd<0)
      return -1;

    size = read (fd, contents, size);
    close (fd);
  }
  if (contents[0]=='.' 
      && contents[1]=='R' 
      && contents[2]=='M' 
      && contents[3]=='F')
  {
    /* this is a real media file ... */
    /* add it directly to the playlist. */

    pos = playlist_add_int (mrl, ins_pos);

    return pos;
  }
    
  if (! (contents_ptr = read_entire_file (mrl, &size)) )
    return -1;

  lines = g_strsplit (contents_ptr, "\n", 0);
  g_free (contents_ptr);

  for (i = 0; lines[i] != NULL; ++i) {
    
    /* ignore comments */
    if (lines[i][0]=='#') continue;
    
    /* honor --stop-- lines */
    if (strstr(lines[i], "--stop--") != NULL) break;
    
    /* Either it's a rtsp, or a pnm mrl, but we also match http mrls here. */
    if (strstr(lines[i], "rtsp://") != NULL
	|| strstr(lines[i], "pnm://") != NULL
	|| strstr(lines[i], "http://") != NULL)  {
      /* we assume real media */

      remove_trailing_cr (lines[i]);

      if (first >= 0) {
	playlist_add_mrl (lines[i], ins_pos);
      }
      else {
	first = playlist_add_mrl (lines[i], ins_pos);
      }
      retval = TRUE;
    }
    
  }

  g_strfreev (lines);

  if (retval)
    return first;
  else
    return -1;
}

static int playlist_load_pls (const char *mrl, gint ins_pos) {

  gboolean  retval = FALSE;
  char     *contents=NULL, **lines;
  int       size, i, num_entries;
  char      mime_type[256];

  /* http or local file ? */

  if (!strncmp (mrl, "http://", 7)) {

    /* get mime type */
    http_peek(mrl, 0, contents, mime_type);

    if (!strcmp(mime_type, "audio/mpeg")) {
      /* add directly to playlist */

      return playlist_add_int (mrl, ins_pos);
    }

    if (! (contents = http_download (mrl, &size)))
      return -1;
  } else {

    if (! (contents = read_entire_file (mrl, &size)) )
      return -1;
  }

  lines = g_strsplit (contents, "\n", 0);
  g_free (contents);

  /* [playlist] */
  if (g_strncasecmp (lines[0], "[playlist]",
		     (gsize)strlen ("[playlist]")) != 0)
    goto bail;

  /* numberofentries=? */
  num_entries = read_ini_line_int (lines, "numberofentries");
  if (num_entries == -1)
    goto bail;

  for (i = 1; i <= num_entries; i++) {
    char *file, *title;
    char *file_key, *title_key;

    file_key = g_strdup_printf ("file%d", i);
    title_key = g_strdup_printf ("title%d", i);

    file = read_ini_line_string (lines, (const char*)file_key);
    title = read_ini_line_string (lines, (const char*)title_key);

    g_free (file_key);
    g_free (title_key);

    if (file != NULL) {

      int pos = strlen(file)-1;

      while ( (pos>0) && (file[pos]<32) ) {
	file[pos] = 0;
	pos --;
      }

      return playlist_add_mrl (file, ins_pos);

      g_free (file);
      g_free (title);
    } else {
      g_free (title);
    }
  }

bail:
  g_strfreev (lines);

  return retval;
}

static int playlist_load_asx (const char *mrl, gint ins_pos) {

  char             *contents;
  char            **lines;
  int               size, res, ret, i;
  xml_node_t       *xml_tree;
  xml_node_t       *current_node_l1;
  xml_node_t       *current_node_l2;
  xml_property_t   *property;
  int               asx_version;

  ret = -1;

  if (! (contents = read_entire_file (mrl, &size)) )
    return ret;

  xml_parser_init (contents, size, XML_PARSER_CASE_INSENSITIVE);
  res = xml_parser_build_tree (&xml_tree);

  if (res) {
    /* humm, maybe the asx contains no xml? */
    /* try to find mms urls by hand */
    lines = g_strsplit (contents, "\n", 0);
    g_free (contents);

    for (i = 0; lines[i] != NULL; ++i) {
      /* look for mms:// in a line */
      if (strstr(lines[i], "mms://"))  {
        remove_trailing_cr (lines[i]);
        /* add to playlist */
	if (ret >= 0)
          playlist_add_mrl (strstr(lines[i], "mms://"), ins_pos);
	else
	  ret = playlist_add_mrl (strstr(lines[i], "mms://"), ins_pos);
      }
    }
    return ret;
  }

  /* check ASX */
  if (strcmp(xml_tree->name, "ASX") == 0) {
    logprintf("playlist: ASX tag detected\n");

    /* check version */
    asx_version = xml_parser_get_property_int (xml_tree, "VERSION", 3);

    if (asx_version == 3) {

      logprintf("playlist: VERSION 3.0 detected\n");

      /* play each entry */
      current_node_l1 = xml_tree->child;
      while (current_node_l1) {
	/* entry */
	if (strcmp(current_node_l1->name, "ENTRY") == 0) {
	  char *title = NULL;

	  logprintf("playlist: ENTRY detected\n");

	  /* play the first ref which is playable */
	  current_node_l2 = current_node_l1->child;
	  while (current_node_l2) {
	    if (strcmp(current_node_l2->name, "REF") == 0) {
	      logprintf("playlist: REF detected\n");
	      /* find href property */
	      property = current_node_l2->props;
	      while ((property) && (strcmp(property->name, "HREF"))) {
		property = property->next;
	      }
	      if (property) {
		play_item_t *item;

		logprintf("playlist: HREF property detected\n"
			  "playlist: found an mrl: %s\n", property->value);

		item = play_item_new (title, property->value, 0);

		if (ret >= 0)
		  playlist_add (item, ins_pos);
		else
		  ret = playlist_add (item, ins_pos);
		
		/* jump to next entry */
		current_node_l2 = NULL;
		
#if 0 /* FIXME */
		/* try to play REF */
		if (demux_asx_play(this, property->value) == 1) {
		  /* jump to next entry */
		  printf("playlist: play next entry or entryref\n");
		  current_node_l2 = NULL;
		} else {
		  /* try next REF */
		  printf("playlist: try next REF\n");
		}
		
#endif
		
	      } else {
		printf(_("playlist: no HREF property\n"));
	      } /* if (property) */
	      
	    } else if (strcmp(current_node_l2->name, "TITLE") == 0) {
	      title = current_node_l2->data;
	    } else {
	      logprintf("playlist: unknown tag %s detected\n", 
			current_node_l2->name);
	    } /* end if (strcmp(current_node_l2->name, "REF") == 0) */
	    if (current_node_l2) {
	      current_node_l2 = current_node_l2->next;
	    }
	  } /* end while */
	} else {
	  
	  /* entryref */
	  if (strcmp(current_node_l1->name, "ENTRYREF") == 0) {
	    logprintf("playlist: ENTRYREF detected\n");
	    property = current_node_l1->props;
	    while ((property) && (strcmp(property->name, "HREF"))) {
	      property = property->next;
	    }
	    if (property) {
	      logprintf ("playlist: HREF property detected\n"
			 "playlist: found an MRL: %s\n", property->value);

	      if (ret >= 0)
		playlist_add_mrl (property->value, ins_pos);
	      else
		ret = playlist_add_mrl (property->value, ins_pos); 
	      
	      /* jump to next entry */
	      current_node_l2 = NULL;
	      
#if 0 /* FIXME */
	      
	      /* try to play HREF */
	      if (demux_asx_play(this, property->value) == 1) {
		/* jump to next entry */
		printf("playlist: play next entry or entryref\n");
		current_node_l1 = NULL;
	      } else {
		/* try next REF */
		printf("playlist: try next REF\n");
	      }
#endif
	    } else {
	      printf(_("playlist: no HREF property\n"));
	    } /* if (property) */
	  } else {
	    
	    /* title */
	    if (strcmp(current_node_l1->name, "TITLE") == 0) {
	      logprintf("playlist: TITLE detected\n");
	      /* change title */
	    } else {
	      printf(_("playlist: unknown tag %s detected\n"),
		     current_node_l1->name);
	    }
	  }
	}
	current_node_l1 = current_node_l1->next;
      }
      
    } else {
      printf(_("playlist: sorry, ASX version %d not implemented yet\n"),
	     asx_version);
    }
  } else {
    printf(_("playlist: no ASX tag\n"));
  }
  return ret;
}

static int playlist_add_playlist_mrl (const char *mrl, gint ins_pos)
{
  const char *extension = strrchr(mrl, '.');
  if (extension)
  {
    if (!strncasecmp (extension, ".pls", 4))
      return playlist_load_pls (mrl, ins_pos);

    if (!strncasecmp (extension, ".m3u", 4))
      return playlist_load_m3u (mrl, ins_pos);

    if (!strcasecmp (extension, ".ra") ||
	!strcasecmp (extension, ".rm") ||
	!strncasecmp (extension, ".ram", 4) ||
	!strncasecmp (extension, ".rpm", 4) ||
	!strncasecmp (extension, ".lsc", 4) ||
	!strncasecmp (extension, ".pl", 3))
      return playlist_load_ram (mrl, ins_pos);

    if (!strncasecmp (extension, ".asx", 4))
      return playlist_load_asx (mrl, ins_pos);

    if (!strncasecmp (extension, ".smi", 4))
      return playlist_load_smil (mrl, ins_pos);
  }

  return -2; /* not recognised */
}

static int playlist_add_mrl_internal (const char *mrl, gint ins_pos)
{
  /* Helper function. Call only from playlist_add_mrl(). */
  int ret;

  /*
   * is this mrl in fact a simple filename?
   */

  if ( !strstr (mrl, ":/") ) {

    /* is it a playlist file? */
    ret = playlist_add_playlist_mrl (mrl, ins_pos);
    if (ret != -2)
      return ret;

    if (!strncasecmp (mrl, "cdda:", 5))
      return playlist_add_int (mrl, ins_pos);

    if (mrl[0]!='/') {

      /* make filename an absolute pathname */

      char  buf[PATH_MAX];
      char *pathname;
      int   l1, l2;

      getcwd (buf, sizeof (buf));

      l1 = strlen(buf);
      l2 = strlen(mrl);

      pathname = g_malloc (l1+l2+2);
      memcpy (pathname, buf, l1);
      pathname[l1]='/';
      memcpy (&pathname[l1+1], mrl, l2);
      pathname[l1+l2+1]=0;

      return playlist_add_int (pathname, ins_pos);
    }

    return playlist_add_int (mrl, ins_pos);
  }

  if (!strncasecmp (mrl, "http://", 7)) {
    /*
     * could still be a pls/asx playlist, but if so we
     * need to download it first
     */

    /* is it a playlist file? */
    ret = playlist_add_playlist_mrl (mrl, ins_pos);
    if (ret != -2)
      return ret;

    return playlist_add_int (mrl, ins_pos);
  }

  if (!strncasecmp (mrl, "mmshttp://", 10)) {
    logprintf ("playlist: adding regular ASX from mmshttp %s\n", mrl);
    return playlist_load_asx (&mrl[3], ins_pos);
  }

  logprintf ("playlist: adding regular mrl %s\n", mrl);
  return playlist_add_int (mrl, ins_pos);
}

int playlist_add_mrl (const char *mrl, gint ins_pos /* -1 => append */)
{
  static GSList *queued = NULL;
  int ret;

  /* printf ("playlist_add called >%s<\n", mrl); */

  /* handle recursive playlists */
  if (g_slist_find_custom (queued, mrl, (GCompareFunc) strcmp))
  {
    printf (_("recursion in playlist: %s\n"), mrl);
    return 0;
  }

  queued = g_slist_prepend (queued, (char *)mrl);
  ret = playlist_add_mrl_internal (mrl, ins_pos);
  queued = g_slist_remove (queued, mrl);

  return ret;
}

void playlist_play (int list_pos) {
  play_item_t *item;

  if (list_pos<0)
    list_pos = 0;

  item = playlist_get_item (list_pos);
  if (!item) {
    playlist_logo ();
    return;
  }  

  playlist_play_from (list_pos, 0, item->start_time); 
}

static void set_mark_play_item (gboolean state)
{
  if (cur_list_pos >= 0 && cur_list_pos < playlist_size ())
  {
    GtkTreeIter iter;
    gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (pl_store), &iter, NULL,
				   cur_list_pos);
    gtk_list_store_set (pl_store, &iter, 3, state ? "•" : "", -1);
  }
}

void playlist_logo (void) {

  /* int err; */

  unmark_play_item ();
  logo_mode = 1;

  player_launch (NULL, logo_mrl, 0, 0);
}

void playlist_show (void) {

  if (is_visible) {
    is_visible = FALSE;
    gtk_widget_hide (dlg);
  } else {
    is_visible = TRUE;
    gtk_widget_show_all (dlg);
    gtk_tree_view_columns_autosize (GTK_TREE_VIEW(tree_view));
  }
}

int playlist_showing_logo (void)
{
  return logo_mode;
}

static gboolean row_activated_lcb (GtkWidget *widget, GtkTreePath *path,
				   GtkTreeViewColumn *col, gpointer data)
{
  GtkTreeSelection *sel;
  GtkTreeIter iter;

  sel = gtk_tree_view_get_selection (GTK_TREE_VIEW(tree_view));
  if (gtk_tree_selection_get_selected (sel, NULL, &iter))
    playlist_play (gtk_tree_path_get_indices (path)[0]);

  return FALSE;
}

static int has_been_played (int pos) {
  return playlist_get_item (pos)->played;
}

static void play_next (void) {

  int is_repeat, is_random;

  is_repeat = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (repeat_button));
  is_random = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (random_button));

  unmark_play_item ();
  ref_list_pos = -1;

  if (is_random) {

    int count;

    cur_list_pos = rand() % playlist_size();

    count = 0;
    while ( (count<playlist_size()) && has_been_played (cur_list_pos) ) {
      cur_list_pos = (cur_list_pos + 1) % playlist_size();
      
      count++;
    }

    if (has_been_played (cur_list_pos))
      cur_list_pos = playlist_size()+1;

  } else {
    cur_list_pos++;
  }

  if ( is_repeat && (cur_list_pos>=playlist_size())) {
    if (is_random) {
      int i;

      /* reset all played entries */

      for (i=0; i<playlist_size(); i++) {
	play_item_t *item;
	item = playlist_get_item (i);
	item->played = 0;
      }

      cur_list_pos = rand() % playlist_size();
      
    } else {
      cur_list_pos = 0; 
    }
  }

  if (cur_list_pos<playlist_size()) {
    play_item_t *item;
    item = playlist_get_item (cur_list_pos);
    item->played = 1;
    mark_play_item ();
    playlist_play (cur_list_pos);
  } else {
    cur_list_pos = 0;
    playlist_logo ();
  }
}

static void xine_event_cb (void *user_data, const xine_event_t *event) {

  /*
  printf ("playlist: got an event %d\n", event->type); 
  */

  switch (event->type) {
  case XINE_EVENT_UI_PLAYBACK_FINISHED:
    gdk_threads_enter();
    if (!logo_mode)
      play_next ();
    gdk_threads_leave();
    break;

  case XINE_EVENT_PROGRESS:
    {
      xine_progress_data_t *prg = event->data;

      gdk_threads_enter();
      if (prg->percent>99)
	infobar_show_metadata ();
      else {
	struct timeval  tv;
	int             age;

	/* filter out old events */
	gettimeofday (&tv, NULL);
	age = abs (tv.tv_sec - event->tv.tv_sec);
	if (age>0) {
	  gdk_threads_leave();
	  break;
	}

	infobar_line (bar, 0, "%s %d%%", prg->description, prg->percent);
      }
      gdk_flush();
      gdk_threads_leave();
    }
    break;

  case XINE_EVENT_UI_SET_TITLE:
    {
      gdk_threads_enter();
      infobar_show_metadata ();
      playlist_check_set_title ();
      gdk_flush();
      gdk_threads_leave();
    }
    break;

  case XINE_EVENT_MRL_REFERENCE:
    {
      xine_mrl_reference_data_t *ref = (xine_mrl_reference_data_t *) event->data;

      logprintf ("playlist: got a reference event (ref MRL: %s)\n",
	      ref->mrl);

      if (ref->alternative == 0)
      {
	play_item_t *item = play_item_new (NULL, ref->mrl, 0);
	if (ref_list_pos == -1)
	  ref_list_pos = playlist_get_list_pos ();
	playlist_add (item, ++ref_list_pos);
      }
    }
    break;

  case XINE_EVENT_UI_MESSAGE:
    {
      xine_ui_message_data_t *data = (xine_ui_message_data_t *) event->data;
      void (*msg)(const gchar *, const gchar *, ...);

      gdk_threads_enter();

      switch (data->type)
      {
      default:
        puts ("xine_event_cb: unknown XINE_MSG_*");
      case XINE_MSG_AUDIO_OUT_UNAVAILABLE:
      case XINE_MSG_ENCRYPTED_SOURCE:
        msg = display_info;
        break;

      case XINE_MSG_GENERAL_WARNING:
      case XINE_MSG_NETWORK_UNREACHABLE:
      case XINE_MSG_UNKNOWN_HOST:
        msg = display_warning;
        break;

      case XINE_MSG_READ_ERROR:
      case XINE_MSG_LIBRARY_LOAD_ERROR:

      case XINE_MSG_CONNECTION_REFUSED:
      case XINE_MSG_FILE_NOT_FOUND:
      case XINE_MSG_PERMISSION_ERROR:
      case XINE_MSG_SECURITY:
      case XINE_MSG_UNKNOWN_DEVICE:
        msg = display_error;
        break;
      }

      /* TODO: provide customized messages, hints... */
      if (data->num_parameters) {
        int i;
	const char *param = data->messages + strlen (data->messages) + 1;
	char *txt = strdup (data->messages);
	for (i = 0; i < data->num_parameters; ++i)
	{
	  asreprintf (&txt, "%s %s", txt, param);
	  param += strlen (param) + 1;
	}
	msg (_("xine engine message"), "%s", txt);
	free (txt);
      }
      else
        msg (_("xine engine message"), "%s", data->messages);
      gdk_threads_leave();
    }
    break;

  case XINE_EVENT_AUDIO_LEVEL:
    {
      xine_audio_level_data_t *data = event->data;
      gdk_threads_enter ();
      ui_set_control_adjustment (Control_VOLUME,
				 (data->left + data->right) / 2);
      ui_set_control_button (Control_MUTE, !data->mute);
      gdk_threads_leave ();
    }
  }
}

void playlist_check_set_title (void)
{
  const char *title;
  if ((title = xine_get_meta_info (stream, XINE_META_INFO_TITLE)))
  {
    GtkTreeIter iter;
    play_item_t *play_item;
    GValue v;

    gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (pl_store), &iter,
				   NULL, cur_list_pos);

    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store), &iter, 2, &v);
    play_item = g_value_peek_pointer (&v);

    if (play_item->untitled)
    {
      free (play_item->title);
      play_item->title = strdup (title);
      play_item->untitled = FALSE;
      gtk_list_store_set (pl_store, &iter, 0, play_item->title, -1);
    }

    g_value_unset (&v);
  }
}

/*
 * drag and drop
 * FIXME: should subclass GtkListStore
 */

static void drop_cb (GtkTreeView	*widget,
		     GdkDragContext     *context,
		     gint                x,
		     gint                y,
		     GtkSelectionData   *data,
		     guint               info,
		     guint		 time)
{
  gint ins_pos = treeview_get_drop_index (widget, context, x, y);

  logprintf ("drag_drop: drop callback, length=%d, format=%d, pos=%d,%d, insert=%d\n",
	     data->length, data->format, x, y, ins_pos);

  if (data->format == 8)
  {
    /* Drag is from an external source */
    dnd_add_mrls (data, &cur_list_pos, ins_pos);
    gtk_drag_finish (context, TRUE, FALSE, time);
    return;
  }

  if (drag_mrl.index == -1)
  {
    /* below here, we're only interested in intra-app drags */
    gtk_drag_finish (context, FALSE, FALSE, time);
    return;
  }

  if (drag_mrl.model == (GtkTreeModel *)pl_store)
  {
    /* Moving a playlist item */
    GtkTreeIter from, to;

    if (drag_mrl.index == ins_pos)
    {
      /* no need to move the item */
      gtk_drag_finish (context, TRUE, FALSE, time);
      return;
    }

    gtk_tree_model_iter_nth_child (drag_mrl.model, &from, NULL, drag_mrl.index);
    if (ins_pos == -1)
    {
      gtk_list_store_move_before (pl_store, &from, NULL);
      ins_pos = gtk_tree_model_iter_n_children (drag_mrl.model, NULL);
    }
    else
    {
      gtk_tree_model_iter_nth_child (drag_mrl.model, &to, NULL, ins_pos);
      gtk_list_store_move_before (pl_store, &from, &to);
    }

    /* adjust index of currently-being-played item if necessary */
    if (cur_list_pos != -1)
    {
      if (cur_list_pos == drag_mrl.index)
	cur_list_pos = ins_pos;
      if (drag_mrl.index < ins_pos && drag_mrl.index <= cur_list_pos
	  && cur_list_pos <= ins_pos)
	--cur_list_pos;
      if (drag_mrl.index > ins_pos && drag_mrl.index >= cur_list_pos
	  && cur_list_pos >= ins_pos)
	++cur_list_pos;
    }

    gtk_drag_finish (context, TRUE, FALSE, time);
    return;
  }

  gtk_drag_finish (context, FALSE, FALSE, time);
}

/*
 * js functions
 */

static JSBool js_playlist_get_item (JSContext *cx, JSObject *obj, uintN argc, 
				    jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  int   speed;

  se_log_fncall ("playlist_get_item");

  speed = xine_get_param (stream, XINE_PARAM_SPEED);

  *rval = INT_TO_JSVAL (playlist_get_list_pos());

  return JS_TRUE;
}

static JSBool js_playlist_clear (JSContext *cx, JSObject *obj, uintN argc, 
				 jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

  se_log_fncall ("playlist_clear");

  playlist_clear ();

  return JS_TRUE;
}

static JSBool js_playlist_add (JSContext *cx, JSObject *obj, uintN argc, 
			       jsval *argv, jsval *rval) {

  se_t     *se = (se_t *) JS_GetContextPrivate(cx);
  JSString *str;
  char     *mrl;
  int       item;

  se_log_fncall ("playlist_add");

  se_argc_check (1, "playlist_add");
  se_arg_is_string (0, "playlist_add");

  str = JS_ValueToString (cx, argv[0]);
  mrl = JS_GetStringBytes (str);

  logprintf ("playlist_add: MRL=%s\n", mrl);
  item = playlist_add_mrl (mrl, -1);

  *rval = INT_TO_JSVAL (item);

  return JS_TRUE;
}

static JSBool js_playlist_play (JSContext *cx, JSObject *obj, uintN argc, 
				jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   item;

  se_log_fncall ("playlist_play");

  se_argc_check (1, "playlist_play");
  se_arg_is_int (0, "playlist_play");

  JS_ValueToInt32 (cx, argv[0], &item);

  if (item<0)
    item = 0;

  playlist_play (item);
  
  return JS_TRUE;
}

static JSBool js_playlist_show (JSContext *cx, JSObject *obj, uintN argc, 
				jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

  se_log_fncall ("playlist_show");

  playlist_show ();
  
  return JS_TRUE;
}

void playlist_play_from (int list_pos, int pos, int pos_time) {

  /* int          err; */
  GtkTreePath *path;
  char         indices[10];
  play_item_t *play_item;

  logo_mode = 0;

  if (list_pos<0)
    list_pos = 0;

  unmark_play_item ();

  play_item = playlist_get_item (list_pos);

  if (!play_item) {
    cur_list_pos = 0;
    playlist_logo ();
    return;
  }

  play_item_play (play_item);
  
  snprintf (indices, 10, "%d", list_pos);
  path = gtk_tree_path_new_from_string (indices);
  gtk_tree_view_set_cursor (GTK_TREE_VIEW (tree_view), 
			    path, NULL, FALSE);
  gtk_tree_path_free (path);

  cur_list_pos  = list_pos;
  mark_play_item ();


}

void playlist_init (void) {

  GtkWidget            *hbox, *button, *scrolled_window;
  GtkCellRenderer      *cell;
  GtkTreeViewColumn    *column;

  is_visible     = 0;
  cur_list_pos   = -1;

  srand (time (NULL));

  /*
   * window
   */

  dlg = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dlg), _("Edit playlist..."));
  gtk_window_set_default_size (GTK_WINDOW (dlg), 500, 400);
  g_signal_connect (GTK_OBJECT (dlg), "delete-event",
		    G_CALLBACK (close_cb), NULL );

  /*
   * top button bar (load/save/clear)
   */

  hbox = gtk_hbox_new (0, 2);
  
  button = gtk_button_new_from_stock (GTK_STOCK_OPEN);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(open_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX(hbox), button, FALSE, FALSE, 2);
  button = gtk_button_new_from_stock (GTK_STOCK_SAVE_AS);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(save_as_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX(hbox), button, FALSE, FALSE, 2);
  button = gtk_button_new_from_stock (GTK_STOCK_CLEAR);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(clear_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX(hbox), button, FALSE, FALSE, 2);

  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), hbox,
		      FALSE, FALSE, 2);

  /*
   * init tree store
   */
 
  pl_store = gtk_list_store_new (4, G_TYPE_STRING, G_TYPE_STRING,
				 G_TYPE_POINTER, G_TYPE_STRING);

  /*
   * tree view widget to display playlist
   */

  tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL(pl_store));  
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);
  g_signal_connect (G_OBJECT(tree_view), "row-activated",
		    G_CALLBACK(row_activated_lcb), NULL);

  cell = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (" ", cell,
						     "text", 3, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));

  if (GTK_TEST_COMPAT_2_6)
  {
    cell = gtk_cell_renderer_text_new ();
    g_object_set (G_OBJECT(cell),
		  "ellipsize", 2, "width_chars", 16, NULL); /* foo...bar */
  }
  column = gtk_tree_view_column_new_with_attributes (_("Title"), cell,
						     "text", 0, NULL);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN(column), TRUE);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));

  if (GTK_TEST_COMPAT_2_6)
  {
    cell = gtk_cell_renderer_text_new ();
    g_object_set (G_OBJECT(cell),
		  "ellipsize", 1, "width_chars", 16, NULL); /* ...bar */
  }
  column = gtk_tree_view_column_new_with_attributes ("MRL", cell,
						     "text", 1, NULL);
  gtk_tree_view_column_set_sizing (GTK_TREE_VIEW_COLUMN(column),
				   GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));

  treeview_drag_drop_setup (GTK_TREE_VIEW(tree_view), drop_cb);

  gtk_widget_show (tree_view);

  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (scrolled_window), tree_view);

  gtk_box_pack_start (GTK_BOX(GTK_DIALOG (dlg)->vbox), scrolled_window,
		      TRUE, TRUE, 2);

  /*
   * checkboxes (repeat, random)
   */

  hbox = gtk_hbox_new (0,2);
  
  repeat_button = gtk_check_button_new_with_mnemonic (_("_Repeat"));
  gtk_box_pack_start (GTK_BOX (hbox), repeat_button, FALSE, FALSE, 2);

  random_button = gtk_check_button_new_with_mnemonic (_("_Random"));
  gtk_box_pack_start (GTK_BOX (hbox), random_button, FALSE, FALSE, 2);

  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), hbox,
		      FALSE, FALSE, 2);

  /*
   * button bar
   */

  hbox = gtk_hbox_new (1, 2);
  
  button = gtk_button_new_from_stock (GTK_STOCK_ADD);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(add_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 2);
  button = GTK_COMPAT_2_6 (gtk_button_new_from_stock ("gtk-edit"),
			   gtk_button_new_with_mnemonic (_("_Edit")));
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(edit_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 2);
  button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(del_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, TRUE, 2);
  button = gtk_button_new_from_stock (GTK_STOCK_GO_UP);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(up_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 2);
  button = gtk_button_new_from_stock (GTK_STOCK_GO_DOWN);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(down_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 2);

  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), hbox,
		      FALSE, FALSE, 2);

  button = gtk_dialog_add_button (GTK_DIALOG (dlg), GTK_STOCK_CLOSE, 1);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(close_cb), 
		    NULL);

  sprintf (logo_mrl, "%s/logo.mpv", logodir);
  logo_mode = 1;

  playlist_load (NULL);

  /*
   * event handling
   */
  {
    xine_event_queue_t *queue;

    queue = xine_event_new_queue (stream);

    xine_event_create_listener_thread (queue, xine_event_cb, NULL);
  }

  {
    static const se_f_def_t defs[] = {
      { "playlist_show", js_playlist_show, 0, 0,
	SE_GROUP_DIALOGUE, NULL, NULL },
      { "playlist_clear", js_playlist_clear, 0, 0,
	SE_GROUP_PLAYLIST, NULL, NULL },
      { "playlist_add", js_playlist_add, 0, 0,
	SE_GROUP_PLAYLIST, N_("MRL"), NULL },
      { "playlist_play", js_playlist_play, 0, 0,
	SE_GROUP_PLAYLIST, N_("int"), N_("playlist entry number") },
      { "playlist_get_item", js_playlist_get_item, 0, 0,
	SE_GROUP_PLAYLIST, NULL, NULL },
      { NULL }
    };
    se_defuns (gse, NULL, defs);
  }
}
