/*
 * Copyright (C) 2002-2004 the xine-project
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: skin_window.c,v 1.15 2004/12/17 00:33:00 dsalt Exp $
 *
 * cool fully-user-skinned windows, generate from xml/js code
 */

#include "globals.h"

#ifdef EXP_STUFF

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <errno.h>

#include <gdk/gdk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <X11/Xlib.h>

#include "skin_window.h"
#include "utils.h"
#include "xmlparser.h"
#include "script_engine.h"
#include "skin.h"
#include "widget_video.h"

GtkWidget *gtv = NULL;

typedef struct buttongroup_s buttongroup_t;

struct buttongroup_s {

  skin_widget_t  widget;

  GdkPixbuf     *pix, *pix_hover, *pix_down, *pix_hover_down;
  GdkPixbuf     *pix_map;
  
  GList         *buttons;
};

typedef struct button_s button_t;

struct button_s {

  skin_widget_t  widget;

  GdkPixbuf     *pix, *pix_hover, *pix_down, *pix_hover_down;

  gboolean       pressed, hovered;

  char          *on_click;
};

typedef struct {

  buttongroup_t *parent;

  uint32_t       color;
  char          *id;
  char          *script;

  GdkPixbuf     *pix, *pix_hover, *pix_down, *pix_hover_down;

  gboolean       pressed, hovered;

} button_element_t;


#define MAXPATH 1024

static void print_cb (void *user_data, char *str, ...)
{
  va_list ap;
  printf ("skin_window: < ");
  va_start (ap, str);
  vprintf (str, ap);
  puts ("*********************************** ERROR\n");
  abort();
}

static int se_eval_int (char *script, se_o_t *o) {

  if (!script)
    return 0;

  logprintf ("skin_window: evaluating '%s'\n", script);

  if (!strncasecmp (script, "jscript:", 8)) {

    int num;

    se_eval (gse, script, o, print_cb, NULL);

    if ( se_result_num (gse, &num) ) {
      logprintf ("skin_window: result: %d\n", num);
      return num;
    } else
      printf ("skin_window: error, non-numeric result\n");

    return 0;
  } else 
    return atoi (script);
}

gint widgetzcomp (gconstpointer a, gconstpointer b) {

  skin_widget_t *wa, *wb;

  wa = (skin_widget_t *) a;
  wb = (skin_widget_t *) b;

  return wa->zIndex - wb->zIndex;
}

static void my_gdk_pixbuf_render_threshold_alpha (GdkPixbuf *pixbuf,
						  GdkBitmap *bitmap,
						  int src_x,  int src_y,
						  int dest_x, int dest_y,
						  int width,  int height,
						  int alpha_threshold) {
  GdkGC    *gc;
  GdkColor  color;
  int       x, y;
  guchar   *p;
  int       start, start_status;
  int       status;

  if (width == -1) 
    width = gdk_pixbuf_get_width(pixbuf);
  if (height == -1)
    height = gdk_pixbuf_get_height(pixbuf);

  gc = gdk_gc_new (bitmap);

  if (!gdk_pixbuf_get_has_alpha (pixbuf)) {
    color.pixel = (alpha_threshold == 255) ? 0 : 1;
    gdk_gc_set_foreground (gc, &color);
    gdk_draw_rectangle (bitmap, gc, TRUE, dest_x, dest_y, width, height);
    g_object_unref (gc);
    return;
  }

  color.pixel = 0;
  gdk_gc_set_foreground (gc, &color);

  color.pixel = 1;
  gdk_gc_set_foreground (gc, &color);

  for (y = 0; y < height; y++) {
    p = (gdk_pixbuf_get_pixels(pixbuf) 
	 + (y + src_y) * gdk_pixbuf_get_rowstride(pixbuf) 
	 + src_x * gdk_pixbuf_get_n_channels(pixbuf)
	 + gdk_pixbuf_get_n_channels(pixbuf) - 1);
    
    start = 0;
    start_status = *p < alpha_threshold;
    
    for (x = 0; x < width; x++) {
      status = *p < alpha_threshold;
      
      if (status != start_status) {
	if (!start_status)
	  gdk_draw_line (bitmap, gc,
			 start + dest_x, y + dest_y,
			 x - 1 + dest_x, y + dest_y);
	
	start = x;
	start_status = status;
      }
      
      p += gdk_pixbuf_get_n_channels(pixbuf);
    }
    
    if (!start_status)
      gdk_draw_line (bitmap, gc,
		     start + dest_x, y + dest_y,
		     x - 1 + dest_x, y + dest_y);
  }
  
  g_object_unref (gc);
}

static GdkPixbuf *load_pix_trans (xml_node_t *xml, char *attr, char *basename) {

  char      *img_name;
  GdkPixbuf *pix;
  char       filename[MAXPATH];

  img_name = xml_parser_get_property (xml, attr);
  if (!img_name)
    return NULL;

  snprintf (filename, MAXPATH, "%s/%s", basename, img_name);

  pix = gdk_pixbuf_new_from_file (filename, NULL);

  if (!pix) 
    return NULL;
  else {
    char     *transparency_color_str;
    uint32_t  color;

    transparency_color_str = xml_parser_get_property (xml, "transparencyColor");
    if ( transparency_color_str 
	 && (sscanf (transparency_color_str, "#%x", &color) == 1) ) {
      int r, g, b;
      
      r = (color & 0xff0000) >> 16;
      g = (color & 0x00ff00) >>  8;
      b =  color & 0x0000ff ;
      
      pix = gdk_pixbuf_add_alpha (pix, TRUE, r, g, b);
    } 
  }
  return pix;
}

static GdkPixbuf *load_pix (xml_node_t *xml, char *attr, char *basename) {

  char      *img_name;
  GdkPixbuf *pix;
  char       filename[MAXPATH];

  img_name = xml_parser_get_property (xml, attr);
  if (!img_name)
    return NULL;

  snprintf (filename, MAXPATH, "%s/%s", basename, img_name);

  pix = gdk_pixbuf_new_from_file (filename, NULL);

  return pix;
}

static int is_inside (int px, int py, int x, int y, int w, int h) {

  return ( (px>=x) && (py>=y) && (px<=(x+w)) && (py<=(y+h)) );
}

static int is_inside_map (GdkPixbuf *map, uint32_t color, int px, int py) {

  uint8_t *p;
  uint32_t c;

  p = (gdk_pixbuf_get_pixels (map) 
       + py * gdk_pixbuf_get_rowstride (map)
       + px * gdk_pixbuf_get_n_channels (map)) ;

  c = *(p+2) | (*(p+1) << 8) | (*p) << 16;

  return (c==color);
}

static GdkPixbuf *gdk_pixmap_extract_map (GdkPixbuf *pix_src, GdkPixbuf *map,
					  uint32_t color) {

  GdkPixbuf *pix;
  int        x,y,width,height,n_channels, nc;
  uint8_t   *ps, *pd, *pm;
  
  if (!pix_src)
    return NULL;
  if (!map)
    return NULL;

  width      = gdk_pixbuf_get_width      (pix_src);
  height     = gdk_pixbuf_get_height     (pix_src);
  n_channels = gdk_pixbuf_get_n_channels (pix_src);

  pix = gdk_pixbuf_new (gdk_pixbuf_get_colorspace (pix_src),
			TRUE, /* alpha */
			gdk_pixbuf_get_bits_per_sample (pix_src),
			width, height);

  for (y = 0; y < height; y++) {
    int map_color;

    ps = (gdk_pixbuf_get_pixels (pix_src) 
	  + y * gdk_pixbuf_get_rowstride (pix_src)) ;
    pd = (gdk_pixbuf_get_pixels (pix) 
	  + y * gdk_pixbuf_get_rowstride (pix)) ;
    pm = (gdk_pixbuf_get_pixels (map) 
	  + y * gdk_pixbuf_get_rowstride (map)) ;

    for (x = 0; x < width; x++) {

      map_color = *(pm+2) | (*(pm+1) << 8) | (*pm) << 16;
    
      pm += gdk_pixbuf_get_n_channels (map);
    
      for (nc = 0; nc<n_channels; nc++) {
	if (map_color == color)
	  *pd = *ps;
	else
	  *pd = 0x00;
      
	pd++; ps++;
      }

      if (!gdk_pixbuf_get_has_alpha (pix_src)) {
	if (map_color == color)
	  *pd = 0xff;
	else
	  *pd = 0x00;
	pd++; 
      }
    }
  }

  return pix;
}

/*
 *
 * button element stuff
 *
 */

static button_element_t *create_button_element (xml_node_t *xbutton_element, 
						buttongroup_t *group) {

  button_element_t *button_element;
  char             *color_str;

  button_element = (button_element_t *) malloc (sizeof (button_element_t));

  button_element->parent  = group;
  button_element->id      = xml_parser_get_property     (xbutton_element, "id");
  button_element->script  = xml_parser_get_property     (xbutton_element, "onClick");
  
  color_str = xml_parser_get_property (xbutton_element, "mappingColor");

  button_element->color = 0;

  if ( color_str )
    sscanf (color_str, "#%x", &button_element->color);

  button_element->pix = gdk_pixmap_extract_map (group->pix, group->pix_map,
						button_element->color);
  button_element->pix_hover = gdk_pixmap_extract_map (group->pix_hover, group->pix_map,
						button_element->color);
  button_element->pix_down = gdk_pixmap_extract_map (group->pix_down, group->pix_map,
						button_element->color);
  button_element->pix_hover_down = gdk_pixmap_extract_map (group->pix_hover_down, 
							   group->pix_map,
							   button_element->color);

  button_element->pressed = FALSE;
  button_element->hovered = FALSE;

  return button_element;
}


/*
 *
 * buttongroup stuff
 *
 */

static void buttongroup_paint (skin_widget_t *w, int xo, int yo) {
  buttongroup_t  *bg = (buttongroup_t *) w;
  view_t         *view = bg->widget.view;

  if (bg->pix) {

    GList *n;
    
    n = g_list_first (bg->buttons);
    while (n) {
      button_element_t *b;

      b = (button_element_t *) n->data;

      if (b->hovered) {
	
	if (b->pressed) {
#if 0 /* FIXME */
	  if (b->pix_hover_down) {
	    printf ("using hover down image\n");
	    gdk_pixbuf_render_to_drawable (b->pix_hover_down, view->bg, NULL, 0, 0,
					   bg->widget.x+xo, 
					   bg->widget.y+yo,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	  } else 
#endif
	    if (b->pix_down)
	    gdk_pixbuf_render_to_drawable (b->pix_down, view->bg, NULL, 0, 0,
					   bg->widget.x+xo, 
					   bg->widget.y+yo,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	  else if (b->pix)
	    gdk_pixbuf_render_to_drawable (b->pix, view->bg, NULL, 0, 0,
					   bg->widget.x+xo, 
					   bg->widget.y+yo,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	} else { /* !pressed */

	  if (b->pix_hover)
	    gdk_pixbuf_render_to_drawable (b->pix_hover, view->bg, NULL, 0, 0,
					   bg->widget.x+xo, 
					   bg->widget.y+yo,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	  else if (b->pix)
	    gdk_pixbuf_render_to_drawable (b->pix, view->bg, NULL, 0, 0,
					   bg->widget.x+xo, 
					   bg->widget.y+yo,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	}

      } else { /* !hovered */

	if (b->pressed) {
	  if (b->pix_down)
	    gdk_pixbuf_render_to_drawable (b->pix_down, view->bg, NULL, 0, 0,
					   bg->widget.x+xo, 
					   bg->widget.y+yo,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	  else if (b->pix)
	    gdk_pixbuf_render_to_drawable (b->pix, view->bg, NULL, 0, 0,
					   bg->widget.x+xo, 
					   bg->widget.y+yo,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	  
	} else { /* !pressed */

	  if (b->pix)
	    gdk_pixbuf_render_to_drawable (b->pix, view->bg, NULL, 0, 0,
					   bg->widget.x+xo, 
					   bg->widget.y+yo,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	}

      }

      n = g_list_next (n);
    }
  }
}

static void buttongroup_layout (skin_widget_t *w) {
  buttongroup_t *bg = (buttongroup_t *) w;

  if (bg->pix) {
    bg->widget.x  = se_eval_int (bg->widget.xscript, w->se_obj);
    bg->widget.y  = se_eval_int (bg->widget.yscript, w->se_obj);
  } 
}

static gboolean buttongroup_handle_button_press (skin_widget_t *w, int px, int py) {

  buttongroup_t *bg = (buttongroup_t *) w;
  gboolean       is_inside_group;
  GList         *n;

  is_inside_group = is_inside (px, py, w->x, w->y, w->w, w->h);

  n = g_list_first (bg->buttons);
  while (n) {
    button_element_t *b;
    
    b = (button_element_t *) n->data;
    
    if (is_inside_group && is_inside_map (bg->pix_map, b->color, px-w->x, py-w->y)) {

      if (!b->pressed) {
	b->pressed = TRUE;
	bg->widget.view->repaint_needed = TRUE;
      }

    } else {
      
      if (b->pressed) {
	b->pressed = FALSE;
	bg->widget.view->repaint_needed = TRUE;
      }
    }
    
    n = g_list_next (n);
  }

  return is_inside_group;
}

static void buttongroup_handle_button_release (skin_widget_t *w, int px, int py) {

  buttongroup_t *bg = (buttongroup_t *) w;
  gboolean       is_inside_group;
  GList         *n;

  is_inside_group = is_inside (px, py, w->x, w->y, w->w, w->h);

  n = g_list_first (bg->buttons);
  while (n) {
    button_element_t *b;
    
    b = (button_element_t *) n->data;
    
    if (is_inside_group && is_inside_map (bg->pix_map, b->color, px-w->x, py-w->y)) {

      printf ("button from buttongroup released\n");

      printf ("executing '%s' ...\n", b->script);

      se_eval (gse, b->script, w->se_obj, print_cb, NULL);

    }

    if (b->pressed) {
      b->pressed = FALSE;
      bg->widget.view->repaint_needed = TRUE;
    }
    
    n = g_list_next (n);
  }
}

static void buttongroup_handle_motion (skin_widget_t *w, int px, int py) {

  buttongroup_t *bg = (buttongroup_t *) w;
  gboolean       is_inside_group;
  GList         *n;

  is_inside_group= is_inside (px, py, w->x, w->y, w->w, w->h);

  n = g_list_first (bg->buttons);
  while (n) {
    button_element_t *b;
    
    b = (button_element_t *) n->data;
    
    if (is_inside_group && is_inside_map (bg->pix_map, b->color, px-w->x, py-w->y)) {

      if (!b->hovered) {
	b->hovered = TRUE;
	bg->widget.view->repaint_needed = TRUE;
      }
      
    } else {
      
      if (b->hovered) {
	b->hovered = FALSE;
	bg->widget.view->repaint_needed = TRUE;
      }
      
    }
    
    n = g_list_next (n);
  }
}

static buttongroup_t *create_buttongroup (xml_node_t *xbuttongroup, skin_widget_t *parent) {

  buttongroup_t *buttongroup;
  xml_node_t    *node;

  buttongroup = (buttongroup_t *) malloc (sizeof (buttongroup_t));

  buttongroup->widget.parent  = parent;
  buttongroup->widget.view    = parent->view;
  buttongroup->widget.id      = xml_parser_get_property     (xbuttongroup, "id");
  buttongroup->widget.zIndex  = xml_parser_get_property_int (xbuttongroup, "zIndex", 0);

  buttongroup->widget.xscript = xml_parser_get_property     (xbuttongroup, "left");
  buttongroup->widget.yscript = xml_parser_get_property     (xbuttongroup, "top");

  buttongroup->widget.layout                = buttongroup_layout;
  buttongroup->widget.paint                 = buttongroup_paint;
  buttongroup->widget.handle_motion         = buttongroup_handle_motion;
  buttongroup->widget.handle_button_press   = buttongroup_handle_button_press;
  buttongroup->widget.handle_button_release = buttongroup_handle_button_release;

  buttongroup->pix       = load_pix_trans (xbuttongroup, "image", parent->view->basename);
  buttongroup->pix_hover = load_pix_trans (xbuttongroup, "hoverImage", parent->view->basename);
  buttongroup->pix_down  = load_pix_trans (xbuttongroup, "downImage", parent->view->basename);
  buttongroup->pix_hover_down = load_pix_trans (xbuttongroup, "hoverDownImage", 
						parent->view->basename);
  buttongroup->pix_map   = load_pix (xbuttongroup, "mappingImage", parent->view->basename);

  if (buttongroup->pix) {
    buttongroup->widget.w = gdk_pixbuf_get_width  (buttongroup->pix);
    buttongroup->widget.h = gdk_pixbuf_get_height (buttongroup->pix);
  } else {
    buttongroup->widget.w = 0;
    buttongroup->widget.h = 0;
  }

  buttongroup->buttons = NULL;

  if (!buttongroup->widget.id)
    buttongroup->widget.id = unique_name ("buttongroup");

  buttongroup->widget.se_obj = se_create_object (gse, parent->se_obj,
						 buttongroup->widget.id,
						 buttongroup);

  node = xbuttongroup->child;
  while (node) {

    if (!strcasecmp (node->name, "buttonelement")
	|| !strcasecmp (node->name, "playelement")
	|| !strcasecmp (node->name, "nextelement")
	|| !strcasecmp (node->name, "stopelement")
	|| !strcasecmp (node->name, "prevelement")) {

      button_element_t *button_element;

      button_element = create_button_element (node, buttongroup);

      buttongroup->buttons = g_list_append (buttongroup->buttons, button_element);
    } 

    node = node->next;
  }


  return buttongroup;
}

/*
 *
 * button stuff
 *
 */

static void button_paint (skin_widget_t *w, int xo, int yo) {
  button_t  *b = (button_t *) w;
  view_t    *view = b->widget.view;

  if (b->hovered) {
	
    if (b->pressed) {
      if (b->pix_hover_down) {
	gdk_pixbuf_render_to_drawable (b->pix_hover_down, view->bg, NULL, 0, 0,
				       b->widget.x+xo, 
				       b->widget.y+yo,
				       -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
      } else 
	if (b->pix_down)
	  gdk_pixbuf_render_to_drawable (b->pix_down, view->bg, NULL, 0, 0,
					 b->widget.x+xo, 
					 b->widget.y+yo,
					 -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	else if (b->pix)
	  gdk_pixbuf_render_to_drawable (b->pix, view->bg, NULL, 0, 0,
					 b->widget.x+xo, 
					 b->widget.y+yo,
					 -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
    } else { /* !pressed */
      
      if (b->pix_hover)
	gdk_pixbuf_render_to_drawable (b->pix_hover, view->bg, NULL, 0, 0,
				       b->widget.x+xo, 
				       b->widget.y+yo,
				       -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
      else if (b->pix)
	gdk_pixbuf_render_to_drawable (b->pix, view->bg, NULL, 0, 0,
				       b->widget.x+xo, 
				       b->widget.y+yo,
				       -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
    }
    
  } else { /* !hovered */
    
    if (b->pressed) {
      if (b->pix_down)
	gdk_pixbuf_render_to_drawable (b->pix_down, view->bg, NULL, 0, 0,
				       b->widget.x+xo, 
				       b->widget.y+yo,
				       -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
      else if (b->pix)
	gdk_pixbuf_render_to_drawable (b->pix, view->bg, NULL, 0, 0,
				       b->widget.x+xo, 
				       b->widget.y+yo,
				       -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
      
    } else { /* !pressed */
      
      if (b->pix)
	gdk_pixbuf_render_to_drawable (b->pix, view->bg, NULL, 0, 0,
				       b->widget.x+xo, 
				       b->widget.y+yo,
				       -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
    }
  }
}

static void button_layout (skin_widget_t *w) {
  button_t *button = (button_t *) w;

  button->widget.x  = se_eval_int (w->xscript, w->se_obj);
  button->widget.y  = se_eval_int (w->yscript, w->se_obj);
}

static gboolean button_handle_button_press (skin_widget_t *w, int px, int py) {

  button_t *button = (button_t *) w;
  gboolean       is_inside_group;

  is_inside_group = is_inside (px, py, w->x, w->y, w->w, w->h);

  if (button->pressed != is_inside_group) {
    w->view->repaint_needed = TRUE;
    button->pressed = is_inside_group;
  }

  return is_inside_group;
}

static void button_handle_button_release (skin_widget_t *w, int px, int py) {

  button_t *button = (button_t *) w;
  gboolean       is_inside_group;

  is_inside_group = is_inside (px, py, w->x, w->y, w->w, w->h);

  if (is_inside_group) {
    if (button->pressed) {
      
      button->pressed = FALSE;
      w->view->repaint_needed = TRUE;

      printf ("button '%s' released\n", w->id);

      printf ("executing '%s' ...\n", button->on_click);
      
      se_eval (gse, button->on_click, w->se_obj, print_cb, NULL);
    }
  }
}

static void button_handle_motion (skin_widget_t *w, int px, int py) {

  button_t      *button = (button_t *) w;
  gboolean       is_inside_group;

  is_inside_group= is_inside (px, py, w->x, w->y, w->w, w->h);

  if (button->hovered != is_inside_group) {
    w->view->repaint_needed = TRUE;
    button->hovered = is_inside_group;
  }
}

static button_t *create_button (xml_node_t *xbutton, skin_widget_t *parent) {

  button_t *button;

  button = (button_t *) malloc (sizeof (button_t));

  button->widget.parent  = parent;
  button->widget.view    = parent->view;
  button->widget.id      = xml_parser_get_property     (xbutton, "id");
  button->on_click       = xml_parser_get_property     (xbutton, "onClick");
  button->widget.zIndex  = xml_parser_get_property_int (xbutton, "zIndex", 0);

  button->widget.xscript = xml_parser_get_property     (xbutton, "left");
  button->widget.yscript = xml_parser_get_property     (xbutton, "top");

  button->widget.layout                = button_layout;
  button->widget.paint                 = button_paint;
  button->widget.handle_motion         = button_handle_motion;
  button->widget.handle_button_press   = button_handle_button_press;
  button->widget.handle_button_release = button_handle_button_release;

  button->pix       = load_pix_trans (xbutton, "image", parent->view->basename);
  button->pix_hover = load_pix_trans (xbutton, "hoverImage", parent->view->basename);
  button->pix_down  = load_pix_trans (xbutton, "downImage", parent->view->basename);
  button->pix_hover_down = load_pix_trans (xbutton, "hoverDownImage", 
					   parent->view->basename);

  button->hovered = FALSE;
  button->pressed = FALSE;

  if (button->pix) {
    button->widget.w = gdk_pixbuf_get_width  (button->pix);
    button->widget.h = gdk_pixbuf_get_height (button->pix);
  } else {
    button->widget.w = 0;
    button->widget.h = 0;
  }

  if (!button->widget.id)
    button->widget.id = unique_name ("button");

  button->widget.se_obj = se_create_object (gse, parent->se_obj,
					    button->widget.id,
					    button);

  return button;
}

/*
 *
 * subview stuff
 *
 */

static void subview_layout (skin_widget_t *w) {
  subview_t  *s = (subview_t *) w;
  GList      *n;

  s->widget.x  = se_eval_int (s->widget.xscript, w->se_obj);
  s->widget.y  = se_eval_int (s->widget.yscript, w->se_obj);

  /*
   * layout children
   */

  n = g_list_first (s->widgets);
  while (n) {
    skin_widget_t *w;

    w = (skin_widget_t *) n->data;

    w->layout (w);

    n = g_list_next (n);
  }
}


static void subview_paint (skin_widget_t *w, int parent_xo, int parent_yo) {
  subview_t *s = (subview_t *) w;
  view_t    *view = w->view;
  GList      *n;

  if (s->pix) {
    gboolean finished;
    gint     xoff, yoff;
    
    if (s->halign && !strcasecmp (s->halign, "right")) 
      xoff = w->parent->w - w->x;
    else
      xoff = w->x;

    if (s->valign && !strcasecmp (s->valign, "bottom")) 
      yoff = w->parent->h - w->y;
    else
      yoff = w->y;

    xoff = w->x;
    yoff = w->y;
    finished = 0; 
    do {
      gdk_pixbuf_render_to_drawable (s->pix, view->bg, NULL, 0, 0,
				     xoff+parent_xo, 
				     yoff+parent_yo,
				     -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
      
      my_gdk_pixbuf_render_threshold_alpha (s->pix, view->mask,
					    0, 0, 
					    xoff+parent_xo, 
					    yoff+parent_yo,
					    -1, -1, 1);
      if (s->tiled) {
	if (s->halign && !strcasecmp (s->halign, "stretch")) {
	  xoff += gdk_pixbuf_get_width (s->pix);
	  
	  finished = (xoff > (w->parent->w - w->x)); 
	} else if (s->valign && !strcasecmp (s->valign, "stretch")) {
	  yoff += gdk_pixbuf_get_height (s->pix);
	  
	  finished = (yoff > (w->parent->h - w->y)); 
	} else
	  finished = TRUE;
      } else
	finished = TRUE;
    } while (!finished);
  }

  /*
   * paint children
   */

  n = g_list_first (s->widgets);
  while (n) {
    skin_widget_t *cw;

    cw = (skin_widget_t *) n->data;

    cw->paint (cw, parent_xo + s->widget.x, parent_yo + s->widget.y);

    n = g_list_next (n);
  }
}

static void subview_handle_button_release (skin_widget_t *w, int px, int py) {

  subview_t *s = (subview_t *) w;
  GList     *n;

  px -= w->x;
  py -= w->y;

  n = g_list_first (s->widgets);
  while (n) {
    skin_widget_t *cw;

    cw = (skin_widget_t *) n->data;

    cw->handle_button_release (cw, px, py);

    n = g_list_next (n);
  }
}

static gboolean subview_handle_button_press (skin_widget_t *w, int px, int py) {

  subview_t *s = (subview_t *) w;
  GList     *n;

  px -= w->x;
  py -= w->y;

  n = g_list_first (s->widgets);
  while (n) {
    skin_widget_t *cw;

    cw = (skin_widget_t *) n->data;

    if (cw->handle_button_press (cw, px, py))
      return TRUE;

    n = g_list_next (n);
  }
  return FALSE;
}

static void subview_handle_motion (skin_widget_t *w, int px, int py) {
  subview_t *s = (subview_t *) w;
  GList     *n;

  px -= w->x;
  py -= w->y;

  n = g_list_first (s->widgets);
  while (n) {
    skin_widget_t *cw;

    cw = (skin_widget_t *) n->data;

    cw->handle_motion (cw, px, py);

    n = g_list_next (n);
  }
}

static int subview_prop_int_cb (void *subview_gen, char *prop, int *num) {

  subview_t *subview = (subview_t *) subview_gen;

  if (!strcasecmp (prop, "left")) {
    subview->widget.x  = se_eval_int (subview->widget.xscript, subview->widget.se_obj);
    *num = subview->widget.x;
    return 1;
  } else if (!strcasecmp (prop, "top")) {
    subview->widget.y  = se_eval_int (subview->widget.yscript, subview->widget.se_obj);
    *num = subview->widget.y;
    return 1;
  }

  logprintf ("skin_window: looking for %s in subview %s\n",
	     prop, subview->widget.id);

  return 0;
}

static subview_t *create_subview (xml_node_t *xsubview, skin_widget_t *parent) {

  subview_t  *subview;
  xml_node_t *node;

  subview = (subview_t *) malloc (sizeof (subview_t));

  subview->widget.parent  = parent;
  subview->widget.view    = parent->view;
  subview->widget.id      = xml_parser_get_property     (xsubview, "id");
  subview->widget.zIndex  = xml_parser_get_property_int (xsubview, "zIndex", 0);

  subview->widget.xscript = xml_parser_get_property     (xsubview, "left");
  subview->widget.yscript = xml_parser_get_property     (xsubview, "top");

  subview->widget.layout                = subview_layout;
  subview->widget.paint                 = subview_paint;
  subview->widget.handle_motion         = subview_handle_motion;
  subview->widget.handle_button_press   = subview_handle_button_press;
  subview->widget.handle_button_release = subview_handle_button_release;

  subview->tiled   = xml_parser_get_property_bool(xsubview, "backgroundTiled", FALSE);
  subview->halign  = xml_parser_get_property     (xsubview, "horizontalAlignment");
  subview->valign  = xml_parser_get_property     (xsubview, "verticalAlignment");

  subview->pix = load_pix_trans (xsubview, "backgroundImage", parent->view->basename);

  if (!subview->widget.id)
    subview->widget.id = unique_name("subview");

  subview->widget.se_obj = se_create_object (gse, parent->se_obj,
					     subview->widget.id,
					     subview);
  subview->widgets = NULL;

  node = xsubview->child;
  while (node) {

    if (!strcasecmp (node->name, "subview")) {

      subview_t *s;

      s = create_subview (node, &subview->widget);

      subview->widgets = g_list_insert_sorted (subview->widgets, s, widgetzcomp);
    } else if (!strcasecmp (node->name, "buttongroup")) {
      buttongroup_t *buttongroup;

      buttongroup = create_buttongroup (node, &subview->widget);

      subview->widgets = g_list_insert_sorted (subview->widgets, buttongroup, widgetzcomp);
      
    } else if (!strcasecmp (node->name, "button") 
	       || !strcasecmp (node->name, "pausebutton")) {
      button_t *button;

      button = create_button (node, &subview->widget);

      subview->widgets = g_list_insert_sorted (subview->widgets, button, widgetzcomp);
    } else if (!strcasecmp (node->name, "video")) {
      /*video_t *video =*/ widget_video_create (node, &subview->widget);
    } else {
      printf ("*** error, '%s' unknown node type\n", node->name);
    }

    node = node->next;
  }


  return subview;
}

static void paint_view (view_t *view) {

  GList      *n;

  /*
   * compute widget positions
   */

  n = g_list_first (view->widgets);
  while (n) {
    skin_widget_t *w;

    w = (skin_widget_t *) n->data;

    w->layout (w);

    n = g_list_next (n);
  }

  /*
   * render widgets
   */

  n = g_list_first (view->widgets);
  while (n) {
    skin_widget_t *w;

    w = (skin_widget_t *) n->data;

    w->paint (w, 0, 0);

    n = g_list_next (n);
  }
}

static void view_motion_cb (GtkWidget *widget, GdkEventMotion *event, 
			    gpointer callback_data) {
  view_t    *view = (view_t *) callback_data;
  XEvent     ev; 
  gint       i = 0;

  XSync (gdk_display, False);
  
  while (XCheckTypedEvent (GDK_DISPLAY(), MotionNotify, &ev)) {
    event->x = ev.xmotion.x;
    event->y = ev.xmotion.y;
    i++;
  }
  
  if (view->is_moving) {
    gint mx, my, newx, newy;
    GdkModifierType modmask;
    
    gdk_window_get_pointer (NULL, &mx, &my, &modmask);
    
    newx = mx - view->widget.x;
    newy = my - view->widget.y;
    gdk_window_move (view->window->window, newx, newy);

  } else {
    GList *n;
    gint mx, my;

    view->repaint_needed = FALSE;

    mx = event->x; my = event->y;
    
    n = g_list_first (view->widgets);
    while (n) {
      skin_widget_t *w;

      w = (skin_widget_t *) n->data;
      
      w->handle_motion (w, mx, my);
      
      n = g_list_next (n);
    }

    if (view->repaint_needed) {
      paint_view (view);

      gdk_window_set_back_pixmap (view->window->window, view->bg, 0);
      gdk_window_shape_combine_mask (view->window->window, view->mask, 0, 0); 
      gdk_window_clear (view->window->window);
    }
  }
  
  gdk_flush();
}

static void view_button_release_cb (GtkWidget * widget, GdkEventButton * event, 
				    gpointer callback_data) {

  view_t    *view = (view_t *) callback_data;
  GList    *n;
  gint      mx, my;

  view->repaint_needed = FALSE;

  mx = event->x; my = event->y; 
  
  n = g_list_first (view->widgets);
  while (n) {
    skin_widget_t *w;
    
    w = (skin_widget_t *) n->data;
      
    w->handle_button_release (w, mx, my);
      
    n = g_list_next (n);
  }

  if (view->repaint_needed) {
    paint_view (view);
    
    gdk_window_set_back_pixmap (view->window->window, view->bg, 0);
    gdk_window_shape_combine_mask (view->window->window, view->mask, 0, 0); 
    gdk_window_clear (view->window->window);
  }

  gdk_pointer_ungrab (GDK_CURRENT_TIME);
  gdk_flush ();
  
  view->is_moving = FALSE;
}

static gboolean view_button_press_cb (GtkWidget *widget, GdkEventButton *event, 
				      gpointer data) {

  view_t *view = (view_t *) data;

  if (event && (event->type==GDK_BUTTON_PRESS)) {
    
    if (event->button == 1) {

      GList    *n;
      gint      mx, my;
      gboolean  handled;

      view->repaint_needed = FALSE;

      mx = event->x; my = event->y; handled = FALSE;
    
      n = g_list_first (view->widgets);
      while (n) {
	skin_widget_t *w;
	
	w = (skin_widget_t *) n->data;
      
	handled |= w->handle_button_press (w, mx, my);
      
	n = g_list_next (n);
      }

      if (view->repaint_needed) {
	paint_view (view);
	
	gdk_window_set_back_pixmap (view->window->window, view->bg, 0);
	gdk_window_shape_combine_mask (view->window->window, view->mask, 0, 0); 
	gdk_window_clear (view->window->window);
      }
      
      if (!handled) {
	view->is_moving = TRUE;
	view->widget.x = event->x;
	view->widget.y = event->y;
      }
    }
  }

  return TRUE;
}

static view_t *create_view (theme_t *theme, xml_node_t *xview, const char *basename,
			    int view_num) {

  xml_node_t *node;
  view_t     *view;
  char       *scriptfile;
  
  view = (view_t *) malloc (sizeof (view_t));

  view->widget.parent  = NULL;
  view->widget.view    = view;
  view->widget.zIndex  = 0;

#if 0
  view->widget.xscript = xml_parser_get_property     (xsubview, "left");
  view->widget.yscript = xml_parser_get_property     (xsubview, "top");
#endif

  view->widget.w         = xml_parser_get_property_int (xview, "width", 0);
  view->widget.h         = xml_parser_get_property_int (xview, "height", 0);
  view->widget.id        = xml_parser_get_property     (xview, "id");

  view->is_moving = FALSE;

  if (!view->widget.w || !view->widget.h) {
    free (view);
    return NULL;
  }

  view->basename = strdup (basename);
  view->widgets  = NULL;

  /*
   * create corresponding javascript object
   */

  if (!view->widget.id)
    view->widget.id = unique_name ("view");
  view->widget.se_obj = se_create_object (gse, NULL, "view", view);

  se_prop_create_int (gse, view->widget.se_obj, "width", view->widget.w);
  se_prop_create_int (gse, view->widget.se_obj, "height", view->widget.w);
  
  /*
   * compile associated js file
   */

  scriptfile = xml_parser_get_property  (xview, "scriptfile");

  if (scriptfile) {
    char  jsfile[MAXPATH];
    char *js_code, *onload, *semi;

    snprintf (jsfile, MAXPATH, "%s/%s", 
	      basename,
	      scriptfile);

    semi = strchr (jsfile, ';');
    if (semi)
      *semi = 0;
    
    printf ("skin_window: loading js file %s\n", jsfile);

    js_code = read_entire_file (jsfile);

    printf ("skin_window: executing it...\n");

    se_eval (gse, js_code, NULL, print_cb, NULL);

    onload = xml_parser_get_property  (xview, "onload");
    
    if (onload) {
      printf ("skin_window: queueing '%s' (onload)\n", onload);

      theme->jobs = g_list_append (theme->jobs, onload);
    }
  }

  view->window =  gtk_window_new(GTK_WINDOW_TOPLEVEL);

  gtk_widget_set_app_paintable (view->window, TRUE);

  gtk_widget_set_size_request (view->window, view->widget.w, view->widget.h);

  gtk_widget_add_events (view->window, GDK_POINTER_MOTION_MASK 
			 | GDK_BUTTON_MOTION_MASK    
			 | GDK_BUTTON1_MOTION_MASK
			 | GDK_BUTTON_PRESS_MASK 
			 | GDK_BUTTON_RELEASE_MASK 
			 | GDK_KEY_PRESS_MASK);

  g_signal_connect (G_OBJECT (view->window), "button_press_event",
		    G_CALLBACK (view_button_press_cb), view);
  g_signal_connect (G_OBJECT (view->window), "button_release_event",
		    G_CALLBACK (view_button_release_cb), view);
  g_signal_connect (G_OBJECT (view->window), "motion_notify_event",
		    G_CALLBACK (view_motion_cb), view);

  gtk_widget_realize (view->window);

  view->bg   = gdk_pixmap_new (view->window->window, view->widget.w, view->widget.h, -1);
  view->mask = gdk_pixmap_new (view->window->window, view->widget.w, view->widget.h, 1);

  view->fixed = gtk_fixed_new ();
  gtk_container_add (GTK_CONTAINER (view->window), view->fixed);
  
  node = xview->child;
  while (node) {

    if (!strcasecmp (node->name, "subview")) {

      subview_t *subview;

      subview = create_subview (node, &view->widget);

      view->widgets = g_list_insert_sorted (view->widgets, subview, widgetzcomp);
    } else if (!strcasecmp (node->name, "buttongroup")) {
      buttongroup_t *buttongroup;

      buttongroup = create_buttongroup (node, &view->widget);

      view->widgets = g_list_insert_sorted (view->widgets, buttongroup, widgetzcomp);
      
    }

    node = node->next;
  }

  {
    GdkGC *gc;
    
    gc = gdk_gc_new (view->mask);
    
    gdk_draw_rectangle (view->mask, gc, TRUE, 0, 0, view->widget.w, view->widget.h);
  }

  paint_view (view);

  gdk_window_set_back_pixmap (view->window->window, view->bg, 0);

  gdk_window_shape_combine_mask (view->window->window, view->mask, 0, 0); 

  gdk_window_set_decorations (view->window->window, 0);
  gtk_window_set_policy (GTK_WINDOW (view->window), FALSE, FALSE, FALSE);

  /* if (!view_num) */
    gtk_widget_show_all (view->window);

  if (gtv) {
    /* FIXME: bad hack */
    printf ("gtv is: %p\n", gtv);
    gtk_widget_hide (gtv);
    gtk_widget_show (gtv);
  }

  return view;
}

static int theme_prop_int_cb (void *theme_gen, char *prop, int *num) {

  /* buttongroup_t *buttongroup = (buttongroup_t *) buttongroup_gen; */

  printf ("skin_window: looking for %s in theme\n", prop);

  return 0;
}

static JSBool load_pref_cb (JSContext *cx, JSObject *obj, uintN argc, 
			    jsval *argv, jsval *rval) {

  JSString *str;
  
  printf ("FIXME: load_pref_cb called, not implemented yet\n");

  str = JS_NewString(cx, "false", 5);
  
  *rval = STRING_TO_JSVAL (str);

  return JS_TRUE;
}

static theme_t *create_theme (xml_node_t *xtheme, char *basename) {

  xml_node_t *node;
  int         view_num;
  theme_t    *theme;

  theme = malloc (sizeof (theme_t));

  theme->views = NULL;
  theme->jobs = NULL;

  theme->se_obj = se_create_object (gse, NULL,
				    "theme",
				    theme);

  se_defun (gse, theme->se_obj, "loadPreference", load_pref_cb, 0, 0,
	    SE_GROUP_HIDDEN, NULL, NULL);

  node = xtheme->child; view_num = 0;
  while (node) {

    if (!strcasecmp (node->name, "view")) {
      view_t *view;

      view = create_view (theme, node, basename, view_num++);

      theme->views = g_list_append (theme->views, view);

    }

    node = node->next;
  }

  return theme;
}

void create_skin_window (void) {

  /* char       *skinfile = "/home/guenter/projects/video/gnome-xine/skins/plus/bionic_d.wms";  */
  char       *skinfile = "/home/guenter/projects/video/gnome-xine/skins/xbox_live/xbox.wms";  
  /* char       *skinfile = "/home/guenter/projects/video/gnome-xine/skins/kenwood/kenwood.wms"; */
  /* char       *skinfile = "/home/guenter/projects/video/gnome-xine/skins/media_center/skin.gxs"; */
  char       *xml_data;
  xml_node_t *tree, *node;

  printf ("skin_window: loading %s...\n", skinfile);

  xml_data = read_entire_file (skinfile);
  if (!xml_data) {
    printf ("failed to open %s: %s\n", skinfile, strerror (errno));
    abort ();
  }

  /* printf ("skin_window: ascii: %s\n", xml_data);  */

  printf ("skin_window: parsing...\n");

  xml_parser_init (xml_data, strlen(xml_data), XML_PARSER_CASE_INSENSITIVE);

  if (xml_parser_build_tree (&tree)<0) {
    printf ("skin_window: error: xml parser failed\n");
    return;
  }

  /* xml_parser_dump_tree (tree); */

#if 0
  printf ("skin_window: loading %s...\n", jsfile);

  js_code = read_entire_file (jsfile);

  printf ("skin_window: javascript code:\n---------------------------------\n%s\n---------------------\n",
	  js_code);
#endif

  node = tree;
  while (node) {

    if (!strcasecmp (node->name, "theme")) {
      gchar *skin_dir, *dir_sep;

      skin_dir = strdup (skinfile);
      dir_sep  = strrchr (skin_dir, '/');
      if (dir_sep)
	*dir_sep = '\0';

      create_theme (node, skin_dir);
      break;
    }

    node = node->next;
  }

}

void window_toolbar_toggle (void)
{
}

void window_toolbar_reset (void)
{
}

void window_toolbar_restore (void)
{
}

#endif /* EXP_STUFF */
