/* 
 * Copyright (C) 2003-2004 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: key_events.c,v 1.46 2004/12/17 00:32:59 dsalt Exp $
 *
 * key event handler and keymap editor
 */

#include "globals.h"

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <xine.h>

#include "key_events.h"

#include "engine.h"
#include "utils.h"
#include "xmlparser.h"
#include "menu.h"
#include "gtkvideo.h"
#include "playlist.h"

/*
#define LOG
*/

static gboolean        list_visible = FALSE, edit_visible = FALSE,
		       is_new_binding = FALSE;
static GtkListStore   *kb_store;
static GtkWidget      *kb_bindings_list_dlg, *tree_view,
		      *kb_binding_edit_dlg, *kb_binding_desc,
		      *kb_binding_command, *kb_binding_key, *kb_binding_quote;
static GtkAccelGroup  *kb_binding_edit_accel;

typedef struct {
  const gchar *desc, *cmd;
  guint        keyval, state;
  const gchar *keyname;
} key_binding_t;

static key_binding_t editkey;
static const key_binding_t null_binding = { 0, 0, GDK_VoidSymbol, 0 };
static key_binding_t default_binding = { N_("New binding"), "", GDK_VoidSymbol, 0 };

static GtkTreeIter catch_key_iter;

static void delete_key_binding (GtkTreeIter *);


static const key_binding_t default_bindings[] = {
  { N_("Play"), "vdr ('PLAY') || play ();", GDK_Return, 0 },
  { N_("Exit gxine"), "exit ();", GDK_q, 0 },
  { N_("Pause"), "if (!vdr ('PAUSE') && !is_live_stream ()) pause ();", GDK_space, 0 },
  { N_("Stop"), "vdr ('STOP') || stop ();", 0, 0 },
  { N_("Windowed mode"), "set_fullscreen (0);", GDK_Escape, 0 },
  { N_("Fullscreen mode"), "set_fullscreen ();", GDK_f, 0 },
  { N_("Aspect ratio"), "set_aspect ();", GDK_a, 0 },
  { N_("Toggle deinterlace"), "set_deinterlace ();", GDK_i, 0 },
  { N_("Rewind / Back 1 min"), "if (!vdr ('FASTREW') && !is_live_stream ()) play (0, get_time()-60000);", GDK_Left, 0 },
  { N_("Fast forward / Forward 1 min"), "if (!vdr ('FASTFWD') && !is_live_stream ()) play (0, get_time()+60000);", GDK_Right, 0 },
  { N_("Faster"), "if (!is_live_stream ()) set_speed (get_speed()+1);", GDK_Up, 0 },
  { N_("Slower"), "if (!is_live_stream ()) set_speed (get_speed()-1);", GDK_Down, 0 },
  { N_("Up"), "input_up ();", GDK_KP_8, 0 },
  { N_("Down"), "input_down ();", GDK_KP_2, 0 },
  { N_("Left"), "input_left ();", GDK_KP_4, 0 },
  { N_("Right"), "input_right ();", GDK_KP_6, 0 },
  { N_("Select/OK"), "input_select ();", GDK_KP_Enter, 0 },
  { N_("Menu 1 [main]"), "input_menu1 ();", GDK_F1, 0 },
  { N_("Menu 2 [schedule]"), "vdr ('SCHEDULE') || input_menu2 ();", GDK_F2, 0 },
  { N_("Menu 3 [channels]"), "vdr ('CHANNELS') || input_menu3 ();", GDK_F3, 0 },
  { N_("Menu 4 [timers]"), "vdr ('TIMERS') || input_menu (4);", GDK_F4, 0 },
  { N_("Menu 5 [recordings]"), "vdr ('RECORDINGS') || input_menu (5);", GDK_F5, 0 },
  { N_("Menu 6 [setup]"), "vdr ('SETUP') || input_menu (6);", GDK_F6, 0 },
  { N_("Menu 7 [user]"), "vdr ('COMMANDS') || input_menu (7);", GDK_F7, 0 },
  { N_("Previous"), "vdr ('CHANNELMINUS') || input_previous ();", GDK_KP_Page_Up, 0 },
  { N_("Next"), "vdr ('CHANNELPLUS') || input_next ();", GDK_KP_Page_Down, 0 },
  { N_("1 / Play, skip first 10%"), "vdr ('1') || play (10, 0);", GDK_1, 0 },
  { N_("2 / Play, skip first 20%"), "vdr ('2') || play (20, 0);", GDK_2, 0 },
  { N_("3 / Play, skip first 30%"), "vdr ('3') || play (30, 0);", GDK_3, 0 },
  { N_("4 / Play, skip first 40%"), "vdr ('4') || play (40, 0);", GDK_4, 0 },
  { N_("5 / Play, skip first half"), "vdr ('5') || play (50, 0);", GDK_5, 0 },
  { N_("6 / Play last 40%"), "vdr ('6') || play (60, 0);", GDK_6, 0 },
  { N_("7 / Play last 30%"), "vdr ('7') || play (70, 0);", GDK_7, 0 },
  { N_("8 / Play last 20%"), "vdr ('8') || play (80, 0);", GDK_8, 0 },
  { N_("9 / Play last 10%"), "vdr ('9') || play (90, 0);", GDK_9, 0 },
  { N_("0 / Play from start"), "vdr ('0') || play (0, 0);", GDK_0, 0 },
  { N_("Red"), "vdr ('RED');", 0, 0 },
  { N_("Green"), "vdr ('GREEN');", 0, 0 },
  { N_("Yellow"), "vdr ('YELLOW');", 0, 0 },
  { N_("Blue"), "vdr ('BLUE');", 0, 0 },
  { N_("Record"), "vdr ('RECORD');", 0, 0 },
  { N_("Power"), "vdr ('POWER');", 0, 0 },
  { N_("Back"), "vdr ('BACK');", 0, 0 },
  { N_("Playlist next"), "playlist_play (playlist_get_item()+1);", GDK_Page_Down, 0 },
  { N_("Playlist previous"), "playlist_play (playlist_get_item()-1);", GDK_Page_Up, 0 },
  { N_("Zoom in"), "set_zoom (get_zoom()+5);", GDK_Z, GDK_SHIFT_MASK },
  { N_("Zoom out"), "set_zoom (get_zoom()-5);", GDK_z, 0 },
  { N_("Zoom 100%"), "set_zoom (100);", GDK_z, GDK_CONTROL_MASK },
  { N_("Volume +"), "set_mute (0); set_volume (get_volume()+5);", 0, 0 },
  { N_("Volume -"), "set_mute (0); set_volume (get_volume()-5);", 0, 0 },
  { N_("Mute"), "set_mute ();", 0, 0 },
  { 0 }
};


static key_binding_t *lookup_binding (GtkTreeIter *iter)
{
  key_binding_t *k;
  GValue v;

  memset (&v, 0, sizeof (GValue));
  gtk_tree_model_get_value (GTK_TREE_MODEL (kb_store), iter, 2, &v);
  k = g_value_peek_pointer (&v);
  g_value_unset (&v);
  return k;
}


gint keypress_cb (GtkWidget *win, GdkEventKey *event)
{
  GtkTreeIter iter;
  guint keyval = gdk_keyval_to_lower (event->keyval);
  guint state = event->state & GXINE_MODIFIER_MASK;

  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL(kb_store), &iter)) {
    do {
      key_binding_t *key_binding = lookup_binding (&iter);
      if ( (key_binding->keyval == keyval)
	   && (key_binding->state == state) ) {
	engine_exec (key_binding->cmd, NULL, NULL);
	return TRUE;
      }
    } while (gtk_tree_model_iter_next (GTK_TREE_MODEL (kb_store), &iter));
  }

  /* Popup menu handling (F10 or right button) */
  if (state == 0 && popup_menu)
    switch (keyval)
    {
    case GDK_F10:
      if (menubar && !gtk_video_is_fullscreen (GTK_VIDEO(gtv)))
	return FALSE; /* let GTK handle this internally */
      gtk_menu_popup (GTK_MENU(popup_menu), NULL, NULL, NULL, NULL, 0,
		      gtk_get_current_event_time ());
      return TRUE;
    case GDK_Pointer_Button2:
      if (gtk_video_is_fullscreen (GTK_VIDEO(gtv)))
	ui_toolbar_toggle ();
      return TRUE;
    }

  if (gtk_video_is_fullscreen (GTK_VIDEO(gtv)) &&
      gtk_accel_groups_activate (G_OBJECT(app), event->keyval, event->state))
    return TRUE;

  logprintf ("key_events: key %d not bound\n", event->keyval);

  return FALSE;
}


gint keyrelease_cb (GtkWidget *win, GdkEventKey *event)
{
  guint keyval = gdk_keyval_to_lower (event->keyval);
  guint state = event->state & GXINE_MODIFIER_MASK;

  if (state == 0 && popup_menu)
    switch (keyval)
    {
    case GDK_Pointer_Button3:
      gtk_menu_popup (GTK_MENU(popup_menu), NULL, NULL, NULL, NULL, 0,
		      gtk_get_current_event_time ());
      return TRUE;
    }

  return FALSE;
}


static void close_list_window (void)
{
  list_visible = FALSE;
  gtk_widget_hide (kb_bindings_list_dlg);
}


static void close_edit_window (void)
{
  edit_visible = FALSE;
  if (is_new_binding)
    delete_key_binding (&catch_key_iter);
  gtk_widget_hide (kb_binding_edit_dlg);
}


static gboolean close_cb (GtkWidget* widget, gpointer data) {
  close_edit_window ();
  close_list_window ();
  return TRUE;
}


static key_binding_t *find_key_binding (const gchar *desc, const gchar *cmd,
					GtkTreeIter *iter)
{
  if (!desc && !cmd)
    return NULL;

  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL(kb_store), iter)) {

    do {
      key_binding_t *key_binding = lookup_binding (iter);

      /* allow either to be null, in which case we match on the other rather
       * than on both
       */
      if ((!desc || !strcasecmp (key_binding->desc, desc)) &&
	  (!cmd || !strcasecmp (key_binding->cmd, cmd)))
	return key_binding;

    } while (gtk_tree_model_iter_next (GTK_TREE_MODEL (kb_store),
				       iter));
  }

  return NULL;
}

static gchar* kb2str (key_binding_t *kb) {

  static const gchar *text_shift = NULL, *text_control,
		     *text_mod1, *text_mod3, *text_mod4;
  guint              l;
  gchar             *name;
  const gchar       *key;

  if (!text_shift)
  {
    text_shift   = _("<Shift>");
    text_control = _("<Control>");
    text_mod1    = _("<Alt>");
    text_mod3    = _("<Mod3>");
    text_mod4    = _("<Mod4>");
  }

  /*
   * calc length of resulting string
   */

  l = 1;
  if (kb->state & GDK_SHIFT_MASK)
    l += strlen (text_shift);
  if (kb->state & GDK_CONTROL_MASK)
    l += strlen (text_control);
  if (kb->state & GDK_MOD1_MASK)
    l += strlen (text_mod1);
  if (kb->state & GDK_MOD3_MASK)
    l += strlen (text_mod3);
  if (kb->state & GDK_MOD4_MASK)
    l += strlen (text_mod4);

  if (kb->keyval == GDK_VoidSymbol)
    key = _("undef");
  else if (!(key = gdk_keyval_name (kb->keyval)))
    key = _("unknown");

  l += strlen (key);
  name = malloc (l);
  *name = 0;

  if (kb->state & GDK_SHIFT_MASK)
    strcat (name, text_shift);
  if (kb->state & GDK_CONTROL_MASK)
    strcat (name, text_control);
  if (kb->state & GDK_MOD1_MASK)
    strcat (name, text_mod1);
  if (kb->state & GDK_MOD3_MASK)
    strcat (name, text_mod3);
  if (kb->state & GDK_MOD4_MASK)
    strcat (name, text_mod4);

  strcat (name, key);

  return name;
}

static void modify_key_binding (GtkTreeIter *iter, key_binding_t *key_binding,
				const key_binding_t *new_binding)
{
  free ((char *)key_binding->keyname);
  free ((char *)key_binding->cmd);
  free ((char *)key_binding->desc);

  if (new_binding->keyval != GDK_VoidSymbol)
  {
    key_binding->keyval = gdk_keyval_to_lower (new_binding->keyval);
    key_binding->state  = new_binding->state & GXINE_MODIFIER_MASK;
  }

  key_binding->keyname = kb2str (key_binding);
  key_binding->cmd     = strdup (new_binding->cmd);
  if (!new_binding->desc)
  {
    int i = -1;
    while (default_bindings[++i].desc)
      if (!strcasecmp (default_bindings[i].cmd, new_binding->cmd))
        break;
    key_binding->desc  = strdup (default_bindings[i].desc ? : new_binding->cmd);
  }
  else
    key_binding->desc  = strdup (new_binding->desc);

  gtk_list_store_set (kb_store, iter, 0, key_binding->desc, 1,
		      key_binding->keyname, 2, key_binding, -1);
}


static void set_key_binding (const key_binding_t *new_binding)
{
  key_binding_t *key_binding;
  GtkTreeIter    iter;

  key_binding = new_binding->desc
		? find_key_binding (new_binding->desc, NULL, &iter)
		: find_key_binding (NULL, new_binding->cmd, &iter);

  if (!key_binding) {

    /* add new key binding */

    key_binding = malloc (sizeof (key_binding_t));
    memset (key_binding, 0, sizeof (key_binding_t));
    key_binding->keyval = GDK_VoidSymbol;

    gtk_list_store_append (kb_store, &iter);
  }

  modify_key_binding (&iter, key_binding, new_binding);
}


static void delete_key_binding (GtkTreeIter *iter)
{
  key_binding_t *key_binding = lookup_binding (iter);
  free ((char *)key_binding->keyname);
  free ((char *)key_binding->cmd);
  free ((char *)key_binding->desc);
  gtk_list_store_remove (kb_store, iter);
  free (key_binding);
}


static void load_default_kb (void)
{
  int i = -1;
  while (default_bindings[++i].desc)
  {
    key_binding_t binding = default_bindings[i];
    binding.desc = gettext (binding.desc);
    set_key_binding (&binding);
  }
}


void save_key_bindings (void) {

  gchar *fname;
  FILE  *f;

  fname = g_strconcat (g_get_home_dir(), "/.gxine/keybindings", NULL);

  f = fopen (fname, "w");
  if (f) {
    GtkTreeIter iter;

    fprintf (f, "<GXINEKB VERSION=\"2\">\n");

    if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL(kb_store), &iter)) {

      do {
	key_binding_t *key_binding = lookup_binding (&iter);

	fprintf (f, "  <KEYBINDING>\n"
		    "    <DESCRIPTION>%s</DESCRIPTION>\n"
		    "    <COMMAND>%s</COMMAND>\n"
		    "    <KEYVAL>%d</KEYVAL>\n"
		    "    <STATE>%d</STATE>\n"
		    "  </KEYBINDING>\n",
		    key_binding->desc, key_binding->cmd,
		    key_binding->keyval, key_binding->state);
      
      } while (gtk_tree_model_iter_next (GTK_TREE_MODEL (kb_store),
					 &iter));
    }

    fprintf (f, "</GXINEKB>\n");

    if (ferror (f))
      printf (_("key_events: bindings save failed: %s\n"), strerror (errno));
    if (fclose (f))
      printf (_("key_events: bindings save failed: %s\n"), strerror (errno));
  } else 
    printf (_("key_events: bindings save failed: %s\n"), strerror (errno));

  g_free (fname);
}

typedef struct kb_upgrade_s { const char *oldcmd, *newcmd; } kb_upgrade_t;

static const char *kb_upgrade (const char *oldcmd,
			       const kb_upgrade_t *upgrade)
{
  while (upgrade->oldcmd)
  {
    if (!strcmp (oldcmd, upgrade->oldcmd))
      return upgrade->newcmd;
    ++upgrade;
  }
  return oldcmd;
}

static const kb_upgrade_t vdr_upgrade[] = {
  /* upgrade from 0.3.3 */
  { "play ();",		"vdr ('PLAY') || play ();"  },
  { "pause ();",	"if (!vdr ('PAUSE') && !is_live_stream ()) pause ();" },
  { "stop ();",		"vdr ('STOP') || stop ();" },
  { "play (0, get_time()-60000);", "if (!vdr ('FASTREW') && !is_live_stream ()) play (0, get_time()-60000);" },
  { "play (0, get_time()+60000);", "if (!vdr ('FASTFWD') && !is_live_stream ()) play (0, get_time()+60000);" },
  { "set_speed (get_speed()+1);", "if (!is_live_stream ()) set_speed (get_speed()+1);" },
  { "set_speed (get_speed()-1);", "if (!is_live_stream ()) set_speed (get_speed()-1);" },
  { "input_menu2 ();",	"vdr ('SCHEDULE') || input_menu2 ();" },
  { "input_menu3 ();",	"vdr ('CHANNELS') || input_menu3 ();" },
  { "input_menu (4);",	"vdr ('TIMERS') || input_menu (4);" },
  { "input_menu (5);",	"vdr ('RECORDINGS') || input_menu (5);" },
  { "input_menu (6);",	"vdr ('SETUP') || input_menu (6);" },
  { "input_menu (7);",	"vdr ('COMMANDS') || input_menu (7);" },
  { "input_previous ();", "vdr ('CHANNELMINUS') || input_previous ();" },
  { "input_next ();",	"vdr ('CHANNELPLUS') || input_next ();" },
  { "play (10, 0);",	"vdr ('1') || play (10, 0);" },
  { "play (20, 0);",	"vdr ('2') || play (20, 0);" },
  { "play (30, 0);",	"vdr ('3') || play (30, 0);" },
  { "play (40, 0);",	"vdr ('4') || play (40, 0);" },
  { "play (50, 0);",	"vdr ('5') || play (50, 0);" },
  { "play (60, 0);",	"vdr ('6') || play (60, 0);" },
  { "play (70, 0);",	"vdr ('7') || play (70, 0);" },
  { "play (80, 0);",	"vdr ('8') || play (80, 0);" },
  { "play (90, 0);",	"vdr ('9') || play (90, 0);" },
  { "play (0, 0);",	"vdr ('0') || play (0, 0);" },
  /* fix breakage in 0.4.0-rc* */
  { "vdr ('PAUSE') || if (!is_live_stream ()) pause ();",	"if (!vdr ('PAUSE') && !is_live_stream ()) pause ();" },
  { "vdr ('FASTREW') || if (!is_live_stream ()) play (0, get_time()-60000);", "if (!vdr ('FASTREW') && !is_live_stream ()) play (0, get_time()-60000);" },
  { "vdr ('FASTFWD') || if (!is_live_stream ()) play (0, get_time()+60000);", "if (!vdr ('FASTFWD') && !is_live_stream ()) play (0, get_time()+60000);" },
  { NULL }
};

static void xml2kb (xml_node_t *node, int version)
{
  key_binding_t binding = { 0 };
  int i;

  while (node) {
    if (!strcasecmp (node->name, "command"))
      binding.cmd = node->data;
    else if (!strcasecmp (node->name, "description"))
      binding.desc = node->data;
    else if (!strcasecmp (node->name, "keyval"))
      binding.keyval = atoi (node->data);
    else if (!strcasecmp (node->name, "state"))
      binding.state = atoi (node->data);

    node = node->next;
  }

  if (!binding.cmd || !binding.keyval)
    return;

  if (version < 2)
    binding.cmd = kb_upgrade (binding.cmd, vdr_upgrade);

  /* translate from English if a new translation is present and in use */
  if (binding.desc)
    for (i = 0; default_bindings[i].desc; ++i)
      if (!strcmp (binding.desc, default_bindings[i].desc) &&
	!strcmp (binding.cmd, default_bindings[i].cmd))
      {
	binding.desc = gettext (binding.desc);
	break;
      }

  logprintf ("key_events: desc='%s', cmd='%s', keyval=%d, state=%d\n",
	     binding.desc ? binding.desc : "", binding.cmd, binding.keyval,
	     binding.state);

  set_key_binding (&binding);
}

static void load_key_bindings (void)
{
  gchar *fname = g_strconcat (g_get_home_dir(), "/.gxine/keybindings", NULL);
  gchar *kbfile = read_entire_file (fname, NULL);
  int version = 1;

  if (kbfile) {
    xml_node_t *node;

    xml_parser_init (kbfile, strlen (kbfile), XML_PARSER_CASE_INSENSITIVE);

    if (xml_parser_build_tree (&node)>=0) {

      if (!strcasecmp (node->name, "gxinekb")) {

	xml_property_t *prop = node->props;

	while (prop)
	{
	  if (!strcasecmp (prop->name, "version") && prop->value)
	  {
	    version = atoi (prop->value);
	    logprintf ("key_events: keybindings version %d\n", version);
	    break;
	  }
	  prop = prop->next;
	}

        if (version < 2)
        {
          display_info (_("Upgrading your key bindings"),
			_("Your key bindings are being upgraded. This adds:\n"
			" • Key binding descriptions\n"
			" • VDR support\n"
			"\n"
			"Some old deleted bindings may have been restored.\n"
			"You should check your bindings now.\n")
			);
          load_default_kb ();
        }

	node = node->child;
	while (node) {
	  if (!strcasecmp (node->name, "KEYBINDING")) {

	    xml2kb (node->child, version);

	  }
	  node = node->next;
	}

      } else {
	printf (_("key_events: error, %s is not a valid gxine keybindings file\n"),
		fname);
	load_default_kb();
      }
    } else {
      printf (_("key_events: error, cannot load keybindings file %s (xml parsing failed)\n"),
	      fname);
      load_default_kb();
    }

    free (kbfile);
  } else {
    if (errno != ENOENT)
      printf (_("key_events: error, cannot open keybindings file %s\n"),
	      fname);
    load_default_kb();
  }

  g_free(fname);
}

void kb_edit_show (void) {

  if (list_visible) {
    close_edit_window ();
    close_list_window ();
  } else {
    list_visible = TRUE;
    gtk_widget_show_all (kb_bindings_list_dlg);
    gtk_tree_view_columns_autosize (GTK_TREE_VIEW(tree_view));
  }
}


static void do_edit_binding (void)
{
  editkey = *lookup_binding (&catch_key_iter);
  char title[64];

  gtk_entry_set_text (GTK_ENTRY (kb_binding_desc), editkey.desc);
  gtk_entry_set_text (GTK_ENTRY (kb_binding_command), editkey.cmd);
  gtk_entry_set_text (GTK_ENTRY (kb_binding_key), editkey.keyname);

  snprintf (title, sizeof (title), _("Keybinding: %s"), editkey.desc);
  gtk_window_set_title (GTK_WINDOW (kb_binding_edit_dlg), title);

  gtk_widget_show_all (kb_binding_edit_dlg);
  gtk_widget_grab_focus (kb_binding_key);
}


static void kb_new_binding (GtkWidget *widget, gpointer data)
{
  
  set_key_binding (&default_binding);
  find_key_binding (default_binding.desc, default_binding.cmd, &catch_key_iter);
  do_edit_binding ();
}


static void kb_delete_binding (GtkWidget *widget, gpointer data)
{
  GtkTreeIter iter;
  GtkTreeSelection *sel =
    gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));

  if (gtk_tree_selection_get_selected (sel, NULL, &iter))
  {
    GtkTreePath *path =
      gtk_tree_model_get_path (GTK_TREE_MODEL (kb_store), &iter);
    if (path)
    {
      gtk_tree_view_set_cursor (GTK_TREE_VIEW (tree_view), path, NULL, FALSE);
      gtk_tree_path_free (path);
    }

    if (edit_visible
	&& lookup_binding (&iter) == lookup_binding (&catch_key_iter))
      close_edit_window ();
    delete_key_binding (&iter);
  }
}


static void kb_edit_binding (GtkWidget *widget, gpointer data)
{
  GtkTreeSelection *sel =
    gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));

  if (sel && gtk_tree_selection_get_selected (sel, NULL, &catch_key_iter))
    do_edit_binding ();
}


static void kb_sig_edit_binding (GtkTreeView *tree, GtkTreeIter *iter,
				 GtkTreePath *path, gpointer data)
{
  kb_edit_binding (GTK_WIDGET (tree), data);
}


static void kb_merge_defaults (GtkWidget *widget, gpointer data)
{
  if (!edit_visible)
    load_default_kb ();
}


static void kb_reload_keymap (GtkWidget *widget, gpointer data)
{
  if (!edit_visible)
  {
    gtk_list_store_clear (kb_store);
    load_key_bindings ();
  }
}


static void kb_reset_keymap (GtkWidget *widget, gpointer data)
{
  if (!edit_visible)
  {
    gtk_list_store_clear (kb_store);
    load_default_kb ();
  }
}


static void set_binding_quote (gboolean state)
{
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (kb_binding_quote))
      == state)
    return;

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (kb_binding_quote), state);
}


static gboolean bq_lock = FALSE;

static gboolean start_edit_key_cb (GtkWidget *widget, GdkEventFocus *event,
				   gpointer user_data)
{
  if (!bq_lock)
    set_binding_quote (FALSE);
  gtk_entry_select_region (GTK_ENTRY (kb_binding_key), 0, 256);
  return TRUE;
}


static gboolean end_edit_key_cb (GtkWidget *widget, GdkEventFocus *event,
				 gpointer user_data)
{
  set_binding_quote (FALSE);
  return FALSE;
}


static gboolean do_edit_key_cb (GtkWidget *widget, GdkEventKey *event,
				gpointer user_data)
{
  gchar *name;

  if (   event->keyval == GDK_Control_L
      || event->keyval == GDK_Control_R
      || event->keyval == GDK_Meta_L
      || event->keyval == GDK_Meta_R
      || event->keyval == GDK_Alt_L
      || event->keyval == GDK_Alt_R
      || event->keyval == GDK_Super_L
      || event->keyval == GDK_Super_R
      || event->keyval == GDK_Hyper_L
      || event->keyval == GDK_Hyper_R
      || event->keyval == GDK_Shift_L
      || event->keyval == GDK_Shift_R
      || gtk_window_get_focus (GTK_WINDOW (kb_binding_edit_dlg)) != kb_binding_key)
    return FALSE;

  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (kb_binding_quote)))
  {
    if (event->keyval == GDK_Tab && event->state == 0)
    {
      gtk_widget_grab_focus (kb_binding_quote);
      return TRUE;
    }

    if (gtk_window_activate_key (GTK_WINDOW (kb_binding_edit_dlg), event))
      return TRUE;
  }

  set_binding_quote (FALSE);

  editkey.keyval = gdk_keyval_to_lower (event->keyval);
  editkey.state = event->state & GXINE_MODIFIER_MASK;

  name = kb2str (&editkey);
  gtk_entry_set_text (GTK_ENTRY (kb_binding_key), name);
  free (name);
  gtk_entry_select_region (GTK_ENTRY (kb_binding_key), 0, 256);

  return TRUE;
}


static gboolean null_edit_key_cb (GtkWidget *widget, GdkEventKey *event,
				  gpointer user_data)
{
  return TRUE;
}


static void quote_edit_key_cb (GtkButton *widget, gpointer user_data)
{
  bq_lock = TRUE;
  gtk_widget_grab_focus (kb_binding_key);
  bq_lock = FALSE;
}


static gboolean close_edit_cb (GtkWidget* widget, gpointer data)
{
  close_edit_window ();
  return TRUE;
}


static void apply_edit (void)
{
  key_binding_t       *key_binding;
  const key_binding_t *found_binding;
  GtkTreeIter	       iter;

  key_binding = lookup_binding (&catch_key_iter);

  editkey.desc = gtk_entry_get_text (GTK_ENTRY (kb_binding_desc));
  editkey.cmd = gtk_entry_get_text (GTK_ENTRY (kb_binding_command));

  found_binding = find_key_binding (editkey.desc, NULL, &iter);
  if (!found_binding)
    found_binding = find_key_binding (NULL, editkey.cmd, &iter);

  if (found_binding && found_binding != key_binding)
  {
    GtkEntry *sel = GTK_ENTRY (strcasecmp (found_binding->desc, editkey.desc)
			       ? kb_binding_command : kb_binding_desc);
    gtk_entry_select_region (sel, 0, 256);
  }
  else
    modify_key_binding (&catch_key_iter, key_binding, &editkey);
}


static JSBool js_input_up (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

  se_log_fncall ("input_up");

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_UP;

  xine_event_send (stream, &event);

  return JS_TRUE;
}

static JSBool js_input_down (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

  se_log_fncall ("input_down");

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_DOWN;

  xine_event_send (stream, &event);

  return JS_TRUE;
}

static JSBool js_input_left (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

  se_log_fncall ("input_left");

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_LEFT;

  xine_event_send (stream, &event);

  return JS_TRUE;
}

static JSBool js_input_right (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

  se_log_fncall ("input_right");

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_RIGHT;

  xine_event_send (stream, &event);

  return JS_TRUE;
}

static JSBool js_input_select (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

  se_log_fncall ("input_select");

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_SELECT;

  xine_event_send (stream, &event);

  return JS_TRUE;
}

static JSBool js_input_menu (JSContext *cx, JSObject *obj, uintN argc,
			     jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  xine_event_t event;
  int menu;

  se_log_fncall ("input_menu");

  se_argc_check (1, "input_menu");
  se_arg_is_int (0, "input_menu");

  JS_ValueToInt32 (cx, argv[0], &menu);

  if (menu < 1 || menu > 7) {
    se->print_cb (se->print_cb_data,
		  _("error: input_menu() expects an int between 1 and 7.\n"));
    return JS_TRUE;
  }

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_MENU1 - 1 + menu;

  xine_event_send (stream, &event);

  return JS_TRUE;
}

static JSBool js_input_menu1 (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

  se_log_fncall ("input_menu1");

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_MENU1;

  xine_event_send (stream, &event);

  return JS_TRUE;
}

static JSBool js_input_menu2 (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

  se_log_fncall ("input_menu2");

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_MENU2;

  xine_event_send (stream, &event);

  return JS_TRUE;
}

static JSBool js_input_menu3 (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

  se_log_fncall ("input_menu3");

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_MENU3;

  xine_event_send (stream, &event);

  return JS_TRUE;
}

static JSBool js_input_previous (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

  se_log_fncall ("input_previous");

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_PREVIOUS;

  xine_event_send (stream, &event);

  return JS_TRUE;
}

static JSBool js_input_next (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

  se_log_fncall ("input_next");

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_NEXT;

  xine_event_send (stream, &event);

  return JS_TRUE;
}

static JSBool js_keybindings_show (JSContext *cx, JSObject *obj, uintN argc, 
				   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

  se_log_fncall ("keybindings_show");

  kb_edit_show ();
  
  return JS_TRUE;
}

/* Xine event mapping data */

static const kb_xine_event_map_t xine_input = {
  "INPUT",
  {
    { "ANGLE_NEXT", XINE_EVENT_INPUT_ANGLE_NEXT },
    { "ANGLE_PREVIOUS", XINE_EVENT_INPUT_ANGLE_PREVIOUS },
    { "DOWN", XINE_EVENT_INPUT_DOWN },
    { "LEFT", XINE_EVENT_INPUT_LEFT },
    { "MENU1", XINE_EVENT_INPUT_MENU1 },
    { "MENU2", XINE_EVENT_INPUT_MENU2 },
    { "MENU3", XINE_EVENT_INPUT_MENU3 },
    { "MENU4", XINE_EVENT_INPUT_MENU4 },
    { "MENU5", XINE_EVENT_INPUT_MENU5 },
    { "MENU6", XINE_EVENT_INPUT_MENU6 },
    { "MENU7", XINE_EVENT_INPUT_MENU7 },
    { "NEXT", XINE_EVENT_INPUT_NEXT },
    { "NUMBER_0", XINE_EVENT_INPUT_NUMBER_0 },
    { "NUMBER_1", XINE_EVENT_INPUT_NUMBER_1 },
    { "NUMBER_10_ADD", XINE_EVENT_INPUT_NUMBER_10_ADD },
    { "NUMBER_2", XINE_EVENT_INPUT_NUMBER_2 },
    { "NUMBER_3", XINE_EVENT_INPUT_NUMBER_3 },
    { "NUMBER_4", XINE_EVENT_INPUT_NUMBER_4 },
    { "NUMBER_5", XINE_EVENT_INPUT_NUMBER_5 },
    { "NUMBER_6", XINE_EVENT_INPUT_NUMBER_6 },
    { "NUMBER_7", XINE_EVENT_INPUT_NUMBER_7 },
    { "NUMBER_8", XINE_EVENT_INPUT_NUMBER_8 },
    { "NUMBER_9", XINE_EVENT_INPUT_NUMBER_9 },
    { "PREVIOUS", XINE_EVENT_INPUT_PREVIOUS },
    { "RIGHT", XINE_EVENT_INPUT_RIGHT },
    { "SELECT", XINE_EVENT_INPUT_SELECT },
    { "UP", XINE_EVENT_INPUT_UP },
    { "0", XINE_EVENT_INPUT_NUMBER_0 },
    { "1", XINE_EVENT_INPUT_NUMBER_1 },
    { "10", XINE_EVENT_INPUT_NUMBER_10_ADD },
    { "2", XINE_EVENT_INPUT_NUMBER_2 },
    { "3", XINE_EVENT_INPUT_NUMBER_3 },
    { "4", XINE_EVENT_INPUT_NUMBER_4 },
    { "5", XINE_EVENT_INPUT_NUMBER_5 },
    { "6", XINE_EVENT_INPUT_NUMBER_6 },
    { "7", XINE_EVENT_INPUT_NUMBER_7 },
    { "8", XINE_EVENT_INPUT_NUMBER_8 },
    { "9", XINE_EVENT_INPUT_NUMBER_9 },
    { "-", -1 }, /* null event */
    { "" }
  }
};


int kb_xine_event_lookup (const kb_xine_event_map_t *map, const char *token)
{
  const char *prefix = strchr (token, '_');
  const kb_xine_event_id_t *lookup;
  char t0;

  if (!map)
    map = &xine_input;

  if (prefix && strlen (prefix) == prefix - token &&
      !strncasecmp (token, map->prefix, prefix - token))
    token = prefix + 1;

  t0 = toupper (token[0]);
  lookup = &map->id[0];

  while (lookup->name[0])
  {
    if (lookup->name[0] == t0 && !strcasecmp (lookup->name, token))
      return lookup->number;
    ++lookup;
  }

  return 0;
}


static const kb_xine_event_map_t xine_vdr = {
  "VDR",
  {
    { "BACK", XINE_EVENT_VDR_BACK },
    { "BLUE", XINE_EVENT_VDR_BLUE },
    { "CHANNELMINUS", XINE_EVENT_VDR_CHANNELMINUS },
    { "CHANNELPLUS", XINE_EVENT_VDR_CHANNELPLUS },
    { "CHANNELS", XINE_EVENT_VDR_CHANNELS },
    { "COMMANDS", XINE_EVENT_VDR_COMMANDS },
    { "FASTFWD", XINE_EVENT_VDR_FASTFWD },
    { "FASTREW", XINE_EVENT_VDR_FASTREW },
    { "GREEN", XINE_EVENT_VDR_GREEN },
    { "MUTE", XINE_EVENT_VDR_MUTE },
    { "PAUSE", XINE_EVENT_VDR_PAUSE },
    { "PLAY", XINE_EVENT_VDR_PLAY },
    { "POWER", XINE_EVENT_VDR_POWER },
    { "RECORD", XINE_EVENT_VDR_RECORD },
    { "RECORDINGS", XINE_EVENT_VDR_RECORDINGS },
    { "RED", XINE_EVENT_VDR_RED },
    { "SCHEDULE", XINE_EVENT_VDR_SCHEDULE },
    { "SETUP", XINE_EVENT_VDR_SETUP },
    { "STOP", XINE_EVENT_VDR_STOP },
    { "TIMERS", XINE_EVENT_VDR_TIMERS },
    { "USER1", XINE_EVENT_VDR_USER1 },
    { "USER2", XINE_EVENT_VDR_USER2 },
    { "USER3", XINE_EVENT_VDR_USER3 },
    { "USER4", XINE_EVENT_VDR_USER4 },
    { "USER5", XINE_EVENT_VDR_USER5 },
    { "USER6", XINE_EVENT_VDR_USER6 },
    { "USER7", XINE_EVENT_VDR_USER7 },
    { "USER8", XINE_EVENT_VDR_USER8 },
    { "USER9", XINE_EVENT_VDR_USER9 },
    { "VOLMINUS", XINE_EVENT_VDR_VOLMINUS },
    { "VOLPLUS", XINE_EVENT_VDR_VOLPLUS },
    { "YELLOW", XINE_EVENT_VDR_YELLOW },
    { "" }
  }
};

static JSBool js_vdr (JSContext *cx, JSObject *obj, uintN argc,
		      jsval *argv, jsval *rval)
{
  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  xine_event_t event;
  JSString *str;
  char *evstr;
  play_item_t *item;

  *rval = JSVAL_TRUE;

  se_log_fncall ("vdr");
  se_argc_check (1, "vdr");
  se_arg_is_string (0, "vdr");

  *rval = JSVAL_FALSE;

  item = playlist_get_item (playlist_get_list_pos());
  if (strncasecmp (item->mrl, "vdr:/", 5))
    return JS_TRUE;

  str = JS_ValueToString (cx, argv[0]);
  evstr = JS_GetStringBytes (str);

  event.type = kb_xine_event_lookup (&xine_vdr, evstr);
  if (!event.type)
    event.type = kb_xine_event_lookup (NULL, evstr);

  if (!event.type)
    return JS_FALSE;

  if (event.type != -1)
  {
    event.data = NULL;
    event.data_length = 0;
    logprintf ("js_vdr: sending event %d\n", event.type);
    xine_event_send (stream, &event);
  }

  *rval = JSVAL_TRUE;
  return JS_TRUE;
}

static void kb_response_cb (GtkDialog *dbox, int response, gpointer data)
{
  switch (response)
  {
  case GTK_RESPONSE_ACCEPT:
    save_key_bindings ();
    break;
  default:
    close_edit_window ();
    close_list_window ();
  }
}

static void edit_response_cb (GtkDialog *dbox, int response, gpointer data)
{
  switch (response)
  {
  case GTK_RESPONSE_REJECT:
    do_edit_binding ();
    break;
  case GTK_RESPONSE_OK:
    apply_edit (); /* and hide */
  default:
    close_edit_window ();
  }
}

static GtkItemFactoryEntry kb_menu_data[] = {
  { N_("/_Reload..."),				NULL,	NULL,			0, "<Branch>" },
  { N_("/Reload.../_Reload bindings"),		NULL,	kb_reload_keymap,	0 },
  { N_("/Reload.../_Merge default bindings"),	NULL,	kb_merge_defaults,	0 },
  { N_("/Reload.../Replace with _default bindings"), NULL, kb_reset_keymap,	0 },
};

void key_events_init (void) {

  GtkWidget            *b, *scrolled_window, *hbox;
  GtkCellRenderer      *cell;
  GtkTreeViewColumn    *column;
  GtkItemFactory *kb_menu_factory;
   
  /*
   * init list store
   */

  kb_store = gtk_list_store_new (3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER);

  default_binding.desc = gettext (default_binding.desc);
  load_key_bindings ();

  /*
   * create (for now invisible) kb_edit dialog
   */

  kb_bindings_list_dlg =
    gtk_dialog_new_with_buttons (_("Keybindings editor"), NULL, 0,
				GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
				GTK_STOCK_CLOSE, GTK_RESPONSE_DELETE_EVENT,
				NULL);
  gtk_window_set_default_size (GTK_WINDOW (kb_bindings_list_dlg), 400, 250);
  g_signal_connect( GTK_OBJECT (kb_bindings_list_dlg), "delete_event",
		      G_CALLBACK (close_cb), NULL );
  g_signal_connect (GTK_OBJECT(kb_bindings_list_dlg), "response",
		    G_CALLBACK (kb_response_cb), NULL);

  /* add a nice tree view widget here */

  tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL(kb_store));  
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);
  gtk_tree_view_set_reorderable (GTK_TREE_VIEW (tree_view), TRUE);

  cell = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (_("Action"), cell,
						     "text", 0, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));

  column = gtk_tree_view_column_new_with_attributes (_("Accelerator key"), cell,
						     "text", 1, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));

  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (scrolled_window), tree_view);

  g_signal_connect (GTK_OBJECT (tree_view), "row-activated",
		    G_CALLBACK (kb_sig_edit_binding), kb_bindings_list_dlg);

  hbox = gtk_hbox_new (FALSE, 2);

  b = gtk_button_new_from_stock (GTK_STOCK_NEW);
  gtk_box_pack_start (GTK_BOX(hbox), b, FALSE, FALSE, 0);
  g_signal_connect (GTK_OBJECT(b), "clicked",
		    G_CALLBACK(kb_new_binding), kb_bindings_list_dlg);

  b = gtk_button_new_with_mnemonic (_("_Edit"));
  gtk_box_pack_start (GTK_BOX(hbox), b, FALSE, FALSE, 0);
  g_signal_connect (GTK_OBJECT(b), "clicked",
		    G_CALLBACK(kb_edit_binding), kb_bindings_list_dlg);

  b = gtk_button_new_from_stock (GTK_STOCK_DELETE);
  gtk_box_pack_start (GTK_BOX(hbox), b, FALSE, FALSE, 0);
  g_signal_connect (GTK_OBJECT(b), "clicked",
		    G_CALLBACK(kb_delete_binding), kb_bindings_list_dlg);

  b = create_menu_tree (&kb_menu_factory, kb_bindings_list_dlg, NULL,
			kb_menu_data, num_menu_items (kb_menu_data));
  gtk_box_pack_end_defaults (GTK_BOX(hbox), b);

  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(kb_bindings_list_dlg)->vbox), hbox,
		      FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(kb_bindings_list_dlg)->vbox), scrolled_window,
		      TRUE, TRUE, 2);

  list_visible = FALSE;

  /*
   * create kb_binding_edit dialogue
   */

  kb_binding_edit_dlg = gtk_dialog_new_with_buttons (_("Keybinding"), NULL, 0,
				GTK_STOCK_UNDO, GTK_RESPONSE_REJECT,
				GTK_STOCK_CANCEL, GTK_RESPONSE_DELETE_EVENT,
				GTK_STOCK_OK, GTK_RESPONSE_OK,
				NULL);
  gtk_dialog_set_default_response (GTK_DIALOG(kb_binding_edit_dlg),
				   GTK_RESPONSE_OK);
  gtk_window_set_default_size (GTK_WINDOW (kb_binding_edit_dlg), 320, 80);
  g_signal_connect (GTK_OBJECT (kb_binding_edit_dlg), "delete_event",
		    G_CALLBACK (close_edit_cb), NULL);
  g_signal_connect (GTK_OBJECT (kb_binding_edit_dlg), "response",
		    G_CALLBACK (edit_response_cb), NULL);

  kb_binding_edit_accel = gtk_accel_group_new ();
  gtk_window_add_accel_group (GTK_WINDOW (kb_binding_edit_dlg),
			      kb_binding_edit_accel);

  b = gtk_table_new (3, 3, FALSE);

  kb_binding_desc = gtk_entry_new_with_max_length (32);
  add_table_row_items (b, 0, _("Description"), 1, kb_binding_desc);

  kb_binding_command = gtk_entry_new_with_max_length (256);
  add_table_row_items (b, 1, _("Command"), 1, kb_binding_command);

  kb_binding_key = gtk_entry_new ();
  gtk_entry_set_editable (GTK_ENTRY (kb_binding_key), FALSE);
  kb_binding_quote = gtk_toggle_button_new_with_mnemonic (_("_Grab"));
  gtk_entry_set_editable (GTK_ENTRY (kb_binding_key), FALSE);
  add_table_row_items (b, 2, _("Accelerator key"), 2, kb_binding_key,
		       kb_binding_quote);
  g_signal_connect (GTK_OBJECT (kb_binding_key), "focus_in_event",
		    G_CALLBACK (start_edit_key_cb), NULL);
  g_signal_connect (GTK_OBJECT (kb_binding_edit_dlg), "key_press_event",
		    G_CALLBACK (do_edit_key_cb), NULL);
  g_signal_connect (GTK_OBJECT (kb_binding_key), "focus_out_event",
		    G_CALLBACK (end_edit_key_cb), NULL);
  g_signal_connect (GTK_OBJECT (kb_binding_key), "key_press_event",
		    G_CALLBACK (null_edit_key_cb), NULL);
  g_signal_connect (GTK_OBJECT (kb_binding_quote), "clicked",
		    G_CALLBACK (quote_edit_key_cb), NULL);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (kb_binding_edit_dlg)->vbox ), b);

  /* script engine functions */

  {
    static const se_f_def_t defs[] = {
      { "input_up", js_input_up, 0, 0, SE_GROUP_INPUT, NULL, NULL },
      { "input_down", js_input_down, 0, 0, SE_GROUP_INPUT, NULL, NULL },
      { "input_left", js_input_left, 0, 0, SE_GROUP_INPUT, NULL, NULL },
      { "input_right", js_input_right, 0, 0, SE_GROUP_INPUT, NULL, NULL },
      { "input_select", js_input_select, 0, 0, SE_GROUP_INPUT, NULL, NULL },
      { "input_menu", js_input_menu, 0, 0,
	SE_GROUP_INPUT, N_("int"), N_("range is 1 to 7") },
      { "input_menu1", js_input_menu1, 0, 0, SE_GROUP_INPUT, NULL, NULL },
      { "input_menu2", js_input_menu2, 0, 0, SE_GROUP_INPUT, NULL, NULL },
      { "input_menu3", js_input_menu3, 0, 0, SE_GROUP_INPUT, NULL, NULL },
      { "input_previous", js_input_previous, 0, 0, SE_GROUP_INPUT, NULL, NULL },
      { "input_next", js_input_next, 0, 0, SE_GROUP_INPUT, NULL, NULL },
      { "keybindings_show", js_keybindings_show, 0, 0,
	SE_GROUP_DIALOGUE, NULL, NULL },
      { "vdr", js_vdr, 0, 0,
	SE_GROUP_EXTERNAL, N_("string"), N_("event; returns true if sent") },
      { NULL }
    };
    se_defuns (gse, NULL, defs);
  }
}
