/*
 * Copyright (C) 2002-2003 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: playlist.c,v 1.64 2003/04/08 22:07:47 guenter Exp $
 *
 * playlist implementation
 *
 * .pls parser stolen from totem (C) 2002 Bastien Nocera
 */

#include <config.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <X11/Xlib.h>

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>

#include <pthread.h>

#include "globals.h"
#include "gtkxine.h"
#include "playlist.h"
#include "http.h"
#include "utils.h"
#include "play_item.h"
#include "actions.h"

#include "xmlparser.h"

/*
#define LOG
*/

static int             is_visible;
static GtkListStore   *pl_store;
static GtkWidget      *dlg, *tree_view;
static GtkWidget      *repeat_button, *random_button;
static int             goto_pos, goto_time;
static char            logo_mrl[1024];
static int             logo_mode;
static pthread_t       play_thread;
static int             cur_list_pos;
static char           *cur_mrl;

static gboolean close_cb (GtkWidget* widget, gpointer data) {
  is_visible = FALSE;
  gtk_widget_unmap (dlg);

  return TRUE;
}

void playlist_clear (void) {
  gtk_xine_stop (GTK_XINE(gtx));
  gtk_list_store_clear (pl_store);
}

static void clear_cb (GtkWidget* widget, gpointer data) {
  playlist_clear ();
  playlist_logo ();
}

static void del_cb (GtkWidget* widget, gpointer data) {

  GtkTreeView       *tree = (GtkTreeView *) data;
  GtkTreeSelection  *sel;
  GtkTreeIter        iter;  

  sel = gtk_tree_view_get_selection (tree);

  if (gtk_tree_selection_get_selected (sel, NULL, &iter)) {

    gtk_list_store_remove (pl_store, &iter);
  }
}

static void add_cb (GtkWidget* widget, gpointer data) {

  gchar **files;

  files = modal_multi_file_dialog ("Select files to add to playlist", NULL);

  if (files) {

    int i;
    gchar *mrl;
    
    i = 0;
    while ((mrl = files[i]) ) {

      playlist_add_mrl (mrl);

      i++;
    }

    g_strfreev (files);

  }
}

static void edit_cb (GtkWidget* widget, gpointer data) {

  GtkTreeView       *tree = (GtkTreeView *) data;
  GtkTreeSelection  *sel;
  GtkTreeIter        iter;  

  sel = gtk_tree_view_get_selection (tree);

  if (gtk_tree_selection_get_selected (sel, NULL, &iter)) {
    GValue       v;
    play_item_t *play_item;

    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store),
			      &iter, 2, &v);
    play_item = g_value_peek_pointer (&v);
      
    play_item_edit (play_item);

    gtk_list_store_set (pl_store, &iter, 
			0, play_item->title, 
			1, play_item->mrl, 2, play_item, -1);

    g_value_unset (&v);
  }
}

static int playlist_load (char *fname) {

  char  *plfile;

  if (!fname)
    fname = g_strconcat(g_get_home_dir(), "/.gxine/playlist", NULL);

  plfile = read_entire_file_ascii (fname);

  if (plfile) {

    xml_node_t *node;

    xml_parser_init (plfile, strlen (plfile), XML_PARSER_CASE_INSENSITIVE);

    if (xml_parser_build_tree (&node)>=0) {

      if (!strcasecmp (node->name, "asx")) {

	playlist_clear ();

	node = node->child;

	while (node) {

	  if (!strcasecmp (node->name, "entry")) {

	    play_item_t *play_item = play_item_load (node->child);
	    GtkTreeIter  iter;

	    gtk_list_store_append (pl_store, &iter);

	    gtk_list_store_set (pl_store, &iter, 0, play_item->title, 1, 
				play_item->mrl, 2, play_item, -1);
	  }

	  node = node->next;
	}
      } else {
	display_error ("Playlist load failed",
		       "Loading of playlist file failed:\n"
		       "%s is not an asx file", fname);
	return 0;
      }
    } else {
      display_error ("Playlist load failed",
		     "Loading of playlist file failed:\n"
		     "%s is not a valid xml/asx file", fname);
      return 0;
    }

  } else {
    display_error ("Playlist load failed",
		   "Loading of playlist file failed:\n"
		   "Failed to open file '%s'", fname);
    return 0;
  }
  return 1;
}

void playlist_save (char *fname) {

  FILE *f;

  if (!fname)
    fname = g_strconcat(g_get_home_dir(), "/.gxine/playlist", NULL);

  f = fopen (fname, "w");
  if (f) {

    GtkTreeIter iter;

    fprintf (f, "<ASX VERSION=\"3.0\">\n");

    if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (pl_store), &iter)) {

      do {

	GValue vi;
	play_item_t  *play_item;

	memset (&vi, 0, sizeof (GValue));
	gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store),
				  &iter, 2, &vi);
	play_item = g_value_peek_pointer (&vi);
	
	play_item_save (play_item, f, 2);

	g_value_unset (&vi);
      } while (gtk_tree_model_iter_next (GTK_TREE_MODEL(pl_store), &iter));
    }

    fprintf (f, "</ASX>\n");
    fclose (f);
  } else {
    printf ("playlist: save failed!\n");
  }
}

static void open_cb (GtkWidget* widget, gpointer data) {

  char *filename;

  filename = modal_file_dialog ("select playlist...", "*.asx");

  if (!filename)
    return;

  playlist_load (filename);
    
}

static void save_as_cb (GtkWidget* widget, gpointer data) {

  char *filename;

  filename = modal_file_dialog ("save playlist as...", "*.asx");

  if (!filename)
    return;

  playlist_save (filename);
}

play_item_t *playlist_get_item (int pos) {

  GtkTreeIter        iter;  

  if (pos<0)
    pos = 0;

  if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (pl_store),
				     &iter, NULL, pos)) {

    GValue       v;
    play_item_t *play_item;
    
    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (pl_store),
			      &iter, 2, &v);
    play_item = g_value_peek_pointer (&v);
    g_value_unset (&v);
      
    return play_item;
 } else
   return NULL;
}

int playlist_get_list_pos (void) {
  return cur_list_pos;
}

int playlist_size (void) {
  return gtk_tree_model_iter_n_children (GTK_TREE_MODEL (pl_store), NULL);
}

static int read_ini_line_int (char **lines, const char *key) {
  int retval = -1;
  int i;

  if (lines == NULL || key == NULL)
    return -1;

  for (i = 0; (lines[i] != NULL && retval == -1); i++) {
    if (g_strncasecmp (lines[i], key, strlen (key)) == 0) {
      char **bits;

      bits = g_strsplit (lines[i], "=", 2);
      if (bits[0] == NULL || bits [1] == NULL) {
	g_strfreev (bits);
	return -1;
      }

      retval = (gint) g_strtod (bits[1], NULL);
      g_strfreev (bits);
    }
  }

  return retval;
}

static char* read_ini_line_string (char **lines, const char *key) {
  char *retval = NULL;
  int i;

  if (lines == NULL || key == NULL)
    return NULL;

  for (i = 0; (lines[i] != NULL && retval == NULL); i++) {
    if (g_strncasecmp (lines[i], key, strlen (key)) == 0) {
      char **bits;

      bits = g_strsplit (lines[i], "=", 2);
      if (bits[0] == NULL || bits [1] == NULL) {
	g_strfreev (bits);
	return NULL;
      }

      retval = g_strdup (bits[1]);

      g_strfreev (bits);
    }
  }

  return retval;
}

static int playlist_load_m3u (const char *mrl) {
  gboolean  retval = FALSE;
  char     *contents, **lines;
  int       size, i, first;
  char     *extension;
  char     *suffix = NULL;

  first = -1;

  if (! (contents = read_entire_file (mrl, &size)) )
    return FALSE;

  lines = g_strsplit (contents, "\n", 0);
  g_free (contents);

  for (i = 0; lines[i] != NULL; ++i) {
    /* Either it's a URI, or it has a proper path */
    if (strstr(lines[i], "://") != NULL
	|| lines[i][0] == G_DIR_SEPARATOR)  {
      extension = strrchr(lines[i], '.');
      if (extension && !strncasecmp (extension, ".mp3", 4)) {
          suffix="#demux:mp3";
      }
      else if (extension && !strncasecmp (extension, ".ogg", 4)) {
          suffix="#demux:ogg";
      }

      if (first >= 0) {
          playlist_add_mrl (g_strconcat(lines[i],suffix,NULL));
      }
      else {
          first = playlist_add_mrl (g_strconcat(lines[i],suffix,NULL));
      }

      retval = TRUE;
    }
  }

  g_strfreev (lines);

  if (retval)
    return first;
  else
    return -1;
}

static int playlist_load_smil (const char *mrl) {
  gboolean  retval = FALSE;
  char     *contents, **lines;
  int       size, i, first;
  char     *begin, *end;

  first = -1;

  if (! (contents = read_entire_file (mrl, &size)) )
    return FALSE;

  lines = g_strsplit (contents, "\n", 0);
  g_free (contents);

  for (i = 0; lines[i] != NULL; ++i) {

    if (strstr(lines[i], "<audio src=") != NULL
	|| lines[i][0] == G_DIR_SEPARATOR)  {
      begin=strstr(lines[i], "<audio src=")+12;
      end=strchr(begin, '"');
      if (!end) end=strchr(begin, '\'');
      if (!end) return -1;
      *end=0;
#ifdef LOG
      printf("playlist: got '%s'\n", begin);
#endif
      if (first >= 0) {
          playlist_add_mrl (begin);
      }
      else {
          first = playlist_add_mrl (begin);
      }

      retval = TRUE;
    }
    if (strstr(lines[i], "<video src=") != NULL
	|| lines[i][0] == G_DIR_SEPARATOR)  {
      begin=strstr(lines[i], "<video src=")+12;
      end=strchr(begin, '"');
      if (!end) end=strchr(begin, '\'');
      if (!end) return -1;
      *end=0;
#ifdef LOG
      printf ("playlist: got '%s'\n", begin);
#endif
      if (first >= 0) {
          playlist_add_mrl (begin);
      }
      else {
          first = playlist_add_mrl (begin);
      }

      retval = TRUE;
    }

  }

  g_strfreev (lines);

  if (retval)
    return first;
  else
    return -1;
}

int playlist_add (play_item_t *play_item) {

  GtkTreeIter  iter;
  GtkTreePath *path;
  gint        *indices;
  gint         ret;

  gtk_list_store_append (pl_store, &iter);
  gtk_list_store_set (pl_store, &iter, 0, play_item->title, 
		      1, play_item->mrl, 2, play_item, -1);

  /*
   * find out entry number in list
   */

  path = gtk_tree_model_get_path (GTK_TREE_MODEL (pl_store), &iter);
  indices = gtk_tree_path_get_indices (path);
  ret = indices[0];
  gtk_tree_path_free (path);

  return ret;
}

void playlist_insert  (play_item_t *play_item, int pos) {

  GtkTreeIter  iter;

  gtk_list_store_insert (pl_store, &iter, pos);
  gtk_list_store_set (pl_store, &iter, 0, play_item->title, 
		      1, play_item->mrl, 2, play_item, -1);
}


static gint playlist_add_int (const char *mrl) {

  play_item_t *play_item;

  play_item = play_item_new (mrl, mrl, 0);

  return playlist_add (play_item);
}

static int playlist_load_ram (const char *mrl) {
  gboolean  retval = FALSE;
  char      contents[1024], mime_type[256], **lines;
  char      *contents_ptr;
  int       size, i, first;
#if 0 /* FIXME: remove ? */
  char     *extension;
  char     *suffix;
#endif
  int       fd;
  int       pos; 

  first = -1;
  sprintf(mime_type,"none");

  /* get first 4 bytes to decide whether it is a real media
   * file or a playlist.
   */
  size = 4;

  if (!strncasecmp (mrl, "http://", 7)) {
  
    size = http_peek (mrl,size, contents, mime_type);
    if (size <= 0)
      return -1;

  } else {

    fd = open (mrl, O_RDONLY);
    if (fd<0)
      return -1;

    size = read (fd, contents, size);
    close (fd);
  }
  if (contents[0]=='.' 
      && contents[1]=='R' 
      && contents[2]=='M' 
      && contents[3]=='F')
  {
    /* this is a real media file ... */
    /* add it directly to the playlist. */

    pos = playlist_add_int (mrl);

    return pos;
  }
    
  if (! (contents_ptr = read_entire_file (mrl, &size)) )
    return FALSE;

  lines = g_strsplit (contents_ptr, "\n", 0);
  g_free (contents_ptr);

  for (i = 0; lines[i] != NULL; ++i) {
    
    /* ignore comments */
    if (lines[i][0]=='#') continue;
    
    /* honor --stop-- lines */
    if (strstr(lines[i], "--stop--") != NULL) break;
    
    /* Either it's a rtsp, or a pnm mrl, but we also match http mrls here. */
    if (strstr(lines[i], "rtsp://") != NULL
	|| strstr(lines[i], "pnm://") != NULL
	|| strstr(lines[i], "http://") != NULL)  {
      /* we assume real media */
      
      /* delete \r at the end */
      contents_ptr=strchr(lines[i],'\r');
      if (contents_ptr) *contents_ptr=0;

      if (first >= 0) {
	playlist_add_mrl (lines[i]);
      }
      else {
	first = playlist_add_mrl (lines[i]);
      }
      retval = TRUE;
    }
    
  }

  g_strfreev (lines);

  if (retval)
    return first;
  else
    return -1;
}

static int playlist_load_pls (const char *mrl) {

  gboolean  retval = FALSE;
  char     *contents=NULL, **lines;
  int       size, i, num_entries;
  char      mime_type[256];

  /* http or local file ? */

  if (!strncmp (mrl, "http://", 7)) {

    /* get mime type */
    http_peek(mrl, 0, contents, mime_type);

    if (!strcmp(mime_type, "audio/mpeg")) {
      /* add directly to playlist */

      return playlist_add_int (mrl);
    }

    if (! (contents = http_download (mrl, &size)))
      return FALSE;
  } else {

    if (! (contents = read_entire_file (mrl, &size)) )
      return FALSE;
  }

  lines = g_strsplit (contents, "\n", 0);
  g_free (contents);

  /* [playlist] */
  if (g_strncasecmp (lines[0], "[playlist]",
		     (gsize)strlen ("[playlist]")) != 0)
    goto bail;

  /* numberofentries=? */
  num_entries = read_ini_line_int (lines, "numberofentries");
  if (num_entries == -1)
    goto bail;

  for (i = 1; i <= num_entries; i++) {
    char *file, *title;
    char *file_key, *title_key;

    file_key = g_strdup_printf ("file%d", i);
    title_key = g_strdup_printf ("title%d", i);

    file = read_ini_line_string (lines, (const char*)file_key);
    title = read_ini_line_string (lines, (const char*)title_key);

    g_free (file_key);
    g_free (title_key);

    if (file != NULL) {

      int pos = strlen(file)-1;

      while ( (pos>0) && (file[pos]<32) ) {
	file[pos] = 0;
	pos --;
      }

      return playlist_add_mrl (file);

      g_free (file);
      g_free (title);
    } else {
      g_free (title);
    }
  }

bail:
  g_strfreev (lines);

  return retval;
}

static int playlist_load_asx (const char *mrl) {

  char             *contents;
  char            **lines;
  int               size, res, ret, i;
  xml_node_t       *xml_tree;
  xml_node_t       *current_node_l1;
  xml_node_t       *current_node_l2;
  xml_property_t   *property;
  int               asx_version;

  ret = -1;

  if (! (contents = read_entire_file (mrl, &size)) )
    return ret;

  xml_parser_init (contents, size, XML_PARSER_CASE_INSENSITIVE);
  res = xml_parser_build_tree (&xml_tree);

  if (res) {
    /* humm, maybe the asx contains no xml? */
    /* try to find mms urls by hand */
    lines = g_strsplit (contents, "\n", 0);
    g_free (contents);

    for (i = 0; lines[i] != NULL; ++i) {
      /* look for mms:// in a line */
      if (strstr(lines[i], "mms://"))  {
        /* delete \r at the end */
        contents=strchr(lines[i],'\r');
        if (contents) *contents=0;
        /* add to playlist */
	if (ret >= 0)
          playlist_add_mrl (strstr(lines[i], "mms://"));
	else
	  ret = playlist_add_mrl (strstr(lines[i], "mms://"));
      }
    }
    return ret;
  }

  /* check ASX */
  if (strcmp(xml_tree->name, "ASX") == 0) {
#ifdef LOG
    printf("playlist: ASX tag detected\n");
#endif

    /* check version */
    asx_version = xml_parser_get_property_int (xml_tree, "VERSION", 3);

    if (asx_version == 3) {

#ifdef LOG
      printf("playlist: VERSION 3.0 detected\n");
#endif

      /* play each entry */
      current_node_l1 = xml_tree->child;
      while (current_node_l1) {
	/* entry */
	if (strcmp(current_node_l1->name, "ENTRY") == 0) {
	  char *title;
	  
	  title = NULL;
	  
#ifdef LOG
	  printf("playlist: ENTRY detected\n");
#endif
	  
	  /* play the first ref which is playable */
	  current_node_l2 = current_node_l1->child;
	  while (current_node_l2) {
	    if (strcmp(current_node_l2->name, "REF") == 0) {
#ifdef LOG
	      printf("playlist: REF detected\n");
#endif
	      /* find href property */
	      property = current_node_l2->props;
	      while ((property) && (strcmp(property->name, "HREF"))) {
		property = property->next;
	      }
	      if (property) {
		play_item_t *item;
		
#ifdef LOG
		printf("playlist: HREF property detected\n");
		
		printf ("playlist: found an mrl: %s\n",
			property->value);
#endif
		if (title)
		  item = play_item_new (title, property->value, 0);
		else
		  item = play_item_new (property->value, property->value, 0);
		
		
		if (ret >= 0)
		  playlist_add (item);
		else
		  ret = playlist_add (item);
		
		/* jump to next entry */
		current_node_l2 = NULL;
		
#if 0 /* FIXME */
		/* try to play REF */
		if (demux_asx_play(this, property->value) == 1) {
		  /* jump to next entry */
		  printf("playlist: play next entry or entryref\n");
		  current_node_l2 = NULL;
		} else {
		  /* try next REF */
		  printf("playlist: try next REF\n");
		}
		
#endif
		
	      } else {
		printf("playlist: no HREF property\n");
	      } /* if (property) */
	      
	    } else if (strcmp(current_node_l2->name, "TITLE") == 0) {
	      title = current_node_l2->data;
	    } else {
#ifdef LOG
	      printf("playlist: unknown tag %s detected\n", 
		     current_node_l2->name);
#endif
	    } /* end if (strcmp(current_node_l2->name, "REF") == 0) */
	    if (current_node_l2) {
	      current_node_l2 = current_node_l2->next;
	    }
	  } /* end while */
	} else {
	  
	  /* entryref */
	  if (strcmp(current_node_l1->name, "ENTRYREF") == 0) {
#ifdef LOG
	    printf("playlist: ENTRYREF detected\n");
#endif
	    property = current_node_l1->props;
	    while ((property) && (strcmp(property->name, "HREF"))) {
	      property = property->next;
	    }
	    if (property) {
#ifdef LOG
	      printf("playlist: HREF property detected\n");
	      
	      printf ("playlist: found an mrl: %s\n",
		      property->value);
#endif
	      
	      if (ret >= 0)
		playlist_add_mrl (property->value);
	      else
		ret = playlist_add_mrl (property->value);
	      
	      /* jump to next entry */
	      current_node_l2 = NULL;
	      
#if 0 /* FIXME */
	      
	      /* try to play HREF */
	      if (demux_asx_play(this, property->value) == 1) {
		/* jump to next entry */
		printf("playlist: play next entry or entryref\n");
		current_node_l1 = NULL;
	      } else {
		/* try next REF */
		printf("playlist: try next REF\n");
	      }
#endif
	    } else {
	      printf("playlist: no HREF property\n");
	    } /* if (property) */
	  } else {
	    
	    /* title */
	    if (strcmp(current_node_l1->name, "TITLE") == 0) {
#ifdef LOG
	      printf("playlist: TITLE detected\n");
#endif
	      /* change title */
	    } else {
	      printf("playlist: unknown tag %s detected\n", 
		     current_node_l1->name);
	    }
	  }
	}
	current_node_l1 = current_node_l1->next;
      }
      
    } else {
      printf("playlist: sorry, asx version %d not implemented yet\n",
	     asx_version);
    }
  } else {
    printf("playlist: no ASX tag\n");
  }


  return ret;
}

int playlist_add_mrl (const char *mrl) {

  /* printf ("playlist_add called >%s<\n", mrl); */

  /*
   * is this mrl in fact a simple filename?
   */

  if ( !strstr (mrl, ":/") ) {

    char *extension;

    /* is it a playlist file? */

    extension = strrchr(mrl, '.');

    if (extension && !strncasecmp (extension, ".pls", 4)) {

      return playlist_load_pls (mrl);

    } else if (extension && !strncasecmp (extension, ".m3u", 4)) {

      return playlist_load_m3u (mrl);

    } else if (extension && !strncasecmp (extension, ".ra", 4)) {

      return playlist_load_ram (mrl);

    } else if (extension && !strncasecmp (extension, ".rm", 4)) {

      return playlist_load_ram (mrl);

    } else if (extension && !strncasecmp (extension, ".ram", 4)) {

      return playlist_load_ram (mrl);

    } else if (extension && !strncasecmp (extension, ".rpm", 4)) {

      return playlist_load_ram (mrl);

    } else if (extension && !strncasecmp (extension, ".lsc", 4)) {

      return playlist_load_ram (mrl);

    } else if (extension && !strncasecmp (extension, ".pl", 3)) {

      return playlist_load_ram (mrl);

    } else if (extension && !strncasecmp (extension, ".asx", 4)) {

      return playlist_load_asx (mrl);

    } else if (extension && !strncasecmp (extension, ".smi", 4)) {

      return playlist_load_smil (mrl);

    } else if (!strncmp (mrl, "cdda:", 5)) {

      return playlist_add_int (strdup(mrl));

    } else if (mrl[0]!='/') {

      /* make filename an absolute pathname */

      char  buf[1024];
      char *pathname;
      int   l1, l2;

      getcwd (buf, 1024);

      l1 = strlen(buf);
      l2 = strlen(mrl);

      pathname = g_malloc (l1+l2+2);
      memcpy (pathname, buf, l1);
      pathname[l1]='/';
      memcpy (&pathname[l1+1], mrl, l2);
      pathname[l1+l2+1]=0;

      return playlist_add_int (pathname);

    } else
      return playlist_add_int (strdup(mrl));


  } else if (!strncasecmp (mrl, "http://", 7)) {

    /*
     * could still be a pls/asx playlist, but if so we
     * need to download it first
     */

    char *extension;

    /* is it a playlist file? */

    extension = strrchr(mrl, '.');

    if (extension && !strncasecmp (extension, ".pls", 4)) {

      return playlist_load_pls (mrl);

    } else if (extension && !strncasecmp (extension, ".m3u", 4)) {

      return playlist_load_m3u (mrl);

    } else if (extension && !strncasecmp (extension, ".ra", 4)) {

      return playlist_load_ram (mrl);

    } else if (extension && !strncasecmp (extension, ".rm", 4)) {

      return playlist_load_ram (mrl);

    } else if (extension && !strncasecmp (extension, ".ram", 4)) {

      return playlist_load_ram (mrl);

    } else if (extension && !strncasecmp (extension, ".rpm", 4)) {

      return playlist_load_ram (mrl);

    } else if (extension && !strncasecmp (extension, ".lsc", 4)) {

      return playlist_load_ram (mrl);

    } else if (extension && !strncasecmp (extension, ".pl", 3)) {

      return playlist_load_ram (mrl);

    } else if (extension && !strncasecmp (extension, ".asx", 4)) {

      return playlist_load_asx (mrl);

    } else if (extension && !strncasecmp (extension, ".smi", 4)) {

      return playlist_load_smil (mrl);

    } else
      return playlist_add_int (strdup(mrl));

  } else if (!strncasecmp (mrl, "mmshttp://", 10)) {

#ifdef LOG
    printf ("playlist: adding regular asx from mmshttp %s\n", mrl);
#endif

    return playlist_load_asx (&mrl[3]);

  } else {
#ifdef LOG
    printf ("playlist: adding regular mrl %s\n", mrl);
#endif
    return playlist_add_int (strdup(mrl));
  }

#ifdef LOG
  printf ("playlist: help, how did i get here? \n");
#endif
  
  return 0;
}

static void show_metadata () {

  char  title[2048];
  const char *t;

  if ( (t=gtk_xine_get_meta_info (GTK_XINE(gtx), XINE_META_INFO_TITLE)) ) {
    const char *a;

    if ( (a=gtk_xine_get_meta_info (GTK_XINE(gtx), XINE_META_INFO_ARTIST)) ) {
      snprintf (title, 2048, "%s (%s) - gxine %s", t, a, VERSION);
      infobar_line1 (bar, "%s (%s)", t, a);
    } else {
      snprintf (title, 2048, "%s - gxine %s", t, VERSION);
      infobar_line1 (bar, "%s", t);
    }
  } else {
    if (strlen (cur_mrl) > 0) {

      char *str = strrchr (cur_mrl, '/');

      if (str) {
	snprintf (title, 2048, "%s - gxine %s", str+1, VERSION);
	infobar_line1 (bar, "%s", str+1);
      } else {
	snprintf (title, 2048, "%s - gxine %s", cur_mrl, VERSION);
	infobar_line1 (bar, "%s", cur_mrl);
      }
    } else {
      snprintf (title, 2048, "gxine %s", VERSION);
      infobar_line1 (bar, "%s", title);
    }
  }
  normalize_to_ascii (title);
  gtk_window_set_title (GTK_WINDOW (app), title);

  /*
   * display some metainfo
   */

  if (gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_HAS_AUDIO)
      && gtk_xine_get_meta_info (GTK_XINE(gtx), XINE_META_INFO_VIDEOCODEC)
      && gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_HAS_VIDEO)
      && gtk_xine_get_meta_info (GTK_XINE(gtx), XINE_META_INFO_AUDIOCODEC))
    infobar_line2 (bar, "%d kBit/s %s, %dx%d %s, %d kHz %d kBit/s %s",
		   gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_BITRATE)/1000,
		   gtk_xine_get_meta_info   (GTK_XINE(gtx), XINE_META_INFO_SYSTEMLAYER),
		   gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_VIDEO_WIDTH),
		   gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_VIDEO_HEIGHT),
		   gtk_xine_get_meta_info   (GTK_XINE(gtx), XINE_META_INFO_VIDEOCODEC),
		   gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_AUDIO_SAMPLERATE)/1000,
		   gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_AUDIO_BITRATE)/1000,
		   gtk_xine_get_meta_info   (GTK_XINE(gtx), XINE_META_INFO_AUDIOCODEC));
  else if (gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_HAS_AUDIO)
	   && gtk_xine_get_meta_info (GTK_XINE(gtx), XINE_META_INFO_AUDIOCODEC))
    infobar_line2 (bar, "%s, %d kHz %d kBit/s %s",
		   gtk_xine_get_meta_info   (GTK_XINE(gtx), XINE_META_INFO_SYSTEMLAYER),
		   gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_AUDIO_SAMPLERATE)/1000,
		   gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_AUDIO_BITRATE)/1000,
		   gtk_xine_get_meta_info   (GTK_XINE(gtx), XINE_META_INFO_AUDIOCODEC));
  else if (gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_HAS_VIDEO)
	   && gtk_xine_get_meta_info (GTK_XINE(gtx), XINE_META_INFO_VIDEOCODEC))
    infobar_line2 (bar, "%s, %dx%d %s",
		   gtk_xine_get_meta_info   (GTK_XINE(gtx), XINE_META_INFO_SYSTEMLAYER),
		   gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_VIDEO_WIDTH),
		   gtk_xine_get_stream_info (GTK_XINE(gtx), XINE_STREAM_INFO_VIDEO_HEIGHT),
		   gtk_xine_get_meta_info   (GTK_XINE(gtx), XINE_META_INFO_VIDEOCODEC));

}

static void *play_exec (void *queue_gen) {

  pthread_mutex_lock (&engine_lock);

  gdk_threads_enter();
  infobar_line1 (bar, "opening...");
  infobar_line2 (bar, "");
  gdk_threads_leave();

  if (!gtk_xine_open (GTK_XINE(gtx), cur_mrl)) {

    gdk_threads_enter();
    switch (gtk_xine_get_error (GTK_XINE (gtx))) {
    case XINE_ERROR_NO_INPUT_PLUGIN:

      display_error ("xine engine error",
		     "xine engine failed to start.\n\n"
		     "No input plugin found.\n"
		     "Maybe the file does not exist, has wrong "
		     "permissions or\nurl syntax error");

      break;
    case XINE_ERROR_NO_DEMUX_PLUGIN:

      display_error ("xine engine error",
		     "xine engine failed to start.\n\n"
		     "No demuxer found - stream format not recognized");

      break;

    default:
      display_error ("xine engine error",
		     "xine engine failed to start.\n\n"
		     "unknown error.");
    }

    infobar_line1 (bar, "gxine %s", VERSION);
    infobar_line2 (bar, "");

    gdk_threads_leave();

    pthread_mutex_unlock (&engine_lock);
    pthread_exit (NULL);

    return NULL;
  }

  gdk_threads_enter();
  if (logo_mode) {
    char  title[2048];
    sprintf (title, "gxine %s", VERSION);
    infobar_line1 (bar, title);
    infobar_line2 (bar, "");
    gtk_window_set_title (GTK_WINDOW (app), title);
  } else
    show_metadata ();
  gdk_threads_leave();

  gtk_xine_play (GTK_XINE(gtx), goto_pos, goto_time);
  goto_pos = 0;
  goto_time = 0;

  pthread_mutex_unlock (&engine_lock);

  pthread_exit (NULL);

  return NULL;
}

void playlist_play_from (int list_pos, int pos, int pos_time) {

  int          err;
  GtkTreePath *path;
  char         indices[10];
  play_item_t *play_item;
  GList       *option;

  logo_mode = 0;

  if (list_pos<0)
    list_pos = 0;

  play_item = playlist_get_item (list_pos);

  if (!play_item) {
    cur_list_pos = 0;
    playlist_logo ();
    return;
  }

  cur_mrl   = play_item->mrl;
  goto_pos  = pos;
  goto_time = pos_time;

  snprintf (indices, 10, "%d", list_pos);
  path = gtk_tree_path_new_from_string (indices);
  gtk_tree_view_set_cursor (GTK_TREE_VIEW (tree_view), 
			    path, NULL, FALSE);
  gtk_tree_path_free (path);

  cur_list_pos  = list_pos;

  /*
   * execute any optional commands which may be associated with this play item
   * (e.g. set volume etc.)
   */

  option = play_item->options;

  while (option) {

    char *cmd = (char *) option->data;

    action_exec (cmd, NULL, NULL);

    option = g_list_next (option);
  }

  /*
   * start seperate thread, so xine_open will not block ui
   */

  if ((err = pthread_create (&play_thread,
			     NULL, play_exec, NULL)) != 0) {
    printf ("playlist: can't create new thread (%s)\n",
	    strerror(err));
    abort();
  }

  pthread_detach (play_thread);
}

void playlist_play (int list_pos) {
  play_item_t *item;

  if (list_pos<0)
    list_pos = 0;

  item = playlist_get_item (list_pos);
  if (!item) {
    playlist_logo ();
    return;
  }  

  playlist_play_from (list_pos, 0, item->start_time);
}


void playlist_logo (void) {

  int err;

  logo_mode = 1;

  cur_mrl = logo_mrl;

  /*
   * start seperate thread, so xine_open will not block ui
   */

  if ((err = pthread_create (&play_thread,
			     NULL, play_exec, NULL)) != 0) {
    fprintf (stderr, "events: can't create new thread (%s)\n",
	     strerror(err));
    abort();
  }

  pthread_detach (play_thread);

}

void playlist_show (void) {

  if (is_visible) {
    is_visible = FALSE;
    gtk_widget_hide (dlg);
  } else {
    is_visible = TRUE;
    gtk_widget_show_all (dlg);
    gtk_widget_map (dlg);
  }
}

static gboolean button_press_lcb (GtkWidget *widget, GdkEventButton *event, 
				  gpointer data) {

  if (event && (event->type==GDK_2BUTTON_PRESS)) {
    GtkTreeSelection  *sel;
    GtkTreeIter        iter;  

    sel = gtk_tree_view_get_selection (GTK_TREE_VIEW(tree_view));

    if (gtk_tree_selection_get_selected (sel, NULL, &iter)) {

      GtkTreePath *path;
      gint        *indices, item;

      path = gtk_tree_model_get_path (GTK_TREE_MODEL (pl_store), &iter);
      indices = gtk_tree_path_get_indices (path);
      item = indices[0];
      gtk_tree_path_free (path);

      playlist_play (item); 
    }
  }
  return FALSE;
}

static int has_been_played (int pos) {

  play_item_t *item;

  item = playlist_get_item (pos);

  return item->played;
}

static void play_next (void) {

  int is_repeat, is_random;

  is_repeat = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (repeat_button));
  is_random = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (random_button));

  if (is_random) {

    int count;

    cur_list_pos = rand() % playlist_size();

    count = 0;
    while ( (count<playlist_size()) && has_been_played (cur_list_pos) ) {
      cur_list_pos = (cur_list_pos + 1) % playlist_size();
      
      count++;
    }

    if (has_been_played (cur_list_pos))
      cur_list_pos = playlist_size()+1;

  } else {
    cur_list_pos++;
  }

  if ( is_repeat && (cur_list_pos>=playlist_size())) {
    if (is_random) {
      int i;

      /* reset all played entries */

      for (i=0; i<playlist_size(); i++) {
	play_item_t *item;
	item = playlist_get_item (i);
	item->played = 0;
      }

      cur_list_pos = rand() % playlist_size();
      
    } else {
      cur_list_pos = 0; 
    }
  }

  if (cur_list_pos<playlist_size()) {
    play_item_t *item;
    item = playlist_get_item (cur_list_pos);
    item->played = 1;

    playlist_play (cur_list_pos);
  } else {
    cur_list_pos = 0;
    playlist_logo ();
  }
}

static void xine_event_cb (void *user_data, const xine_event_t *event) {

  /*
  printf ("playlist: got an event %d\n", event->type); 
  */

  switch (event->type) {
  case XINE_EVENT_UI_PLAYBACK_FINISHED:

    gdk_threads_enter();
    if (!logo_mode) {

      play_next ();

    }
    gdk_threads_leave();
    break;
  case XINE_EVENT_PROGRESS:
    {
      xine_progress_data_t *prg = event->data;

      gdk_threads_enter();
      if (prg->percent>99)
	show_metadata ();
      else {
	struct timeval  tv;
	int             age;

	/* filter out old events */
	gettimeofday (&tv, NULL);
	age = abs (tv.tv_sec - event->tv.tv_sec);
	if (age>0) {
	  gdk_threads_leave();
	  break;
	}

	infobar_line1 (bar, "%s %d%%", prg->description, prg->percent);
      }
      gdk_flush();
      gdk_threads_leave();
    }
    break;
  case XINE_EVENT_UI_SET_TITLE:
    {
      gdk_threads_enter();
      show_metadata ();
      gdk_flush();
      gdk_threads_leave();
    }
    break;
  case XINE_EVENT_MRL_REFERENCE:
    {
      xine_mrl_reference_data_t *ref = (xine_mrl_reference_data_t *) event->data;

      printf ("playlist: got an reference event (ref mrl: %s)\n",
	      ref->mrl);
      
      if (ref->alternative == 0) {

	play_item_t *item;

	item = play_item_new (ref->mrl, ref->mrl, 0);

	playlist_insert (item, playlist_get_list_pos()+1);
      }
    }
    break;
  case XINE_EVENT_UI_MESSAGE: 
    {
      xine_ui_message_data_t *data = (xine_ui_message_data_t *) event->data;

      gdk_threads_enter();

      /* TODO: use all parameters, provide customized messages, hints... */
      if (data->num_parameters) {

	char *param;

	param = data->messages + strlen (data->messages) +1;

	display_info ("xine engine message", "%s %s",
		      data->messages, param);
      } else {
	display_info ("xine engine message", data->messages);
      }
      gdk_threads_leave();
    }
    break;
    
  }

}

void playlist_init (void) {

  GtkWidget            *hbox, *button, *scrolled_window;
  GtkCellRenderer      *cell;
  GtkTreeViewColumn    *column;

  is_visible     = 0;
  cur_mrl        = NULL;
  cur_list_pos   = -1;

  srand (time (NULL));

  /*
   * window
   */

  dlg = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dlg), "Edit Playlist...");
  gtk_window_set_default_size (GTK_WINDOW (dlg), 500, 400);
  g_signal_connect (GTK_OBJECT (dlg), "delete_event",
		    G_CALLBACK (close_cb), NULL );

  /*
   * init tree store
   */
 
  pl_store = gtk_list_store_new (3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER);

  /*
   * tree view widget to display playlist
   */

  tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL(pl_store));  
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);
  g_signal_connect (G_OBJECT(tree_view), "button_press_event",
		    G_CALLBACK(button_press_lcb), NULL);

  cell = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes ("title",
						     cell,
						     "text", 0,
						     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));

  column = gtk_tree_view_column_new_with_attributes ("mrl",
						     cell,
						     "text", 1,
						     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));
  
  gtk_tree_view_set_reorderable (GTK_TREE_VIEW (tree_view), TRUE);

  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (scrolled_window), tree_view);

  gtk_box_pack_start (GTK_BOX(GTK_DIALOG (dlg)->vbox), scrolled_window,
		      TRUE, TRUE, 2);

  /*
   * checkboxes (repeat, random)
   */

  hbox = gtk_hbox_new (0,2);
  
  repeat_button = gtk_check_button_new_with_label ("repeat");
  gtk_box_pack_start (GTK_BOX (hbox), repeat_button, TRUE, TRUE, 2);

  random_button = gtk_check_button_new_with_label ("random");
  gtk_box_pack_start (GTK_BOX (hbox), random_button, TRUE, TRUE, 2);

  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), hbox,
		      FALSE, FALSE, 2);
  

  /*
   * button bar
   */

  hbox = gtk_hbox_new (1, 2);
  
  button = gtk_button_new_from_stock (GTK_STOCK_ADD);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(add_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 2);
  button = gtk_button_new_from_stock (GTK_STOCK_FIND_AND_REPLACE);
  gtk_button_set_label (GTK_BUTTON (button), "Edit");
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(edit_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 2);
  button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(del_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, TRUE, 2);
  button = gtk_button_new_from_stock (GTK_STOCK_CLEAR);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(clear_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, TRUE, 2);
  button = gtk_button_new_from_stock (GTK_STOCK_OPEN);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(open_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, TRUE, 2);
  button = gtk_button_new_from_stock (GTK_STOCK_SAVE_AS);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(save_as_cb), 
		    tree_view);
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, TRUE, 2);

  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), hbox,
		      FALSE, FALSE, 2);

  button = gtk_dialog_add_button (GTK_DIALOG (dlg), GTK_STOCK_CLOSE, 1);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(close_cb), 
		    NULL);

  playlist_load (NULL);

  sprintf (logo_mrl, "%s/logo.mpv", GXINE_LOGODIR);

  logo_mode = 1;

  /*
   * event handling
   */
  {
    xine_event_queue_t *queue;

    queue = xine_event_new_queue (GTK_XINE(gtx)->stream);

    xine_event_create_listener_thread (queue, xine_event_cb, NULL);

  }
}
