documented: #t
module:  c-lexer
author:  Robert Stockton (rgs@cs.cmu.edu)
synopsis: Encapsulates the lexical conventions of the C language.  Along with
          c-lexer-cpp.dylan, this file also incorporates most of the
          functionality of CPP.
copyright: see below
rcs-header: $Header: /var/lib/cvs/src/tools/melange/c-lexer.dylan,v 1.14 2003/04/12 23:36:41 andreas Exp $

//======================================================================
//
// Copyright (c) 1994, 1995, 1996, 1997  Carnegie Mellon University
// Copyright (c) 1998 - 2003  Gwydion Dylan Maintainers
// All rights reserved.
// 
// Use and copying of this software and preparation of derivative
// works based on this software are permitted, including commercial
// use, provided that the following conditions are observed:
// 
// 1. This copyright notice must be retained in full on any copies
//    and on appropriate parts of any derivative works.
// 2. Documentation (paper or online) accompanying any system that
//    incorporates this software, or any part of it, must acknowledge
//    the contribution of the Gwydion Project at Carnegie Mellon
//    University, and the Gwydion Dylan Maintainers.
// 
// This software is made available "as is".  Neither the authors nor
// Carnegie Mellon University make any warranty about the software,
// its performance, or its conformity to any specification.
// 
// Bug reports should be sent to <gd-bugs@gwydiondylan.org>; questions,
// comments and suggestions are welcome at <gd-hackers@gwydiondylan.org>.
// Also, see http://www.gwydiondylan.org/ for updates and documentation. 
//
//======================================================================

//======================================================================
// Module c-lexer performs lexical analysis and macro preprocessing for C
// header files.  The implementation is divided into two files --
// c-lexer.dylan handles basic lexing, while c-lexer-cpp.dylan contains most
// of the support for macro preprocessing.
//
// The module exports two major classes -- <tokenizer> and <token> -- along
// with assorted operations, subclasses, and constants.
//
//   <tokenizer>
//      Given some input source, produces a stream of tokens.  Tokenizers 
//      maintain local state.  At present this consists of the current
//      position in the input stream and the list of "typedefs" which
//      have been recognized thus far.  (The latter would be unnecessary
//      in a rational language, but C's lexical analysis is context dependent.
//
//   Tokenizers support the following operations:
//     make(<tokenizer>, #key source) -- source may either be a file name or a
//       stream. 
//     get-token(tokenizer) -- returns the next token
//     unget-token(tokenizer, token) -- returns a previously "gotten" token to
//       the beginning of the sequence of available tokens.
//     add-typedef(tokenizer, value) -- given either a token or string value,
//       will register that value in the tokenizer as the name of a valid
//       typedef.
//     cpp-decls(tokenizer) -- if at the "top-level" (i.e. not recursively
//       processing a #include) will contain the names of all "simple" macros
//       defined in the top level token stream.
//
//    <token>
//      Represents a single input token generated by a tokenizer.
//      Encapsulates: the position in the original source; the character
//      string which generated the token; and the typed "value" of the token.
//      There are numerous subclasses of <token> representing specific
//      reserved words or semantic types (e.g. <semicolon-token> or
//      <string-literal-token>). 
//
//    All tokens support the following operations:
//      value(token) -- returns the abstract "value" of the token.  The
//        type depends upon class of the specific token.
//      string-value(token) -- returns the sequence of characters from which
//        the token's value was derived
//      generator(token) -- returns the tokenizer which generated the
//        token. 
//      source-location(token) -- translate a token into a source location,
//        for use with error reporting.
//======================================================================

//------------------------------------------------------------------------
// Definitions specific to <tokenizer>s
//------------------------------------------------------------------------


// tokenizer -- exported class.
//
// The public view of tokenizers is described above.  The additional fields
// "cpp-stack" and "include-tokenizer" are used for handling the functionality
// of CPP.  The "parent" keyword for make is used to create tokenizers for
// "#included"ed files.
//
define /* exported */ primary class <tokenizer> (<object>)
  keyword name:, init-value: #f, type: false-or(<string>);
  keyword contents:, init-value: #f, type: false-or(<string>);
  keyword parent:, init-value: #f, type: false-or(<tokenizer>);
  keyword typedefs-from:, init-value: #f,
	type: false-or(<tokenizer>);
  slot file-name :: <string> = "<unknown-file>";
  slot contents :: <long-byte-string> = make(<long-byte-string>);
  slot position :: <integer> = 0;
  slot unget-stack :: <deque>, init-function: curry(make, <deque>);
  /* exported */ slot cpp-table :: <table>;
  slot cpp-stack :: <list> = #();
  /* exported */ slot cpp-decls :: type-union(<deque>, <false>) = #f;
  slot include-tokenizer :: false-or(<tokenizer>) = #f;
  slot typedefs :: <table>;
end class <tokenizer>;

//======================================================================
// Class definitions for and operations upon <token>s
//======================================================================
// Tokens types follow this hierarchy:
//   <token>
//     <simple-token> -- value is the token itself
//       <reserved-word-token>
//         ... -- lots of different tokens, distinguished via the
//                "reserved-words" table.
//       <punctuation-token>
//         ... -- lots of different tokens, distinguished via the
//                "reserved-words" table.
//       <eof-token>
//       <error-token>
//       <begin-include-token>
//     <end-include-token> -- value is a sequence of macro names
//     <name-token> -- value is the token string
//       <identifier-token>
//       <type-name-token> -- distinguished via the "typedefs" table
//     <literal-token>
//        <integer-token> -- value is an integer
//        <string-literal-token> -- value is the token string, minus
//                                  bracketing '"'s and character escapes
//        <character-token> -- value is a single character
//        <cpp-token> -- value is the token string minus the initial "#".
//                       This is only used internally.
//======================================================================

// <token> -- exported class.
//
define /* exported */ abstract primary class <token> (<object>)
  constant /* exported */ slot token-id :: <integer> = -1;
  slot string-value :: <byte-string>, required-init-keyword: #"string";
  /* exported */ slot generator, required-init-keyword: #"generator";
  slot position, init-value: #f, init-keyword: #"position";
end;

//----------------------------------------------------------------------
// General classes of tokens.
//----------------------------------------------------------------------

define /* exported */ abstract class <simple-token> (<token>) end class;
define /* exported */ abstract class <reserved-word-token> (<simple-token>)
end class;
define /* exported */ abstract class <punctuation-token> (<simple-token>)
end class;
define /* exported */ abstract class <literal-token> (<token>) end class;

// <end-include-token> will be a subclass of this.
//
define /* exported */ class <ei-token> (<token>)
  slot value :: <deque>, required-init-keyword: #"value";
end class;
define /* exported */ abstract class <name-token> (<token>) end class;

define /* exported */ class <type-specifier-token> (<reserved-word-token>)
end class;

//----------------------------------------------------------------------
// Specific token types
//----------------------------------------------------------------------

#if (~mindy)
define macro token-definer
  { define token ?:name :: ?super:expression = ?value:expression }
    => { define class ?name (?super)
	   inherited slot token-id = ?value;
	 end class ?name;
       }
end macro;

// Magic tokens
define token <eof-token> :: <simple-token> = 0;
define token <error-token> :: <simple-token> = 1;
define token <begin-include-token> :: <simple-token> = 2;
define token <end-include-token> :: <ei-token> = 3;
define /* exported */ token <identifier-token> :: <name-token> = 4;
define token <type-name-token>  :: <name-token> = 5;
// literals
define /* exported */ token <integer-token> :: <literal-token> = 6;
define /* exported */ token <character-token> :: <literal-token> = 7;
define token <string-literal-token> :: <literal-token> = 8;
define token <cpp-token> :: <literal-token> = 9;
// A whole bunch of reserved words
define /* exported */ token <struct-token> :: <reserved-word-token> = 10;
define token <typedef-token> :: <reserved-word-token> = 11;
define /* exported */ token <short-token> :: <type-specifier-token> = 12;
define /* exported */ token <long-token> :: <type-specifier-token> = 13;
define /* exported */ token <int-token> :: <type-specifier-token> = 14;
define /* exported */ token <char-token> :: <type-specifier-token> = 15;
define /* exported */ token <signed-token> :: <type-specifier-token> = 16;
define /* exported */ token <unsigned-token> :: <type-specifier-token> = 17;
define /* exported */ token <float-token> :: <type-specifier-token> = 18;
define /* exported */ token <double-token> :: <type-specifier-token> = 19;
// "const" and "volatile" will be preprocessed away by the cpp code.  They
// were being used in too many different odd places by various different
// compilers.  
//
// define class <const-token> (<reserved-word-token>) end class;
// define class <volatile-token> (<reserved-word-token>) end class;
define /* exported */ token <void-token> :: <type-specifier-token> = 20;
define token <inline-token> :: <reserved-word-token> = 21;
define token <extern-token> :: <reserved-word-token> = 22;
define token <static-token> :: <reserved-word-token> = 23;
define token <auto-token> :: <reserved-word-token> = 24;
define token <register-token> :: <reserved-word-token> = 25;
define token <dummy-token> :: <reserved-word-token> = 26;
define /* exported */ token <union-token> :: <reserved-word-token> = 27;
define /* exported */ token <enum-token> :: <reserved-word-token> = 28;
define token <constant-token> :: <reserved-word-token> = 29;
define token <mul-assign-token> :: <reserved-word-token> = 30;
define token <div-assign-token> :: <reserved-word-token> = 31;
define token <mod-assign-token> :: <reserved-word-token> = 32;
define token <add-assign-token> :: <reserved-word-token> = 33;
define token <sub-assign-token> :: <reserved-word-token> = 34;
define token <left-assign-token> :: <reserved-word-token> = 35;
define token <right-assign-token> :: <reserved-word-token> = 36;
define token <and-assign-token> :: <reserved-word-token> = 37;
define token <xor-assign-token> :: <reserved-word-token> = 38;
define token <or-assign-token> :: <reserved-word-token> = 39;
// A whole bunch of puctuation
define token <ellipsis-token> :: <punctuation-token> = 40;
define token <sizeof-token> :: <punctuation-token> = 41;
define token <dec-op-token> :: <punctuation-token> = 42;
define token <inc-op-token> :: <punctuation-token> = 43;
define token <ptr-op-token> :: <punctuation-token> = 44;
define token <semicolon-token> :: <punctuation-token> = 45;
define token <comma-token> :: <punctuation-token> = 46;
define token <dot-token> :: <punctuation-token> = 47;
define token <lparen-token> :: <punctuation-token> = 48;
define token <rparen-token> :: <punctuation-token> = 49;
define token <lbracket-token> :: <punctuation-token> = 50;
define token <rbracket-token> :: <punctuation-token> = 51;
define token <ampersand-token> :: <punctuation-token> = 52;
define token <star-token> :: <punctuation-token> = 53;
define token <carat-token> :: <punctuation-token> = 54;
define token <bar-token> :: <punctuation-token> = 55;
define token <percent-token> :: <punctuation-token> = 56;
define token <slash-token> :: <punctuation-token> = 57;
define token <plus-token> :: <punctuation-token> = 58;
define /* exported */ token <minus-token> :: <punctuation-token> = 59;
define /* exported */ token <tilde-token> :: <punctuation-token> = 60;
define /* exported */ token <bang-token> :: <punctuation-token> = 61;
define token <lt-token> :: <punctuation-token> = 62;
define token <gt-token> :: <punctuation-token> = 63;
define token <question-token> :: <punctuation-token> = 64;
define token <colon-token> :: <punctuation-token> = 65;
define token <eq-op-token> :: <punctuation-token> = 66;
define token <le-op-token> :: <punctuation-token> = 67;
define token <ge-op-token> :: <punctuation-token> = 68;
define token <ne-op-token> :: <punctuation-token> = 69;
define token <and-op-token> :: <punctuation-token> = 70;
define token <or-op-token> :: <punctuation-token> = 71;
define token <pound-pound-token> :: <punctuation-token> = 72;
define token <left-op-token> :: <punctuation-token> = 73;
define token <right-op-token> :: <punctuation-token> = 74;
define token <assign-token> :: <punctuation-token> = 75;
define token <lcurly-token> :: <punctuation-token> = 76;
define token <rcurly-token> :: <punctuation-token> = 77;
// "Magic" tokens which provide alternate entry points to the parser
define /* exported */ token <alien-name-token> :: <token> = 78;
define /* exported */ token <macro-parse-token> :: <token> = 79;
define /* exported */ token <cpp-parse-token> :: <token> = 80;
define /* exported */ token <machine-token> :: <token> = 81;
#else
// The mindy declarations have to be a lot clumsier since we don't have macros.

// Magic tokens
define class <eof-token> (<simple-token>) 
  inherited slot token-id = 0;
end class;
define class <error-token> (<simple-token>) 
  inherited slot token-id = 1;
end class;
define class <begin-include-token> (<simple-token>) 
  inherited slot token-id = 2;
end class;
define class <end-include-token> (<ei-token>) 
  inherited slot token-id = 3;
end class;
define class <identifier-token> (<name-token>) 
  inherited slot token-id = 4;
end class;
define class <type-name-token>  (<name-token>) 
  inherited slot token-id = 5;
end class;
// literals
define class <integer-token> (<literal-token>) 
  inherited slot token-id = 6;
end class;
define class <character-token> (<literal-token>) 
  inherited slot token-id = 7;
end class;
define class <string-literal-token> (<literal-token>) 
  inherited slot token-id = 8;
end class;
define class <cpp-token> (<literal-token>) 
  inherited slot token-id = 9;
end class;
// A whole bunch of reserved words
define class <struct-token> (<reserved-word-token>) 
  inherited slot token-id = 10;
end class;
define class <typedef-token> (<reserved-word-token>) 
  inherited slot token-id = 11;
end class;
define class <short-token> (<type-specifier-token>) 
  inherited slot token-id = 12;
end class;
define class <long-token> (<type-specifier-token>) 
  inherited slot token-id = 13;
end class;
define class <int-token> (<type-specifier-token>) 
  inherited slot token-id = 14;
end class;
define class <char-token> (<type-specifier-token>) 
  inherited slot token-id = 15;
end class;
define class <signed-token> (<type-specifier-token>) 
  inherited slot token-id = 16;
end class;
define class <unsigned-token> (<type-specifier-token>) 
  inherited slot token-id = 17;
end class;
define class <float-token> (<type-specifier-token>) 
  inherited slot token-id = 18;
end class;
define class <double-token> (<type-specifier-token>) 
  inherited slot token-id = 19;
end class;
// "const" and "volatile" will be preprocessed away by the cpp code.  They
// were being used in too many different odd places by various different
// compilers.  
//
// define class <const-token> (<reserved-word-token>) end class;
// define class <volatile-token> (<reserved-word-token>) end class;
define class <void-token> (<type-specifier-token>) 
  inherited slot token-id = 20;
end class;
define class <inline-token> (<reserved-word-token>) 
  inherited slot token-id = 21;
end class;
define class <extern-token> (<reserved-word-token>) 
  inherited slot token-id = 22;
end class;
define class <static-token> (<reserved-word-token>) 
  inherited slot token-id = 23;
end class;
define class <auto-token> (<reserved-word-token>) 
  inherited slot token-id = 24;
end class;
define class <register-token> (<reserved-word-token>) 
  inherited slot token-id = 25;
end class;
define class <dummy-token> (<reserved-word-token>) 
  inherited slot token-id = 26;
end class;
define class <union-token> (<reserved-word-token>) 
  inherited slot token-id = 27;
end class;
define class <enum-token> (<reserved-word-token>) 
  inherited slot token-id = 28;
end class;
define class <constant-token> (<reserved-word-token>) 
  inherited slot token-id = 29;
end class;
define class <mul-assign-token> (<reserved-word-token>) 
  inherited slot token-id = 30;
end class;
define class <div-assign-token> (<reserved-word-token>) 
  inherited slot token-id = 31;
end class;
define class <mod-assign-token> (<reserved-word-token>) 
  inherited slot token-id = 32;
end class;
define class <add-assign-token> (<reserved-word-token>) 
  inherited slot token-id = 33;
end class;
define class <sub-assign-token> (<reserved-word-token>) 
  inherited slot token-id = 34;
end class;
define class <left-assign-token> (<reserved-word-token>) 
  inherited slot token-id = 35;
end class;
define class <right-assign-token> (<reserved-word-token>) 
  inherited slot token-id = 36;
end class;
define class <and-assign-token> (<reserved-word-token>) 
  inherited slot token-id = 37;
end class;
define class <xor-assign-token> (<reserved-word-token>) 
  inherited slot token-id = 38;
end class;
define class <or-assign-token> (<reserved-word-token>) 
  inherited slot token-id = 39;
end class;
// A whole bunch of puctuation
define class <ellipsis-token> (<punctuation-token>) 
  inherited slot token-id = 40;
end class;
define class <sizeof-token> (<punctuation-token>) 
  inherited slot token-id = 41;
end class;
define class <dec-op-token> (<punctuation-token>) 
  inherited slot token-id = 42;
end class;
define class <inc-op-token> (<punctuation-token>) 
  inherited slot token-id = 43;
end class;
define class <ptr-op-token> (<punctuation-token>) 
  inherited slot token-id = 44;
end class;
define class <semicolon-token> (<punctuation-token>) 
  inherited slot token-id = 45;
end class;
define class <comma-token> (<punctuation-token>) 
  inherited slot token-id = 46;
end class;
define class <dot-token> (<punctuation-token>) 
  inherited slot token-id = 47;
end class;
define class <lparen-token> (<punctuation-token>) 
  inherited slot token-id = 48;
end class;
define class <rparen-token> (<punctuation-token>) 
  inherited slot token-id = 49;
end class;
define class <lbracket-token> (<punctuation-token>) 
  inherited slot token-id = 50;
end class;
define class <rbracket-token> (<punctuation-token>) 
  inherited slot token-id = 51;
end class;
define class <ampersand-token> (<punctuation-token>) 
  inherited slot token-id = 52;
end class;
define class <star-token> (<punctuation-token>) 
  inherited slot token-id = 53;
end class;
define class <carat-token> (<punctuation-token>) 
  inherited slot token-id = 54;
end class;
define class <bar-token> (<punctuation-token>) 
  inherited slot token-id = 55;
end class;
define class <percent-token> (<punctuation-token>) 
  inherited slot token-id = 56;
end class;
define class <slash-token> (<punctuation-token>) 
  inherited slot token-id = 57;
end class;
define class <plus-token> (<punctuation-token>) 
  inherited slot token-id = 58;
end class;
define class <minus-token> (<punctuation-token>) 
  inherited slot token-id = 59;
end class;
define class <tilde-token> (<punctuation-token>) 
  inherited slot token-id = 60;
end class;
define class <bang-token> (<punctuation-token>) 
  inherited slot token-id = 61;
end class;
define class <lt-token> (<punctuation-token>) 
  inherited slot token-id = 62;
end class;
define class <gt-token> (<punctuation-token>) 
  inherited slot token-id = 63;
end class;
define class <question-token> (<punctuation-token>) 
  inherited slot token-id = 64;
end class;
define class <colon-token> (<punctuation-token>) 
  inherited slot token-id = 65;
end class;
define class <eq-op-token> (<punctuation-token>) 
  inherited slot token-id = 66;
end class;
define class <le-op-token> (<punctuation-token>) 
  inherited slot token-id = 67;
end class;
define class <ge-op-token> (<punctuation-token>) 
  inherited slot token-id = 68;
end class;
define class <ne-op-token> (<punctuation-token>) 
  inherited slot token-id = 69;
end class;
define class <and-op-token> (<punctuation-token>) 
  inherited slot token-id = 70;
end class;
define class <or-op-token> (<punctuation-token>) 
  inherited slot token-id = 71;
end class;
define class <pound-pound-token> (<punctuation-token>) 
  inherited slot token-id = 72;
end class;
define class <left-op-token> (<punctuation-token>) 
  inherited slot token-id = 73;
end class;
define class <right-op-token> (<punctuation-token>) 
  inherited slot token-id = 74;
end class;
define class <assign-token> (<punctuation-token>) 
  inherited slot token-id = 75;
end class;
define class <lcurly-token> (<punctuation-token>) 
  inherited slot token-id = 76;
end class;
define class <rcurly-token> (<punctuation-token>) 
  inherited slot token-id = 77;
end class;
// "Magic" tokens which provide alternate entry points to the parser
define class <alien-name-token> (<token>) 
  inherited slot token-id = 78;
end class;
define class <macro-parse-token> (<token>) 
  inherited slot token-id = 79;
end class;
define class <cpp-parse-token> (<token>) 
  inherited slot token-id = 80;
end class;
// An extra token to handle Solaris's "#machine(foo)" construct
define class <machine-token> (<token>) 
  inherited slot token-id = 81;
end class;

#endif

//----------------------------------------------------------------------
// Support code
//----------------------------------------------------------------------

// string-value -- exported generic.
//
define /* exported */ sealed generic string-value
  (token :: <token>) => (result :: <string>);

// value -- exported generic.
//
define /* exported */ sealed generic value
  (token :: <token>) => (result :: <object>);

// Literal tokens (and those not otherwise modified) evaluate to themselves.
//
define method value (token :: <token>) => (result :: <token>);
  token;
end method value;

// Name tokens evaluate to the string-value.
//
define method value (token :: <name-token>) => (result :: <string>);
  token.string-value;
end method value;

// Integer tokens can be in one of three different radices.  Figure out which
// and then compute an integer value.
//
define method value (token :: <integer-token>) => (result :: <general-integer>);
  let string = token.string-value;
  // Strip trailing markers from string.
  while (member?(string.last, "uUlL"))
    string := copy-sequence(string, end: string.size - 1);
  end while;

  case
    string.first ~= '0' => string-to-integer(string);
    string.size == 1 => 0;
    string.second.digit? =>
      string-to-integer(copy-sequence(string, start: 1), base: 8);
    otherwise =>
      string-to-integer(copy-sequence(string, start: 2), base: 16);
  end case;
end method value;

// Both string and character literals allow you to use '\\' to get certain
// non-alphanumeric characters.  This routine translates the second character
// of such a sequence into the appropriate "escaped character".
//
define method escaped-character (char :: <character>) => (esc :: <character>);
  select (char)
    'a' => as(<character>, 7);
    'b' => as(<character>, 8);
    't' => as(<character>, 9);
    'n' => as(<character>, 10);
    'v' => as(<character>, 11);
    'f' => as(<character>, 12);
    'r' => as(<character>, 13);
    otherwise => char;
  end select;
end method escaped-character;

// Single-character character literals evaluate to characters.  We
// must handle two character "escape sequences" as well as simple
// literals.
// Multi-character character literals are a non-standard extension.
// They evaluate to integers.
//
define method value
    (token :: <character-token>)
 => (result :: type-union(<character>, <integer>));
  let string = token.string-value;

  let result :: <integer> = 0;
  let multi? :: <boolean> = #f;

  local method process-char (position :: <integer>) => ()
          let char = string[position];
          if (char == '\\')
            result := ash(result, 8)
              + as(<integer>, escaped-character(string[position + 1]));
            if(string[position + 2] ~== '\'')
              multi? := #t;
              process-char(position + 2);
            end if;
          else
            result := ash(result, 8) + as(<integer>, char);
            if(string[position + 1] ~== '\'')
              multi? := #t;
              process-char(position + 1);
            end if;
          end if;
        end method;
  process-char(1);
  if(multi?) result else as(<character>, result) end;
end method value;
  
// String literals evaluate to strings (without the bracketing quotation
// marks).  Handling is complicated by the possibility that there will be
// "character escape"s in the string.
//
define method value (token :: <string-literal-token>) => (result :: <string>);
  let string = token.string-value;
  let new = make(<stretchy-vector>);

  local method process-char (position :: <integer>) => ();
	  let char = string[position];
	  if (char == '\\')
	    add!(new, escaped-character(string[position + 1]));
	    process-char(position + 2);
	  elseif (char ~= '"')
	    add!(new, char);
	    process-char(position + 1);
          // else we're done, so fall through
	  end if;
	end method process-char;
  process-char(1);
  as(<string>, new);
end method value;

// "Cpp" tokens evaluate to the string minus the initial "#".
//
define method value (token :: <cpp-token>) => (result :: <string>);
  copy-sequence(token.string-value, start: 1);
end method value;

// When we have a specific token that triggered an error, this routine can
// used saved character positions to precisely identify the location.
//
define method source-location (token :: <token>)
 => (srcloc :: <source-location>)
  let source-string = token.generator.contents;
  let line-num = 1;
  let last-CR = -1;

  for (i from 0 below token.position | 0)
    if (source-string[i] == '\n')
      line-num := line-num + 1;
      last-CR := i;
    end if;
  end for;

  let char-num = (token.position | 0) - last-CR;
  make(<file-source-location>,
       file: token.generator.file-name,
       line: line-num,
       line-position: char-num);
end method;

//========================================================================
// Support type -- <long-byte-string>
//========================================================================

// $long-string-component-size -- private constant.
//
// The size of each substring in a <long-byte-string>.  Useful even if we use
// normal <byte-string>s, cause it gives us an idea of the max chunk we can
// read in on any architecture.
//
define constant $long-string-component-size = 16384;

#if (~mindy)
define constant <long-byte-string> = <byte-string>;
#else
// <long-byte-string> -- private class.
//
// This class represents arbitrary length strings of <byte-character>.  We
// introduce it because Mindy cannot support <byte-string>s above a certain
// length.  It is also an interesting test for built in assumptions concerning
// strings.
//
define sealed class <long-byte-string> (<string>, <vector>)
  sealed slot size :: <integer> = 0, init-keyword: #"size";
  slot components :: <simple-object-vector> /* of <byte-string> */ = #[];
end class <long-byte-string>;

define sealed inline method initialize
    (value :: <long-byte-string>, #key fill = ' ')
  if (value.size > 0)
    let (max-component, final-size)
      = floor/(value.size, $long-string-component-size);
    value.components := make(<simple-object-vector>, size: max-component + 1);
    for (i from 0 below max-component)
      value.components[i] := make(<byte-string>,
				  size: $long-string-component-size,
				  fill: fill);
    end for;
    value.components[max-component]
      := make(<byte-string>, size: final-size, fill: fill);
  end if;
end method initialize;

// type-for-copy -- method on imported generic.
//
// We copy <long-byte-string>s as <byte-string>s because we will want to pass
// pieces of the string out to restrictive functions such as "write" and
// "format".  Ideally, these should not be so narrow-minded.
//
define sealed inline method type-for-copy 
    (string :: <long-byte-string>) => (result :: <type>);
  <byte-string>;
end method type-for-copy;
  
// copy-sequence -- method on imported generic.
//
// Relying on the "standard" definition of copy-sequence makes melange run 20
// times slower.  It's worth spending a few lines on a 20-fold speedup.
//
define sealed method copy-sequence
    (vector :: <long-byte-string>, #key start :: <integer> = 0, end: last)
 => (result :: <byte-string>);
  let last :: <integer> = last | size(vector);
  let sz :: <integer> = last - start;
  let (start-component, start-index)
    = floor/(start, $long-string-component-size);
  if (start-index + sz < $long-string-component-size)
    let subseq :: <byte-string> = vector.components[start-component];
    copy-sequence(subseq, start: start-index, end: start-index + sz);
  else
    let result :: <byte-string> = make(<byte-string>, size: sz);
    for (from-index :: <integer> from start below last,
	 to-index :: <integer> from 0)
      element(result, to-index) := element(vector, from-index);
    end for;
    result;
  end if;
end method copy-sequence;

define sealed inline method element
    (string :: <long-byte-string>, key :: <integer>,
     #key default = $not-supplied)
 => (result :: <byte-character>);
  if (key >= 0 & key < string.size)
    let (component, index) = floor/(key, $long-string-component-size);
    let substr :: <byte-string> =  string.components[component];
    substr[index];
  elseif (default == $not-supplied)
    error("No such element in %=: %=", string, key);
  else
    default;
  end if;
end method element;

define sealed inline method element-setter
    (value :: <byte-character>, string :: <long-byte-string>, key :: <integer>)
 => (value :: <byte-character>);
   if (key >= 0 & key < string.size)
    let (component, index) = floor/(key, $long-string-component-size);
    let substr :: <byte-string> =  string.components[component];
    substr[index] := value;
   else
     error("No such element in %=: %=", string, key);
   end if;
end method element-setter;

// forward-iteration-protocol -- method on imported generic.
//
// This method is identical to the one in "array.dylan", except that it
// is more tightly specialized to a single sealed class.  If you need to 
// make a general change, you should probably grep for "outlined-iterator" 
// and change all matching locations.
//
define inline method forward-iteration-protocol
    (array :: <long-byte-string>)
    => (initial-state :: <integer>,
	limit :: <integer>,
	next-state :: <function>,
	finished-state? :: <function>,
	current-key :: <function>,
	current-element :: <function>,
	current-element-setter :: <function>,
	copy-state :: <function>);
  values(0,
	 array.size,
	 method (array :: <long-byte-string>, state :: <integer>)
	     => new-state :: <integer>;
	   state + 1;
	 end,
	 method (array :: <long-byte-string>, state :: <integer>,
		 limit :: <integer>)
	     => done? :: <boolean>;
	   // We use >= instead of == so that the constraint propagation
	   // stuff can tell that state is < limit if this returns #f.
	   state >= limit;
	 end,
	 method (array :: <long-byte-string>, state :: <integer>)
	     => key :: <integer>;
	   state;
	 end,
	 method (array :: <long-byte-string>, state :: <integer>)
	     => element :: <object>;
	   element(array, state);
	 end,
	 method (new-value :: <object>, array :: <long-byte-string>,
		 state :: <integer>)
	     => new-value :: <object>;
	   element(array, state) := new-value;
	 end,
	 method (array :: <long-byte-string>, state :: <integer>)
	     => state-copy :: <integer>;
	   state;
	 end);
end;
#endif

//========================================================================
// "Simple" operations on tokenizers
//========================================================================

// Tokenizers can be created in a number of ways.  It must be passed a
// "source", but this may be either a file name or a stream.  The "name:"
// keyword can override the file name for the purposes of error reporting.
// We also accept an optional set of preprocessor definitions (and
// un-definitions.) 
//
// If "parent:" is specified then we inherit context sensitivities (i.e.
// typedefs and #defines) from the parent tokenizer.  Note that changes made
// to these tables will be reflected in the parent tokenizer.
// "Typedefs-from:" is similar, except that we only inherit the typedefs
// table.
//
define method initialize (value :: <tokenizer>,
			  #key name :: false-or(<string>),
			       contents: stuff :: false-or(<string>),
			       parent, typedefs-from, 
			       defines :: false-or(<table>))
  // We just read the entire file into a string for the tokenizer to use.
  // This simplifies things since we can use regexp searches to find things,
  // even across line boundaries.
  if (name) value.file-name := name end if;

  if (stuff)
    value.contents := as(<long-byte-string>, stuff);
  else
    let source-stream :: <stream>
      = make(<file-stream>, locator: as(<byte-string>, name),
	     direction: #"input");
    let components = make(<stretchy-vector>);
    block ()
      while (#t)
	add!(components, read(source-stream, $long-string-component-size));
      end;
    exception (err :: <incomplete-read-error>)
      add!(components, err.incomplete-read-sequence);
    exception (err :: <end-of-stream-error>)
      #t;
    end block;
    value.contents := apply(concatenate-as, <long-byte-string>, components);
    close(source-stream);
  end if;

  if (parent)
    value.typedefs := (typedefs-from | parent).typedefs;
    value.cpp-table := parent.cpp-table;
    value.cpp-decls := make(<deque>);
  else
    value.cpp-table := make(<string-table>);
    value.typedefs := if (typedefs-from)
			typedefs-from.typedefs;
		      else
			make(<string-table>);
		      end if;
    value.cpp-decls := make(<deque>);
  end if;

  local method parse-define-rhs(cpp-value)
	  if (cpp-value.empty?)
	    // Work around bug/misfeature in streams
	    #();
	  else
	    let sub-tokenizer
	      = make(<tokenizer>, contents: cpp-value);
	    for (list = #() then pair(token, list),
		 token = get-token(sub-tokenizer, expand: #f)
		   then get-token(sub-tokenizer, expand: #f),
		 until: instance?(token, <eof-token>))
	      if (instance?(token, <error-token>))
		parse-error(value, "Error in cpp defines");
	      end if;
	    finally
	      list;
	    end for;
	  end if;
	end method parse-define-rhs;

  if (defines)
    // We must be able to initialize the cpp-table with user supplied
    // additions (or subtractions).  If the supplied value is a
    // string, we need to create a temporary tokenizer to convert it
    // to a sequence of tokens.  Like all such sequences of "cpp"
    // tokens, this one will be in reverse order.
    for (cpp-value keyed-by key in defines)
      value.cpp-table[key] := 
	select (cpp-value by instance?)
	  <integer> =>
	    list(make(<integer-token>,
		      string: cpp-value.integer-to-string,
		      generator: value));
	  <string> =>
	    parse-define-rhs(cpp-value);
	  <list> =>
	    pair(map(method (param)
		       let tokenizer = make(<tokenizer>, contents: param);
		       get-token(tokenizer, expand: #f);
		       // XXX - Should check for end of stream, but this
		       // data came from foo-portability.dylan.
		     end,
		     head(cpp-value)),
		 parse-define-rhs(head(tail(cpp-value))));
	end select;
    end for;
  end if;
end method initialize;

// unget-token -- exported function.
//
// Stores a previously analyzed token for later return
//
define /* exported */ method unget-token
    (state :: <tokenizer>, token :: <token>)
 => (result :: <false>);
  push(state.unget-stack, token);
  #f;
end method unget-token;

// add-typedef -- exported generic.
//
// Record the given name as a valid type specifier.
//
define /* exported */ generic add-typedef
    (tokenizer :: <tokenizer>, name :: <object>) => (result :: <false>);

define method add-typedef (tokenizer :: <tokenizer>, token :: <token>)
 => (result :: <false>);
  tokenizer.typedefs[token.value] := <type-name-token>;
  #f;
end method add-typedef;

define method add-typedef (tokenizer :: <tokenizer>, name :: <string>)
 => (result :: <false>);
  tokenizer.typedefs[name] := <type-name-token>;
  #f;
end method add-typedef;

//======================================================================
// A bunch of specialized routines which together support "get-token"
//======================================================================

// Internal error messages generated by the lexer.  We have to go through some
// messy stuff to get to the "current" file name.
// 
define method source-location (tokenizer :: <tokenizer>)
 => (srcloc :: <source-location>)
  for (gen = tokenizer then gen.include-tokenizer,
       while: gen.include-tokenizer)
  finally 
    let source-string = gen.contents;
    let line-num = 1;
    let last-CR = -1;
    
    for (i from 0 below gen.position | 0)
      if (source-string[i] == '\n')
	line-num := line-num + 1;
	last-CR := i;
      end if;
    end for;

    let char-num = (gen.position | 0) - last-CR;
    make(<file-source-location>,
	 file: gen.file-name,
	 line: line-num,
	 line-position: char-num);
  end for;
end method;

// Each pair of elements in this vector specifies a literal constant
// corresponding to a C reserved word and the token class it belongs to.
define constant reserved-words
  = vector("struct", <struct-token>,
	   "typedef", <typedef-token>,
	   "short", <short-token>,
	   "long", <long-token>,
	   "int", <int-token>,
	   "char", <char-token>,
	   "signed", <signed-token>,
	   "unsigned", <unsigned-token>,
	   "float", <float-token>,
	   "double", <double-token>,
//	   "const", <const-token>,
//	   "volatile", <volatile-token>,
	   "void", <void-token>,
	   "inline", <inline-token>,
	   "__inline__", <inline-token>,
	   "__inline", <inline-token>,
	   "extern", <extern-token>,
	   "static", <static-token>,
	   "auto", <auto-token>,
	   "register", <register-token>,
	   "union", <union-token>,
	   "enum", <enum-token>,
	   "...", <ellipsis-token>,
	   "sizeof", <sizeof-token>,
//         There's no C dialect where "constant" is a reserved word...
//	   "constant", <constant-token>,
	   "#machine", <machine-token>,
	   "--", <dec-op-token>,
	   "++", <inc-op-token>,
	   "->", <ptr-op-token>,
	   "*=", <mul-assign-token>,
	   "/=", <div-assign-token>,
	   "%=", <mod-assign-token>,
	   "+=", <add-assign-token>,
	   "-=", <sub-assign-token>,
	   "<<=", <left-assign-token>,
	   ">>=", <right-assign-token>,
	   "&=", <and-assign-token>,
	   "^=", <xor-assign-token>,
	   "|=", <or-assign-token>,
	   ">=", <ge-op-token>,
	   "<=", <le-op-token>,
	   "!=", <ne-op-token>,
	   "&&", <and-op-token>,
	   "||", <or-op-token>,
	   "##", <pound-pound-token>,
	   ";", <semicolon-token>,
	   ",", <comma-token>,
	   ".", <dot-token>,
	   "*", <star-token>,
	   "%", <percent-token>,
	   "+", <plus-token>,
	   "-", <minus-token>,
	   "~", <tilde-token>,
	   "!", <bang-token>,
	   "/", <slash-token>,
	   "<", <lt-token>,
	   ">", <gt-token>,
	   "^", <carat-token>,
	   "|", <bar-token>,
	   "&", <ampersand-token>,
	   "?", <question-token>,
	   ":", <colon-token>,
	   "=", <assign-token>,
	   "==", <eq-op-token>,
	   "<<", <left-op-token>,
	   ">>", <right-op-token>,
	   "{", <lcurly-token>,
	   "}", <rcurly-token>,
	   "[", <lbracket-token>,
	   "]", <rbracket-token>,
	   "(", <lparen-token>,
	   ")", <rparen-token>);

// This table maps reserved words (as "symbol" literals) to the corresponding
// token class.  It is initialized from the "reserved-words" vector defined
// above
//
define constant reserved-word-table =
  make(<string-table>, size: truncate/(reserved-words.size, 2));

// Do the actual initialization of reserved-word-table at load time.
//
for (index from 0 below reserved-words.size by 2)
  reserved-word-table[reserved-words[index]] := reserved-words[index + 1];
end for;

// Looks for special classes of tokens and acts appropriately.  This includes
// macro names, typedefs, reserved words, and punctuation.  These are
// identified by entries in cpp-table, tokenizer.typedefs, and reserved-word
// table.
//
define function lex-identifier
    (tokenizer :: <tokenizer>, position :: <integer>, string :: <byte-string>,
     #key expand = #t, cpp-line = #f)
 => (token :: <token>);
  case
    expand & check-cpp-expansion(string, tokenizer,
				 forbidden-expansions: if (instance?(expand, <boolean>))
							 #()
						       else
							 expand
						       end if)
	=> get-token(tokenizer);

    element(tokenizer.typedefs, string, default: #f) =>
      make(<type-name-token>, string: string, position: position,
	   generator: tokenizer);

    otherwise =>
      let default
	= if (string.first == '#') <cpp-token> else <identifier-token> end if;
      let cls = if(~cpp-line)
		  element(reserved-word-table, string, default: default);
		else
		  default;
		end if;
      make(cls, position: position, string: string, generator: tokenizer);
  end case;
end function lex-identifier;

// Attempts to match "words" (i.e. identifiers or reserved words).  Returns a
// token if the match is succesful and #f otherwise.
//
define function try-identifier
    (state :: <tokenizer>, position :: <integer>, #key expand = #t, cpp-line = #f)
 => (result :: type-union(<token>, <false>));
  let contents :: <long-byte-string> = state.contents;

  let pos = if (contents[position] == '#') position + 1 else position end if;
  if (alpha?(contents[pos]) | contents[pos] == '_')
    for (index from pos + 1 below contents.size,
	 until: ~alphanumeric?(contents[index]) & contents[index] ~= '_')
    finally
      state.position := index;
      let string-value = copy-sequence(contents,
				       start: position, end: index);
      lex-identifier(state, position, string-value, expand: expand, cpp-line: cpp-line);
    end for;
  end if;
end function try-identifier;

#if (~mindy)
define multistring-checker match-punctuation
  ("-=", "*=", "/=", "%=", "+=", "<=", ">=", "&=", "^=", "|=", "==", "!=",
   "++", "--", "->", "...", ">>", ">>=", "<<", "<<=", "||", "&&", "##", 
   ";", ",", "(", ")", ".", "&", "*", "+", "~", "!", "/", "%", "<", ">", "^",
   "|", "?", ":", "=", "{", "}", "-", "[", "]");
#else
define constant match-punctuation
  = make-multistring-checker("-=", "*=", "/=", "%=", "+=", "<=", ">=", "&=",
			     "^=", "|=", "==", "!=",
			     "++", "--", "->", "...", ">>", ">>=", "<<",
			     "<<=", "||", "&&", "##",
			     ";", ",", "(", ")", ".", "&", "*", "+", "~",
			     "!", "/", "%", "<", ">", "^", "|", "?", ":",
			     "=", "{", "}", "-", "[", "]");
#endif

// Attempts to match "punctuation".  Returns a token if the match is succesful
// and #f otherwise.
//
define method try-punctuation (state :: <tokenizer>, position :: <integer>)
 => result :: type-union(<token>, <false>);
  let contents :: <long-byte-string> = state.contents;

  if (punctuation?(contents[position]))
    let match = match-punctuation(contents, start: position);
    if (match)
      state.position := position + match.size;
      lex-identifier(state, position, match, expand: #f);
    end if;
  end if;
end method try-punctuation;

define constant match-comment-end = make-substring-positioner("*/");

// *handle-c++-comments* -- xported variable.
//
define /* xported */ variable *handle-c++-comments* :: <boolean> = #f;

#if (~mindy)
define multistring-checker comment-matcher("/*", "//", "\\\n", "\\\r\n");
#else
define constant comment-matcher
  = make-multistring-checker("/*", "//", "\\\n", "\\\r\n");
#endif

// Skip over whitespace characters (including newlines) and comments.
//
define method skip-whitespace
    (contents :: <long-byte-string>, position :: <integer>)
 => (position :: <integer>);
  let sz = contents.size;

  local method skip-comments (index :: <integer>)
	 => end-index :: type-union(<integer>, <false>);
	  for (i from index,
	       until: (i >= sz | ~whitespace?(contents[i])))
	  finally
	    select (comment-matcher(contents, start: i) by \=)
	      "/*" =>
		let end-index = match-comment-end(contents, start: i + 2);
		if (~end-index)
		  error("Incomplete comment in C header file.");
		end if;
		skip-comments(end-index + 2);
	      "//" =>
                if (*handle-c++-comments*)
  	          while (i < sz & contents[i] ~== '\n')
		    i := i + 1;
		  end while;
  	          // i points at the newline if it ain't eof, but we need
	          // to recurse anyway to handle multiple // comments in a
	          // row, and that'll handle whitespace.
	          skip-comments(i);
		else
		  i;
		end if;
	      otherwise => 
	        i;
	    end select;
	  end for;
	end method skip-comments;
  skip-comments(position);
end method skip-whitespace;

// Skip over whitespace characters (excluding newlines) and comments in
// "#preprocessor" lines.  Handles the "\\\n" special case.
//
define method skip-cpp-whitespace
    (contents :: <long-byte-string>, position :: <integer>)
 => (position :: <integer>);
  let sz = contents.size;

  local method skip-comments (index :: <integer>)
	 => end-index :: type-union(<integer>, <false>);
	  for (i from index,
	       until: (i >= sz 
			 | ~(whitespace?(contents[i]) & contents[i] ~== '\n')))
	  finally
	    select (comment-matcher(contents, start: i) by \=)
	      "/*" =>
		let end-index = match-comment-end(contents, start: i + 2);
		if (~end-index)
		  error("Incomplete comment in C header file.");
		end if;
		skip-comments(end-index + 2);
	      "//" =>
                if (*handle-c++-comments*)
		  while (i < sz & contents[i] ~== '\n')
		    i := i + 1;
		  end while;
	          // i points at the newline now.
		end if;
	        i;
	      "\\\n" =>
		skip-comments(i + 2);
	      "\\\r\n" =>
		skip-comments(i + 3);
	      otherwise =>
		i;
  	    end select;
	  end for;
	end method skip-comments;
  skip-comments(position);
end method skip-cpp-whitespace;

// This matcher is used to match various literals.  Marks will be generated as
// follows:
//   [0, 1] and [2, 3] -- start and end of the entire match
//   [4, 5] -- start and end of character literal contents
//   [6, 7] -- start and end of string literal contents
//   [10, 11] -- start and end of integer literal
//
define constant match-literal
  = make-regexp-positioner("^('(\\\\?.)+'|"
			     "\"(([^\\\\\"]|\\\\.)*)\"|"
			     "((([1-9][0-9]*)|(0[xX][0-9a-fA-F]+)|(0[0-7]*))[lLuU]*))",
			   byte-characters-only: #t, case-sensitive: #t);

// get-token -- exported function.
//
// Returns a <token> object and updates state to reflect the token's
// consumption. 
//
define /* exported */ method get-token
    (state :: <tokenizer>,
     #key cpp-line, position: init-position, expand = ~cpp-line)
 => (token :: <token>);
  block (return)
    let pos = init-position | state.position;

    // If we are recursively including another file, defer to the tokenizer
    // for that file.
    let sub-tokenizer = state.include-tokenizer;
    if (sub-tokenizer)
      let token = get-token(sub-tokenizer, expand: expand,
			    cpp-line: cpp-line, position: init-position);
      if (instance?(token, <eof-token>))
	let macros = sub-tokenizer.cpp-decls;
	let old-file = sub-tokenizer.file-name;
	state.include-tokenizer := #f;
	let ei-token = make(<end-include-token>, position: pos,
			    generator: state, string: old-file,
			    value: macros);
	parse-progress-report(ei-token, "<<< exiting header <<<");
        return(ei-token);
      else
	return(token);
      end if;
    end if;

    // If we have old tokens, just pop them from the stack and return them
    if (~state.unget-stack.empty?)
      let stack = state.unget-stack;
      let token = pop(stack);
      if (~stack.empty? & instance?(stack.first, <pound-pound-token>))
	// The pound-pound construct is nasty.  We must concatenate two tokens
	// and then get a new token from the resulting string.  If this isn't
	// a single token, we just ignore the rest -- "the results are
	// undefined".
	pop(stack);		// Get rid of the pound-pound-token
	let new-string = concatenate(token.string-value,
				     get-token(state).string-value);
	if (new-string = "/" "/" & *handle-c++-comments*)
	  // Cruft to handle VC++ 4.2, which attempts to make a comment out of
	  // a boolean declaration.  Don't ask -- you don't wan to know.
	  return(make(<char-token>, position: token.position,
		      generator: token.generator, string: "char"));
	else 
	  let sub-tokenizer
	    = make(<tokenizer>, contents: new-string);
	  return(get-token(sub-tokenizer));
	end if;
      elseif (instance?(token, <identifier-token>)
		& element(state.typedefs, token.value, default: #f))
	// This is our last chance to deal with recently declared typedefs, so
	// we check one more time.
	return(make(<type-name-token>, position: token.position,
		    generator: token.generator, string: token.string-value));
      else
	return(token);
      end if;
    end if;

    let contents = state.contents;
    local method string-value(start-index, end-index)
	    copy-sequence(contents, start: start-index, end: end-index);
	  end method string-value;

    // There are different whitespace conventions for normal input and for
    // preprocessor directives.
    let pos = if (cpp-line)
		     skip-cpp-whitespace(contents, pos);
		   else
		     skip-whitespace(contents, pos);
		   end if;
    if (pos = contents.size | (cpp-line & contents[pos] == '\n'))
      state.position := pos;
      return(make(<eof-token>, position: pos, generator: state,
		  string: ""));
    end if;

    // Deal with preprocessor lines.  Since these may change the state, we
    // will simply re-call "get-token" after invoking the appropriate
    // processing.  We don't look for preprocessor lines in the middle of
    // other preprocessor lines.
    if (~cpp-line & try-cpp(state, pos))
      return(get-token(state));
    end if;

    // Do the appropriate matching, and return an <error-token> if we don't
    // find a match.
    let token? =
      try-identifier(state, pos, expand: expand, cpp-line: cpp-line) | try-punctuation(state, pos);
    if (token?) return(token?) end if;

    let (start-index, end-index, dummy1, dummy2, char-start, char-end,
	 string-start, string-end, dummy3, dummy4, int-start, int-end)
      = match-literal(contents, start: pos);

    if (start-index)
      // At most one of the specialized start indices will be non-false.  Look
      // for that one and build the appropriate token.
      state.position := end-index;
      let token-type = case
			 char-start => <character-token>;
			 string-start => <string-literal-token>;
			 int-start => <integer-token>;
		       end case;
      return(make(token-type, position: pos,
		  string: string-value(pos, end-index), generator: state));
    end if;

    // None of our searches matched, so we haven't the foggiest what this is.
    parse-error(state, "Major botch in get-token.");
  end block;
end method get-token;

// Seals for file c-lexer.dylan

// <tokenizer> -- subclass of <object>
define sealed domain make(singleton(<tokenizer>));
define sealed domain initialize(<tokenizer>);
// <eof-token> -- subclass of <simple-token>
define sealed domain make(singleton(<eof-token>));
// <error-token> -- subclass of <simple-token>
define sealed domain make(singleton(<error-token>));
// <begin-include-token> -- subclass of <simple-token>
define sealed domain make(singleton(<begin-include-token>));
// <end-include-token> -- subclass of <token>
define sealed domain make(singleton(<end-include-token>));
// <identifier-token> -- subclass of <name-token>
define sealed domain make(singleton(<identifier-token>));
// <type-name-token> -- subclass of <name-token>
define sealed domain make(singleton(<type-name-token>));
// <integer-token> -- subclass of <literal-token>
define sealed domain make(singleton(<integer-token>));
// <character-token> -- subclass of <literal-token>
define sealed domain make(singleton(<character-token>));
// <string-literal-token> -- subclass of <literal-token>
define sealed domain make(singleton(<string-literal-token>));
// <cpp-token> -- subclass of <literal-token>
define sealed domain make(singleton(<cpp-token>));
// <struct-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<struct-token>));
// <typedef-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<typedef-token>));
// <type-specifier-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<type-specifier-token>));
// <short-token> -- subclass of <type-specifier-token>
define sealed domain make(singleton(<short-token>));
// <long-token> -- subclass of <type-specifier-token>
define sealed domain make(singleton(<long-token>));
// <int-token> -- subclass of <type-specifier-token>
define sealed domain make(singleton(<int-token>));
// <char-token> -- subclass of <type-specifier-token>
define sealed domain make(singleton(<char-token>));
// <signed-token> -- subclass of <type-specifier-token>
define sealed domain make(singleton(<signed-token>));
// <unsigned-token> -- subclass of <type-specifier-token>
define sealed domain make(singleton(<unsigned-token>));
// <float-token> -- subclass of <type-specifier-token>
define sealed domain make(singleton(<float-token>));
// <double-token> -- subclass of <type-specifier-token>
define sealed domain make(singleton(<double-token>));
// <const-token> -- subclass of <reserved-word-token>
// define sealed domain make(singleton(<const-token>));
// <volatile-token> -- subclass of <reserved-word-token>
// define sealed domain make(singleton(<volatile-token>));
// <void-token> -- subclass of <type-specifier-token>
define sealed domain make(singleton(<void-token>));
// <inline-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<inline-token>));
// <extern-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<extern-token>));
// <static-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<static-token>));
// <auto-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<auto-token>));
// <register-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<register-token>));
// <dummy-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<dummy-token>));
// <union-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<union-token>));
// <enum-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<enum-token>));
// <constant-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<constant-token>));
// <mul-assign-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<mul-assign-token>));
// <div-assign-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<div-assign-token>));
// <mod-assign-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<mod-assign-token>));
// <add-assign-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<add-assign-token>));
// <sub-assign-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<sub-assign-token>));
// <left-assign-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<left-assign-token>));
// <right-assign-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<right-assign-token>));
// <and-assign-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<and-assign-token>));
// <xor-assign-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<xor-assign-token>));
// <or-assign-token> -- subclass of <reserved-word-token>
define sealed domain make(singleton(<or-assign-token>));
// <ellipsis-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<ellipsis-token>));
// <sizeof-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<sizeof-token>));
// <dec-op-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<dec-op-token>));
// <inc-op-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<inc-op-token>));
// <ptr-op-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<ptr-op-token>));
// <semicolon-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<semicolon-token>));
// <comma-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<comma-token>));
// <dot-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<dot-token>));
// <lparen-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<lparen-token>));
// <rparen-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<rparen-token>));
// <lbracket-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<lbracket-token>));
// <rbracket-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<rbracket-token>));
// <ampersand-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<ampersand-token>));
// <star-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<star-token>));
// <carat-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<carat-token>));
// <bar-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<bar-token>));
// <percent-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<percent-token>));
// <slash-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<slash-token>));
// <plus-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<plus-token>));
// <minus-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<minus-token>));
// <tilde-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<tilde-token>));
// <bang-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<bang-token>));
// <lt-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<lt-token>));
// <gt-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<gt-token>));
// <question-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<question-token>));
// <colon-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<colon-token>));
// <eq-op-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<eq-op-token>));
// <le-op-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<le-op-token>));
// <ge-op-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<ge-op-token>));
// <ne-op-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<ne-op-token>));
// <and-op-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<and-op-token>));
// <or-op-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<or-op-token>));
// <pound-pound-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<pound-pound-token>));
// <left-op-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<left-op-token>));
// <right-op-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<right-op-token>));
// <assign-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<assign-token>));
// <lcurly-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<lcurly-token>));
// <rcurly-token> -- subclass of <punctuation-token>
define sealed domain make(singleton(<rcurly-token>));
// <alien-name-token> -- subclass of <token>
define sealed domain make(singleton(<alien-name-token>));
define sealed domain initialize(<alien-name-token>);
// <macro-parse-token> -- subclass of <token>
define sealed domain make(singleton(<macro-parse-token>));
define sealed domain initialize(<macro-parse-token>);
// <cpp-parse-token> -- subclass of <token>
define sealed domain make(singleton(<cpp-parse-token>));
define sealed domain initialize(<cpp-parse-token>);
