"""

Gwibber Client v2.0
SegPhault (Ryan Paul) - 06/06/2009

"""

import gtk, gtk.glade, gobject, dbus, dbus.service, hashlib
import re, urlparse, urllib
import functools, traceback, webbrowser, mx.DateTime, json, time
from . import gwui, config, configui, gintegration
from . import urlshorter, resources, table, actions
import os

from . import microblog

import gettext, locale
from gettext import lgettext as _

# Set this way as in setup.cfg we have prefix=/usr/local
LOCALEDIR = "/usr/share/locale"
DOMAIN = "gwibber"
MAX_MESSAGE_LENGTH = 140

locale.setlocale(locale.LC_ALL, "")

if "gwibber" not in urlparse.uses_query:
  urlparse.uses_query.append("gwibber")

for module in gtk.glade, gettext:
  module.bindtextdomain(DOMAIN, LOCALEDIR)
  module.textdomain(DOMAIN)

gtk.gdk.threads_init()

VERSION_NUMBER = "2.0.0"

SELECTED_STREAM = 0
SELECTED_PATH = 1
SELECTED_ACCOUNT = 2
SELECTED_OPNAME = 3
SELECTED_ICON = 4
SELECTED_TRANSIENT = 5
SELECTED_OPID = 6

CONFIGURABLE_UI_ELEMENTS = {
  "input": "_Editor",
  "statusbar": "_Statusbar",
  "tree": "Account _Tree",
  "tray_icon": "Tray _Icon",
  "spellcheck": "Spell_check",
  "toolbar": "Tool_bar",
}

CONFIGURABLE_ACCOUNT_ACTIONS = {
  # Translators: these are checkbox
  "receive": "_Receive Messages",
  "send": "_Send Messages",
  "search": "Search _Messages",
  }

DEFAULT_PREFERENCES = {
  "version": VERSION_NUMBER,
  "show_notifications": True,
  "refresh_interval": 2,
  "minimize_to_tray": False,
  "hide_taskbar_entry": False,
  "spellcheck_enabled": True,
  "theme": "default",
  "urlshorter": "is.gd",
  "retweet_style_via": False,
  "global_retweet": False,
  "override_font_options": False,
  "default_font": "Sans 14",
  "show_fullname_in_messages": True,
  "saved_position": [0, 0],
  "saved_size": [630, 765],
  "saved_input_position": 650,
  "saved_tree_position":150
}

for _i in list(CONFIGURABLE_UI_ELEMENTS.keys()):
  DEFAULT_PREFERENCES["show_%s" % _i] = True

try:
  import indicate
  import wnck
except:
  indicate = None
  wnck = None

class Client(dbus.service.Object):
  __dbus_object_path__ = "/com/GwibberClient"

  def __init__(self):
    # Setup a Client dbus interface
    self.bus = dbus.SessionBus()
    self.bus_name = dbus.service.BusName("com.GwibberClient", self.bus)
    dbus.service.Object.__init__(self, self.bus_name, self.__dbus_object_path__)

    # Methods the client exposes via dbus, return from the list method
    self.exposed_methods = [
                       'focus_client',
                       'show_replies',
                      ]

    self.w = GwibberClient()

  @dbus.service.method("com.GwibberClient", in_signature="", out_signature="")
  def focus_client(self):
    """
    This method focuses the client UI displaying the default view.
    Currently used when the client is activated via dbus activation.
    """
    self.w.present()
    try:
      self.w.move(*self.w.last_position)
    except:
      pass
    self.w.account_tree.get_selection().unselect_all()
    self.w.account_tree.get_selection().select_path((1,))

  @dbus.service.method("com.GwibberClient", in_signature="", out_signature="")
  def show_replies(self):
    """
    This method focuses the client UI and displays the replies view.
    Currently used when activated via the messaging indicator.
    """
    self.w.present()
    self.w.account_tree.get_selection().unselect_all()
    self.w.account_tree.get_selection().select_path((2,))

  @dbus.service.method("com.GwibberClient")
  def list(self):
    """
    This method returns a list of exposed dbus methods
    """
    return self.exposed_methods

class Message:
  def __init__(self, d):
    self.__dict__ = d

class GwibberClient(gtk.Window):
  def __init__(self):
    gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)
    self.connect("focus-in-event", self.on_focus)
    self.connect("delete-event", self.on_window_close)

    dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)
    bus = dbus.SessionBus()

    db_mb_obj = bus.get_object("com.Gwibber", "/com/gwibber/Microblog",
      follow_name_owner_changes = True)
    db_msg_obj = bus.get_object("com.Gwibber", "/com/gwibber/Messages",
      follow_name_owner_changes = True)
 
    self.microblog = dbus.Interface(db_mb_obj, "com.Gwibber")
    self.messages = dbus.Interface(db_msg_obj, "com.Gwibber")
    self.messages.connect_to_signal("operations_started", self.on_operation_started)
    self.messages.connect_to_signal("loading_complete", self.on_loading_complete)
    self.messages.connect_to_signal("finish_loading", self.on_finish_loading)
    self.messages.connect_to_signal("error", self.on_error)

    self.protocols = self.microblog.protocols()
    self.features = self.microblog.features()

    config.GCONF.add_dir(config.GCONF_PREFERENCES_DIR, config.gconf.CLIENT_PRELOAD_NONE)
    
    self.accounts = configui.AccountManager()
    self.accounts.connect("account-updated", self.on_account_update)
    self.accounts.connect("account-deleted", self.on_account_delete)
    self.preferences = config.Preferences()
    self.selected = [None, "home", None, None, None]
    
    self.last_focus_time = None
    self.last_update = None
    self.load_callbacks = {}
    self.message_target = None
    self.extended_view = False

    for key, value in list(DEFAULT_PREFERENCES.items()):
      if self.preferences[key] == None: 
        self.preferences[key] = value

    # Handle cases where the theme might not exist
    if self.preferences["theme"] not in resources.get_themes():
      self.preferences["theme"] = DEFAULT_PREFERENCES["theme"]

    self.setup_ui()

    self.preferences.notify("theme", self.update_view)
    config.GCONF.add_dir("/desktop/gnome/interface", config.gconf.CLIENT_PRELOAD_NONE)
    config.GCONF.notify_add("/desktop/gnome/interface/gtk_theme", self.update_view)
    config.GCONF.notify_add("/desktop/gnome/interface/gtk_color_scheme", self.update_view)

    for i in CONFIGURABLE_UI_ELEMENTS:
      self.preferences.notify("show_%s" % i,
        self.apply_ui_element_settings)

    self.apply_ui_element_settings()

    # Set the starting path
    # TODO: load the user's path from the previous session
    self.account_tree.get_selection().select_path((0,))

    # Call refresh when we start
    self.refresh()
    
    # Delay resizing input area or else it doesn't work
    gobject.idle_add(lambda: self.content.set_position(self.preferences["saved_input_position"]))

  def setup_ui(self):
    self.set_title(_("Gwibber"))
    x,y = self.preferences["saved_position"]
    w,h = self.preferences["saved_size"]
    self.set_default_size(w.get_int(), h.get_int())
    self.move(x.get_int(), y.get_int())

    menu_ui = self.setup_menu()
    self.add_accel_group(menu_ui.get_accel_group())

    def on_tray_menu_popup(i, b, a):
      menu_ui.get_widget("/menu_tray").popup(None, None,
        gtk.status_icon_position_menu, b, a, self.tray_icon)

    gtk.icon_theme_add_builtin_icon("gwibber", 22,
      gtk.gdk.pixbuf_new_from_file_at_size(
        resources.get_ui_asset("gwibber.svg"), 24, 24))

    self.set_icon_name("gwibber")
    self.tray_icon = gtk.status_icon_new_from_icon_name("gwibber")
    self.tray_icon.connect("activate", self.on_toggle_window_visibility)
    self.tray_icon.connect("popup-menu", on_tray_menu_popup)

    self.account_store = gwui.AccountTreeStore(self.accounts, self.microblog)
    self.account_store.populate_tree()
    
    self.account_tree = gwui.AccountTreeView(self.account_store)
    self.account_tree.get_selection().connect("changed", self.change_view, self.account_tree)
    self.account_tree.expand_all()

    self.tree = gtk.ScrolledWindow()
    self.tree.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    self.tree.add(self.account_tree)
    self.tree.set_shadow_type(gtk.SHADOW_IN)

    self.account_combobox = gwui.AccountComboBoxView(self.account_store)
    self.account_combobox.connect("changed", self.change_view, self.account_combobox)
    
    self.content_view = gwui.MessageStreamView(self.preferences["theme"], self)
    self.content_view.link_handler = self.on_link_clicked

    warning_icon = gtk.image_new_from_stock(gtk.STOCK_DIALOG_WARNING, gtk.ICON_SIZE_MENU)
    self.status_icon = gtk.EventBox()
    self.status_icon.add(warning_icon)
    self.status_icon.connect("button-press-event", self.on_errors_show)
    
    self.statusbar = gtk.Statusbar()
    self.statusbar.pack_start(self.status_icon, False, False)

    self.input = gwui.Input(self)
    self.input.connect("changed", self.on_input_change)
    self.input.connect("submit", self.on_input_activate)

    self.target_bar = gwui.TargetBar(self.account_store)
    self.target_bar.connect("canceled", self.on_target_cancel)

    input_area = gtk.VBox()
    input_area.pack_start(self.target_bar, False)
    input_area.pack_start(self.input, True, True)

    scroll = gtk.ScrolledWindow()
    scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    scroll.add(self.content_view)
    scroll.set_shadow_type(gtk.SHADOW_IN)

    self.content = gtk.VPaned()
    self.content.pack1(scroll, resize=True)
    self.content.pack2(input_area, resize=False)

    self.split = gtk.HPaned()
    self.split.add1(self.tree)
    self.split.add2(self.content)
    self.split.set_border_width(5)
    self.split.set_position(self.preferences["saved_tree_position"])

    if self.messages.get_errors():
      self.account_store.add_errors()

    ## Toolbar
    sw = menu_ui.get_widget("/toolbar_main/test")
    sw.set_draw(False)
    sw.set_expand(True)

    self.search_entry = gwui.GwibberSearch(self.accounts, self.microblog)
    self.search_entry.connect("activate", self.on_search_enter)
    
    ti = gtk.ToolItem()
    ti.add(self.search_entry)
    self.toolbar = menu_ui.get_widget("/toolbar_main")
    self.toolbar.insert(ti,5)

    ## Load spinner
    self.throbber = gtk.Image()
    menuspin = gtk.ImageMenuItem("")
    menuspin.set_right_justified(True)
    menuspin.set_sensitive(False)
    menuspin.set_image(self.throbber)
    if hasattr(menuspin, "set_always_show_image"):
      menuspin.set_always_show_image(True)
    main_menu = menu_ui.get_widget("/menubar_main")
    main_menu.append(menuspin)
    
    ## Main layout
    layout = gtk.VBox()
    layout.pack_start(main_menu, False)
    layout.pack_start(self.toolbar, False)
    layout.pack_start(self.account_combobox, False)
    layout.pack_start(self.split, True, True)
    layout.pack_start(self.statusbar, False)

    self.add(layout)
    self.show_all()
    self.content_view.load_fonts(resources.get_font())
    self.account_combobox.set_active(0)
    self.account_combobox.hide()
    self.target_bar.hide()
    self.status_icon.hide()

    ## Handle Facebook Auth
    for a in self.accounts:
      if a["protocol"] == "facebook":
        try: 
          fbc = microblog.facebook.Client(a)
          fbuid = fbc.facebook.users.getLoggedInUser()
          fbperms = "publish_stream,read_stream,status_update,offline_access"
          permstat = fbc.facebook.fql.query("SELECT %s FROM permissions WHERE uid=%s" % (fbperms, fbuid))[0]
          fbsuccess = a["session_key"].split("-")[1] == str(fbuid) and sum(permstat.values()) == 4
        except: fbsuccess = False
        
        if not fbsuccess:
          d = gtk.MessageDialog(None, gtk.DIALOG_MODAL, gtk.MESSAGE_INFO,
            gtk.BUTTONS_OK, "Gwibber requires Facebook authorization and permission to access your Facebook stream.")
          if d.run(): d.destroy()
          self.accounts.facebook_authorize(a)


  def setup_menu(self):
    ui_string = """
    <ui>
      <menubar name="menubar_main">
        <menu action="menu_gwibber">
          <menuitem action="refresh" />
          <menuitem action="search" />
          <separator/>
          <menuitem action="close_window" />
          <menuitem action="close_stream" />
          <separator/>
          <menuitem action="clear_window" />
          <menuitem action="clear_stream" />
          <separator/>
          <menuitem action="preferences" />
          <separator/>
          <menuitem action="quit" />
        </menu>

        <menu action="menu_view">
          <menuitem action="view_input" />
          <menuitem action="view_statusbar" />
          <menuitem action="view_toolbar" />
          <menuitem action="view_tree" />
          <menuitem action="view_tray_icon" />
          <menuitem action="view_spellcheck" />
        </menu>

        <menu action="menu_accounts">
        </menu>

        <menu action="menu_help">
          <menuitem action="help_online" />
          <menuitem action="help_translate" />
          <menuitem action="help_report" />
          <separator/>
          <menuitem action="about" />
        </menu>
      </menubar>

      <toolbar name="toolbar_main">
        <toolitem action="refresh" />
        <separator />
        <toolitem action="preferences" />
        <toolitem action="quit" />
        <separator name="test" />
        <toolitem action="close_stream" />
      </toolbar>

      <popup name="menu_tray">
        <menuitem action="refresh" />
        <separator />
        <menuitem action="preferences" />
        <menuitem action="about" />
        <separator />
        <menuitem action="quit" />
      </popup>
    </ui>
    """

    self.actions = gtk.ActionGroup("Actions")
    self.actions.add_actions([
      ("menu_gwibber", None, "_Gwibber"),
      ("menu_view", None, "_View"),
      ("menu_accounts", None, "_Accounts"),
      ("menu_help", None, "_Help"),

      ("refresh", gtk.STOCK_REFRESH, "_Refresh", "<ctrl>R", None, self.on_refresh),
      ("search", gtk.STOCK_FIND, "_Search", "<ctrl>F", None, self.on_search),
      ("preferences", gtk.STOCK_PREFERENCES, "_Preferences", "<ctrl>P", None, self.on_preferences),
      ("about", gtk.STOCK_ABOUT, "_About", None, None, self.on_about),
      ("quit", gtk.STOCK_QUIT, "_Quit", "<ctrl>Q", None, self.on_quit),

      ("close_window", gtk.STOCK_CLOSE, "_Close Window", "<ctrl><shift>W", None, self.on_window_close),
      ("close_stream", gtk.STOCK_CLOSE, "_Close Stream", "<ctrl>W", None, self.on_close_stream),
      ("clear_window", gtk.STOCK_CLEAR, "C_lear Window", "<ctrl><shift>L", None, self.on_clear),
      ("clear_stream", gtk.STOCK_CLEAR, "Clear _Stream", "<ctrl>L", None, self.on_clear_tab),

      ("help_online", None, "Get Help Online...", None, None, None),
      ("help_translate", None, "Translate This Application...", None, None, None),
      ("help_report", None, "Report A Problem...", None, None, None),
    ])

    view_icons = {
      "statusbar": None,
      "toolbar": None,
      "tray_icon": None,
      "input": "gtk-edit",
      "tree": gtk.STOCK_INDEX,
      "spellcheck": gtk.STOCK_SPELL_CHECK,
    }

    for w, n in list(CONFIGURABLE_UI_ELEMENTS.items()):
      ta = gtk.ToggleAction("view_%s" % w, _(n), None, view_icons[w])
      self.preferences.bind(ta, "show_%s" % w)
      self.actions.add_action(ta)

    ui = gtk.UIManager()
    ui.insert_action_group(self.actions, pos=0)
    ui.add_ui_from_string(ui_string)
    ui.get_widget("/menubar_main/menu_accounts").connect("select", self.on_accounts_menu)
    
    refresh_item =  ui.get_widget("/toolbar_main/refresh")
    preferences_item =  ui.get_widget("/toolbar_main/preferences")
    quit_item =  ui.get_widget("/toolbar_main/quit")
    refresh_item.set_tooltip_text(_("Refresh"))
    preferences_item.set_tooltip_text(_("Preferences"))
    quit_item.set_tooltip_text(_("Quit"))

    return ui

  def on_accounts_menu(self, amenu):
    amenu.emit_stop_by_name("select")
    menu = amenu.get_submenu()
    for c in menu: menu.remove(c)

    menuAccountsManage = gtk.MenuItem(_("_Manage"))
    menuAccountsManage.connect("activate", lambda *a: self.accounts.show_account_list())
    menu.append(menuAccountsManage)

    menuAccountsCreate = gtk.MenuItem(_("_Create"))
    menu.append(menuAccountsCreate)
    mac = gtk.Menu()

    for pid in self.protocols:
      mi = gtk.ImageMenuItem(self.protocols[pid]["name"])
      img = resources.get_ui_asset("icons/%s.png" % pid)
      if img: mi.set_image(gtk.image_new_from_file(img))
      mi.connect("activate", self.accounts.on_account_create, pid)
      mac.append(mi)

    menuAccountsCreate.set_submenu(mac)
    menu.append(gtk.SeparatorMenuItem())

    for acct in self.accounts:
      if acct["protocol"] in self.protocols:
        sm = gtk.Menu()

        for key in CONFIGURABLE_ACCOUNT_ACTIONS:
          if key in self.protocols[acct["protocol"]]["features"]:
            mi = gtk.CheckMenuItem(_(CONFIGURABLE_ACCOUNT_ACTIONS[key]))
            acct.bind(mi, "%s_enabled" % key)
            sm.append(mi)

        sm.append(gtk.SeparatorMenuItem())

        mi = gtk.ImageMenuItem(gtk.STOCK_PROPERTIES)
        mi.connect("activate", lambda w, a: self.accounts.show_properties_dialog(a), acct)
        sm.append(mi)

        if acct["username"]: aname = acct["username"]
        else: aname = "(None)"

        mi = gtk.ImageMenuItem(aname) #"%s (%s)" % (aname, self.protocols[acct["protocol"]]["name"]))
        img = resources.get_ui_asset("icons/%s.png" % acct["protocol"])
        if img: mi.set_image(gtk.image_new_from_file(img))
        if hasattr(mi, "set_always_show_image"): mi.set_always_show_image(True)
        mi.set_submenu(sm)
        menu.append(mi)

    menu.show_all()

  def on_message_action_menu(self, msg):
    theme = gtk.icon_theme_get_default()
    menu = gtk.Menu()
    
    for a in actions.MENU_ITEMS:
      if a.include(self, msg):
        mi = gtk.Action("gwibberMessage%s" % a.__name__, a.label, None, None).create_menu_item()
        mi.get_image().set_from_icon_name(a.icon, gtk.ICON_SIZE_MENU)
        mi.connect("activate", a.action, self, msg)
        menu.append(mi)

    menu.show_all()
    menu.popup(None, None, None, 3, 0)

  def on_account_delete(self, w, acctid):
    self.account_store.remove_account(acctid)
    self.update_view()

  def on_account_update(self, w, acctid):
    self.account_store.update_account(acctid)
    self.update_view()
  
  def change_view(self, source, widget):
    if hasattr(widget, "get_selection") and \
        not widget.get_selection().count_selected_rows():
      return

    self.selected = widget.get_selected_data()
    self.extended_view = False
    
    iter = widget.get_selected_iter()
    self.account_combobox.set_active_iter(iter)
    self.account_tree.get_selection().select_iter(iter)

    if isinstance(widget, gtk.ComboBox):
      opname = self.selected[SELECTED_OPNAME]
      acctid = self.selected[SELECTED_ACCOUNT]
      
      csa = self.actions.get_action("close_stream")
      csa.set_visible(self.features[opname]["transient"] == True if opname else False)

      if opname and self.features[opname]["dynamic"]:
        self.microblog.operation({
          "opname": opname,
          "accountid": acctid,
          "source": "gwibber",
          "id": "gwibber-%s-%s-tree" % (acctid, opname)})
      
      """
      path = self.path.split("/")
      if len(path) > 1 and path[2] not in self.protocols.support("send"):
        self.input.hide()
      else: self.input.show()
      """

    if self.preferences["show_tree"] or isinstance(widget, gtk.ComboBox):
      self.update_view()
  
  def on_focus(self, *a):
    self.messages.mark_all_read()
    if self.last_update:
      self.last_focus_time = self.last_update
    else:
      self.last_focus_time= mx.DateTime.gmt()
    if self.last_focus_time:
      self.preferences["last_focus_time"] = self.last_focus_time.ticks()

  def on_errors_show(self, *a):
    self.account_tree.get_selection().unselect_all()
    self.account_tree.get_selection().select_iter(self.account_store.error_item)

  def is_gwibber_active(self):
    if wnck:
      screen = wnck.screen_get_default()
      screen.force_update()
      w = screen.get_active_window()
      if w: return self.window.xid == w.get_xid()
      return False
    else: return True

  def on_error(self, operation, error):
    self.status_icon.show()
    self.account_store.add_errors()

  def on_operation_started(self):
    self.throbber.set_from_animation(
      gtk.gdk.PixbufAnimation(resources.get_ui_asset("progress.gif")))

  def on_loading_complete(self):
    self.throbber.clear()
    self.update_view()

  def on_start_loading(self, source, id, accountid, protocol, opname):
    self.throbber.set_from_animation(
      gtk.gdk.PixbufAnimation(resources.get_ui_asset("progress.gif")))

  def on_finish_loading(self, opdata):
    for cb, fn in self.load_callbacks.items():
      if source == "gwibber" and cb == id:
        self.populate_view(messages)

  def populate_view(self, messages, count=None, total=None):
    messages = [self.post_process_message(Message(m)) for m in messages]
    self.flag_duplicates(messages)
    self.content_view.load_messages(messages, count, total)

    if self.is_gwibber_active():
      self.messages.mark_all_read()

  def update_view(self, *a):
    self.content_view.load_theme(self.preferences["theme"])
    if hasattr(self, "content_view"):
      if self.preferences["show_tree"]:
        paths = [row[1] for row in self.account_tree.get_multiselected_data()]
      else:
        paths = [self.account_combobox.get_selected_data()[SELECTED_PATH]]

      if len(paths) == 1 and paths[0] == "home":
        data = self.messages.get_messages(["replies"], 5)["messages"]
        messages = [self.post_process_message(Message(m)) for m in data]
        self.content_view.display_page("home.mako", messages)
      elif len(paths) == 1 and paths[0] == "errors":
        self.status_icon.hide()
        data = self.messages.get_errors()
        for e in data: e["time_string"] = generate_time_string(
            mx.DateTime.DateTimeFromTicks(e["time"]))
        self.content_view.display_page("errors.mako", data[::-1])
      else:
        data = self.messages.get_messages(paths, 0 if self.extended_view else 100)
        self.populate_view(data["messages"], data["count"], data["total"])

  def on_target_cancel(self, w):
    self.reply_target = None

  def on_link_clicked(self, uri, view):
    if uri.startswith("gwibber:"):
      u = urlparse.urlparse(uri)
      cmd = u.path.split("/")[1]
      query = dict((x,y[0]) for x,y in urlparse.parse_qs(u.query).items())

      if hasattr(actions, cmd):
        act = getattr(actions, cmd)

        if "msg" in query:
          query["msg"] = self.get_message(query["msg"])

        act.action(None, self, **query)
        return True

  def search(self, query):
    ti = self.account_store.add_search(query)
    self.account_tree.select_iter(ti)

    for acct in self.accounts:
      if "search" in self.protocols[acct["protocol"]]["features"]:
        self.microblog.schedule_operation(5, {
          "id": "search-%s-%s" % (query, acct["id"]),
          "source": "gwibber",
          "opname": "search",
          "accountid": acct["id"],
          "args": [query],
          "immediate": True,
        })

  def on_search_enter(self, w):
    self.search(w.get_text())
    self.apply_ui_element_settings()
    w.set_text("")

  def on_search(self, a):
    self.toolbar.show()
    self.search_entry.grab_focus()

  def add_transient_stream(self, acct, opname, query, name=None): 
    for row in self.account_tree.get_model():
      if row[2] == acct:
        opid = "%s-%s-%s" % (acct, opname, urllib.quote_plus(query))
        feature = self.features[opname]
        proto = self.accounts[acct]["protocol"]
        path = feature["path"] % {
            "protocol": proto, "accountid": acct, "query": urllib.quote_plus(query), "stream": feature["stream"]}
        icon = gtk.gdk.pixbuf_new_from_file(resources.icon(feature["icon"], False))
        it = self.account_tree.get_model().append(row.iter, [name or query, path, acct, opname, icon, True, opid])
        self.account_tree.select_iter(it)
        self.microblog.schedule_operation(5, {
          "id": opid,
          "source": "gwibber",
          "accountid": acct,
          "opname": opname,
          "args": [query],
          "immediate": True,
        })

  def on_close_stream(self, a):
    self.microblog.unschedule_operation("gwibber", self.selected[SELECTED_OPID])
    iter = self.account_tree.get_selected_iter()
    self.account_tree.get_selection().select_path((1,))
    self.account_store.remove(iter)

  def on_input_change(self, w, text, cnt):
    self.statusbar.pop(1)
    if cnt > 0:
      self.statusbar.push(1,
        _("Characters remaining: %s" % (MAX_MESSAGE_LENGTH - cnt)))

  def send(self, acctid, args, threaded=False):
    self.microblog.operation({
      "source": "gwibber",
      "id": "send-%s-%s" % (acctid, time.time()),
      "accountid": acctid,
      "args": args,
      "opname": "send_thread" if threaded else "send",
    })

  def on_input_activate(self, w, text, cnt):
    if len(text) > MAX_MESSAGE_LENGTH:
      pass # Warn users that the message is too long

    if self.target_bar.get_property("visible"):
      args = [text]
      
      if self.reply_target:
        args.append("message://" + self.reply_target.gwibber_path)

      self.send(self.target_bar.get_account(), args, True)
      self.reply_target = None
      self.target_bar.hide()
    
    else:
      for acct in self.accounts:
        if self.can("send", acct["protocol"]):

          if self.preferences["show_tree"]:
            if (not self.account_tree.get_selected_accounts() and acct["send_enabled"]) or \
                (acct["id"] in self.account_tree.get_selected_accounts()):
              self.send(acct["id"], [text])
          else:
            if (not self.selected[SELECTED_ACCOUNT] and acct["send_enabled"]) or \
                (self.selected[SELECTED_ACCOUNT] == acct["id"]):
              self.send(acct["id"], [text])
          
    w.clear()

  def get_message(self, path):
    return Message(self.messages.get_messages([path], 1)["messages"][0])

  def can(self, feature, protocol):
    return feature in self.protocols[protocol]["features"]

  def refresh(self):
    self.register_operations()
    self.microblog.perform_scheduled_operations()

  def register_operations(self):
    for a in self.accounts:
      for o in ["receive", "responses", "private"]:
        if o in self.protocols[a["protocol"]]["features"]:
          self.microblog.schedule_operation(5, {
            "source": "gwibber",
            "id": "%s-%s" % (a["id"], o),
            "accountid": a["id"],
            "opname": o,
          })

  ######

  def on_oldsearch(self, *a):
    dialog = gtk.MessageDialog(None,
      gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT, gtk.MESSAGE_QUESTION,
      gtk.BUTTONS_OK_CANCEL, None)

    entry = gtk.Entry()
    entry.connect("activate", lambda *a: dialog.response(gtk.RESPONSE_OK))

    dialog.set_markup(_("Enter a search query:"))
    dialog.vbox.pack_end(entry, True, True, 0)
    dialog.show_all()
    ret = dialog.run()
    dialog.hide()

    if ret == gtk.RESPONSE_OK:
      query = entry.get_text()
      view = None
      if query.startswith("#"):
        view = self.add_msg_tab(functools.partial(self.client.tag, query),
          query.replace("#", ""), True, gtk.STOCK_INFO, True, query)
      elif microblog.support.LINK_PARSE.match(query):
        view = self.add_msg_tab(functools.partial(self.client.search_url, query),
          urlparse.urlparse(query)[1], True, gtk.STOCK_FIND, True, query)
      elif len(query) > 0:
        title = _("Search") + " '" + query[:12] + "...'"
        view = self.add_msg_tab(functools.partial(self.client.search, query),
          title, True, gtk.STOCK_FIND, True, query)

      if view:
        self.update([view.get_parent()])

  def on_account_change(self, client, junk, entry, *args):
    if "color" in entry.get_key():
      for tab in self.tabs.get_children():
        view = tab.get_child()
        view.load_messages()

  def on_window_close(self, w, *args):
    self.save_position()
    if self.preferences["minimize_to_tray"]:
      self.on_toggle_window_visibility(w)
      return True
    else:
      self.on_window_close_quit()

  def on_cancel_reply(self, w, *args):
    self.cancel_button.hide()
    self.message_target = None
    self._reply_acct = None
    self.input.set_text("")

  def on_toggle_window_visibility(self, w):
    if self.get_property("visible"):
      self.last_position = self.get_position()
      self.hide()
    else:
      self.present()
      self.move(*self.last_position)

  def on_indicator_activate(self, w):
    tab_num = w.get_property("gwibber_tab")
    if tab_num is not None:
      self.tabs.set_current_page(int(tab_num))
    visible = self.get_property("visible")
    self.present()
    if not visible:
      self.move(*self.last_position)

  def external_invoke(self):
    logging.info("Invoked by external")
    if not self.get_property("visible"):
      logging.debug("Not visible..")
      self.present()
      self.move(*self.last_position)

  def apply_ui_element_settings(self, *a):
    for i in CONFIGURABLE_UI_ELEMENTS:
      if hasattr(self, i):
        getattr(self, i).set_property(
          "visible", self.preferences["show_%s" % i])

    if hasattr(self, "account_combobox"):
      if self.preferences["show_tree"] and self.account_combobox.get_property("visible"):
        self.account_tree.get_selection().unselect_all()
        self.account_tree.get_selection().select_iter(self.account_combobox.get_selected_iter())
      self.account_combobox.set_property("visible", not self.preferences["show_tree"])

    self.set_property("skip-taskbar-hint",
      self.preferences["hide_taskbar_entry"])

    #if gintegration.SPELLCHECK_ENABLED:
    #  self.input.set_checked(self.preferences["spellcheck_enabled"])

  def on_refresh_interval_changed(self, *a):
    gobject.source_remove(self.timer)
    self.timer = gobject.timeout_add(
      60000 * int(self.preferences["refresh_interval"]), self.update)

  def reply_old(self, message):
    acct = message.account
    # store which account we replied to first so we know when not to allow further replies
    if not self._reply_acct:
        self._reply_acct = acct
    if acct.supports(microblog.can.REPLY) and acct==self._reply_acct:
      self.input.grab_focus()
      if hasattr(message, 'is_private') and message.is_private:
        self.input.set_text("d %s " % (message.sender_nick))
      else:
        # Allow replying to more than one person by clicking on the reply button.
        current_text = self.input.get_text()
        # If the current text ends with ": ", strip the ":", it's only taking up space
        text = current_text[:-2] + " " if current_text.endswith(": ") else current_text
        # do not add the nick if it's already in the list
        if not text.count("@%s" % message.sender_nick):
          self.input.set_text("%s@%s%s" % (text, message.sender_nick, self.preferences['reply_append_colon'] and ': ' or ' '))

      self.input.set_position(-1)
      self.message_target = message
      self.cancel_button.show()

  def on_theme_change(self, *args):
    for tab in self.tabs:
      view = tab.get_child()
      view.load_theme(self.preferences["theme"])
      if len(view.message_store) > 0:
        view.load_messages()

  def get_themes(self):
    for base in xdg.BaseDirectory.xdg_data_dirs:
      theme_root = os.path.join(base, "gwibber", "ui", "themes")
      if os.path.exists(theme_root):

        for p in os.listdir(theme_root):
          if not p.startswith('.'):
            theme_dir = os.path.join(theme_root, p)
            if os.path.isdir(theme_dir):
              yield theme_dir

  def save_position(self):
    config.GCONF.set_list("%s/%s" % (config.GCONF_PREFERENCES_DIR, "saved_position"),
       config.gconf.VALUE_INT, list(self.get_position()))
    config.GCONF.set_list("%s/%s" % (config.GCONF_PREFERENCES_DIR, "saved_size"),
       config.gconf.VALUE_INT, list(self.get_size()))
    self.preferences["saved_input_position"] = self.content.get_position()
    self.preferences["saved_tree_position"] = self.split.get_position()

  def on_window_close_quit(self):
    gtk.main_quit()

  def on_quit(self, *a):
    self.save_position()
    cmd = "pidof -x gwibber-daemon"
    try:
      for line in os.popen(cmd):
        data = line.split()
        os.kill(int(data[0]), 15)
    except:
      pass
    gtk.main_quit()

  def on_refresh(self, *a):
    self.refresh()

  def on_about(self, mi):
    glade = gtk.glade.XML(resources.get_ui_asset("preferences.glade"))
    dialog = glade.get_widget("about_dialog")
    dialog.set_version(str(VERSION_NUMBER))
    dialog.connect("response", lambda *a: dialog.hide())
    dialog.show_all()

  def on_clear(self, mi):
    self.last_clear = mx.DateTime.gmt()
    for tab in self.tabs.get_children():
      view = tab.get_child()
      view.execute_script("clearMessages()")

  def on_clear_tab(self, mi):
    self.last_clear = mx.DateTime.gmt()
    n = self.tabs.get_current_page()
    view = self.tabs.get_nth_page(n).get_child()
    view.execute_script("clearMessages()")

  def old_on_errors_show(self, *args):
    self.status_icon.hide()
    errorwin = gtk.Window()
    errorwin.set_title(_("Errors"))
    errorwin.set_border_width(10)
    errorwin.resize(600, 300)

    def on_row_activate(tree, path, col):
      w = gtk.Window()
      w.set_title(_("Debug Output"))
      w.resize(800, 800)

      text = gtk.TextView()
      text.get_buffer().set_text(tree.get_selected().error)

      scroll = gtk.ScrolledWindow()
      scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
      scroll.add_with_viewport(text)

      w.add(scroll)
      w.show_all()

    errors = table.View(self.errors.tree_style,
      self.errors.tree_store, self.errors.tree_filter)
    errors.connect("row-activated", on_row_activate)

    scroll = gtk.ScrolledWindow()
    scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    scroll.add_with_viewport(errors)

    buttons = gtk.HButtonBox()
    buttons.set_layout(gtk.BUTTONBOX_END)

    def on_click_button(w, stock):
      if stock == gtk.STOCK_CLOSE:
        errorwin.destroy()
      elif stock == gtk.STOCK_CLEAR:
        self.errors.tree_store.clear()

    for stock in [gtk.STOCK_CLEAR, gtk.STOCK_CLOSE]:
      b = gtk.Button(stock=stock)
      b.connect("clicked", on_click_button, stock)
      buttons.pack_start(b)

    vb = gtk.VBox(spacing=5)
    vb.pack_start(scroll)
    vb.pack_start(buttons, False, False)

    errorwin.add(vb)
    errorwin.show_all()

  def on_preferences(self, mi):
    # reset theme to default if no longer available
    if self.preferences['theme'] not in resources.get_themes():
      config.GCONF.set_string("%s/%s" % (config.GCONF_PREFERENCES_DIR, "theme"), 'default')

    glade = gtk.glade.XML(resources.get_ui_asset("preferences.glade"))
    dialog = glade.get_widget("pref_dialog")
    dialog.show_all()

    for widget in [
        "show_notifications",
        "refresh_interval",
        "minimize_to_tray",
        "hide_taskbar_entry",
        "shorten_urls",
        "reply_append_colon",
        "retweet_style_via",
        "global_retweet",
        "show_fullname_in_messages",
        "override_font_options",
        "default_font"]:
      self.preferences.bind(glade.get_widget("pref_%s" % widget), widget)

    def on_toggle_font_override(*a):
      glade.get_widget("pref_default_font").set_sensitive(self.preferences["override_font_options"])
    
    glade.get_widget("pref_override_font_options").connect("toggled", on_toggle_font_override)
    on_toggle_font_override()

    self.preferences.bind(glade.get_widget("show_tray_icon"), "show_tray_icon")

    theme_selector = gtk.combo_box_new_text()
    for theme_name in sorted(resources.get_themes()): theme_selector.append_text(theme_name)
    glade.get_widget("containerThemeSelector").pack_start(theme_selector, True, True)
    if self.preferences["theme"] not in resources.get_themes():
      self.preferences["theme"] = DEFAULT_PREFERENCES["theme"]
    self.preferences.bind(theme_selector, "theme")
    theme_selector.show_all()

    # add combo box with url shorter services
    urlshorter_selector = gtk.combo_box_new_text()
    for us in sorted(urlshorter.PROTOCOLS.keys()):
      urlshorter_name = urlshorter.PROTOCOLS[us].PROTOCOL_INFO["name"]
      urlshorter_selector.append_text(urlshorter_name)
    glade.get_widget("containerURLShorterSelector").pack_start(urlshorter_selector, True, True)
    if self.preferences["urlshorter"] not in [x[0].strip() for x in urlshorter_selector.get_model()]:
      self.preferences["urlshorter"] = DEFAULT_PREFERENCES["urlshorter"]
    self.preferences.bind(urlshorter_selector, "urlshorter")
    urlshorter_selector.show_all()

    glade.get_widget("button_close").connect("clicked", lambda *a: dialog.destroy())

  def handle_error(self, acct, err, msg = None):
    self.status_icon.show()
    self.errors += {
      "time": mx.DateTime.gmt(),
      "username": acct["username"],
      "protocol": acct["protocol"],
      "message": "%s\n<i><span foreground='red'>%s</span></i>" % (msg, microblog.support.xml_escape(err.split("\n")[-2])),
      "error": err,
    }

  def old_on_input_activate(self, e):
    text = self.input.get_text().strip()
    if text:
      # don't allow submission if text length is greater than allowed
      if len(text) > MAX_MESSAGE_LENGTH:
         return
      # check if reply and target accordingly
      if self.message_target:
        account = self.message_target.account
        if account:
          # temporarily send_enable the account to allow reply to be posted
          is_send_enabled = account["send_enabled"]
          account["send_enabled"] = True
          if account.supports(microblog.can.THREAD_REPLY) and hasattr(self.message_target, "id"):
            result = self.client.send_thread(text, self.message_target, [account["protocol"]])
          else:
            result = self.client.reply(text, [account["protocol"]])
          # restore send_enabled choice after replying
          account["send_enabled"] = is_send_enabled
      # else standard post
      else:
        result = self.client.send(text, list(microblog.PROTOCOLS.keys()))

      # Strip empties out of the result
      result = [x for x in result if x]

      # if we get returned message info for the posts we should be able
      # to display them to the user immediately
      if result:
        for msg in result:
          if hasattr(msg, 'text'):
            self.post_process_message(msg)
            msg.is_new = msg.is_unread = False
        self.flag_duplicates(result)
        self.messages_view.message_store = result + self.messages_view.message_store
        self.messages_view.load_messages()

      self.on_cancel_reply(None)

  def post_process_message(self, message):
    def remove_url(s):
      return ' '.join([x for x in s.strip('.').split()
        if not x.startswith('http://') and not x.startswith("https://") ])

    if message.text.strip() == "": message.gId = None
    else: message.gId = hashlib.sha1(remove_url(message.text)[:140]).hexdigest()

    message.aId = message.account
    message.dupes = []

    if config.Account(message.account)[message.bgcolor]:
      message.color = gwui.Color.from_gtk_color(config.Account(message.account)[message.bgcolor])

    if self.last_focus_time:
      message.is_unread = (message.time > self.last_focus_time) or (hasattr(message, "is_unread") and message.is_unread)

    if self.last_update:
      message.is_new = message.time > self.last_update
    else: message.is_new = False

    message.time_string = generate_time_string(mx.DateTime.DateTimeFromTicks(message.time))

    if not hasattr(message, "html_string"):
      message.html_string = '<span class="text">%s</span>' % \
        LINK_PARSE.sub('<a href="\\1">\\1</a>', message.text)

    message.can_reply = True #message.account.supports(microblog.can.REPLY)
    return message

  def show_notification_bubbles(self, messages):
    new_messages = []
    for message in messages:
      if message.is_new and self.preferences["show_notifications"] and \
        message.first_seen and gintegration.can_notify and \
          message.username != message.sender_nick:
          new_messages.append(message)

    new_messages.reverse()
    gtk.gdk.threads_enter()
    if len(new_messages) > 0:
        for index, message in enumerate(new_messages):
            body = microblog.support.xml_escape(message.text)
            image = hasattr(message, "image_path") and message.image_path or ''
            expire_timeout = 5000 + (index*2000) # default to 5 second timeout and increase by 2 second for each notification
            n = gintegration.notify(message.sender, body, image, ["reply", "Reply"], expire_timeout)
            self.notification_bubbles[n] = message
    gtk.gdk.threads_leave()

  def flag_duplicates(self, data):
    seen = {} 
    for n, message in enumerate(data):
      if hasattr(message, "gId") and message.gId:
        message.is_duplicate = message.gId in seen
        message.first_seen = False
        if message.is_duplicate:
          data[seen[message.gId]].dupes.append(message)

        if not message.is_duplicate:
          message.first_seen = True
          seen[message.gId] = n
      else:
        message.is_duplicate = False
        message.first_seen = True

  def manage_indicator_items(self, data, tab_num=None):
    if not self.is_gwibber_active():
      for msg in data:
        if hasattr(msg, "first_seen") and msg.first_seen and \
            hasattr(msg, "is_unread") and msg.is_unread and \
            hasattr(msg, "gId") and msg.gId not in self.indicator_items:
          indicator = indicate.IndicatorMessage()
          indicator.set_property("subtype", "im")
          indicator.set_property("sender", msg.sender_nick)
          indicator.set_property("body", msg.text)
          indicator.set_property_time("time", msg.time.gmticks())
          if hasattr(msg, "image_path"):
            pb = gtk.gdk.pixbuf_new_from_file(msg.image_path)
            indicator.set_property_icon("icon", pb)

          if tab_num is not None:
            indicator.set_property("gwibber_tab", str(tab_num))
          indicator.connect("user-display", self.on_indicator_activate)

          self.indicator_items[msg.gId] = indicator
          indicator.show()

  def update_view_contents(self, view):
    view.load_messages()

  def update(self, tabs = None):
    self.throbber.set_from_animation(
      gtk.gdk.PixbufAnimation(resources.get_ui_asset("progress.gif")))
    self.target_tabs = tabs

    def process():
      try:
        next_update = mx.DateTime.gmt()
        if not self.target_tabs:
          self.target_tabs = self.tabs.get_children()

        for tab in self.target_tabs:
          view = tab.get_child()
          if view:
            view.message_store = [m for m in
              view.data_retrieval_handler() if not self.last_clear or m.time > self.last_clear
              and m.time <= mx.DateTime.gmt()]
            self.flag_duplicates(view.message_store)
            gtk.gdk.threads_enter()
            gobject.idle_add(self.update_view_contents, view)
            
            if indicate and hasattr(view, "add_indicator") and view.add_indicator:
              self.manage_indicator_items(view.message_store, tab_num=self.tabs.page_num(tab))

            gtk.gdk.threads_leave()
            self.show_notification_bubbles(view.message_store)

        self.statusbar.pop(0)
        self.statusbar.push(0, _("Last update: %s") % time.strftime("%X"))
        self.last_update = next_update

      finally: gobject.idle_add(self.throbber.clear)

    t = threading.Thread(target=process)
    t.setDaemon(True)
    t.start()

    return True

SCHEMES = ('http', 'https', 'ftp', 'mailto', 'news', 'gopher',
                'nntp', 'telnet', 'wais', 'prospero', 'aim', 'webcal')
URL_FORMAT = (r'(?<!\w)((?:%s):' # protocol + :
    '/*(?!/)(?:' # get any starting /'s
    '[\w$\+\*@&=\-/]' # reserved | unreserved
    '|%%[a-fA-F0-9]{2}' # escape
    '|[\?\.:\(\),;!\'\~](?!(?:\s|$))' # punctuation
    '|(?:(?<=[^/:]{2})#)' # fragment id
    '){2,}' # at least two characters in the main url part
    ')') % ('|'.join(SCHEMES),)
LINK_PARSE = re.compile(URL_FORMAT)

def generate_time_string(t):
  if isinstance(t, str): return t

  d = mx.DateTime.gmt() - t

  # Aliasing the function doesn't work here with intltool...
  if d.days >= 365:
    years = round(d.days / 365)
    return gettext.ngettext("%(year)d year ago", "%(year)d years ago", years) % {"year": years}
  elif d.days >= 1 and d.days < 365:
    days = round(d.days)
    return gettext.ngettext("%(day)d day ago", "%(day)d days ago", days) % {"day": days}
  elif d.seconds >= 3600 and d.days < 1:
    hours = round(d.seconds / 60 / 60)
    return gettext.ngettext("%(hour)d hour ago", "%(hour)d hours ago", hours) % {"hour": hours}
  elif d.seconds < 3600 and d.seconds >= 60:
    minutes = round(d.seconds / 60)
    return gettext.ngettext("%(minute)d minute ago", "%(minute)d minutes ago", minutes) % {"minute": minutes}
  elif round(d.seconds) < 60:
    seconds = round(d.seconds)
    return gettext.ngettext("%(sec)d second ago", "%(sec)d seconds ago", seconds) % {"sec": seconds}
  else: return _("BUG: %s") % str(d)

#if __name__ == "__main__":
#  Client()
#  w = GwibberClient()
#  gtk.main()
