/*  GWhere
 *  Copyright (C) 2000  Sbastien LECACHEUR
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*!
 * @header	gwcatalogfileinfo.h
 *		GWhere<br>
 *		Copyright (C) 2000  Sbastien LECACHEUR<br><br>
 *		This Distributed under the terms of the <a href="http://www.gnu.org/copyleft/gpl.html">GNU General Public Licence</a><br><br>
 *		This component was developed for the GWhere project.<br>
 *		It's an object for catalog file representation. His structure gets many informations as file name, catalog name, file version, source path,...<br><br>
 *		<b>Warning :</b><br>
 *		All get functions which returns a <i>gchar*</i>, return string "" when the attribute is <i>NULL</i>.
 * @abstract	This is the catalog file data structure.
 * @author	Sbastien LECACHEUR <zero@gwhere.org>
 * @version	1.0 2001/08/21
 * @see		gwcategory.h
 * @see		gwsupport.h
 * @link		GWhere Web Site <http://www.gwhere.org>
 * @link		GTK+ Web Site <http://www.gtk.org>
 * @ref		gwcatalogfileinfo.c
 * @author	Author <zero@gwhere.org>
 */


#ifndef GW_DATA_CATALOG_FILE_INFO_H
#define GW_DATA_CATALOG_FILE_INFO_H


#include "gwcategory.h"


/*!
 * @struct	catalog_file_info
 * @abstract	Catalog file information data structure
 * @discussion	Catalog file information is an opaque data structure containing information about catalog file properties and informations.
 */
struct catalog_file_info;


/*!
 * @function	catalog_file_info_new
 * @abstract	Creates a new catalog file information.
 * @discussion	This function creates a new catalog file information. It's an allocation function.
 * @result	a pointer to the catalog file information structure.<br>
 *		Returns <i>NULL</i> when an erro occured.<br>
 */
struct catalog_file_info * catalog_file_info_new ( void);


/*!
 * @function	catalog_file_info_set_name
 * @abstract	Sets the catalog's name.
 * @discussion	This function sets the catalog's name. It's the name displayed in the tree (it's not the file name).<br>
 *		New name may be <i>NULL</i>, in this case the catalog hasn't got name.
 * @param	p	the catalog file info to set the name
 * @param	name	the new catalog name
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint catalog_file_info_set_name ( struct catalog_file_info *p, gchar *name);


/*!
 * @function	catalog_file_info_set_title
 * @abstract	Sets the catalog's file name.
 * @discussion	This function sets the catalog's file name. Why file name is call title? In the first version of GWhere file name and title were the same data.<br>
 *		<b>Warning</b> : New file name may be <i>NULL</i>, in this case the catalog hasn't got file name. In this case the catalog doesn't exist!!
 * @param	p		the catalog file info to set the file name
 * @param	title		the new catalog file name
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint catalog_file_info_set_title ( struct catalog_file_info *p, gchar *title);


/*!
 * @function	catalog_file_info_set_full_name
 * @abstract	Sets the catalog's full file name.
 * @discussion	This function sets the catalog's full file name. The full file name is the file name with path.<br>
 *		<b>Warning</b> : New full file name may be <i>NULL</i>, in this case the catalog hasn't got file name. In this case the catalog doesn't exist!!
 * @param	p		the catalog file info to set the full file name
 * @param	full_name	the new catalog full file name
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint catalog_file_info_set_full_name ( struct catalog_file_info *p, gchar *full_name);


/*!
 * @function	catalog_file_info_set_version
 * @abstract	Sets the catalog's file format version.
 * @discussion	This function sets the catalog's file format version.<br>
 * @param	p	the catalog file info to set the file format version
 * @param	version	the new catalog file format version
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint catalog_file_info_set_version ( struct catalog_file_info *p, gchar *version);


/*!
 * @function	catalog_file_info_set_program_maker
 * @abstract	Sets the catalog's program maker.
 * @discussion	This function sets the catalog's program maker. Program maker is the program that built this catalog<br>
 * @param	p		the catalog file info to set the program maker
 * @param	program_maker	the new catalog program maker
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint catalog_file_info_set_program_maker ( struct catalog_file_info *p, gchar *program_maker);


/*!
 * @function	catalog_file_info_set_categories
 * @abstract	Sets the catalog's categories list.
 * @discussion	This function sets the catalog's categories list.<br>
 * @param	p		the catalog file info to set the categories list
 * @param	categories	the new catalog categories list
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint catalog_file_info_set_categories ( struct catalog_file_info *p, GList *categories);


/*!
 * @function	catalog_file_info_append_category
 * @abstract	Appends a category to the catalog's categories list.
 * @discussion	This function appends a category to the catalog's categories list.<br>
 * @param	p		the catalog file info to set the categories list
 * @param	category	the new catalog category
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint catalog_file_info_append_category ( struct catalog_file_info *p, struct category *category);




/*!
 * @function	catalog_file_info_remove_category
 * @abstract	Removes a category to the catalog's categories list.
 * @discussion	This function removes a category to the catalog's categories list.<br>
 * @param	p		the catalog file info to remove the categories list
 * @param	category	the category to remove
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint catalog_file_info_remove_category ( struct catalog_file_info *p, struct category *category);


/*!
 * @function	catalog_file_info_set_description
 * @abstract	Sets the catalog's description.
 * @discussion	This function sets the catalog's description.<br>
 * @param	p		<i>struct catalog_file_info*</i> the catalog file info to set the description
 * @param	program_maker	<i>gchar*</i> the new catalog description
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint catalog_file_info_set_description ( struct catalog_file_info *p, gchar *description);


/*!
 * @function	catalog_file_info_set_ismodified
 * @abstract	Sets the catalog modification state.
 * @discussion	This function sets the catalog modification state. Sets true if the catalog is modified else sets false if catalog isn't modified.<br>
 * @param	p		the catalog file info to set the modification state
 * @param	b		the new catalog program maker
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint catalog_file_info_set_ismodified ( struct catalog_file_info *p, gboolean b);


/*!
 * @function	catalog_file_info_get_name
 * @abstract	Gets the catalog's name.
 * @discussion	This function gets the catalog's name. It's the name displayed in the tree (it's not the file name).<br>
 * @param	p		the catalog file info to get the name
 * @result	the catalog's name.<br>
 *		Returns "" if the catalog's name is <i>NULL</i>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * catalog_file_info_get_name ( struct catalog_file_info *p);


/*!
 * @function	catalog_file_info_get_title
 * @abstract	Gets the catalog's file name.
 * @discussion	This function gets the catalog's file name. Why file name is call title? In the first version of GWhere file name and title were the same data.<br>
 * @param	p		the catalog file info to get the file name
 * @result	the catalog's file name.<br>
 *		Returns "" if the catalog's file name is <i>NULL</i>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * catalog_file_info_get_title ( struct catalog_file_info *p);


/*!
 * @function	catalog_file_info_get_full_name
 * @abstract	Gets the catalog's full file name.
 * @discussion	This function gets the catalog's full file name. The full file name is the file name with path.<br>
 * @param	p		the catalog file info to get the full file name
 * @result	the catalog's full file name.<br>
 *		Returns "" if the catalog's full file name is <i>NULL</i>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * catalog_file_info_get_full_name ( struct catalog_file_info *p);


/*!
 * @function	catalog_file_info_get_version
 * @abstract	Gets the catalog's file format version.
 * @discussion	This function gets the catalog's file format version.<br>
 * @param	p		the catalog file info to get the file format version
 * @result	the catalog's file format version.<br>
 *		Returns "" if the catalog's file format is <i>NULL</i>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * catalog_file_info_get_version ( struct catalog_file_info *p);


/*!
 * @function	catalog_file_info_get_program_maker
 * @abstract	Gets the catalog's program maker.
 * @discussion	This function gets the catalog's program maker. Program maker is the program that built this catalog<br>
 * @param	p		the catalog file info to get the program maker
 * @result	the catalog's program maker.<br>
 *		Returns "" if the catalog's program maker is <i>NULL</i>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * catalog_file_info_get_program_maker ( struct catalog_file_info *p);


/*!
 * @function	catalog_file_info_get_categories
 * @abstract	Gets the catalog's categories list.
 * @discussion	This function gets the catalog's categories list.<br>
 * @param	p		the catalog file info to get the categories list
 * @result	the catalog's categories list.<br>
 *		Returns <i>NULL</i> if catalog hasn't got categories.<br>
*		Returns <i>NULL</i> when an error occured.<br>
  */
GList * catalog_file_info_get_categories ( struct catalog_file_info *p);


/*!
 * @function	catalog_file_info_get_category
 * @abstract	Gets a special category in the catalog
 * @discussion	This function gets the category with a special name from the catalog.
 * @param	p		the catalog file info to get the category
 *		category	the category name
 * @result	a pointer to the category.<br>
 *		Returns the default category if don't find the category.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
struct category * catalog_file_info_get_category ( struct catalog_file_info *p, gchar *category);

/*!
 * @function	catalog_file_info_get_category_by_index
 * @abstract	Gets a special category in the catalog
 * @discussion	This function gets the category with a special index from the catalog.
 * @param	p		the catalog file info to get the category
 *		index		the category index
 * @result	a pointer to the category.<br>
 *		Returns the default category if don't find the category.<br>
*		Returns <i>NULL</i> when an error occured.<br>
  */
struct category * catalog_file_info_get_category_by_index ( struct catalog_file_info *p, guint index);


/*!
 * @function	catalog_file_info_get_description
 * @abstract	Gets the catalog's description.
 * @discussion	This function gets the catalog's description.<br>
 * @param	p		the catalog file info to get the description
 * @result	the catalog's description.<br>
 *		Returns "" if the catalog's description is <i>NULL</i>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * catalog_file_info_get_description ( struct catalog_file_info *p);


/*!
 * @function	catalog_file_info_get_size
 * @abstract	Gets the catalog file size.
 * @discussion	This function gets the size of catalog file.<br>
 *		<b>Warning</b> : this function doesn't control if full file name exist!!<br>
 * @param	p		the catalog file info to get file size
 * @result	the size of catalog file.<br>
 *		Returns <i>0</i> when an error occured.<br>
 */
gulong catalog_file_info_get_size ( struct catalog_file_info *p);


/*!
 * @function	catalog_file_info_get_ismodified
 * @abstract	Gets the catalog modification state.
 * @discussion	This function gets the catalog modification state. Gets <i>true</i> if the catalog is modified else gets <i>false</i> if catalog isn't modified.<br>
 * @param	p		the catalog file info to get the modification state
 * @result	is modified or no.<br>
 *		Returns <i>FALSE</i> when an error occured.<br>
 */
gboolean catalog_file_info_get_ismodified ( struct catalog_file_info *p);


/*!
 * @function	catalog_file_info_category_free
 * @abstract	Frees a category from catalog file info.
 * @discussion	This function frees a category from catalog file info. It is used by catalog_file_info_free().
 * @param	category		category to free from catalog file info
 * @param	data		unused pointer
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint catalog_file_info_category_free ( struct category *category, gpointer data);


/*!
 * @function	catalog_file_info_free
 * @abstract	Frees datas structure.
 * @discussion	This function frees the catalog file information's datas.
 * @param	p		the catalog file info to free
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gboolean catalog_file_info_free ( struct catalog_file_info *p);


#endif
