""" RUR-PLE: Roberge's Used Robot - a Python Learning Environment
    rur_world_display.py - Displays "world" with "robot" and allow
                      keyboard-based interactions
    Version 0.7
    Author: Andre Roberge    Copyright  2005
    andre.roberge@gmail.com
    
    Adapted for gvr by Stas Zytkiewicz. stas@linux.isbeter.nl
    
"""
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import rur_misc
from utils import WXVERSION
rur_misc.WX_VERSION = WXVERSION

import wx
from wxPython.wx import *

if __name__ == '__main__':
    app = wxPySimpleApp()

from rur_world_creation import Visible_world
from rur_robot_factory import Used_robot, New_improved_robot
#import rur_translatable
import rur_dialogs


class WorldGUI(wxScrolledWindow):
    def __init__(self, parent, id = -1, size = wxDefaultSize):
        self.world = Visible_world()
        wxScrolledWindow.__init__(self, parent, id, (0, 0),
                                size=(self.world.maxWidth+8, self.world.maxHeight+8),
                                style=wxSUNKEN_BORDER)
        self.SetBackgroundColour("WHITE")
        
        self.editor = False  #
        self.InitialiseVariables()
        self.MakePopupMenu()
        self.bindEvents()
        self.SetFocus()

    def InitialiseVariables(self):
        # Create the world image
        self.world.updateImage = False
        self.world.DoDrawing()
        # Initialize the buffer bitmap.  No real DC is needed at this point.
        self.buffer = wxEmptyBitmap(self.world.maxWidth, self.world.maxHeight)
        self.drawImage()

        # Set the size of the total window, of which only a small part
        # will be displayed; apparently SetVirtualSize needs
        # a single (tuple) argument, which explains the double (( )).
        #self.SetVirtualSize((self.world.maxWidth, self.world.maxHeight))

        # Set the scrolling rate; use same value in both horizontal and
        # vertical directions.
        scrollRate = self.world.tile_narrow + self.world.tile_wide
        self.SetScrollRate(scrollRate, scrollRate)

        self.SetScrollbars(1, 1, self.world.maxWidth,
                           self.world.maxHeight)
        # allow for testing "new and improved" robot
        self.newRobot = False

    def bindEvents(self):
        wx.EVT_PAINT(self, self.OnPaint)
        wx.EVT_LEFT_DOWN(self, self.OnLeftDown) # for "drawing"
        wx.EVT_CHAR(self, self.MyKeys) # to test Robot actions
        wx.EVT_IDLE(self, self.OnIdle)
        wx.EVT_RIGHT_UP(self, self.OnRightUp)

    def OnIdle(self, event=None):
        """
        Do any needed updating on Idle time ...
        """
        if self.world.updateImage:
            self.drawImage()
        self.world.updateImage = False

    def OnPaint(self, event=None):
        # This works flawlessly on my Windows XP computer with wxPython 2.4
        # Create a buffered paint DC.  It will create the real
        # wx.PaintDC and then blit the bitmap to it when dc is
        # deleted.  Since we don't need to draw anything else
        # here that's all there is to it.
        dc = wxBufferedPaintDC(self, self.buffer)

    def isActive(self, event):
        xView, yView = self.GetViewStart()
        xDelta, yDelta = self.GetScrollPixelsPerUnit()
        self.x, self.y = event.GetPositionTuple()
        self.x += xView*xDelta
        self.y += yView*yDelta
        # Test to see if in "active" zone i.e. inside world borders
        if (self.x > self.world.xOffset and
           self.y > self.world.yTopOffset and
           self.y < self.world.maxHeight - self.world.yOffset and
           self.x < self.world.maxWidth - self.world.right_scroller_space):
            return True
        else:
            return False

    def OnLeftDown(self, event):
        """Called when the left mouse button is pressed and build
           or remove walls."""
        if self.isActive(event):
            col, row = self.world.CalculatePosition(self.x, self.y)
            self.world.ChangeWall(col, row)
            if self.world.updateImage:
                self.drawImage()
                self.Refresh()
        else:
            pass   # added for clarity; we are on the outside

    def MakePopupMenu(self):
        """Make a menu for beepers that can be popped up later;
        this menu will offer the possibility of
        putting beepers from 0 (none) to 20, in steps of 1,
        at the current location (street/avenue intersection)."""
        menu = wxMenu()
        self.maxNumberOfBeepers = 20
        for x in range(0, self.maxNumberOfBeepers+1):
            menu.Append(x, str(x))
        wx.EVT_MENU_RANGE(self, 0, self.maxNumberOfBeepers, self.OnMenuSetBeepers)
        self.menu = menu

    def OnMenuSetBeepers(self, event):
        self._num = event.GetId()

    def OnRightUp(self, event):
        """called when the right mouse button is released,
        will call the popup menu, offering the possibility of
        putting beepers from 0 (none) to 20, in steps of 1,
        at the current location (street/avenue intersection)."""
        if self.isActive(event):
            col, row = self.world.CalculatePosition(self.x, self.y)
            if row%2:
                if col%2:
                    pt = event.GetPosition()
                    self.PopupMenu(self.menu, pt)
                    av = (col+1)/2
                    st = (row+1)/2
                    self.world.setBeepers((av, st), self._num)
                    if self.world.updateImage:
                        self.drawImage()
                        self.Refresh()
        else:
            pass   # added for clarity; we are on the outside

    def MyKeys(self, event):
        message = """
            F5: Help (i.e. this message)

Editing world:
            lower case e: toggle in/out of edit walls mode
            Left mouse button: add or remove walls (if in edit walls mode)
            Right mouse button: put beepers at intersection

Robot actions:
            Up arrow:     move robot forward
            Left arrow:   turn robot left
            lower case p: pick_beeper
            upper case P: put_beeper

            Errors will occur if you attempt to move through a wall,
            put a beeper when you carry none,
            or try to pick up a beeper where there is none.

For testing only (future features):
            F7: creates New_improved_robot
            Right arrow:   turn robot right (only New_improved_robot)
            F8: creates Used_robot (default)"""
        code = event.KeyCode()
        if code == WXK_UP:           # up arrow
            if 'robot' in self.world.robot_dict:
                try:
                    self.world.MoveRobot('robot') # may raise an exception
                    self.scrollWorld('robot')
                except rur_dialogs.HitWallException, mesg:
                    rur_dialogs.DialogHitWallError(mesg)
        elif code == WXK_LEFT:       # left arrow
            self.world.TurnRobotLeft('robot')
        elif code == 112:            # p - lower case
            if 'robot' in self.world.robot_dict:
                try:
                    self.world.robot_dict['robot'].pick_beeper()
                    if self.editor:
                        self.editor.DestroyChildren()
                        wxStaticText(self.editor, -1, self.UpdateEditor(), (10, 10))
                except rur_dialogs.PickBeeperException, mesg:
                    rur_dialogs.DialogPickBeeperError(mesg)
        elif code == 80:             # P - upper case
            if 'robot' in self.world.robot_dict:
                try:
                    self.world.robot_dict['robot'].put_beeper()
                    if self.editor:
                        self.editor.DestroyChildren()
                        wxStaticText(self.editor, -1, self.UpdateEditor(), (10, 10))
                except rur_dialogs.PutBeeperException, mesg:
                    rur_dialogs.DialogPutBeeperError(mesg)
        elif code == 101:            # e - lower case
            if self.world.editWalls:
                self.world.editWalls = False
                self.world.DoDrawing()
            else:
                self.world.editWalls = True
                self.world.DoDrawing()
#######
##        Resizing world and giving robot beepers moved out to rur_dialogs.py
#####
        elif code == WXK_F5:
            rur_dialogs.rurMessageDialog(message, "Help :-)")
        elif code == WXK_F7:
            self.world.robot_dict['robot'] = New_improved_robot(parent = self.world)
            self.world.DoDrawing()
            self.newRobot = True
        elif code == WXK_F8:
            self.world.robot_dict['robot'] = Used_robot(parent = self.world)
            self.world.DoDrawing()
            self.newRobot = False
        elif code == WXK_RIGHT:       # right arrow
            if self.newRobot:         # only New_improved_robot can turn right.
                self.world.TurnRobotRight('robot')
            else:
                pass
        else:
            pass
        if self.world.updateImage:
            self.drawImage()
            self.Refresh()
    #- end of MyKeys()-----------------------------------------

    def scrollWorld(self, name):
        '''Determines if window needs scrolling and does necessary.'''
        # determine robot image bounding box
        x0, y0 = self.world.robot_image_origin
        x1, y1 = self.world.robot_dict[name]._image_size
        x1 += x0
        y1 += y0

        # marginal space around robot
        margin = self.world.tile_narrow + self.world.tile_wide
        margin *= 2

        # position of top left visible window in "scrollrate" units
        xView, yView = self.GetViewStart()
        xViewOld, yViewOld = xView, yView

        # corresponding amount of pixel per "scroll"
        xDelta, yDelta = self.GetScrollPixelsPerUnit()

        # size of wisible window
        width, height = self.GetSizeTuple()

        #-- Determine if window needs to be scrolled so that object
        # remains visible.  Assume that the object fits entirely
        # in the visible view.
        if x0 - margin < xView*xDelta:
            xView = max(0, -1 + (x0-margin)/xDelta)
        elif x1 + margin > xView*xDelta + width:
            xView = (x1 + margin - width)/xDelta
        if y0 - margin < yView*yDelta:
            yView = max(0, -1 + (y0-margin)/yDelta)
        elif y1 + margin > yView*yDelta + height:
            yView = (y1 + margin - height)/yDelta
        # if needed, scroll window by required amount to keep image in view
        if xView != xViewOld or yView != yViewOld:
            self.Scroll(xView, yView)

    def drawImage(self):
        dc = wxBufferedDC(None, self.buffer)
        dc.BeginDrawing()
        # Simply copy the world image onto the buffer
        dc.DrawBitmap(self.world.world_image, 0, 0, True)
        # update the editor
        if self.editor:
            self.editor.DestroyChildren() # removes the old wxStaticText
            wxStaticText(self.editor, -1, self.UpdateEditor(), (10, 10))
        dc.EndDrawing()


    def UpdateEditor(self):
        av_string = "avenues = " + str(self.world.num_cols//2)
        st_string = "streets = " + str(self.world.num_rows//2)
        robot_string = ''
        for name in self.world.robot_dict:
            x, y = self.world.robot_dict[name].getPos()
            beepers = self.world.robot_dict[name]._beeper_bag
            orientation = self.world.robot_dict[name]._getOrientationKey()
            robot_string += name + " = " + \
                  self.world.robot_dict[name]._getInfoString() + "\n"
        if len(self.world.walls_list) > 0:
            wall_string = "walls = [\n"
            for item in self.world.walls_list:
                wall_string = wall_string + ("    " + str(item)
                                         + ', \n')
            wall_string = wall_string[0:-3]+'\n]'
        else:
            wall_string = "walls = []"
        if len(self.world.beepers_dict) > 0:
            beeper_list = "beepers = {\n"
            for key in self.world.beepers_dict:
                beeper_list = beeper_list + ("    " + str(key) + ': '
                       + str(self.world.beepers_dict[key])+ ', \n')
            beeper_list = beeper_list[0:-3]+'\n}'
        else:
            beeper_list = "beepers = {}"

        worldmap = av_string + '\n' + st_string + '\n' + robot_string \
                  + wall_string + '\n' +  beeper_list
        return worldmap

#--- popup windows

class MyPopupWindow(wxPopupWindow):
    '''Allows user to reposition the popup window.'''
    def __init__(self, parent, mesg, width, height, colour="wheat"):
        wxPopupWindow.__init__(self, parent, wxSIMPLE_BORDER)
        self.SetBackgroundColour(colour)
        st = wxStaticText(self, -1, mesg, pos=(10,10))

        EVT_LEFT_DOWN(self, self.OnMouseLeftDown)
        EVT_MOTION(self, self.OnMouseMotion)
        EVT_LEFT_UP(self, self.OnMouseLeftUp)
        self.SetSize( (width, height) )
        self.SetFocus()

    def OnMouseLeftDown(self, evt):
        self.Refresh()
        self.ldPos = evt.GetEventObject().ClientToScreen(evt.GetPosition())
        self.wPos = self.ClientToScreen((0,0))
        self.CaptureMouse()


    def OnMouseMotion(self, evt):
        if evt.Dragging() and evt.LeftIsDown():
            dPos = evt.GetEventObject().ClientToScreen(evt.GetPosition())
            nPos = (self.wPos.x + (dPos.x - self.ldPos.x),
                    self.wPos.y + (dPos.y - self.ldPos.y))
            self.Move(nPos)

    def OnMouseLeftUp(self, evt):
        self.ReleaseMouse()



class AddBeeperPopup(MyPopupWindow):
    """Allows user to give robot between 0 and 1000 beepers to carry."""
    def __init__(self, parent):
        mesg = "Select the desired value, or sum of " +\
               "of values as the number of beepers in\n" +\
               "the robot's beeper bag."
        MyPopupWindow.__init__(self, parent, mesg, 380, 200)
        self.parent = parent
        self.robot = parent.world.robot_dict
        self.max1 = 20
        self.max2 = 980
        if self.robot['robot']._beeper_bag > self.max1:
            self.beepers1 = self.max1
            self.beepers2 = self.robot['robot']._beeper_bag - self.max1
        else:
            self.beepers1 = self.robot['robot']._beeper_bag
            self.beepers2 = 0

        self.slider1 = wxSlider(
            # id, value, min, max, (x, y), (length, height)
            self, -1, self.beepers1, 0, self.max1, (60, 70), (250, -1),
            wxSL_HORIZONTAL | wxSL_AUTOTICKS | wxSL_LABELS
            )
        self.slider1.SetTickFreq(1, 1)
        self.slider2 = wxSlider(
            # id, value, min, max, (x, y), (length, height)
            self, -1, self.beepers2, 0, self.max2, (60, 110), (250, -1),
            wxSL_HORIZONTAL | wxSL_AUTOTICKS | wxSL_LABELS
            )
        self.slider2.SetTickFreq(35, 1)

        b = wxButton(self, -1, "Ok", (150, 160))
        if rur_misc.WX_VERSION == 2.5:
            self.Bind(wx.EVT_BUTTON, self.OnOK, b)
        else:
            EVT_BUTTON(self, -1, self.OnOK)
        wxCallAfter(self.Refresh)

    def OnOK(self, evt):
        num = self.slider1.GetValue()
        num += self.slider2.GetValue()
        self.robot['robot']._beeper_bag = num
        if self.parent.editor:
            self.parent.editor.DestroyChildren()
            self.parent.static = wxStaticText(self.parent.editor, -1,
                                 self.parent.UpdateEditor(), (10, 10))
        self.Show(False)
        self.Destroy()

#--- Self-testing part (visual testing)-------
class RURworldParent(wxFrame):
    def __init__(self, parent):
        wxFrame.__init__(self, parent, -1, "Press F5 for help", size=(400, 400))
        gui = WorldGUI(self, -1)  # create and draw an empty world; no robot
        gui.world.addOneRobot()
        gui.world.DoDrawing() # redraw to include robot on canvas
        gui.drawImage()   # redraw to include robot on buffer


