""" RUR-PLE: Roberge's Used Robot - a Python Learning Environment
    rur_dialogs.py - dialogs, messages and exceptions used to communicate
                 with user.
    Version 0.7
    Author: Andre Roberge    Copyright  2005
    andre.roberge@gmail.com
"""
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import os
#import rur_translatable
import rur_misc
from utils import L2U,get_rootdir

from wxPython.wx import *

class MyDialogs(wxDialog):
    '''Custom dialogs that include "interesting" image and text.'''
    def __init__(self, parent, ID, title, mesg, btn_txt, img,
                 pos=wxDefaultPosition, size=wxDefaultSize,
                 style=wxDEFAULT_DIALOG_STYLE):

        x, y = img.GetWidth(), img.GetHeight()
        # leave room for text
        size = wxSize(x, y+100)

        # Instead of calling wxDialog.__init__ we precreate the dialog
        # so we can set an extra style that must be set before
        # creation, and then we create the GUI dialog using the Create
        # method.
        pre = wxPreDialog()
        pre.Create(parent, ID, title, pos, size, style)
        # This next step is the most important, it turns this Python
        # object into the real wrapper of the dialog (instead of pre)
        # as far as the wxPython extension is concerned.
        self.this = pre.this

        # Add the message from the exception
        wxStaticText(self,-1, mesg, wxPoint(0,0))
        # custom image
        wxStaticBitmap(self, -1, img, wxPoint(0, 40))
        btnId = wxNewId()
        button = wxButton(self, btnId, btn_txt)
        x, y = size
        button.SetPosition(wxPoint(x/2-30, y-60))
        EVT_BUTTON(self, btnId, self.OnCloseMe)
        EVT_CLOSE(self, self.OnCloseWindow)

        self.CenterOnScreen()
        self.ShowModal()

    def OnCloseMe(self, event):
        self.Close(True)

    def OnCloseWindow(self, event):
        self.Destroy()


#--- Error dialogs

class DialogHitWallError(MyDialogs):
    """ Custom dialog """
    def __init__(self, excpt):
        MyDialogs.__init__(self,None, -1,"Error", excpt.mesg,
                              L2U(_("Turn off")), L2U(_("Ouch! I hit a wall!")))

class DialogPutBeeperError(MyDialogs):
    """ Will need to have its own image """
    def __init__(self, excpt):
        MyDialogs.__init__(self,None, -1,"Error", excpt.mesg,
                              L2U(_("I don't have any beepers left!")))

class DialogPickBeeperError(MyDialogs):

    """ Will need to have its own image """
    def __init__(self, excpt):
        MyDialogs.__init__(self,None, -1,"Error", excpt.mesg,
                              L2U(_("Hey! There are no beepers here!")))

class NormalEndDialog(MyDialogs):
    def __init__(self, excpt):
        MyDialogs.__init__(self,None, -1,l2U(_("Success!")), excpt.mesg,
                              L2U(_("Ok")), MINI_SPLASH)
#--- Exceptions

class LogicException(Exception):
    '''General class to deal with problematic instructions given to robot.'''
    def __init__(self,mesg):
        self.mesg = mesg
    def __str__(self):
        return repr(self.mesg)

class HitWallException(LogicException):
    pass

class PutBeeperException(LogicException):
    pass

class PickBeeperException(LogicException):
    pass

class UserStopException(LogicException):
    pass

class NoTurnOffException(LogicException):
    pass

class NormalEnd(LogicException):
    """ This is a normal condition raised when the program ends properly."""
    pass


#--- General Message

def rurMessageDialog(text, title):

    messageDialog = wxMessageDialog(None, text, title,
                                     wxOK | wxICON_INFORMATION)
    messageDialog.ShowModal()
    messageDialog.Destroy()

class RobotBeeperDialog(wxDialog):
    def __init__(self, parent, ID, title, pos=wxDefaultPosition,
                 size=wxDefaultSize, style=wxDEFAULT_DIALOG_STYLE):

        # leave room for text
        size = wxSize(380, 250)

        # Instead of calling wxDialog.__init__ we precreate the dialog
        # so we can set an extra style that must be set before
        # creation, and then we create the GUI dialog using the Create
        # method.
        pre = wxPreDialog()
        pre.Create(parent, ID, title, pos, size, style)
        # This next step is the most important, it turns this Python
        # object into the real wrapper of the dialog (instead of pre)
        # as far as the wxPython extension is concerned.
        self.this = pre.this
        #self.SetBackgroundColour("wheat")
        mesg = L2U(_("Select the desired value, or sum of values as the number\nof beepers in the robot's beeper bag."))
        wxStaticText(self,-1, mesg, wxPoint(10,10))
        self.parent = parent
        self.robot = parent.world.robot_dict
        self.max1 = 20
        self.max2 = 980
        if self.robot['robot']._beeper_bag > self.max1:
            self.beepers1 = self.max1
            self.beepers2 = self.robot['robot']._beeper_bag - self.max1
        else:
            self.beepers1 = self.robot['robot']._beeper_bag
            self.beepers2 = 0
        wxStaticText(self,-1, L2U(_("Number of beepers < 100")), wxPoint(100,60))
        self.slider1 = wxSlider(self,
            # id, value, min, max, (x, y), (length, height)
            -1, self.beepers1, 0, self.max1, (60, 70), (250, -1),
            wxSL_HORIZONTAL | wxSL_AUTOTICKS | wxSL_LABELS
            )
        wxStaticText(self,-1, L2U(_("Number of beepers > 100")), wxPoint(100,120))
        self.slider1.SetTickFreq(1, 1)
        self.slider2 = wxSlider(self,
            # id, value, min, max, (x, y), (length, height)
             -1, self.beepers2, 0, self.max2, (60, 130), (250, -1),
            wxSL_HORIZONTAL | wxSL_AUTOTICKS | wxSL_LABELS
            )
        self.slider2.SetTickFreq(35, 1)

        btnId = wxNewId()
        button = wxButton(self, btnId,'Ok',(220, 190))
        #button.SetPosition(wxPoint(180, 170))
        EVT_BUTTON(self, btnId, self.OnCloseMe)
        
        btnId = wxNewId()
        button = wxButton(self, btnId,'Cancel',(40, 190))
        #button.SetPosition(wxPoint(180, 170))
        EVT_BUTTON(self, btnId, self.OnCloseWindow)
        
        EVT_CLOSE(self, self.OnCloseWindow)

        self.CenterOnScreen()
        self.ShowModal()

    def OnCloseMe(self, event):
        num = self.slider1.GetValue()
        num += self.slider2.GetValue()
        self.robot['robot']._beeper_bag = num
        # retrieve backup value
        x, y, key, beep = self.parent.backup_dict['robot']
        # temporary set to new value
        self.parent.backup_dict['robot'] = x, y, key, num
        self.parent.UpdateWorld(num)
        # recover backup value after updating World file window
        self.parent.backup_dict['robot'] = x, y, key, beep
        self.Show(False)
        self.Destroy()

    def OnCloseWindow(self, event):
        self.Destroy()

class ResizeWorldDialog(wxDialog):
    def __init__(self, parent, ID, title, pos=wxDefaultPosition,
                 size=wxDefaultSize, style=wxDEFAULT_DIALOG_STYLE):


        # leave room for text
        size = wxSize(380, 420)

        # Instead of calling wxDialog.__init__ we precreate the dialog
        # so we can set an extra style that must be set before
        # creation, and then we create the GUI dialog using the Create
        # method.
        pre = wxPreDialog()
        pre.Create(parent, ID, title, pos, size, style)
        # This next step is the most important, it turns this Python
        # object into the real wrapper of the dialog (instead of pre)
        # as far as the wxPython extension is concerned.
        self.this = pre.this
        #self.SetBackgroundColour("wheat")

        mesg = L2U(_("Select the desired world dimensions (streets and avenues).\n"\
               "WARNING: The existing walls and beepers will be removed."))
        wxStaticText(self,-1, mesg, wxPoint(10,10))
        self.parent = parent

        try:
            edit_world_image = wxImage(os.path.join(get_rootdir(),'bitmaps','edit_world.png'),
                                        wxBITMAP_TYPE_PNG).ConvertToBitmap()
        except Exception,info:
            print __name__, info
            print "Problem loading edit_world.png in WorldDisplay.py"

        self.world = parent.world
        self.WorldDisplay = parent.WorldDisplay
        self.robot = parent.world.robot_dict

        min_avenues = 7
        min_streets = 7
        max_avenues = 30
        max_streets = 30
        self.streets = parent.world.st
        self.avenues = parent.world.av


        if rur_misc.WX_VERSION == 2.5:
            self.slider_street = wxSlider(
                # id, value, min, max, (x, y), (length, height)
                self, -1, self.streets, min_streets, max_streets,
                (40, 90), (60, 260), 
                wxSL_VERTICAL | wxSL_AUTOTICKS | wxSL_LABELS
                )
        else:
            self.slider_street = wxSlider(
                # id, value, min, max, (x, y), (length, height)
                self, -1, self.streets, min_streets, max_streets,
                (40, 70), (-1, 300),
                wxSL_VERTICAL | wxSL_AUTOTICKS | wxSL_LABELS
                )

        self.slider_street.SetTickFreq(1, 1)
        self.slider_avenue = wxSlider(
            # id, value, min, max, (x, y), (length, height)
            self, -1, self.avenues, min_avenues, max_avenues,
            (60, 50), (300, -1),
            wxSL_HORIZONTAL | wxSL_AUTOTICKS | wxSL_LABELS
            )
        self.slider_avenue.SetTickFreq(1, 1)
        #
        wxStaticBitmap(self, -1, edit_world_image, (120, 110))
        
        btnId = wxNewId()
        button = wxButton(self, btnId,'Ok',(220, 380))
        #button.SetPosition(wxPoint(180, 170))
        EVT_BUTTON(self, btnId, self.OnCloseMe)
        
        btnId = wxNewId()
        button = wxButton(self, btnId,'Cancel',(40, 380))
        #button.SetPosition(wxPoint(180, 170))
        EVT_BUTTON(self, btnId, self.OnCloseWindow)
        
        EVT_CLOSE(self, self.OnCloseWindow)

        self.CenterOnScreen()
        self.ShowModal()

    def OnCloseMe(self, event):
        st = self.slider_street.GetValue()
        av = self.slider_avenue.GetValue()

        self.world.resetDimensions(av, st)
        if 'robot' in self.robot:
            self.robot['robot']._setPos(1, 1) # brings robot back to the origin
            self.robot['robot']._facing = 3   # facing east
            self.robot['robot'].robot_image = \
                     self.robot['robot']._image[self.robot['robot']._facing]
        # prepare to recreate the background images
        self.world.background_images_created = False
        self.world.AdjustWorldSize()
        self.world.InitTileSizes()
        self.WorldDisplay.InitialiseVariables()
        self.world.DoDrawing()
        # create a new bitmap image
        self.parent.buffer = wxEmptyBitmap(self.world.maxWidth,
                                           self.world.maxHeight)
        self.WorldDisplay.drawImage()
        self.parent.Refresh()
        self.Show(False)
        self.Destroy()

    def OnCloseWindow(self, event):
        self.Destroy()

class SetDelayDialog(wxDialog):
    def __init__(self, parent, ID, title, pos=wxDefaultPosition,
                 size=wxDefaultSize, style=wxDEFAULT_DIALOG_STYLE):

        # leave room for text
        size = wxSize(450, 300)

        # Instead of calling wxDialog.__init__ we precreate the dialog
        # so we can set an extra style that must be set before
        # creation, and then we create the GUI dialog using the Create
        # method.
        pre = wxPreDialog()
        pre.Create(parent, ID, title, pos, size, style)
        # This next step is the most important, it turns this Python
        # object into the real wrapper of the dialog (instead of pre)
        # as far as the wxPython extension is concerned.
        self.this = pre.this
        #self.SetBackgroundColour("white")
        self.parent = parent

        mesg = "Select the desired robot speed"
        wxStaticText(self,-1, mesg, wxPoint(10,10))
        # here speed is somewhat inversely related to the time delay
        try:
            snail_img = wxImage('snail.png',
                                wxBITMAP_TYPE_PNG).ConvertToBitmap()
        except Exception,info:
            print __name__, info
            print "Problem loading snail.png in WorldDisplay.py"

        try:
            cheetah_img = wxImage('cheetah.png',
                                  wxBITMAP_TYPE_PNG).ConvertToBitmap()
        except Exception,info:
            print __name__, info
            print "Problem loading cheetah.png in WorldDisplay.py"

        self.min_speed = 1
        self.max_speed = 100

        actual_delay = self.parent.world.robot_dict['robot'].delay
        default_speed = int(1.0/actual_delay)
        self.slider_speed = wxSlider(
            # id, value, min, max, (x, y), (length, height)
            self, -1, default_speed, self.min_speed, self.max_speed,
            (100, 80), (200, -1),
            wxSL_HORIZONTAL | wxSL_AUTOTICKS | wxSL_LABELS
            )
        self.slider_speed.SetTickFreq(10, 1)

        wxStaticBitmap(self, -1, snail_img, (10, 50))
        wxStaticBitmap(self, -1, cheetah_img, (320, 50))
        btnId = wxNewId()
        button = wxButton(self, btnId, "ok")
        button.SetPosition(wxPoint(200, 200))
        EVT_BUTTON(self, btnId, self.OnCloseMe)
        EVT_CLOSE(self, self.OnCloseWindow)

        self.CenterOnScreen()
        self.ShowModal()

    def OnCloseMe(self, event):
        speed = self.slider_speed.GetValue()
        self.parent.world.robot_dict['robot'].delay = 1./speed
        self.Show(False)
        self.Destroy()

    def OnCloseWindow(self, event):
        self.Destroy()


