# Index.pm - Project Gutenberg index class definition for Gutenbook
# http://www.gutenbook.org/
# Copyright (C) 2000  Lee "Lefty" Burgess
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

package PGB::Index;

use PGB;
use PGB::File;
use PGB::Network;
use PGB::PropList;

use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

require Exporter;
require AutoLoader;

@ISA = qw(Exporter AutoLoader);
$VERSION = '0.01';

sub new {
  my ($class) = @_;
  my $self = {};
  bless($self, $class);
  $self->_initialize();
  return($self);
}

sub _initialize {
  my ($self) = @_;
  my (@array, $index);
  my $http_index = $plist_hash->{"http_index"};

  opendir(DIR, $plist_hash->{"library_dir"}) || 
    BUG_ME(1, "Cannot open directory " . $plist_hash->{"library_dir"} . ": $!");
  @array = grep(/[txt|zip]$/, readdir(DIR));
  closedir(DIR);
  
  $self->{"local_files"} = \@array;
    
  if (-f $local_index && $plist_hash->{"cache_local_index"}) {
    $self->{"library_hash"} = retrieve($local_index);
  }
  else {
    my $index = http_get($http_index);
    
    foreach (@{$index}) {
      chomp;
      
      # NEW INDEX looks like this:
      #Mon Year [filename.ext]####  Title and Author
      #Apr 2000 [poe5vxxx.xxx]2151  The Sphinx, by Edgar Allan Poe
      #Oct 2000  [dorat.xxx]2374  Dora Thorne, by Charlotte M. Braeme
      #Oct 2000 [sbeafx.xxx]2359  Stories by English Authors in France, Scribners Ed
      
      if (my ($month, $year, $file, $number, $title) = 
	  /^(\w{3})\s(\d{4})\s+\[(.{8}\..{3})\]([\w|\d]+)\s+(.+)$/) {
	# key = etext number; value = hash ref (keys = file, title, month, year, local)
	my $key;
	my $copyright = "";
	my $num_len;
	$number =~ s/^\s+//;
	$number =~ s/\s+$//;
	$key = $number;
	if ($number =~ /[a-zA-Z]/) {
	  if ($number =~ /c/i) {
	    $copyright = "yes";
	  }
	  $number =~ s/[a-zA-Z]//g;
	}
	$num_len = length($number);
	if ($num_len < 5) {
	  my $diff = 5 - $num_len;
	  for (my $i = 0; $i < $diff; $i++) {
	    $number = "0" . $number;
	  }
	}
	$title =~ s/\s+$//;
	$self->{"library_hash"}->{$key} = { 'FILE' => $file,
					    'COPYRIGHT' => $copyright,
					    'NUMBER' => $number,
					    'KEY' => $key,
					    'TITLE' => $title,
					    'MONTH' => $month,
					    'YEAR' => $year,
					    'LOCAL' => $self->local_file_exists($file) };
      }
    }
  }
  
  if ($plist_hash->{"cache_local_index"}) {
    $self->write_index();
  }

  $self->{"search_holder"} = undef;
  return;
}

sub destroy {
  my ($self) = @_;
  undef($self->{"library_hash"});
  undef($self->{"local_files"});
  undef($self->{"search_holder"});
  undef($self);
  return;
}

sub write_index {
  my ($self) = @_;
  
  store($self->{"library_hash"}, $local_index);

  return;
}

sub library_hash {
  my ($self) = @_;
  return($self->{"library_hash"});
}

sub total_titles {
  my ($self) = @_;
  return(scalar(keys(%{$self->{"library_hash"}})));
}

sub local_file_exists {
  my ($self, $file) = @_;
  my @matches;
  
  $file =~ s/\.x+$//;
  $file =~ s/x+$//;
  
  # TODO: this is not working completely
  # the file field of GUTINDEX.ALL is not unique
  # this leads to erroneous indications of present files that
  # really are not
  @matches = grep(/$file.*\.[txt|zip]/, @{$self->{"local_files"}});
  
  if (@matches) {
    return($matches[0]);
  }
  return("");
}

sub search_in_title {
  my ($self, $string, $notebook, $hashref) = @_;
  my $page = $notebook->get_current_page();
  my $key = (sort(keys(%{$hashref})))[$page];
  my $pages = scalar(keys(%{$hashref})) - 1;
  my ($title, $srow, $spage);

  if (!defined($self->{"search_holder"})) {
    # holding row and page set to first row and page
    $self->{"search_holder"} = [ 0, $page ];
  }
  else {
    # increment so we don't get stuck on the last match
    $self->{"search_holder"}[0]++;
  }
 
  $title = $hashref->{$key}[1]->get_text($self->{"search_holder"}[0], 1);
  ($srow, $spage) = @{$self->{"search_holder"}};

  while (defined($title)) {
    my $test;
    if ($title =~ /$string/i) {
      BUG_ME(0, "Found $string in row " . $self->{"search_holder"}[0] . ": $title");
      $notebook->set_page($self->{"search_holder"}[1]);
      if ($self->{"search_holder"}[0] > 7) {
	$hashref->{$key}[1]->moveto($self->{"search_holder"}[0] - 7, 0, 0, 0);
      }
      else {
	$hashref->{$key}[1]->moveto(0, 0, 0, 0);
      }
      $hashref->{$key}[1]->select_row($self->{"search_holder"}[0], 0);

      return($self->{"search_holder"}[0]);
    }
    $self->{"search_holder"}[0]++;
    $test = $hashref->{$key}[1]->get_text($self->{"search_holder"}[0], 1);
    if (!defined($test)) {
      $self->{"search_holder"}[0] = 0;
      $self->{"search_holder"}[1]++;
      if ($self->{"search_holder"}[1] > $pages) {
	$self->{"search_holder"}[1] = 0;
      }
      $key = (sort(keys(%{$hashref})))[$self->{"search_holder"}[1]];
      BUG_ME(0, "Wrapping search back to row " . $self->{"search_holder"}[0] . ", page " . $self->{"search_holder"}[1]);
      $test = $hashref->{$key}[1]->get_text($self->{"search_holder"}[0], 1);
    }
    $title = $test;
    if ($srow == $self->{"search_holder"}[0] && $spage == $self->{"search_holder"}[1]) {
      # search not found
      undef($self->{"search_holder"});
      $hashref->{$key}[1]->unselect_all();
      return(undef);
    }
  }
}

1;


