/*
 * Guifications - The end all, be all, toaster popup plugin
 * Copyright (C) 2003-2004 Gary Kramlich
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <glib.h>

#include <debug.h>
#include <notify.h>
#include <prefs.h>
#include <util.h>

#ifdef HAVE_CONFIG_H
# include "../gf_config.h"
#endif

#include "gf_internal.h"
#include "gf_preferences.h"
#include "gf_release.h"

static void
gf_release_check_cb(void *userdata, const char *data, size_t len) {
	const gchar *changelog = data;
	GString *notification;
	gchar *cur_ver, *formatted;
	gint i = 0;

	if (!changelog || !len)
		return;

	while (changelog[i] && changelog[i] != '\n')
		i++;

	cur_ver = g_strndup(changelog, i);
	changelog += i;

	while (*changelog == '\n')
		changelog++;

	notification = g_string_new("");
	g_string_append_printf(notification,
						   _("Guifications %s is available, "
							 "you are running version %s."),
						   cur_ver,
#ifdef _WIN32
						   GF_VERSION
#else
						   VERSION
#endif
						   );

	notification = g_string_append(notification, "<hr>");

	if (*changelog) {
		formatted = gaim_strdup_withhtml(changelog);
		g_string_append_printf(notification, "<b>%s</b><br>%s",
							   _("ChangeLog:"), formatted);
		g_free(formatted);
	}

	formatted = g_strdup_printf(_("You can download version %s from"), cur_ver);
	g_string_append_printf(notification,
						   "<hr>%s <a href=\"%s\">%s</a>.",
						   formatted, GF_WEBSITE, GF_WEBSITE);
	g_free(formatted);

	gaim_notify_formatted(NULL, _("New version available"),
						  _("There is a new version of Guifications available!"), NULL,
						  notification->str, NULL, NULL);

	g_string_free(notification, TRUE);
	g_free(cur_ver);
}

void
gf_release_check() {
	int last_checked;
	gchar *url;
	time_t t;

	if(gaim_prefs_get_bool(GF_PREF_ADVANCED_RELEASE_NOTIFICATION)) {
		last_checked = gaim_prefs_get_int(GF_PREF_ADVANCED_RELEASE_LAST_CHECK);
		t = time(NULL);

		if (t - last_checked > 86400 /* one day */) {
			gaim_debug_info("Guifications", "Checking for a new version\n");

			url = g_strdup_printf("http://guifications.sourceforge.net/version.php?module=guifications&version=%s",
#ifdef _WIN32
								  GF_VERSION);
#else
								  VERSION);
#endif
			gaim_url_fetch(url, TRUE, NULL, FALSE, gf_release_check_cb, NULL);
			gaim_prefs_set_int(GF_PREF_ADVANCED_RELEASE_LAST_CHECK, t);
			g_free(url);
		}
	}
}
