/*
 * Parser for standard guessnet configuration file
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#define APPNAME PACKAGE
#else
#warning No config.h found: using fallback values
#define APPNAME __FILE__
#define VERSION "unknown"
#endif

#include "GuessnetParser.h"
#include "stringf.h"
#include "Regexp.h"
#include "Environment.h"

#include <string>

using namespace std;
using namespace stringf;

/* Parse the input from `input'
 * To make it simple, use regexps on input lines instead of implementing a real
 * parser.
 */
void GuessnetParser::parse(FILE* input, ScanConsumer& sc) throw (Exception)
{
	ExtendedRegexp null_line("^[[:blank:]]*(#.*)?$");
	ExtendedRegexp old_input_line(
		"^[[:blank:]]*[0-9]+\\.[0-9]+\\.[0-9]+\\.[0-9]+[[:blank:]]+"
		"([0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2})"
		"[[:blank:]]+([0-9]+\\.[0-9]+\\.[0-9]+\\.[0-9]+)[[:blank:]]+([[:alnum:]_+-]+)"
		"[[:blank:]]*$", 4);
	ExtendedRegexp peer_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"peer[[:blank:]]+"
		"([0-9]+\\.[0-9]+\\.[0-9]+\\.[0-9]+)[[:blank:]]+"
		"([0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2})"
		"[[:blank:]]*$", 4);
	ExtendedRegexp peer_nomac_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"peer[[:blank:]]+"
		"([0-9]+\\.[0-9]+\\.[0-9]+\\.[0-9]+)"
		"[[:blank:]]*$", 3);
	ExtendedRegexp script_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"script[[:blank:]]+(.+)$", 3);
	ExtendedRegexp cable_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"missing-cable[[:blank:]]*$", 2);
	ExtendedRegexp pppoe_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"pppoe[[:blank:]]*$", 2);
	ExtendedRegexp wireless_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"wireless-(ap|id|scan)[[:blank:]]*(.+)$", 4);

	string line;
	int linenum = 1;
	int found = 0;
	int c;
	while ((c = fgetc(input)) != EOF)
	{
		if (c != '\n')
			line += c;
		else
		{
			if (null_line.match(line))
			{
				//fprintf(stderr, "EMPTY\n");
			}
			else if (old_input_line.match(line))
			{	
				string mac = old_input_line[1];
				string ip = old_input_line[2];
				string name = old_input_line[3];

				//debug("Will scan network %.*s for %.*s (%.*s)\n",
				//		PFSTR(name), PFSTR(ip), PFSTR(mac));

				struct ether_addr macAddr;
				parse_mac(&macAddr, mac);

				IPAddress ipAddr(ip);

				sc.handleScan(new PeerScan(name, macAddr, ipAddr));
				found++;
			}
			else if (peer_line.match(line))
			{
				string name = peer_line[1];
				string ip = peer_line[2];
				string mac = peer_line[3];
				//debug("Will scan network %.*s for %.*s (%.*s)\n",
				//		PFSTR(name), PFSTR(ip), PFSTR(mac));

				struct ether_addr macAddr;
				parse_mac(&macAddr, mac);

				IPAddress ipAddr(ip);

				//fprintf(stderr, "  GOT: %.*s: %.*s, %.*s\n",
						//PFSTR(profileName), PFSTR(ip), PFSTR(mac));

				sc.handleScan(new PeerScan(name, macAddr, ipAddr));
				found++;
			}
			else if (peer_nomac_line.match(line))
			{
				string name = peer_line[1];
				string ip = peer_line[2];
				//debug("Will scan network %.*s for %.*s (%.*s)\n",
				//		PFSTR(name), PFSTR(ip), PFSTR(mac));

				struct ether_addr macAddr;
				bzero(&macAddr, sizeof(struct ether_addr));

				IPAddress ipAddr(ip);

				//fprintf(stderr, "  GOT: %.*s: %.*s, %.*s\n",
						//PFSTR(profileName), PFSTR(ip), PFSTR(mac));

				sc.handleScan(new PeerScan(name, macAddr, ipAddr));
				found++;
			}
			else if (script_line.match(line))
			{
				string name = script_line[1];
				string cmd = script_line[2];
				//fprintf(stderr, "TEST: %.*s\n", PFSTR(cmd));
				//debug("Will use script %.*s to test %.*s\n",
				//	PFSTR(cmd), PFSTR(name));

				sc.handleScan(new ScriptScan(name, cmd));
				found++;
			}
			else if (cable_line.match(line))
			{
				string name = script_line[1];
				//fprintf(stderr, "TEST: %.*s\n", PFSTR(cmd));
				//debug("Will use script %.*s to test %.*s\n",
				//	PFSTR(cmd), PFSTR(name));

				sc.handleScan(new LinkBeatScan(name));
				found++;
			}
#ifdef HAVE_PPPOE
			else if (pppoe_line.match(line))
			{
				string name = script_line[1];
				//fprintf(stderr, "TEST: %.*s\n", PFSTR(cmd));
				//debug("Will use script %.*s to test %.*s\n",
				//	PFSTR(cmd), PFSTR(name));

				sc.handleScan(new ScriptScan(name, string(PPPOE " -I ") + Environment::get().iface() + " -A"));
				found++;
			}
#endif
			else if (wireless_line.match(line))
			{
				string name = script_line[1];
				string type = script_line[2];
				string parms = script_line[3];

				sc.handleScan(new ScriptScan(name, string(SCRIPTDIR "/test-wifi-") + type + " " + parms));
				found++;
			}
			else
			{
				warning("Parse error at line %d: line ignored\n", linenum);
			}
			line = string();
			linenum++;
		}
	}
	debug("%d candidates found in input\n", found);
}

// vim:set ts=4 sw=4:
