/* This is the implementation of our primitive virtual filesystem
 * for the unix filesystem.
 * It will map everything straight down to the operating system */

/* uncomment for debugging */
//#define DEBUG
//
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <stdlib.h>

#include "int.h"

#include "vfs.h"
#include "vfs_unixtree.h"

#include "helpings.h"
#include "datacopydlg.h"
#include "streamcopy.h"
#include "dirlow.h"

#include "multisession.h"

#define ROOTDIR ((char*)((vfs_filesystem*)fs)->data)

#define TRANSLATE \
   char *path=((ROOTDIR)?helpings_fileinpath(ROOTDIR,rpath):strdup(rpath))

int vfs_unixtree_createdir(gpointer fs,char *rpath)
{
   int result;
   TRANSLATE;
   result=mkdir(path,0770);
   free(path);
   return result;
};

typedef struct
{
   DIR *handle;
   char *path;
   virtualdir_VDIR *vhandle;
}
vfs_unixtree_dirhandle;

gtoaster_handle_t vfs_unixtree_opendir(gpointer fs,char *rpath)
{
   DIR *handle;
   vfs_unixtree_dirhandle *result;
   TRANSLATE;
   handle=opendir(path);
   result=(vfs_unixtree_dirhandle*)((handle)
				    ?malloc(sizeof(vfs_unixtree_dirhandle))
				    :NULL);
   if (handle)
     {
	result->handle=handle;
	result->path=strdup(rpath);
	result->vhandle=NULL;
     };
   free(path);
   return (gtoaster_handle_t)result;
};

char *vfs_unixtree_readdirentry(gpointer fs,gtoaster_handle_t handle)
{
   struct dirent *current;
   char *result=NULL;
   vfs_unixtree_dirhandle *myhandle=(vfs_unixtree_dirhandle*)handle;
   if (myhandle->handle)
     {
	current=readdir(myhandle->handle);
	if (current)
	  {
	     if (strcmp(current->d_name,".")&&strcmp(current->d_name,".."))
	       result=helpings_fileinpath(myhandle->path,
					  current->d_name);
	     else
	       result=vfs_unixtree_readdirentry(fs,handle);
	  }
	else
	  {
	     closedir(myhandle->handle);
	     myhandle->handle=NULL;
	     if ((multisession_lastsession)&&(ROOTDIR))
	       {
#ifdef DEBUG
		  printf("vfs_unixtree: opening virtual multisession fs, path=%s\n",myhandle->path);
#endif
		  myhandle->vhandle=virtualdir_open(multisession_lastsession,
						    myhandle->path);

	       };
	  }
     };
   if ((!result)&&(myhandle->vhandle))
     {
	virtualdir_dir *entry=virtualdir_read(myhandle->vhandle);
	if (entry)
	  {
#ifdef DEBUG
	     printf("vfs_unixtree_readdirentry processing '%s'\n",entry->name);
#endif
	     /* skip directories. Those are already covered by "real" ones */
	     if (S_ISDIR(entry->mode))
	       {
#ifdef DEBUG
		  printf("vfs_unixtree_readdirentry skipping directory '%s'\n",entry->name);
#endif
		  result=vfs_unixtree_readdirentry(fs,handle);
	       }
	     else
	       result=helpings_fileinpath(myhandle->path,
					  entry->name);
	     if ((result)&&(!vfs_isreference(fs,result)))
	       {
#ifdef DEBUG
		  printf("vfs_unixtee_readdirentry skipping ms file overwritten by new session '%s'\n",entry->name);
#endif		  
		  result=vfs_unixtree_readdirentry(fs,handle);
	       };
	  }
	else
	  {
	     virtualdir_close(myhandle->vhandle);
	     myhandle->vhandle=NULL;
	  };
     };
#ifdef DEBUG
   printf("vfs_unixtree_readdirentry returning '%s'\n",result);
#endif   
   return result;
};

void vfs_unixtree_closedir(gpointer fs,gtoaster_handle_t handle)
{
   vfs_unixtree_dirhandle *myhandle=(vfs_unixtree_dirhandle*)handle;
   if (myhandle->handle)
     closedir(myhandle->handle);
   if (myhandle->vhandle)
     virtualdir_close(myhandle->vhandle);
   free(myhandle->path);
   free(myhandle);
};

vfs_direntry *vfs_unixtree_filestat(gpointer fs,const char *rpath,int getfilelinks)
{
   struct stat buf;
   vfs_direntry *result;
   TRANSLATE;
#ifdef DEBUG
   printf("vfs_unixtree_filestat: statting '%s'\n",path);
#endif
   result=(vfs_direntry*)(((ROOTDIR)
			   ?(!stat(path,&buf))
			   :(!lstat(path,&buf)))
			  ?malloc(sizeof(vfs_direntry))
			  :NULL);
   if (result)
     {
	char *rpc=helpings_rpathcomponent(path);
	strncpy(result->name,
		rpc,VFS_MAXFILENAMESIZE);
	free(rpc);
	result->name[VFS_MAXFILENAMESIZE]=0;
	result->isdir=S_ISDIR(buf.st_mode);
	result->islink=S_ISLNK(buf.st_mode);
	result->filesize=buf.st_size;
	result->filelinks=buf.st_nlink;
	/* Something's wrong here.
	 * Calculate correct value */
	if (result->isdir&&(result->filelinks<2))
	  result->filelinks=dirlow_subdirentries(path)+2;
	/* FIXME: precaching might actually make sense when
	 * files are taken from certain devices (like /dev/fd0) */
	result->precache=0;

	result->is_reference=0;

	/* just a precaution measure.
	 * Symbolic links to directories are not allowed within a
	 * Symlink tree */
	if ((ROOTDIR)&&(result->isdir))
	  {
	     lstat(path,&buf);
	     if (S_ISLNK(buf.st_mode))
	       {
#ifdef DEBUG
		  printf("WARNING! Found '%s',which is a symlink to a directory within a symlink tree\n",
			 path);
#endif
		  free(result);
		  result=NULL;
	       };
	  };
     }
   else
     {
	/* might as well try if there's a virtual one */
	if (multisession_lastsession)
	  {
	     virtualdir_dir *entry=virtualdir_finddir(multisession_lastsession,
						      rpath);
	     if (entry)
	       {
		  result=(vfs_direntry*)malloc(sizeof(vfs_direntry));
		  strcpy(result->name,entry->name);
		  result->filesize=entry->filesize;
		  result->isdir=S_ISDIR(entry->mode);
		  result->islink=S_ISLNK(entry->mode);
		  // normal file, there ain't no virtual directories
		  result->filelinks=1;
		  result->precache=0;
		  result->is_reference=1;
	       };
	  };
     };

   free(path);
   return result;
};

char *vfs_unixtree_getlinkdestination(gpointer fs,char *path)
{
   if (!ROOTDIR)
     {
	int result;
	char *buffer=(char*)malloc(PATH_MAX+1);
	if ((result=readlink(path,buffer,PATH_MAX))==-1)
	  {
	     free(buffer);
	     return NULL;
	  }
	else
	  {
	     buffer[result]=0;
	     return buffer;
	  };
     }
   else
     return NULL;
};

int vfs_unixtree_symlink(gpointer fs,char *linkdest,char *linkname)
{
   if (!ROOTDIR)
     return symlink(linkdest,linkname);
   else
     return 1;
};

typedef struct
{
   streamcopy_state *copyprocess;
   vfs_operationcomplete callback;
   gpointer callback_data;
}
vfs_unixtree_copyinfo;

void vfs_unixtree_streamcopy_donecb(gpointer state,
				    gpointer data)
{
   vfs_unixtree_copyinfo *info=(vfs_unixtree_copyinfo*)data;
   info->callback(info->copyprocess->fatalerror,info->callback_data);
   free(info);
};

int vfs_unixtree_dummycb(gpointer data)
{
   vfs_unixtree_copyinfo *info=(vfs_unixtree_copyinfo*)data;
   /* abuse the copyprocess field for status report here */
   info->callback((int)info->copyprocess,info->callback_data);
   free(info);
   return 0;
};

/* We cannot call the callback immediately because we would risk
 * display update failures this way */
void vfs_unixtree_queuecallback(vfs_operationcomplete callback,int result,gpointer data)
{
   vfs_unixtree_copyinfo *info=(vfs_unixtree_copyinfo*)malloc(sizeof(vfs_unixtree_copyinfo));

   info->callback=callback;
   info->callback_data=data;
   info->copyprocess=(streamcopy_state*)result;

   /* add a 20 ms timeout roughly every 8th file, so's the display
    * can be updated */
   gtk_timeout_add(((random()<(RAND_MAX>>3))?20:0),
		   vfs_unixtree_dummycb,info);
};

void vfs_unixtree_copy(gpointer fs,
		       char *rpath,
		       gpointer sourcefs,
		       char *sourcepath,
		       datacopydlg_dlginfo *progress,
		       vfs_operationcomplete callback,gpointer data,
		       int link_requested,int copy_move)
{
   /* Get informations about the source file */
   vfs_direntry *source=vfs_filestat((vfs_filesystem*)sourcefs,
				     sourcepath,0);
   TRANSLATE;
   if (source)
     {
	char *realname=vfs_getrealname(sourcefs,sourcepath);

	/* if we're supposed to remove the file afterwards and
	 * are not copying within the same fs,
	 * use the default file remover for it */
	if ((copy_move)&&(fs!=sourcefs))
	  vfs_install_autodelete(sourcefs,
				 sourcepath,
				 &callback,
				 &data);
	/* are we within the same fs ? */
	if (copy_move&&(fs==sourcefs)&&realname)
	  {
	     if (progress)
	       datacopydlg_newdatablock(progress,source->filesize);
	     vfs_unixtree_queuecallback(callback,
					rename(realname,path),
					data);
	  }
	else // not move within the same fs
	  {
	     char *linkdestination=NULL;
	     if (vfs_islink(sourcefs,sourcepath))
	       linkdestination=vfs_getlinkdestination(sourcefs,sourcepath);
	     // reproduce symlink if possible
	     if ((linkdestination)&&(!ROOTDIR))
	       {
		  if (progress)
		    datacopydlg_newdatablock(progress,source->filesize);
		  vfs_unixtree_queuecallback(callback,
					     vfs_symlink(fs,linkdestination,path),
					     data);
	       }
	     // is link and we're in unixtree mode
	     else
	       {
		  char *linkpath=helpings_pathonly(sourcepath);
	     /* Link might be relative. We have to deal with that */
		  char *reallinkdestination=
		    ((linkdestination)
		     ?((linkdestination[0]!='/')
		       ?helpings_fileinpath(linkpath,linkdestination)
		       :strdup(linkdestination))
		     :NULL);
		  free(linkpath);

		  if ((ROOTDIR)&&reallinkdestination&&(vfs_isdirectory(sourcefs,reallinkdestination)))
		    {
		       // sorry, can't do it. Report error
		       vfs_adderrorstring(fs,_("Can't add Symbolic link to a Symlinktree Filesystem"));
		       vfs_unixtree_queuecallback(callback,1,data);
		    }
		  else
		    {
		       if ((ROOTDIR)&&(link_requested))
			 {

			    if (progress)
			      datacopydlg_newdatablock(progress,source->filesize);
#ifdef DEBUG
			    printf("vfs_unixtree_copy: creating symlink in Symlink tree (%s -> %s)\n",
				   path,realname);
#endif

			    vfs_unixtree_queuecallback(callback,
						       symlink(realname,path),
						       data);
			 }
		       // we're in symlink mode and can get the real filename
		       else
			 {

	     /* we have to copy every file as linking into another filesystem
	      * is not supported by the unixtree (don't confuse normal
	      * symlinking with the kind of linking we're requested to do */
			    int fd=vfs_openfile((vfs_filesystem*)sourcefs,sourcepath);
			    int destfd=open(path,O_WRONLY|O_CREAT,0770);
			    if ((fd!=-1)&&(destfd!=-1))
			      {
				 vfs_unixtree_copyinfo *info=(vfs_unixtree_copyinfo*)malloc(sizeof(vfs_unixtree_copyinfo));
				 info->callback=callback;
				 info->callback_data=data;
				 info->copyprocess=streamcopy_create(destfd,
								     fd,
								     -1,
								     source->filesize,
								     progress,
								     STREAMCOPY_OPT_CUT|
								     STREAMCOPY_OPT_CLOSESOURCE|
								     STREAMCOPY_OPT_CLOSEDEST,
								     vfs_unixtree_streamcopy_donecb,
								     (gpointer)info);
			      }
			    // copying can be done
			    else
			      {

				 /* report error */
				 vfs_adderrorstring(fs,_("Read/Write error copying file"));

				 vfs_unixtree_queuecallback(callback,1,data);
				 if (fd!=-1)
				   close(fd);
				 if (destfd!=-1)
				   close(destfd);
			      }; // error copying files
			 }; // we're not in symlink mode so we had to copy
		    }; // we're either not in symlink mode or
		  // source is not a link
		  // or link doesn't aim at a directory
		  if (reallinkdestination)
		    free(reallinkdestination);
	       };
	     if (linkdestination)
	       free(linkdestination);
	     free(source);
	  };
	if (realname)
	  free(realname);
     }
   else
     /* report error */
     vfs_unixtree_queuecallback(callback,1,data);
   free(path);
};

char *vfs_unixtree_getrealname(gpointer fs,char *rpath)
{
   TRANSLATE;
   return path;
};

int vfs_unixtree_openfile(gpointer fs,char *rpath)
{
   int result;
   TRANSLATE;
   result=open(path,O_RDONLY);
   free(path);
   return result;
};

int vfs_unixtree_deletefile(gpointer fs,char *rpath)
{
   int result;
   TRANSLATE;
   result=remove(path);
   free(path);
   return result;
};

int vfs_unixtree_renamefile(gpointer fs,char *rpath,char *newname)
{
   char *pathonly;
   char *newpath;
   int result;
   TRANSLATE;
   pathonly=helpings_pathonly(path);
   newpath=helpings_fileinpath(pathonly,newname);
   result=rename(path,newpath);
   free(pathonly);
   free(newpath);
   free(path);
   return result;
};

void vfs_unixtree_destroy(gpointer fs)
{
   if (ROOTDIR)
     free(ROOTDIR);
};

/* create a vfs based on the standard unix tree
 *
 * This is pretty straight forward, the complicated part comes with all those
 * vfs_unixtree_* functions *g* */
vfs_filesystem *vfs_unixtree_create(char *root)
{
   vfs_filesystem *fs;

   fs=vfs_create(vfs_unixtree_createdir,
		 vfs_unixtree_opendir,
		 vfs_unixtree_readdirentry,
		 vfs_unixtree_closedir,
		 vfs_unixtree_filestat,
		 vfs_unixtree_getlinkdestination,
		 vfs_unixtree_symlink,
		 vfs_unixtree_copy,
		 // no separate implementation of createfile_getrealname
		 // in other words: it doesn't matter what the user's
		 // about to do with it :-)
		 vfs_unixtree_getrealname,
		 vfs_unixtree_getrealname,
		 vfs_unixtree_openfile,
		 vfs_unixtree_deletefile,
		 vfs_unixtree_renamefile,
		 vfs_unixtree_destroy,
		 (root
		  ?strdup(root)
		  :NULL));

   return fs;
};

void vfs_unixtree_setroot(vfs_filesystem *fs,const char *root)
{
   if (fs->data)
     free(fs->data);
   fs->data=((root)?strdup(root):NULL);
};
