/*
 *   Copyright (C) 2007 Tristan Heaven <tristanheaven@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License along
 *   with this program; if not, write to the Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <gtk/gtk.h>

#include "list.h"
#include "main.h"
#include "hash.h"
#include "gui.h"

#define COL_TOTAL (COL_HASH + HASH_N)

enum {
	COL_PATH,
	COL_BASENAME,
	COL_HASH
};

static struct {
	GtkTreeView *view;
	GtkTreeModel *model;
	GtkListStore *store;
} list;

void list_init(void)
{
	list.view = GTK_TREE_VIEW(gui_widget("treeview"));
	GType types[COL_TOTAL];
	GtkTreeViewColumn *col;

	gtk_tree_selection_set_mode(gtk_tree_view_get_selection(list.view),
		GTK_SELECTION_MULTIPLE);

	gtk_tree_view_insert_column_with_attributes(list.view, -1, _("Path"),
		gtk_cell_renderer_text_new(), "text", COL_PATH, NULL);
	// TODO: Make this configurable
	col = gtk_tree_view_get_column(list.view, COL_PATH);
	gtk_tree_view_column_set_visible(col, false);

	gtk_tree_view_insert_column_with_attributes(list.view, -1, _("Filename"),
		gtk_cell_renderer_text_new(), "text", COL_BASENAME, NULL);

	for (int i = 0; i < HASH_N; i++)
		gtk_tree_view_insert_column_with_attributes(list.view, -1, hash[i].name,
			gtk_cell_renderer_text_new(), "text", COL_HASH + i, NULL);

	for (int i = 0; i < COL_TOTAL; i++) {
		types[i] = G_TYPE_STRING;
		col = gtk_tree_view_get_column(list.view, i);
		gtk_tree_view_column_set_resizable(col, true);
	}

	list.store = gtk_list_store_newv(COL_TOTAL, types);
	list.model = GTK_TREE_MODEL(list.store);

	gtk_tree_view_set_model(list.view, list.model);
}

void list_update(void)
{
	GtkTreeViewColumn *col;
	bool active;

	for (int i = 0; i < HASH_N; i++) {
		col = gtk_tree_view_get_column(list.view, COL_HASH + i);
		active = gtk_toggle_button_get_active(hash[i].button);
		gtk_tree_view_column_set_visible(col, active);
	}

	list_clear_digests();
}

void list_append_row(const char *path)
{
	GtkTreeIter iter;

	if (!file_exists(path) || (list_find_row(path) >= 0))
		return;

	// New row
	gtk_list_store_append(list.store, &iter);

	// Set path
	gtk_list_store_set(list.store, &iter, COL_PATH, path, -1);

	// Set filename
	char *basename = g_path_get_basename(path);
	gtk_list_store_set(list.store, &iter, COL_BASENAME, basename, -1);
	g_free(basename);

	gtk_widget_set_sensitive(gui_widget("button_hash"), true);

	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
		gui_widget("radiomenuitem_file_list")), true);

	list_clear_digests();
}

void list_remove_selection(void)
{
	GtkTreeSelection *selection = gtk_tree_view_get_selection(list.view);
	GList *rows = gtk_tree_selection_get_selected_rows(selection, &list.model);

	for (GList *i = rows; i != NULL; i = i->next) {
		GtkTreePath *path = i->data;
		GtkTreeRowReference *ref = gtk_tree_row_reference_new(list.model, i->data);
		i->data = ref;
		gtk_tree_path_free(path);
	}

	for (GList *i = rows; i != NULL; i = i->next) {
		GtkTreeRowReference *ref = i->data;
		GtkTreePath *path = gtk_tree_row_reference_get_path(ref);
		GtkTreeIter iter;

		if (gtk_tree_model_get_iter(list.model, &iter, path))
			gtk_list_store_remove(list.store, &iter);

		gtk_tree_path_free(path);
		gtk_tree_row_reference_free(ref);
	}

	g_list_free(rows);
}

char *list_get_path(const int row)
{
	GtkTreeIter iter;
	char *path;
	GValue value;
	value.g_type = 0;

	if (!gtk_tree_model_get_iter_first(list.model, &iter))
		return NULL;

	for (int i = 0; i < row; i++)
		gtk_tree_model_iter_next(list.model, &iter);

	gtk_tree_model_get_value(list.model, &iter, COL_PATH, &value);
	path = g_strdup(g_value_get_string(&value));
	g_value_unset(&value);

	return path;
}

char *list_get_basename(const int row)
{
	char *path, *basename;

	path = list_get_path(row);
	basename = g_path_get_basename(path);
	g_free(path);

	return basename;
}

// Returns the row number containing path or -1 if not found
int list_find_row(const char *path)
{
	GtkTreeIter iter;

	if (gtk_tree_model_get_iter_first(list.model, &iter)) {
		int row = 0;
		do {
			GValue value;
			value.g_type = 0;
			gtk_tree_model_get_value(list.model, &iter, COL_PATH, &value);
			const char *string = g_value_get_string(&value);
			if (strcmp(string, path) == 0) {
				g_value_unset(&value);
				return row;
			} else {
				g_value_unset(&value);
				row++;
			}
		} while (gtk_tree_model_iter_next(list.model, &iter));
	}

	return -1;
}

unsigned int list_count_rows(void)
{
	GtkTreeIter iter;
	unsigned int count = 0;

	if (gtk_tree_model_get_iter_first(list.model, &iter)) {
		count++;
		while(gtk_tree_model_iter_next(list.model, &iter))
			count++;
	}

	return count;
}

void list_set_digest(const char *path, const int id, const char *digest)
{
	GtkTreeIter iter;
	int row = list_find_row(path);

	if (!gtk_tree_model_get_iter_first(list.model, &iter))
		return;

	for (int i = 0; i < row; i++)
		gtk_tree_model_iter_next(list.model, &iter);

	gtk_list_store_set(list.store, &iter, COL_HASH + id, digest, -1);
}

char *list_get_digest(const int row, const int id)
{
	GtkTreeIter iter;
	char *digest;
	GValue value;
	value.g_type = 0;

	if (!gtk_tree_model_get_iter_first(list.model, &iter))
		return NULL;

	for (int i = 0; i < row; i++)
		gtk_tree_model_iter_next(list.model, &iter);

	gtk_tree_model_get_value(list.model, &iter, COL_HASH + id, &value);
	digest = g_strdup(g_value_get_string(&value));
	g_value_unset(&value);

	return digest;
}

void list_clear_digests(void)
{
	GtkTreeIter iter;

	if (!gtk_tree_model_get_iter_first(list.model, &iter))
		return;

	do {
		for (int i = 0; i < HASH_N; i++)
			gtk_list_store_set(list.store, &iter, COL_HASH + i, "", -1);
	} while (gtk_tree_model_iter_next(list.model, &iter));
}

void list_clear(void)
{
	gtk_list_store_clear(list.store);
	gtk_widget_set_sensitive(gui_widget("button_hash"), false);
}
