/* 
** main.c: Starting up the program.
** copyright: (c) 2003 by László Pere
** email: pipas@linux.pte.hu
**
** This program is free software; you can redistribute it and/or 
** modify  it under the terms of the GNU General Public License as
** published by the Free Software Foundation; either version 2 of
** the License, or (at your option) any later version.
**
** $Id: main.c,v 1.1 2004/11/16 20:47:40 root Exp root $
** $Log: main.c,v $
** Revision 1.1  2004/11/16 20:47:40  root
** Initial revision
*/

#define _GNU_SOURCE
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <locale.h>
#include <getopt.h>
#include <gtk/gtk.h>
#include "main.h"
#include "variables.h"
#include "automaton.h"

char *program_src;
char *program_name;

char *includefile = NULL;
int source = PRG_UNKNOWN;
int charsreaded = 0;
FILE *sourcefile = NULL;
int debug = 0;

gboolean have_geometry_xy = FALSE;
gboolean have_geometry_dxdy = FALSE;
gint geometry_dx = 0;
gint geometry_dy = 0;
gint geometry_x = 0;
gint geometry_y = 0;
gint no_warning = FALSE;

static struct option const long_options[] = {
	{ "help", no_argument, NULL, 'h' },
	{ "version", no_argument, NULL, 'v' },
	{ "program", optional_argument, NULL, 'p' },
	{ "filename", required_argument, NULL, 'f' },
	{ "include", required_argument, NULL, 'i' },
	{ "debug", no_argument, NULL, 'd' },
	{ "stdin", no_argument, NULL, 's' },
	{ "no-warning", no_argument, NULL, 'w' },
	{ "geometry", required_argument, NULL, 'g'},
	{ NULL, 0, NULL, 0 }
};

int first_free_argument;


static gboolean get_geometry(const char *argument){
	g_assert(argument != NULL);

	if (sscanf(optarg, "%dx%d+%d+%d", &geometry_dx, &geometry_dy, 
				&geometry_x, &geometry_y) == 4){
		have_geometry_dxdy = TRUE;
		have_geometry_xy = TRUE;
		return TRUE;
	}
	
	if (sscanf(optarg, "%dx%d", &geometry_dx, &geometry_dy) == 2){
		have_geometry_dxdy = TRUE;
		return TRUE;
	}
	
	if (sscanf(optarg, "+%d+%d", &geometry_x, &geometry_y) == 2){
		have_geometry_xy = TRUE;
		return TRUE;
	}
	
	return FALSE;
}

void gtkdialog_init( int argc, char *argv[] )
{
	int c;
	while ((c=getopt_long_only(argc, argv, "whvf:i:dsg:p::", 
					long_options, NULL)) != -1 ) {
		switch (c) {
			case 'h':
				print_help_exit( EXIT_SUCCESS );
				break;
			case 'v':
				print_version_exit( EXIT_SUCCESS );
				break;
			case 'd':
				debug = 1;
				break;
			case 'i':
				includefile = optarg;
				break;
			case 's':	
				program_name = "STDIN";
				source = PRG_STDIN;
				break;
			case 'p':
				if (optarg != NULL){
					get_program_from_variable(optarg);
				}else{
					get_program_from_variable("MAIN_DIALOG");
				}
				break;
			case 'f':
				get_program_from_file(optarg);
				break;
			case 'g':
				get_geometry(optarg);
				break;
			case 'w':
				no_warning = TRUE;
				break;
			default:
				exit (EXIT_FAILURE);
		}
	}
	first_free_argument = optind;
}

int get_program_from_variable( char *name )
{
	program_name = name;

	g_assert(program_name != NULL);
	program_src = getenv(program_name);
	if (!program_src) {
		fprintf(stderr,
		"Could not find environment variable '%s'.\n",
			program_name);
		exit(EXIT_FAILURE);
	}
	source = PRG_MEMORY;
}
	
int get_program_from_file( char *name )
{
	int tmp;
	
	program_name = "MAIN_WINDOW";
	sourcefile = fopen(name, "r");
	if (!sourcefile) {
		fprintf(stderr, "Error: %m\n" );
		exit(EXIT_FAILURE);
	}
	/* FIXME: We read the first line of file and drop it. It is
	 * required because the #! is not included in the language.
	 */
	program_src = NULL;
	getline(&program_src, &tmp, sourcefile);
	free(program_src);
	program_src = NULL;
	source = PRG_FILE;
}

int print_help_exit( int exitcode )
{
	fprintf(stderr,
"Usage: gtdkdialog [options]...\n"
"Create dialog boxes and windows according to the given dialog description. \n" 
"\n"
"For more information try 'info gtkdialog'.                                 \n"
"\n"
"-h,  --help                        Print usage information and exit.       \n"
"-v,  --version                     Print version information and exit.     \n"
"-p[=VARNAME],  --program[=VARNAME] Get program from an environment variable\n"
"                                   (only the name of the variable).        \n"
" -s,  --stdin                      Read program from the standard input.   \n"
" -f filename                       Get program from the given file.        \n"
" -i <filename>                     Include file as a bash script           \n" 
" -d,  --debug                      Debug mode.                             \n"
" -w,  --no-warning                 Suppress warning messages.              \n"
);
	exit( exitcode );
}

int print_version_exit( int exitcode )
{	
	printf("%s version %s\n", PACKAGE_NAME, PACKAGE_VERSION);
	exit( exitcode );
}


int getnextchar(void)
{
	int c;
	if (source == PRG_MEMORY) {
		if (program_src == NULL
		    || charsreaded > strlen(program_src))
			return (EOF);
		c = (int) program_src[charsreaded];
		if (c == '\0')
			c = EOF;
		++charsreaded;
	}
	if (source == PRG_STDIN) {
		c = (int) getchar();
	}
	if (source == PRG_FILE) {
		c = (int) getc(sourcefile);
	}
	if (debug != 0)
		printf("%c", c);

	return (c);
}

int main(int argc, char *argv[])
{
#ifdef DEBUG
	fprintf(stderr, "%s: Starting.\n", __func__);
	fflush(stderr);
#endif
	setlocale(LC_ALL, "");
	gtk_init(&argc, &argv);
	gtkdialog_init(argc, argv);
	yyparse();
}
