// GtkSharp.Generation.CallbackGen.cs - The Callback Generatable.
//
// Author: Mike Kestner <mkestner@speakeasy.net>
//
// Copyright (c) 2002-2003 Mike Kestner
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public
// License as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.


namespace GtkSharp.Generation {

	using System;
	using System.IO;
	using System.Xml;

	public class CallbackGen : GenBase, IGeneratable  {

		private Parameters parms;
		private Signature sig = null;
		private ImportSignature isig = null;

		public CallbackGen (XmlElement ns, XmlElement elem) : base (ns, elem) 
		{
			if (elem ["parameters"] != null) {
				parms = new Parameters (elem ["parameters"], NS);
				parms.HideData = true;
			}
		}

		public string MarshalType {
			get
			{
				return NS + "Sharp." + Name + "Native";
			}
		}

		public string MarshalReturnType {
			get
			{
				return MarshalType;
			}
		}

		public string CallByName (string var_name)
		{
			return var_name + ".NativeDelegate";
		}

		public string FromNative(string var)
		{
			return var;
		}

		public string FromNativeReturn(string var)
		{
			return FromNative (var);
		}

		public virtual string ToNativeReturn(string var)
		{
			return CallByName (var);
		}
		
		public string GenWrapper (string ns, GenerationInfo gen_info)
		{
			string wrapper = Name + "Native";
			string qualname = ns + "Sharp." + wrapper;

			isig = new ImportSignature (parms, NS);

			StreamWriter sw = gen_info.OpenStream (qualname);

			sw.WriteLine ("namespace " + ns + "Sharp {");
			sw.WriteLine ();
			sw.WriteLine ("\tusing System;");
			sw.WriteLine ();

			sw.WriteLine ("#region Autogenerated code");

			SymbolTable table = SymbolTable.Table;

			XmlElement ret_elem = Elem["return-type"];
			string rettype = ret_elem.GetAttribute("type");
			string m_ret = table.GetMarshalReturnType (rettype);
			string s_ret = table.GetCSType (rettype);
			ClassBase ret_wrapper = table.GetClassGen (rettype);

			sw.WriteLine ("\tinternal delegate " + m_ret + " " + wrapper + "(" + isig.ToString() + ");");
			sw.WriteLine ();
			
			sw.WriteLine ("\tinternal class " + Name + "Wrapper : GLib.DelegateWrapper {");
			if (m_ret != "void") {
				if (table.IsEnum (rettype)) {
					sw.WriteLine ("\t\tstatic int _dummy;");
				} else if (ret_wrapper != null && (ret_wrapper is ObjectGen || ret_wrapper is OpaqueGen)) {
					// Do nothing
				} else if (!table.IsStruct (rettype) && !table.IsBoxed (rettype)) {
					sw.WriteLine ("\t\tstatic {0} _dummy;", m_ret);
				}
			}
			
			sw.WriteLine ();
			sw.WriteLine ("\t\tpublic " + m_ret + " NativeCallback (" + isig.ToString() + ")");
			sw.WriteLine ("\t\t{");

			int count = (parms != null) ? parms.Count : 0;
			int idx = 0;
			bool need_sep = false;
			string call_str = "";
			string cleanup_str = "";
			for (int i = 0; i < count; i++)
			{
				string parm_name = parms[i].Name;
				string ctype = parms[i].CType;

				if (i > 0 && parms[i].IsLength && parms[i-1].IsString)
					continue;

				if ((i == count - 1) && ctype == "gpointer" && (parm_name.EndsWith ("data") || parm_name.EndsWith ("data_or_owner"))) 
					continue;

				if (ctype == "GError**") {
					sw.WriteLine ("\t\t\t" + parm_name + " = IntPtr.Zero;");
					continue;
				}

				string cstype = parms[i].CSType;
				ClassBase parm_wrapper = table.GetClassGen (ctype);
				sw.Write("\t\t\t" + cstype + " _arg" + idx);
				if (parms[i].PassAs == "out") {
					sw.WriteLine(";");
					cleanup_str += "\t\t\t" + parm_name + " = " + table.CallByName (ctype, "_arg" + idx) + ";\n";
				} else
					sw.WriteLine(" = " + table.FromNative (ctype, parm_name) + ";");

				if (need_sep)
					call_str += ", ";
				else
					need_sep = true;
				call_str += String.Format ("{0} _arg{1}", parms[i].PassAs, idx);
				idx++;
			}

			sw.Write ("\t\t\t");
			string invoke = "_managed (" + call_str + ")";
			if (m_ret != "void") {
				if (cleanup_str == "")
					sw.Write ("return ");
				else {
					sw.Write (m_ret + " ret = ");
					cleanup_str += "\t\t\treturn ret;\n";
				}

				if (ret_wrapper != null && (ret_wrapper is ObjectGen || ret_wrapper is OpaqueGen))
					sw.WriteLine ("(({0}) {1}).Handle;", s_ret, invoke);
				else if (table.IsStruct (rettype) || table.IsBoxed (rettype)) {
					// Shoot. I have no idea what to do here.
					sw.WriteLine ("IntPtr.Zero;"); 
				} else if (table.IsEnum (rettype))
					sw.WriteLine ("(int) {0};", invoke);
				else
					sw.WriteLine ("({0}) {1};", m_ret, table.ToNativeReturn (rettype, invoke));
			} else
				sw.WriteLine (invoke + ";");

			if (cleanup_str != "")
				sw.Write (cleanup_str);
			sw.WriteLine ("\t\t}");
			sw.WriteLine ();

			sw.WriteLine ("\t\tinternal {0} NativeDelegate;", wrapper);
			sw.WriteLine ("\t\tprotected {0} _managed;", NS + "." + Name);
			sw.WriteLine ();

			sw.WriteLine ("\t\tpublic {0} ({1} managed, object o) : base (o)", Name + "Wrapper", NS + "." + Name);
			sw.WriteLine ("\t\t{");

			sw.WriteLine ("\t\t\tNativeDelegate = new {0} (NativeCallback);", wrapper);
			sw.WriteLine ("\t\t\t_managed = managed;");
			sw.WriteLine ("\t\t}");
			sw.WriteLine ("\t}");
			sw.WriteLine ("#endregion");
			sw.WriteLine ("}");
			sw.Close ();
			return ns + "Sharp." + Name + "Wrapper";
		}
		
		public void Generate ()
		{
			GenerationInfo gen_info = new GenerationInfo (NSElem);
			Generate (gen_info);
		}

		public void Generate (GenerationInfo gen_info)
		{
			XmlElement ret_elem = Elem["return-type"];
			if (ret_elem == null) {
				Console.WriteLine("No return type in callback " + CName);
				Statistics.ThrottledCount++;
				return;
			}

			SymbolTable table = SymbolTable.Table;

			string rettype = ret_elem.GetAttribute("type");
			string s_ret = table.GetCSType (rettype);
			if (s_ret == "") {
				Console.WriteLine("rettype: " + rettype + " in callback " + CName);
				Statistics.ThrottledCount++;
				return;
			}

			if ((parms != null) && !parms.Validate ()) {
				Console.WriteLine(" in callback " + CName + " **** Stubbing it out ****");
				Statistics.ThrottledCount++;
				parms = null;
			}

			sig = new Signature (parms);

			StreamWriter sw = gen_info.OpenStream (Name);

			sw.WriteLine ("namespace " + NS + " {");
			sw.WriteLine ();
			sw.WriteLine ("\tusing System;");
			sw.WriteLine ();
			sw.WriteLine ("\tpublic delegate " + s_ret + " " + Name + "(" + sig.ToString() + ");");
			sw.WriteLine ();
			sw.WriteLine ("}");

			sw.Close ();
			
			GenWrapper (NS, gen_info);

			Statistics.CBCount++;
		}
	}
}

