/*  gtk-imonc - imond client for fli4l
 *  timed calls
 *  Copyright (C) 2001-2003 Stefan Strigler <steve@zeank.in-berlin.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>

#include <sys/types.h>
#include <sys/wait.h>

#define _XOPEN_SOURCE /* glibc2 needs this */
#include <time.h>
#include <math.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "imonc.h"
#include "timer.h"
#include "news.h"
#include "docklet.h"

/* set titles of calls tab */
/* needed for csv export - makes it easier to get the headings */
void
init_calls (GtkCList * clist)
{
	gtk_clist_set_column_title (clist, 0, _("Date"));
	gtk_clist_set_column_title (clist, 1, _("Time"));
	gtk_clist_set_column_title (clist, 2, _("Calling Number"));
	gtk_clist_set_column_title (clist, 3, _("Called Number"));
}

/* set titles for connections tab */
/* needed for csv export - makes it easier to get the headings */
void
init_imondlog (GtkCList * clist)
{
	gtk_clist_set_column_title (clist, 0, _("Circuit"));
	gtk_clist_set_column_title (clist, 1, _("StartDate"));
	gtk_clist_set_column_title (clist, 2, _("StartTime"));
	gtk_clist_set_column_title (clist, 3, _("EndDate"));
	gtk_clist_set_column_title (clist, 4, _("EndTime"));
	gtk_clist_set_column_title (clist, 5, _("OnlTime"));
	gtk_clist_set_column_title (clist, 6, _("ChTime"));
	gtk_clist_set_column_title (clist, 7, _("Charge"));
	gtk_clist_set_column_title (clist, 8, _("IBytes"));
	gtk_clist_set_column_title (clist, 9, _("OBytes"));
	gtk_clist_set_column_title (clist, 10, _("Device"));
	gtk_clist_set_column_title (clist, 11, _("ChInt"));
	gtk_clist_set_column_title (clist, 12, _("Ch/Min"));
}

/*----------------------------------------------------------------------------
 * get timetable and fill tab
 *----------------------------------------------------------------------------
 */
void
get_timetable (GtkCList * timetable_clist, GtkCList * circuits_clist)
{
	GSList *list;
	gpointer data;
	gchar *timetable_row[26];
	gchar *circuits_row[7];
	guint i, j;
	GdkColor *gcolors[4];
	gchar *colors[4];
	GtkStyle *cstyle;

	/* setup colors */
	colors[0] = "lightyellow";
	colors[1] = "lightblue";
	colors[2] = "lightgreen";
	colors[3] = "red";

	for (i=0;i<4;i++) {
		gcolors[i] = (GdkColor *) malloc (sizeof(GdkColor));

		gdk_color_parse(colors[i],gcolors[i]);
		gdk_colormap_alloc_color(gtk_widget_get_colormap(GTK_WIDGET(timetable_clist)),gcolors[i],TRUE,TRUE);
	}
		
	/* get timetable from imond */
	list = get_list (fd, "timetable");

	/* clear clists */
	gtk_clist_clear(timetable_clist);
	gtk_clist_clear(circuits_clist);

	if (list == NULL)
		return;

	for (i = 2; i < 9; i++)
	{
		int row;
		data = g_slist_nth_data (list, i);

		timetable_row[0] = strtok (data, " ");
		for (j = 1; j < 25; j++)
			timetable_row[j] = strtok (NULL, " ");
		/* last col is empty */
		timetable_row[25] = "";
		row = gtk_clist_append (timetable_clist, timetable_row);

		/* color cells */
		for (j=1;j<25;j++) {
			int colnr = atoi(timetable_row[j])-1;
			if (colnr < 4) {
				cstyle = gtk_style_new();
				cstyle->base[GTK_STATE_NORMAL] = *gcolors[colnr];
				gtk_clist_set_cell_style(timetable_clist,row,j,cstyle);
			}
		}

		g_free (data);
	}


	list = g_slist_nth (list, 10);

	while ((list = g_slist_next (list)) != NULL)
	{
		int row;
		circuits_row[0] = strtok (list->data, " ");
		for (j = 1; j < 7; j++)
			circuits_row[j] = strtok (NULL, " ");

		row = gtk_clist_append (circuits_clist, circuits_row);
		g_free (list->data);

		/* color row */
		if (row<4) // only got 4 colors
			gtk_clist_set_background(circuits_clist, row, gcolors[row]);
			
	}
	g_slist_free (list);
}

/*----------------------------------------------------------------------------
 * get telmondlog file from imond and fill clist
 *----------------------------------------------------------------------------
 */
void
get_calls (GtkCList * calls_clist)
{
	GSList *list;
	gchar *row[4];
	gpointer tmp;

	gtk_clist_clear(calls_clist);

	list = get_list (fd, "telmond-log-file");

	if (list == NULL)
		return;

	gtk_clist_freeze(calls_clist);
	
	do
	{
		row[0] = strtok (list->data, " ");
		if (!(row[1] = strtok (NULL, " ")))
				continue;
		if (!(tmp = strtok (NULL, " ")))
				continue;
		if ((row[2] = g_hash_table_lookup (phonebook, tmp)) == NULL)
		{
			row[2] = tmp;
		}
		if (!(tmp = strtok (NULL, " ")))
				continue;
		if ((row[3] = g_hash_table_lookup (phonebook, tmp)) == NULL)
		{
			row[3] = tmp;
		}
		gtk_clist_prepend (calls_clist, row);
		g_free (list->data);
	}
	while ((list = g_slist_next (list)) != NULL);

	gtk_clist_thaw(calls_clist);
	
	g_slist_free (list);	/* free up memory */
}

/*----------------------------------------------------------------------------
 * get imondlog from imond and fill clist
 *----------------------------------------------------------------------------
 */
void
get_connections (GtkCList * imondlog_clist)
{
	GSList *list;
	gchar *row[IMONDLOG_CLIST_COLS];
	guint64 i, j, tmpbytes;
	gchar ibytebuf[64], obytebuf[64],
		chargebuf[64], onl_time_buf[64], ch_time_buf[64];

	GtkWidget *imondlog_onltime_label, *imondlog_ibytes_label,
		*imondlog_obytes_label, *imondlog_charge_label;

	imondlog_onltime_label =
		lookup_widget (GTK_WIDGET (imondlog_clist),
			       "imondlog_onltime_label");
	imondlog_ibytes_label =
		lookup_widget (GTK_WIDGET (imondlog_clist),
			       "imondlog_ibytes_label");
	imondlog_obytes_label =
		lookup_widget (GTK_WIDGET (imondlog_clist),
			       "imondlog_obytes_label");
	imondlog_charge_label =
		lookup_widget (GTK_WIDGET (imondlog_clist),
			       "imondlog_charge_label");

	gtk_clist_clear(imondlog_clist);

	list = get_list (fd, "imond-log-file");

	if (list == NULL)
		return;

	gtk_clist_freeze(imondlog_clist);
	
	j = 0;			/* count lines */
	do
	{
		if (j++ < con_rows)
		{		/* only prepend new lines */
			g_free (list->data);	/* free up used string elem */
			continue;
		}
		else
		{
			con_rows++;
		}
		row[0] = strtok (list->data, " ");
		for (i = 1; i < IMONDLOG_CLIST_COLS; i++)
		{
			switch (i)
			{	/* split it up here for special treatment */
			case 5:	/* OnlTime */
				strcpy (onl_time_buf, strtok (NULL, " "));
				onltime += atoi (onl_time_buf);
				row[i] = get_timestr (atoi (onl_time_buf),
						      onl_time_buf);
				break;
			case 6:	/* ChTime */
				row[i] = get_timestr (atoi
						      (strtok (NULL, " ")),
						      ch_time_buf);
				break;
			case 7:	/* Charge */
				strcpy (chargebuf, strtok (NULL, " "));
				row[i] = chargebuf;
				break;
			case 8:	/* IBytes */
				/* q'n'd hack to get imond>10 working */
				if (imond_versionnr > 10)
				{
					char in[128], out[128], buf[1024];
					char *inh = strtok (NULL, " ");
					char *inl = strtok (NULL, " ");
					char *outh = strtok (NULL, " ");
					char *outl = strtok (NULL, " ");

					/* reconstruct string ... >:-/ */
					snprintf (buf, 1024, "%s %s %s %s",
						  inh, inl, outh, outl);
					get_bytes_64 (buf, in, out);
					row[i++] = in;
					row[i] = out;
					ibytes +=
						atoi (inh) * (2 ^ 32) +
						atoi (inl);
					obytes +=
						atoi (outh) * (2 ^ 32) +
						atoi (outl);
				}
				else
				{
					tmpbytes = atoll (strtok (NULL, " "));
					ibytes += tmpbytes;
					row[i] = get_bytes (tmpbytes,
							    ibytebuf);
				}
				break;
			case 9:	/* OBytes */
				tmpbytes = atoll (strtok (NULL, " "));
				obytes += tmpbytes;
				row[i] = get_bytes (tmpbytes, obytebuf);
				break;
			default:
				row[i] = strtok (NULL, " ");
			}
		}
		/* add the row */
		gtk_clist_prepend (imondlog_clist, row);

		/* split up charge string 
		 * we must do this here cause strtok uses a static buffer
		 */
		charge +=
			((atoi (strtok (chargebuf, "."))) * 100) +
			(atoi (strtok (NULL, ".")));
		g_free (list->data);	/* free up used string elem */

	}
	while ((list = g_slist_next (list)) != NULL);

	gtk_clist_thaw(imondlog_clist);

	g_slist_free (list);	/* free up useless list */

	/* autosize the whole list */
	gtk_clist_columns_autosize (imondlog_clist);

	/* set labels of summaries */
	gtk_label_set_text (GTK_LABEL (imondlog_onltime_label),
			    get_timestr (onltime, onl_time_buf));

	gtk_label_set_text (GTK_LABEL (imondlog_ibytes_label),
			    get_bytes (ibytes, ibytebuf));

	gtk_label_set_text (GTK_LABEL (imondlog_obytes_label),
			    get_bytes (obytes, obytebuf));

	sprintf (chargebuf, "%d,%.2d", (charge / 100), (charge % 100));
	gtk_label_set_text (GTK_LABEL (imondlog_charge_label), chargebuf);

}				/* end get_connections */


/*----------------------------------------------------------------------------
 * initializes some settings of the interface 
 *----------------------------------------------------------------------------
 */
void
imonc_init (gpointer window)
{
	gint i;
	char buf[64];
	gchar imond_version[64];
	gchar *clist_row[ROW_LENGTH];
	gchar about[128];
	GtkWidget *dial_button, *hangup_button, *reboot_button,
		*halt_button, *remlink_button,
		*default_route_menu, *default_route_menu_menu,
		*dialmode, *new_menu_item, *clist,
		*timetable_clist, *circuits_clist, *calls_clist,
		*imondlog_clist, *remote_status_frame, *notebook,
		*about_label, *version_error, *gtk_imonc;
  GtkStyle *style;
	
	/* menu items */
	GtkWidget *dial_item, *hangup_item, *reboot_item, *halt_item, *remlink_item;

	gint imondlog_tab_nr;

	/* get gui elements */
	dial_button = lookup_widget (window, "dial_button");
	hangup_button = lookup_widget (window, "hangup_button");
	remlink_button = lookup_widget (window, "remlink_button");
	reboot_button = lookup_widget (window, "reboot_button");
	halt_button = lookup_widget (window, "halt_button");
	default_route_menu = lookup_widget (window, "default_route_menu");
	default_route_menu_menu = GTK_OPTION_MENU (default_route_menu)->menu;
	dialmode = lookup_widget (window, "dialmode");
	clist = lookup_widget (window, "channel_clist");
	timetable_clist = lookup_widget (window, "timetable_clist");
	circuits_clist = lookup_widget (window, "circuits_clist");
	calls_clist = lookup_widget (window, "calls_clist");
	imondlog_clist = lookup_widget (window, "imondlog_clist");
	notebook = lookup_widget (window, "notebook");
	about_label = lookup_widget (window, "about_label");
	remote_status_frame = lookup_widget (window, "remote_status_frame");
	gtk_imonc = lookup_widget (window, "gtk_imonc");
	traffic_info = lookup_widget (window, "traffic_info");

	/* menu items */
	dial_item = lookup_widget (rightclick_menu, "dial");
	hangup_item = lookup_widget (rightclick_menu, "hangup");
	reboot_item = lookup_widget (rightclick_menu, "reboot");
	halt_item = lookup_widget (rightclick_menu, "halt");
	remlink_item = lookup_widget (rightclick_menu, "rem_link");

	/* get size of traffic info */
	gdk_window_get_size (traffic_info->window, &traffic_info_width,
			     &traffic_info_height);

	/* setup data chunks for storing traffic info */
	for (i = 0; i < 6; i++)
	{
		chan_data[i] = createTrafficInfoData (traffic_info_width / 2);
		if (i >= 2)	/* isdn channel */
			traffic_info_set_max (chan_data[i], (64 / 8) * 1024);	/* 64kbit/s */
	}

	/* setup colors for traffic info */
	for (i = 0; i < 6; i++)
	{
		gint red, green, blue;
		GdkColor *color;

		color = (GdkColor *) malloc (sizeof (GdkColor));

		red = green = blue = 0;
		gc[i] = gdk_gc_new (traffic_info->window);

		switch (i)
		{
		case 0:
			red = 0;
			green = 255;
			blue = 0;
			break;
		case 1:
			red = 255;
			green = 255;
			blue = 0;
			break;
		case 2:
			red = 255;
			green = 0;
			blue = 255;
			break;
		case 3:
			red = 255;
			green = 255;
			blue = 255;
			break;
		case 4:
			red = 0;
			green = 0;
			blue = 255;
			break;
		case 5:
			red = 255;
			green = 0;
			blue = 0;
			break;
		}
		color->red = red * (65535 / 255);
		color->green = green * (65535 / 255);
		color->blue = blue * (65535 / 255);
		color->pixel = (gulong) (red * 65536 + green * 256 + blue);
		gdk_color_alloc (gtk_widget_get_colormap (traffic_info),
				 color);
		gdk_gc_set_foreground (gc[i], color);
	}

	gf = gdk_font_load
		("-b&h-lucida-medium-r-normal-*-10-*-*-*-p-*-iso8859-1");

		/* first, create a GC to draw on */
	gcb = gdk_gc_new (traffic_info->window);
	gcg = gdk_gc_new (traffic_info->window);

	/* the color we want to use */
	color_black = (GdkColor *) malloc (sizeof (GdkColor));
	color_green = (GdkColor *) malloc (sizeof (GdkColor));

	/* red, green, and blue are passed values, indicating the RGB triple
	 * of the color we want to draw. Note that the values of the RGB components
	 * within the GdkColor are taken from 0 to 65535, not 0 to 255.
	 */
	color_black->red = color_black->green = color_black->blue = 0;
	color_green->red = color_green->blue = 0;
	color_green->green = 120 * (65535 / 255);

	/* the pixel value indicates the index in the colormap of the color.
	 * it is simply a combination of the RGB values we set earlier
	 */
	color_black->pixel = (gulong) (0);
	color_green->pixel = (gulong) (256 * 120);

	/* However, the pixel valule is only truly valid on 24-bit (TrueColor)
	 * displays. Therefore, this call is required so that GDK and X can
	 * give us the closest color available in the colormap
	 */
	gdk_color_alloc (gtk_widget_get_colormap (traffic_info), color_black);
	gdk_color_alloc (gtk_widget_get_colormap (traffic_info), color_green);

	/* set the foreground to our color */
	gdk_gc_set_foreground (gcb, color_black);
	gdk_gc_set_foreground (gcg, color_green);


	/* set window title with connected router's name in it :) */
	sprintf (buf, "GTK-Imonc - %s", host_name);
	gtk_window_set_title (GTK_WINDOW (gtk_imonc), buf);

	/* init icons */
  style = gtk_widget_get_style( GTK_WIDGET(window) );
  gtk_widget_realize(window);
  online_icon =	gdk_pixmap_create_from_xpm (GTK_WIDGET(window)->window, &online_icon_mask, &style->bg[GTK_STATE_NORMAL], find_pixmap_file("online.png"));
  offline_icon = gdk_pixmap_create_from_xpm (GTK_WIDGET(window)->window, &offline_icon_mask, &style->bg[GTK_STATE_NORMAL], find_pixmap_file("offline.png"));

	/* get imond version */
	send_command (fd, "version");
	imond_versionnr = atoi (strtok (get_answer (fd), " "));
	strcpy (imond_version, strtok (NULL, " "));

	/* check version and abort if imond to old */
	if (imond_versionnr < 9)
	{
		g_print ("imond to old\n");
		version_error = create_version_error ();
		gtk_widget_show (version_error);
		return;
	}

	/*****
	 * interface adjustments based on different imond version
	 */
	if (imond_versionnr >= 11) {
		GtkWidget *checkbox_vbox, *syslinux_checkbutton;
		/* add _syslinux.cfg_ to remote update tab */
		
		checkbox_vbox = lookup_widget (window, "checkbox_vbox");

		syslinux_checkbutton = gtk_check_button_new_with_label (_("syslinux.cfg"));
		gtk_box_pack_end_defaults (GTK_BOX (checkbox_vbox), syslinux_checkbutton);
		gtk_widget_show (syslinux_checkbutton);
	}


	/*****
	 * dynamic data based on remote host
	 */

	/* get imond hostname - this might be different from connected host
	 * due to used tunnels or so */
	send_command (fd, "hostname");
	sprintf (buf, "[ %s - v%s ]", get_answer (fd), imond_version);

	gtk_frame_set_label (GTK_FRAME (remote_status_frame), buf);

	/* check which functions are available */
	send_command (fd, "is-allowed dial");
	if (!(atoi (get_answer (fd))))
	{
		gtk_widget_set_sensitive (dial_button, FALSE);
		gtk_widget_set_sensitive (hangup_button, FALSE);
		gtk_widget_set_sensitive (dial_item, FALSE);
		gtk_widget_set_sensitive (hangup_item, FALSE);
	}

	send_command (fd, "is-allowed reboot");
	if (!(atoi (get_answer (fd))))
	{
		gtk_widget_set_sensitive (reboot_button, FALSE);
		gtk_widget_set_sensitive (halt_button, FALSE);
		gtk_widget_set_sensitive (reboot_item, FALSE);
		gtk_widget_set_sensitive (halt_item, FALSE);
	}

	send_command (fd, "is-allowed route");
	if (!(atoi (get_answer (fd))))
		gtk_widget_set_sensitive (default_route_menu, FALSE);

	admintab_pos = 5;
	if (!is_admin)
	{
		gtk_widget_set_sensitive (dialmode, FALSE);
		/* hide remoteupdate */
		gtk_widget_hide (gtk_notebook_get_nth_page
				 (GTK_NOTEBOOK (notebook), admintab_pos));
	}

	/* hide rem link button/menu item */
	gtk_widget_hide (remlink_button);
	gtk_widget_hide (remlink_item);

	/* setting up menu for circuits (default route) */
	send_command (fd, "circuits");
	circuits = atoi (get_answer (fd));

	for (i = 1; i <= circuits; i++)
	{
		sprintf (buf, "circuit %i", i);
		send_command (fd, buf);

		new_menu_item =
			gtk_menu_item_new_with_label (get_answer (fd));
		gtk_widget_show (new_menu_item);
		gtk_menu_append (GTK_MENU (default_route_menu_menu),
				 new_menu_item);
	}

	/* clear list to be sure */
	gtk_clist_clear (GTK_CLIST (clist));

	/* make a clean row for later usage */
	for (i = 0; i < ROW_LENGTH; i++)
	{
		clist_row[i] = "";
	}

	send_command (fd, "pppoe");
	pppoe = atoi (get_answer (fd));

	if (pppoe)
	{
		/* add a row for pppoe */
		gtk_clist_append (GTK_CLIST (clist), clist_row);
	}

	send_command (fd, "channels");
	channels = atoi (get_answer (fd));
	for (i = 1; i <= channels; i++)
	{
		/* foresee a clist_row for every circuit */
		gtk_clist_append (GTK_CLIST (clist), clist_row);
	}

	gtk_signal_connect (GTK_OBJECT (default_route_menu_menu),
			    "deactivate",
			    GTK_SIGNAL_FUNC (on_default_route_selected),
			    NULL);

	/* setting up menu for dialmode */
	gtk_signal_connect (GTK_OBJECT (GTK_OPTION_MENU (dialmode)->menu),
			    "deactivate",
			    GTK_SIGNAL_FUNC (on_dialmode_selected), NULL);

	/* init timetable */
	get_timetable (GTK_CLIST (timetable_clist),
		       GTK_CLIST (circuits_clist));


	/* check if we should display the news tab */
	if (!use_news_tab)
    gtk_widget_hide (gtk_notebook_get_nth_page
                     (GTK_NOTEBOOK (notebook), 4));
    //		gtk_notebook_remove_page (GTK_NOTEBOOK (notebook), 4);
    //		admintab_pos--;
  else 
    gtk_widget_show (gtk_notebook_get_nth_page
                     (GTK_NOTEBOOK (notebook), 4));

	/* init calls */
	imondlog_tab_nr = 2;
	send_command (fd, "is-allowed telmond-log");
	if (atoi (get_answer (fd)))
	{
		init_calls (GTK_CLIST (calls_clist));
		get_calls (GTK_CLIST (calls_clist));
	}
	else
	{
		gtk_notebook_remove_page (GTK_NOTEBOOK (notebook), 1);
		imondlog_tab_nr = 1;
		admintab_pos--;
	}

	/* init connections overview */
	send_command (fd, "is-allowed imond-log");
	if (atoi (get_answer (fd)))
	{
		init_imondlog (GTK_CLIST (imondlog_clist));
		con_rows = onltime = obytes = ibytes = charge = 0;	/* init global values */
		gtk_clist_column_titles_passive (GTK_CLIST (imondlog_clist));
		get_connections (GTK_CLIST (imondlog_clist));
		imond_log = TRUE;
	}
	else
	{
		gtk_notebook_remove_page (GTK_NOTEBOOK (notebook),
					  imondlog_tab_nr);
		imond_log = FALSE;
		admintab_pos--;
	}

	/* insert version nr. into about label */
  gtk_label_set_justify(GTK_LABEL(about_label), GTK_JUSTIFY_CENTER); /* fixing a glade bug */
 	sprintf (about, "<big><b>GTK-Imonc</b></big>\nVersion %s\n\n© 2001-2004 by Stefan Strigler &lt;steve@zeank.in-berlin.de&gt;",VERSION);
  gtk_label_set_markup (GTK_LABEL (about_label), about);


	/* make sure status is shown at startup */
	show_status = TRUE;
	show_status_old = TRUE;
	init_show_status = TRUE;

	/* set news init */
	news_init = 1;

	/* init tray icon */
	if (use_trayicon)
		docklet_init();

}				/* imonc_init (gpointer window) */

/*---------------------------------------------------------------------------- 
 * this is THE timer function called every TIMEOUT_MSEC msec
 * gets data from router and updates interface accordingly
 *----------------------------------------------------------------------------
 */
gint
imonc_update (gpointer window)
{
	guint i, in, out, was_traffic;
	gchar buf[64];
	gfloat cpu_usage;
	gchar *answer;
	gchar answers[5][8192];
	extern EggTrayIcon *docklet;

	GtkWidget *dial_button, *hangup_button,
		*addlink_button, *remlink_button,
		*admin_mode_button, *calls_reset_button,
		*imondlog_reset_button, *imondlog_clist,
		*config_circuits_button, *news_text, *remote_cpu_progressbar,
		*support_button, *lastcall_label;

	GtkWidget *dial_menuitem, *hangup_menuitem, 
		*addlink_menuitem, *remlink_menuitem;

	/* check for exited child processes */
	waitpid(0,NULL,WNOHANG);
	
	/* get ui elements */
	dial_button = lookup_widget (window, "dial_button");
	hangup_button = lookup_widget (window, "hangup_button");
	addlink_button = lookup_widget (window, "addlink_button");
	remlink_button = lookup_widget (window, "remlink_button");
	admin_mode_button = lookup_widget (rightclick_menu, "admin_mode_item");
	calls_reset_button = lookup_widget (window, "calls_reset_button");
	imondlog_reset_button =
		lookup_widget (window, "imondlog_reset_button");
	imondlog_clist = lookup_widget (window, "imondlog_clist");
	config_circuits_button =
		lookup_widget (window, "config_circuits_button");
	news_text = lookup_widget (window, "news_text");
	support_button = lookup_widget (window, "support_button");
	remote_cpu_progressbar =
		lookup_widget (window, "remote_cpu_progressbar");
	lastcall_label = lookup_widget (window, "lastcall_label");

	/* menu items */
	dial_menuitem = lookup_widget (rightclick_menu, "dial");
	hangup_menuitem = lookup_widget (rightclick_menu, "hangup");
	addlink_menuitem = lookup_widget(rightclick_menu, "add_link");
	remlink_menuitem = lookup_widget(rightclick_menu, "rem_link");

	/* get online-status */
	is_online = FALSE;

	if (pppoe)
	{			/* DSL */
		send_command (fd, "status pppoe");
		if (strcmp (get_answer (fd), "Online") == 0)
		{
			/* we are online */
			chan_online[0] = TRUE;
			send_command (fd, "rate pppoe");

      in = atoi (strtok (get_answer (fd), " "));
      out = atoi (strtok (NULL, " "));

      was_traffic = in + out;

			traffic_info_insert (chan_data[0],in);
			traffic_info_insert (chan_data[1],out);

			is_online = TRUE;	/* remember online-status */
		}
		else
		{
			chan_online[0] = FALSE;
		}
	}

	for (i = 1; i <= channels; i++)
	{			/* ISDN */
		sprintf (buf, "status %i", i);
		send_command (fd, buf);
		if (strcmp (get_answer (fd), "Online") == 0)
		{
			chan_online[i] = TRUE;

			sprintf (buf, "rate %i", i);
			send_command (fd, buf);

      in = atoi (strtok (get_answer (fd), " "));
      out = atoi (strtok (NULL, " "));

      was_traffic = in + out;

			traffic_info_insert (chan_data[(i * 2)],in);
			traffic_info_insert (chan_data[(i * 2) + 1],out);

			is_online = TRUE;	/* remember state */
		}
		else
		{
			chan_online[i] = FALSE;
		}
	}

	/* update traffic info and cpu */
	if (!init_show_status && (show_status || is_mini))
	{
		send_command (fd, "cpu");

    answer = get_answer(fd);
    sprintf(buf,"%s %%", answer);

    gtk_progress_bar_set_text(GTK_PROGRESS_BAR(remote_cpu_progressbar),buf);

		cpu_usage = (gfloat) (atof(answer) / 100.0);
		gtk_progress_bar_update (GTK_PROGRESS_BAR
					 (remote_cpu_progressbar), cpu_usage);
		sprintf (buf, "CPU: %.0f%%", (cpu_usage * 100));
		gtk_window_set_title (GTK_WINDOW (gtk_miniimonc), buf);

		traffic_info_update (window);
	}

	if (show_status)	/* true if status page is visible */
		status_display_update (window);

	/* set dial|hangup button */
	if (is_online)
	{
		gtk_widget_hide (dial_button);
		gtk_widget_show (hangup_button);

		gtk_widget_hide (dial_menuitem);
		gtk_widget_show (hangup_menuitem);

		was_online = TRUE;	/* remember again */
		if (news_init && use_news_tab)
		{
			get_news (GTK_TEXT_VIEW (news_text));
			news_init = 0;
		}

		if (docklet != NULL) {
      if (was_traffic)
        docklet_update_icon(online);
      else
        docklet_update_icon(inactive);
    }
	}
	else
	{
		gtk_widget_hide (hangup_button);
		gtk_widget_show (dial_button);

		gtk_widget_show (dial_menuitem);
		gtk_widget_hide (hangup_menuitem);

		if (docklet != NULL)
			docklet_update_icon(offline);

		if (was_online)
		{		/* we just hang up */
			int i;
			was_online = FALSE;

			/* update connections overview */
			if (imond_log)
				get_connections (GTK_CLIST (imondlog_clist));

			/* clear traffic info */
			for (i = 0; i < 6; i++)
				traffic_info_clear (chan_data[i]);
		}
	}

	/* set admin-mode button */
	send_command (fd, "pass");
	if ((atoi (get_answer (fd))) >= 4)
	{
		gtk_widget_set_sensitive (admin_mode_button, FALSE);
		gtk_widget_set_sensitive (calls_reset_button, TRUE);
		gtk_widget_set_sensitive (imondlog_reset_button, TRUE);
		gtk_widget_set_sensitive (config_circuits_button, TRUE);
		gtk_widget_set_sensitive (support_button, TRUE);

		/* set channel-bundling stuff */
		send_command (fd, "links 0");

		switch (atoi (get_answer (fd)))
		{
		case 1:
			gtk_widget_set_sensitive (addlink_button, TRUE);
			gtk_widget_hide (remlink_button);
			gtk_widget_show (addlink_button);
			gtk_widget_set_sensitive (addlink_menuitem, TRUE);
			gtk_widget_hide (remlink_menuitem);
			gtk_widget_show (addlink_menuitem);
			break;
		case 2:
			gtk_widget_set_sensitive (remlink_button, TRUE);
			gtk_widget_hide (addlink_button);
			gtk_widget_show (remlink_button);
			gtk_widget_set_sensitive (remlink_menuitem, TRUE);
			gtk_widget_hide (addlink_menuitem);
			gtk_widget_show (remlink_menuitem);
			break;
		default:
			gtk_widget_set_sensitive (addlink_button, FALSE);
			gtk_widget_set_sensitive (remlink_button, FALSE);
			gtk_widget_set_sensitive (addlink_menuitem, FALSE);
			gtk_widget_set_sensitive (remlink_menuitem, FALSE);
		}

	}
	else
	{
		gtk_widget_set_sensitive (addlink_button, FALSE);
		gtk_widget_set_sensitive (addlink_menuitem, FALSE);
		gtk_widget_set_sensitive (calls_reset_button, FALSE);
		gtk_widget_set_sensitive (imondlog_reset_button, FALSE);
		gtk_widget_set_sensitive (config_circuits_button, FALSE);
	}

	/* 
	 * get last incoming call from telmond on (hopefully) fixed port
	 * 5001 
	 */
	if ((answer = get_lastcall (TELMOND_PORT)))
	{
		gchar *tok, *tmp;
		strcpy (answers[0], strtok (answer, " "));	/* date */
		strcpy (answers[1], strtok (NULL, " "));	/* time */
		tok = strtok (NULL, " ");
		if ((tmp = g_hash_table_lookup (phonebook, tok)) == NULL) {
			tmp = tok;
		}
		strcpy (answers[2], tmp);	/* caller nr. */
		tok = strtok (NULL, " ");
		if ((tmp = g_hash_table_lookup (phonebook, tok)) == NULL) {
			tmp = tok;
		}
		strcpy (answers[3], tmp);	/* callee nr. */
		sprintf (answers[4], "%s -> %s (%s %s)", answers[2],
			 answers[3], answers[0], answers[1]);
		gtk_label_set_text (GTK_LABEL (lastcall_label), answers[4]);

		if (new_call && telmond_popup) {
			if (call_notification) {
				pid_t child;
				if (!(child = fork())) {
					extern char **environ;
					snprintf(buf,sizeof(buf),"%s %s", answers[0], answers[1]);
					setenv("TELMOND_CALL_DATE",buf,1);
					setenv("TELMOND_CALLER",answers[2],1);
					setenv("TELMOND_CALLEE",answers[3],1);
					
					if(!execle(call_notification,call_notification,NULL,environ))
						perror("execlp");
				}

			} else {
				GtkWidget *new_call_dialog, *new_call_date_label,
					*new_call_time_label, *new_call_caller_label,
					*new_call_callee_label;
				
				new_call_dialog = create_new_call_dialog ();
				
				new_call_date_label =
					lookup_widget (new_call_dialog,
												 "new_call_date_label");
				new_call_time_label =
					lookup_widget (new_call_dialog,
												 "new_call_time_label");
				new_call_caller_label =
					lookup_widget (new_call_dialog,
												 "new_call_caller_label");
				new_call_callee_label =
					lookup_widget (new_call_dialog,
												 "new_call_callee_label");
				
				gtk_label_set_text (GTK_LABEL (new_call_date_label),
														answers[0]);
				gtk_label_set_text (GTK_LABEL (new_call_time_label),
														answers[1]);
				gtk_label_set_text (GTK_LABEL (new_call_caller_label),
														answers[2]);
				gtk_label_set_text (GTK_LABEL (new_call_callee_label),
														answers[3]);
				gtk_widget_show (new_call_dialog);
			}
		}
	}
	else
	{
		gtk_label_set_text (GTK_LABEL (lastcall_label), _("none"));
	}


	return 1;
}				/* imonc_update (gpointer window) */

/*----------------------------------------------------------------------------
 * makes update to values set by imonc_init
 * called when user switches to admin-mode
 *----------------------------------------------------------------------------
 */
void
imonc_auth_update (gpointer window)
{
	GtkWidget *dial_button,
		*hangup_button,
		*reboot_button,
		*halt_button,
		*default_route_menu, *support_button, *notebook, *dialmode;
	/* menu items */
	GtkWidget *dial_item, *hangup_item, *reboot_item, *halt_item;

	dial_button = lookup_widget (window, "dial_button");
	hangup_button = lookup_widget (window, "hangup_button");
	reboot_button = lookup_widget (window, "reboot_button");
	halt_button = lookup_widget (window, "halt_button");
	default_route_menu = lookup_widget (window, "default_route_menu");
	support_button = lookup_widget (window, "support_button");
	notebook = lookup_widget (window, "notebook");
	dialmode = lookup_widget (window, "dialmode");

	/* menu items */
	dial_item = lookup_widget (rightclick_menu, "dial");
	hangup_item = lookup_widget (rightclick_menu, "hangup");
	reboot_item = lookup_widget (rightclick_menu, "reboot");
	halt_item = lookup_widget (rightclick_menu, "halt");

	/* check which functions are available */
	send_command (fd, "is-allowed dial");
	if (atoi (get_answer (fd)))
	{
		gtk_widget_set_sensitive (dial_button, TRUE);
		gtk_widget_set_sensitive (hangup_button, TRUE);
		gtk_widget_set_sensitive (dial_item, TRUE);
		gtk_widget_set_sensitive (hangup_item, TRUE);
	}

	send_command (fd, "is-allowed reboot");
	if (atoi (get_answer (fd)))
	{
		gtk_widget_set_sensitive (reboot_button, TRUE);
		gtk_widget_set_sensitive (halt_button, TRUE);
		gtk_widget_set_sensitive (reboot_item, TRUE);
		gtk_widget_set_sensitive (halt_item, TRUE);
	}

	send_command (fd, "is-allowed route");
	if (atoi (get_answer (fd)))
		gtk_widget_set_sensitive (default_route_menu, TRUE);

	gtk_widget_set_sensitive (support_button, TRUE);
	gtk_widget_set_sensitive (dialmode, TRUE);
	gtk_widget_show (gtk_notebook_get_nth_page
			 (GTK_NOTEBOOK (notebook), admintab_pos));

}				/* imonc_auth_update (gpointer window) */


/*----------------------------------------------------------------------------
 * updates items on "status" notebook page
 *----------------------------------------------------------------------------
 */
void
status_display_update (gpointer window)
{
	guint i, j;
	gchar buf[64];
  static struct tm tm;

	GtkWidget *clist,
		*default_route_menu, *dialmode, *remote_date, *remote_uptime;

	/* get ui elements */
	clist = lookup_widget (window, "channel_clist");
	default_route_menu = lookup_widget (window, "default_route_menu");
	dialmode = lookup_widget (window, "dialmode");
	remote_date = lookup_widget (window, "remote_date");
	remote_uptime = lookup_widget (window, "remote_uptime");

	/* updating remote machine status */
	send_command (fd, "date");	/* get date */
  strptime(get_answer(fd)+3, "%d/%m/%Y %T", &tm);
  strftime(buf, sizeof(buf), "%c", &tm);

	gtk_label_set_text (GTK_LABEL (remote_date), g_convert(buf,strlen(buf),"UTF-8","ISO-8859-1",NULL,NULL,NULL));

	send_command (fd, "uptime");
	gtk_label_set_text (GTK_LABEL (remote_uptime),
			    get_datestr (atoi (get_answer (fd)), buf));

	i = j = 0;

	/* set dialmode */
	send_command (fd, "dialmode");
	strcpy (buf, get_answer (fd));
	if (strcmp (buf, "auto") == 0)
		i = 0;
	else if (strcmp (buf, "manual") == 0)
		i = 1;
	else if (strcmp (buf, "off") == 0)
		i = 2;
	gtk_option_menu_set_history (GTK_OPTION_MENU (dialmode), i);

	/* set default route menu */
	send_command (fd, "route");
	gtk_option_menu_set_history (GTK_OPTION_MENU (default_route_menu),
				     atoi (get_answer (fd)));

	/* updating clist-window */
	if (!is_online && !was_online && !init_show_status)
		return;

	/* freeze clists */
	gtk_clist_freeze (GTK_CLIST (clist));

	/* update status display for DSL (if DSL) */
	if (pppoe)
	{
		if (chan_online[0])
		{
			/* we are online */
      send_command (fd, "phone pppoe");
 			gtk_clist_set_pixtext (GTK_CLIST (clist), j, 0, get_answer (fd), 0, online_icon, online_icon_mask); 

      send_command (fd, "inout pppoe");
      gtk_clist_set_text (GTK_CLIST (clist), j, 1, get_answer (fd));
		}
		else
		{
			/* we are offline - don't show state information */
 			gtk_clist_set_pixmap (GTK_CLIST (clist), j, 0, offline_icon, offline_icon_mask);
      /* FIX: direction wrong in imond */
      gtk_clist_set_text (GTK_CLIST (clist), j, 1, "");
		}

		send_command (fd, "ip pppoe");
		gtk_clist_set_text (GTK_CLIST (clist), j, 2, get_answer (fd));

		send_command (fd, "quantity pppoe");
		if (imond_versionnr > 10)
		{
			char in[128], out[128];
			get_bytes_64 (get_answer (fd), in, out);
			gtk_clist_set_text (GTK_CLIST (clist), j, 3, in);
			gtk_clist_set_text (GTK_CLIST (clist), j, 4, out);
		}
		else
		{
			gtk_clist_set_text (GTK_CLIST (clist), j, 3, get_bytes(atoll(strtok(get_answer (fd)," ")), buf));
			gtk_clist_set_text (GTK_CLIST (clist), j, 4, get_bytes(atoll(strtok(NULL, " ")), buf));
		}

		send_command (fd, "online-time pppoe");
		gtk_clist_set_text (GTK_CLIST (clist), j, 5, get_answer (fd));

		send_command (fd, "time pppoe");
		gtk_clist_set_text (GTK_CLIST (clist), j, 6, get_answer (fd));

		send_command (fd, "chargetime pppoe");
		gtk_clist_set_text (GTK_CLIST (clist), j, 7, get_answer (fd));

		send_command (fd, "charge pppoe");
		gtk_clist_set_text (GTK_CLIST (clist), j, 8, get_answer (fd));

		j++;		/* remember that we have a dsl-circuit */
	}

	/* update status display for ISDN channels */
	for (i = 1; i <= channels; i++)
	{

		/* set online-status icons and connection name */
		if (chan_online[i])
		{		/* we are online on this channel */
			/* get connection name */
			sprintf (buf, "phone %i", i);
      send_command (fd, buf);
 			gtk_clist_set_pixtext (GTK_CLIST (clist),(j + i - 1),0,get_answer (fd),5,online_icon,online_icon_mask); 
    }
		else
		{
 			gtk_clist_set_pixmap (GTK_CLIST (clist),(j + i - 1),0, offline_icon,offline_icon_mask);
		}

		sprintf (buf, "inout %i", i);
		send_command (fd, buf);
		gtk_clist_set_text (GTK_CLIST (clist), (j + i - 1), 1, get_answer (fd));

		sprintf (buf, "ip %i", i);
		send_command (fd, buf);
		gtk_clist_set_text (GTK_CLIST (clist), (j + i - 1), 2, get_answer (fd));

		sprintf (buf, "quantity %i", i);
		send_command (fd, buf);
		if (imond_versionnr > 10)
		{
			char in[128], out[128];
			get_bytes_64 (get_answer (fd), in, out);
			gtk_clist_set_text (GTK_CLIST (clist), (j + i - 1), 3, in);
			gtk_clist_set_text (GTK_CLIST (clist), (j + i - 1), 4, out);
		}
		else
		{
			gtk_clist_set_text (GTK_CLIST (clist), (j + i - 1), 3, get_bytes(atoi(strtok(get_answer(fd)," ")), buf));
			gtk_clist_set_text (GTK_CLIST (clist), (j + i - 1), 4, get_bytes(atoi(strtok (NULL, " ")),buf));
		}

		sprintf (buf, "online-time %i", i);
		send_command (fd, buf);
		gtk_clist_set_text (GTK_CLIST (clist), (j + i - 1), 5, get_answer (fd));

		sprintf (buf, "time %i", i);
		send_command (fd, buf);
		gtk_clist_set_text (GTK_CLIST (clist), (j + i - 1), 6, get_answer (fd));

		sprintf (buf, "chargetime %i", i);
		send_command (fd, buf);
		gtk_clist_set_text (GTK_CLIST (clist), (j + i - 1), 7, get_answer (fd));

		sprintf (buf, "charge %i", i);
		send_command (fd, buf);
		gtk_clist_set_text (GTK_CLIST (clist), (j + i - 1), 8, get_answer (fd));

	}

	/* release clists */
	gtk_clist_thaw (GTK_CLIST (clist));

	/* done initialization */
	if (init_show_status)
		init_show_status = FALSE;

}				/* status_display_update (gpointer window) */


/*----------------------------------------------------------------------------
 * tries to establish a new connection up to RECON_TRIES times
 *----------------------------------------------------------------------------
 */
gint
try_reconnect() {
	gchar buf[64];
  gint tries = 0;

	while ((fd = service_connect (host_name, port)) < 0 && tries++ < RECON_TRIES) {
		g_print ("Reconnect failed. Trying again ...\n");
		sleep(RECON_TIMEOUT);
	}
	
  if (fd<0) // nothing happend :(
    return 0;

	g_print ("Reconnected to %s\n", host_name);

	/* 
	 * not doing init - gives us two tray icons and 
	 * shouldn't be necessary anyway 
	 */
	/* imonc_init(gtk_imonc); */
	
	/* make auth to imond */
	if (strlen(pass)) {
		sprintf (buf, "pass %s", pass);
	 	send_command (fd, buf);
		get_answer(fd);
	}
	if (admin_passwd != NULL) {
		sprintf (buf, "pass %s", admin_passwd);
  	send_command (fd, buf);
		get_answer(fd);
	}
  imonc_auth_update (gtk_imonc);
	
	timeout_tag = gtk_timeout_add (TIMEOUT_MSEC, imonc_update, gtk_imonc);

  return 1;
}
/*----------------------------------------------------------------------------
 * called if connection was lost
 *----------------------------------------------------------------------------
 */
void
fli4l_reconnect (int signo)
{

	if (reconnect_dialog == NULL)
	{
		gtk_timeout_remove (timeout_tag);
		usleep(500000); // wait for timer to terminate
		g_print ("Connection to %s lost.\n", host_name);
		service_disconnect (fd);

    if (try_reconnect()) /* actually try to connect again */
      return;

    /* seems as if it didn't work - tell user that we're in trouble */
		reconnect_dialog = create_reconnect_dialog ();
		gtk_object_set_data (GTK_OBJECT (reconnect_dialog),
				     "gtk_imonc", gtk_imonc);
		gtk_widget_show (reconnect_dialog);
	}
}

/*----------------------------------------------------------------------------
 * some helper functions
 *----------------------------------------------------------------------------
 */

/* converts int representing bytes into a nice formated string */
void
get_bytes_64 (gchar * reply, gchar * in, gchar * out)
{
	guint64 outl, outh, inl, inh, num;
	gchar buf[128];

	outl = outh = inl = inh = 0;
	num = sscanf (reply, "%lu %lu %lu %lu", &inh, &inl, &outh, &outl);
	if (num != 4)
	{
		g_print("error while converting %s\n", reply);
		return;
	}
	if (outh && outh != (unsigned long int) pow(2.0,32.0)-1)
	{
		snprintf (out, 64,"%luG+%s", (4 * outh), get_bytes (outl, buf));
	}
	else
	{
		get_bytes (outl, out);
	}
	if (inh && inh != (unsigned long int) pow(2.0,32.0)-1)
	{
		snprintf (in, 64, "%luG+%s", (4 * inh), get_bytes (inl, buf));
	}
	else
	{
		get_bytes (inl, in);
	}
}


gchar *
get_bytes (guint64 bytes, gchar * buf)
{
	gfloat flobytes;

	if (bytes >= (1024 * 1024 * 1024))
	{
		flobytes = ((bytes / 1024.0) / 1024.0) / 1024.0;
		snprintf (buf, 64,"%.2fG", flobytes);
		return buf;
	}

	if (bytes >= (1024 * 1024))
	{
		flobytes = (bytes / 1024.0) / 1024.0;
		snprintf (buf, 64,"%.2fM", flobytes);
		return buf;
	}

	if (bytes >= 1024)
	{
		flobytes = bytes / 1024.0;
		snprintf (buf, 64,"%.2fK", flobytes);
		return buf;
	}

	snprintf (buf, 64, "%lu", (gulong) bytes);
	return buf;

}				/* get_bytes (bytes, *buf) */

/* converts int representing seconds into a string in the format HH:MM:SS */
gchar *
get_timestr (gint time, gchar * buf)
{
	gint hour, min, sec;

	hour = time / 3600;
	min = (time - hour * 3600) / 60;
	sec = time - hour * 3600 - min * 60;

	if (hour < 10)
		sprintf (buf, "0%d", hour);
	else
		sprintf (buf, "%d", hour);

	if (min < 10)
		sprintf (buf, "%s:0%d", buf, min);
	else
		sprintf (buf, "%s:%2d", buf, min);

	if (sec < 10)
		sprintf (buf, "%s:0%d", buf, sec);
	else
		sprintf (buf, "%s:%2d", buf, sec);

	return buf;
}				/* get_timestr (time, *buf) */

/* converts int representing seconds into a string in the format "days HH:MM:SS" */
gchar *
get_datestr (gint time, gchar * buf)
{
	gint days, hour, min, sec;

	days = time / (3600 * 24);
	hour = (time - days * 3600 * 24) / 3600;
	min = (time - days * 3600 * 24 - hour * 3600) / 60;
	sec = time - days * 3600 * 24 - hour * 3600 - min * 60;

	if (days > 0)
		sprintf (buf, _("%d days %d"), days, hour);
	else
		sprintf (buf, "%d", hour);

	if (min < 10)
		sprintf (buf, "%s:0%d", buf, min);
	else
		sprintf (buf, "%s:%2d", buf, min);

	if (sec < 10)
		sprintf (buf, "%s:0%d", buf, sec);
	else
		sprintf (buf, "%s:%2d", buf, sec);

	return buf;
}				/* get_datestr (time, *buf) */


/*----------------------------------------------------------------------------
 * dumps clist as csv to file
 *----------------------------------------------------------------------------
 */
void
csv_dump_clist (GtkCList * clist, FILE * fd)
{
	gint i, j;

	for (i = 0; i < clist->columns; i++)
	{
		if (i > 0)
			fprintf (fd, ",");
		fprintf (fd, "\"%s\"", gtk_clist_get_column_title (clist, i));
	}

	fprintf (fd, "\n");

	for (i = 0; i < clist->rows; i++)
	{
		for (j = 0; j < clist->columns; j++)
		{
			gchar *cellstr;
			if (j > 0)
				fprintf (fd, ",");
			gtk_clist_get_text (clist, i, j, &cellstr);
			fprintf (fd, "\"%s\"", cellstr);
		}
		fprintf (fd, "\n");
	}
}


/*----------------------------------------------------------------------------
 * draws the traffic info graph
 *----------------------------------------------------------------------------
 */
void
traffic_info_update (gpointer window)
{
	GtkWidget *widget = traffic_info;
	gint k;			/* channel number */
	gint i;
	gchar d_buf[64];
	gchar buf[64];
	gint baseline, color;

	/* hint on traffic info data structures 
	 * 0 = Download pppoe
	 * 1 = Upload pppoe
	 * 2 = Download chan1
	 * 3 = Upload chan1
	 * 4 = Download chan2
	 * 5 = Upload chan2
	 */

	d_buf[0] = '\0';
	buf[0] = '\0';

	/* find proper dimensions for rectangle */
	gdk_window_get_size (widget->window, &traffic_info_width,
			     &traffic_info_height);

	/* draw black background */
	gdk_draw_rectangle (widget->window, gcb, 1, 0, 0, traffic_info_width,
			    traffic_info_height);
	/* draw green lines */
	for (i = 0; i < traffic_info_width; i += 7)
		gdk_draw_line (widget->window, gcg, i, 0, i,
			       traffic_info_height - 2);
	for (i = (traffic_info_height - 2); i > 0; i -= 7)
		gdk_draw_line (widget->window, gcg, 0, i, traffic_info_width,
			       i);

	if (!is_online)
		return;

	baseline = 10;
	color = 0;
	for (k = 5; k >= 0; k--)
	{
		gfloat x, old_x;	/* index im widget */
		gint j = chan_data[k]->head;	/* zeigt auf aktuellen wert in der liste */
		gint old_height = 0;
		x = old_x = 2;

		if (!chan_online[k / 2])
			continue;

		while (j !=
		       (chan_data[k]->head - 1 +
			chan_data[k]->width) % chan_data[k]->width)
		{

			/* draw graph */
			gint draw_height =
				traffic_info_height - 2 -
				(chan_data[k]->cells[j] /
				 (gfloat) chan_data[k]->max) *
				(traffic_info_height - 6);
			if (x == 2)
				old_height = draw_height;
			gdk_draw_line (widget->window, gc[color],
				       (gint) old_x, old_height, (gint) x,
				       draw_height);

			old_height = draw_height;
			old_x = x;
			x += ((gfloat) (traffic_info_width - 2) /
			      (gfloat) (chan_data[k]->width - 2));
			j = (j + 1) % chan_data[k]->width;

		}

		/* draw up-/download rate */
		get_bytes (chan_data[k]->
			   cells[(chan_data[k]->head - 1 +
				  chan_data[k]->width) % chan_data[k]->width],
			   buf);
		if (k % 2 == 0)
		{		/* download */
			sprintf (d_buf, "D: %sB/s", buf);
			gdk_draw_text (widget->window, gf, gc[color], 2,
				       baseline, d_buf, strlen (d_buf));
			baseline += 12;
		}
		else
		{		/* upload */
			sprintf (d_buf, "U: %sB/s", buf);
			gdk_draw_text (widget->window, gf, gc[color],
				       (traffic_info_width / 2) + 2, baseline,
				       d_buf, strlen (d_buf));
		}
		color++;
	}
	/* draw a green at bottom of graphs */
	gdk_draw_line (widget->window, gcg, 0, traffic_info_height - 2,
		       traffic_info_width, traffic_info_height - 2);
}

struct TrafficInfoData *
createTrafficInfoData (gint width)
{
	gint i;
	struct TrafficInfoData *tData =
		(struct TrafficInfoData *)
		malloc (sizeof (struct TrafficInfoData));

	tData->cells = (guint *) malloc (width * sizeof (guint));

	for (i = 0; i < width; i++)
		tData->cells[i] = 0;

	tData->width = width;
	tData->head = 0;
	tData->max = 1;
	return tData;
}

void
traffic_info_insert (struct TrafficInfoData *tData, guint val)
{
	tData->cells[tData->head] = val;
	tData->head = (tData->head + 1) % tData->width;
	if (tData->max < val)
		tData->max = val;
}

void
traffic_info_clear (struct TrafficInfoData *tData)
{
	gint i;
	for (i = 0; i < tData->width; i++)
		tData->cells[i] = 0;
	tData->max = 1;
	tData->head = 0;
}

void
traffic_info_set_max (struct TrafficInfoData *tData, guint max)
{
	tData->max = max;
}

void
fill_phonebook (GHashTable *phonebook, const gchar *filename)
{
	FILE *fp;

	g_hash_table_freeze (phonebook);
	
	if ((fp = fopen (filename, "r")) != NULL)
	{
		gchar line[256];
		gchar *key, *val;
		
		while (fgets(line, 256, fp) != NULL) {

			if (!strncmp(line,"#",1)) /* skip comments */
				continue;

			if ((key = strtok(line,"=")) != NULL) {
				if ((val = strtok(NULL,"=")) != NULL) {
					val[strlen(val)-1] = '\0';
					if (val[strlen(val)-1] == '\r') /* probably dos file - convert to unix */
						val[strlen(val)-1] = '\0';
					g_hash_table_insert(phonebook,g_convert(key,strlen(key),"UTF-8","ISO-8859-1",NULL,NULL,NULL),g_convert(val,strlen(val),"UTF-8","ISO-8859-1",NULL,NULL,NULL));
				}
			}
		}

		fclose (fp);
	}
	else
	{
		g_print ("no phonebook found at %s\n", filename);
	}
	g_hash_table_thaw (phonebook);
}

void fli4l_disconnect() {
  if (GTK_WIDGET_VISIBLE(gtk_imonc))
		gtk_widget_set_sensitive(gtk_imonc,FALSE); // probably bad idea
	if (timeout_tag) {
		gtk_timeout_remove(timeout_tag);
		usleep(500000);
	}
	if (fd)
		service_disconnect(fd);
}

gboolean fli4l_connect()
{

	if ((fd = service_connect (host_name, port))<0) {
		GtkWidget *error, *err_label;
		gchar buf[1024];

		/* raise error */
		error = create_error_dialog();
		err_label = lookup_widget(error,"error_label");
		snprintf(buf,1024,"Can't connect!\nHost: %s\nPort: %d", host_name, port);
		gtk_label_set_text(GTK_LABEL(err_label),buf);

		on_preferences_activate(NULL,NULL);

		gtk_widget_show(error);

		return FALSE;
	}
	if (GTK_WIDGET_VISIBLE(gtk_imonc))
		gtk_widget_set_sensitive(gtk_imonc,TRUE);
	else
		gtk_widget_show(gtk_imonc);
	imonc_init(gtk_imonc);
	imonc_update(gtk_imonc);
	timeout_tag = gtk_timeout_add (TIMEOUT_MSEC, imonc_update, gtk_imonc);

	return TRUE;
}
