#include "mac2_theme.h"
#include <gtk/gtk.h>
#include <gmodule.h>

#define SCROLL_DELAY_LENGTH  300
#define SCROLLBAR_SPACING(w) (GTK_SCROLLED_WINDOW_CLASS (GTK_OBJECT (w)->klass)->scrollbar_spacing)
#define RANGE_CLASS(widget)  (GTK_RANGE_CLASS(GTK_OBJECT(widget)->klass))

/* Theme functions to export */
void theme_init(GtkThemeEngine * engine);
void theme_exit(void);

/* Exported vtable from th_draw */
extern GtkStyleClass mac2_default_class;

/* Local prototypes */
static void mac2_range_remove_timer(GtkRange *range);
static void mac2_range_vslider_update(GtkRange *range);
static void mac2_range_hslider_update(GtkRange *range);
static gint mac2_range_vtrough_click(GtkRange *range, gint x, gint y, gfloat *jump_perc);
static gint mac2_range_htrough_click(GtkRange *range, gint x, gint y, gfloat *jump_perc);
static void mac2_range_vmotion(GtkRange *range, gint xdelta, gint ydelta);
static void mac2_range_hmotion(GtkRange *range, gint xdelta, gint ydelta);
static void mac2_range_trough_vdims(GtkRange *range, gint *top,  gint *bottom);
static void mac2_range_trough_hdims(GtkRange *range, gint *left, gint *right);
static void mac2_vscrollbar_calc_slider_size(GtkVScrollbar *vscrollbar);
static void mac2_vscrollbar_slider_update(GtkRange *range);
static void mac2_vscrollbar_realize (GtkWidget *widget);
static void mac2_vscrollbar_size_allocate(GtkWidget *widget, GtkAllocation *allocation);
static void mac2_hscrollbar_calc_slider_size(GtkHScrollbar *hscrollbar);
static void mac2_hscrollbar_slider_update(GtkRange *range);
static void mac2_hscrollbar_realize (GtkWidget *widget);
static void mac2_hscrollbar_size_allocate (GtkWidget *widget, GtkAllocation *allocation);
static void mac2_scrolled_window_size_allocate (GtkWidget *widget, GtkAllocation *allocation);
static void mac2_scrolled_window_relative_allocation (GtkWidget *widget, GtkAllocation *allocation);
static void hls_to_rgb(gdouble * h, gdouble * l, gdouble * s);
static void rgb_to_hls(gdouble * r, gdouble * g, gdouble * b);

/* Theme parsing */
static struct
{
    gchar *name;
    guint  token;
}
theme_symbols[] =
{
  { "black_and_white",		TOKEN_BLACKANDWHITE },

  { "lightness_multiplier",	TOKEN_LIGHTNESSMULTIPLIER },
  { "darkness_multiplier",	TOKEN_DARKNESSMULTIPLIER },
  
  { "TRUE",			TOKEN_TRUE },
  { "FALSE",			TOKEN_FALSE },
};

static guint n_theme_symbols = sizeof(theme_symbols) / sizeof(theme_symbols[0]);

static guint
theme_parse_boolean(GScanner * scanner, ThemeRcData * theme_data)
{
    guint token, which_token;
    gboolean value;

    which_token = g_scanner_get_next_token(scanner);
    if (token != TOKEN_BLACKANDWHITE)
	return TOKEN_BLACKANDWHITE;

    token = g_scanner_get_next_token(scanner);
    if (token != G_TOKEN_EQUAL_SIGN)
	return G_TOKEN_EQUAL_SIGN;

    token = g_scanner_get_next_token(scanner);
    if (token == TOKEN_TRUE)
	value = TRUE;
    else if (token == TOKEN_FALSE)
	value = FALSE;
    else
	return TOKEN_TRUE;

    switch(which_token)
    {
	case TOKEN_BLACKANDWHITE:
	    theme_data->black_and_white = value;
	    break;

	default:
	    break;
    }
    return G_TOKEN_NONE;
}

static guint
theme_parse_multiplier(GScanner * scanner, ThemeRcData * theme_data)
{
    guint token, which_token;

    which_token = g_scanner_get_next_token(scanner);
    if ((which_token != TOKEN_LIGHTNESSMULTIPLIER) &&
	(which_token != TOKEN_DARKNESSMULTIPLIER))
	return TOKEN_LIGHTNESSMULTIPLIER;

    token = g_scanner_get_next_token(scanner);
    if (token != G_TOKEN_EQUAL_SIGN)
	return G_TOKEN_EQUAL_SIGN;

    token = g_scanner_get_next_token(scanner);
    if (token == G_TOKEN_FLOAT)
    {
	if (which_token == TOKEN_LIGHTNESSMULTIPLIER)
	    theme_data->lightness_multiplier = scanner->value.v_float;
	else
	    theme_data->darkness_multiplier = scanner->value.v_float;
    }
    else
	return G_TOKEN_FLOAT;

    return G_TOKEN_NONE;
}

static void
mac2_style_shade(GdkColor * a, GdkColor * b, gdouble k)
{
    gdouble red;
    gdouble green;
    gdouble blue;

    red = (gdouble) a->red / 65535.0;
    green = (gdouble) a->green / 65535.0;
    blue = (gdouble) a->blue / 65535.0;

    rgb_to_hls(&red, &green, &blue);

    green *= k;
    if (green > 1.0)
	green = 1.0;
    else if (green < 0.0)
	green = 0.0;

    blue *= k;
    if (blue > 1.0)
	blue = 1.0;
    else if (blue < 0.0)
	blue = 0.0;

    hls_to_rgb(&red, &green, &blue);

    b->red = red * 65535.0;
    b->green = green * 65535.0;
    b->blue = blue * 65535.0;
}

static void
hls_to_rgb(gdouble * h, gdouble * l, gdouble * s)
{
    gdouble hue;
    gdouble lightness;
    gdouble saturation;
    gdouble m1, m2;
    gdouble r, g, b;

    lightness = *l;
    saturation = *s;

    if (lightness <= 0.5)
	m2 = lightness * (1 + saturation);
    else
	m2 = lightness + saturation - lightness * saturation;
    m1 = 2 * lightness - m2;

    if (saturation == 0)
    {
	*h = lightness;
	*l = lightness;
	*s = lightness;
    }
    else
    {
	hue = *h + 120;
	while (hue > 360)
	    hue -= 360;
	while (hue < 0)
	    hue += 360;

	if (hue < 60)
	    r = m1 + (m2 - m1) * hue / 60;
	else if (hue < 180)
	    r = m2;
	else if (hue < 240)
	    r = m1 + (m2 - m1) * (240 - hue) / 60;
	else
	    r = m1;

	hue = *h;
	while (hue > 360)
	    hue -= 360;
	while (hue < 0)
	    hue += 360;

	if (hue < 60)
	    g = m1 + (m2 - m1) * hue / 60;
	else if (hue < 180)
	    g = m2;
	else if (hue < 240)
	    g = m1 + (m2 - m1) * (240 - hue) / 60;
	else
	    g = m1;

	hue = *h - 120;
	while (hue > 360)
	    hue -= 360;
	while (hue < 0)
	    hue += 360;

	if (hue < 60)
	    b = m1 + (m2 - m1) * hue / 60;
	else if (hue < 180)
	    b = m2;
	else if (hue < 240)
	    b = m1 + (m2 - m1) * (240 - hue) / 60;
	else
	    b = m1;

	*h = r;
	*l = g;
	*s = b;
    }
}

static void
rgb_to_hls(gdouble * r, gdouble * g, gdouble * b)
{
    gdouble min;
    gdouble max;
    gdouble red;
    gdouble green;
    gdouble blue;
    gdouble h, l, s;
    gdouble delta;

    red = *r;
    green = *g;
    blue = *b;

    if (red > green)
    {
	if (red > blue)
	    max = red;
	else
	    max = blue;

	if (green < blue)
	    min = green;
	else
	    min = blue;
    }
    else
    {
	if (green > blue)
	    max = green;
	else
	    max = blue;

	if (red < blue)
	    min = red;
	else
	    min = blue;
    }

    l = (max + min) / 2;
    s = 0;
    h = 0;

    if (max != min)
    {
	if (l <= 0.5)
	    s = (max - min) / (max + min);
	else
	    s = (max - min) / (2 - max - min);

	delta = max - min;
	if (red == max)
	    h = (green - blue) / delta;
	else if (green == max)
	    h = 2 + (blue - red) / delta;
	else if (blue == max)
	    h = 4 + (red - green) / delta;

	h *= 60;
	if (h < 0.0)
	    h += 360;
    }

    *r = h;
    *g = l;
    *b = s;
}

guint
theme_parse_rc_style(GScanner * scanner, GtkRcStyle * rc_style)
{
    static GQuark scope_id = 0;
    ThemeRcData *theme_data;
    guint old_scope;
    guint token;
    guint i;

    /* Set up a new scope in this scanner. */

    if (!scope_id)
	scope_id = g_quark_from_string("theme_engine");

    /*
     * If we bail out due to errors, we *don't* reset the scope, so the
     * error messaging code can make sense of our tokens.
     */
    old_scope = g_scanner_set_scope(scanner, scope_id);

    /*
     * Now check if we already added our symbols to this scope
     * (in some previous call to theme_parse_rc_style for the
     * same scanner.
     */

    if (!g_scanner_lookup_symbol(scanner, theme_symbols[0].name))
    {
	g_scanner_freeze_symbol_table(scanner);
	for (i = 0; i < n_theme_symbols; i++)
	{
	    g_scanner_scope_add_symbol(scanner, scope_id,
				       theme_symbols[i].name,
				       GINT_TO_POINTER(theme_symbols[i].token));
	}
	g_scanner_thaw_symbol_table(scanner);
    }

    /* We're ready to go, now parse the top level */

    theme_data = g_new(ThemeRcData, 1);
    theme_data->black_and_white = DEFAULT_BLACKANDWHITE;

    token = g_scanner_peek_next_token(scanner);
    while (token != G_TOKEN_RIGHT_CURLY)
    {
	switch (token)
	{
	    case TOKEN_BLACKANDWHITE:
		token = theme_parse_boolean(scanner, theme_data);
		break;
	    case TOKEN_LIGHTNESSMULTIPLIER:
	    case TOKEN_DARKNESSMULTIPLIER:
		token = theme_parse_multiplier(scanner, theme_data);
		break;
	    default:
		g_scanner_get_next_token(scanner);
		token = G_TOKEN_RIGHT_CURLY;
		break;
	}

	if (token != G_TOKEN_NONE)
	{
	    g_free(theme_data);
	    return token;
	}
	token = g_scanner_peek_next_token(scanner);
    }

    g_scanner_get_next_token(scanner);

    rc_style->engine_data = theme_data;
    g_scanner_set_scope(scanner, old_scope);

    return G_TOKEN_NONE;
}

void
theme_merge_rc_style(GtkRcStyle * dest, GtkRcStyle * src)
{
    ThemeRcData *src_data = src->engine_data;
    ThemeRcData *dest_data = dest->engine_data;

    if (!dest_data)
    {
	dest_data = g_new(ThemeRcData, 1);
	dest->engine_data = dest_data;
    }

    dest_data->lightness_multiplier = src_data->lightness_multiplier;
    dest_data->darkness_multiplier = src_data->darkness_multiplier;
    dest_data->black_and_white = src_data->black_and_white;
}

void
theme_rc_style_to_style(GtkStyle * style, GtkRcStyle * rc_style)
{
    ThemeRcData *data = rc_style->engine_data;
    ThemeStyleData *style_data;

    style_data = g_new(ThemeStyleData, 1);
    style_data->lightness_multiplier = data->lightness_multiplier;
    style_data->darkness_multiplier = data->darkness_multiplier;
    style_data->black_and_white = data->black_and_white;

    style->klass = &mac2_default_class;
    style->engine_data = style_data;
}

void
theme_duplicate_style(GtkStyle * dest, GtkStyle * src)
{
    ThemeStyleData *dest_data;
    ThemeStyleData *src_data = src->engine_data;

    dest_data = g_new(ThemeStyleData, 1);
    dest_data->lightness_multiplier = src_data->lightness_multiplier;
    dest_data->darkness_multiplier = src_data->darkness_multiplier;
    dest_data->black_and_white = src_data->black_and_white;

    dest->klass = &mac2_default_class;
    dest->engine_data = dest_data;
}

void
theme_realize_style(GtkStyle * style)
{
    ThemeStyleData *th_data = style->engine_data;

    GdkGCValuesMask gc_values_mask = GDK_GC_FOREGROUND | GDK_GC_FONT;
    GdkGCValues gc_values;
    gint i;
    extern GdkFont *default_font;

    if (!default_font)
	default_font
	    =
	    gdk_font_load
	    ("-adobe-helvetica-medium-r-normal--*-120-*-*-*-*-*-*");

    if (style->font->type == GDK_FONT_FONT)
	gc_values.font = style->font;
    else if (style->font->type == GDK_FONT_FONTSET)
	gc_values.font = default_font;

    for (i = 0; i < 5; i++)
    {
	/*
	 *  Release old GC
	 */
	gtk_gc_release(style->light_gc[i]);
	gtk_gc_release(style->dark_gc[i]);

	/*
	 *  Compute new colors
	 */
	mac2_style_shade(&style->bg[i], &style->light[i],
			  th_data->lightness_multiplier);
	mac2_style_shade(&style->bg[i], &style->dark[i],
			  th_data->darkness_multiplier);

	style->mid[i].red = (style->light[i].red + style->dark[i].red) / 2;
	style->mid[i].green =
	    (style->light[i].green + style->dark[i].green) / 2;
	style->mid[i].blue = (style->light[i].blue + style->dark[i].blue) / 2;

	/*
	 *  Allocate new gc
	 */
	if (!gdk_color_alloc(style->colormap, &style->light[i]))
	    g_warning("unable to allocate color: ( %d %d %d )",
		      style->light[i].red, style->light[i].green,
		      style->light[i].blue);
	if (!gdk_color_alloc(style->colormap, &style->dark[i]))
	    g_warning("unable to allocate color: ( %d %d %d )",
		      style->dark[i].red, style->dark[i].green,
		      style->dark[i].blue);
	if (!gdk_color_alloc(style->colormap, &style->mid[i]))
	    g_warning("unable to allocate color: ( %d %d %d )",
		      style->mid[i].red, style->mid[i].green,
		      style->mid[i].blue);

	gc_values.foreground = style->light[i];
	style->light_gc[i] = gtk_gc_get(style->depth, style->colormap,
					&gc_values, gc_values_mask);

	gc_values.foreground = style->dark[i];
	style->dark_gc[i] = gtk_gc_get(style->depth, style->colormap,
				       &gc_values, gc_values_mask);

	gc_values.foreground = style->mid[i];
	style->mid_gc[i] = gtk_gc_get(style->depth, style->colormap,
				      &gc_values, gc_values_mask);
    }
}

void
theme_unrealize_style(GtkStyle * style)
{
}

void
theme_destroy_rc_style(GtkRcStyle * rc_style)
{
    ThemeRcData *data = rc_style->engine_data;

    if (data)
    {
	g_free(data);
    }
}

void
theme_destroy_style(GtkStyle * style)
{
    ThemeStyleData *data = style->engine_data;

    if (data)
    {
	g_free(data);
    }
}

void
theme_set_background(GtkStyle * style,
		     GdkWindow * window, GtkStateType state_type)
{
    GdkPixmap *pixmap;
    gint parent_relative;

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);

    if (style->bg_pixmap[state_type])
    {
	if (style->bg_pixmap[state_type] == (GdkPixmap *) GDK_PARENT_RELATIVE)
	{
	    pixmap = NULL;
	    parent_relative = TRUE;
	}
	else
	{
	    pixmap = style->bg_pixmap[state_type];
	    parent_relative = FALSE;
	}

	gdk_window_set_back_pixmap(window, pixmap, parent_relative);
    }
    else
	gdk_window_set_background(window, &style->bg[state_type]);
}

void
theme_init(GtkThemeEngine * engine)
{
    GtkRangeClass *rangeclass;
    GtkHScrollbarClass *hscrollbarclass;
    GtkVScrollbarClass *vscrollbarclass;
    GtkWidgetClass *widgetclass;

    engine->parse_rc_style = theme_parse_rc_style;
    engine->merge_rc_style = theme_merge_rc_style;
    engine->rc_style_to_style = theme_rc_style_to_style;
    engine->duplicate_style = theme_duplicate_style;
    engine->realize_style = theme_realize_style;
    engine->unrealize_style = theme_unrealize_style;
    engine->destroy_rc_style = theme_destroy_rc_style;
    engine->destroy_style = theme_destroy_style;
    engine->set_background = theme_set_background;

    /*
     * This code was ad[oa]pted from GTKstep theme. It moves both scrollbar
     * buttons of a horizontal scrollbar to the right and of a vertical
     * scrollbar to the bottom.
     */

    hscrollbarclass =
	(GtkHScrollbarClass *) gtk_type_class(gtk_hscrollbar_get_type());
    rangeclass = (GtkRangeClass *) hscrollbarclass;
    widgetclass = (GtkWidgetClass *) hscrollbarclass;

    rangeclass->slider_width = DEFAULT_SLIDER_WIDTH;
    rangeclass->min_slider_size = DEFAULT_MIN_SLIDER_SIZE;
    rangeclass->stepper_size = DEFAULT_SLIDER_WIDTH;
    rangeclass->stepper_slider_spacing = 0;

    rangeclass->motion = mac2_range_hmotion;
    rangeclass->trough_click = mac2_range_htrough_click;
    rangeclass->slider_update = mac2_hscrollbar_slider_update;
    widgetclass->realize = mac2_hscrollbar_realize;
    widgetclass->size_allocate = mac2_hscrollbar_size_allocate;

    vscrollbarclass =
	(GtkVScrollbarClass *) gtk_type_class(gtk_vscrollbar_get_type());
    rangeclass = (GtkRangeClass *) vscrollbarclass;
    widgetclass = (GtkWidgetClass *) vscrollbarclass;

    rangeclass->slider_width = DEFAULT_SLIDER_WIDTH;
    rangeclass->min_slider_size = DEFAULT_MIN_SLIDER_SIZE;
    rangeclass->stepper_size = DEFAULT_SLIDER_WIDTH;
    rangeclass->stepper_slider_spacing = 0;

    rangeclass->motion = mac2_range_vmotion;
    rangeclass->trough_click = mac2_range_vtrough_click;
    rangeclass->slider_update = mac2_vscrollbar_slider_update;
    widgetclass->realize = mac2_vscrollbar_realize;
    widgetclass->size_allocate = mac2_vscrollbar_size_allocate;
}

void
theme_exit(void)
{
}

/*
 * The following function will be called by GTK+ when the module
 * is loaded and checks to see if we are compatible with the
 * version of GTK+ that loads us.
 */
G_MODULE_EXPORT const gchar *g_module_check_init(GModule * module);

const gchar *
g_module_check_init(GModule * module)
{
    return gtk_check_version(GTK_MAJOR_VERSION,
			     GTK_MINOR_VERSION,
			     GTK_MICRO_VERSION - GTK_INTERFACE_AGE);
}

/*
 * Following is the scrollbar code
 */

static void
mac2_hscrollbar_realize(GtkWidget * widget)
{
    GtkRange *range;
    GdkWindowAttr attributes;
    gint attributes_mask;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_HSCROLLBAR(widget));

    GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);
    range = GTK_RANGE(widget);

    attributes.x = widget->allocation.x;
    attributes.y =
	widget->allocation.y + (widget->allocation.height -
				widget->requisition.height) / 2;
    attributes.width = widget->allocation.width;
    attributes.height = widget->requisition.height;
    attributes.wclass = GDK_INPUT_OUTPUT;
    attributes.window_type = GDK_WINDOW_CHILD;
    attributes.visual = gtk_widget_get_visual(widget);
    attributes.colormap = gtk_widget_get_colormap(widget);
    attributes.event_mask = gtk_widget_get_events(widget);
    attributes.event_mask |= (GDK_EXPOSURE_MASK |
			      GDK_BUTTON_PRESS_MASK |
			      GDK_BUTTON_RELEASE_MASK |
			      GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);

    attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
    widget->window =
	gdk_window_new(gtk_widget_get_parent_window(widget), &attributes,
		       attributes_mask);

    range->trough = widget->window;
    gdk_window_ref(range->trough);


    /* back arrow */
    attributes.x = widget->allocation.width -
	widget->style->klass->xthickness -
	RANGE_CLASS(widget)->stepper_size * 2;
    attributes.y = widget->style->klass->ythickness;
    attributes.width = RANGE_CLASS(widget)->stepper_size;
    attributes.height = RANGE_CLASS(widget)->stepper_size;
    range->step_back =
	gdk_window_new(range->trough, &attributes, attributes_mask);

    /* forward arrow */
    attributes.x = widget->allocation.width -
	widget->style->klass->xthickness - RANGE_CLASS(widget)->stepper_size;
    range->step_forw =
	gdk_window_new(range->trough, &attributes, attributes_mask);



    /* slider */
    attributes.x = 0;
    attributes.y = widget->style->klass->ythickness;
    attributes.width = RANGE_CLASS(widget)->min_slider_size;
    attributes.height = RANGE_CLASS(widget)->slider_width;
    attributes.event_mask |= (GDK_BUTTON_MOTION_MASK |
			      GDK_POINTER_MOTION_HINT_MASK);
    range->slider = gdk_window_new(range->trough, &attributes, attributes_mask);

    mac2_hscrollbar_calc_slider_size(GTK_HSCROLLBAR(widget));
    gtk_range_slider_update(GTK_RANGE(widget));

    widget->style = gtk_style_attach(widget->style, widget->window);

    gdk_window_set_user_data(range->trough, widget);
    gdk_window_set_user_data(range->slider, widget);
    gdk_window_set_user_data(range->step_forw, widget);
    gdk_window_set_user_data(range->step_back, widget);

    gtk_style_set_background(widget->style, range->trough, GTK_STATE_ACTIVE);
    gtk_style_set_background(widget->style, range->slider, GTK_STATE_NORMAL);
    gtk_style_set_background(widget->style, range->step_forw, GTK_STATE_ACTIVE);
    gtk_style_set_background(widget->style, range->step_back, GTK_STATE_ACTIVE);

    gdk_window_show(range->slider);
    gdk_window_show(range->step_back);
    gdk_window_show(range->step_forw);
}

static void
mac2_vscrollbar_realize(GtkWidget * widget)
{
    GtkRange *range;
    GdkWindowAttr attributes;
    gint attributes_mask;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_VSCROLLBAR(widget));

    GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);
    range = GTK_RANGE(widget);

    attributes.x =
	widget->allocation.x + (widget->allocation.width -
				widget->requisition.width) / 2;
    attributes.y = widget->allocation.y;
    attributes.width = widget->requisition.width;
    attributes.height = widget->allocation.height;
    attributes.wclass = GDK_INPUT_OUTPUT;
    attributes.window_type = GDK_WINDOW_CHILD;
    attributes.visual = gtk_widget_get_visual(widget);
    attributes.colormap = gtk_widget_get_colormap(widget);
    attributes.event_mask = gtk_widget_get_events(widget);
    attributes.event_mask |= (GDK_EXPOSURE_MASK |
			      GDK_BUTTON_PRESS_MASK |
			      GDK_BUTTON_RELEASE_MASK |
			      GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);

    attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
    widget->window =
	gdk_window_new(gtk_widget_get_parent_window(widget), &attributes,
		       attributes_mask);

    range->trough = widget->window;
    gdk_window_ref(range->trough);

    attributes.x = widget->style->klass->xthickness;
    attributes.y = (widget->allocation.height -
		    widget->style->klass->ythickness -
		    2 * RANGE_CLASS(widget)->stepper_size);
    attributes.width = RANGE_CLASS(widget)->stepper_size;
    attributes.height = RANGE_CLASS(widget)->stepper_size;

    range->step_back =
	gdk_window_new(range->trough, &attributes, attributes_mask);

    attributes.y = (widget->allocation.height -
		    widget->style->klass->ythickness -
		    RANGE_CLASS(widget)->stepper_size);

    range->step_forw =
	gdk_window_new(range->trough, &attributes, attributes_mask);

    attributes.x = widget->style->klass->ythickness;
    attributes.y = 0;
    attributes.width = RANGE_CLASS(widget)->slider_width;
    attributes.height = RANGE_CLASS(widget)->min_slider_size;
    attributes.event_mask |= (GDK_BUTTON_MOTION_MASK |
			      GDK_POINTER_MOTION_HINT_MASK);

    range->slider = gdk_window_new(range->trough, &attributes, attributes_mask);

    mac2_vscrollbar_calc_slider_size(GTK_VSCROLLBAR(widget));
    gtk_range_slider_update(GTK_RANGE(widget));

    widget->style = gtk_style_attach(widget->style, widget->window);

    gdk_window_set_user_data(range->trough, widget);
    gdk_window_set_user_data(range->slider, widget);
    gdk_window_set_user_data(range->step_forw, widget);
    gdk_window_set_user_data(range->step_back, widget);

    gtk_style_set_background(widget->style, range->trough, GTK_STATE_ACTIVE);
    gtk_style_set_background(widget->style, range->slider, GTK_STATE_NORMAL);
    gtk_style_set_background(widget->style, range->step_forw, GTK_STATE_ACTIVE);
    gtk_style_set_background(widget->style, range->step_back, GTK_STATE_ACTIVE);

    gdk_window_show(range->slider);
    gdk_window_show(range->step_back);
    gdk_window_show(range->step_forw);
}

static void
mac2_hscrollbar_size_allocate(GtkWidget * widget, GtkAllocation * allocation)
{
    GtkRange *range;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_HSCROLLBAR(widget));
    g_return_if_fail(allocation != NULL);

    widget->allocation = *allocation;
    if (GTK_WIDGET_REALIZED(widget))
    {
	range = GTK_RANGE(widget);

	gdk_window_move_resize(range->trough,
			       allocation->x,
			       allocation->y + (allocation->height -
						widget->requisition.height) /
			       2, allocation->width,
			       widget->requisition.height);
	gdk_window_move_resize(range->step_back,
			       widget->allocation.width -
			       widget->style->klass->xthickness -
			       RANGE_CLASS(widget)->stepper_size * 2,
			       widget->style->klass->ythickness,
			       RANGE_CLASS(widget)->stepper_size,
			       widget->requisition.height -
			       widget->style->klass->ythickness * 2);

	gdk_window_move_resize(range->step_forw,
			       widget->allocation.width -
			       widget->style->klass->xthickness -
			       RANGE_CLASS(widget)->stepper_size,
			       widget->style->klass->ythickness,
			       RANGE_CLASS(widget)->stepper_size,
			       widget->requisition.height -
			       widget->style->klass->ythickness * 2);

	gdk_window_resize(range->slider,
			  RANGE_CLASS(widget)->min_slider_size,
			  widget->requisition.height -
			  widget->style->klass->ythickness * 2);

	gtk_range_slider_update(GTK_RANGE(widget));
    }
}

static void
mac2_vscrollbar_size_allocate(GtkWidget * widget, GtkAllocation * allocation)
{
    GtkRange *range;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_VSCROLLBAR(widget));
    g_return_if_fail(allocation != NULL);

    widget->allocation = *allocation;
    if (GTK_WIDGET_REALIZED(widget))
    {
	range = GTK_RANGE(widget);

	gdk_window_move_resize(range->trough,
			       allocation->x + (allocation->width -
						widget->requisition.width) /
			       2, allocation->y, widget->requisition.width,
			       allocation->height);
	gdk_window_move_resize(range->step_back,
			       widget->style->klass->xthickness,
			       allocation->height -
			       widget->style->klass->ythickness -
			       2 * RANGE_CLASS(widget)->stepper_size,
			       widget->requisition.width -
			       widget->style->klass->xthickness * 2,
			       RANGE_CLASS(widget)->stepper_size);
	gdk_window_move_resize(range->step_forw,
			       widget->style->klass->xthickness,
			       allocation->height -
			       widget->style->klass->ythickness -
			       RANGE_CLASS(widget)->stepper_size,
			       widget->requisition.width -
			       widget->style->klass->xthickness * 2,
			       RANGE_CLASS(widget)->stepper_size);
	gdk_window_resize(range->slider,
			  widget->requisition.width -
			  widget->style->klass->xthickness * 2,
			  RANGE_CLASS(range)->min_slider_size);

	gtk_range_slider_update(GTK_RANGE(widget));
    }
}

static void
mac2_hscrollbar_slider_update(GtkRange * range)
{
    g_return_if_fail(range != NULL);
    g_return_if_fail(GTK_IS_HSCROLLBAR(range));

    mac2_hscrollbar_calc_slider_size(GTK_HSCROLLBAR(range));
    mac2_range_hslider_update(range);
}

static void
mac2_vscrollbar_slider_update(GtkRange * range)
{
    g_return_if_fail(range != NULL);
    g_return_if_fail(GTK_IS_VSCROLLBAR(range));

    mac2_vscrollbar_calc_slider_size(GTK_VSCROLLBAR(range));
    mac2_range_vslider_update(range);
}

static void
mac2_range_trough_hdims(GtkRange * range, gint * left, gint * right)
{
    gint trough_width;
    gint slider_length;
    gint tmp_width;
    gint tleft;
    gint tright;

    g_return_if_fail(range != NULL);

    gdk_window_get_size(range->trough, &trough_width, NULL);
    gdk_window_get_size(range->slider, &slider_length, NULL);

    tleft = GTK_WIDGET(range)->style->klass->xthickness;
    tright =
	trough_width - 1 - slider_length -
	GTK_WIDGET(range)->style->klass->xthickness;




    if (range->step_back)
    {
	gdk_window_get_size(range->step_back, &tmp_width, NULL);
	tright -= (tmp_width + RANGE_CLASS(range)->stepper_slider_spacing);
    }

    if (range->step_forw)
    {
	gdk_window_get_size(range->step_forw, &tmp_width, NULL);
	tright -= (tmp_width + RANGE_CLASS(range)->stepper_slider_spacing);
    }

    if (left)
	*left = tleft;
    if (right)
	*right = tright;
}

static void
mac2_range_trough_vdims(GtkRange * range, gint * top, gint * bottom)
{
    gint trough_height;
    gint slider_length;
    gint tmp_height;
    gint ttop;
    gint tbottom;

    g_return_if_fail(range != NULL);

    gdk_window_get_size(range->trough, NULL, &trough_height);
    gdk_window_get_size(range->slider, NULL, &slider_length);

    ttop = GTK_WIDGET(range)->style->klass->ythickness;
    tbottom =
	trough_height - 1 - slider_length -
	GTK_WIDGET(range)->style->klass->ythickness;

    if (range->step_back)
    {
	gdk_window_get_size(range->step_back, NULL, &tmp_height);
	tbottom -= (tmp_height + RANGE_CLASS(range)->stepper_slider_spacing);
    }

    if (range->step_forw)
    {
	gdk_window_get_size(range->step_forw, NULL, &tmp_height);
	tbottom -= (tmp_height + RANGE_CLASS(range)->stepper_slider_spacing);
    }

    if (top)
	*top = ttop;
    if (bottom)
	*bottom = tbottom;
}

static void
mac2_range_hmotion(GtkRange * range, gint xdelta, gint ydelta)
{
    gdouble old_value;
    gint left, right;
    gint slider_x, slider_y;
    gint new_pos;

    g_return_if_fail(range != NULL);
    g_return_if_fail(GTK_IS_RANGE(range));

    range = GTK_RANGE(range);

    gdk_window_get_position(range->slider, &slider_x, &slider_y);
    mac2_range_trough_hdims(range, &left, &right);

    if (left == right)
	return;

    new_pos = slider_x + xdelta;

    if (new_pos < left)
	new_pos = left;
    else if (new_pos > right)
	new_pos = right;

    old_value = range->adjustment->value;
    range->adjustment->value = ((range->adjustment->upper -
				 range->adjustment->lower -
				 range->adjustment->page_size) *
				(new_pos - left) / (right - left) +
				range->adjustment->lower);

    if (range->digits >= 0)
    {
	char buffer[64];

	sprintf(buffer, "%0.*f", range->digits, range->adjustment->value);
	sscanf(buffer, "%f", &range->adjustment->value);
    }

    if (old_value != range->adjustment->value)
    {
	if (range->policy == GTK_UPDATE_CONTINUOUS)
	{
	    gtk_signal_emit_by_name(GTK_OBJECT(range->adjustment),
				    "value_changed");
	}
	else
	{
	    gtk_range_slider_update(range);
	    gtk_range_clear_background(range);

	    if (range->policy == GTK_UPDATE_DELAYED)
	    {
		mac2_range_remove_timer(range);
		range->timer = gtk_timeout_add(SCROLL_DELAY_LENGTH,
					       (GtkFunction)
					       RANGE_CLASS(range)->timer,
					       (gpointer) range);
	    }
	}
    }
}

static void
mac2_range_vmotion(GtkRange * range, gint xdelta, gint ydelta)
{
    gdouble old_value;
    gint top, bottom;
    gint slider_x, slider_y;
    gint new_pos;

    g_return_if_fail(range != NULL);
    g_return_if_fail(GTK_IS_RANGE(range));

    range = GTK_RANGE(range);

    gdk_window_get_position(range->slider, &slider_x, &slider_y);
    mac2_range_trough_vdims(range, &top, &bottom);

    if (bottom == top)
	return;

    new_pos = slider_y + ydelta;

    if (new_pos < top)
	new_pos = top;
    else if (new_pos > bottom)
	new_pos = bottom;

    old_value = range->adjustment->value;
    range->adjustment->value = ((range->adjustment->upper -
				 range->adjustment->lower -
				 range->adjustment->page_size) *
				(new_pos - top) / (bottom - top) +
				range->adjustment->lower);

    if (range->digits >= 0)
    {
	char buffer[64];

	sprintf(buffer, "%0.*f", range->digits, range->adjustment->value);
	sscanf(buffer, "%f", &range->adjustment->value);
    }

    if (old_value != range->adjustment->value)
    {
	if (range->policy == GTK_UPDATE_CONTINUOUS)
	{
	    gtk_signal_emit_by_name(GTK_OBJECT(range->adjustment),
				    "value_changed");
	}
	else
	{
	    gtk_range_slider_update(range);
	    gtk_range_clear_background(range);

	    if (range->policy == GTK_UPDATE_DELAYED)
	    {
		mac2_range_remove_timer(range);
		range->timer = gtk_timeout_add(SCROLL_DELAY_LENGTH,
					       (GtkFunction)
					       RANGE_CLASS(range)->timer,
					       (gpointer) range);
	    }
	}
    }
}

static gint
mac2_range_htrough_click(GtkRange * range, gint x, gint y, gfloat * jump_perc)
{
    gint ythickness;
    gint trough_width;
    gint trough_height;
    gint slider_x;
    gint slider_length;
    gint left, right;

    g_return_val_if_fail(range != NULL, GTK_TROUGH_NONE);
    g_return_val_if_fail(GTK_IS_RANGE(range), GTK_TROUGH_NONE);

    ythickness = GTK_WIDGET(range)->style->klass->ythickness;

    mac2_range_trough_hdims(range, &left, &right);
    gdk_window_get_size(range->slider, &slider_length, NULL);
    right += slider_length;

    if ((x > left) && (y > ythickness))
    {
	gdk_window_get_size(range->trough, &trough_width, &trough_height);

	if ((x < right) && (y < (trough_height - ythickness)))
	{
	    if (jump_perc)
	    {
		*jump_perc =
		    ((gdouble) (x - left)) / ((gdouble) (right - left));
		return GTK_TROUGH_JUMP;
	    }

	    gdk_window_get_position(range->slider, &slider_x, NULL);

	    if (x < slider_x)
		return GTK_TROUGH_START;
	    else
		return GTK_TROUGH_END;
	}
    }

    return GTK_TROUGH_NONE;
}

static gint
mac2_range_vtrough_click(GtkRange * range, gint x, gint y, gfloat * jump_perc)
{
    gint xthickness;
    gint trough_width;
    gint trough_height;
    gint slider_y;
    gint top, bottom;
    gint slider_length;

    g_return_val_if_fail(range != NULL, GTK_TROUGH_NONE);
    g_return_val_if_fail(GTK_IS_RANGE(range), GTK_TROUGH_NONE);

    xthickness = GTK_WIDGET(range)->style->klass->xthickness;

    mac2_range_trough_vdims(range, &top, &bottom);
    gdk_window_get_size(range->slider, NULL, &slider_length);
    bottom += slider_length;

    if ((x > xthickness) && (y > top))
    {
	gdk_window_get_size(range->trough, &trough_width, &trough_height);

	if ((x < (trough_width - xthickness) && (y < bottom)))
	{
	    if (jump_perc)
	    {
		*jump_perc = ((gdouble) (y - top)) / ((gdouble) (bottom - top));

		return GTK_TROUGH_JUMP;
	    }

	    gdk_window_get_position(range->slider, NULL, &slider_y);

	    if (y < slider_y)
		return GTK_TROUGH_START;
	    else
		return GTK_TROUGH_END;
	}
    }

    return GTK_TROUGH_NONE;
}

static void
mac2_range_hslider_update(GtkRange * range)
{
    gint left;
    gint right;
    gint x;

    g_return_if_fail(range != NULL);
    g_return_if_fail(GTK_IS_RANGE(range));

    if (GTK_WIDGET_REALIZED(range))
    {
	mac2_range_trough_hdims(range, &left, &right);
	x = left;

	if (range->adjustment->value < range->adjustment->lower)
	{
	    range->adjustment->value = range->adjustment->lower;
	    gtk_signal_emit_by_name(GTK_OBJECT(range->adjustment),
				    "value_changed");
	}
	else if (range->adjustment->value > range->adjustment->upper)
	{
	    range->adjustment->value = range->adjustment->upper;
	    gtk_signal_emit_by_name(GTK_OBJECT(range->adjustment),
				    "value_changed");
	}

	if (range->adjustment->lower !=
	    (range->adjustment->upper - range->adjustment->page_size))
	    x +=
		((right - left) *
		 (range->adjustment->value -
		  range->adjustment->lower) / (range->adjustment->upper -
					       range->adjustment->lower -
					       range->adjustment->page_size));

	if (x < left)
	    x = left;
	else if (x > right)
	    x = right;

	gdk_window_move(range->slider, x,
			GTK_WIDGET(range)->style->klass->ythickness);
    }
}

static void
mac2_range_vslider_update(GtkRange * range)
{
    gint top;
    gint bottom;
    gint y;

    g_return_if_fail(range != NULL);
    g_return_if_fail(GTK_IS_RANGE(range));

    if (GTK_WIDGET_REALIZED(range))
    {
	mac2_range_trough_vdims(range, &top, &bottom);
	y = top;

	if (range->adjustment->value < range->adjustment->lower)
	{
	    range->adjustment->value = range->adjustment->lower;
	    gtk_signal_emit_by_name(GTK_OBJECT(range->adjustment),
				    "value_changed");
	}
	else if (range->adjustment->value > range->adjustment->upper)
	{
	    range->adjustment->value = range->adjustment->upper;
	    gtk_signal_emit_by_name(GTK_OBJECT(range->adjustment),
				    "value_changed");
	}

	if (range->adjustment->lower !=
	    (range->adjustment->upper - range->adjustment->page_size))
	{
	    y +=
		((bottom - top) *
		 (range->adjustment->value -
		  range->adjustment->lower) / (range->adjustment->upper -
					       range->adjustment->lower -
					       range->adjustment->page_size));
	}

	if (y < top)
	    y = top;
	else if (y > bottom)
	    y = bottom;

	gdk_window_move(range->slider,
			GTK_WIDGET(range)->style->klass->xthickness, y);
    }
}

static void
mac2_hscrollbar_calc_slider_size(GtkHScrollbar * hscrollbar)
{
    GtkRange *range;
    gint step_back_x;
    gint step_back_width;
    gint step_forw_x;
    gint step_forw_width;
    gint trough_width;
    gint slider_width;
    gint slider_height;
    gint left, right;
    gint width;

    g_return_if_fail(hscrollbar != NULL);
    g_return_if_fail(GTK_IS_HSCROLLBAR(hscrollbar));

    if (GTK_WIDGET_REALIZED(hscrollbar))
    {
	range = GTK_RANGE(hscrollbar);

	gdk_window_get_size(range->step_back, &step_back_width, NULL);
	gdk_window_get_size(range->step_forw, &step_forw_width, NULL);
	gdk_window_get_size(range->trough, &trough_width, NULL);
	gdk_window_get_position(range->step_back, &step_back_x, NULL);
	gdk_window_get_position(range->step_forw, &step_forw_x, NULL);

	left = step_back_width + step_forw_width +
		RANGE_CLASS (hscrollbar)->stepper_slider_spacing +
		GTK_WIDGET(hscrollbar)->style->klass->xthickness;
	right = GTK_WIDGET(hscrollbar)->allocation.width
	    - 1 - GTK_WIDGET(hscrollbar)->style->klass->xthickness;
	width = right - left;

	if ((range->adjustment->page_size > 0) &&
	    (range->adjustment->lower != range->adjustment->upper))
	{
	    if (range->adjustment->page_size >
		(range->adjustment->upper - range->adjustment->lower))
		range->adjustment->page_size =
		    range->adjustment->upper - range->adjustment->lower;

	    width = (width * range->adjustment->page_size /
		     (range->adjustment->upper - range->adjustment->lower));

	    if (width < RANGE_CLASS(hscrollbar)->min_slider_size)
		width = RANGE_CLASS(hscrollbar)->min_slider_size;
	}

	gdk_window_get_size(range->slider, &slider_width, &slider_height);

	if (slider_width != width)
	    gdk_window_resize(range->slider, width, slider_height);
    }
}

static void
mac2_vscrollbar_calc_slider_size(GtkVScrollbar * vscrollbar)
{
    GtkRange *range;
    gint step_back_y;
    gint step_back_height;
    gint step_forw_y;
    gint step_forw_width;
    gint trough_width;
    gint slider_width;
    gint slider_height;
    gint top, bottom;
    gint height;

    g_return_if_fail(vscrollbar != NULL);
    g_return_if_fail(GTK_IS_VSCROLLBAR(vscrollbar));

    if (GTK_WIDGET_REALIZED(vscrollbar))
    {
	range = GTK_RANGE(vscrollbar);

	gdk_window_get_size(range->step_back, NULL, &step_back_height);
	gdk_window_get_size(range->step_forw, &step_forw_width, NULL);
	gdk_window_get_size(range->trough, &trough_width, NULL);
	gdk_window_get_position(range->step_back, NULL, &step_back_y);
	gdk_window_get_position(range->step_forw, NULL, &step_forw_y);

	top = GTK_WIDGET(vscrollbar)->style->klass->ythickness;
	bottom = step_back_y -
	    RANGE_CLASS(vscrollbar)->stepper_slider_spacing * 2 - 1;
	height = bottom - top;

	if ((range->adjustment->page_size > 0) &&
	    (range->adjustment->lower != range->adjustment->upper))
	{
	    if (range->adjustment->page_size >
		(range->adjustment->upper - range->adjustment->lower))
		range->adjustment->page_size =
		    range->adjustment->upper - range->adjustment->lower;

	    height = (height * range->adjustment->page_size /
		      (range->adjustment->upper - range->adjustment->lower));

	    if (height < RANGE_CLASS(vscrollbar)->min_slider_size)
		height = RANGE_CLASS(vscrollbar)->min_slider_size;
	}

	gdk_window_get_size(range->slider, &slider_width, &slider_height);

	if (slider_height != height)
	    gdk_window_resize(range->slider, slider_width, height);
    }
}

static void
mac2_scrolled_window_relative_allocation(GtkWidget * widget,
					  GtkAllocation * allocation)
{
    GtkScrolledWindow *scrolled_window;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(allocation != NULL);

    scrolled_window = GTK_SCROLLED_WINDOW(widget);

    allocation->x = GTK_CONTAINER(widget)->border_width;
    allocation->y = GTK_CONTAINER(widget)->border_width;
    allocation->width =
	MAX(1, (gint) widget->allocation.width - allocation->x * 2);
    allocation->height =
	MAX(1, (gint) widget->allocation.height - allocation->y * 2);

    if (scrolled_window->vscrollbar_visible)
    {
	GtkRequisition vscrollbar_requisition;

	gtk_widget_get_child_requisition(scrolled_window->vscrollbar,
					 &vscrollbar_requisition);

	if (scrolled_window->window_placement == GTK_CORNER_TOP_RIGHT ||
	    scrolled_window->window_placement == GTK_CORNER_BOTTOM_RIGHT)
	    allocation->x += (vscrollbar_requisition.width +
			      SCROLLBAR_SPACING(scrolled_window));

	allocation->width = MAX(1, (gint) allocation->width -
				((gint) vscrollbar_requisition.width +
				 (gint) SCROLLBAR_SPACING(scrolled_window)));
    }
    if (scrolled_window->hscrollbar_visible)
    {
	GtkRequisition hscrollbar_requisition;

	gtk_widget_get_child_requisition(scrolled_window->hscrollbar,
					 &hscrollbar_requisition);

	if (scrolled_window->window_placement == GTK_CORNER_BOTTOM_LEFT ||
	    scrolled_window->window_placement == GTK_CORNER_BOTTOM_RIGHT)
	    allocation->y += (hscrollbar_requisition.height +
			      SCROLLBAR_SPACING(scrolled_window));

	allocation->height = MAX(1, (gint) allocation->height -
				 ((gint) hscrollbar_requisition.height +
				  (gint) SCROLLBAR_SPACING(scrolled_window)));
    }
}

static void
mac2_scrolled_window_size_allocate(GtkWidget * widget,
				    GtkAllocation * allocation)
{
    GtkScrolledWindow *scrolled_window;
    GtkBin *bin;
    GtkAllocation relative_allocation;
    GtkAllocation child_allocation;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_SCROLLED_WINDOW(widget));
    g_return_if_fail(allocation != NULL);

    scrolled_window = GTK_SCROLLED_WINDOW(widget);
    bin = GTK_BIN(scrolled_window);

    widget->allocation = *allocation;

    if (scrolled_window->hscrollbar_policy == GTK_POLICY_ALWAYS)
	scrolled_window->hscrollbar_visible = TRUE;
    else if (scrolled_window->hscrollbar_policy == GTK_POLICY_NEVER)
	scrolled_window->hscrollbar_visible = FALSE;
    if (scrolled_window->vscrollbar_policy == GTK_POLICY_ALWAYS)
	scrolled_window->vscrollbar_visible = TRUE;
    else if (scrolled_window->vscrollbar_policy == GTK_POLICY_NEVER)
	scrolled_window->vscrollbar_visible = FALSE;

    if (bin->child && GTK_WIDGET_VISIBLE(bin->child))
    {
	gboolean previous_hvis;
	gboolean previous_vvis;
	guint count = 0;

	do
	{
	    mac2_scrolled_window_relative_allocation(widget,
						      &relative_allocation);

	    child_allocation.x = relative_allocation.x + allocation->x;
	    child_allocation.y = relative_allocation.y + allocation->y;
	    child_allocation.width = relative_allocation.width;
	    child_allocation.height = relative_allocation.height;

	    previous_hvis = scrolled_window->hscrollbar_visible;
	    previous_vvis = scrolled_window->vscrollbar_visible;

	    gtk_widget_size_allocate(bin->child, &child_allocation);

	    /* If, after the first iteration, the hscrollbar and the
	     * vscrollbar flip visiblity, then we need both.
	     */
	    if (count &&
		previous_hvis != scrolled_window->hscrollbar_visible &&
		previous_vvis != scrolled_window->vscrollbar_visible)
	    {
		scrolled_window->hscrollbar_visible = TRUE;
		scrolled_window->vscrollbar_visible = TRUE;

		/* a new resize is already queued at this point,
		 * so we will immediatedly get reinvoked
		 */
		return;
	    }

	    count++;
	}
	while (previous_hvis != scrolled_window->hscrollbar_visible ||
	       previous_vvis != scrolled_window->vscrollbar_visible);
    }
    else
	mac2_scrolled_window_relative_allocation(widget, &relative_allocation);

    if (scrolled_window->hscrollbar_visible)
    {
	GtkRequisition hscrollbar_requisition;

	gtk_widget_get_child_requisition(scrolled_window->hscrollbar,
					 &hscrollbar_requisition);

	if (!GTK_WIDGET_VISIBLE(scrolled_window->hscrollbar))
	    gtk_widget_show(scrolled_window->hscrollbar);

	child_allocation.x = relative_allocation.x;
	if (scrolled_window->window_placement == GTK_CORNER_TOP_LEFT ||
	    scrolled_window->window_placement == GTK_CORNER_TOP_RIGHT)
	    child_allocation.y = (relative_allocation.y +
				  relative_allocation.height +
				  SCROLLBAR_SPACING(scrolled_window));
	else
	    child_allocation.y = GTK_CONTAINER(scrolled_window)->border_width;

	child_allocation.width = relative_allocation.width;
	child_allocation.height = hscrollbar_requisition.height;
	child_allocation.x += allocation->x;
	child_allocation.y += allocation->y;

	gtk_widget_size_allocate(scrolled_window->hscrollbar,
				 &child_allocation);
    }
    else if (GTK_WIDGET_VISIBLE(scrolled_window->hscrollbar))
	gtk_widget_hide(scrolled_window->hscrollbar);

    if (scrolled_window->vscrollbar_visible)
    {
	GtkRequisition vscrollbar_requisition;

	if (!GTK_WIDGET_VISIBLE(scrolled_window->vscrollbar))
	    gtk_widget_show(scrolled_window->vscrollbar);

	gtk_widget_get_child_requisition(scrolled_window->vscrollbar,
					 &vscrollbar_requisition);

	if (scrolled_window->window_placement == GTK_CORNER_TOP_LEFT ||
	    scrolled_window->window_placement == GTK_CORNER_BOTTOM_LEFT)
	    child_allocation.x = (relative_allocation.x +
				  relative_allocation.width +
				  SCROLLBAR_SPACING(scrolled_window));
	else
	    child_allocation.x = GTK_CONTAINER(scrolled_window)->border_width;

	child_allocation.y = relative_allocation.y;
	child_allocation.width = vscrollbar_requisition.width;
	child_allocation.height = relative_allocation.height;
	child_allocation.x += allocation->x;
	child_allocation.y += allocation->y;
	if (scrolled_window->window_placement == GTK_CORNER_TOP_RIGHT
	    && scrolled_window->hscrollbar_visible)
	    child_allocation.height += SCROLLBAR_SPACING(scrolled_window)
		+ scrolled_window->hscrollbar->requisition.height;

	gtk_widget_size_allocate(scrolled_window->vscrollbar,
				 &child_allocation);
    }
    else if (GTK_WIDGET_VISIBLE(scrolled_window->vscrollbar))
	gtk_widget_hide(scrolled_window->vscrollbar);
}

static void
mac2_range_remove_timer(GtkRange * range)
{
    g_return_if_fail(range != NULL);
    g_return_if_fail(GTK_IS_RANGE(range));

    if (range->timer)
    {
	gtk_timeout_remove(range->timer);
	range->timer = 0;
    }
    range->need_timer = FALSE;
}
