/* gt_display.c - Generic display driver
 *
 * Copyright (C) 1997, 1998, 1999 Free Software Foundation
 * Copyright (C) 1995, 1996 Eric M. Ludlam
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 * 
 *  This handles the driving of a generic display.  All display
 * management is handled either through curses, or X, depending on the
 * local flag set during intialization.
 * 
 * $Log: gt_display.c,v $
 * Revision 1.33  1999/11/29 17:02:45  zappo
 * Converted old crypt code into generic filter code.
 *
 * Revision 1.32  1999/11/23 14:24:15  zappo
 * Fixed refresh problem when runing in curses mode.
 *
 * Revision 1.31  1999/08/26 12:13:27  zappo
 * Fixed tracking eight_bit flag to be in the right direction.
 *
 * Revision 1.30  1999/08/26 11:57:04  zappo
 * Use 8bit flag to determine how to handle the highbit. (As meta or
 * not.)
 *
 * Revision 1.29  1998/10/29 17:11:20  zappo
 * GTK doesn't trim the status line right.  Trim here.
 *
 * Revision 1.28  1998/01/04 13:31:30  zappo
 * Fixed warnings.
 * DELETE and BS both cause the editkey[0] (delchar) in talkmode.
 *
 * Revision 1.27  1997/12/14 19:15:45  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.26  1997/11/01 13:28:22  zappo
 * Changed email.  Fixed send-file to correctly add the username into the prompt.
 *
 * Revision 1.25  1997/10/25 01:47:12  zappo
 * The add-text-to-window now will do it in batch, and also expand control
 * characters to their ^A notation.
 * Added user names to all prompts that affect them.
 *
 * Revision 1.24  1997/10/17 02:32:41  zappo
 * Changed interactive command to set the encryption type to use crypt_index
 *
 * Revision 1.23  1997/10/14 02:13:03  zappo
 * Added the arbitrary message command which sends a minibuffer message
 * to the remote (the kind of message long supported in the emacs interface)
 *
 * Revision 1.22  1997/10/07 00:05:48  zappo
 * Added many more interactive commands for all those things that were hard to
 * do in the past.  Commands that take a user are smart about asking which user
 * if there is only one other person connected.
 *
 * Revision 1.21  1997/07/27 17:13:23  zappo
 * Changed to use the generic list for all windows.
 *
 * Revision 1.20  1997/03/23 15:42:45  zappo
 * The abort command now falls-through to fix-windows.
 *
 * Revision 1.19  1997/03/12 02:27:44  zappo
 * Added new method suspend, and that is used for C-z.
 *
 * Revision 1.18  1997/03/12 01:32:01  zappo
 * Added preprocessing to input keys so both C-h and DEL turn into the TERM
 * specified RUBOUT character, and changed many ANSI decls into stdc decls.
 *
 * Revision 1.17  1997/03/12 00:24:16  zappo
 * Added checks for the new AutoClean flag.  Changed how some items are freed.
 * C-c will now act as M-a if a call is active.  If no call is active, it
 * exits.
 *
 * Revision 1.16  1997/02/23 03:20:00  zappo
 * Added API so that X could cancel prompts when needed.
 *
 * Revision 1.15  1997/02/20 02:32:24  zappo
 * Ctxt less printing now uses the same method as a Socket print.
 *
 * Revision 1.14  1997/02/01 14:26:53  zappo
 * Added active encryption types to the mode line.  USER_send/read now
 * take Ctxt.  Removed X differential on what is displayed in minibuffer
 * (was adding a cursor).  Added C-g/c exit to prompts with beep & Quit msg.
 * Added editchars to minibuffer (delete word/line)
 *
 * Revision 1.13  1996/07/27  21:28:25  zappo
 * Replaced send routines with USER_send so encryption is maintained
 *
 * Revision 1.12  1996/03/29  04:26:19  zappo
 * Screen out C-l sent by remote, and send them in curses or X modes
 *
 * Revision 1.11  1996/02/26  00:17:59  zappo
 * Added include of gtproc.h, and fixed some warnings
 *
 * Revision 1.10  1996/02/01  01:15:02  zappo
 * Added the hook to allow shared apps to capture some keyboard elements.
 *
 * Revision 1.9  1996/01/20  19:08:33  zappo
 * Updated display code to be cleaner by using a Methods structure.
 *
 * Revision 1.8  1995/12/09  23:58:28  zappo
 * Added blurb buffer.  Removed query for personal name (now in context),
 * added context storage in DISP_message so it can be called from library
 * routines. YORN is now BLOCKS ALL OTHER IO preventing allowing inline
 * user queries.  Fixed hangup to use this method for each active user.
 *
 * Revision 1.7  1995/11/21  03:40:47  zappo
 * Updated yes_or_no to block all other threads while waiting for an
 * answer.   This allows asking questions during potentially
 * polysynchronous (real word?) situations.
 *
 * Revision 1.6  1995/09/29  09:23:47  zappo
 * Added YORN (Yes OR No) state and prompt code, updated references to
 * user state, and client type, updated printing of statusline, and
 * CURS_init call, and _beep function.  Added DIPS_window_displayed to
 * return true if window is being displayed.  Added bell to C-g input.
 *
 * Revision 1.5  1995/09/22  13:40:22  zappo
 * Added codes needed to optionally have the X toolkit, or curses, or
 * both interfaces removed from the binary.
 *
 * Revision 1.4  1995/09/20  23:17:06  zappo
 * Added the X interface in next to the curses interface.  Added some new
 * commands (like fix-windows, and hangup)
 *
 * Revision 1.3  1995/09/14  11:33:29  zappo
 * Added reply command.
 *
 * Revision 1.2  1995/09/14  10:41:12  zappo
 * Successfully extracted the curses interface, and left a few stubs in
 * for use with X11 interface when I get time for that task.
 *
 * Revision 1.1  1995/08/25  16:46:23  zappo
 * Initial revision
 *
 * Tokens: ::Header:: gtalkc.h
 */

#include "gtalklib.h"
#include "gtalkc.h"
#include "gtproc.h"

#ifndef NO_IFACE

#if HAVE_SIGNAL_H == 1
#include <signal.h>
#endif

#if defined USE_CURSES && defined X_TOOLKIT
#define IELSE else
#else
#define IELSE
#endif /* more than one interface */

static void DISP_redraw(struct TalkContext *Ctxt);
static void DISP_text_to_window(struct TalkContext *Ctxt,
				char *editkeys, char fill, 
				char *text, int size,
				struct WindowList *win);

/* The io state determines what is currently going on on the screen.
 * IDLE means normal talk things are going on, META means we are
 * waiting for a meta-key to be hit.  PROMPT means we are collecting a
 * string for input.  DIALOG means we a dialog is up, and ESC will exit
 */
enum IOState { TALKMODE, META, PROMPT, DIALOG, YORN };
static enum IOState state;
/* some interfaces may wish to perform actions thruogh keyboard
 * events. Thia lets them suppress the usual feedback generated
 * by doing that.
 */
int suppress_feedback = 0;

enum PromptType { METAX, CALLUSER, 
		  ADD_FILTER_WHO, ADD_FILTER_TYPE,
		  APPLICATION_WHO, APPLICATION_NAME,
		  SEND_WHO, SEND_FILE,
		  DOWNLOAD_PATH,
		  ARBITRARY_WHO,
		  ARBITRARY_MESSAGE
};
static enum PromptType prompt_state;

struct UserObject *entered_user;

static void *popup = NULL;	/* popup window which may be around */
static int   popup_full = 0;	/* set when functions which fill popups 
				 * are done */

static int num_windows = 0;	/* keep track of the number of windows    */
static MakeList(list);		/* List of windows */

static int MSG = 0;

/* this is a callback installed when the user needs to answer a Y/N question 
 */
static char *yorn_prompt = NULL; /* prompt for re-display */
static int   yorn_answer = -1;	 /* answer to be returned */

/* This structure pointer will contain all the currently valid methods
 * for performing certain types of operations
 */
static struct IfaceMethods *CurrentMethods = NULL;

/* This is our copyright information we want everyone to see.
 */
static char blurb[] =
"   Thanks for using GNU talk!\n\n\
This program is free under the GNU general public license.\n\
For copyright and warranty information type:\n\
       <ESC>-x SHOW COPYRIGHT <RET>\n\n\
See the GNU COPYING file for additional details.\n\n\
Please report Bugs/Problems/Suggestions to: Eric Ludlam via\n\
                                            zappo@ultranet.com";

/*
 * Function: set_window_statusline
 *
 *   Locally defined function which will set the contents of the
 * status line to be relavent to the person in that window.
 *
 * Returns:     static void  - 
 * Parameters:  TalkContext -  Talk Context
 *              user        - Pointer to user
 * History:
 * zappo   8/6/95     Created
 */
static void set_window_statusline(Ctxt, win)
     struct TalkContext *Ctxt;
     struct WindowList *win;
{
  /* Most of the strings in these two items will never be used,
   * but I leave them in for completeness
   */
  static char *states[] = { "NEW", "OK", "WAITING", "CLOSED",
			    "CALLING" };
  static char *clients[] = { "Unknown", "Data Link", "Subprocess", 
			     "BSD talk", "Emacs talk", "GNU talk", "ytalk" };
  char buffname[100];
  char statusbuff[100];
  char totalbuff[200];
  int            i;

  if(win->user)
    {
      strcpy(buffname, win->user->name);

      if(win->user->remote)
	{
	  strcat(buffname, "@");
	  strcat(buffname, win->user->remote->host->name);
	}

      if(win->user->longname)
	{
	  strcat(buffname, " {");
	  strcat(buffname, win->user->longname);
	  strcat(buffname, "}");
	}

      strcpy(statusbuff, states[win->user->state]);

      /* display the current filters on the different
       * connections to this user.  Conserve space whenever possible
       */
      if(win->user->outbound_filter != 0)
	{
	  if(win->user->inbound_filter == win->user->outbound_filter)
	    strcat(statusbuff, " ><");
	  else
	    strcat(statusbuff, " >");
	  strcat(statusbuff, FILTER_name(win->user->outbound_filter));
	}

      if((win->user->inbound_filter != 0) && 
	 ((win->user->inbound_filter != win->user->outbound_filter)))
	{
	  if(win->user->outbound_filter != 0)
	    strcat(statusbuff, "<");
	  else
	    strcat(statusbuff, " <");
	  strcat(statusbuff, FILTER_name(win->user->inbound_filter));
	}

      if(win->user->state == USER_CONNECTED)
	{
	  strcat(statusbuff, " : ");
	  strcat(statusbuff, clients[win->user->type]);
	}
    }
  else
    {
      strcpy(buffname, Ctxt->myprefname);

      sprintf(statusbuff, "%s : ESC-h for help",
	      (Ctxt->runstate == Curses)?"Curses":"X");
    }

  /* Create the status line text */
  sprintf(totalbuff, " %-30s (%s)---------------------------",
	  buffname, statusbuff);
  i = strlen(totalbuff);

  if(!CurrentMethods) return;

  while(CurrentMethods->win_width(win->statusline) > i)
    totalbuff[i++] = '-';
  totalbuff[i++] = '-';
  totalbuff[i++] = 0;

  /* Now that we did all that work, lets force the line to be short */
  totalbuff[CurrentMethods->win_width(win->statusline)] = 0;

  CurrentMethods->set_label(win->statusline, totalbuff);
}

/*
 * Function: alloc_win
 *
 *   Locally defined function which allocates and initializes a new
 * window struct, which can be stacked by special stacking routines.
 *
 * Returns:     static void  - 
 * Parameters:  Ctxt - Context
 *              user - Pointer to user
 * History:
 * zappo   8/3/95     Created
 */
static void alloc_win(Ctxt, user)
     struct TalkContext *Ctxt;
     struct UserObject *user;
{
  struct WindowList *new;

  new = (struct WindowList *)LIST_alloc(&list, sizeof(struct WindowList));

  if(!new)
    {
      gtalk_shutdown("malloc failure in alloc_win!\n");
    }
  num_windows++;

  new->user = user;
  if(user) user->window = new;

  DISP_fix_windows(Ctxt);

  set_window_statusline(Ctxt, new);

  DISP_redraw(Ctxt);

  return;
}


/*
 * Function: DISP_init
 *
 *   Initializes the curses and Xt or GTK display.
 *  Only called when not running as an etalk subprocess.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   8/3/95     Created
 */
void DISP_init(Ctxt)
     struct TalkContext *Ctxt;
{
#ifdef USE_CURSES
  /* If command line parameter defaults us to Curses, then short
     circuit prevents X from initializing */
  if((Ctxt->runstate == Curses)
#ifdef X_TOOLKIT
     || ((CurrentMethods = X_init(Ctxt)) == NULL)
#endif
     )
    {
      Ctxt->runstate = Curses;
      CurrentMethods = CURS_init(Ctxt);
    }
#else
#ifdef X_TOOLKIT
  /* If there is no curses interface, error on failed init */
  if((CurrentMethods = X_init(Ctxt)) == NULL)
    {
      gtalk_shutdown("Could not open display.\n");
    }
#endif /* X_TOOLKIT */
#endif /* USE_CURSES */

#ifdef X_TOOLKIT
  if(Ctxt->runstate == X)
    {
      Ctxt->runstate = X;
    }
#endif

  if(!CurrentMethods)
    gtalk_shutdown("Unexpected error initializing interface.");

  /* This is the local information buffer */
  alloc_win(Ctxt, NULL);
  ((struct WindowList *)FirstElement(list))->autofill = 1;

  CurrentMethods->put_text(((struct WindowList *)FirstElement(list))->window,
			   blurb, sizeof(blurb)-1);
  CurrentMethods->refresh_window
    (((struct WindowList *)FirstElement(list))->window, TRUE);
}


/*
 * Function: DISP_close
 *
 *   Cleanly closes any user interfaces which may be currently active.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   8/3/95     Created
 */
void DISP_close(Ctxt)
     struct TalkContext *Ctxt;
{
  if(CurrentMethods) CurrentMethods->close();
}

/*
 * Function: DISP_beep
 *
 *   Locally defined function which lets us ring the bell a little
 *
 * Returns:     static void  - 
 * Parameters:  None
 *
 * History:
 * zappo   9/21/95    Created
 */
static void DISP_beep(Ctxt)
     struct TalkContext *Ctxt;
{
  if(CurrentMethods) CurrentMethods->beep(Ctxt);
}

/*
 * Function: DISP_fix_windows
 *
 *   Do what is needed to fix the windows so the sizes are correct
 * based on some change.
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   9/14/95    Created
 */
void DISP_fix_windows(Ctxt)
     struct TalkContext *Ctxt;
{
  if(Ctxt->runstate == Socket) return;

  if(CurrentMethods) CurrentMethods->fix_windows(Ctxt, list.first);

  /* Whenever a hangup or similar activity occurs, this function is
   * called.  If autoclean is active, then the user struct and other
   * items have already been cleaned up, so lets ditch the last of
   * these windows too. We do this after updating so that if we
   * have deleted windows, it can get rid of them. */
  if(Ctxt->cleanup == AutoClean) DISP_clean();
}


/*
 * Function: DISP_window_displayed
 *
 *   Returns non-0 iff the window passed in is displayable, meaning,
 * it has windows in it's structure, is currently attached to a
 * network, and is of a type which displays data.
 *   The local window is identified with PREV == NULL.  Windows where
 * USER == NULL have been hung up upon. (Except for local window).
 * If the user flag is still active, make sure we should display it based
 * on it's flags.
 *
 * Returns:     int  - 
 * Parameters:  win - Pointer toWindow win
 *
 * History:
 * zappo   9/26/95    Created
 */
int DISP_window_displayed(win)
     struct WindowList *win;
{
  if(win == NULL) {
    fprintf(stderr, "DISP_window_displayed: win == NULL. Bad!\n");
  }

  return ((win                 != NULL)             &&
	  /* win->link.prev == NULL means the local window (always up) */
	  ((win->link.prev     == NULL)             || 
	   ((win->user           != NULL)             &&
	    (win->user->type     != DATALINK)         &&
	    (win->user->type     != SUBPROCESS)       &&
	    (win->user->state    != USER_UNREACHABLE) &&
	    (win->user->state    != USER_CLOSED))));
}


/*
 * Function: DISP_new_user
 *
 *   Prepare a window for a new user to be added into the list of windows
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              user - Pointer to user
 * History:
 * zappo   8/6/95     Created
 */
void DISP_new_user(Ctxt, user)
     struct TalkContext *Ctxt;
     struct UserObject *user;
{
  alloc_win(Ctxt, user);
}


/*
 * Function: DISP_clean, IsDead
 *
 *   Goes through the list of windows, and frees up all extra space.
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   8/31/95    Created
 */
static unsigned char IsDead(w, criteria)
     struct WindowList *w;
     void *criteria;
{
  return (DISP_window_displayed(w) == 0);
}

void DISP_clean()
{
  struct WindowList *found;

  /* loop over all dead windows */
  while((found = (struct WindowList *)LIST_find(&list, IsDead, NULL)) != NULL)
    {
      if (found->user != NULL) 
	{
	  /* Make sure that user struct doesn't backreference us. */
	  found->user->window = NULL;
	}
      LIST_deallocate(&list, found);
    }  
}
#endif /* NO_IFACE */

/*
 * Function: DISP_update_user, MatchName
 *
 *   Updates the status line for a given user.  If the state is
 * closed, we can remove the window from the screen.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              user - Pointer to user
 * History:
 * zappo   8/26/95    Created
 */
static unsigned char MatchName(w, user)
     struct WindowList *w;
     struct UserObject *user;
{
  return (w->user == user);
}

void DISP_update_user(Ctxt, user)
     struct TalkContext *Ctxt;
     struct UserObject *user;
{
#ifndef NO_IFACE
  struct WindowList *found;

  if(Ctxt->runstate == Socket) return;

  found = (struct WindowList *)LIST_find(&list, MatchName, user);

  if(found && DISP_window_displayed(found))
    set_window_statusline(Ctxt, found);
  else
    {
      DISP_fix_windows(Ctxt);
      DISP_redraw(Ctxt);
    }
#endif /* NO_IFACE */
}
#ifndef NO_IFACE


/*
 * Function: DISP_text_to_window
 *
 *   Locally defined function which handles the placing of a string of
 * characters into a window list.  The string is an input with embeded
 * edit characters.
 *
 * Returns:     static void  - 
 * Parameters:  Ctxt - Context
 *              text - Pointer toCharacter of text
 *              size - Number of size
 * History:
 * zappo   8/29/95    Created
 */
static void DISP_text_to_window(Ctxt, editkeys, fill, text, size, win)
     struct TalkContext *Ctxt;
     char *editkeys, fill, *text;
     int size;
     struct WindowList *win;
{
  int i;
  int start, end;

  if(!CurrentMethods) return;

  /* Add text to the window... */
  for(i = 0, start = 0, end = 0; i < size; i++)
    {
      if((text[i] < ' ') || (text[i] == 127))
	{
	  /* Here we have encountered a special char we have to deal with.
	   * In this case, we either convert it to text, or we will
	   * expand it.  Either way, the print-buffer needs to be flushed.
	   */

	  if(end > start)
	    {
	      CurrentMethods->put_text(win->window, text+start, end-start);
	      start = end;
	    }
	  
#ifdef CHAR_PLAY
	  /* Turn this off.  I think it's wrong */
	  if((win == FirstElement(&list)) &&
	     /*
	      * Delete char is usually C-h or C-?, but with all the new shells
	      * these days, you can never tell which.  Preprocess each 
	      * character and turn it into whatever our editkey is.
	      * Only do this to our local window though...
	      */
	     ((text[i] == 8) || (text[i] == 127)))  /* C-h and DEL */
	    {
	      text[i] = editkeys[0]; /* preprocess that value in */
	    }
#endif
	  /*
	   * Now examine my edit keys
	   */
	  if(text[i] == editkeys[0]) /* delete backwards one */
	    {
	      CurrentMethods->delete_char(win->window);
	    }
	  else if(text[i] == editkeys[1]) /* delete this line */
	    {
	      CurrentMethods->delete_line(win->window);
	    }
	  else if(text[i] == editkeys[2]) /* delete a word */
	    {
	      CurrentMethods->delete_word(win->window);
	    }
	  else if(text[i] == 7)	/* Bell */
	    {
	      DISP_beep(Ctxt);
	    }
	  else if(text[i] == 12)	/* C-l (refresh screen) */
	    {
	      /* DISP_message(Ctxt, "Remote had messy screen."); */
	    }
	  else if((text[i] == 13) || (text[i] == 10))
	    {
	      /* Add linefeed because moving the cursor won't force a scroll.
		 Then change the text field too.  If we are interactive, the
		 send is next, and will place this into the input stream
		 instead. */
	      CurrentMethods->put_text(win->window, "\n", 1);
	      text[i] = 10;
	    }
	  else
	    {
	      /* Here we take the control character, and expand it to
	       * a couple of characters.
	       */
	      char ctrlbuff[3];
	      ctrlbuff[0] = '^';
	      ctrlbuff[1] = text[i] + 'A' - 1;
	      ctrlbuff[2] = 0;
	      CurrentMethods->put_text(win->window, ctrlbuff, 2);
	    }
	}
      /* The below test could fail if end-start + column > fill_column */
      else if(fill && (text[i] == ' ') && 
	      CurrentMethods->at_fill_column(win->window))
	{
	  if(end > start)
	    {
	      CurrentMethods->put_text(win->window, text+start, end-start);
	      start = end;
	    }
	  /* Add linefeed because moving the cursor won't force a scroll.
	     Then change the text field too.  If we are interactive, the
	     send is next, and will place this into the input stream
	     instead. */
	  CurrentMethods->put_text(win->window, "\n", 1);
	  text[i] = 10;
	}
      else
	{
	  end++;
	}
    }
  /* After scanning the string, pump it out to the window... */
  if(end > start)
    {
      CurrentMethods->put_text(win->window, text+start, end-start);
      start = end;
    }
  CurrentMethods->refresh_window(win->window, FALSE);
}

/*
 * Function: DISP_local_string
 *
 *   Displays TEXT into the local users window, and also sends that
 * text to all remotes currently attached.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              text - Pointer toCharacter of text
 *              size - Number of size
 * History:
 * zappo   8/26/95    Created
 */
void DISP_local_string(Ctxt, text, size)
     struct TalkContext *Ctxt;
     char *text;
     int size;
{
  struct WindowList *loop;

  /* Local is always the FIRST window */

  /* Add to window */
  DISP_text_to_window(Ctxt, Ctxt->editkeys, 
		      ((struct WindowList *)FirstElement(list))->autofill,
		      text, size, FirstElement(list));
  
  /* Send this text to all subwindows... */
  for(loop = FirstElement(list); loop; loop = NextElement(loop))
    {
      if(loop->user && DISP_window_displayed(loop))
	{
	  USER_send(Ctxt, loop->user, text, size);
	  /* GT_send(loop->user->remote, text, size); */
	}
    }
}

/*
 * Function: DISP_remote_string
 *
 *   Display's characters in the CURSES window owned by USER.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              text - Pointer to Character of text
 *              size - size of text
 *              user - Pointer to user
 * History:
 * zappo   8/26/95    Created
 */
void DISP_remote_string(Ctxt, text, size, user)
     struct TalkContext *Ctxt;
     char *text;
     int size;
     struct UserObject *user;
{
  static char *errmsg = "etalk error: No window binding for TCP socket.\n";
  struct WindowList *loop;

  /* Get the window for this user... */
  loop = (struct WindowList *)LIST_find(&list, MatchName, user);

  if(!loop) 
    {
      USER_send(Ctxt, user, errmsg, strlen(errmsg));
      return;
    }

  /* Put in this users window */
  DISP_text_to_window(Ctxt, loop->user->editkeys, 0, text, size, loop);

  /* X needs a little help pushing changes to the display */
  CurrentMethods->flush_io(Ctxt, NULL);
}

/*
 * Function: DISP_create_popup
 *
 *   Creates a new  window which will contain all messages about to be
 * recieved.  Waits for the input ESC before disapearing.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt   - Context
 *              width  - Number of width
 *              height - Number of height
 * History:
 * zappo   8/25/95    Created
 */
void DISP_create_popup(Ctxt, width, height)
     struct TalkContext *Ctxt;
     int width, height;
{
  /* Only do this if we arn't in curses mode! */
  if(Ctxt->runstate == Socket) return;

  popup = CurrentMethods->create_popup(width, height);

  state = DIALOG;
}
#endif /* NO_IFACE */

/*
 * Function: DISP_message
 *
 *   Displays a message in the echo area below the talk windows.  this
 * makes it appear vaguely like an EMACS window...
 *
 *   This should really take printf parameters, but not all systems
 * support varargs macros (like linux 0.99p14 kernel slackware...)
 *
 * Returns:     Nothing
 * Parameters:  Ctxt    - Context
 *              message - Pointer toCharacter of message
 * History:
 * zappo   8/6/95     Created
 */
void DISP_message(Ctxt, message)
     struct TalkContext *Ctxt;
     char *message;
{
  static struct TalkContext *BkCtxt = NULL;

  /* Make sure we have a backup context */
  if(Ctxt) BkCtxt = Ctxt;

  /* Can't do much without a context.  Stdout is the only way.
   * This is the same method as using a Socket. */
  if((!BkCtxt) || (BkCtxt->runstate == Socket))
    {
      printf(message);
      if(message[strlen(message)] != '\n')
	printf("\n");
    }
#ifndef NO_IFACE
  /* Only put stuff in popup if we havn't overrun the size... */
  else if(popup && !popup_full)
    {
      /* If there is an embeded code, remove it... */
      while(*message && (*message < ' ')) message++;

      CurrentMethods->put_text(popup, message, strlen(message));
    }
  else
    {
      /* If there is an embeded code, remove it... */
      while(*message && (*message < ' ')) message++;

      CurrentMethods->echo_area_message(message, FALSE);
    }
#endif /* NO_IFACE */
}
#ifndef NO_IFACE

/*
 * Function: mention_user, DISP_prompt
 *
 *   Display a promted for string in the echo_area. Basically calls
 * DISP_message with the combined version of prompt and collected.
 *
 * Mention_user is for formatting a prompt w/ a UserObject's name
 * formatted in it.  It is a convenience.
 * 
 * Returns:     Nothing
 * Parameters:  Ctxt      - Context
 *              prompt    - Pointer toCharacter of prompt
 *              collected - Pointer toCharacter of collected
 * History:
 * zappo   8/24/95    Created
 */
static char *mention_user(fmt, uo)
     char             *fmt;
     struct UserObject *uo;
{
  static char buffer[256];
  if(uo->name == NULL) return NULL;
  sprintf(buffer,fmt,uo->name);
  return buffer;
}
void DISP_prompt_string(Ctxt, prompt, collected)
     struct TalkContext *Ctxt;
     char *prompt, *collected;
{
  static char *lastprompt = NULL;
  char *msg;
  
  if(prompt)
    lastprompt = prompt;

  msg = (char *)malloc(strlen(lastprompt) + strlen(collected) + 1);

  sprintf(msg, "%s%s", lastprompt, collected);

  /* Safe because we never prompt outside of a controlled interface */
  CurrentMethods->echo_area_message(msg, TRUE);

  free(msg);
}


/*
 * Function: DISP_redraw
 *
 *   Locally defined function which will do magic curses things to
 * redraw the screen when someone types ^L
 *
 * Returns:     static void  - 
 * Parameters:  None
 *
 * History:
 * zappo   8/25/95    Created
 */
static void DISP_redraw(Ctxt)
     struct TalkContext *Ctxt;
{
  CurrentMethods->refresh_screen(FirstElement(list));
}

/*
 * Function: DISP_help
 *
 *   Provides help for curses interface specifics
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   8/25/95    Created
 */
void DISP_help(Ctxt)
     struct TalkContext *Ctxt;
{
  DISP_create_popup(Ctxt, 56, 18);
		    
  DISP_message(Ctxt, "                   Interface help:");
  DISP_message(Ctxt, " ESC   removes dialog boxes  C-z   suspend etalk");
  DISP_message(Ctxt, " C-l   refresh display       C-c   exit etalk");
  DISP_message(Ctxt, "                   META commands:");
  DISP_message(Ctxt, " ESC-c Call user             ESC-H Hangup connections");
  DISP_message(Ctxt, " ESC-r Reply to talk request ESC-a Abort current call");
  DISP_message(Ctxt, " ESC-f Fix window sizes      ");
  DISP_message(Ctxt, " ");
  DISP_message(Ctxt, " ESC-F Filter a connection   ESC-R Toggle ringer on/off");  
  DISP_message(Ctxt, " ESC-A Start Application     ESC-T Toggle TCP Accept");
  DISP_message(Ctxt, " ESC-s Send a file           ESC-D Set download path");
  DISP_message(Ctxt, " ESC-h this help message     ESC-M Send mini message");
  DISP_message(Ctxt, " ESC-q Hangup & quit.");
  DISP_message(Ctxt, " ");
  DISP_message(Ctxt, " ESC-x COMMAND   enter etalk commands");
  DISP_message(Ctxt, " ESC-x HELP      help on etalk commands");
}
#endif /* NO_IFACE */


/*
 * Function: DISP_reset_any_prompts
 *
 *   Resets any existing prompt modes.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   2/22/97    Created
 */
void DISP_reset_any_prompts(Ctxt)
     struct TalkContext *Ctxt;
{
  static void DISP_prompt(struct TalkContext *Ctxt, char in);

  /* If in a prompt, erase it */
  if(state == PROMPT) DISP_prompt(Ctxt, 21);
  state = TALKMODE;
 }

void DISP_suppress_feedback(Ctxt, suppress)
     struct TalkContext *Ctxt;
     int suppress;
{
  suppress_feedback = suppress;
}

/*
 * Function: DISP_yes_or_no
 *
 *   Ask a Y or N question.  Waits for user response, blocking all
 * other IO until the choice has been made.  In Socket mode, have
 * interface ask the question and read from it.
 *
 * Returns:     int  - 
 * Parameters:  Ctxt  - Context
 *              query - Pointer toCharacter of query
 * History:
 * zappo   9/24/95    Created
 */
int DISP_yes_or_no(Ctxt, query)
     struct TalkContext *Ctxt;
     char               *query;
{
  if(Ctxt->runstate == Socket)
    {
      char buff[100];
      /* In subprocess mode: Output query, read from stdin one answer */
      printf("%c%c%s\n", ETALK_ESCAPE, TTY_YORN, query);
      if(GT_recv(Ctxt->tty, buff, sizeof(buff)) == Fail)
	return Fail;
      else
	return ((buff[0] == 'y') || (buff[0] == 'Y'));
    }
#ifndef NO_IFACE
  else
    {
      if(state != TALKMODE)
	{
	  return Fail;
	}
      
      state = YORN;
      yorn_prompt = query;
      yorn_answer = -1;
      DISP_prompt_string(Ctxt, yorn_prompt, " (Y/n)");
      /* Now prepare to halt other threads of execution.  The Ctxt->tty is
       * either emacs's tty, the Curses input device, or the X windows
       * device (where refresh, and user interaction is recieved)
       */
      if(GT_select_one(Ctxt, Ctxt->tty))
	{
	  /* In this case, we successfully read something */
	  return yorn_answer;
	}
      else
	{
	  /* Timeout is simulated as a NO answer to our query, as we can't
	   * always wait forever! */
	  return FALSE;
	}
    }
#endif /* NO_IFACE */
}
#ifndef NO_IFACE

/*
 * Function: DISP_meta, prompt, talkmode, dialog
 *
 *   Locally defined functions which interpret an input character
 * based on the current input mode of the talk application
 *
 * Returns:     static void  - 
 * Parameters:  Ctxt - Context
 *              in   - Character of in
 * History:
 * zappo   8/25/95    Created
 */
static void DISP_meta(Ctxt, in)
     struct TalkContext *Ctxt;
     char in;
{
  char o[60];
  struct UserObject *uo;
  sprintf(o, "M-%c", in);
  DISP_message(Ctxt, o);	/* display char before using... */
  
  switch(in)
    {
    case 'c':
      DISP_prompt_string(Ctxt, "Call user: ", "");
      state = PROMPT;
      prompt_state = CALLUSER;
      MSG = TRUE;
      break;
    case 'F':
      uo = USER_OnlyOne();
      if(uo == (struct UserObject *)1) {
	DISP_message(Ctxt, "No active user to add a filter to.");
	DISP_beep(Ctxt);
	state = TALKMODE;
      } else {
	if(uo == NULL) {
	  DISP_prompt_string(Ctxt, "Add Filter for User: ", "");
	  prompt_state = ADD_FILTER_WHO;
	} else {
	  entered_user = uo;
	  DISP_prompt_string(Ctxt, mention_user("%s Filter Method: ",
						entered_user),
			     "");
	  prompt_state = ADD_FILTER_TYPE;
	}
	state = PROMPT;
      }
      MSG = TRUE;
      break;
    case 'A':
      uo = USER_OnlyOne();
      if(uo == (struct UserObject *)1) {
	DISP_message(Ctxt, "No active user to share an application with.");
	DISP_beep(Ctxt);
	state = TALKMODE;
      } else {
	if(uo == NULL) {
	  DISP_prompt_string(Ctxt, "Start Application with: ", "");
	  prompt_state = APPLICATION_WHO;
	} else {
	  entered_user = uo;
	  DISP_prompt_string(Ctxt, mention_user("%s's Application: ",
						entered_user), "");
	  prompt_state = APPLICATION_NAME;
	}
	state = PROMPT;
      }
      MSG = TRUE;
      break;
    case 's':
      uo = USER_OnlyOne();
      if(uo == (struct UserObject *)1) {
	DISP_message(Ctxt, "No active user to send a file to.");
	DISP_beep(Ctxt);
	state = TALKMODE;
      } else {
	if(uo == NULL) {
	  entered_user = uo;
	  DISP_prompt_string(Ctxt, "Send File To: ", "");
	  prompt_state = SEND_WHO;
	} else {
	  entered_user = uo;
	  DISP_prompt_string(Ctxt, mention_user("%s file to send: ",
						entered_user), "");
	  prompt_state = SEND_FILE;
	}
	state = PROMPT;
      }
      MSG = TRUE;
      break;
    case 'D':
      DISP_prompt_string(Ctxt, "New Download path: ", 
			 /* Display current path only temporarily */
			 Ctxt->downloadpath?Ctxt->downloadpath:"");
      state = PROMPT;
      prompt_state = DOWNLOAD_PATH;
      MSG = TRUE;
      break;
    case 'M':
      uo = USER_OnlyOne();
      if(uo == (struct UserObject *)1) {
	DISP_message(Ctxt, "No active user to send a message to.");
	DISP_beep(Ctxt);
	state = TALKMODE;
      } else {
	if(uo == NULL) {
	  DISP_prompt_string(Ctxt, "Send Message to: ", "");
	  prompt_state = ARBITRARY_WHO;
	} else {
	  entered_user = uo;
	  DISP_prompt_string(Ctxt, mention_user("%s Message: ", 
						entered_user), "");
	  prompt_state = ARBITRARY_MESSAGE;;
	}
	state = PROMPT;
      }
      MSG = TRUE;
      break;
    case 'T':
      /* Toggle the accept method */
      switch (Ctxt->querynewtcp) {
      case RefuseConnect:
	Ctxt->querynewtcp = QueryConnect;
	DISP_message(Ctxt, "New Connect method: Query before accepting.");
	break;
      case AlwaysConnect:
	Ctxt->querynewtcp = RefuseConnect;
	DISP_message(Ctxt, "New Connect method: Never accept new connections.");
	break;
      default:
	Ctxt->querynewtcp = AlwaysConnect;
	DISP_message(Ctxt, "New Connect method: Always accept new connections.");
	break;
      }
      state = TALKMODE;
      MSG = TRUE;
      break;
    case 'h':
      DISP_help(Ctxt);
      MSG = TRUE;
      break;
    case 'q':
      state = TALKMODE;
      if(DISP_yes_or_no(Ctxt, "Really exit GNU talk?"))
	{
	  gtalk_shutdown(NULL);
	}
      break;
    case 'r':
      /* Replies require NO extra information... */
      PROTOCOL_reply(Ctxt);
      /* We always go into talk mode! */
      state = TALKMODE;
      MSG = TRUE;		/* the response of the reply. */
      break;
    case 'R':
      /* Toggle whatever it is currently set to */
      if(Ctxt->ringerflag == TRUE)
	{
	  if(RING_deactivate(Ctxt->myname))
	    {
	      DISP_message(Ctxt, "Ringer successfully deactivated.");
	    }
	  else
	    {
	      DISP_message(Ctxt, "Unable to deactivate ringer.");
	    }
	}
      else
	{
	  if(RING_activate(Ctxt->myname, &Ctxt->ringerflag, Ctxt->udp_ring, ETR_read))
	    {
	      DISP_message(Ctxt, "Ringer successfully activated.");
	    }
	  else
	    {
	      DISP_message(Ctxt, "Unable to activate ringer.");
	    }
	}
      state = TALKMODE;
      MSG = TRUE;
      break;
    case 'a':
      PROTOCOL_abort(Ctxt);
    case 'f':
      DISP_fix_windows(Ctxt);
      state = TALKMODE;
      break;
    case 'H':
      /* Hanup on all active connections */
      state = TALKMODE;
      {
	struct WindowList *loop;
	for(loop = FirstElement(list); loop;)
	  {
	    if(loop->user && 
	       ((loop->user->state == USER_CONNECTED) ||
		(loop->user->state == USER_CALLING)))
	      {
		if(DISP_yes_or_no(Ctxt, mention_user("Hangup on %s?",
						     loop->user)))
		  {
		    USER_hangup(Ctxt, loop->user->id);
		    /* Advance the loop here, then call fix windows... */
		    loop = NextElement(loop);
		    DISP_fix_windows(Ctxt);
		    DISP_redraw(Ctxt);
		  } else {
		    loop = NextElement(loop);
		  }
	      } else {
		loop = NextElement(loop);
	      }
	  }
      }
      MSG = TRUE;
      break;
    case 'x':
      DISP_prompt_string(Ctxt, "M-x ", "");
      state = PROMPT;
      prompt_state = METAX;
      MSG = TRUE;
      break;
    case 3:			/* C-c */
    case 7:			/* c-g */
    case 27:			/* ESC */
      state = TALKMODE;
      DISP_beep(Ctxt);
      DISP_message(Ctxt, "Quit");
      MSG = TRUE;
      break;
    default:
      strcat(o, " ... Invalid escape");
      DISP_message(Ctxt, o);
      state = TALKMODE;
      MSG = TRUE;
      break;
    }
}

static void DISP_prompt(Ctxt, in)
     struct TalkContext *Ctxt;
     char in;
{
#define COLLECTSIZE 80
  static char collect[COLLECTSIZE] = "";

  /* Permutate our edit keys into known constants */
  if(in == Ctxt->editkeys[0])
    in = 8;			/* backspace */
  else if(in == Ctxt->editkeys[1])
    in = 21;			/* C-u */
  else if(in == Ctxt->editkeys[2])
    in = 23;			/* C-w */


  switch(in) 
    {
    case 3:			/* C-c */
    case 7:			/* c-g */
    case 27:			/* ESC */
      state = TALKMODE;
      collect[0] = 0;	    
      DISP_beep(Ctxt);
      DISP_message(Ctxt, "Quit");
      break;
    case 13:			/* CR */
    case 10:			/* LF */
      {
	char col_buff[COLLECTSIZE];
	strcpy(col_buff, collect);
	collect[0] = 0;
	state = TALKMODE;
	/* Clear the prompt area... */
	DISP_message(Ctxt, "");
	if(col_buff[0] != 0)	/* only do something if they typed something */
	  {
	    /* Run this through the command interpreter... */
	    switch(prompt_state)
	      {
	      case METAX:
		GTC_interpret_string(Ctxt, col_buff);
		break;
	      case CALLUSER:
		LOCAL_new_user_tty(Ctxt, col_buff, NULL);
		break;
	      case ADD_FILTER_WHO:
		entered_user = USER_finduser(col_buff);
		if(entered_user == NULL)
		  {
		    char buffer[100];

		    DISP_beep(Ctxt);
		    sprintf(buffer, "FILTER: Unknown user id %s", col_buff);
		    DISP_message(Ctxt, buffer);
		    break;
		  }
		DISP_prompt_string(Ctxt, mention_user("%s Filter Method: ",
						      entered_user), "");
		prompt_state = ADD_FILTER_TYPE;
		state = PROMPT;
		MSG = TRUE;
		break;
	      case ADD_FILTER_TYPE:
		{
		  int idx = FILTER_index(col_buff);

		  if(idx == -1) {
		    char buff[100];
		    sprintf(buff, "\03Unsupported filter type %s.",
			    col_buff);
		    DISP_message(Ctxt, buff);
		    FILTER_describe(Ctxt);
		  } else {
		    FILTER_init(Ctxt, entered_user, idx);
		  }
		  break;
		}
	      case APPLICATION_WHO:
		entered_user = USER_finduser(col_buff);
		if(entered_user == NULL)
		  {
		    char buffer[100];

		    DISP_beep(Ctxt);
		    sprintf(buffer, "APPLICATION: Unknown user id %s", col_buff);
		    DISP_message(Ctxt, buffer);
		    break;
		  }
		DISP_prompt_string(Ctxt, mention_user("%s's Application: ",
						      entered_user), "");
		prompt_state = APPLICATION_NAME;
		state = PROMPT;
		MSG = TRUE;
		break;
	      case APPLICATION_NAME:
		SHAPP_fork_shared(Ctxt, col_buff, entered_user);
		break;
	      case SEND_WHO:
		entered_user = USER_finduser(col_buff);
		if(entered_user == NULL)
		  {
		    char buffer[100];

		    DISP_beep(Ctxt);
		    sprintf(buffer, "SEND: Unknown user id %s", col_buff);
		    DISP_message(Ctxt, buffer);
		    break;
		  }
		DISP_prompt_string(Ctxt, mention_user("%s File to send: ", 
						      entered_user), "");
		prompt_state = SEND_FILE;
		state = PROMPT;
		MSG = TRUE;
		break;
	      case SEND_FILE:
		DATA_send_file(Ctxt, entered_user, col_buff);
		break;
	      case DOWNLOAD_PATH:
		if(Ctxt->downloadpath != NULL)
		  free(Ctxt->downloadpath);
		Ctxt->downloadpath = strdup(col_buff);
		break;
	      case ARBITRARY_WHO:
		entered_user = USER_finduser(col_buff);
		if(entered_user == NULL)
		  {
		    char buffer[100];

		    DISP_beep(Ctxt);
		    sprintf(buffer, "SEND: Unknown user id %s", col_buff);
		    DISP_message(Ctxt, buffer);
		    break;
		  }
		DISP_prompt_string(Ctxt, mention_user("%s Message: ",
						      entered_user), "");
		prompt_state = ARBITRARY_MESSAGE;
		state = PROMPT;
		MSG = TRUE;
	      case ARBITRARY_MESSAGE:
		/* First, make sure we can contact them */
		if(!((entered_user->type == GNUTALK) ||
		     (entered_user->type == ETALK)))
		  {
		    DISP_message(Ctxt, "Selected user client does not messages.");
		    break;
		  }
		USER_send(Ctxt, entered_user, "\03", 1);
		USER_send(Ctxt, entered_user, col_buff, strlen(col_buff));
		USER_send(Ctxt, entered_user, "\n", 1);
		break;
	      default:
		break;
	      }
	  }
      }
      break;	    
    case 127:			/* DEL */
    case 8:			/* C-h */
      if(collect[0])
	{
	  collect[strlen(collect)-1] = 0;
	  /* The last used prompt is remembered */
	  if(suppress_feedback == 0)
	    DISP_prompt_string(Ctxt, NULL, collect);
	}
      else
	{
	  DISP_beep(Ctxt);
	}
      break;
    case 21:			/* C-u - delete line */
      collect[0] = 0;
      if(suppress_feedback == 0)
	DISP_prompt_string(Ctxt, NULL, collect);
      break;
    case 23:			/* C-w - delete word */
      /* Zap over left over spaces */
      while((collect[0] != 0) && (collect[strlen(collect)-1] == ' '))
	collect[strlen(collect)-1] = 0;
      /* Delete the word itself... */
      while((collect[0] != 0) && (collect[strlen(collect)-1] != ' '))
	collect[strlen(collect)-1] = 0;
      if(suppress_feedback == 0)
	DISP_prompt_string(Ctxt, NULL, collect);
      break;
    default:
      {
	int len;
	      
	len= strlen(collect);
	if(len >= COLLECTSIZE)
	  {
	    DISP_beep(Ctxt);
	  }
	else
	  {
	    collect[len] = in;
	    collect[len+1] = 0;
	    /* The last used prompt is remembered */
	    if(suppress_feedback == 0)
	      DISP_prompt_string(Ctxt, NULL, collect);
	  }
	break;
      }
    }
  /* We have a prompt, always set this! */
  MSG = 1;
}

static void DISP_talkmode(Ctxt, in)
     struct TalkContext *Ctxt;
     char in;
{
  switch(in)
    {
    case 3:			/* C-c */
      /* If we are currently dialing out, then do a hangup.  If the user
       * really wants to abort, they'll hit C-c twice for us.
       */
      if(PROTOCOL_abort(Ctxt) == 0) {
	/* Ask if they really want to quit as an option? */
	gtalk_shutdown(NULL);
      } else {
	DISP_message(Ctxt, "Call canceled.");
      }
      break;
    case 26:			/* C-z */
      CurrentMethods->suspend(Ctxt);
      /* We must redraw when we return for curses... */
      DISP_redraw(Ctxt);
      break;
    case 27:			/* ESC */
      if(suppress_feedback == 0)
	DISP_message(Ctxt, "ESC - ");
      state = META;
      MSG = 1;
      /* don't set message, next char always prints something new... */
      break;
    case 8:    /* BACKSPACE */
    case 127:  /* DELETE */
      /* If the user hits BS or DELETE, chances are they don't want to
       * send it, and in fact probably want "delchar backwards" instead!
       * Knowing this, convert it to the currently used DELCHAR character.
       */
      in = Ctxt->editkeys[0];
      /* Now flow into the below test addchar thing... */
    default:			/* transfer this character... */
      if(!Ctxt->eight_bit && (in & 128))	/* is it a meta character? */
	{
	  DISP_meta(Ctxt, in & 127); /* call meta keys */
	}
      else
	{
	  char buffer[2];
	  sprintf(buffer, "%c", in);
	  DISP_local_string(Ctxt, buffer, 1);
	}
      break;
    }
}

static void DISP_dialog(Ctxt, in)
     struct TalkContext *Ctxt;
     char in;
{
  switch(in)
    {
    case 27:			/* ESC  */
    case 3:			/* C-c */
    case 7:			/* C-g */
      state = TALKMODE;
      CurrentMethods->nuke_popup(popup);
      popup = NULL;
      popup_full = 0;
      DISP_redraw(Ctxt);
      break;
    default:
      break;
    }
}

static void DISP_yorn(Ctxt, in)
     struct TalkContext *Ctxt;
     char in;
{
  MSG = 0;
  switch(in)
    {
    case 27:			/* ESC  */
    case 3:			/* C-c */
    case 7:			/* C-g */
    case 'n':
    case 'N':
      state = TALKMODE;
      yorn_answer = 0;
      GT_end_recursion();
      break;
    case 10:			/* LFD */
    case 13:			/* RET */
    case ' ':			/* SPCBAR */
    case 'y':
    case 'Y':
      state = TALKMODE;
      yorn_answer = 1;
      GT_end_recursion();
      break;
    default:
      DISP_beep(Ctxt);
      DISP_prompt_string(Ctxt, yorn_prompt, "...Please answer with (Y/N)");
      MSG = 1;
      break;
    }
}

/*
 * Function: DISP_input
 *
 *   Read input using curses functions, and do stuff based on that
 * information.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              dev  - Pointer to device
 * History:
 * zappo   8/3/95     Created
 */
void DISP_input(Ctxt, dev)
     struct TalkContext *Ctxt;
     struct InputDevice *dev;
{
  static int first = TRUE;
  int need_refresh = FALSE;
  char in;

  MSG = 0;

  in = CurrentMethods->read_next_char();

  if(first)
    {
      /* Clear the window */
      if(CurrentMethods) 
	{
	  CurrentMethods->clear_window
	    (((struct WindowList *)FirstElement(list))->window);
	  CurrentMethods->refresh_window
	    (((struct WindowList *)FirstElement(list))->window);
	}

      need_refresh = TRUE;

      first = 0;
    }

  if(in == 12)			/* C-l */
    {
      /* refresh the entire screen at user's request... Since this is
	 the same for all modes, do it here. */
      DISP_redraw(Ctxt);
      /* If in draw mode then send to remote for compatibility with other
	talk programs. */
      if(state == TALKMODE)
	{
	  DISP_talkmode(Ctxt, in);
	}
    }
  else if(FORK_grabkey(Ctxt, in, NULL))
    {
      /* Do nothing */
    }
  else
    {
      switch(state)
	{
	case META:
	  DISP_meta(Ctxt, in);
	  /* If there was no new message, then nuke anything that may have
	     been there... */
	  if(MSG == 0)
	    DISP_message(Ctxt, "");
	  break;
	case PROMPT:
	  DISP_prompt(Ctxt, in);
	  break;
	case TALKMODE:
	  DISP_message(Ctxt, "");
	  DISP_talkmode(Ctxt, in);
	  break;
	case DIALOG:
	  DISP_dialog(Ctxt, in);
	  break;
	case YORN:
	  DISP_yorn(Ctxt, in);
	  /* If there was no new message, then nuke anything that may have
	     been there... */
	  if(MSG == 0)
	    DISP_message(Ctxt, "");
	  break;
	}
      /* Set flag saying popup is now done being filled up, and then 
	 display it */
      if(popup && !popup_full)
	{
	  popup_full = 1;

	  CurrentMethods->finish_popup(popup, FALSE);
	}
    }
  if(need_refresh) {
    DISP_redraw(Ctxt);
  }
}
		
#endif /* NO_IFACE */
