/* GStreamer Element
 * Copyright (C) 2008 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * avidemux filter:
 * Copyright (C) Mean - 2004
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-soften
 *
 * <refsect2>
 * <para>
 * Light-weight "single-frame" smoothing/denoising, that is,
 * it only works with the current frame,
 * it does not need the next or the previous frame.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * avidemux soften filter [Mean] (based/inspired on avisynth spatial soften)
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-ad.h"


#define GST_TYPE_SOFTEN \
  (gst_soften_get_type())
#define GST_SOFTEN(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_SOFTEN,GstSoften))
#define GST_SOFTEN_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_SOFTEN,GstSoftenClass))
#define GST_IS_SOFTEN(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_SOFTEN))
#define GST_IS_SOFTEN_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_SOFTEN))

typedef struct _GstSoften GstSoften;
typedef struct _GstSoftenClass GstSoftenClass;


struct _GstSoften
{
  GstVideoFilter videofilter;

  gint width, height;

  /* soften parameters */
  guint radius, luma_threshold;
};

struct _GstSoftenClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (smooth_debug);
#define GST_CAT_DEFAULT smooth_debug

/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_RADIUS,
  PROP_LUMA_THRESHOLD
      /* FILL ME */
};

#define DEFAULT_RADIUS             2
#define DEFAULT_LUMA_THRESHOLD     5

static GstStaticPadTemplate gst_soften_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_soften_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstFlowReturn gst_soften_transform (GstBaseTransform * btrans,
    GstBuffer * in, GstBuffer * out);
static gboolean gst_soften_start (GstBaseTransform * btrans);
static gboolean gst_soften_stop (GstBaseTransform * btrans);

static void gst_soften_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_soften_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstSoften, gst_soften, GstVideoFilter, GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE (GstSoften, gst_soften);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_soften);

static void
gst_soften_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Soften",
      "Filter/Effect/Video", "Softening",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Mean");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_soften_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_soften_src_template));
}

static void
gst_soften_class_init (GstSoftenClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (smooth_debug, "smooth", 0, "smooth");

  gobject_class->set_property = gst_soften_set_property;
  gobject_class->get_property = gst_soften_get_property;

  g_object_class_install_property (gobject_class, PROP_RADIUS,
      g_param_spec_uint ("radius", "Radius", "Radius",
          1, 60, DEFAULT_RADIUS, G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_LUMA_THRESHOLD,
      g_param_spec_uint ("luma-threshold", "Luma Threshold", "Luma Threshold",
          0, 255, DEFAULT_LUMA_THRESHOLD,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_soften_set_caps);
  trans_class->get_unit_size = GST_DEBUG_FUNCPTR (gst_soften_get_unit_size);
  trans_class->transform = GST_DEBUG_FUNCPTR (gst_soften_transform);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_soften_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_soften_stop);
}

static void
gst_soften_init (GstSoften * filter, GstSoftenClass * g_class)
{
  filter->radius = DEFAULT_RADIUS;
  filter->luma_threshold = DEFAULT_LUMA_THRESHOLD;
}


static void
gst_soften (guint8 * src, guint8 * dst, gint width, gint height, gint radius,
    gint threshold)
{
  guint8 *rsrc, *rdst;
  guint32 val, cur, coef, x, y;
  gint blockx, blocky;

  for (y = radius; y < height - radius; y++) {
    rsrc = src + y * width + radius;
    rdst = dst + y * width + radius;

    for (x = radius; x < width - radius; x++) {
      coef = 0;
      val = 0;

      for (blocky = -radius; blocky <= radius; blocky++) {
        for (blockx = -radius; blockx <= radius; blockx++) {
          cur = *(rsrc + blockx + blocky * width);

          if (distMatrix[cur][*rsrc] <= threshold) {
            coef++;
            val += cur;
          }
        }
      }

      if (coef != 1)
        val = (val + (coef >> 1) - 1) / coef;
      *rdst++ = val;
      rsrc++;
    }
  }
}

/* see below for specific (loop) context */
#define CHECK(x) G_STMT_START { \
      cur = *x; \
      if (dist[cur] <= threshold) { \
        coef++; \
        val+=cur; \
      } \
      x++; \
    } G_STMT_END

static void
gst_soften3 (guint8 * src, guint8 * dst, gint width, gint height, gint radius,
    gint threshold)
{
  guint8 *rsrc, *rdst, *c0, *c1, *c2, *dist;
  guint32 val, cur, coef, x, y;

  for (y = 1; y < height - 1; y++) {
    rsrc = src + y * width + 1;
    rdst = dst + y * width + 1;

    for (x = 1; x < width - 1; x++) {
      coef = 0;
      val = 0;
      c0 = rsrc - 1 - width;
      c1 = c0 + width;
      c2 = c1 + width;

      dist = distMatrix[*rsrc];

      CHECK (c0);
      CHECK (c0);
      CHECK (c0);

      CHECK (c1);
      CHECK (c1);
      CHECK (c1);

      CHECK (c2);
      CHECK (c2);
      CHECK (c2);

      if (coef != 1)
        val = (val + (coef >> 1) - 1) / coef;
      *rdst++ = val;
      rsrc++;
    }
  }
}

static void
gst_soften5 (guint8 * src, guint8 * dst, gint width, gint height, gint radius,
    gint threshold)
{
  guint8 *rsrc, *rdst, *c0, *c1, *c2, *c3, *c4, *dist;
  guint32 val, cur, coef, x, y;

  for (y = 2; y < height - 2; y++) {
    rsrc = src + y * width + 2;
    rdst = dst + y * width + 2;

    for (x = 2; x < width - 2; x++) {
      coef = 0;
      val = 0;
      c0 = rsrc - 2 - 2 * width;
      c1 = c0 + width;
      c2 = c1 + width;
      c3 = c2 + width;
      c4 = c3 + width;

      dist = distMatrix[*rsrc];

      CHECK (c0);
      CHECK (c0);
      CHECK (c0);
      CHECK (c0);
      CHECK (c0);

      CHECK (c1);
      CHECK (c1);
      CHECK (c1);
      CHECK (c1);
      CHECK (c1);

      CHECK (c2);
      CHECK (c2);
      CHECK (c2);
      CHECK (c2);
      CHECK (c2);

      CHECK (c3);
      CHECK (c3);
      CHECK (c3);
      CHECK (c3);
      CHECK (c3);

      CHECK (c4);
      CHECK (c4);
      CHECK (c4);
      CHECK (c4);
      CHECK (c4);

      if (coef != 1)
        val = (val + (coef >> 1) - 1) / coef;
      *rdst++ = val;
      rsrc++;
    }
  }
}

static GstFlowReturn
gst_soften_transform (GstBaseTransform * btrans, GstBuffer * in,
    GstBuffer * out)
{
  GstSoften *filter;
  guint8 *src, *dest;
  guint radius;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_SOFTEN (btrans);

  src = (guint8 *) GST_BUFFER_DATA (in);
  dest = (guint8 *) GST_BUFFER_DATA (out);

  oil_memcpy (dest, src, GST_VIDEO_I420_SIZE (filter->width, filter->height));

  radius = MIN (filter->radius, MIN (filter->width, filter->height));

  /* only luma is processed */
  switch (radius) {
    case 1:
      gst_soften3 (src, dest, GST_VIDEO_I420_Y_ROWSTRIDE (filter->width),
          filter->height, radius, filter->luma_threshold);
    case 2:
      gst_soften5 (src, dest, GST_VIDEO_I420_Y_ROWSTRIDE (filter->width),
          filter->height, radius, filter->luma_threshold);
    default:
      gst_soften (src, dest, GST_VIDEO_I420_Y_ROWSTRIDE (filter->width),
          filter->height, radius, filter->luma_threshold);
  }

  return GST_FLOW_OK;
}

static gboolean
gst_soften_start (GstBaseTransform * btrans)
{

  return TRUE;
}

static gboolean
gst_soften_stop (GstBaseTransform * btrans)
{

  return TRUE;
}

static void
gst_soften_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstSoften *src;

  g_return_if_fail (GST_IS_SOFTEN (object));
  src = GST_SOFTEN (object);

  switch (prop_id) {
    case PROP_RADIUS:
      src->radius = g_value_get_uint (value);
      break;
    case PROP_LUMA_THRESHOLD:
      src->luma_threshold = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_soften_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstSoften *src;

  g_return_if_fail (GST_IS_SOFTEN (object));
  src = GST_SOFTEN (object);

  switch (prop_id) {
    case PROP_RADIUS:
      g_value_set_uint (value, src->radius);
      break;
    case PROP_LUMA_THRESHOLD:
      g_value_set_uint (value, src->luma_threshold);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
