//*****************************************************************************
//                                FrmMain.cpp                                 *
//                               -------------                                *
//  Started     : 18/08/2003                                                  *
//  Last Update : 14/04/2010                                                  *
//  Copyright   : (C) 2003 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "main/FrmMain.hpp"

// The application icon
#if defined( __WXGTK__ ) || defined( __WXMOTIF__ ) || defined( __WXMAC__ ) || \
    defined( __WXMGL__ ) || defined( __WXX11__ )
  #include "icons/gspiceui-32x32.xpm"
  #include "icons/file-open.xpm"
  #include "icons/file-import.xpm"
  #include "icons/file-reload.xpm"
  #include "icons/file-close.xpm"
  #include "icons/sim-create.xpm"
  #include "icons/sim-run.xpm"
  #include "icons/sim-stop.xpm"
  #include "icons/sim-edit.xpm"
  #include "icons/sim-plot.xpm"
  #include "icons/help.xpm"
#endif

//*****************************************************************************
// Implement an event table in which the events are routed to their respective
// handler functions in the class. If -1 is given as the ID, the given handler
// will be invoked for any event of the specified type.

BEGIN_EVENT_TABLE( FrmMain, wxFrame )

  EVT_MENU( ID_MNU_OPEN,    FrmMain::OnOpen      )
  EVT_MENU( ID_MNU_IMPORT,  FrmMain::OnImport    )
  EVT_MENU( ID_MNU_RELOAD,  FrmMain::OnReload    )
  EVT_MENU( ID_MNU_CLOSE,   FrmMain::OnClose     )
  EVT_MENU( ID_MNU_EXIT,    FrmMain::OnAppExit   )
  EVT_MENU( ID_MNU_CREATE,  FrmMain::OnSimCreate )
  EVT_MENU( ID_MNU_RUN,     FrmMain::OnSimRun    )
  EVT_MENU( ID_MNU_STOP,    FrmMain::OnSimStop   )
  EVT_MENU( ID_MNU_SCHEM,   FrmMain::OnSchematic )
  EVT_MENU( ID_MNU_VIEWER,  FrmMain::OnViewData  )
  EVT_MENU( ID_MNU_GNUCAP,  FrmMain::OnSelSimEng )
  EVT_MENU( ID_MNU_NGSPICE, FrmMain::OnSelSimEng )
  EVT_MENU( ID_MNU_PREFS,   FrmMain::OnPrefs     )
  EVT_MENU( ID_MNU_MANUAL,  FrmMain::OnManual    )
  EVT_MENU( ID_MNU_ABOUT,   FrmMain::OnAbout     )

  EVT_TOOL( ID_TBR_OPEN,    FrmMain::OnOpen      )
  EVT_TOOL( ID_TBR_IMPORT,  FrmMain::OnImport    )
  EVT_TOOL( ID_TBR_RELOAD,  FrmMain::OnReload    )
  EVT_TOOL( ID_TBR_CLOSE,   FrmMain::OnClose     )
  EVT_TOOL( ID_TBR_CREATE,  FrmMain::OnSimCreate )
  EVT_TOOL( ID_TBR_RUN,     FrmMain::OnSimRun    )
  EVT_TOOL( ID_TBR_STOP,    FrmMain::OnSimStop   )
  EVT_TOOL( ID_TBR_SCHEM,   FrmMain::OnSchematic )
  EVT_TOOL( ID_TBR_VIEWER,  FrmMain::OnViewData  )
  EVT_TOOL( ID_TBR_HELP,    FrmMain::OnManual    )

  EVT_TOOL_ENTER( -1,       FrmMain::OnToolEnter )

  EVT_CLOSE(                FrmMain::OnSysExit   )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Argument List :
//   poApp - A pointer to the class that created this object

FrmMain::FrmMain( const wxApp * poApp ) :
             wxFrame( (wxFrame *) NULL, -1, wxT(""), wxDefaultPosition,
                      wxDefaultSize, wxDEFAULT_FRAME_STYLE & ~wxMAXIMIZE_BOX ),
             m_oFileTsks( this ), m_oHelpTsks( this ),
             m_oSimnNgs( ), m_oSimnGcp( )
{
  // Indicate that the main frame is open
  m_bIsOpen = TRUE;

  // Set pointer to parent application
  m_poApp = poApp;

  // Get the global configuration object
  m_poCfg = (wxConfig *) wxConfig::Get( );

  // Initialize pointers
  m_poSimultn   = NULL;
  m_poPrcSimEng = NULL;
  m_poNbkSimEng = NULL;

  // Create everything
  Initialize( );
}

//*****************************************************************************
// Destructor.

FrmMain::~FrmMain( )
{
  if( m_poNbkSimEng != NULL ) { delete m_poNbkSimEng; m_poNbkSimEng = NULL; }
  if( m_poPrcSimEng != NULL ) { delete m_poPrcSimEng; m_poPrcSimEng = NULL; }
}

//*****************************************************************************
// Initialize the main frame.

void  FrmMain::Initialize( void )
{
  long  li1;

  // This line just suppresses the following compiler warning :
  //   ./icons/gspiceui-32x32.xpm:3: warning: gspiceui_xpm defined but not used
  char  c1; c1 = gspiceui_xpm[ 0 ][ 0 ]; // ???

  // Set the frame icon, font and title
  SetIcon( wxICON( gspiceui ) );
  SetFont( FONT_NORM );
  m_oFileTsks.bSetTitle( );

  // Automatically update the frame layout when it is resized
  SetAutoLayout( TRUE );

  // Initialize the FileTasks object eg. load schematic or netlist file/s
  m_oFileTsks.Initialize( );

  // Call all the initialization functions
  InitMenuBar ( );
  InitToolBar ( );
  InitLstBoxs ( );
  InitNbkTCtls( );
  InitStatBar ( );
  InitToolTips( );
  InitSimEng  ( );
  InitLogFiles( );
  DoLayout( );
  InitPosnSize( );

  // Set the sweep source synchronization flag
  m_poCfg->SetPath( wxT("/Main") );
  m_poCfg->Read( wxT("SyncSwpSrcs"), &li1, 0 );
  PnlAnaBase::SetSyncSwpSrcs( li1 != 0 ? TRUE : FALSE ); // ???

  // Display the desired analysis page
  m_poCfg->SetPath( wxT("/Simulator") );
  m_poNbkSimEng->bSetPage( m_poCfg->Read( wxT("Analysis"), wxT("OP") ) );

  // If necessary load the simulation or netlist
  if( ! m_oNetLst.bIsEmpty( ) )
  {
    bSimLoad( );
    m_poNbkSimEng->bLoad( *m_poSimultn );
  }

  // Display the appropriate text control page
  if(      ! m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_SIMULTN )->bIsEmpty( ) )
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );
  else if( ! m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_NETLIST )->bIsEmpty( ) )
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
  else
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_CONSOLE );
}

//*****************************************************************************
// Initialize the menu bar.

void  FrmMain::InitMenuBar( void )
{
  // Create the menu bar and menus
  wxMenuBar * poMenuBar  = new wxMenuBar;
  wxMenu    * poMenuFile = new wxMenu;
  wxMenu    * poMenuSimu = new wxMenu;
  wxMenu    * poMenuSets = new wxMenu;
  wxMenu    * poMenuHelp = new wxMenu;

  // Set the tool bar font
  poMenuBar->SetFont( FONT_NORM );

  // Load the menus with items
  poMenuFile->Append( ID_MNU_OPEN,             wxT(" &Open ... ")        );
  poMenuFile->Append( ID_MNU_IMPORT,           wxT(" &Import ... ")      );
  poMenuFile->Append( ID_MNU_RELOAD,           wxT(" &Reload ")          );
  poMenuFile->Append( ID_MNU_CLOSE,            wxT(" &Close ")           );
  poMenuFile->AppendSeparator( );
  poMenuFile->Append( ID_MNU_EXIT,             wxT(" E&xit ")            );

  poMenuSimu->Append( ID_MNU_CREATE,           wxT(" &Create ")          );
  poMenuSimu->Append( ID_MNU_RUN,              wxT(" &Run ")             );
  poMenuSimu->Append( ID_MNU_STOP,             wxT(" &Stop ")            );
  poMenuSimu->AppendSeparator( );
  poMenuSimu->Append( ID_MNU_SCHEM,            wxT(" Sc&hematic ... ")   );
  poMenuSimu->Append( ID_MNU_VIEWER,           wxT(" &Data Viewer ... ") );

  poMenuSets->AppendRadioItem( ID_MNU_GNUCAP,  wxT("  &GNU-Cap ")        );
  poMenuSets->AppendRadioItem( ID_MNU_NGSPICE, wxT("  &NG-Spice ")       );
  poMenuSets->AppendSeparator( );
  poMenuSets->Append( ID_MNU_PREFS,            wxT("&Preferences ... ")  );

  poMenuHelp->Append( ID_MNU_MANUAL,           wxT(" &User Manual ... ") );
  poMenuHelp->AppendSeparator( );
  poMenuHelp->Append( ID_MNU_ABOUT,            wxT(" &About ... ")       );

  // Load the menubar with menus
  poMenuBar ->Append( poMenuFile,              wxT(" &File ")            );
  poMenuBar ->Append( poMenuSimu,              wxT(" &Simulate ")        );
  poMenuBar ->Append( poMenuSets,              wxT(" S&ettings ")        );
  poMenuBar ->Append( poMenuHelp,              wxT(" &Help ")            );

  // Check the appropriate simulation engine menu item
  m_poCfg->SetPath( wxT("/Simulator") );
  if( m_poCfg->Read( wxT("Engine"), CLP_GNUCAP ) == CLP_NGSPICE )
       poMenuSets->Check( ID_MNU_NGSPICE, TRUE );
  else poMenuSets->Check( ID_MNU_GNUCAP , TRUE );

  // Attach the menu bar to the frame
  SetMenuBar( poMenuBar );
}

//*****************************************************************************
// Initialize the tool bar.

void  FrmMain::InitToolBar( void )
{
  wxBitmap  * poPixMap[ 10 ];
  wxToolBar * poToolBar;

  // Create the tool bar
  poToolBar = CreateToolBar( wxHORIZONTAL | wxTB_FLAT );

  // Create the bitmaps for the tools
  poPixMap[ 0 ] = new wxBitmap( file_open_xpm   );
  poPixMap[ 1 ] = new wxBitmap( file_import_xpm );
  poPixMap[ 2 ] = new wxBitmap( file_reload_xpm );
  poPixMap[ 3 ] = new wxBitmap( file_close_xpm  );
  poPixMap[ 4 ] = new wxBitmap( sim_create_xpm  );
  poPixMap[ 5 ] = new wxBitmap( sim_run_xpm     );
  poPixMap[ 6 ] = new wxBitmap( sim_stop_xpm    );
  poPixMap[ 7 ] = new wxBitmap( sim_edit_xpm    );
  poPixMap[ 8 ] = new wxBitmap( sim_plot_xpm    );
  poPixMap[ 9 ] = new wxBitmap( help_xpm        );

  // Add the tools to the toolbar
  poToolBar->SetToolSeparation( 10 );

  poToolBar->AddTool( ID_TBR_OPEN,   wxT(""), *(poPixMap[ 0 ]),
                      wxT("Open a netlist file") );
  poToolBar->AddTool( ID_TBR_IMPORT, wxT(""), *(poPixMap[ 1 ]),
                      wxT("Import schematic file/s") );
  poToolBar->AddTool( ID_TBR_RELOAD, wxT(""), *(poPixMap[ 2 ]),
                      wxT("Reload netlist / schematic file/s") );
  poToolBar->AddTool( ID_TBR_CLOSE,  wxT(""), *(poPixMap[ 3 ]),
                      wxT("Close netlist / schematic file/s") );

  poToolBar->AddSeparator( );

  poToolBar->AddTool( ID_TBR_CREATE, wxT(""), *(poPixMap[ 4 ]),
                      wxT("Create a netlist file containing simulation instructions") );
  poToolBar->AddTool( ID_TBR_RUN,    wxT(""), *(poPixMap[ 5 ]),
                      wxT("Run the simulation engine process") );
  poToolBar->AddTool( ID_TBR_STOP,   wxT(""), *(poPixMap[ 6 ]),
                      wxT("Stop the simulation engine process") );

  poToolBar->AddSeparator( );

  poToolBar->AddTool( ID_TBR_SCHEM,  wxT(""), *(poPixMap[ 7 ]),
                      wxT("Edit / view schematic/s") );
  poToolBar->AddTool( ID_TBR_VIEWER, wxT(""), *(poPixMap[ 8 ]),
                      wxT("View results data") );

  poToolBar->AddSeparator( );

  poToolBar->AddTool( ID_TBR_HELP,   wxT(""), *(poPixMap[ 9 ]),
                      wxT("View the user manual") );

  // Realize the toolbar
  poToolBar->Realize( );

  // Delete the bitmaps
  for( int i1=0; i1<10; i1++ ) delete poPixMap[ i1 ];
}

//*****************************************************************************
// Initialize the test node and component list boxes.

void  FrmMain::InitLstBoxs( void )
{
  long  lStyle;

  // Create the node and component list labels
  lStyle = wxALIGN_CENTRE;
  m_oLblNodes.Create( this, ID_UNUSED, wxT("Nodes"),      wxDefaultPosition,
                      wxDefaultSize, lStyle );
  m_oLblCpnts.Create( this, ID_UNUSED, wxT("Components"), wxDefaultPosition,
                      wxDefaultSize, lStyle );

  // Create the node and component list boxes
  lStyle = wxLB_MULTIPLE | wxLB_ALWAYS_SB | wxLB_SORT;
  m_oLbxNodes.Create( this, ID_LBX_NODES, wxDefaultPosition,
                      wxSize(  90, -1 ), 0, NULL, lStyle );
  m_oLbxCpnts.Create( this, ID_LBX_CPNTS, wxDefaultPosition,
                      wxSize( 110, -1 ), 0, NULL, lStyle );
}

//*****************************************************************************
// Initialize the console notebook.

void  FrmMain::InitNbkTCtls( void )
{
  long  li1;

  // Create the text control notebook
  m_oNbkTxtCtls.bCreate( this, ID_NBK_TXTCTLS );

  // Set the text control maximum lines
  m_poCfg->SetPath( wxT("/Main") );
  m_poCfg->Read( wxT("NbkMaxLines"), &li1, TXTCTL_LINESDEF );
  m_oNbkTxtCtls.bSetLinesMax( li1 );

  // Set the precision of the results data
  m_poCfg->Read( wxT("Precision"),   &li1, CNVTYPE_DFLT_FLTRES );
  ConvertType::bSetFltRes( (int) li1 );
}

//*****************************************************************************
// Initialize the status bar.
//
// Note : The first field in the status bar has benn effectively disable by
//        setting it's width to near zero. It is not used since the frame
//        insists on writing text into this field as the user moves the mouse
//        over display controls.

void  FrmMain::InitStatBar( void )
{
  wxString  os1;
  int       ia1[ 4 ];

  // Create the status bar
  CreateStatusBar( 4 );

  // Determine the status bar field widths
  ia1[ 0 ] = 3;   // Vanishingly thin
  ia1[ 1 ] = -1;  // Variable width
  ia1[ 2 ] = 140; // Fixed width
  ia1[ 3 ] = 120; // Fixed width
  SetStatusWidths( 4, ia1 );

  // Get the viewer application name from the configuration object
  m_poCfg->SetPath( wxT("/Viewer") );
  m_poCfg->Read( wxT("Name"), &os1, CLP_GWAVE );

  // Set status bar text
  SetStatusText( wxT(" Welcome to GNU Spice GUI"), 1 );
  SetStatusText( wxT(" Simulator : None")        , 2 );
  SetStatusText( wxT(" Viewer : ") + os1         , 3 );
}

//*****************************************************************************
// Initialize the tool tips.

void  FrmMain::InitToolTips( void )
{
  long  li1;

  // Define tool tips for each control
  m_oLbxNodes.SetToolTip( wxT("Test Nodes") );
  m_oLbxCpnts.SetToolTip( wxT("Test Components") );

  // Set global tool tip attributes
  m_poCfg->SetPath( wxT("/Main") );
  m_poCfg->Read( wxT("ToolTips"), &li1, (long) 0 );
  wxToolTip::Enable( li1 != 0 ? TRUE : FALSE );
  wxToolTip::SetDelay( 700 );
}

//*****************************************************************************
// Set the electronic circuit simulator engine to use.

void  FrmMain::InitSimEng( void )
{
  // Delete any existing simulation engine notebook object
  if( m_poNbkSimEng != NULL )
  {
    GetSizer( )->Detach( m_poNbkSimEng );
    delete m_poNbkSimEng;
    m_poNbkSimEng = NULL;
  }

  // Delete any existing simulation engine process object
  if( m_poPrcSimEng != NULL )
  {
    delete m_poPrcSimEng;
    m_poPrcSimEng = NULL;
  }

  // Create the new simulation engine objects and update the status text
  if( GetMenuBar( )->IsChecked( ID_MNU_NGSPICE ) )
  {
    m_poNbkSimEng = new NbkNgSpice( this, ID_NBK_ANALYS );
    m_poPrcSimEng = new PrcNgSpice( );
    m_poSimultn   = &m_oSimnNgs;

    SetStatusText( wxT(" Simulator : NG-Spice"), 2 );
  }
  else
  {
    m_poNbkSimEng = new NbkGnuCap( this, ID_NBK_ANALYS );
    m_poPrcSimEng = new PrcGnuCap( );
    m_poSimultn   = &m_oSimnGcp;

    SetStatusText( wxT(" Simulator : GNU-Cap"), 2 );
  }
}

//*****************************************************************************
// Initialize the log file names in the process objects.

void  FrmMain::InitLogFiles( void )
{
  wxFileName  ofn1;
  wxString    os1;

  // Check that a netlist file has been defined
  ofn1 = m_oFileTsks.rosGetNetLstFile( );
  if( ! ofn1.IsOk( )       ) return;
  if( ! ofn1.FileExists( ) ) return;

  // Get the path to the schematic or netlist file
  os1 = ofn1.GetPath( ) + wxT("/gspiceui.log");

  // Set the log file path for the gNetList process object
  m_oFileTsks.bSetLogFile( os1 );

  // Set the log file path for the simulation process object
  if( m_poPrcSimEng != NULL )
    m_poPrcSimEng->bSetLogFile( os1 );
}

//*****************************************************************************
// Initialize the frames position amd size.

void  FrmMain::InitPosnSize( void )
{
  int  ix, iy, iw, ih;

  // Get the default position and size
  GetPosition( &ix, &iy );
  GetClientSize( &iw, &ih );

  // Get the position and size from the configuration object
  m_poCfg->SetPath( wxT("/Main") );
  ix = m_poCfg->Read( wxT("PosnX"), 1      );
  iy = m_poCfg->Read( wxT("PosnY"), iy     );
  iw = m_poCfg->Read( wxT("SizeW"), iw     );
  ih = m_poCfg->Read( wxT("SizeH"), ih+100 );

  // Set the position and size
  Move( ix, iy );
  SetClientSize( iw, ih );
}

//*****************************************************************************
// Layout the main frame display objects.

void  FrmMain::DoLayout( void )
{
  wxGridBagSizer * poSizer;
  wxGBPosition     oGBPosn;
  wxGBSpan         oGBSpan;
  int              iFlags = wxALL | wxALIGN_CENTER | wxEXPAND;
  int              iBorder = 3;

  if( GetSizer( ) == NULL )
  { // Create and set the frame's sizer (at startup)
    poSizer = new wxGridBagSizer( 1, 1 );
    SetSizer( poSizer );
  }
  else
  { // Detach the display objects from the sizer
    poSizer = (wxGridBagSizer *) GetSizer( );
    poSizer->Detach( &m_oLblNodes   );
    poSizer->Detach( &m_oLbxNodes   );
    poSizer->Detach( &m_oLblCpnts   );
    poSizer->Detach( &m_oLbxCpnts   );
    poSizer->Detach(  m_poNbkSimEng );
    poSizer->Detach( &m_oNbkTxtCtls );
  }

  // Add the nodes label
  oGBPosn.SetCol( 0 );          oGBPosn.SetRow( 0 );
  oGBSpan.SetColspan( 1 );      oGBSpan.SetRowspan( 1 );
  poSizer->Add( &m_oLblNodes,   oGBPosn, oGBSpan, iFlags, iBorder );

  // Add the nodes list box
  oGBPosn.SetCol( 0 );          oGBPosn.SetRow( 1 );
  oGBSpan.SetColspan( 1 );      oGBSpan.SetRowspan( 19 );
  poSizer->Add( &m_oLbxNodes,   oGBPosn, oGBSpan, iFlags, iBorder );

  // Add the components label
  oGBPosn.SetCol( 1 );          oGBPosn.SetRow( 0 );
  oGBSpan.SetColspan( 1 );      oGBSpan.SetRowspan( 1 );
  poSizer->Add( &m_oLblCpnts,   oGBPosn, oGBSpan, iFlags, iBorder );

  // Add the components list box
  oGBPosn.SetCol( 1 );          oGBPosn.SetRow( 1 );
  oGBSpan.SetColspan( 1 );      oGBSpan.SetRowspan( 19 );
  poSizer->Add( &m_oLbxCpnts,   oGBPosn, oGBSpan, iFlags, iBorder );

  // Add the simulator notebook
  oGBPosn.SetCol( 2 );          oGBPosn.SetRow( 0 );
  oGBSpan.SetColspan( 1 );      oGBSpan.SetRowspan( 13 );
  poSizer->Add( m_poNbkSimEng,  oGBPosn, oGBSpan, iFlags, iBorder );
  poSizer->SetItemMinSize( m_poNbkSimEng, 575, 266 );

  // Add the console notebook
  oGBPosn.SetCol( 2 );          oGBPosn.SetRow( 13 );
  oGBSpan.SetColspan( 1 );      oGBSpan.SetRowspan( 7 );
  poSizer->Add( &m_oNbkTxtCtls, oGBPosn, oGBSpan, iFlags, iBorder );

  // Specify how the sizer will grow when resized
  poSizer->SetNonFlexibleGrowMode( wxFLEX_GROWMODE_SPECIFIED );
  poSizer->AddGrowableCol( 0 );
  poSizer->AddGrowableCol( 1 );
  for( size_t szt1=13; szt1<20; szt1++ )
    poSizer->AddGrowableRow( szt1 );

  // Set minimum and initial sizes as calculated by the frame's sizer
  poSizer->SetSizeHints( this );
}

//*****************************************************************************
// Load information from the Simulation object attribute m_poSimultn.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bSimLoad( void )
{
  TextCtrl * poTxtCtl;
  size_t     sz1;

  // Load all the nodes (except ground) into the "Nodes" list box
  for( sz1=0; sz1<m_poSimultn->m_osaNodeLbls.GetCount( ); sz1++ )
    m_oLbxNodes.Append( m_poSimultn->m_osaNodeLbls.Item( sz1 ) );
  // Select the test nodes
  for( sz1=0; sz1<m_poSimultn->rosaGetTstNodes( ).GetCount( ); sz1++ )
    m_oLbxNodes.SetStringSelection( m_poSimultn->rosaGetTstNodes().Item(sz1) );

  // Load all 2-port components into the "Components" list box
  for( sz1=0; sz1<m_poSimultn->m_oaCpnts.GetCount( ); sz1++ )
  {
    const Component & roCpnt = m_poSimultn->m_oaCpnts.Item( sz1 );
    switch( roCpnt.m_eType )     // Only collect two terminal components
    {
      case eCPNT_SCJN  :         // Super-conductor Junction
      case eCPNT_CAP   :         // Capacitor
      case eCPNT_RES   :         // Resistor
      case eCPNT_ADM   :         // Admittance
      case eCPNT_IND   :         // Inductor
      case eCPNT_DIODE :         // Diode
      case eCPNT_ICS   :         // Independent Current Source
      case eCPNT_IVS   : break;  // Independent Voltage Source
      default          : continue;
    }
    m_oLbxCpnts.Append( roCpnt.m_osName );
  }
  // Select the test components
  for( sz1=0; sz1<m_poSimultn->rosaGetTstCpnts( ).GetCount( ); sz1++ )
    m_oLbxCpnts.SetStringSelection( m_poSimultn->rosaGetTstCpnts().Item(sz1) );

  // Load the raw circuit description into the NetList text control
  if( ! m_oNetLst.m_osaNetLst.IsEmpty( ) )
  {
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
    m_oNbkTxtCtls.bClear( NbkTxtCtls::ePAGE_NETLIST );
    poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_NETLIST );
    poTxtCtl->bClear( );
    for( sz1=0; sz1<m_oNetLst.m_osaNetLst.GetCount( ); sz1++ )
      poTxtCtl->bAppendLine( m_oNetLst.m_osaNetLst.Item( sz1 ) );
    m_oNbkTxtCtls.bSetPosn( 0 );
  }

  // Load the analysis notebook with simulation information
  if( ! m_poNbkSimEng->bLoad( *m_poSimultn ) ) return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Reload information from the Simulation object attribute m_poSimultn.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bSimReload( void )
{
  wxArrayString  osaTstNodes;
  wxArrayString  osaTstCpnts;
  size_t         sz1;
  int            i1;

  // Record the nodes that are currently selected
#if wxCHECK_VERSION( 2,8,0 )
  for( sz1=0; sz1<m_oLbxNodes.GetCount( ); sz1++ )
#else
  for( sz1=0; sz1<(size_t)m_oLbxNodes.GetCount( ); sz1++ )
#endif
    if( m_oLbxNodes.IsSelected( sz1 ) )
      osaTstNodes.Add( m_oLbxNodes.GetString( sz1 ) );

  // Record the components that are currently selected
#if wxCHECK_VERSION( 2,8,0 )
  for( sz1=0; sz1<m_oLbxCpnts.GetCount( ); sz1++ )
#else
  for( sz1=0; sz1<(size_t)m_oLbxNodes.GetCount( ); sz1++ )
#endif
    if( m_oLbxCpnts.IsSelected( sz1 ) )
      osaTstCpnts.Add( m_oLbxCpnts.GetString( sz1 ) );

  // Clear the lists of nodes and components
  m_oLbxNodes.Clear( );
  m_oLbxCpnts.Clear( );

  bSimLoad( );  // Load the Simulation object

  // Select nodes that where previously selected
  if( ! osaTstNodes.IsEmpty( ) )
  {
    for( sz1=0; sz1<osaTstNodes.GetCount( ); sz1++ )
    {
      i1 = m_oLbxNodes.FindString( osaTstNodes.Item( sz1 ) );
      if( i1 != wxNOT_FOUND ) m_oLbxNodes.Select( i1 );
    }
  }

  // Select components that where previously selected
  if( ! osaTstCpnts.IsEmpty( ) )
  {
    for( sz1=0; sz1<osaTstCpnts.GetCount( ); sz1++ )
    {
      i1 = m_oLbxCpnts.FindString( osaTstCpnts.Item( sz1 ) );
      if( i1 != wxNOT_FOUND ) m_oLbxCpnts.Select( i1 );
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Save information to the Simulation object attribute m_poSimultn.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bSimSave( void )
{
  wxArrayInt  oaiSelNodes, oaiSelCpnts;
  size_t      sz1;
  int         iSelNo;
  wxString    osSel;

  // Clear all previously test points and command objects
  m_poSimultn->bClrCmds( );

  // Get an array of indicies to the currently selected test points
  m_oLbxNodes.GetSelections( oaiSelNodes );
  m_oLbxCpnts.GetSelections( oaiSelCpnts );

  // Load the test nodes into the simulation object
  for( sz1=0; sz1<oaiSelNodes.GetCount( ); sz1++ )
  {
    iSelNo = oaiSelNodes.Item( sz1 );
    osSel = m_oLbxNodes.GetString( iSelNo );
    m_poSimultn->bAddTstNode( osSel );
  }

  // Load the test components into the simulation object
  for( sz1=0; sz1<oaiSelCpnts.GetCount( ); sz1++ )
  {
    iSelNo = oaiSelCpnts.Item( sz1 );
    osSel = m_oLbxCpnts.GetString( iSelNo );
    m_poSimultn->bAddTstCpnt( osSel );
  }

  // Transfer the simulation parameters into the simulation object
  if( ! m_poNbkSimEng->bSave( *m_poSimultn ) ) return( FALSE );

  // Create the simulation file
  if( ! m_poSimultn->bSaveFile( ) )            return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Display a dialog box containing an error message.
//
// Argument List :
//   rosTitle - The dialogue box title
//   rosMsg   - The error message

void  FrmMain::DlgErrMsg( const wxString & rosTitle, const wxString & rosMsg )
{
  wxStringTokenizer  ostk1;
  wxString           os1, os2, os3;

  // Tokenize the message into lines
  ostk1.SetString( rosMsg, wxT(" \n\r"), wxTOKEN_STRTOK );

  // Use the first complete sentence as the status bar message
  os1 << wxT(" ERROR :");
  while( ostk1.HasMoreTokens( ) )
  {
    // Get the next line
    os2 = ostk1.GetNextToken( ).Strip( wxString::both );

    // Remove the path from file names
    if( os2.Freq( wxT('/') ) > 1 )
    {
      os3 << os2.AfterLast( wxT('/') ) << wxT(',');
      os2 = os3;
    }

    // Look for the end of the sentence
    if( os2==wxT('.') || os2==wxT(':') ) break;

    // Append the line to the message
    os1 << wxT(' ') << os2;
  }

  // Set the status line message
  SetStatusText( os1, 1 );

  // Display the error message dialog
  os1 = wxT('\n') + rosMsg;
  wxMessageDialog  oMsgDlg( this, os1, rosTitle, wxOK | wxICON_ERROR );
  oMsgDlg.ShowModal( );
}

//*****************************************************************************
// Check that the electronic circuit simulator engine is present and accounted
// for, if not display an error message.
//
// Argument List :
//   poPrcSimEng - The utility object to be tested
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bIsOkSimEng( PrcBase * poPrcSimEng )
{
  wxString  os1;

  // Check that gnetlist exists and is accessible
  if( ! poPrcSimEng->bBinExists( ) )
  {
    os1 << wxT("Can't find ") << poPrcSimEng->rofnGetBinary( ).GetFullName( )
        << wxT(" which is required to run electronic simulation.\n")
        << wxT("There is no path to it or it has not been installed.\n\n");
    DlgErrMsg( wxT("Configuration Fault"), os1 );

    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Check that the waveform viewer utility is present and accounted for, if not
// display an error message.
//
// Argument List  :
//   poPrcViewer - The utility object to be tested
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bIsOkViewer( PrcBase * poPrcViewer )
{
  wxString  os1;

  // Check that the waveform viewer utility exists and is accessible
  if( ! poPrcViewer->bBinExists( ) )
  {
    os1 << wxT("Can't find ") << poPrcViewer->rofnGetBinary( ).GetFullName( )
        << wxT(" which is required to plot simulation results.\n")
        << wxT("There is no path to it or it has not been installed.\n\n");
    DlgErrMsg( wxT("Configuration Fault"), os1 );

    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Clear the object attributes.

bool  FrmMain::bClear( void )
{
  bool  bRtn = TRUE;

  // Clear all previously selected test points
  m_oLbxNodes.Clear( );
  m_oLbxCpnts.Clear( );

  // Clear simulation object attributes
  m_oSimnGcp.bClear( );
  m_oSimnNgs.bClear( );

  // Clear the simulation object and the analysis panels
  if( m_poNbkSimEng != NULL )
    if( ! m_poNbkSimEng->bClear( ) )  bRtn = FALSE;

  // Clear the text controls
  if( ! m_oNbkTxtCtls.bClear( ) )     bRtn = FALSE;
  m_oNbkTxtCtls.bInitialize( );

  // Terminate any simulation process
  if( m_poPrcSimEng != NULL )
    if( ! m_poPrcSimEng->bKill( ) )   bRtn = FALSE;

  // Terminate any schematic editor process
  if( ! m_oPrcGSchem.bKill( ) )       bRtn = FALSE;

  // Terminate any waveform viewer process
  m_poCfg->SetPath( wxT("/Viewer") );
  if( m_poCfg->Read( wxT("Name") ) == CLP_GWAVE )
       { if( ! m_oPrcGWave.bKill( ) ) bRtn = FALSE; }
  else { if( ! m_oPrcGaw  .bKill( ) ) bRtn = FALSE; }

  return( bRtn );
}

//*****************************************************************************
//                                                                            *
//                               Event Handlers                               *
//                                                                            *
//*****************************************************************************
// Open a circuit description file.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnOpen( wxCommandEvent & roEvtCmd )
{
  // Attempt to get a netlist file name from the user
  if( ! m_oFileTsks.bDlgOpen( ) ) return;

  // Attempt to open and read the netlist file
  if( ! m_oFileTsks.bOpen( ) )    return;

  bSimLoad( );      // Load the simulation information
  InitLogFiles( );  // Initialize the process log files
}

//*****************************************************************************
// Import a schematic file using gnetlist to convert the schematic to a circuit
// description.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnImport( wxCommandEvent & roEvtCmd )
{
  // Attempt to get the schematic file name/s from the user
  if( ! m_oFileTsks.bDlgImport( ) ) return;

  // Attempt to import the schematic file/s and read the netlist file
  if( ! m_oFileTsks.bImport( ) )    return;

  bSimLoad( );      // Load the simulation information
  InitLogFiles( );  // Initialize the process log files
}

//*****************************************************************************
// Reload the schematic or netlist file.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnReload( wxCommandEvent & roEvtCmd )
{
  m_oFileTsks.bReload( );  // Reload schematic file/s or the netlist file
  bSimReload( );           // Reload the simulation information
}

//*****************************************************************************
// Close the circuit description file.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnClose( wxCommandEvent & roEvtCmd )
{
  m_oFileTsks.bClose( );               // Delete temporary files
  bClear( );                           // Clear the object attributes
  m_oFileTsks.bSetTitle( );            // Set the frame title
  m_poNbkSimEng->bSetPage( eCMD_OP );  // Display first analysis panel
}

//*****************************************************************************
// Exit the application.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnAppExit( wxCommandEvent & roEvtCmd )
{
  Close( TRUE ); // Generates a wxCloseEvent which is handled by OnSysExit( )
}

//*****************************************************************************
// Create the simulation.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSimCreate( wxCommandEvent & roEvtCmd )
{
  TextCtrl * poTxtCtl;
  wxString   os1;

  // Change the cursor to the wait symbol
  ::wxBeginBusyCursor( );

  if( ! m_oNetLst.m_osaNetLst.IsEmpty( ) )
  { // Save the simualtion info. in the simulation object
    if( bSimSave( ) )
    {
      // Print the simulation to the simulation text control
      m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );
      poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_SIMULTN );
      poTxtCtl->bLoadFile( m_poSimultn->rofnGetSaveFile( ).GetFullPath( ) );
      poTxtCtl->SetEditable( TRUE );
    }
    else
    {
      if(      ! m_poNbkSimEng->bIsOk()  ) os1 = m_poNbkSimEng->rosGetErrMsg();
      else if( ! m_poSimultn->bIsValid() ) os1 = m_poSimultn->  rosGetErrMsg();
      else                                 os1 = wxT("Unknown error.");
      DlgErrMsg( wxT("Create Simulation Error"), os1 );
    }
  }
  else
  { // Display an error message dialogue
    os1 = wxT("There is no netlist loaded.");
    DlgErrMsg( wxT("Create Simulation Error"), os1 );
  }

  // Change the cursor to the default
  ::wxEndBusyCursor( );

  // Set the status line text
  if( os1.IsEmpty( ) )
       SetStatusText( wxT(" Simulation created successfully")        , 1 );
  else SetStatusText( wxT(" Error/s encountered creating simulation"), 1 );
}

//*****************************************************************************
// Run the simulation.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSimRun( wxCommandEvent & roEvtCmd )
{
  TextCtrl             * poTxtCtl;
  NbkTxtCtls::ePageType  ePage;
  wxFileName             ofn1;
  wxString               os1;

  // Check if a schematic or netlist file is currently open
  ofn1 = m_oFileTsks.rosGetNetLstFile( );
  if( ofn1.GetFullPath( ).IsEmpty( ) )
  {
    os1 = wxT("There is no schematic or netlist file currently loaded.");
    DlgErrMsg( wxT("Run Simulation Error"), os1 );
    return;
  }

  // Change the cursor to the wait symbol eg. the hour glass
  ::wxBeginBusyCursor( );

  //*****************************************************************
  // Prepare to run the simulation (and update the Simulation text page)
  m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );
  poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_SIMULTN );
  if( poTxtCtl->IsModified( ) )
  { // Save the simulation in the text page to file, the user has modified it
    poTxtCtl->SaveFile( m_poSimultn->rofnGetSaveFile( ).GetFullPath( ) );
  }
  else if( bSimSave( ) )
  { // Load the simulation on file into the Simulation text page
    poTxtCtl->bLoadFile( m_poSimultn->rofnGetSaveFile( ).GetFullPath( ) );
    // poTxtCtl is saved immediately after loading to clear IsModified flag
    poTxtCtl->SaveFile( m_poSimultn->rofnGetSaveFile( ).GetFullPath( ) );
    poTxtCtl->SetEditable( TRUE ); // ??? 26/11/2008
  }
  else
  { // There's been an error, report it to the user and return
    if(      ! m_poNbkSimEng->bIsOk( )  ) os1 = m_poNbkSimEng->rosGetErrMsg( );
    else if( ! m_poSimultn->bIsValid( ) ) os1 = m_poSimultn  ->rosGetErrMsg( );
    else                                  os1 = wxT("Unknown error.");
    DlgErrMsg( wxT("Run Simulation Error"), os1 );
    ::wxEndBusyCursor( ); // Change the cursor to the default
    return;
  }

  //*****************************************************************
  // Run the simulation (and update the Console text page)
  m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_CONSOLE );
  poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_CONSOLE );
  poTxtCtl->bClear( );                // Clear the Console text page
  if( ! m_poPrcSimEng->bMakeArgLst( *m_poSimultn ) )
  {
    DlgErrMsg( wxT( "Run Simulation Error" ), m_poPrcSimEng->rosGetErrMsg( ) );
    ::wxEndBusyCursor( );             // Change the cursor to the default
    return;
  }
  m_poPrcSimEng->PrintCmd( *poTxtCtl ); // Print simulator cmd to Console page
  SetStatusText( wxT(" Running the simulation ..."), 1 );
  if( ! m_poPrcSimEng->bExec( ) )
  {
    DlgErrMsg( wxT( "Run Simulation Error" ), m_poPrcSimEng->rosGetErrMsg( ) );
    ::wxEndBusyCursor( );             // Change the cursor to the default
    return;
  }
  if( GetStatusBar( )->GetStatusText( 1 ).Contains( wxT("aborted") ) )
  { // Just exit ammediately if the user aborts the simulation
    ::wxEndBusyCursor( );             // Change the cursor to the default
    return;
  }
  m_poPrcSimEng->PrintRsp( *poTxtCtl ); // Print simulator output to Console page

  //*****************************************************************
  // Format the simulation results (and update the Results text page)
  SetStatusText( wxT(" Formatting the simulation results ..."), 1 );
  wxYield( );
  if( ! m_poPrcSimEng->bFmtResults( ) )
  {
    DlgErrMsg( wxT( "Run Simulation Error" ), m_poPrcSimEng->rosGetErrMsg( ) );
    ::wxEndBusyCursor( );             // Change the cursor to the default
    return;
  }
  if( m_poNbkSimEng->eGetSimEng( ) == eSIMR_GNUCAP )
       ePage = NbkTxtCtls::ePAGE_GNUCAP;
  else ePage = NbkTxtCtls::ePAGE_NGSPICE;
  poTxtCtl = m_oNbkTxtCtls.poGetPage( ePage );
  os1 = m_poPrcSimEng->roGetResultsFile( ).GetFullPath( );
  // Load the simulation output into the results text page
  if( ! poTxtCtl->bLoadFile( os1 ) )
  {
    os1.Prepend( wxT("Couldn't load the results file : \n\n") );
    DlgErrMsg( wxT( "Run Simulation Error" ), os1 );
    ::wxEndBusyCursor( );             // Change the cursor to the default
    return;
  }
  m_oNbkTxtCtls.bSetPage( ePage );    // Change to the appropriate results page

  // Change the cursor to the default
  ::wxEndBusyCursor( );

  SetStatusText( wxT(" Simulation ran successfully"), 1 );
}

//*****************************************************************************
// Stop the simulation.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSimStop( wxCommandEvent & roEvtCmd )
{
  if( m_poPrcSimEng->bIsExec( ) )
  {
    m_poPrcSimEng->bKill( );

    SetStatusText( wxT(" Simulation aborted by user"), 1 );

    wxMessageDialog  oMsgDlg( this, wxT("The simulation has been aborted."),
                              wxT("Run Simulation Event"), wxICON_INFORMATION );
    oMsgDlg.ShowModal( );
  }
}

//*****************************************************************************
// Edit / view the schematic file.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSchematic( wxCommandEvent & roEvtCmd )
{
  wxString  os1;

  // Check if the schematic editor process is already running
  if( m_oPrcGSchem.bIsExec( ) )
  {
    os1 = wxT("The schematic editor process is already running.");
    DlgErrMsg( wxT("Schematic Editor Error"), os1 );
    return;
  }

  // Check if a schematic file is currently set
  m_oPrcGSchem.bSetSchems( m_oFileTsks.rosaGetSchemFiles( ) );
  if( m_oPrcGSchem.rosGetSchems( ).IsEmpty( ) )
  {
    os1 = wxT("The netlist has no associated schematic file to edit.");
    DlgErrMsg( wxT("Schematic Editor Error"), os1 );
    return;
  }

  // Execute the schematic editor process
  if( ! m_oPrcGSchem.bExec( ) )
    // Something went wrong, display an error message
    DlgErrMsg( os1, m_oPrcGSchem.rosGetErrMsg( ) );
}

//*****************************************************************************
// View the simulation results.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnViewData( wxCommandEvent & roEvtCmd )
{
  wxFileName  ofn1;
  wxString    os1 = wxT("Plot Results Error");
  wxString    os2;

  // Check if the waveform viewer process is already running
  if( m_oPrcGaw.bIsExec( ) || m_oPrcGWave.bIsExec( ) )
  {
    DlgErrMsg( os1, wxT("The waveform viewer process is already running.") );
    return;
  }

  // Check if a schematic or netlist file is currently open
  ofn1 = m_oFileTsks.rosGetNetLstFile( );
  if( ofn1.GetFullPath( ).IsEmpty( ) )
  {
    DlgErrMsg( os1, wxT("There is no schematic or netlist file currently loaded.") );
    return;
  }

  // Create the appropriate name for the results file
  os2 = ofn1.GetName( );
  switch( m_poNbkSimEng->eGetSimEng( ) )
  {
    case eSIMR_GNUCAP :
      if( m_oNbkTxtCtls.eGetPage( ) == NbkTxtCtls::ePAGE_NGSPICE )
           os2 << wxT(".ngspice");
      else os2 << wxT(".gnucap");
      break;

    case eSIMR_NGSPICE :
      if( m_oNbkTxtCtls.eGetPage( ) == NbkTxtCtls::ePAGE_GNUCAP )
           os2 << wxT(".gnucap");
      else os2 << wxT(".ngspice");
      break;

    default :
      return;
  }
  ofn1.SetName( os2 );
  ofn1.SetExt( m_poNbkSimEng->rosGetPage( ) );

  // Set the results file in the waveform viewer process
  if( ! ofn1.FileExists( ) )
  {
    os2.Empty( );
    os2 << wxT("The simulation results file :\n\n") << ofn1.GetFullPath( )
        << wxT("\n\nhasn't yet been generated.");
    DlgErrMsg( os1, os2 );
    return;
  }
  m_oPrcGaw  .bSetResults( ofn1.GetFullPath( ) );
  m_oPrcGWave.bSetResults( ofn1.GetFullPath( ) );

  // Execute the waveform viewer process
  m_poCfg->SetPath( wxT("/Viewer") );
  if( m_poCfg->Read( wxT("Name") ) == CLP_GWAVE )
  {
    if( ! m_oPrcGWave.bExec( ) )
      DlgErrMsg( os1, m_oPrcGWave.rosGetErrMsg( ) );
  }
  else
  {
    if( ! m_oPrcGaw.bExec( ) )
      DlgErrMsg( os1, m_oPrcGaw.rosGetErrMsg( ) );
  }
}

//*****************************************************************************
// Select which simulator to use.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSelSimEng( wxCommandEvent & roEvtCmd )
{
  int  ix, iy, iw, ih;

  // Temporarily store the current frame position and size
  GetPosition( &ix, &iy );
  GetClientSize( &iw, &ih );

  // Change cursor to the wait symbol and hide the GUI while re-constructing it
  ::wxBeginBusyCursor( );
  Show( FALSE );

  // Create and display a progress dialog
  wxProgressDialog  oDlgProgress( wxT(" GNU Spice GUI"),
                             wxT("Changing simulation engines ..."), 100, this,
                             wxPD_AUTO_HIDE | wxPD_APP_MODAL | wxPD_SMOOTH );
  oDlgProgress.CenterOnParent( );
  oDlgProgress.Update( 0 ); ::wxMilliSleep( 50 );

  // Record the simulation engine change in the configuration file
  m_poCfg->SetPath( wxT("/Simulator") );
  if( GetMenuBar( )->IsChecked( ID_MNU_GNUCAP ) )
       m_poCfg->Write( wxT("Engine"), CLP_GNUCAP  );
  else m_poCfg->Write( wxT("Engine"), CLP_NGSPICE );
  m_poCfg->Flush( ); // Write changes to the configuration file

  // Update the progress dialog
  oDlgProgress.Update( 10 );

  // Change the simulator
  InitSimEng( );

  // Update the progress dialog
  oDlgProgress.Update( 50 ); ::wxMilliSleep( 50 );

  // Set the simulator process log file name
  InitLogFiles( );

  // Update the progress dialog
  oDlgProgress.Update( 60 ); ::wxMilliSleep( 50 );

  // Update the GUI components
  DoLayout( );

  // Update the progress dialog
  oDlgProgress.Update( 70 ); ::wxMilliSleep( 50 );

  // Transfer info. from one simulation object to the other
  if( ! m_oNetLst.bIsEmpty( ) )
  {
    if( GetMenuBar( )->IsChecked( ID_MNU_GNUCAP ) ) m_oSimnGcp = m_oSimnNgs;
    else                                            m_oSimnNgs = m_oSimnGcp;
  }

  // Update the progress dialog
  oDlgProgress.Update( 80 ); ::wxMilliSleep( 50 );

  // Load the analysis notebook with the simulation values
  m_poNbkSimEng->bLoad( *m_poSimultn );

  // Update the progress dialog
  oDlgProgress.Update( 90 ); ::wxMilliSleep( 50 );

  // Restore the frame's original position and size
  Move( ix, iy );
  SetClientSize( iw, ih );

  // Update the progress dialog
  oDlgProgress.Update( 99 ); ::wxMilliSleep( 50 );
  oDlgProgress.Update( 100 ); // Kill the progress dialog

  // Show the GUI again and change the cursor to the default
  Show( TRUE );
  ::wxEndBusyCursor( );
}

//*****************************************************************************
// Show the application preferences dialog.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnPrefs( wxCommandEvent & roEvtCmd )
{
  DlgPrefs  oDlgPrefs( this );;
  wxString  os1;

  oDlgPrefs.CenterOnParent( );
  oDlgPrefs.ShowModal( );

  if( oDlgPrefs.GetReturnCode( ) == wxID_OK )
  {
    // Set the waveform viewer
    os1 = wxT(" Viewer : ") + oDlgPrefs.rosGetViewer( );
    SetStatusText( os1, 3 );

    // Set the text control maximum lines
    m_oNbkTxtCtls.bSetLinesMax( oDlgPrefs.iGetNbkMaxLines( ) );
  }
}

//*****************************************************************************
// Display the gSpiceUI manual.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnManual( wxCommandEvent & roEvtCmd )
{
  m_oHelpTsks.Manual( );
}

//*****************************************************************************
// Display about message dialog.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnAbout( wxCommandEvent & roEvtCmd )
{
  m_oHelpTsks.About( );
}

//*****************************************************************************
// Change the cursor when a simulation is running and the mouse is over the
// stop button.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnToolEnter( wxCommandEvent & roEvtCmd )
{
  if( roEvtCmd.GetSelection( ) == ID_TBR_STOP )
  {
    if( m_poPrcSimEng->bIsExec( ) &&  ::wxIsBusy( ) )
    {
      ::wxEndBusyCursor( );    // Change the cursor to the default
      ::wxYield( );            // Allow the display to update
    }
  }
  else
  {
    if( m_poPrcSimEng->bIsExec( ) && !::wxIsBusy( ) )
    {
      ::wxBeginBusyCursor( );  // Change the cursor to the hour glass
      ::wxYield( );            // Allow the display to update
    }
  }
}

//*****************************************************************************
// Event handler for system close.
//
// Argument List :
//   roEvtClose - The event to be processed

void  FrmMain::OnSysExit( wxCloseEvent & roEvtClose )
{
  wxString  os1;
  int       ix, iy, iw, ih;

  // Save the frame size and position
  m_poCfg->SetPath( wxT("/Main") );
  GetClientSize( &iw, &ih );
  GetPosition( &ix, &iy );
  m_poCfg->Write( wxT("PosnX"), (long) ix );
  m_poCfg->Write( wxT("PosnY"), (long) iy );
  m_poCfg->Write( wxT("SizeW"), (long) iw );
  m_poCfg->Write( wxT("SizeH"), (long) ih );
  m_poCfg->Flush( );

  // Record the simulation engine currently selected
  m_poCfg->SetPath( wxT("/Simulator") );
  os1 = m_poPrcSimEng->eGetSimEng( )==eSIMR_GNUCAP ? CLP_GNUCAP : CLP_NGSPICE;
  m_poCfg->Write( wxT("Engine"), os1 );

  // Record the analysis type last used
  os1 = m_poNbkSimEng->rosGetPage( );
  m_poCfg->Write( wxT("Analysis"), os1.Upper( ) );

  // Record the waveform veiwer currently selected
  m_poCfg->SetPath( wxT("/Viewer") );
  if( m_poCfg->Read( wxT("Name") ).IsEmpty( ) )
    m_poCfg->Write( wxT("Name"), CLP_GWAVE );

  bClear( );            // Kill any processes currently running

  m_oFileTsks.bExit( ); // Delete temporary files

  m_bIsOpen = FALSE;    // Indicate that the main frame has been closed

  // Destroys the window safely (used instead of the delete operator)
  Destroy( );
}

//*****************************************************************************
