//*****************************************************************************
//                                 NbkGnuCap.cpp                              *
//                                ---------------                             *
//  Started     : 05/09/2003                                                  *
//  Last Update : 12/04/2010                                                  *
//  Copyright   : (C) 2003 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/NbkGnuCap.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( NbkGnuCap, wxNotebook )

  EVT_NOTEBOOK_PAGE_CHANGED( -1,          NbkSimrBase::OnPageChanged )

  EVT_BUTTON( PnlAnaBase::ID_BTN_OPTIONS, NbkGnuCap::OnBtnOptions    )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Arguments:
//   poParent - The parent window
//   oWinID   - The display object ID
//   roPosn   - The notebook position
//   roSize   - The notebook size

NbkGnuCap::NbkGnuCap( wxWindow * poParent, wxWindowID oWinID )
                    : NbkSimrBase( poParent, oWinID ), m_oDlgCfgOPT( poParent )
{
  // Set the simulator engine type specifier
  m_eSimEng = eSIMR_GNUCAP;

  // Create the various display objects
  m_poPnlGnuCapOP = new PnlGnuCapOP( this );
  m_poPnlGnuCapDC = new PnlGnuCapDC( this );
  m_poPnlGnuCapAC = new PnlGnuCapAC( this );
  m_poPnlGnuCapTR = new PnlGnuCapTR( this );
  m_poPnlGnuCapFO = new PnlGnuCapFO( this );

  // Add the display objects to the note book
  AddPage( m_poPnlGnuCapOP, wxT( " Quiescent  "  ) );
  AddPage( m_poPnlGnuCapDC, wxT( "     DC      " ) );
  AddPage( m_poPnlGnuCapAC, wxT( "     AC      " ) );
  AddPage( m_poPnlGnuCapTR, wxT( " Transient  "  ) );
  AddPage( m_poPnlGnuCapFO, wxT( "   Fourier   " ) );

  // Specify the default page to be displayed
  SetSelection( 0 );
}

//*****************************************************************************
// Default constructor.
// (Used for two stage creation ie. must call Create( ) method.

NbkGnuCap::NbkGnuCap( void ) : NbkSimrBase( ), m_oDlgCfgOPT( this )
{
  // Set the simulator engine type specifier
  m_eSimEng = eSIMR_GNUCAP;
}

//*****************************************************************************
// Destructor.

NbkGnuCap::~NbkGnuCap( )
{
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkGnuCap::bClear( void )
{
  bool  bRtn=TRUE;

  // Clear the base class
  if( ! NbkSimrBase    ::bClear( ) ) bRtn = FALSE;

  if( ! m_poPnlGnuCapOP->bClear( ) ) bRtn = FALSE;
  if( ! m_poPnlGnuCapDC->bClear( ) ) bRtn = FALSE;
  if( ! m_poPnlGnuCapAC->bClear( ) ) bRtn = FALSE;
  if( ! m_poPnlGnuCapTR->bClear( ) ) bRtn = FALSE;
  if( ! m_poPnlGnuCapFO->bClear( ) ) bRtn = FALSE;

  if( ! m_oDlgCfgOPT    .bClear( ) ) bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Load information from a simulation object.
//
// Argument List:
//   roSimn - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkGnuCap::bLoad( const SimnBase & roSimn )
{
  bool  bRtn=TRUE;

  if( roSimn.eGetSimEng( ) != eSIMR_GNUCAP )             return( FALSE );

  SimnGnuCap & roSimnGcp = (SimnGnuCap &) roSimn;

  if( ! bSetPage( roSimnGcp.eGetAnaType( ) ) )           bRtn = FALSE;

  if( ! m_poPnlGnuCapOP->bLoad( roSimnGcp ) )            bRtn = FALSE;
  if( ! m_poPnlGnuCapDC->bLoad( roSimnGcp ) )            bRtn = FALSE;
  if( ! m_poPnlGnuCapAC->bLoad( roSimnGcp ) )            bRtn = FALSE;
  if( ! m_poPnlGnuCapTR->bLoad( roSimnGcp ) )            bRtn = FALSE;
  if( ! m_poPnlGnuCapFO->bLoad( roSimnGcp ) )            bRtn = FALSE;

  if( ! m_oDlgCfgOPT.bSetValues( roSimnGcp.m_oCmdOPT ) ) bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Save information to a simulation object.
//
// Argument List:
//   roSimn - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkGnuCap::bSave( SimnBase & roSimn )
{
  wxString  os1;

  // Only proceed if this is an GNU-Cap simulation object
  if( roSimn.eGetSimEng( ) != eSIMR_GNUCAP ) return( FALSE );

  SimnGnuCap & roSimnGcp = (SimnGnuCap &) roSimn;

  // Clear the error string
  NbkSimrBase::m_osErrMsg.Empty( );

  // Create the analysis command/s
  switch( GetSelection( ) )
  {
    case 0 : // OP analysis
      if( m_poPnlGnuCapOP->bSave( roSimnGcp ) )
           os1.Empty( );
      else SetErrMsg( m_poPnlGnuCapOP->rosGetErrMsg( ) );
      break;

    case 1 : // DC analysis
      if( m_poPnlGnuCapDC->bSave( roSimnGcp ) )
           os1 = roSimnGcp.m_oCmdDC.m_osTempC;
      else SetErrMsg( m_poPnlGnuCapDC->rosGetErrMsg( ) );
      break;

    case 2 : // AC analysis
      if( m_poPnlGnuCapAC->bSave( roSimnGcp ) )
           os1 = roSimnGcp.m_oCmdAC.m_osTempC;
      else SetErrMsg( m_poPnlGnuCapAC->rosGetErrMsg( ) );
      break;

    case 3 : // TR analysis
      if( m_poPnlGnuCapTR->bSave( roSimnGcp ) )
           os1 = roSimnGcp.m_oCmdTR.m_osTempC;
      else SetErrMsg( m_poPnlGnuCapTR->rosGetErrMsg( ) );
      break;

    case 4 : // FO analysis
      if( m_poPnlGnuCapFO->bSave( roSimnGcp ) )
           os1 = roSimnGcp.m_oCmdFO.m_osTempC;
      else SetErrMsg( m_poPnlGnuCapFO->rosGetErrMsg( ) );
      break;

    default:                                 return( FALSE );
  }
  if( ! bIsOk( ) )                           return( FALSE );

  // Create the OPTIONS command
  m_oDlgCfgOPT.bGetValues( roSimnGcp.m_oCmdOPT );
  if( ! os1.IsEmpty( ) ) roSimnGcp.m_oCmdOPT.m_osTEMPAMB = os1;
  roSimnGcp.m_oCmdOPT.bFormat( );
  if( ! roSimnGcp.m_oCmdOPT.bIsValid( ) )
  {
    os1 = wxT("OPTIONS command fault :\n\n  ");
    SetErrMsg( os1 + roSimnGcp.m_oCmdOPT.rosGetErrMsg( ) );
    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Set the page to be displayed.
//
// Argument List:
//   eCmdType - The enumerated analysis type specifier
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkGnuCap::bSetPage( eCmdType eAnalysis )
{
  int  iPage;

  switch( eAnalysis )
  {
    case eCMD_NONE :
    case eCMD_OP   : iPage = 0; break;
    case eCMD_DC   : iPage = 1; break;
    case eCMD_AC   : iPage = 2; break;
    case eCMD_TR   : iPage = 3; break;
    case eCMD_FO   : iPage = 4; break;
    default        : return( FALSE );
  }

  SetSelection( iPage ); // Specify the page to be displayed

  return( TRUE );
}

//*****************************************************************************
// Set the page to be displayed.
//
// Argument List:
//   psAnalysis - The two letter analysis type specifier (case ignored)
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NbkGnuCap::bSetPage( const wxChar * psAnalysis )
{
  eCmdType  ePage=eCMD_NONE;
  wxString  osAnalysis;

  osAnalysis = wxString( psAnalysis ).Upper( );

  if( osAnalysis.Length( ) == 2 )
  {
    if(      osAnalysis == wxT("OP") ) ePage = eCMD_OP;
    else if( osAnalysis == wxT("DC") ) ePage = eCMD_DC;
    else if( osAnalysis == wxT("AC") ) ePage = eCMD_AC;
    else if( osAnalysis == wxT("TR") ) ePage = eCMD_TR;
    else if( osAnalysis == wxT("FO") ) ePage = eCMD_FO;
    else if( osAnalysis == wxT("DI") ) ePage = eCMD_DI;
    else if( osAnalysis == wxT("NO") ) ePage = eCMD_NO;
    else if( osAnalysis == wxT("PZ") ) ePage = eCMD_PZ;
    else if( osAnalysis == wxT("SE") ) ePage = eCMD_SE;
    else if( osAnalysis == wxT("TF") ) ePage = eCMD_TF;
  }

  return( bSetPage( ePage ) );
}

//*****************************************************************************
// Get the two letter page specifier.
//
// Return Values:
//   Success - The two letter analysis type specifier (lower case)
//   Failure - An empty string

const wxString & NbkGnuCap::rosGetPage( void )
{
  static  wxString  osPage;

  switch( eGetPage( ) )
  {
    case eCMD_OP : osPage = wxT("op"); break;
    case eCMD_DC : osPage = wxT("dc"); break;
    case eCMD_AC : osPage = wxT("ac"); break;
    case eCMD_TR : osPage = wxT("tr"); break;
    case eCMD_FO : osPage = wxT("fo"); break;
    default      : osPage.Empty( );
  }

  return( osPage );
}

//*****************************************************************************
// Get the enumerated page specifier.
//
// Return Values:
//   Success - The enumerated analysis specifier
//   Failure - Simulation::eANA_NONE

eCmdType  NbkGnuCap::eGetPage( void )
{
  eCmdType  ePage;

  switch( GetSelection( ) )
  {
    case  0 : ePage = eCMD_OP; break;
    case  1 : ePage = eCMD_DC; break;
    case  2 : ePage = eCMD_AC; break;
    case  3 : ePage = eCMD_TR; break;
    case  4 : ePage = eCMD_FO; break;
    default : ePage = eCMD_NONE;
  }

  return( ePage );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// .OPTIONS command setup button control event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  NbkGnuCap::OnBtnOptions( wxCommandEvent & roEvtCmd )
{
  PnlAnaBase * poPnlAna;
  wxString     os1;
  int          i1;

  // Set the temperature in the OPTIONS dialog
  poPnlAna = (PnlAnaBase *) GetPage( GetSelection( ) );
  if( poPnlAna->bIsCreated( PnlAnaBase::eCTLGRP_TEMP ) )
  {
    os1 = poPnlAna->m_oPnlTemp.rosGetValue( );
    m_oDlgCfgOPT.bSetValue( DlgGcpCfgOPT::ID_PNL_TEMPAMB, os1 );
  }

  // Display the OPTIONS dialog
  m_oDlgCfgOPT.CenterOnParent( );
  i1 = m_oDlgCfgOPT.ShowModal( );

  // Set the temperature in the analysis panel
  if( poPnlAna->bIsCreated( PnlAnaBase::eCTLGRP_TEMP ) && i1==wxID_OK )
  {
    os1 = m_oDlgCfgOPT.rosGetValue( DlgGcpCfgOPT::ID_PNL_TEMPAMB );
    poPnlAna->m_oPnlTemp.bSetValue( os1 );
  }
}

//*****************************************************************************
