//*****************************************************************************
//                              CmdNgSpiceTR.cpp                              *
//                             ------------------                             *
//  Started     : 15/10/2007                                                  *
//  Last Update : 21/02/2008                                                  *
//  Copyright   : (C) 2007 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/commands/CmdNgSpiceTR.hpp"

//*****************************************************************************
// Constructor.

CmdNgSpiceTR::CmdNgSpiceTR( void )
{
  bClear( );
}

//*****************************************************************************
// Destructor.

CmdNgSpiceTR::~CmdNgSpiceTR( )
{
}

//*****************************************************************************
// Parse the command string.
//
// Eg.s : .TRAN 10.00m 100.00m 0.00 10.00m UIC
//        .TRAN 10.00m 100.00m 0.00 10.00m
//        .TRAN 10.00m 100.00m 0.00
//        .TRAN 10.00m 100.00m
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceTR::bParse( void )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  double             df1;
  int                i1;

  // Clear the object attributes
  os1 = *this;
  bClear( );
  *this = os1;

  // Tokenize the command string
  ostk1.SetString( *this );
  i1 = ostk1.CountTokens( );
  if( i1<3 || i1>6 )        return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( ).Left( 5 ).Upper( );
  if( os1 != wxT(".TRAN") ) return( FALSE );
  m_osName = wxT("TRAN");

  m_bIsOk = TRUE;

  // Extract the step increment (must be present)
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df1 ) )   m_fStep  = (float) df1;
  else                      m_bIsOk = FALSE;

  // Extract the stop value (must be present)
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df1 ) )   m_fStop  = (float) df1;
  else                      m_bIsOk = FALSE;

  // Extract the start value (may be omitted)
  m_fStart = 0.0;
  if( ostk1.HasMoreTokens( ) )
  {
    os1 = ostk1.GetNextToken( );
    if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fStart = (float) df1;
    else                    m_bIsOk = FALSE;
  }

  // Extract the initial conditions specifier
  m_eInitC = eINITC_WARM;
  while( ostk1.HasMoreTokens( ) )
  {
    os1 = ostk1.GetNextToken( ).Upper( );
    if( os1 == wxT("UIC") )
    {
      m_eInitC = eINITC_UICS;
      break;
    }
  }

  return( m_bIsOk );
}

//*****************************************************************************
// Format the command string.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceTR::bFormat( void )
{
  wxString  osCmd, os1;

  m_bIsOk = FALSE;

  osCmd = wxT(".TRAN");

  if( ! ConvertType::bFltToStrEng( (double) m_fStep,  os1 ) ) return( FALSE );
  osCmd << wxT(' ') << os1;

  if( ! ConvertType::bFltToStrEng( (double) m_fStop,  os1 ) ) return( FALSE );
  osCmd << wxT(' ') << os1;

  if( ! ConvertType::bFltToStrEng( (double) m_fStart, os1 ) ) return( FALSE );
  osCmd << wxT(' ') << os1;

  if( m_eInitC == eINITC_UICS ) osCmd << wxT(" UIC");

  *this = osCmd;

  m_bIsOk = TRUE;

  return( m_bIsOk );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceTR::bClear( void )
{
  CmdBase::bClear( );

  m_fStart = -FLT_MAX;
  m_fStop  = -FLT_MAX;
  m_fStep  = -FLT_MAX;

  m_eInitC = eINITC_NONE;

  return( TRUE );
}

//*****************************************************************************
//                                                                            *
//                                 Test Utility                               *
//                                                                            *
//*****************************************************************************

#ifdef TEST_UTIL

// System include files


// Application includes


// Function prototypes

void  Usage( char * psAppName );

//*****************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osCmd;
  wxString  os1;

  // Validate the argument count passed to the application
  if( argc > 2 )           { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1 == wxT("-h") ) { Usage( argv[ 0 ] ); exit( EXIT_SUCCESS ); }
    else                   { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }
  }

  // Display the utility banner
  cout << "\n  Class CmdNgSpiceTR Test Utility"
       << "\n     Version 1.00 (01/02/2008)\n";

  // Create a NG-SPICE TRANSIENT command object
  CmdNgSpiceTR  oCmd_TR;

  // Use the following command example to check the formatter and the parser :
  osCmd = wxT(".TRAN 10.00m 100.00m 0.00 UIC");

  // Set things up for a formatter test
  oCmd_TR.bClear( );
  oCmd_TR.m_fStart = 0.0;
  oCmd_TR.m_fStop  = 100.0E-3;
  oCmd_TR.m_fStep  = 10.0E-3;
  oCmd_TR.m_eInitC = eINITC_UICS;
  cout << "\nRun Formatter    : " << ( oCmd_TR.bFormat( ) ? "OK" : "FAULT" );
  cout << "\nTest Cmd Format  : " << ( oCmd_TR == osCmd   ? "OK" : "FAULT" );
  cout << "\nExample Command  : " << osCmd  .mb_str( );
  cout << "\noCmd_TR Contents : " << oCmd_TR.mb_str( ) << '\n';

  // Set things up for a parser test
  oCmd_TR.bClear( );
  oCmd_TR = osCmd;
  cout << "\nRun Parser       : " << ( oCmd_TR.bParse( ) ? "OK" : "FAULT" );
  oCmd_TR.bFormat( );
  cout << "\nTest Cmd Format  : " << ( oCmd_TR == osCmd  ? "OK" : "FAULT" );
  cout << "\nExample Command  : " << osCmd  .mb_str( );
  cout << "\noCmd_TR Contents : " << oCmd_TR.mb_str( ) << '\n';

  cout << '\n';

  exit( EXIT_SUCCESS );
}

//*****************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)\n";
}

#endif // TEST_UTIL

//*****************************************************************************
