//*****************************************************************************
//                               PrcNgSpice.cpp                               *
//                              ----------------                              *
//  Started     : 07/05/2004                                                  *
//  Last Update : 24/10/2007                                                  *
//  Copyright   : (C) 2004 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "process/PrcNgSpice.hpp"

//*****************************************************************************
// Constructor.

PrcNgSpice::PrcNgSpice( void ) : PrcSimrBase( )
{
  // Set the simulation engine type
  m_eSimrType = eSIMR_NGSPICE;

  // Set the simulator binary file name if it can be found
  bSetBinary( wxT("ngspice") );
}

//*****************************************************************************
// Destructor.

PrcNgSpice::~PrcNgSpice( )
{
}

//*****************************************************************************
// Parse a OP command and put the data in a simulation object.
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseCmdOP( wxString & rosCmd, Simulation & roSim )
{
  // ??? Not yet implemented

  return( FALSE );
}

//*****************************************************************************
// Parse a IC command and put the data in a simulation object.
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseCmdIC( wxString & rosCmd, Simulation & roSim )
{
  // ??? Not yet implemented

  return( FALSE );
}

//*****************************************************************************
// Parse a DC command and put the data in a simulation object.
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseCmdDC( wxString & rosCmd, Simulation & roSim )
{
  bool          bRtnValue=TRUE;
  CmdNgSpiceDC  oCmd_DC;
  wxString      os1;

  // Parse the DC command
  if( ! oCmd_DC.bSetCmd( rosCmd ) )              return( FALSE );

  // Extract the sweep type: source or temperature
  if( oCmd_DC.m_osSource == wxT("TEMP") )
    roSim.bSetSwpScale( 1 );
  else
  {
    os1 = roSim.rosGetCpnt( oCmd_DC.m_osSource );
    if( !os1.IsEmpty( ) && roSim.bSetSigSrc( os1 ) )
         roSim.bSetSwpScale( 0 );
    else                                         bRtnValue = FALSE;
  }

  // Extract the sweep start, stop and step values
  if( ! roSim.bSetSwpStart( oCmd_DC.m_fStart ) ) bRtnValue = FALSE;
  if( ! roSim.bSetSwpStop ( oCmd_DC.m_fStop  ) ) bRtnValue = FALSE;
  if( ! roSim.bSetSwpStep ( oCmd_DC.m_fStep  ) ) bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Parse a AC command and put the data in a simulation object.
// Eg.s: .AC LIN 30 1.00K 300.00K
//       .AC DEC 30 1.00K 300.00K
//       .AC OCT 30 1.00K 300.00K
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseCmdAC( wxString & rosCmd, Simulation & roSim )
{
  wxStringTokenizer  ostk1;
  wxString  os1;
  bool      bRtnValue=TRUE;
  double    df1;
  long      li1;

  // Basic argument checks
  ostk1.SetString( rosCmd.Upper( ) );
  if( ostk1.CountTokens( ) != 5 )             return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( );
  if( os1 != wxT(".AC") )                     return( FALSE );

  // Extract the sweep type: linear or log
  os1 = ostk1.GetNextToken( );
  if(      os1 == wxT("LIN") ) roSim.bSetSwpScale( 0 );
  else if( os1 == wxT("DEC") ) roSim.bSetSwpScale( 2 );
  else if( os1 == wxT("OCT") ) roSim.bSetSwpScale( 3 );
  else                                        bRtnValue = FALSE;

  // Extract the step count
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToInt( os1, &li1 ) ) bRtnValue = FALSE;
  if( ! roSim.bSetSwpStep( (float) li1 ) )    bRtnValue = FALSE;

  // Extract the start frequency
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) ) bRtnValue = FALSE;
  if( ! roSim.bSetSwpStart( (float) df1 ) )   bRtnValue = FALSE;

  // Extract the stop frequency
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) ) bRtnValue = FALSE;
  if( ! roSim.bSetSwpStop( (float) df1 ) )    bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Parse a TRANSIENT command and put the data in a simulation object.
// Eg.s: .TRAN 10.00m 100.00m 0.00 10.00m
//       .TRAN 10.00m 100.00m 0.00 10.00m UIC
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseCmdTR( wxString & rosCmd, Simulation & roSim )
{
  wxStringTokenizer  ostk1;
  wxString  os1;
  bool      bRtnValue=TRUE;
  double    df1;
  int       i1;

  // Basic argument checks
  ostk1.SetString( rosCmd );
  i1 = ostk1.CountTokens( );
  if( i1!=5 && i1!=6 )                        return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( );
  if( os1.Upper( ) != wxT(".TRAN") )          return( FALSE );

  // Extract the step size (must be present)
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) ) bRtnValue = FALSE;
  if( ! roSim.bSetSwpStep( (float) df1 ) )    bRtnValue = FALSE;

  // Extract the stop time (must be present)
  os1 = ostk1.GetNextToken( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) ) bRtnValue = FALSE;
  if( ! roSim.bSetSwpStop( (float) df1 ) )    bRtnValue = FALSE;

  // Extract the start time (may not be specified)
  os1.Empty( );
  if( ostk1.HasMoreTokens( ) )
  {
    os1 = ostk1.GetNextToken( );
    if( ! ConvertType::bStrToFlt( os1, &df1 ) ) df1 = 0.0;
    else os1.Empty( );
    if( ! roSim.bSetSwpStart( (float) df1 ) ) bRtnValue = FALSE;
  }

  // Extract the initial conditions
  while( ostk1.HasMoreTokens( ) ) os1 = ostk1.GetNextToken( );
  if( os1.Upper( ) == wxT("UIC") ) i1 = (int) eINITC_UICS;
  else                             i1 = (int) eINITC_WARM;
  if( ! roSim.bSetSwpScale( i1 ) )            bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
// Parse a PRINT command and put the data in a simulation object.
//
// Argument List:
//   rosCmd - The command line to be parsed
//   roSim  - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseCmdPR( wxString & rosCmd, Simulation & roSim )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  bool               bRtnValue=TRUE;
  size_t             sz1, sz2;

  // Basic argument checks
  ostk1.SetString( rosCmd.Upper( ) );
  if( ostk1.CountTokens( ) < 3 )      return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( ).Left( 6 );
  if( os1 != wxT(".PRINT") ) return( FALSE );

  // Extract the analysis type
  os1 = ostk1.GetNextToken( );
  if(      os1 == wxT("DC")   ) roSim.bSetAnaType( eANA_DC );
  else if( os1 == wxT("AC")   ) roSim.bSetAnaType( eANA_AC );
  else if( os1 == wxT("TRAN") ) roSim.bSetAnaType( eANA_TR );
  else return( FALSE );

  // Extract the parameters to derive, any complex parts and the test component
  // and/or test node labels
  while( ostk1.HasMoreTokens( ) )
  {
    // Extract the next field
    os1 = ostk1.GetNextToken( );
    if( os1.Length( ) < 4 ) { bRtnValue = FALSE; continue; }

    // Standardize the parameter format; there are 3 possible formats eg. V(n),
    // V(n,m) or 0-V(n). The latter is reformatted to : V(0,n).
    if( os1.StartsWith( wxT("0-") ) )
    {
      sz1 = os1.Index( wxT('(') );
      sz2 = os1.Index( wxT(')') );
      if( (int)sz1!=wxNOT_FOUND && (int)sz2!=wxNOT_FOUND && sz2>(sz1+1) )
        os1 = os1.Mid( 2, sz1-1 ) + wxT("0,") + os1.Mid( sz1+1 );
    }

    // Extract the parameter specifier
    switch( os1.GetChar( 0 ) )
    {
      case wxT('V') : roSim.bSetOutPar( ePAR_VLT, TRUE ); break;
      case wxT('I') : roSim.bSetOutPar( ePAR_CUR, TRUE ); break;
      case wxT('P') : roSim.bSetOutPar( ePAR_PWR, TRUE ); break;
      case wxT('R') : roSim.bSetOutPar( ePAR_RES, TRUE ); break;
      default :       bRtnValue = FALSE; break;
    }

    // Extract the complex part if the analysis type is AC
    if( roSim.eGetAnaType( ) == eANA_AC )
    {
      switch( os1.GetChar( 1 ) )
      {
        case wxT('M') : roSim.bSetOutCpx( eCPX_MAG,   TRUE ); break;
        case wxT('P') : roSim.bSetOutCpx( eCPX_PHASE, TRUE ); break;
        case wxT('R') : roSim.bSetOutCpx( eCPX_REAL,  TRUE ); break;
        case wxT('I') : roSim.bSetOutCpx( eCPX_IMAG,  TRUE ); break;
        case wxT('D') :
          if( os1.Mid( 1, 2 ).Upper( ) == wxT("DB") )
          {
            roSim.bSetOutCpx( eCPX_MAG,   TRUE );
            roSim.bSetOutCpx( eCPX_MAGDB, TRUE );
            break;
          }
        default: bRtnValue = FALSE; break;
      }
    }

    // Extract the node label/s, there are 2 possible formats eg. V(n) or
    // V(n,m). The first denotes a node and the second a component.
    sz1 = os1.Index( wxT('(') );
    sz2 = os1.Index( wxT(')') );
    if( (int)sz1==wxNOT_FOUND || (int)sz2==wxNOT_FOUND || (sz1+1)>=sz2 )
       { bRtnValue = FALSE; continue; }
    os1 = os1.Mid( sz1+1, sz2-sz1-1 );
    if( roSim.roasGetNodeLbls( ).Index( os1 ) != wxNOT_FOUND )
      roSim.bAddTstNode( os1 );
    else if( bMatchCpnt( roSim, os1 ) )
      roSim.bAddTstCpnt( os1 );
    else bRtnValue = FALSE;
  }

  return( bRtnValue );
}

//*****************************************************************************
// Match a component to a set of node labels. Component voltages cannot be
// specified using the component label in the Ng-Spice print statement. The two
// node labels connected to the component must be specified. When parsing the
// print statement the component label must be derived from the node labels.
// This may not work where components are connected in parallel. The nodes are
// specified in the form "<Node1>,<Node2>" eg. "1,2".
//
// Argument List:
//   rosToNodes - The nodes to match, also used to return the component label
//
// Return Values:
//   TRUE  - Success (A component with the specified nodes was found)
//   FALSE - Failure

bool  PrcNgSpice::bMatchCpnt( Simulation & roSim, wxString & rosToNodes )
{
  const wxArrayString & roasCpnts=roSim.roasGetCpnts( );
  wxArrayString  oasNodes;
  Component  oCpnt;
  wxString   os1;
  size_t     sz1;

  // Argument validity checks
  if( rosToNodes.IsEmpty( ) )            return( FALSE );
  if( roasCpnts.GetCount( ) <= 0 )       return( FALSE );
  if( rosToNodes.Freq( wxT(',') ) != 1 ) return( FALSE );

  // Extract the node labels
  oasNodes.Add( rosToNodes.BeforeFirst( wxT(',') ) );
  oasNodes.Add( rosToNodes.AfterLast( wxT(',') ) );

  // Attempt to match the nodes with a component
  for( sz1=0; sz1<roasCpnts.GetCount( ); sz1++ )
  {
    os1 = roasCpnts.Item( sz1 );
    oCpnt.bSetDefn( os1 );
    if( oCpnt.roasGetNodes( ) == oasNodes ) break;
    oCpnt.Clear( );
  }

  // Was a match found?
  if( ! oCpnt.bIsOk( ) )                 return( FALSE );

  // A match was found
  rosToNodes = oCpnt.rosGetName( );
  return( TRUE );
}

//*****************************************************************************
// Create the simulator command sequence for a DC analysis.
//
// Argument List:
//   roSim - Simulation object

void  PrcNgSpice::MakeCmdDC( Simulation & roSim )
{
  wxString  osCmdWIDTH;
  wxString  osCmdPR;
  wxString  osCmdDC;
  wxString  osPar, os1;
  uint      uiPar;

  // Clear the field list
  m_osFieldLst.Empty( );

  // Create the DC command
  osCmdDC = wxT(".DC ");
  if( roSim.iGetSwpScale( ) == 0 )
       osCmdDC << roSim.rosGetSigSrc( ).BeforeFirst( wxT(' ') ) << wxT(' ');
  else osCmdDC << wxT("TEMP ");
  osCmdDC << roSim.rosGetSwpStart( ) << wxT(' ')
          << roSim.rosGetSwpStop ( ) << wxT(' ')
          << roSim.rosGetSwpStep ( );

  // Create the PRINT command
  osCmdPR = wxT(".PRINT DC");
  for( uiPar=ePAR_FST; uiPar<=ePAR_LST; uiPar++ )
  {
    if( ! roSim.bGetOutPar( (eParType) uiPar ) ) continue;

    switch( uiPar )
    {
      case ePAR_VLT : osPar = wxT("V("); break;
      case ePAR_CUR : osPar = wxT("I("); break;
      case ePAR_PWR : osPar = wxT("P("); break;
      case ePAR_RES : osPar = wxT("R("); break;
      default :       continue;
    }

    // Add any components and/or nodes to derivation list for this parameter
    os1 = rosMakeArgsPR( roSim, osPar );
    if( ! os1.IsEmpty( ) ) osCmdPR << os1;
  }

  // Create the WIDTH command
  osCmdWIDTH = rosMakeCmdWIDTH( osCmdPR );

  // Add simulation commands to Simulation object
  roSim.bAddSimCmd( osCmdWIDTH );
  roSim.bAddSimCmd( osCmdPR );
  roSim.bAddSimCmd( osCmdDC );
}

//*****************************************************************************
// Create the simulator command sequence for a AC analysis.
//
// Argument List:
//   roSim - Simulation object

void  PrcNgSpice::MakeCmdAC( Simulation & roSim )
{
  wxString  osCmdWIDTH;
  wxString  osCmdPR;
  wxString  osCmdOP;
  wxString  osCmdAC;
  wxString  osPar, osCpx, os1;
  uint      uiPar, uiCpx;

  // Clear the field list
  m_osFieldLst.Empty( );

  // Create the AC command
  osCmdAC = wxT(".AC ");
  switch( roSim.iGetSwpScale( ) )
  {
    case 0 : osCmdAC << wxT("LIN "); break;
    case 2 : osCmdAC << wxT("DEC "); break;
    case 3 : osCmdAC << wxT("OCT "); break;
  }
  osCmdAC << ( (int) roSim.fGetSwpStep( ) ) << wxT(' ')
          << roSim.rosGetSwpStart( ) << wxT(' ')
          << roSim.rosGetSwpStop( );

  // Create the PRINT command
  osCmdPR = wxT(".PRINT AC");
  for( uiPar=ePAR_FST; uiPar<=ePAR_LST; uiPar++ )
  {
    if( ! roSim.bGetOutPar( (eParType) uiPar ) ) continue;

    // Add the parameter prefix
    switch( uiPar )
    {
      case ePAR_VLT : osPar = wxT('V'); break;
      case ePAR_CUR : osPar = wxT('I'); break;
      case ePAR_PWR : osPar = wxT('P'); break;
      case ePAR_RES : osPar = wxT('R'); break;
      default: continue;
    }

    // Add the complex part prefix
    for( uiCpx=eCPX_FST; uiCpx<=eCPX_LST; uiCpx++ )
    {
      if( ! roSim.bGetOutCpx( (eCpxType) uiCpx ) ) continue;

      osCpx = osPar;

      switch( uiCpx )
      {
        case eCPX_MAG :
          if( ! roSim.bGetOutCpx( eCPX_MAGDB ) ) osCpx << wxT('M');
          else                                   osCpx << wxT("DB");
          break;
        case eCPX_PHASE : osCpx << wxT('P'); break;
        case eCPX_REAL  : osCpx << wxT('R'); break;
        case eCPX_IMAG  : osCpx << wxT('I'); break;
        default :         continue;
      }
      osCpx << wxT('(');

      // Add any components and/or nodes to derivation list for this parameter
      os1 = rosMakeArgsPR( roSim, osCpx );
      if( ! os1.IsEmpty( ) ) osCmdPR << os1;
    }
  }

  // Create the WIDTH command
  osCmdWIDTH = rosMakeCmdWIDTH( osCmdPR );

  // Add simulation commands to Simulation object
  roSim.bAddSimCmd( osCmdWIDTH );
  roSim.bAddSimCmd( osCmdPR );
  roSim.bAddSimCmd( osCmdOP );
  roSim.bAddSimCmd( osCmdAC );
}

//*****************************************************************************
// Create the simulator command sequence for a transient analysis.
//
// Argument List:
//   roSim - Simulation object

void  PrcNgSpice::MakeCmdTR( Simulation & roSim )
{
  wxString  osCmdIC;
  wxString  osCmdWIDTH;
  wxString  osCmdPR;
  wxString  osCmdTR;
  wxString  osPar, os1;
  uint      uiPar, ui1;
  size_t    sz1;

  // Clear the field list
  m_osFieldLst.Empty( );

  // Create initial conditions command
  ui1 = (uint) roSim.iGetSwpScale( );
  if( ui1==eINITC_COLD || ui1==eINITC_UICS )
  {
    const wxArrayString & roas1 = roSim.roasGetNodeLbls( );
    if( roas1.GetCount( ) > 0 )
    {
      osCmdIC << wxT(".IC");
      for( sz1=0; sz1<roas1.GetCount( ); sz1++ )
        osCmdIC << wxT(" V(") << roas1.Item( sz1 ) << wxT(")=0.0");
    }
  }

  // Create the TR command
  osCmdTR << wxT(".TRAN ")
          << roSim.rosGetSwpStep( ) << wxT(' ')
          << roSim.rosGetSwpStop( ) << wxT(' ')
          << roSim.rosGetSwpStart( ) << wxT(' ')
          << roSim.rosGetSwpStep( );  // TMAX calculation increment
  if( roSim.iGetSwpScale( ) == eINITC_UICS )
    osCmdTR << wxT(" UIC");

  // Create the PRINT commands
  osCmdPR << wxT(".PRINT TRAN");
  for( uiPar=ePAR_FST; uiPar<=ePAR_LST; uiPar++ )
  {
    if( ! roSim.bGetOutPar( (eParType) uiPar ) ) continue;

    switch( uiPar )
    {
      case ePAR_VLT : osPar = wxT("V("); break;
      case ePAR_CUR : osPar = wxT("I("); break;
      case ePAR_PWR : osPar = wxT("P("); break;
      case ePAR_RES : osPar = wxT("R("); break;
      default :       continue;
    }

    // Add any components and/or nodes to derivation list for this parameter
    os1 = rosMakeArgsPR( roSim, osPar );
    if( ! os1.IsEmpty( ) ) osCmdPR << os1;
  }

  // Create the WIDTH command
  osCmdWIDTH = rosMakeCmdWIDTH( osCmdPR );

  // Add simulation commands to Simulation object
  roSim.bAddSimCmd( osCmdIC );
  roSim.bAddSimCmd( osCmdWIDTH );
  roSim.bAddSimCmd( osCmdPR );
  roSim.bAddSimCmd( osCmdTR );
}

//*****************************************************************************
// Generate the simulator commands to be executed based on the simulation
// specifications and the NetList description.
//
// Arguments:
//   roSim - The net list and simulation specification
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bMakeCmds( Simulation & roSim )
{
  switch( roSim.eGetAnaType( ) )
  {
    case eANA_DC : MakeCmdDC( roSim ); break;
    case eANA_AC : MakeCmdAC( roSim ); break;
    case eANA_TR : MakeCmdTR( roSim ); break;
    default :      return( FALSE );
  }

  return( ! roSim.roasGetSimCmds( ).IsEmpty( ) );
}

//*****************************************************************************
// Create a list of arguments for the Spice PRINT command based on the test
// components and test nodes specified in a Simulation object. A prefix may be
// specified which is prepended to each argument in the list.
//
// Argument List:
//   roSim     - The Simulation object
//   rosPrefix - The prefix to be prepended to each argument
//
// Return Values:
//   Success - A list of correctly formatted arguments
//   Failure - An empty string

wxString & PrcNgSpice::rosMakeArgsPR( Simulation & roSim, wxString & rosPrefix )
{
  static  wxString  osArgs;
  Component  oCpnt;
  wxString   os1;
  wxChar     oc1;
  size_t     sz1;

  osArgs.Empty( );

  // Add any components to derivation list for this parameter
  for( sz1=0; sz1<roSim.roasGetTstCpnts( ).GetCount( ); sz1++ )
  {
    os1 = roSim.roasGetTstCpnts( ).Item( sz1 );
    m_osFieldLst << rosPrefix << os1 << wxT(") ");
    os1 = roSim.rosGetCpnt( os1.c_str( ) );
    if( os1.IsEmpty( ) )                   continue;
    if( ! oCpnt.bSetDefn( os1.c_str( ) ) ) continue;
    const wxArrayString & roas2 = oCpnt.roasGetNodes( );
    if( roas2.GetCount( ) != 2 )           continue;
    osArgs << wxT(' ');
    if( roas2.Item( 0 ).IsSameAs( wxT('0') ) )
         osArgs << wxT("0-") << rosPrefix << roas2.Item( 1 ) << wxT(")");
    else osArgs << rosPrefix << roas2.Item( 0 ) << wxT(',') << roas2.Item( 1 )
                << wxT(')');
  }

  // Add any nodes to derivation list for voltage only
  for( oc1=0, sz1=0; sz1<rosPrefix.Length( ); sz1++ )
  {
    oc1 = rosPrefix.GetChar( sz1 );
    if( ! wxIsspace( oc1 ) ) break;
  }
  if( oc1==wxT('v') || oc1==wxT('V') )
  {
    for( sz1=0; sz1<roSim.roasGetTstNodes( ).GetCount( ); sz1++ )
    {
      os1.Empty( );
      os1 << wxT(' ') << rosPrefix << roSim.roasGetTstNodes( ).Item( sz1 )
          << wxT(')');
      osArgs << os1;
      m_osFieldLst << os1 << wxT(' ');
    }
  }

  return( osArgs );
}

//*****************************************************************************
// Create a width command based on a PRINT command.
// By default raw output from NG-Spice uses an 80 column output width, allows
// about two dependant variables to be displayed. To display more information
// a .WIDTH statement must be added setting the number columns in the output.
//
// Argument List:
//   rosCmdPR - The Print command
//
// Return Values:
//   Success - A .WIDTH command
//   Failure - An empty string

wxString & PrcNgSpice::rosMakeCmdWIDTH( wxString & rosCmdPR )
{
  static  wxString  osCmdWIDTH;
  wxStringTokenizer  ostk1;
  wxString  os1;
  int  i1=0;

  osCmdWIDTH.Empty( );

  ostk1 = rosCmdPR;
  if( ostk1.GetNextToken( ).StartsWith( wxT(".PR") ) )
  {
    os1 = ostk1.GetNextToken( );

    if(      os1.StartsWith( wxT("DC") ) )
      i1 = 8 + 16 + ostk1.CountTokens( ) * 16;
    else if( os1.StartsWith( wxT("AC") ) )
      i1 = 8 + 2 * 16 + ostk1.CountTokens( ) * 16;
    else if( os1.StartsWith( wxT("TR") ) )
      i1 = 8 + 16 + ostk1.CountTokens( ) * 16;

    if( i1 > 80 ) osCmdWIDTH << wxT(".WIDTH OUT=") << i1;
  }

  return( osCmdWIDTH );
}

//*****************************************************************************
// Format the contents of the results file so that gWave can read it ie. a
// header line at the top containing parameter names followed by the columns of
// data.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bFmtResults( void )
{
  bool  bRtnValue;

  m_osErrMsg.Empty( );

  // Attempt to open the results file
  if( ! roGetResultsFile( ).FileExists( ) )
  {
    m_osErrMsg << wxT("Results file doesn't exist : \n\n")
               << roGetResultsFile( ).GetFullPath( );
    return( FALSE );
  }
  if( ! m_oFileResults.Open( roGetResultsFile( ).GetFullPath( ) ) )
  {
    m_osErrMsg << wxT("Results file couldn't be opened : \n\n")
               << roGetResultsFile( ).GetFullPath( );
    return( FALSE );
  }

  // Format the simulation results
  bRtnValue = bFmtResultsAny( );

  // Need last line to end with a '\n' or text controls will not load it
  if( m_oFileResults.GetLastLine( ).Last( ) != wxT('\n') )
    m_oFileResults.GetLastLine( ) << wxT('\n');

  m_oFileResults.Write( );  // Save the changes to disk
  m_oFileResults.Close( );  // Close the file

  return( bRtnValue );
}

//*****************************************************************************
// This function is a generic results file formatter which works in most
// circumstance.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bFmtResultsAny( void )
{
  wxString  os1;
  size_t    sz1, sz2;

  // This function requires the result file to have been opened
  if( ! m_oFileResults.IsOpened( ) )               return( FALSE );

  // Find the beginning of the data area (ie. the 1st line starting with "Index")
  for( sz1=0; sz1<m_oFileResults.GetLineCount( ); sz1++ )
  {
    os1 = m_oFileResults.GetLine( sz1 );
    if( os1.StartsWith( wxT("Index") ) ) break;
  }
  if( sz1 >= (m_oFileResults.GetLineCount( )-1) )
  {
    m_osErrMsg << wxT("Couldn't find the data section in the results file.");
    return( FALSE );
  }

  // Delete everything before the data area
  sz2 = sz1;
  for( sz1=0; sz1<sz2; sz1++ )          m_oFileResults.RemoveLine( 0 );
  if( m_oFileResults.GetLineCount( ) > 0 )  m_oFileResults.RemoveLine( 1 );

  // Format the column header line
  bFmtColumnHdr( );

  // Delete lines other than data lines
  for( sz1=1; sz1<m_oFileResults.GetLineCount( ); sz1++ )
  {
    os1 = m_oFileResults.GetLine( sz1 );
    if( ! wxIsdigit( os1.GetChar( 0 ) ) )
    {
      m_oFileResults.RemoveLine( sz1 );
      sz1--;
    }
  }

  // Format the data lines
  bFmtDataLines( );

  return( TRUE );
}

//*****************************************************************************
// Format the column header in the results file.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bFmtColumnHdr( void )
{
  wxStringTokenizer  ostk1;
  wxString  osLine, os1;
  int  i1;

  // This function requires the result file to have been opened
  if( ! m_oFileResults.IsOpened( ) ) return( FALSE );

  // Get the column header line
  ostk1 = m_oFileResults.GetFirstLine( );
  if( ostk1.CountTokens( ) < 3 )     return( FALSE );

  // Dispose of the first field (ie. "Index")
  ostk1.GetNextToken( );

  // Extract the operator field and replace the Ng-Spice field identifiers
  os1 << ostk1.GetNextToken( ) << wxT(' ') << m_osFieldLst;
  ostk1.SetString( os1 );

  // Format each field
// osLine = wxT("# "); // ??? 06/08/2005
  osLine = wxT('#'); // gWave breaks if there's a space after initial '#'
  while( ostk1.HasMoreTokens( ) )
  {
    if( osLine.Length( ) > 2 )
    { // Pad the column with spaces to the required width
      i1 = NGSPICE_COL_WD - (osLine.Length( ) % NGSPICE_COL_WD) + 1;
      osLine.Append( wxT(' '), i1 );
    }

    // Add the next label to the line
    osLine << ostk1.GetNextToken( );
  }

  osLine.Trim( ); // Remove trailing space characters

  m_oFileResults.GetFirstLine( ) = osLine;

  return( TRUE );
}

//*****************************************************************************
// Format a data lines in the results file.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bFmtDataLines( void )
{
  wxStringTokenizer  ostk1;
  wxString  osLine, os1;
  size_t    sz1;
  double    f1;
  int       i1;

  // This function requires the result file to have been opened
  if( ! m_oFileResults.IsOpened( ) ) return( FALSE );

  for( sz1=1; sz1<m_oFileResults.GetLineCount( ); sz1++ )
  {
    // Get a line of data
    osLine = m_oFileResults.GetLine( sz1 );

    // Tokenize the string
    ostk1.SetString( osLine );
    if( ostk1.CountTokens( ) < 3 )   return( FALSE );

    // Dispose of the first field
    ostk1.GetNextToken( );

    // Format each field
    osLine.Empty( );
    while( ostk1.HasMoreTokens( ) )
    {
      // Add the next parameter to the line
      os1 = ostk1.GetNextToken( );
      if( os1.Last( ) == wxT(',') )
      {
        os1.Last( ) = wxT(' ');
        ostk1.GetNextToken( );
      }
      if( ! ConvertType::bStrToFlt( os1, &f1 ) ) f1  =      -9.999e+99;
      if( ! ConvertType::bFltToStr( f1, os1 ) )  os1 = wxT("-9.999e+99");
      osLine << os1;

      // Pad the column with spaces to the required width
      i1 = NGSPICE_COL_WD - (osLine.Length( ) % NGSPICE_COL_WD);
      osLine.Append( wxT(' '), i1 );
    }

    m_oFileResults.GetLine( sz1 ) = osLine;

    osLine.Trim( ); // Remove trailing space characters
  }

  return( TRUE );
}

//*****************************************************************************
// Format any phase values contained in the results file (this is only
// pertinent for AC analysis). It involves removing the discontinuity at 360
// degree in the data.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bFmtPhaseData( void )
{
  return( TRUE );
}

//*****************************************************************************
// Parse the simulation commands strings found in a Simulation object and load
// the values back into the Simulation object.
//
// Arguments:
//   roSim - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bParseSim( Simulation & roSim )
{
  wxString  os1;
  size_t    sz1;

  // Set the default state of some of the simulation object attributes
  roSim.bSetOutPar( ePAR_VLT,   TRUE );
  roSim.bSetOutCpx( eCPX_MAG,   TRUE );
  roSim.bSetOutCpx( eCPX_MAGDB, TRUE );

  // Go through the simulation commands and attempt to parse them
  const wxArrayString & roasCmds=roSim.roasGetSimCmds( );
  for( sz1=0; sz1<roasCmds.GetCount( ); sz1++ )
  {
    os1 = roasCmds.Item( sz1 );

    if( bParseCmdOP ( os1, roSim ) ) continue;
    if( bParseCmdIC ( os1, roSim ) ) continue;
    if( bParseCmdDC ( os1, roSim ) ) continue;
    if( bParseCmdAC ( os1, roSim ) ) continue;
    if( bParseCmdTR ( os1, roSim ) ) continue;
    if( bParseCmdPR ( os1, roSim ) ) continue;
  }

  return( TRUE );
}

//*****************************************************************************
// Create the simulation commands from the values found in a Simulation object
// and load the command string back into the Simulation object.
//
// Arguments:
//   roSim - The simulation object
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcNgSpice::bMakeSim( Simulation & roSim )
{
  wxFileName  ofn1;
  wxString  os1;
  int       i1;

  // Clear error attributes
  m_osErrMsg.Empty( );

  // Only execute simulation if it isn't already running
  if( bIsExec( ) )
  {
    m_osErrMsg = wxT("Simulation already running");
    return( FALSE );
  }

  // Create the simulator commands
  if( ! bMakeCmds( roSim ) )
  {
    m_osErrMsg = wxT("Couldn't create simulator commands");
    return( FALSE );
  }

  // Save circuit description and SPICE commands to file
  ofn1 = roSim.rofnGetLoadFile( );
  if( ! roSim.bSaveFile( ofn1.GetFullPath( ) ) )
  {
    m_osErrMsg = wxT("Simulation couldn't be saved to file");
    return( FALSE );
  }

  // Create the results file name
  os1 = roSim.rofnGetLoadFile( ).GetPath( ) + wxT('/')
      + roSim.rofnGetLoadFile( ).GetName( );
  i1 = os1.Find( wxT(".gspiceui") );
  if( i1 > 0 ) os1 = os1.Truncate( (size_t) i1 );
  os1 << wxT('.') << rofnGetBinary( ).GetName( );
  switch( roSim.eGetAnaType( ) )
  {
    case eANA_DC : os1 << wxT(".dc"); break;
    case eANA_AC : os1 << wxT(".ac"); break;
    case eANA_DI : os1 << wxT(".di"); break;
    case eANA_NO : os1 << wxT(".no"); break;
    case eANA_PZ : os1 << wxT(".pz"); break;
    case eANA_SE : os1 << wxT(".se"); break;
    case eANA_TR : os1 << wxT(".tr"); break;
    case eANA_TF : os1 << wxT(".tf"); break;
    default :      return( FALSE );
  }
  if( ! bSetResultsFile( os1.c_str( ) ) )
  {
    m_osErrMsg = wxT("Couldn't set results file name");
    return( FALSE );
  }

  // Construct the command line to execute the simulation
  os1 = wxT("-n -b ") + roSim.rofnGetSaveFile( ).GetFullPath( );
  if( ! bSetArgLst( os1.c_str( ) ) )
  {
    m_osErrMsg = wxT("Couldn't set argument list");
    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
