//*****************************************************************************
//                                FileTasks.cpp                               *
//                               ---------------                              *
//  Started     : 28/05/2005                                                  *
//  Last Update : 04/10/2007                                                  *
//  Copyright   : (C) 2005 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "main/FileTasks.hpp"
#include "main/FrmMain.hpp"

//*****************************************************************************
// Constructor.
//
// Argument List:
//   poFrmMain - A pointer to the parent frame

FileTasks::FileTasks( FrmMain * poFrmMain )
{
  // Set the pointer to the parent frame
  m_poFrmMain = poFrmMain;

  // Set default temporary file management strategy
  bSetTmpFileMgt( eTFM_DELETE );

  // Clear object attributes
  bClear( );

  // Get the global configuration object
  m_poConfig = (wxConfig *) wxConfig::Get( );
}

//*****************************************************************************
// Destructor.

FileTasks::~FileTasks( )
{
}

//*****************************************************************************
// Display an open file dialog and set the net list file name.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bDlgOpen( void )
{
  wxFileDialog * poDlgOpen;
  bool           bRtnValue=TRUE;
  wxString       os1, os2;
  long           liStyle;

  // Can't display dialogue unless the application main frame has been created
  if( m_poFrmMain == NULL ) return( FALSE );

  // Create the different file filters
  os1 << wxT("All files (*)|*|")
      << wxT("Circuit files (*.ckt)|*.ckt|")
      << wxT("Circuit files (*.cir)|*.cir|")
      << wxT("Netlist files (*.net)|*.net");

  // Get the net list file path from the global configuration object
  m_poConfig->SetPath( wxT("/Directories") );
  os2 = m_poConfig->Read( wxT("LastAccess"), wxGetHomeDir( ) );

  // Create and configure the file open dialog
#if wxCHECK_VERSION( 2,8,0 )
  liStyle = wxFD_OPEN | wxFD_CHANGE_DIR | wxFD_FILE_MUST_EXIST;
#else
  liStyle = wxOPEN | wxCHANGE_DIR | wxFILE_MUST_EXIST;
#endif
  poDlgOpen = new wxFileDialog( m_poFrmMain, wxT(""), wxT(""), wxT(""),
                                wxT(""), liStyle );
  poDlgOpen->SetMessage( wxT("Open a Circuit Description File") );
  poDlgOpen->SetWildcard( os1 );
  poDlgOpen->SetFilterIndex( 1 );
  poDlgOpen->SetDirectory( os2 );

  // Display file open dialog
  if( poDlgOpen->ShowModal( ) != wxID_OK ) return( FALSE );

  // Set the net list file name
  if( ! bClear( ) )                               bRtnValue = FALSE;
  if( ! bSetNetLstFile( poDlgOpen->GetPath( ) ) ) bRtnValue = FALSE;

  // Set the net list file path in the global configuration object
  m_poConfig->Write( wxT("LastAccess"), poDlgOpen->GetDirectory( ) );
  m_poConfig->Flush( );

  return( bRtnValue );
}

//*****************************************************************************
// Display an import file/s dialog and set the schematic file name/s.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bDlgImport( void )
{
  wxFileDialog * poDlgImport;
  bool           bRtnValue=TRUE;
  wxArrayString  oas1;
  wxString       os1, os2;
  long           liStyle;

  // Can't display dialogue unless the application main frame has been created
  if( m_poFrmMain == NULL ) return( FALSE );

  // Create the different file filters
  os1 << wxT("gSchem files (*.sch)|*.sch|")
      << wxT("Protel II files (*.\?\?\?)|*.\?\?\?");

  // Get the net list file path from the global configuration object
  m_poConfig->SetPath( wxT("/Directories") );
  os2 = m_poConfig->Read( wxT("LastAccess"), wxGetHomeDir( ) );

  // Create and configure the file import dialog
#if wxCHECK_VERSION( 2,8,0 )
  liStyle = wxFD_OPEN | wxFD_CHANGE_DIR | wxFD_MULTIPLE | wxFD_FILE_MUST_EXIST;
#else
  liStyle = wxOPEN | wxCHANGE_DIR | wxMULTIPLE | wxFILE_MUST_EXIST;
#endif
  poDlgImport = new wxFileDialog( m_poFrmMain, wxT(""), wxT(""), wxT(""),
                                  wxT(""), liStyle );
  poDlgImport->SetMessage( wxT("Import Schematic File") );
  poDlgImport->SetWildcard( os1 );
  poDlgImport->SetFilterIndex( 0 );
  poDlgImport->SetDirectory( os2 );

  // Display file import dialog
  if( poDlgImport->ShowModal( ) != wxID_OK ) return( FALSE );

  // Set the guile procedure and schematic file name/s
  if( ! bClear( ) )              bRtnValue = FALSE;
  bSetGuileProc( wxT("spice-sdb") );
  poDlgImport->GetPaths( oas1 );
  if( ! bSetSchemFiles( oas1 ) ) bRtnValue = FALSE;

  // Set the net list file path in the global configuration object
  m_poConfig->Write( wxT("LastAccess"), poDlgImport->GetDirectory( ) );
  m_poConfig->Flush( );

  return( bRtnValue );
}

//*****************************************************************************
// Execute the schematic import process.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bExecImport( void )
{
  bool       bRtnValue;
  TextCtrl * poTxtCtl;

  // Display the console page
  m_poFrmMain->m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_CONSOLE );

  // Convert the schematic file/s to a net list file
  bRtnValue = m_oPrcGNetList.bExec( );

  // Print the results
  poTxtCtl = m_poFrmMain->m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_CONSOLE );
  m_oPrcGNetList.Print( *poTxtCtl );

  // Check for errors in the gnetlist output
  if( poTxtCtl->GetValue( ).Contains( wxT("ERROR") ) ) bRtnValue = FALSE;

  // Delete the process log file
  m_oPrcGNetList.bDelLogFile( );

  // If the schematic/s couldn't be imported clear the net list name
  if( ! bRtnValue ) m_oPrcGNetList.bSetNetList( wxT("") );

  return( bRtnValue );
}

//*****************************************************************************
// Initialize temporary file management strategy.

void  FileTasks::InitTmpFileMgt( void )
{
  long  li1;

  // Set the temporary file management strategy
  m_poConfig->SetPath( wxT("/Main") );
  li1 = m_poConfig->Read( wxT("TmpFileMgt"), (long) FileTasks::eTFM_DELETE );
  bSetTmpFileMgt( (eTmpFMgtType) li1 );
}

//*****************************************************************************
// Initialize the Guile procedure name to be used by gnetlist to import
// schematic files.

void  FileTasks::InitGuileProc( void )
{
  wxString  os1;

  // Set the Guile procedure name
  m_poConfig->SetPath( wxT("/gnetlist") );
  os1 = m_poConfig->Read( wxT("GuileProc"), wxT("spice-sdb") );
  bSetGuileProc( os1 );
}

//*****************************************************************************
// Initialize the schematic file names.

void  FileTasks::InitSchemFiles( void )
{
  wxStringTokenizer  ostk1;
  wxArrayString  oas1;
  wxString  os1;

  // Get the net list name
  m_poConfig->SetPath( wxT("/Files") );
  os1 = m_poConfig->Read( wxT("Schematics"), wxT("") );

  // Set the schematic file name/s
  if( ! os1.IsEmpty( ) )
  {
    ostk1.SetString( os1 );
    while( ostk1.HasMoreTokens( ) )
      oas1.Add( ostk1.GetNextToken( ) );

    if( bSetSchemFiles( oas1 ) )
      if( m_poFrmMain->m_oSim.IsEmpty( ) )
        if( bReload( m_poFrmMain->m_oSim ) )
          if( m_poFrmMain->bReloadSim( ) )
            m_poFrmMain->m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
  }
}

//*****************************************************************************
// Initialize the net list file name.

void  FileTasks::InitNetLstFile( void )
{
  wxString  os1;

  // Get the net list name
  m_poConfig->SetPath( wxT("/Files") );
  os1 = m_poConfig->Read( wxT("NetList"), wxT("") );

  // Load the net list file if it has been specified
  if( ! os1.IsEmpty( ) )
    if( bSetNetLstFile( os1 ) )
      if( bReload( m_poFrmMain->m_oSim ) )
        if( m_poFrmMain->bReloadSim( ) )
          m_poFrmMain->m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
}

//*****************************************************************************
// Display a error message dialog when a net list file can't be loaded.

void  FileTasks::DlgNetLstErr( void )
{
  wxString  os1;

  os1 << wxT("The net list file:\n     ")
      << m_oPrcGNetList.rofnGetNetList( ).GetFullPath( )
      << wxT("\ncouldn't be loaded.\n\n");

  os1 << wxT("This is often because the file is empty or doesn't exist.\n");

  m_poFrmMain->DlgErrMsg( wxT("Load Net List Error"), os1 );
}

//*****************************************************************************
// Display a error message dialog when schematic file/s can't be imported.

void  FileTasks::DlgSchemsErr( void )
{
  wxString  os1;
  size_t    szt1;

  os1 << wxT("The schematic file/s:");
  for( szt1=0; szt1<m_oPrcGNetList.sztGetSchemCnt( ); szt1++ )
    os1 << wxT("\n     ") << m_oPrcGNetList.rofnGetSchem( szt1 ).GetFullPath( );
  os1 << wxT("\ncouldn't be converted to a net list file.\n\n");

  os1 << wxT("gnetlist has terminated with an error. Try examining the\n")
      << wxT("console output to determine where the problem occurred.\n");

  m_poFrmMain->DlgErrMsg( wxT("Import Schematic/s Error"), os1 );
}

//*****************************************************************************
// Delete temporary files which may have been generated by this application.
// Ie. given the circuit description file "<netlist>.ckt" delete files of the
//     following form:
//       <netlist>.ngspice.DC, <netlist>.ngspice.AC, etc.
//       <netlist>.gnucap.OP, <netlist>.gnucap.DC, etc.
//       gnetlist.log

bool  FileTasks::bDelTmpFiles( void )
{
  wxArrayString  oas1;
  wxFileName  ofn1;
  wxString  os1, os2;
  size_t  szt1;
  int  i1;

  if( m_eTmpFileMgt == eTFM_KEEP ) return( TRUE );

  // Get the path to the schematic or circuit description file
  ofn1 = m_oPrcGNetList.rofnGetNetList( );
  ofn1.Normalize( );

  if( ofn1.IsOk( ) && ofn1.FileExists( ) )
  {
    // Look for the gnetlist log file
    os1 = ofn1.GetPath( ) + wxT("/gnetlist.log");
    if( wxFileExists( os1 ) ) oas1.Add( os1 );

    os1 = ofn1.GetName( );
    i1 = os1.Find( wxT(".gspiceui") );
    if( i1 > 0 ) os1 = os1.Truncate( (size_t) i1 );
    ofn1.SetFullName( os1 );

    // Look for any GNU-Cap results file
    os1 = ofn1.GetFullPath( ) + wxT(".gnucap.??");
    for( os2=wxFindFirstFile( os1 ); !os2.IsEmpty( ); os2=wxFindNextFile( ) )
      oas1.Add( os2 );

    // Look for any Ng-Spice results files
    os1 = ofn1.GetFullPath( ) + wxT(".ngspice.??");
    for( os2=wxFindFirstFile( os1 ); !os2.IsEmpty( ); os2=wxFindNextFile( ) )
      oas1.Add( os2 );

    // Prompt the user for permission to delete files
    if( oas1.GetCount( ) > 0 )
    {
      if( m_poFrmMain!=NULL && m_poFrmMain->IsShown( ) )
      {
        if( m_eTmpFileMgt == eTFM_PROMPT )
        {
          os1 = wxT("Delete Temporary Files");
          os2 = wxT("\nDelete the following temporary files:\n\n");
          for( szt1=0; szt1<oas1.GetCount( ); szt1++ )
            os2 << oas1.Item( szt1 ) << wxT("   \n");
          os2 << '\n';
          long lStyle = wxYES_NO | wxICON_QUESTION;
          i1 = wxMessageBox( os2, os1, lStyle, m_poFrmMain );
        }
        else i1 = wxYES;
        if( i1 == wxYES )
        {
          for( szt1=0; szt1<oas1.GetCount( ); szt1++ )
            wxRemoveFile( oas1.Item( szt1 ) );
        }
      }
    }
  }
  else return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bClear( void )
{
  // Clear the GNetList process object attributes
  m_oPrcGNetList.bClear( );

  return( TRUE );
}

//*****************************************************************************
// Do initialization tasks.

void  FileTasks::Initialize( void )
{
  InitTmpFileMgt( );
  InitGuileProc ( );
  InitNetLstFile( );
  InitSchemFiles( );
}

//*****************************************************************************
// Check that the gNetList binary can be found, if not display an error
// message.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bIsOkGNetList( void )
{
  wxString  os1;

  // Check that gnetlist exists and is accessible
  if( ! m_oPrcGNetList.bBinExists( ) )
  {
    if( m_poFrmMain != NULL )
    {
      os1 << wxT("\nCan't find ") << m_oPrcGNetList.rofnGetBinary( ).GetFullName( )
          << wxT(" which is required to import a schematic\nfile. ")
          << wxT("There is no path to it or it has not been installed.\n\n");
      m_poFrmMain->DlgErrMsg( wxT("Configuration Fault"), os1 );
    }
    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Set the application main frame title.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bSetTitle( void )
{
  wxString  os1;
  size_t    szt1;

  if( m_poFrmMain == NULL ) return( FALSE );

  os1 << wxT(" ") << APP_NAME << wxT(",  ") << APP_VERSION;

  const wxArrayString & roas1 = roasGetSchemFiles( );
  if( roas1.GetCount( ) > 0 )
  {
    os1 << wxT("  -  ");
    for( szt1=0; szt1<roas1.GetCount( ); szt1++ )
    {
      if( szt1 > 0 ) os1 << wxT(", ");
      os1 << roas1.Item( szt1 );
    }
  }
  else if( ! rosGetNetLstFile( ).IsEmpty( ) )
    os1 << wxT("  -  ") << rosGetNetLstFile( );

  m_poFrmMain->wxFrame::SetTitle( os1 );

  return( TRUE );
}

//*****************************************************************************
// Set a Guile procedure name to be used for importing schematic files using
// GNetList.
//
// Argument List:
//   rosProcName - The Guile procedure name
//                 (Refer to GNetList documentation for list of procedure names)

bool  FileTasks::bSetGuileProc( const wxString & rosProcName )
{
  return( m_oPrcGNetList.bSetGuileProc( rosProcName ) );
}

//*****************************************************************************
// Set the schematic file name/s.
//
// Argument List:
//   roasFileName - A string array containing the full path and file name/s
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bSetSchemFiles( const wxArrayString & roasFileNames )
{
  wxString  os1;
  size_t  szt1;

  if( ! m_oPrcGNetList.bSetSchems( roasFileNames ) ) return( FALSE );

  if( m_poFrmMain != NULL )
  {
    for( szt1=0; szt1<roasFileNames.GetCount( ); szt1++ )
      os1 << roasFileNames.Item( szt1 ) << wxT(' ');
    m_poFrmMain->m_oSim.bSetSchems( os1.Strip( ) );  // Set sim. schem. file/s
    m_oPrcGNetList.bSetNetList( );                   // Set the net list file
    os1 = m_oPrcGNetList.rofnGetNetList( ).GetFullPath( );
    m_poFrmMain->m_oSim.bSetLoadFile( os1 );         // Set sim. load file

    m_poFrmMain->InitLogFiles( );                    // Init. log file path
    bSetTitle( );                                    // Set frame title text
  }

  return( TRUE );
}

//*****************************************************************************
// Set the net list file name.
//
// Argument List:
//   psFileName - A string containing the full path and file name
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bSetNetLstFile( const wxString & rosFileName )
{
  if( ! m_oPrcGNetList.bSetNetList( rosFileName ) ) return( FALSE );

  if( m_poFrmMain != NULL )
  {
    m_poFrmMain->m_oSim.bSetSchems( wxT("") );       // Clear sim. schem. files
    m_poFrmMain->m_oSim.bSetLoadFile( rosFileName ); // Init. sim. load file

    m_poFrmMain->InitLogFiles( );                     // Init. log file path
    bSetTitle( );                                     // Set frame title text
  }

  return( TRUE );
}

//*****************************************************************************
// Set the temporary file management strategy on close or exit.
//
// Argument List:
//   eTmpFileMgt - An enumerated type indicating what to do with temp. files
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bSetTmpFileMgt( eTmpFMgtType eTFM )
{
  if( eTFM<eTFM_FST || eTFM>eTFM_LST ) return( FALSE );

  m_eTmpFileMgt = eTFM;

  return( TRUE );
}

//*****************************************************************************
// Get the currently selected Guile procedure.
//
// Return Values:
//   The currently selected Guile procedure

const wxString & FileTasks::rosGetGuileProc( void )
{
  return( m_oPrcGNetList.rosGetGuileProc( ) );
}

//*****************************************************************************
// Get an array containing the schematic file name/s.
//
// Return Values:
//   An array of schematic file names

const wxArrayString & FileTasks::roasGetSchemFiles( void )
{
  static  wxArrayString  oas1;
  wxString  os1;
  size_t    szt1;

  oas1.Clear( );

  for( szt1=0; szt1<m_oPrcGNetList.sztGetSchemCnt( ); szt1++ )
  {
    os1 = m_oPrcGNetList.rofnGetSchem( szt1 ).GetFullPath( );
    if( ! os1.IsEmpty( ) ) oas1.Add( os1 );
  }

  return( oas1 );
}

//*****************************************************************************
// Get the net list file name.
//
// Return Values:
//   The net list file name

const wxString & FileTasks::rosGetNetLstFile( void )
{
  static  wxString  osNetLstFile;

  osNetLstFile = m_oPrcGNetList.rofnGetNetList( ).GetFullPath( );

  return( osNetLstFile );
}

//*****************************************************************************
// Open and load a circuit description (net list) file into a Simulation
// object.
//
// Argument List:
//   roSim - The Simulation object to be loaded
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bOpen( Simulation & roSim )
{
  wxString  os1;

  // Get the net list file name from the user
  if( ! bDlgOpen( ) ) return( FALSE );

  // Change the cursor to the wait symbol
  ::wxBeginBusyCursor( );

  // Clear the main frame object attributes
  m_poFrmMain->bClear( );

  // Display the net list page
  m_poFrmMain->m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );

  // Attempt to load the circuit description file
  if( ! roSim.bLoadFile( ) ) DlgNetLstErr( );

  // Set the schematic file name/s if supplied
  if( ! roSim.rosGetSchems( ).IsEmpty( ) )
  {
    m_oPrcGNetList.bSetSchems( roSim.rosGetSchems( ) );
    bSetTitle( );
  }

  // Change the cursor to the default
  ::wxEndBusyCursor( );

  return( os1.IsEmpty( ) );
}

//*****************************************************************************
// Import a schematic file by converting it to a net list using GNetList and
// then loading it into a Simulation object.
//
// Argument List:
//   roSim - The Simulation object to be loaded
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bImport( Simulation & roSim )
{
  bool  bRtnValue=TRUE;

  // Check that gNetList binary exists and is accessible
  if( ! bIsOkGNetList( ) ) return( FALSE );

  // Get the schematic file name/s from the user
  if( ! bDlgImport( ) )    return( FALSE );

  // Change the cursor to the wait symbol
  ::wxBeginBusyCursor( );

  // Clear the main frame object attributes
  m_poFrmMain->bClear( );

  // Convert the schematic file/s to a net list file and load it
  if( ! bExecImport( ) )
  {
    bRtnValue = FALSE;
    DlgSchemsErr( );
  }
  else if( ! roSim.bLoadFile( ) )
  {
    bRtnValue = FALSE;
    DlgNetLstErr( );
  }

  // Change the cursor to the default
  ::wxEndBusyCursor( );

  return( bRtnValue );
}

//*****************************************************************************
// Reload a Simulation object whether it be from a schematic or net list file.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bReload( Simulation & roSim )
{
  bool  bRtnValue=TRUE;

  // Change the cursor to the wait symbol
  ::wxBeginBusyCursor( );

  // Reset the text control notebook
  m_poFrmMain->m_oNbkTxtCtls.Initialize( );

  // If schematic file/s have been specified perform an import operation
  if( m_oPrcGNetList.sztGetSchemCnt( ) > 0 )
  {
    if( ! bExecImport( ) )
    {
      bRtnValue = FALSE;
      DlgSchemsErr( );
    }
  }

  // Attempt to load the net list file
  if( bRtnValue )
  {
    if( ! m_oPrcGNetList.rofnGetNetList( ).GetFullPath( ).IsEmpty( ) )
    {
      if( roSim.bLoadFile( ) )
      { // Set the schematic file name/s if supplied
        if( ! roSim.rosGetSchems( ).IsEmpty( ) )
        {
          m_oPrcGNetList.bSetSchems( roSim.rosGetSchems( ) );
          bSetTitle( );
        }
      }
      else
      {
        bRtnValue = FALSE;
        DlgNetLstErr( );
      }
    }
    else
    {
      bRtnValue = FALSE;
      m_poFrmMain->DlgErrMsg( wxT("Reload Operation Error"),
                              wxT("No file is currently open.") );
    }
  }

  // Change the cursor to the default
  ::wxEndBusyCursor( );

  return( bRtnValue );
}

//*****************************************************************************
// Close the circuit description file.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bClose( void )
{
  bool  bRtnValue=TRUE;

  if( ! bDelTmpFiles( ) ) bRtnValue = FALSE; // Delete temporary files
  if( ! bClear( ) )       bRtnValue = FALSE; // Clear the object attributes

  return( bRtnValue );
}

//*****************************************************************************
// Do the necessary tasks before the application exits.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FileTasks::bExit( void )
{
  wxString  os1;
  bool      bRtnValue=TRUE;

  // Record the Guile procedure name
  m_poConfig->SetPath( wxT("/gnetlist") );
  os1 = rosGetGuileProc( );
  m_poConfig->Write( wxT("GuileProc"), os1 );

  // Record the net list file name
  m_poConfig->SetPath( wxT("/Files") );
  os1 = rosGetNetLstFile( );
  m_poConfig->Write( wxT("NetList"), os1 );

  // Record the schematic file name/s
  os1.Empty( );
  const wxArrayString & roas1 = roasGetSchemFiles( );
  for( size_t szt1=0; szt1<roas1.GetCount( ); szt1++ )
    os1 << roas1.Item( szt1 ) << wxT(' ');
  m_poConfig->Write( wxT("Schematics"), os1 );

  // Write any changes to file
  m_poConfig->Flush( );

  // Delete temporary file
  if( ! bDelTmpFiles( ) ) bRtnValue = FALSE;

  return( bRtnValue );
}

//*****************************************************************************
