/* GSAMBAD, an easy to use GTK+ frontend for the SAMBA file and print server.
 * Copyright (C) 2006 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307, USA.
 *
*/


#include "../config.h"
#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "support.h"
#include "allocate.h"
#include "widgets.h"
#include "functions.h"
#include "show_info.h"
#include "import_functions.h"
#include "get_option_pos.h"
#include "commented.h"

#include "populate_users.h"
#include "select_first_user.h"
#include "populate_user_settings.h"
#include "populate_shares.h"
#include "populate_share_settings.h"
#include "populate_conf_tab.h"


extern int activated;

extern char global_share_name[16384];

extern int global_import_local;
extern int global_import_remote;

extern int global_import_users;
extern int global_import_groups;

extern int global_import_valid;
extern int global_import_write;
extern int global_import_admin;


long num_imported = 0;
int  new_smbuser_added = 0;


/* Locates the share and checks if the name is in its valid/admin or writelist. */
int in_perm_list(gchar *name, gchar *sharename, char where[128])
{
    FILE *fp;
    long conf_size, opt_pos;
    int i, retval = 0;
    int share_found = 0;
    char *temp, *line, *user_line;
    gchar *info, *share_def;    

    if((fp=fopen(SAMBA_CONF, "r"))==NULL)
    {
        info = g_strdup_printf(_("Cant open smb.conf here: %s run gsambad as root.\n"), SAMBA_CONF);
        show_info(info);
        g_free(info);
        return retval;
    }
    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);
    rewind(fp);

    line = allocate(conf_size+1);
    user_line = allocate(conf_size+1);

    share_def = g_strdup_printf("[%s]", sharename);

    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
        if( commented(line) )
          continue;

	/* The share definition is found, break */
	if( strstr(line, share_def) )
	{
	    share_found = 1;
	    break;
	}
    }

    /* Return not found if the share isnt found */
    if( ! share_found )
    {
	free(line);
	free(user_line);
	g_free(share_def);
        return retval;
    }
    
    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
        if( commented(line) )
          continue;

	/* If a new share definition is found, break */
	if( strstr(line, "[") && strstr(line, "]") )
	  break;
	  	

        /* Lines where this user could be listed */
        if( (strstr(line, "admin users =") && strstr(where, "admin")) 
        ||  (strstr(line, "valid users =") && strstr(where, "valid"))
        ||  (strstr(line, "write list =" ) && strstr(where, "write")) )
        {
            /* Only handle the users and groups */
            opt_pos = get_option_pos(line);

            snprintf(user_line, conf_size, "%s", &line[opt_pos]);

            for(i=strlen(user_line)-1; user_line[i]!='\0'; i--)
               if( user_line[i]!='\r' || user_line[i]!='\n' )
               {
                  user_line[i]='\0';
                  break;
               }

            /* There must be an extra whitespace at the end for strtok */
            strcat(user_line, " ");

            temp = strtok(user_line, " ");
            while( temp != NULL )
            {
                if( ! strcmp(temp, name) )
                {
                    /* The user is deleted */
                    retval = 1;
		    break;
                }
                /* We pass NULL to strtok to get the next token in the string */
                temp = strtok(NULL, " ");
            }
        }
    }
    fclose(fp);
    free(line);
    free(user_line);
    g_free(share_def);


    return retval;
}


/* Adds a name to a share, "where" the permission is valid users, write list or admin users */
void add_name_to_share(gchar *name, gchar *sharename, char where[128])
{
    FILE *fp;
    long conf_size;
    int i;
    int share_found = 0;
    char *line, *user_line;
    char *new_conf, *new_user_line;
    gchar *info, *share_def;    


    if((fp=fopen(SAMBA_CONF, "r"))==NULL)
    {
        info = g_strdup_printf(_("Cant open smb.conf here: %s run gsambad as root.\n"), SAMBA_CONF);
        show_info(info);
        g_free(info);
        return;
    }
    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);
    rewind(fp);

    line = allocate(conf_size+1);
    new_conf = allocate(conf_size+1+4096);
    user_line = allocate(conf_size+1);
    new_user_line = allocate(conf_size+1+4096); /* Max 4096/4000 for an imported user/group name */

    share_def = g_strdup_printf("[%s]", sharename);

    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
        if( commented(line) )
          continue;

	/* Collect the old lines */
	strcat(new_conf, line);

	/* The share definition is found, break */
	if( strstr(line, share_def) )
	{
	    share_found = 1;
	    break;
	}
    }

    /* Return if the share isnt found */
    if( ! share_found )
    {
	free(line);
	free(new_conf);
	free(user_line);
	g_free(share_def);
        return;
    }
    
    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
    {
        if( commented(line) )
          continue;

	/* If a new share definition is found, break */
	if( strstr(line, "[") && strstr(line, "]") )
	{
	    /* Collect the old line */
	    strcat(new_conf, line);
	    break;
	}

        /* Lines where this user should be listed */
        if( (strstr(line, "admin users =") && strstr(where, "admin")) 
        ||  (strstr(line, "valid users =") && strstr(where, "valid"))
        ||  (strstr(line, "write list =" ) && strstr(where, "write")) )
        {
	    /* Count how many bytes the line contains */
	    for(i=0; line[i]!='\0' && i <= conf_size; i++)
	      if( line[i]=='\n' || line[i]=='\r' )
	        break;
		
	    /* Current names are conf_size+1 + 4000 for a single new name */
	    snprintf(new_user_line, i+1, "%s", line);
	
	    strcat(new_user_line, " ");
	    strcat(new_user_line, name);
	    strcat(new_user_line, "\n");

	    /* The name is added to the list of names */
	    strcat(new_conf, new_user_line);
        }
	else
	  strcat(new_conf, line);
    }

    /* Get any remaining conf lines */
    if( conf_size > 1 )
    while(fgets(line, conf_size, fp)!=NULL)
	strcat(new_conf, line);

    fclose(fp);
    free(line);
    free(user_line);
    free(new_user_line);
    g_free(share_def);


    /* Write the new conf */
    if((fp=fopen(SAMBA_CONF, "w+"))==NULL)
    {
        info = g_strdup_printf(_("Cant write smb.conf here: %s run gsambad as root.\n"), SAMBA_CONF);
        show_info(info);
        g_free(info);
	free(new_conf);
        return;
    }
    fputs(new_conf, fp);
    fclose(fp);
    free(new_conf);
}


/* The contents of the selections in the list is iterated and imported */
void do_import(GtkTreeModel *model, GtkTreePath *path,
	          GtkTreeIter *iter, struct w *widgets)
{
    FILE *fp;
    int imported = 0;
    gchar *cmd, *name, *mod_name, *sharename;


    /* Where in the list should we import from.. */
    if( global_import_users )
      gtk_tree_model_get(model, iter, 0, &name, -1);
    else
    if( global_import_groups )
      gtk_tree_model_get(model, iter, 1, &name, -1);
    

    /* What are we importing, do we need to add chars infromt of the name.. */    
    if( global_import_users && global_import_local )
      mod_name = g_strdup_printf("%s", name);
    else
    if( global_import_groups && global_import_local )
      mod_name = g_strdup_printf("@%s", name);
    else
      {
// FIXME: global_import_remote
	 mod_name = g_strdup_printf("Avoid compile warning\n");
	 // This is never executed.
	 g_free(name);
	 g_free(mod_name);
         return;
      }    


    /* Doesnt seem to be required: local_username = g_locale_from_utf8(name, -1, NULL, NULL, NULL); */


    /* If the name is empty and its a local user import. */
    if( strlen(mod_name) == 0 && global_import_users && global_import_local )
    {
	printf("The import name was empty\n");
	g_free(name);
	g_free(mod_name);
    	return;
    }

    /* If the name only contains an '@' or is empty and its a local group import. */
    if( strlen(mod_name) < 2 && global_import_groups && global_import_local )
    {
	printf("The import name was empty\n");
	g_free(name);
	g_free(mod_name);
    	return;
    }

// Fixme: Add remote defs..




// Importing root isnt rickdangerous.
//    if( name[0]=='r' && name[1]=='o' && name[2]=='o' && name[3]=='t' && strlen(name)==4 ) 
//    {
//	printf("Refusing to import user or group root\n");
//    	return;
//    }

    /* Machine accounts arent allowed */
    if( strstr((char *)name, "$") ) 
    {
    	printf("An import name containing $ is not allowed as it is a machine account.\n");
	g_free(name);
	g_free(mod_name);
    	return;
    }


    sharename = g_strdup_printf("%s", global_share_name);


    /* If the name is not listed in valid users and thats requested then add it */
    if( ! in_perm_list(mod_name, sharename, "valid") && global_import_valid )
    {
        add_name_to_share(mod_name, sharename, "valid");
	imported = 1;
    }

    /* If the name is not listed in the write list and thats requested then add it */
    if( ! in_perm_list(mod_name, sharename, "write") && global_import_write )
    {
        add_name_to_share(mod_name, sharename, "write");
	imported = 1;
    }

    /* If the name is not listed in admin users, and thats requested then add it */
    if( ! in_perm_list(mod_name, sharename, "admin") && global_import_admin )
    {
        add_name_to_share(mod_name, sharename, "admin");
	imported = 1;
    }

    if( imported )
	num_imported++;


    g_free(sharename);


    /* If its user imports and this smb user doesnt exist, add it with an empty password */
    if( ! smbuser_exists(name) && global_import_local && global_import_users ) 
    {
	new_smbuser_added = 1;

	cmd = g_strdup_printf("%s -a -n '%s'", SMBPASSWD_BINARY, name);
        if((fp=popen(cmd, "w"))==NULL)
          perror("popen");
	else
          pclose(fp);
        
        g_free(cmd);


	/* Also create its /home/profiles/username directory */
	cmd = g_strdup_printf("%s %s/%s", "mkdir -p", "/home/profiles/", name);
        if((fp=popen(cmd, "w"))==NULL)
          perror("popen");
	else
          pclose(fp);
        
        g_free(cmd);

	/* Chown this directory to user:user */
	cmd = g_strdup_printf("%s %s:%s %s/%s", "chown", name, name, "/home/profiles", name);
        if((fp=popen(cmd, "w"))==NULL)
          perror("popen");
	else
          pclose(fp);
        
        g_free(cmd);
    } 


    g_free(name);
    g_free(mod_name);
}



/* Call foreach on the group or user names */
void import_button_clicked(GtkButton *button, struct w *widgets)
{
    GtkTreeSelection *selection;
    gchar *info=NULL;

    if( global_import_remote )
    {
        info = g_strdup_printf(_("Remote operations are not yet supported.\n"));
	show_info(info);
	g_free(info);
	return;
    }

		
    /* Global counter */
    num_imported = 0;
    new_smbuser_added = 0;
			
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(widgets->import_treeview));
     
    gtk_tree_selection_selected_foreach(GTK_TREE_SELECTION(selection),
                     (GtkTreeSelectionForeachFunc) &do_import, widgets);

    gtk_widget_destroy(widgets->import_question_window);
    gtk_widget_destroy(widgets->import_window);

    if( new_smbuser_added )
    {
        info = g_strdup_printf(_("New samba users where added with empty passwords.\nSee server settings and empty passwords.\n"));
	show_info(info);
	g_free(info);
    }


    if( global_import_users )
      info = g_strdup_printf(_("Number of imported users: %ld\n"), num_imported);
    else
      info = g_strdup_printf(_("Number of imported groups: %ld\n"), num_imported);

    show_info(info);
    g_free(info);



    /* If anything was imported, update relevant parts of the gui */
    if( num_imported )
    {
        /* Populate the user tab */
        populate_users(widgets);
        /* Set the first user as selected */
        select_first_user(widgets);

        /* Populate the user settings */
        populate_user_settings(widgets);

        /* Populate the share tab */
        populate_shares(widgets);

        /* Populate the share settings */
        populate_share_settings(widgets);

	populate_conf_tab(widgets);
    }


    /* We need to reset global_* options so that it matches the default settings */
    global_import_local = 1;
    global_import_remote = 0;
    global_import_users = 1;
    global_import_groups = 0;
}
