/*-----------------------------------------------------------------+
 |                                                                 |
 |  Copyright (C) 2002-2003 Grubconf                               |
 |                     http://grubconf.sourceforge.net/            | 
 |                                                                 |
 | This program is free software; you can redistribute it and/or   |
 | modify it under the terms of the GNU General Public License     |
 | as published by the Free Software Foundation; either version 2  |
 | of the License, or (at your option) any later version           |
 |                                                                 |
 | This program is distributed in the hope that it will be useful, |
 | but WITHOUT ANY WARRANTY; without even the implied warranty of  |
 | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   |
 | GNU General Public License for more details.                    |
 |                                                                 |
 | A copy of the GNU General Public License may be found in the    |
 | installation directory named "COPYING"                          |
 |                                                                 |
 +-----------------------------------------------------------------+
 */
/* 
 * This helps out grubconf_dev.c with finding partition information
 *
 * Much of this code is an adaptation from sfdisk,
 * Copyright (C) 1995  Andries E. Brouwer (aeb@cwi.nl) GNU GPL
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/unistd.h>
#include <linux/hdreg.h>
#include <errno.h>

#include <include/partitioning.h>

#define SIZE(a)	(sizeof(a)/sizeof(a[0]))

static int
is_extended(unsigned char type) {
	return (type == EXTENDED_PARTITION
		|| type == LINUX_EXTENDED
		|| type == WIN98_EXTENDED);
}

/* Unfortunately, partitions are not aligned, and non-Intel machines
   are unhappy with non-aligned integers. So, we need a copy by hand. */
static int
copy_to_int(unsigned char *cp) {
    unsigned int m;

    m = *cp++;
    m += (*cp++ << 8);
    m += (*cp++ << 16);
    m += (*cp++ << 24);
    return m;
}

static void
copy_to_part(char *cp, struct partition *p) {
    p->bootable = *cp++;
    p->begin_chs.h = *cp++;
    p->begin_chs.s = *cp++;
    p->begin_chs.c = *cp++;
    p->sys_type = *cp++;
    p->end_chs.h = *cp++;
    p->end_chs.s = *cp++;
    p->end_chs.c = *cp++;
    p->start_sect = copy_to_int(cp);
    p->nr_sects = copy_to_int(cp+4);
}

#if !defined (__alpha__) && !defined (__ia64__) && !defined (__x86_64__) && !defined (__s390x__)
static
_syscall5(int,  _llseek,  uint,  fd, ulong, hi, ulong, lo,
       loff_t *, res, uint, wh);
#endif

static int
sseek(char *dev, unsigned int fd, unsigned long s) {
    loff_t in, out;
    in = ((loff_t) s << 9);
    out = 1;

#if !defined (__alpha__) && !defined (__ia64__) && !defined (__x86_64__) && !defined (__s390x__)
    if (_llseek (fd, in>>32, in & 0xffffffff, &out, SEEK_SET) != 0) {
#else
    if ((out = lseek(fd, in, SEEK_SET)) != in) {
#endif
	perror("llseek");
	fprintf (stderr, "seek error on %s - cannot seek to %lu\n", dev, s);
	return 0;
    }

    if (in != out) {
	fprintf (stderr, "seek error: wanted 0x%08x%08x, got 0x%08x%08x\n",
	       (uint)(in>>32), (uint)(in & 0xffffffff),
	       (uint)(out>>32), (uint)(out & 0xffffffff));
	return 0;
    }
    return 1;
}

/*
 * We preserve all sectors read in a chain - some of these will
 * have to be modified and written back.
 */
struct sector {
    struct sector *next;
    unsigned long sectornumber;
    int to_be_written;
    char data[512];
} *sectorhead;

static void
free_sectors(void) {
    struct sector *s;

    while (sectorhead) {
		s = sectorhead;
		sectorhead = s->next;
		free(s);
    }
}

static struct sector *
get_sector(char *dev, int fd, unsigned long sno) {
    struct sector *s;

    for(s = sectorhead; s; s = s->next)
	if(s->sectornumber == sno)
	    return s;

    if (!sseek(dev, fd, sno))
		return 0;

    if (!(s = (struct sector *) malloc(sizeof(struct sector)))) {
		fprintf (stderr, "grubconf: out of memory - giving up\n");
		exit(0);
	}

    if (read(fd, s->data, sizeof(s->data)) != sizeof(s->data)) {
		if (errno)		/* 0 in case we read past end-of-disk */
			perror("read");
		fprintf (stderr, "read error on %s - cannot read sector %lu\n", dev, sno);
		free(s);
		return 0;
    }

    s->next = sectorhead;
    sectorhead = s;
    s->sectornumber = sno;
    s->to_be_written = 0;

    return s;
}

static int
msdos_signature (struct sector *s) {
    unsigned char *data = s->data;
    if (data[510] == 0x55 && data[511] == 0xaa)
	    return 1;
    fprintf (stderr, "ERROR: sector %lu does not have an msdos signature\n",
	  s->sectornumber);
    return 0;
}

/*
 * We consider several geometries for a disk:
 * B - the BIOS geometry, gotten from the kernel via HDIO_GETGEO
 * F - the fdisk geometry
 * U - the user-specified geometry
 *
 * 0 means unspecified / unknown
 */
struct geometry {
	unsigned long cylindersize;
	unsigned long heads, sectors, cylinders;
	unsigned long start;
} B, F, U;

#define BLKGETSIZE _IO(0x12,96)    /* return device size */

static int
get_cylindersize(char *dev, int fd) {
    struct hd_geometry g;
    long size;
    struct geometry R;

    if (ioctl(fd, BLKGETSIZE, &size)) {
		return 0;
    }
    if (ioctl(fd, HDIO_GETGEO, &g)) {
		return 0;
    }
    R.heads = g.heads;
    R.sectors = g.sectors;
    R.cylindersize = R.heads * R.sectors;
    R.cylinders = (R.cylindersize ? size / R.cylindersize : 0);
    R.start = g.start;

    B.heads = (U.heads ? U.heads : R.heads);
    B.sectors = (U.sectors ? U.sectors : R.sectors);
    B.cylinders = (U.cylinders ? U.cylinders : R.cylinders);

    B.cylindersize = B.heads * B.sectors;

    if (R.start) {
		fprintf (stderr, "Warning: start=%lu - this looks like a partition rather than\n"
			  "the entire disk. Using fdisk on it is probably meaningless.\n"
			  "[Use the --force option if you really want this]\n%s\n", R.start, dev);
		return 0;
    }

    if (R.heads && B.heads != R.heads)
		fprintf (stderr, "Warning: HDIO_GETGEO says that there are %lu heads\n",
	     R.heads);
    if (R.sectors && B.sectors != R.sectors)
		fprintf (stderr, "Warning: HDIO_GETGEO says that there are %lu sectors\n",
	     R.sectors);
    if (R.cylinders && B.cylinders != R.cylinders
		    && B.cylinders < 65536 && R.cylinders < 65536)
		fprintf (stderr, "Warning: BLKGETSIZE/HDIO_GETGEO says that there are %lu cylinders\n",
	     R.cylinders);

    if (B.sectors > 63)
      fprintf (stderr, "Warning: unlikely number of sectors (%lu) - usually at most 63\n"
	   "This will give problems with all software that uses C/H/S addressing.\n",
	   B.sectors);
	
	return 1;
}

static void
extended_partition(char *dev, int fd, struct part_desc *ep, struct disk_desc *z) {
    char *cp;
    struct sector *s;
    unsigned long start, here, next;
    int i, moretodo = 1;
    struct partition p;
    struct part_desc *partitions = &(z->partitions[0]);
    int pno = z->partno;

    here = start = ep->start;

    if (B.cylindersize && start % B.cylindersize) {
	/* This is BAD */
	    printf("Warning: extended partition does not start at a "
		     "cylinder boundary.\n"
		     "DOS and Linux will interpret the contents differently.\n");
    }

    while (moretodo) {
		moretodo = 0;
	
		if (!(s = get_sector(dev, fd, here)))
			break;
	
		if (!msdos_signature(s))
			break;
	
		cp = s->data + 0x1be;
	
		if (pno+4 >= SIZE(z->partitions)) {
			printf("too many partitions - ignoring those past nr (%d)\n",
			   pno-1);
			break;
		}
	
		next = 0;
	
		for (i=0; i<4; i++,cp += sizeof(struct partition)) {
			partitions[pno].sector = here;
			partitions[pno].offset = cp - s->data;
			partitions[pno].ep = ep;
			copy_to_part(cp,&p);
			if (is_extended(p.sys_type)) {
				partitions[pno].start = start + p.start_sect;
				if (next)
					printf("tree of partitions?\n");
				else
				  next = partitions[pno].start;		/* follow `upper' branch */
				moretodo = 1;
			} else {
				partitions[pno].start = here + p.start_sect;
			}
			partitions[pno].size = p.nr_sects;
			partitions[pno].ptype = DOS_TYPE;
			partitions[pno].p = p;
			//printf ("etype: %.2X\n", p.sys_type);
			pno++;
		}
		here = next;
    }

    z->partno = pno;
}


/* retrieves drive information */
struct disk_desc*
get_drive_info(char *drive_str, int drv_count)
{
	unsigned long start = 0;
    struct disk_desc *z = calloc (sizeof (struct disk_desc), 1);
	char *cp;	
    struct partition pt;
    struct sector *s;
    struct part_desc *partitions = &(z->partitions[0]);
    int pno = z->partno;
	int fd = open (drive_str, O_RDONLY);
	int i;

	if (fd < 0) {
		free (z);
		return NULL;
	}
		
	if (!get_cylindersize(drive_str, fd)) {
		free (z);
		close (fd);
		return NULL;
	}

    free_sectors();

    if (!(s = get_sector(drive_str, fd, start))) {
		free (z);
		close (fd);
		return NULL;
	}

    cp = s->data + 0x1be;
    copy_to_part(cp,&pt);

    for (pno = 0; pno < 4; pno++, cp += sizeof(struct partition)) {
		partitions[pno].sector = start;
		partitions[pno].offset = cp - s->data;
		copy_to_part(cp,&pt);
		partitions[pno].start = start + pt.start_sect;
		partitions[pno].size = pt.nr_sects;
		partitions[pno].ep = 0;
		partitions[pno].p = pt;
    }

    z->partno = pno;

    for (i=0; i<4; i++) {
		if (is_extended(partitions[i].p.sys_type)) {
			if (!partitions[i].size) {
				printf("strange..., an extended partition of size 0?\n");
				continue;
			}
			extended_partition(drive_str, fd, &partitions[i], z);
		}
	}
	
	close(fd);
	return z;
}

/* returns the filesystem type ID for part starting at 1 from part_table
 */
unsigned char
valid_type (unsigned char p_type)
{
	/* check for valid type */
	switch (p_type)
	{
	case 0x01:
	case 0x06:
	case 0x07:
	case 0x0B:
	case 0x0C:
	case 0x0E:
	case 0x83:
		return p_type;
	default:
		return 0x00;
	}
	return 0x00;
}

/* Returns the human readable fileystem type, or null if cant
 * be used by GRUB (or not implemented by grubconf)
 */
char *
type_to_name (unsigned char type)
{
	switch (type)
	{
	case 0x01:
		return "DOS";
	case 0x07:
	case 0x0B:
	case 0x0C:
	case 0x0E:
		return "Windows";
	case 0x83:
		return "Linux";
	case 0x05:
	case 0x0F:
	case 0x85:
		return "Extended";
	default:
		return NULL;
	}
	return NULL;
}
