/*-----------------------------------------------------------------+
 |                                                                 |
 |  Copyright (C) 2002-2003 Grubconf                               |
 |                     http://grubconf.sourceforge.net/            | 
 |                                                                 |
 | This program is free software; you can redistribute it and/or   |
 | modify it under the terms of the GNU General Public License     |
 | as published by the Free Software Foundation; either version 2  |
 | of the License, or (at your option) any later version           |
 |                                                                 |
 | This program is distributed in the hope that it will be useful, |
 | but WITHOUT ANY WARRANTY; without even the implied warranty of  |
 | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   |
 | GNU General Public License for more details.                    |
 |                                                                 |
 | A copy of the GNU General Public License may be found in the    |
 | installation directory named "COPYING"                          |
 |                                                                 |
 +-----------------------------------------------------------------+
 */
/* 
 * This is the heart of grubconf. Performs initializiation and cleanup
 * as well as deals with parsing the grub.conf file
 */

#include <string.h>
#include <fstab.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <fstab.h>

// for _(String)
#include <gnome.h>

#include <include/grubconf_global.h>

// used to convert a '0' to 0 keeping it in bounds of os_type
#define OS_NUM_TYPE(a) (enum os_type) ((a - 0x30) < 0 || (a - 0x30) > 2 ? 2 : a - 0x30)

// makes grubconf load a test configuration file instead
//  of the system grub.conf
#define GC_TESTING 0 /* 1 for testing, 0 for normal */

extern int fcloseall (void);

/* Perform all data initialization. This is called before
 * gtk is initialied. 
 */
void
init_global ()
{
	init_grubconf_dev ();
	grubconf_fname_bak = NULL;
	find_menu_lst();
}

/* Called before the program exits. Cleanup anything that needs
 * it here
 */
void
clean_global ()
{
	os_list_clean ();
	dev_list_clean ();
	fcloseall ();
	//filename strings need freed especially when user defines them as they get malloc'd
	free(grubconf_fname); 
}

/* Finds the grub configuration file and returns an open
 * file pointer
 */
void
find_menu_lst ()
{
//	FILE *fp;

	if (!grubconf_fname)
		grubconf_fname = "/boot/grub/menu.lst";
	
	if (strncmp (grubconf_fname, "/boot", 5) == 0) {
		/* attemp to mount /boot */
		mount ("/boot");
	}

	/* now that we did our best to make sure /boot was mounted
	 * we see if fname exists
	 */
	if (access (grubconf_fname, W_OK) == 0) {
		grubconf_fname_bak = malloc (strlen (grubconf_fname) + 2);
		sprintf (grubconf_fname_bak, "%s~", grubconf_fname);
	} else if (strcmp (grubconf_fname, "/boot/grub/menu.lst") == 0) {
		/* last ditch effort search for a config file */
		printf ("grubconf: Cant find %s, trying /etc/grub.conf\n", grubconf_fname);
		if (access ("/etc/grub.conf", W_OK) == 0) {
			grubconf_fname = "/etc/grub.conf";
			grubconf_fname_bak = malloc (strlen (grubconf_fname) + 2);
			sprintf (grubconf_fname_bak, "%s~", grubconf_fname);
		} else {
			printf ("grubconf: Cant find /etc/grub.conf\n");
			grubconf_fname = NULL;
		}
	} else {
		grubconf_fname = NULL;
	}
}

/* Loads the grub configuration file
 *  Please excuse this funciton. It is huge and uncommented.
 */
int
load_from_file ()
{
	GtkWidget *timeout_spin = lookup_widget (winMain, "timeout_spin");
	GtkWidget *chkGenHidden = lookup_widget (winMain, "chkGenHidden");
	GtkWidget *tvGenExtra = lookup_widget (diaMore, "tvGenExtra");
	GtkTextBuffer *tbGenExtra =
		gtk_text_view_get_buffer (GTK_TEXT_VIEW (tvGenExtra));

#if USE_SPLASHIMAGE

	GtkWidget *optGenSplashDev = lookup_widget (winMain, "optGenSplashDev");
	GtkWidget *txtGenSplash = lookup_widget (winMain, "txtGenSplash");
	
#else
	
	GtkWidget *optNFor = lookup_widget (winMain, "optNFor");
	GtkWidget *optNBak = lookup_widget (winMain, "optNBak");
	GtkWidget *optHFor = lookup_widget (winMain, "optHFor");
	GtkWidget *optHBak = lookup_widget (winMain, "optHBak");
	enum color_enum nFor, nBak, hFor, hBak;

#endif
	
	char *line = NULL;
	char *tmp_str = NULL, *tmp_str_mid = NULL;
	int tmp_int = 0;
	int def = 0;
	char *unknown = NULL;
	int i = 0;
	int line_num = -1;
	enum os_type type = OTHER;
	int os_index;
	gboolean in_title = FALSE;
	char *title = NULL;
	int root_num = -1;
	int kern_num = -1;
	char *kernel = NULL;
	char *kern_parms = NULL;

	if (!grubconf_fname) {
		gtk_message (GTK_MESSAGE_ERROR, _("Configuration file not available\n\nFatal error, quitting"));
		return 0;
	}
		
	FILE *grubconf_fp = fopen (grubconf_fname, "r");
	if (!grubconf_fp) {
		gtk_message (GTK_MESSAGE_ERROR, _("Configuration file not available\n\nFatal error, quitting"));
		return 0;
	}
	line_num = 0;
	while (!feof (grubconf_fp)) {
		line_num++;
		i = 0;
		if (line) {
			free (line);
			line = NULL;
		}
		if (my_getline (&line, grubconf_fp) < 0) {
			break;
		}
		if (!line || feof (grubconf_fp)) {
			break;
		}
		if (*line == '#' || *line == '\n' || *line == '\0') {
			free (line);
			line = NULL;
			continue;
		}
		switch (get_command (&line)) {
		case UNKNOWN:
			if (unknown) {
				unknown = realloc (unknown, 
						strlen (unknown) + strlen (line) + 2);
				strcat (unknown, "\n");
				strcat (unknown, line);
			} else {
				unknown = malloc (strlen (line) + 1);
				strcpy (unknown, line);
			}
			break;
		case DEFAULT:
			tmp_str = get_value ((line) + 7);
			if (tmp_str) {
				def = atoi (tmp_str);
				free (tmp_str);
			} else {
				gtk_message (GTK_MESSAGE_ERROR, _("[%s line %i]\nInvalid default operating system"), grubconf_fname, line_num);
				return 0;
			}
			break;
		case TIMEOUT:
			tmp_str = get_value ((line) + 7);
			if (tmp_str) {
				gtk_spin_button_set_value (GTK_SPIN_BUTTON (timeout_spin), 
							atoi (tmp_str));
				free (tmp_str);
			} else {
				gtk_message (GTK_MESSAGE_ERROR, _("[%s line %i]\nInvalid Timeout value"), grubconf_fname, line_num);
				return 0;
			}
			break;
		case SPLASHIMAGE:

#if USE_SPLASHIMAGE

			tmp_str = get_value ((line) + 11);
			if (!tmp_str || strlen (tmp_str) < 8) {
				if (tmp_str) {
					free (tmp_str);
				}
				gtk_message (GTK_MESSAGE_ERROR, _("[%s line %i]\nInvalid splash image"), grubconf_fname, line_num);
				return 0;
			}
			tmp_int = get_index_grub (tmp_str);
			if (tmp_int == -1) {
				gtk_message (GTK_MESSAGE_ERROR, _("[%s line %i]\nCould not find splash image device, given %s"), grubconf_fname, line_num, tmp_str);
				return 0;
			}
			gtk_option_menu_set_history (GTK_OPTION_MENU
						     (optGenSplashDev),
						     tmp_int);
			gtk_entry_set_text (GTK_ENTRY (txtGenSplash), tmp_str + 7);
			gtk_toggle_button_set_active (
			       GTK_TOGGLE_BUTTON (lookup_widget (winMain, "chkSplash")),
			       TRUE);
			free (tmp_str);

#else

			if (unknown) {
				unknown = realloc (unknown, 
						strlen (unknown) + strlen (line) + 2);
				strcat (unknown, "\n");
				strcat (unknown, line);
			} else {
				unknown = malloc (strlen (line) + 1);
				strcpy (unknown, line);
			}

#endif


			break;
		case COLOR:

#if USE_SPLASHIMAGE==0

			tmp_str_mid = tmp_str = get_value ((line) + 5);
			if (!tmp_str) {
				gtk_message (GTK_MESSAGE_ERROR, _("[%s line %i]\nInvalid color settings"), grubconf_fname, line_num);
				return 0;
			}
			// ignore blinks until I figure out how to 
			// handle this with more grace
			if (strncmp (tmp_str_mid, "blink-", 6) == 0) {
				tmp_str_mid += 6;
			}
			
			// find the normal colors
			nFor = get_color_enum (tmp_str_mid);
			tmp_str_mid = strstr (tmp_str_mid, "/");
			if (!tmp_str_mid) {
				gtk_message (GTK_MESSAGE_ERROR, _("[%s line %i]\nInvalid color settings"), grubconf_fname, line_num);
				free (tmp_str);
				return 0;
			}
			nBak = get_color_enum (++tmp_str_mid);
			
			// find the highlight colors
			// scan for end of normal color		
			for ( ; *tmp_str_mid != '\0' && *tmp_str_mid != '\n' &&
						*tmp_str_mid != ' ' && *tmp_str_mid != '\t'; 
						tmp_str_mid++) { }
			//  scan any extra whitespace
			for ( ; *tmp_str_mid != '\0' && *tmp_str_mid != '\n' &&
					(*tmp_str_mid == ' ' || *tmp_str_mid == '\t'); 
					tmp_str_mid++) { }
			if (*tmp_str_mid == '\0' || *tmp_str_mid == '\n') {
				// if we there was no highlight specified
				hFor = INVERSE;
				hBak = DARK_GRAY; // this is where INVERSE is in the background
			} else {
				// if there was a hightlight specified
				if (strncmp (tmp_str_mid, "blink-", 6) == 0) {
					tmp_str_mid += 6;
				}
				hFor = get_color_enum (tmp_str_mid);
				tmp_str_mid = strstr (tmp_str_mid, "/");
				if (!tmp_str_mid) {
					gtk_message (GTK_MESSAGE_ERROR, _("[%s line %i]\nInvalid color settings"), grubconf_fname, line_num);
					free (tmp_str);
					return 0;
				}
				hBak = get_color_enum (++tmp_str_mid);
			}
			
			gtk_option_menu_set_history (GTK_OPTION_MENU (optNFor), nFor);
			gtk_option_menu_set_history (GTK_OPTION_MENU (optNBak), nBak);
			gtk_option_menu_set_history (GTK_OPTION_MENU (optHFor), hFor);
			gtk_option_menu_set_history (GTK_OPTION_MENU (optHBak), hBak);
			gtk_toggle_button_set_active (
			       GTK_TOGGLE_BUTTON (lookup_widget (winMain, "chkColor")),
			       TRUE);
			free (tmp_str);
			
			// ---------------------------------------------------
#else

			if (unknown) {
				unknown = realloc (unknown, 
						strlen (unknown) + strlen (line) + 2);
				strcat (unknown, "\n");
				strcat (unknown, line);
			} else {
				unknown = malloc (strlen (line) + 1);
				strcpy (unknown, line);
			}

#endif

			break;
		case HIDDENMENU:
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
						      (chkGenHidden), TRUE);
			break;
		case TITLE:
			// now that we found a title, set any unknown commands
			if (unknown) {
				gtk_text_buffer_set_text (tbGenExtra,
							  unknown,
							  strlen (unknown));
				free (unknown);
				unknown = NULL;
			}
			os_index = 0;
			/* deals with start/end of OS. This loop doesnt acutally read in a 
			 * line, becuase to start it already has a title, and in the inner
			 * while it will break when a new title is found. but of couse
			 * check for feof before we go at it again.
			 */
			 while (!feof (grubconf_fp)) {
 				in_title = TRUE;
				title = get_value ((line) + 5);
				if (stristr (title, "linux") != NULL) {
					type = LINUX;
				} else if (stristr (title, "win") != NULL) {
					type = WINDOWS;
				} else {
					type = OTHER;
				}
				// be sure to free up the line before we get another
				if (line) {
					free (line);
					line = NULL;
				}
				// loop to gather OS info.
				while (in_title && my_getline (&line, grubconf_fp)) {
					if (!line || feof (grubconf_fp)) {
						break;
					}
					line_num++;
					// check for comments and 0-length lines
					if (*line == '#') {
						// check for internally used type classification
						if (line[1] == ':') {
							type = OS_NUM_TYPE (line[2]);
						}
					}
					if (*line == '#' || *line == '\n' || *line == '\0') {
						free (line);
						line = NULL;
						continue;
					}
					enum known_commands tmp_cmd = get_command (&line);
					if (type == OTHER && tmp_cmd != TITLE)
						tmp_cmd = UNKNOWN;
					switch (tmp_cmd) {
					case TITLE:
						in_title = FALSE;
						break;
					case ROOT:
						tmp_str = get_value ((line) + 4);
						root_num = get_index_grub (tmp_str);
						free (tmp_str);
						if (root_num  == -2) {
							gtk_message (GTK_MESSAGE_ERROR, _("[%s line %i]\nInvalid kernel syntax"), grubconf_fname, line_num);
							return 0;
						} else if (root_num == -1) {
							gtk_message (GTK_MESSAGE_ERROR, 
								_("[%s line %i]\nWas not able to find root device for %s. This must be fixed!")
								, grubconf_fname, line_num, title);
						}
						break;
					case KERNEL:
						tmp_str_mid = tmp_str = get_value ((line) + 7);
						// parse kernel string
						
						tmp_int = get_index_grub (tmp_str);
						
						if (tmp_int > -2) {
							if (tmp_int != -1) {
								root_num = tmp_int;
							} else {
								gtk_message (GTK_MESSAGE_WARNING, _("[%s line %i]\n%s contains a device that was not detected. This must be fixed!"), grubconf_fname, line_num, title);
							}
							tmp_str_mid = strchr (tmp_str, ')');
							if (!tmp_str_mid) {
								gtk_message (GTK_MESSAGE_ERROR, _("[%s line %i]\nInvalid kernel syntax"), grubconf_fname, line_num);
								free (tmp_str);
								return 0;
							}	
							tmp_str_mid++;
							if (tmp_str_mid[0] == '\0' || tmp_str_mid[1] == '\0') {
								gtk_message (GTK_MESSAGE_ERROR, _("[%s line %i]\nInvalid kernel syntax"), grubconf_fname, line_num);
								free (tmp_str);
								return 0;
							}	
						} else if (tmp_str[0] == '(') {
							gtk_message (GTK_MESSAGE_ERROR, _("[%s line %i]\nInvalid kernel syntax"), grubconf_fname, line_num);
							free (tmp_str);
							return 0;
						}
						
						// first find end of kernel name
						for (tmp_int = 1; *(tmp_str_mid + tmp_int) != '\0'
						     && *(tmp_str_mid + tmp_int) != ' '; tmp_int++) { }
						kernel = malloc (tmp_int + 1);
						strncpy (kernel, tmp_str_mid, tmp_int);
						*(kernel + tmp_int) = '\0';
						if (*(tmp_str_mid + tmp_int) == '\0') {
							free (tmp_str);
							break;
						}
						// now get ourselves a root device
						if (*(tmp_str_mid + ++tmp_int) ==' ') {
							// if there are more spaces, get moving
							for ( ; *(tmp_str_mid + tmp_int) != '\0'
								 && *(tmp_str_mid + tmp_int) == ' '; tmp_int++)	{ }
						}
						if (*(tmp_str_mid + tmp_int) == '\0') {
							free (tmp_str);
							break;
						}
						// now parse any extra kernel parms before root=
						tmp_str_mid += tmp_int;
						tmp_int = 0;
						do {
							// check if the command we're at is root
							if (strncmp (tmp_str_mid + tmp_int, "root=", 5) == 0) {
								break;
							}
							// find the end of the command
							for (tmp_int++; *(tmp_str_mid + tmp_int) != '\0'
								 && *(tmp_str_mid + tmp_int) != ' '; tmp_int++) { }
							if (*(tmp_str_mid + tmp_int) == '\0') {
								break;
							}
							// eat up any extra spaces if any
							if (*(tmp_str_mid + ++tmp_int) == ' ') {
								for (; *(tmp_str_mid + tmp_int) != '\0' 
										&& *(tmp_str_mid + tmp_int) == ' '; tmp_int++) { }
							}
						} while (*(tmp_str_mid + tmp_int) != '\0');
						// if we didnt find root= on our first try ...
						if (tmp_int != 0) { 
							kern_parms = malloc (tmp_int + 1);
							strncpy (kern_parms, tmp_str_mid, tmp_int);
							*(kern_parms + tmp_int) = 0;
							tmp_str_mid += tmp_int;
						}
						// anchor ourselves at the root device
						tmp_str_mid += 5;
						if (*tmp_str_mid == 'L') {
							tmp_str_mid += 6;
							for (tmp_int = 1; *(tmp_str_mid + tmp_int) != '\0'
								 && *(tmp_str_mid + tmp_int) != ' '; tmp_int++) { }
							if (tmp_str_mid[tmp_int] == ' ') {
								tmp_str_mid[tmp_int] = 0;
								kern_num = get_index_label (tmp_str_mid);
								tmp_str_mid[tmp_int] = ' ';
							} else {
								kern_num = get_index_label (tmp_str_mid);
							}
							
						} else {
							kern_num = get_index_dev (tmp_str_mid);
						}
						// see if there are any extra parameters
						for (tmp_int = 1; *(tmp_str_mid + tmp_int) != '\0'
						     && *(tmp_str_mid + tmp_int) != ' '; tmp_int++) { }
						if (*(tmp_str_mid + tmp_int)  == '\0') {
							free (tmp_str);
							break;
						}
						// if we're here there must be extra parameters
						tmp_str_mid += tmp_int + 1;
						if (kern_parms) {
							kern_parms = realloc (kern_parms, strlen (kern_parms) + strlen (tmp_str_mid) + 1);
							strcat (kern_parms, tmp_str_mid);
						} else {
							kern_parms = malloc (strlen (tmp_str_mid) + 1);
							strcpy (kern_parms, tmp_str_mid);
						}
						free (tmp_str);
						break;
					case ROOTNOVERIFY:
						tmp_str = get_value ((line) + 12);
						root_num = get_index_grub (tmp_str);
						if (root_num < 0) {
							gtk_message (GTK_MESSAGE_ERROR, 
								_("[%s line %i]\nWas not able to find device for %s, given %s")
								, grubconf_fname, line_num, title, tmp_str);
							return 0;
						}
						free (tmp_str);
						break;
					case MAKEACTIVE:
					case CHAINLOADER:
						break;
					case UNKNOWN:
						if (unknown) {
							unknown = realloc (unknown, strlen (unknown) + strlen (line) + 2);
							strcat (unknown, "\n");
							strcat (unknown, line);
						} else {
							unknown = malloc (strlen (line) + 1);
							strcpy (unknown, line);
						}
						break;
					default:
						return 0;
						break;
					}
					// be sure to free up the line before we get another
					//  as long as we are in a title (not between titles)
					if (in_title && line) {
						free (line);
						line = NULL;
					}
				}

				// now that we have all the data for this OS, 
				// add it to our internal list
				switch (type) {
				case LINUX:
					if (!os_list_new_linux ((os_index == def), title,
								    kern_num,
								    root_num,
								    kernel,
								    kern_parms,
								    unknown)) {
						gtk_message (GTK_MESSAGE_ERROR, _("Error allocating %s"), title);
						return 0;
					}
						
					break;
				case WINDOWS:
					if (!os_list_new_windows ((os_index == def), title, root_num, unknown)) {
						gtk_message (GTK_MESSAGE_ERROR, _("Error allocating %s"), title);
						return 0;
					}
					break;
				case OTHER:
					if (!os_list_new_other ((os_index == def), title, unknown)) {
						gtk_message (GTK_MESSAGE_ERROR, _("Error allocating %s"), title);
						return 0;
					}
					break;
				default:
					// do nothing
					break;
				}
				kern_parms = NULL;
				unknown = NULL;
				os_index++;	// keep track of index for default OS
			}

			break;
		default:
			gtk_message (GTK_MESSAGE_ERROR, _("Invalid command"));
			return 0;
			break;
		}
	}
	fclose (grubconf_fp);

	// be sure to reset revert button state
	gtk_widget_set_sensitive (cmdOSRevertMain, FALSE);
	return 1;
}

/* Parses a line from the config file and returns an enumeration of the type
 * it also trims any whitespace off the beginning of the line
 */
enum known_commands
get_command (char **line)
{
	char *tmp = *line;
	char *buf;
	// first see if the line has whitespace at the beginning
	while (*tmp != '\0' && (*tmp == '\t' || *tmp == ' ')) {
		tmp++;
	}
	// if we ended up moving tmp, re-allocate a shorter string.
	if (tmp != *line) {
		buf = malloc (strlen (tmp) + 1);
		strcpy (buf, tmp);
		free (*line);
		*line = buf;
	}
	if (strncmp (*line, "default", 7) == 0)
		return DEFAULT;
	if (strncmp (*line, "timeout", 7) == 0)
		return TIMEOUT;
	if (strncmp (*line, "splashimage", 11) == 0)
		return SPLASHIMAGE;
	if (strncmp (*line, "color", 5) == 0)
		return COLOR;
	if (strncmp (*line, "hiddenmenu", 10) == 0)
		return HIDDENMENU;
	if (strncmp (*line, "title", 5) == 0)
		return TITLE;
	if (strncmp (*line, "root", 4) == 0 && *(*line + 4) != 'n')
		return ROOT;
	if (strncmp (*line, "kernel", 6) == 0)
		return KERNEL;
	if (strncmp (*line, "rootnoverify", 12) == 0)
		return ROOTNOVERIFY;
	if (strncmp (*line, "makeactive", 10) == 0)
		return MAKEACTIVE;
	if (strncmp (*line, "chainloader", 11) == 0)
		return CHAINLOADER;
	return UNKNOWN;
}

/* parses a line from the config file and returns the value of that option
 */
char *
get_value (char *line)
{
	char *ret;
	char *tmp = line;
	int size;
	// first point tmp to the start of the value scrolling
	//  past whitespace and =
	while (*tmp != '\0' && (*tmp == ' ' || *tmp == '=' || *tmp == '\t'))
	{
		tmp++;
	}
	if (*tmp == '\0')
		return NULL;
	size = strlen (tmp);
	ret = malloc (size + 1);
	strcpy (ret, tmp);
	if (*(ret + size - 1) == '\n')
		*(ret + size - 1) = '\0';
	return ret;
}

/* searches haystack for needle case-insensitively
 */
char *
stristr (char *haystack, char *needle)
{
	unsigned char *a, *b;
	/* First scan quickly through the two strings looking for a
	 * single-character match.  When it's found, then compare the
	 * rest of the substring.
	 */
	b = needle;
	if (*b == 0) {
		return haystack;
	}
	for (; *haystack != 0; haystack += 1) {
		if (tolower (*haystack) != tolower (*b)) {
			continue;
		}
		a = haystack;
		while (1) {
			if (*b == 0) {
				return haystack;
			}
			if (tolower (*a++) != tolower (*b++)) {
				break;
			}
		}
		b = needle;
	}
	return NULL;
}

/* Saves the grub configuration to file
 */
int
save_to_file ()
{
	if (!grubconf_fname) {
		gtk_message (GTK_MESSAGE_ERROR, "configuration file not available, quitting without saving");
		return 1; // return 1 b/c we want to quit
	}
	FILE *grubconf_fp = fopen (grubconf_fname, "w");
	if (!grubconf_fp) {
		gtk_message (GTK_MESSAGE_ERROR, "configuration file not available, quitting without saving");
		return 1; // return 1 b/c we want to quit
	}
	 
	GtkTextIter start, end;
 //   gchar *warning_msg = (gchar*)malloc(sizeof(gchar)*100);
	
	//start of GUI value strings
	int default_OS = 0;
	int timeout_value = 0;	//holds integer version of timeout
	gboolean hidden_value;	//bool to dtermine whether of not it is a hidden menu
	struct grubconf_dev *selected_device;	//the info behind the selected device
	GtkTextBuffer *tbGenExtra;	//textbox buffer general tab extra options
	gchar *extra_text;	//string that holds the value of the text buffer
	//end of GUI value strings

	GtkWidget *chkGenHidden = lookup_widget (winMain, "chkGenHidden");
	hidden_value = gtk_toggle_button_get_active (
	                      GTK_TOGGLE_BUTTON (chkGenHidden));

#if USE_SPLASHIMAGE
	
	gchar *splash_loc = NULL;	//string to hold location of splash image
	gint selected_device_index;	//index of the selected spash image device 

	GtkWidget *chkSplash = lookup_widget (winMain, "chkSplash");
	gboolean splash = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (chkSplash));

	if (splash) {
		GtkWidget *txtGenSplash = lookup_widget (winMain, "txtGenSplash");
		splash_loc = gtk_editable_get_chars (
		                 GTK_EDITABLE (txtGenSplash), 0, -1);
		if (!splash_loc || splash_loc[0] == '\0') {
			gtk_message (GTK_MESSAGE_WARNING, "Invalid splash image. Please reconfigure");
			return 0;
		}
	}
	
	GtkWidget *optGenSplashDev = lookup_widget (winMain, "optGenSplashDev");
	selected_device_index = get_selected (optGenSplashDev);
	selected_device = get_dev (selected_device_index);

#else
	
	GtkWidget *chkColor = lookup_widget (winMain, "chkColor");
	GtkWidget *optNFor = lookup_widget (winMain, "optNFor");
	GtkWidget *optNBak = lookup_widget (winMain, "optNBak");
	GtkWidget *optHFor = lookup_widget (winMain, "optHFor");
	GtkWidget *optHBak = lookup_widget (winMain, "optHBak");
	enum color_enum nFor, nBak, hFor, hBak;
	
	gboolean color = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (chkColor));
	
	nFor = get_selected (optNFor);
	nBak = get_selected (optNBak);
	hFor = get_selected (optHFor);
	hBak = get_selected (optHBak);

#endif	
	

	GtkWidget *tvGenExtra = lookup_widget (diaMore, "tvGenExtra");
	tbGenExtra = gtk_text_view_get_buffer (GTK_TEXT_VIEW (tvGenExtra));
	gtk_text_buffer_get_bounds (tbGenExtra, &start, &end);
	extra_text =
		gtk_text_buffer_get_text (GTK_TEXT_BUFFER (tbGenExtra),
					  &start, &end, 0);
    
	GtkWidget *timeout_spin = lookup_widget (winMain, "timeout_spin");
	timeout_value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(timeout_spin));

	char *title;
	struct os_list_item *os_data;
	struct os_list_linux *temp_lin;
	struct os_list_windows *temp_win;
	
	
	GtkTreeModel *model = gtk_tree_view_get_model ( GTK_TREE_VIEW (lookup_widget (winMain, "treeview1")) );
	GtkTreeIter iter;
	gboolean def;

	if (!gtk_tree_model_get_iter_first (model, &iter)) {
		gtk_message (GTK_MESSAGE_WARNING, "No operating systems chosen! Please reconfigure.");
		return 0;
	}
	do {
		gtk_tree_model_get (model, &iter,
							DEFAULT_COLUMN, &def,
							-1);
		if (def)
			break;
		
		default_OS++;
	} while (gtk_tree_model_iter_next (model, &iter) || (default_OS = 0));

	
	fprintf (grubconf_fp, "# Generated by grubconf-%s\n", VERSION);
	fprintf (grubconf_fp, "default=%d\n", default_OS);	//print the default line
	if (hidden_value){
	   fprintf (grubconf_fp, "hiddenmenu\n");
	}
	
	fprintf (grubconf_fp, "timeout=%d\n", timeout_value); //print the timeout value

#if USE_SPLASHIMAGE	

	if (splash) {
		fprintf (grubconf_fp, "splashimage=(hd%d,%d)%s\n", 
		         selected_device->drv_num, 
		         selected_device->part_num, 
		         splash_loc);
	}

#else
	
	if (color) {
		fprintf (grubconf_fp, "color=%s/%s", 
		         get_color_string (nFor),
		         get_color_string (nBak));
		if (hFor == INVERSE) {
			fprintf (grubconf_fp, "\n");
		} else {
			fprintf (grubconf_fp, " %s/%s\n",
			         get_color_string (hFor),
			         get_color_string (hBak));
		}
	}			
	
#endif
	
	if (extra_text && extra_text != '\0') {
		fprintf (grubconf_fp, "%s\n", extra_text);
	}
	
	gtk_tree_model_get_iter_first (model, &iter);
	do
	{
		gtk_tree_model_get (model, &iter,
							TITLE_COLUMN, &title,
		                    DATA_COLUMN, &os_data,
							-1);
		
		fprintf (grubconf_fp, "title %s\n", title);
		fprintf (grubconf_fp, "#:%i <-- type: 0 => linux, 1 => windows, 2 => other\n", os_data->type);
		switch (os_data->type) {
		case LINUX:

			temp_lin = os_data->data;

			selected_device =
				get_dev (temp_lin->bdev_index);
			fprintf (grubconf_fp, "\troot (hd%d,%d)\n",
				 selected_device->drv_num,
				 selected_device->part_num);

			selected_device =
				get_dev (temp_lin->rdev_index);
			fprintf (grubconf_fp, "\tkernel %s root=%s %s\n",
				 temp_lin->kernel,
				 selected_device->dev_name,
				 temp_lin->kernel_parms);

			break;
		case WINDOWS:
	
			temp_win = os_data->data;
			
			selected_device = get_dev (temp_win->dev_index);
			fprintf (grubconf_fp, "\trootnoverify (hd%d,%d)\n",
				 selected_device->drv_num,
				 selected_device->part_num);

			fprintf (grubconf_fp, "\tmakeactive\n\tchainloader +1\n");

			break;
		default:
			break;
		} 

		if (os_data->unknown != NULL && *os_data->unknown != '\0') {					
			fprintf (grubconf_fp, "%s\n", os_data->unknown);
		}
	}
	while (gtk_tree_model_iter_next (model, &iter));
	
//   	free(warning_msg);
	fclose (grubconf_fp);
	return 1;
}

/* Resests the OS editing dialog to default values
 */
void
reset_diaOS ()
{
	GtkWidget *txtOSTitle = lookup_widget (diaOS, "txtOSTitle");
	GtkWidget *optOSType = lookup_widget (diaOS, "optOSType");
	GtkWidget *optLinuxBootDev = lookup_widget (diaOS, "optLinuxBootDev");
	GtkWidget *optLinuxRootDev = lookup_widget (diaOS, "optLinuxRootDev");
	GtkWidget *txtOSLinuxKernel =
		lookup_widget (diaOS, "txtOSLinuxKernel");
	GtkWidget *txtOSLinuxParams =
		lookup_widget (diaOS, "txtOSLinuxParams");
	GtkWidget *tvOSLinuxExtra = lookup_widget (diaOS, "tvOSLinuxExtra");
	GtkTextBuffer *tbOSLinuxExtra =
		gtk_text_view_get_buffer (GTK_TEXT_VIEW (tvOSLinuxExtra));
	GtkWidget *optOSWinDev = lookup_widget (diaOS, "optOSWinDev");
	GtkWidget *tvOSOther = lookup_widget (diaOS, "tvOSOther");
	GtkTextBuffer *tbOSOther =
		gtk_text_view_get_buffer (GTK_TEXT_VIEW (tvOSOther));
	GtkWidget *fraOSWindows = lookup_widget (diaOS, "fraOSWindows");
	GtkWidget *fraOSOther = lookup_widget (diaOS, "fraOSOther");
	GtkWidget *fraOSLinux = lookup_widget (diaOS, "fraOSLinux");

	gtk_window_set_title (GTK_WINDOW (diaOS), "[ No Title Entered ]");
	
	gtk_editable_delete_text (GTK_EDITABLE (txtOSTitle), 0, -1);
	gtk_option_menu_set_history (GTK_OPTION_MENU (optOSType), 0);

	gtk_option_menu_set_history (GTK_OPTION_MENU (optLinuxBootDev), 0);
	gtk_option_menu_set_history (GTK_OPTION_MENU (optLinuxRootDev), 0);
	gtk_editable_delete_text (GTK_EDITABLE (txtOSLinuxKernel), 0, -1);
	gtk_editable_delete_text (GTK_EDITABLE (txtOSLinuxParams), 0, -1);
	gtk_text_buffer_set_text (tbOSLinuxExtra, "", 0);

	gtk_option_menu_set_history (GTK_OPTION_MENU (optOSWinDev), 0);

	gtk_text_buffer_set_text (tbOSOther, "", 0);

	gtk_widget_hide (fraOSWindows);
	gtk_widget_hide (fraOSOther);
	gtk_widget_show (fraOSLinux);
}

/* from index, finds the corresponding os_list_item and loads diaOS
 */
void
edit_diaOS (GtkTreeModel *model, GtkTreeIter *iter)
{
	GtkWidget *txtOSTitle = lookup_widget (diaOS, "txtOSTitle");
	GtkWidget *optOSType = lookup_widget (diaOS, "optOSType");
	GtkWidget *optLinuxRootDev = lookup_widget (diaOS, "optLinuxRootDev");
	GtkWidget *optLinuxBootDev = lookup_widget (diaOS, "optLinuxBootDev");
	GtkWidget *txtOSLinuxKernel =
		lookup_widget (diaOS, "txtOSLinuxKernel");
	GtkWidget *txtOSLinuxParams =
		lookup_widget (diaOS, "txtOSLinuxParams");
	GtkWidget *tvOSLinuxExtra = lookup_widget (diaOS, "tvOSLinuxExtra");
	GtkTextBuffer *tbOSLinuxExtra =
		gtk_text_view_get_buffer (GTK_TEXT_VIEW (tvOSLinuxExtra));
	GtkWidget *optOSWinDev = lookup_widget (diaOS, "optOSWinDev");
	GtkWidget *tvOSOther = lookup_widget (diaOS, "tvOSOther");
	GtkTextBuffer *tbOSOther =
		gtk_text_view_get_buffer (GTK_TEXT_VIEW (tvOSOther));
	GtkWidget *fraOSWindows = lookup_widget (diaOS, "fraOSWindows");
	GtkWidget *fraOSOther = lookup_widget (diaOS, "fraOSOther");
	GtkWidget *fraOSLinux = lookup_widget (diaOS, "fraOSLinux");
	
	struct os_list_item *item;
	char *title;

	gtk_tree_model_get (model, iter, TITLE_COLUMN, &title, DATA_COLUMN, &item, -1);	

	struct os_list_linux *tmp_linux;
	struct os_list_windows *tmp_windows;
	if (item != NULL)
	{
		if (title != NULL)
		{
			gtk_entry_set_text (GTK_ENTRY (txtOSTitle), title);
			gtk_window_set_title (GTK_WINDOW (diaOS), title);
		}
		gtk_option_menu_set_history (GTK_OPTION_MENU (optOSType), item->type);
		switch (item->type)
		{
		case LINUX:
			if (item->data != NULL)
			{
				tmp_linux =
					(struct os_list_linux *) item->data;
				gtk_option_menu_set_history (GTK_OPTION_MENU
							     (optLinuxRootDev),
							     tmp_linux->rdev_index);
				gtk_option_menu_set_history (GTK_OPTION_MENU
							     (optLinuxBootDev),
							     tmp_linux->bdev_index);
				if (tmp_linux->kernel != NULL)
				{
					gtk_entry_set_text (GTK_ENTRY (txtOSLinuxKernel),
								  tmp_linux->kernel);
				}
				if (tmp_linux->kernel_parms != NULL)
				{
					gtk_entry_set_text (GTK_ENTRY (txtOSLinuxParams),
								  tmp_linux->kernel_parms);
				}
				if (item->unknown != NULL)
				{
					gtk_text_buffer_set_text
						(tbOSLinuxExtra,
						 item->unknown,
						 strlen (item->unknown));
				}
			}
			gtk_widget_hide (fraOSWindows);
			gtk_widget_hide (fraOSOther);
			gtk_widget_show (fraOSLinux);
			break;
		case WINDOWS:
			if (item->data != NULL)
			{
				tmp_windows =
					(struct os_list_windows *) item->data;
				gtk_option_menu_set_history (GTK_OPTION_MENU (optOSWinDev),
							     tmp_windows->dev_index);
			}
			gtk_widget_hide (fraOSLinux);
			gtk_widget_hide (fraOSOther);
			gtk_widget_show (fraOSWindows);
			break;
		case OTHER:
			if (item->unknown != NULL)
			{
				gtk_text_buffer_set_text (tbOSOther,
							  item->unknown,
							  strlen (item->unknown));
			}
			gtk_widget_hide (fraOSLinux);
			gtk_widget_hide (fraOSWindows);
			gtk_widget_show (fraOSOther);
			break;
		}
	}

}

/* reset the device menus that use paths
 */
void
reset_dev_menus () {
	// retrieve splash device selection and make new menu
	GtkWidget *menu_widget = lookup_widget (winMain, "optGenSplashDev");
	GtkWidget *menu = gtk_menu_new();

	// save history
	int hist = gtk_option_menu_get_history (GTK_OPTION_MENU (menu_widget));

	// reload the menu
	dev_load_menu_path (menu, TRUE);
	
	// restore menu and history
	gtk_option_menu_set_menu (GTK_OPTION_MENU (menu_widget), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (menu_widget), hist);

	
	// retrieve splash device selection and make new menu
	menu_widget = lookup_widget (diaOS, "optLinuxBootDev");
	menu = gtk_menu_new();

	// save history
	hist = gtk_option_menu_get_history (GTK_OPTION_MENU (menu_widget));

	// reload the menu
	dev_load_menu_path (menu, TRUE);
	
	// restore menu and history
	gtk_option_menu_set_menu (GTK_OPTION_MENU (menu_widget), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (menu_widget), hist);
}

/* get color from char*
 */
enum color_enum get_color_enum (char *color) {
	if (strncmp (color, "black", 5) == 0) {
		return BLACK;
	} else if (strncmp (color, "blue", 4) == 0) {
		return BLUE;
	} else if (strncmp (color, "green", 5) == 0) {
		return GREEN;
	} else if (strncmp (color, "cyan", 4) == 0) {
		return CYAN;
	} else if (strncmp (color, "red", 3) == 0) {
		return RED;
	} else if (strncmp (color, "magenta", 7) == 0) {
		return MAGENTA;
	} else if (strncmp (color, "brown", 5) == 0) {
		return BROWN;
	} else if (strncmp (color, "light-gray", 10) == 0) {
		return LIGHT_GRAY;
	} else if (strncmp (color, "dark-gray", 9) == 0) {
		return DARK_GRAY;
	} else if (strncmp (color, "light-blue", 10) == 0) {
		return LIGHT_BLUE;
	} else if (strncmp (color, "light-green", 11) == 0) {
		return LIGHT_GREEN;
	} else if (strncmp (color, "light-cyan", 10) == 0) {
		return LIGHT_CYAN;
	} else if (strncmp (color, "light-red", 9) == 0) {
		return LIGHT_RED;
	} else if (strncmp (color, "light-magenta", 13) == 0) {
		return LIGHT_MAGENTA;
	} else if (strncmp (color, "yellow", 6) == 0) {
		return YELLOW;
	} else if (strncmp (color, "white", 5) == 0) {
		return WHITE;
	}
	return INVERSE;
}

/* get char* from color
 */
char *get_color_string (enum color_enum color) {
	switch (color) {
	case BLACK:
		return "black";
	case BLUE:
		return "blue";
	case GREEN:
		return "green";
	case CYAN:
		return "cyan";
	case RED:
		return "red";
	case MAGENTA:
		return "magenta";
	case BROWN:
		return "brown";
	case LIGHT_GRAY:
		return "light-gray";
	case DARK_GRAY:
		return "dark-gray";
	case LIGHT_BLUE:
		return "light-blue";
	case LIGHT_GREEN:
		return "light-green";
	case LIGHT_CYAN:
		return "light-cyan";
	case LIGHT_RED:
		return "light-red";
	case LIGHT_MAGENTA:
		return "light-magenta";
	case YELLOW:
		return "yellow";
	case WHITE:
		return "white";
	case INVERSE:
	default:
		return NULL;
	}
	return NULL;
}
