/*
 * Grouch.app				Copyright (C) 2006 Andy Sveikauskas
 * ------------------------------------------------------------------------
 * This program is free software under the GNU General Public License
 * --
 * Protocols for GUI <-> portocol bundle interaction.
 */

#import <Foundation/NSString.h>
#include <time.h>

@class NSArray;

// Message flags for IM and chat rooms.
typedef enum
{
	GrouchMessageError = (1<<0),	// error message.  display in red maybe?
	GrouchMessageReflect = (1<<1),	// request that this is sent back to UI
	GrouchMessageAway = (1<<2),	// auto-response
	GrouchMessageEmote = (1<<3)	// this is a /me
} GrouchMessageFlags;

// All user interaction should go through this class.
// To look up users see GrouchClient -getUser:
@protocol GrouchUser

- (NSString*)name;
- (NSString*)alias;
- (void)setAlias:(NSString*)alias;

- (BOOL)online;
- (void)online:(BOOL)o;
- (void)onlineSince:(time_t)t;
- (void)idle:(time_t)t;
- (void)away:(BOOL)a;
- (void)warn:(int)warn;

- (void)message:(NSString*)str withFlags:(GrouchMessageFlags)flags;

// Call this to update the GUI on a status change
- (void)reloadData;

@end

// Same deal, but for chat channels
@protocol GrouchChannel

// You joined.
- (void)welcome;
// You left.
- (void)bye;

- (void)join:(NSString*)user;
- (void)part:(NSString*)user;

- (void)message:(NSString*)msg from:(NSString*)usr
	withFlags:(GrouchMessageFlags)flags;

// Call this after join: or part: to tell the GUI that the user list
// has changed.  If you add/remove a large group of users, call this
// at the very end of all that.
- (void)reloadData;

@end

// The protocol code provides this; the GUI uses it to reply to prompt methods.
@protocol GrouchPrompt
- (void)gotResult:obj;
@end

// This is a protocol that the GUI should speak, that protocol code
// can use to notify the GUI of events.
@protocol GrouchClient

// Login status message
- (void)statusMessage:(NSString*)msg;

- (void)welcome:(NSString*)yourNick;
- (void)bye;

// Note: due to multi-threading, it is likely that you will receive this
// message BEFORE you even have a pointer to an object
- (void)error:(NSString*)str fatal:(BOOL)fatal;

- (id<GrouchUser>)getUser:(NSString*)nick;
- (id<GrouchChannel>)getChannel:(NSString*)chan;

// Call this to refresh the buddy list.
// If newBuddyList is true, the client will ask the protocol code for a
// new copy of the buddy list.  Otherwise, it will just re-draw everything
// using the existing value.  (Protocol code will not want to do this second
// case; it is already done by User-reloadData.)
- (void)reloadData:(BOOL)newBuddyList;

// The prompt reply will call -gotResult:nil if declined, -gotResult:self
// if accepted.
- (void)yesNoPrompt:(id<GrouchPrompt>)promptObject
	title:(NSString*)title message:(NSString*)msg;

@end

// Protocol code will also need to call this.
@protocol GrouchProfile
- (void)profile:(NSString*)str;
- (void)awayMessage:(NSString*)str;
- (void)finishedProfile;
@end

// protocol that the GUI uses to interact with the network
@protocol GrouchSession

- tag;
- (void)tag:t;

- (void)profile:(NSString*)str;
- (void)away:(NSString*)str;
- (void)idle:(time_t)idle;

- (void)sendMessage:(NSString*)msg to:(NSString*)user
  withFlags:(GrouchMessageFlags)f;
- (void)sendMessage:(NSString*)msg toChannel:(NSString*)user
  withFlags:(GrouchMessageFlags)f;
- (void)joinChannel:(NSString*)str;
- (void)partChannel:(NSString*)str;

- (void)getInfo:(id<GrouchProfile>)obj forUser:(NSString*)str;
- (void)getAwayMessage:(id<GrouchProfile>)obj forUser:(NSString*)str;

// The protocol code is responsible for managing buddy list.
// Ask for contact list.  Should be an array of NSArrays (for groups) and
// id<GrouchUser>s (for buddies).  The first element of a group should be
// its title (NSString).
- (NSArray*)getContactList:(BOOL)showOfflineUsers;

// Argument can be an id<GrouchUser> or an NSArray* (one of the group items
// in -getContactList)
- (void)removeFromList:obj;

- (void)addUser:(NSString*)user toGroup:(NSString*)group;

// Object can be id<GrouchUser> or NSArray* as above.
- (void)rename:obj to:(NSString*)str;

@end

// The "main class" in a protocol bundle.  Your bundle will have to
// implement this and set NSPrincipalClass to it.
@protocol GrouchBundle
- (id<GrouchSession>)loadGrouchClient:(id<GrouchClient>)cli
  forHost:(NSString*)host atPort:(int)port
  withLogin:(NSString*)login andPassword:(NSString*)passwd;
@end
