/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */
 
#include "config.h"

#ifdef HAVE_LIBSSH

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <glib/gstdio.h>
#include <pthread.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include "grdcpixmaps.h"
#include "grdcpublic.h"
#include "grdcpref.h"
#include "grdcssh.h"
#include "grdcsftpwindow.h"

G_DEFINE_TYPE (GrdcSFTPWindow, grdc_sftp_window, GRDC_TYPE_FTP_WINDOW)

static void
grdc_sftp_window_class_init (GrdcSFTPWindowClass *klass)
{
}

#define GET_SFTPATTR_TYPE(a,type) \
    if (a->type == 0) \
    { \
        type = ((a->permissions & 040000) ? GRDC_FTP_FILE_TYPE_DIR : GRDC_FTP_FILE_TYPE_FILE); \
    } \
    else \
    { \
        type = (a->type == SSH_FILEXFER_TYPE_DIRECTORY ? GRDC_FTP_FILE_TYPE_DIR : GRDC_FTP_FILE_TYPE_FILE); \
    }

/* ------------------------ The Task Thread routines ----------------------------- */

static gboolean grdc_sftp_window_refresh (GrdcSFTPWindow *window);

#define THREAD_CHECK_EXIT \
    (!window->taskid || window->thread_abort)

static gboolean
grdc_sftp_window_thread_update_task (GrdcSFTPWindow *window, GrdcFTPTask *task)
{
    if (THREAD_CHECK_EXIT) return FALSE;

    THREADS_ENTER
    grdc_ftp_window_update_task (GRDC_FTP_WINDOW (window), task);
    THREADS_LEAVE
    return TRUE;
}

static void
grdc_sftp_window_thread_set_error (GrdcSFTPWindow *window, GrdcFTPTask *task, const gchar *error_format, ...)
{
    va_list args;

    task->status = GRDC_FTP_TASK_STATUS_ERROR;
    g_free (task->tooltip);
    va_start (args, error_format);
    task->tooltip = g_strdup_vprintf (error_format, args);
    va_end (args);

    grdc_sftp_window_thread_update_task (window, task);
}

static void
grdc_sftp_window_thread_set_finish (GrdcSFTPWindow *window, GrdcFTPTask *task)
{
    task->status = GRDC_FTP_TASK_STATUS_FINISH;
    g_free (task->tooltip);
    task->tooltip = NULL;

    grdc_sftp_window_thread_update_task (window, task);
}

static GrdcFTPTask*
grdc_sftp_window_thread_get_task (GrdcSFTPWindow *window)
{
    GrdcFTPTask *task;

    if (window->thread_abort) return NULL;

    THREADS_ENTER
    task = grdc_ftp_window_get_waiting_task (GRDC_FTP_WINDOW (window));
    if (task)
    {
        window->taskid = task->taskid;

        task->status = GRDC_FTP_TASK_STATUS_RUN;
        grdc_ftp_window_update_task (GRDC_FTP_WINDOW (window), task);
    }
    THREADS_LEAVE

    return task;
}

static gboolean
grdc_sftp_window_thread_download_file (GrdcSFTPWindow *window, GrdcSFTP *sftp, GrdcFTPTask *task,
    const gchar *remote_path, const gchar *local_path, guint64 *donesize)
{
    SFTP_FILE *remote_file;
    FILE *local_file;
    gchar *tmp;
    gchar buf[20000];
    gint len;

    if (THREAD_CHECK_EXIT) return FALSE;

    /* Ensure local dir exists */
    g_strlcpy (buf, local_path, sizeof (buf));
    tmp = g_strrstr (buf, "/");
    if (tmp && tmp != buf)
    {
        *tmp = '\0';
        if (g_mkdir_with_parents (buf, 0755) < 0)
        {
            grdc_sftp_window_thread_set_error (window, task, _("Error creating directory %s."), buf);
            return FALSE;
        }
    }

    tmp = grdc_ssh_unconvert (GRDC_SSH (sftp), remote_path);
    remote_file = sftp_open (sftp->sftp_sess, tmp, O_RDONLY, 0);
    g_free (tmp);

    if (!remote_file)
    {
        grdc_sftp_window_thread_set_error (window, task, _("Error opening file %s on server. %s"),
            remote_path, ssh_get_error (GRDC_SSH (window->sftp)->session));
        return FALSE;
    }
    local_file = g_fopen (local_path, "wb");
    if (!local_file)
    {
        sftp_close (remote_file);
        grdc_sftp_window_thread_set_error (window, task, _("Error creating file %s."), local_path);
        return FALSE;
    }

    while (!THREAD_CHECK_EXIT && (len = sftp_read (remote_file, buf, sizeof (buf))) > 0)
    {
        if (THREAD_CHECK_EXIT) break;

        if (fwrite (buf, 1, len, local_file) < len)
        {
            sftp_close (remote_file);
            fclose (local_file);
            grdc_sftp_window_thread_set_error (window, task, _("Error writing file %s."), local_path);
            return FALSE;
        }

        *donesize += (guint64) len;
        task->donesize = (gfloat) (*donesize);

        if (!grdc_sftp_window_thread_update_task (window, task)) break;
    }

    sftp_close (remote_file);
    fclose (local_file);
    return TRUE;
}

static gboolean
grdc_sftp_window_thread_recursive_dir (GrdcSFTPWindow *window, GrdcSFTP *sftp, GrdcFTPTask *task,
    const gchar *rootdir_path, const gchar *subdir_path, GPtrArray *array)
{
    SFTP_DIR *sftpdir;
    SFTP_ATTRIBUTES *sftpattr;
    gchar *tmp;
    gchar *dir_path;
    gchar *file_path;
    gint type;
    gboolean ret = TRUE;

    if (THREAD_CHECK_EXIT) return FALSE;

    if (subdir_path)
    {
        dir_path = grdc_public_combine_path (rootdir_path, subdir_path);
    }
    else
    {
        dir_path = g_strdup (rootdir_path);
    }
    tmp = grdc_ssh_unconvert (GRDC_SSH (sftp), dir_path);
    sftpdir = sftp_opendir (sftp->sftp_sess, tmp);
    g_free (tmp);

    if (!sftpdir)
    {
        grdc_sftp_window_thread_set_error (window, task, _("Error opening directory %s. %s"),
            dir_path, ssh_get_error (GRDC_SSH (window->sftp)->session));
        g_free (dir_path);
        return FALSE;
    }

    g_free (dir_path);

    while ((sftpattr = sftp_readdir (sftp->sftp_sess, sftpdir)))
    {
        if (g_strcmp0 (sftpattr->name, ".") != 0 &&
            g_strcmp0 (sftpattr->name, "..") != 0)
        {
            GET_SFTPATTR_TYPE (sftpattr, type);

            tmp = grdc_ssh_convert (GRDC_SSH (sftp), sftpattr->name);
            if (subdir_path)
            {
                file_path = grdc_public_combine_path (subdir_path, tmp);
                g_free (tmp);
            }
            else
            {
                file_path = tmp;
            }

            if (type == GRDC_FTP_FILE_TYPE_DIR)
            {
                ret = grdc_sftp_window_thread_recursive_dir (window, sftp, task, rootdir_path, file_path, array);
                g_free (file_path);
                if (!ret)
                {
                    sftp_attributes_free (sftpattr);
                    break;
                }
            }
            else
            {
                task->size += (gfloat) sftpattr->size;
                g_ptr_array_add (array, file_path);

                if (!grdc_sftp_window_thread_update_task (window, task))
                {
                    sftp_attributes_free (sftpattr);
                    break;
                }
            }
        }
        sftp_attributes_free (sftpattr);

        if (THREAD_CHECK_EXIT) break;
    }

    sftp_closedir (sftpdir);
    return ret;
}

static gboolean
grdc_sftp_window_thread_upload_file (GrdcSFTPWindow *window, GrdcSFTP *sftp, GrdcFTPTask *task,
    const gchar *remote_path, const gchar *local_path, guint64 *donesize)
{
    SFTP_FILE *remote_file;
    FILE *local_file;
    gchar *tmp;
    gchar buf[20000];
    gint len;

    if (THREAD_CHECK_EXIT) return FALSE;

    tmp = grdc_ssh_unconvert (GRDC_SSH (sftp), remote_path);
    remote_file = sftp_open (sftp->sftp_sess, tmp, O_WRONLY | O_CREAT | O_TRUNC, 0);
    g_free (tmp);

    if (!remote_file)
    {
        grdc_sftp_window_thread_set_error (window, task, _("Error creating file %s on server. %s"),
            remote_path, ssh_get_error (GRDC_SSH (window->sftp)->session));
        return FALSE;
    }
    local_file = g_fopen (local_path, "rb");
    if (!local_file)
    {
        sftp_close (remote_file);
        grdc_sftp_window_thread_set_error (window, task, _("Error opening file %s."), local_path);
        return FALSE;
    }

    while (!THREAD_CHECK_EXIT && (len = fread (buf, 1, sizeof (buf), local_file)) > 0)
    {
        if (THREAD_CHECK_EXIT) break;

        if (sftp_write (remote_file, buf, len) < len)
        {
            sftp_close (remote_file);
            fclose (local_file);
            grdc_sftp_window_thread_set_error (window, task, _("Error writing file %s on server. %s"),
                remote_path, ssh_get_error (GRDC_SSH (window->sftp)->session));
            return FALSE;
        }

        *donesize += (guint64) len;
        task->donesize = (gfloat) (*donesize);

        if (!grdc_sftp_window_thread_update_task (window, task)) break;
    }

    sftp_close (remote_file);
    fclose (local_file);
    return TRUE;
}

static gpointer
grdc_sftp_window_thread_main (gpointer data)
{
    GrdcSFTPWindow *window = GRDC_SFTP_WINDOW (data);
    GrdcSFTP *sftp = NULL;
    GrdcFTPTask *task;
    gchar *remote, *local;
    guint64 size;
    GPtrArray *array;
    gint i;
    gchar *remote_file, *local_file;
    gboolean ret;
    gchar *refreshdir = NULL;
    gchar *tmp;
    gboolean refresh = FALSE;

    task = grdc_sftp_window_thread_get_task (window);
    while (task)
    {
        size = 0;
        if (!sftp)
        {
            sftp = grdc_sftp_new_from_ssh (GRDC_SSH (window->sftp));
            if (!grdc_ssh_init_session (GRDC_SSH (sftp)) ||
                grdc_ssh_auth (GRDC_SSH (sftp), NULL) <= 0 ||
                !grdc_sftp_open (sftp))
            {
                grdc_sftp_window_thread_set_error (window, task, (GRDC_SSH (sftp))->error);
                grdc_ftp_task_free (task);
                break;
            }
        }

        remote = grdc_public_combine_path (task->remotedir, task->name);
        local = grdc_public_combine_path (task->localdir, task->name);

        switch (task->tasktype)
        {
        case GRDC_FTP_TASK_TYPE_DOWNLOAD:
            switch (task->type)
            {
            case GRDC_FTP_FILE_TYPE_FILE:
                if (grdc_sftp_window_thread_download_file (window, sftp, task,
                    remote, local, &size))
                {
                    grdc_sftp_window_thread_set_finish (window, task);
                }
                break;

            case GRDC_FTP_FILE_TYPE_DIR:
                array = g_ptr_array_new ();
                ret = grdc_sftp_window_thread_recursive_dir (window, sftp, task, remote, NULL, array);
                if (ret)
                {
                    for (i = 0; i < array->len; i++)
                    {
                        if (THREAD_CHECK_EXIT)
                        {
                            ret = FALSE;
                            break;
                        }
                        remote_file = grdc_public_combine_path (remote, (gchar*) g_ptr_array_index (array, i));
                        local_file = grdc_public_combine_path (local, (gchar*) g_ptr_array_index (array, i));
                        ret = grdc_sftp_window_thread_download_file (window, sftp, task,
                            remote_file, local_file, &size);
                        g_free (remote_file);
                        g_free (local_file);
                        if (!ret) break;
                    }
                }
                g_ptr_array_free (array, TRUE);
                if (ret)
                {
                    grdc_sftp_window_thread_set_finish (window, task);
                }
                break;
            }
            break;

        case GRDC_FTP_TASK_TYPE_UPLOAD:
            if (grdc_sftp_window_thread_upload_file (window, sftp, task,
                remote, local, &size))
            {
                grdc_sftp_window_thread_set_finish (window, task);
                tmp = grdc_ftp_window_get_dir (GRDC_FTP_WINDOW (window));
                if (g_strcmp0 (tmp, task->remotedir) == 0)
                {
                    refresh = TRUE;
                    g_free (refreshdir);
                    refreshdir = tmp;
                }
                else
                {
                    g_free (tmp);
                }
            }
            break;
        }

        g_free (remote);
        g_free (local);

        grdc_ftp_task_free (task);
        window->taskid = 0;

        if (window->thread_abort) break;

        task = grdc_sftp_window_thread_get_task (window);
    }

    if (sftp)
    {
        grdc_sftp_free (sftp);
    }

    if (!window->thread_abort && refresh)
    {
        tmp = grdc_ftp_window_get_dir (GRDC_FTP_WINDOW (window));
        if (g_strcmp0 (tmp, refreshdir) == 0)
        {
            IDLE_ADD ((GSourceFunc) grdc_sftp_window_refresh, window);
        }
        g_free (tmp);
    }
    g_free (refreshdir);
    window->thread = 0;

    return NULL;
}

/* ------------------------ The SFTP Window routines ----------------------------- */

static gboolean
grdc_sftp_window_confirm_quit (GrdcSFTPWindow *window)
{
    GtkWidget *dialog;
    gint ret;

    if (!window->thread) return TRUE;

    dialog = gtk_message_dialog_new (GTK_WINDOW (window),
        GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
        _("File transfer currently in progress.\nAre you sure to quit?"));
    ret = gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
    if (ret == GTK_RESPONSE_YES) return TRUE;
    return FALSE;
}

static gboolean
grdc_sftp_window_on_delete_event (GrdcSFTPWindow *window, GdkEvent *event, gpointer data)
{
    return !grdc_sftp_window_confirm_quit (window);
}

static void
grdc_sftp_window_on_quit (GrdcSFTPWindow *window, gpointer data)
{
    if (grdc_sftp_window_confirm_quit (window))
    {
        gtk_widget_destroy (GTK_WIDGET (window));
    }
}

static void
grdc_sftp_window_destroy (GrdcSFTPWindow *window, gpointer data)
{
    if (window->sftp)
    {
        grdc_sftp_free (window->sftp);
        window->sftp = NULL;
    }
    window->thread_abort = TRUE;
    /* We will wait for the thread to quit itself, and hopefully the thread is handling things correctly */
    if (window->thread) pthread_join (window->thread, NULL);
    if (window->quit_when_close) gtk_main_quit();
}

static SFTP_DIR*
grdc_sftp_window_sftp_session_opendir (GrdcSFTPWindow *window, const gchar *dir)
{
    SFTP_DIR *sftpdir;
    GtkWidget *dialog;

    sftpdir = sftp_opendir (window->sftp->sftp_sess, (gchar*) dir);
    if (!sftpdir)
    {
        dialog = gtk_message_dialog_new (GTK_WINDOW (window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
            _("Failed to open directory %s. %s"), dir,
            ssh_get_error (GRDC_SSH (window->sftp)->session));
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
        return NULL;
    }
    return sftpdir;
}

static gboolean
grdc_sftp_window_sftp_session_closedir (GrdcSFTPWindow *window, SFTP_DIR *sftpdir)
{
    GtkWidget *dialog;

    if (!sftp_dir_eof (sftpdir))
    {
        dialog = gtk_message_dialog_new (GTK_WINDOW (window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
            _("Failed reading directory. %s"), ssh_get_error (GRDC_SSH (window->sftp)->session));
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
        return FALSE;
    }
    sftp_closedir (sftpdir);
    return TRUE;
}

static void
grdc_sftp_window_on_opendir (GrdcSFTPWindow *window, gchar *dir, gpointer data)
{
    SFTP_DIR *sftpdir;
    SFTP_ATTRIBUTES *sftpattr;
    GtkWidget *dialog;
    gchar *newdir;
    gchar *newdir_conv;
    gchar *tmp;
    gint type;

    if (!dir || dir[0] == '\0')
    {
        newdir = g_strdup (".");
    }
    else if (dir[0] == '/')
    {
        newdir = g_strdup (dir);
    }
    else
    {
        tmp = grdc_ftp_window_get_dir (GRDC_FTP_WINDOW (window));
        if (tmp)
        {
            newdir = grdc_public_combine_path (tmp, dir);
            g_free (tmp);
        }
        else
        {
            newdir = g_strdup_printf ("./%s", dir);
        }
    }

    tmp = grdc_ssh_unconvert (GRDC_SSH (window->sftp), newdir);
    newdir_conv = sftp_canonicalize_path (window->sftp->sftp_sess, tmp);
    g_free (tmp);
    g_free (newdir);
    newdir = grdc_ssh_convert (GRDC_SSH (window->sftp), newdir_conv);
    if (!newdir)
    {
        dialog = gtk_message_dialog_new (GTK_WINDOW (window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
            _("Failed to open directory %s. %s"), dir,
            ssh_get_error (GRDC_SSH (window->sftp)->session));
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
        g_free (newdir_conv);
        return;
    }

    sftpdir = grdc_sftp_window_sftp_session_opendir (window, newdir_conv);
    g_free (newdir_conv);
    if (!sftpdir)
    {
        g_free (newdir);
        return;
    }

    grdc_ftp_window_clear_file_list (GRDC_FTP_WINDOW (window));

    while ((sftpattr = sftp_readdir (window->sftp->sftp_sess, sftpdir)))
    {
        if (g_strcmp0 (sftpattr->name, ".") != 0 &&
            g_strcmp0 (sftpattr->name, "..") != 0)
        {
            GET_SFTPATTR_TYPE (sftpattr, type);

            tmp = grdc_ssh_convert (GRDC_SSH (window->sftp), sftpattr->name);
            grdc_ftp_window_add_file (GRDC_FTP_WINDOW (window),
                GRDC_FTP_FILE_COLUMN_TYPE, type,
                GRDC_FTP_FILE_COLUMN_NAME, tmp,
                GRDC_FTP_FILE_COLUMN_SIZE, (gfloat) sftpattr->size,
                GRDC_FTP_FILE_COLUMN_USER, sftpattr->owner,
                GRDC_FTP_FILE_COLUMN_GROUP, sftpattr->group,
                GRDC_FTP_FILE_COLUMN_PERMISSION, sftpattr->permissions,
                -1);
            g_free (tmp);
        }
        sftp_attributes_free (sftpattr);
    }
    grdc_sftp_window_sftp_session_closedir (window, sftpdir);

    grdc_ftp_window_set_dir (GRDC_FTP_WINDOW (window), newdir);
    g_free (newdir);
}

static void
grdc_sftp_window_on_newtask (GrdcSFTPWindow *window, gpointer data)
{
    if (window->thread) return;

    if (pthread_create (&window->thread, NULL, grdc_sftp_window_thread_main, window))
    {
        window->thread = 0;
    }
}

static gboolean
grdc_sftp_window_on_canceltask (GrdcSFTPWindow *window, gint taskid, gpointer data)
{
    GtkWidget *dialog;
    gint ret;

    if (window->taskid != taskid) return TRUE;

    dialog = gtk_message_dialog_new (GTK_WINDOW (window),
        GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
        _("File transfer currently in progress.\nAre you sure to cancel it?"));
    ret = gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
    if (ret == GTK_RESPONSE_YES)
    {
        /* Make sure we are still handling the same task before we clear the flag */
        if (window->taskid == taskid) window->taskid = 0;
        return TRUE;
    }
    return FALSE;
}

static gboolean
grdc_sftp_window_on_deletefile (GrdcSFTPWindow *window, gint type, gchar *name, gpointer data)
{
    GtkWidget *dialog;
    gint ret = 0;
    gchar *tmp;

    tmp = grdc_ssh_unconvert (GRDC_SSH (window->sftp), name);
    switch (type)
    {
    case GRDC_FTP_FILE_TYPE_DIR:
        ret = sftp_rmdir (window->sftp->sftp_sess, tmp);
        break;

    case GRDC_FTP_FILE_TYPE_FILE:
        ret = sftp_unlink (window->sftp->sftp_sess, tmp);
        break;
    }
    g_free (tmp);

    if (ret != 0)
    {
        dialog = gtk_message_dialog_new (GTK_WINDOW (window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
            _("Failed to delete '%s'. %s"),
            name, ssh_get_error (GRDC_SSH (window->sftp)->session));
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
        return FALSE;
    }
    return TRUE;
}

static void
grdc_sftp_window_init (GrdcSFTPWindow *window)
{
    window->sftp = NULL;
    window->thread = 0;
    window->taskid = 0;
    window->thread_abort = FALSE;

    /* Setup the internal signals */
    g_signal_connect (G_OBJECT (window), "delete-event",
        G_CALLBACK (grdc_sftp_window_on_delete_event), NULL);
    g_signal_connect (G_OBJECT (window), "quit",
        G_CALLBACK (grdc_sftp_window_on_quit), NULL);
    g_signal_connect (G_OBJECT (window), "destroy",
        G_CALLBACK (grdc_sftp_window_destroy), NULL);
    g_signal_connect (G_OBJECT (window), "open-dir",
        G_CALLBACK (grdc_sftp_window_on_opendir), NULL);
    g_signal_connect (G_OBJECT (window), "new-task",
        G_CALLBACK (grdc_sftp_window_on_newtask), NULL);
    g_signal_connect (G_OBJECT (window), "cancel-task",
        G_CALLBACK (grdc_sftp_window_on_canceltask), NULL);
    g_signal_connect (G_OBJECT (window), "delete-file",
        G_CALLBACK (grdc_sftp_window_on_deletefile), NULL);
}

static gboolean
grdc_sftp_window_refresh (GrdcSFTPWindow *window)
{
    gdk_window_set_cursor (GTK_WIDGET (window)->window, gdk_cursor_new (GDK_WATCH));
    gdk_flush ();

    grdc_sftp_window_on_opendir (window, ".", NULL);

    gdk_window_set_cursor (GTK_WIDGET (window)->window, NULL);

    return FALSE;
}

static GtkWidget*
grdc_sftp_window_create_window (GtkWindow *parent, GrdcSFTP *sftp, gboolean quit_when_close)
{
    GrdcSFTPWindow *window;
    gchar buf[200];

    window = GRDC_SFTP_WINDOW (g_object_new (GRDC_TYPE_SFTP_WINDOW, NULL));

    window->sftp = sftp;
    window->quit_when_close = quit_when_close;

    if (parent)
    {
        gtk_window_set_transient_for (GTK_WINDOW (window), parent);
    }
    gtk_window_set_icon (GTK_WINDOW (window), grdc_pixmap (GRDC_PIXMAP_TYPE_SSH, 0));

    /* Title */
    g_snprintf (buf, sizeof (buf), _("Secure File Transfer: %s"), GRDC_SSH (sftp)->server);
    gtk_window_set_title (GTK_WINDOW (window), buf);

    gtk_widget_show (GTK_WIDGET (window));
    gtk_window_present (GTK_WINDOW (window));

    return GTK_WIDGET (window);
}

GtkWidget*
grdc_sftp_window_new (GtkWindow *parent, GrdcSFTP *sftp, gboolean quit_when_close)
{
    GtkWidget *window;

    window = grdc_sftp_window_create_window (parent, sftp, quit_when_close);
    g_idle_add ((GSourceFunc) grdc_sftp_window_refresh, window);
    return window;
}

GtkWidget*
grdc_sftp_window_new_init (GtkWindow *parent, GrdcSFTP *sftp, gboolean quit_when_close)
{
    GtkWidget *window;
    GtkWidget *dialog;

    window = grdc_sftp_window_create_window (parent, sftp, quit_when_close);

    gdk_window_set_cursor (GTK_WIDGET (window)->window, gdk_cursor_new (GDK_WATCH));
    gdk_flush ();

    if (!grdc_ssh_init_session (GRDC_SSH (sftp)) ||
        grdc_ssh_auth (GRDC_SSH (sftp), NULL) <= 0 ||
        !grdc_sftp_open (sftp))
    {
        dialog = gtk_message_dialog_new (GTK_WINDOW (window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
            (GRDC_SSH (sftp))->error, NULL);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
        gtk_widget_destroy (window);
        return NULL;
    }

    gdk_window_set_cursor (GTK_WIDGET (window)->window, NULL);

    g_idle_add ((GSourceFunc) grdc_sftp_window_refresh, window);
    return window;
}

typedef struct _GrdcSFTPWindowInitParam
{
    GrdcFile *grdc_file;
    gboolean quit_when_close;
    GtkWidget *init_dialog;
    pthread_t thread;
    gchar *error;
    GrdcSFTP *sftp;
} GrdcSFTPWindowInitParam;

static void
grdc_sftp_window_init_cancel (GrdcInitDialog *dialog, gint response_id, GrdcSFTPWindowInitParam *param)
{
    if ((response_id == GTK_RESPONSE_CANCEL || response_id == GTK_RESPONSE_DELETE_EVENT)
        && dialog->mode == GRDC_INIT_MODE_CONNECTING)
    {
        gtk_widget_destroy (GTK_WIDGET (dialog));
    }
}

static void
grdc_sftp_window_init_destroy (GrdcInitDialog *dialog, GrdcSFTPWindowInitParam *param)
{
    GtkWidget *msg_dialog;

    if (param->error)
    {
        msg_dialog = gtk_message_dialog_new (NULL,
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
            param->error, NULL);
        gtk_dialog_run (GTK_DIALOG (msg_dialog));
        gtk_widget_destroy (msg_dialog);

        g_free (param->error);
    }
    if (param->thread)
    {
        pthread_cancel (param->thread);
        pthread_join (param->thread, NULL);
    }
    if (param->sftp)
    {
        grdc_sftp_free (param->sftp);
    }
    if (param->quit_when_close)
    {
        gtk_main_quit ();
    }
    if (param->grdc_file)
    {
        grdc_file_free (param->grdc_file);
    }
    g_free (param);
}

static gboolean
grdc_sftp_window_open_thread_quit (GrdcSFTPWindowInitParam *param)
{
    gtk_widget_destroy (param->init_dialog);
    return FALSE;
}

static gboolean
grdc_sftp_window_open_thread_start (GrdcSFTPWindowInitParam *param)
{
    GrdcSFTP *sftp = param->sftp;
    gboolean quit_when_close = param->quit_when_close;

    /* Hand over the sftp session pointer from the initial param to the sftp window */
    param->sftp = NULL;
    param->quit_when_close = FALSE;

    /* Remember recent list for quick connect */
    if (param->grdc_file->filename == NULL)
    {
        grdc_pref_add_recent (param->grdc_file->protocol, param->grdc_file->server);
    }

    gtk_widget_destroy (param->init_dialog);

    grdc_sftp_window_new (NULL, sftp, quit_when_close);
    return FALSE;
}

static gpointer
grdc_sftp_window_open_thread (gpointer data)
{
    GrdcSFTPWindowInitParam *param = (GrdcSFTPWindowInitParam*) data;
    gint ret;

    pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL);
    CANCEL_ASYNC

    param->sftp = grdc_sftp_new_from_file (param->grdc_file);

    if (!grdc_ssh_init_session (GRDC_SSH (param->sftp)))
    {
        param->error = g_strdup (GRDC_SSH (param->sftp)->error);
        IDLE_ADD ((GSourceFunc) grdc_sftp_window_open_thread_quit, data);
        return NULL;
    }

    ret = grdc_ssh_auth_gui (GRDC_SSH (param->sftp), GRDC_INIT_DIALOG (param->init_dialog), TRUE);
    if (ret <= 0)
    {
        if (ret == 0)
        {
            param->error = g_strdup (GRDC_SSH (param->sftp)->error);
        }
        IDLE_ADD ((GSourceFunc) grdc_sftp_window_open_thread_quit, data);
        return NULL;
    }

    if (!grdc_sftp_open (param->sftp))
    {
        param->error = g_strdup (GRDC_SSH (param->sftp)->error);
        IDLE_ADD ((GSourceFunc) grdc_sftp_window_open_thread_quit, data);
        return NULL;
    }

    /* We are done initializing. Unregister the thread from the param struct and open up the sftp window */
    param->thread = 0;

    IDLE_ADD ((GSourceFunc) grdc_sftp_window_open_thread_start, data);
    return NULL;
}

void
grdc_sftp_window_open (GtkWindow *parent, GrdcFile *grdcfile, gboolean quit_when_close)
{
    GrdcSFTPWindowInitParam *param;

    param = g_new (GrdcSFTPWindowInitParam, 1);
    param->grdc_file = grdcfile;
    param->quit_when_close = quit_when_close;
    param->thread = 0;
    param->error = NULL;
    param->sftp = NULL;

    param->init_dialog = grdc_init_dialog_new (parent, _("Connecting to SFTP server '%s'..."),
        (grdcfile->filename ? grdcfile->name : grdcfile->server));
    g_signal_connect (G_OBJECT (param->init_dialog), "response",
        G_CALLBACK (grdc_sftp_window_init_cancel), param);
    g_signal_connect (G_OBJECT (param->init_dialog), "destroy",
        G_CALLBACK (grdc_sftp_window_init_destroy), param);
    gtk_widget_show (param->init_dialog);

    g_free (grdcfile->ssh_server);
    grdcfile->ssh_server = g_strdup (grdcfile->server);

    if (pthread_create (&param->thread, NULL, grdc_sftp_window_open_thread, param))
    {
        param->thread = 0;
        param->error = g_strdup ("Failed to initialize pthread.");
        gtk_widget_destroy (param->init_dialog);
        if (quit_when_close) gtk_main_quit ();
    }
}

#endif

